#! /usr/bin/perl 
use strict;

use Getopt::Long; 

##########################################################################################
#											 #
#	       Pre-process : dealing with options, reading dictionary			 #
#											 #
##########################################################################################

##########################
#Global technic variables#
##########################
my $script = "giec-eval_convertor.pl"; 
my $help = << "end_of_help;";

Usage: giec-eval_convertor.pl -r <responses-file> -d <dictionary-file>
Options :
	-h	help
	-r	path of the response-file
	-d	path of the dictionary
please questioning and reporting bugs at lll05\@jouy.inra.fr

end_of_help;
my %options;

#############################
#Global conceptual variables#
#############################
my @ListOfFormsAndSyns;
my @ListOfCanonicalForms;
my @Responses;
my @OutputLines;

#Dealing with options
GetOptions(\%options, 
	   "h",
	   "r=s",
	   "d=s",
           ); 

if (defined ($options{'h'})) {print $help;exit;};


if (not (defined ($options{'r'}))) {
		print "Caution : please give \"r\" option\n";
		print $help;
		exit;
		};

if (not (defined ($options{'d'}))) {
		print "Caution : please give \"d\" option\n";
		print $help;
		exit;
		};

if ($#ARGV>=0) {die "Error : $script takes no arguments ! \n"};

#Reading Dictionary
open(DIC,$options{'d'}) or die "$script: could not open file ($options{'d'}) ! \n";
my @dicLines=<DIC>;
close DIC;
foreach my $iRD (0..$#dicLines) {
	my @listOfOneFormAndSyns;
	if ($dicLines[$iRD]=~ s/^([^%|\r|\n|\t]+)[\r|\n|\t]//) {
		push(@ListOfCanonicalForms,$1);
		push(@listOfOneFormAndSyns,$1);
		while ($dicLines[$iRD]=~ s/^([^%|\r|\n|\t]+)[\r|\n|\t]//) {
			push(@listOfOneFormAndSyns,$1);
			};
		push (@ListOfFormsAndSyns,\@listOfOneFormAndSyns);
		};
	};


##########################################################################################
#											 #
#	       			       Main program					 #
#											 #
##########################################################################################

readAndSaveResponses($options{'r'});
foreach my $i (0..$#OutputLines) {
	print "$OutputLines[$i]";
	};

##########################################################################################
#											 #
#	 		 Part I : Reading & checking responses				 #
#											 #
##########################################################################################

sub readAndSaveResponses {
#read the response file, and save in @OutputLines the same output, but in canonical forms

	my $nameFile=$_[0];
	open (FILE,$nameFile) or die "$script: could not open file ($nameFile) ! \n"; 
	my @lines=<FILE>;#array of lines from response file
	close FILE;

	my $irASR=0;
	while (not ($irASR>$#lines)) {
	
		# jumping blank lines
		while (not ($lines[$irASR] =~ /^ID/)) {
			push (@OutputLines,$lines[$irASR]);
			$irASR++;
			if ($irASR>$#lines) {last;}
			};
		if ($irASR>$#lines) {last;}
		
		my %response;#response is a hash with 1 ID, 1 list of agents, 1 list of targets, and 1 list of interactions
	
		# reading ID
		if (not ($lines[$irASR] =~ /^ID\t([0-9|-]+)[\r|\n]/)) {die "$script: format error \($nameFile line ".($irASR+1)."\) ! \n";}
			else {$response{'ID'}=$1};
		
		# jumping to next line
		$irASR++;
		if ($irASR>$#lines) {die "$script: format error \($nameFile line ".($irASR+1)."\)! \n";};

		# jumping sentence
		if (not ($lines[$irASR] =~ /^sentence\t/)) {die "$script: format error \($nameFile line ".($irASR+1)."\), sentence waited ! \n";} else {$irASR++;};
		if ($irASR>$#lines) {die "$script: format error \($nameFile line ".($irASR+1)."\)! \n";};

		# reading words and saving words
		my @wordsDeclared;
		if (not ($lines[$irASR] =~ /^words\t/)) {die "$script: format error \($nameFile line ".($irASR+1)."\), words waited ! \n";} else {
			while ($lines[$irASR]=~ s/^words\t+word\([0-9]+,\'([^\']+)\'[^\t|\r|\n]+[\t|\r|\n]/words\t/){
				push (@wordsDeclared,$1);
				};
			$irASR++;};
		if ($irASR>$#lines) {die "$script: format error \($nameFile line ".($irASR+1)."\)! \n";};

		# jumping lemmas (optional)
		if ($lines[$irASR] =~ /^lemmas\t/) {$irASR++;};
		if ($irASR>$#lines) {die "$script: format error \($nameFile line ".($irASR+1)."\)! \n";};

		# jumping syntactic_relations (optional)
		if ($lines[$irASR] =~ /^syntactic_relations\t/) {$irASR++;};
		if ($irASR>$#lines) {die "$script: format error \($nameFile line ".($irASR+1)."\)! \n";};

		#reading agents IDs, finding the correspondant word and taking the canonical form
		my @listAgents;
		if (not ($lines[$irASR] =~ /^agents/)) {die "$script: format error \($nameFile line ".($irASR+1)."\) ! \n";}
			else {
			while ($lines[$irASR] =~ s/^agents\t([^\t|\r|\n]+)[\t|\r|\n]/agents\t/) {
				if (not ($1=~ /^agent\(([0-9]+)\)/)) {die "$script: format error \($nameFile line ".($irASR+1)."\) ! \n";}
					else {  my $canonicalForm=canonicalForm($wordsDeclared[$1]);
						if ($canonicalForm eq "") {die "Error : in $nameFile, for ID ".$response{'ID'}.", agent ".$wordsDeclared[$1]." is not referenced in the dictionary\n"} else {
							push (@listAgents,canonicalForm($wordsDeclared[$1]));
							};
						}
				};
			$response{'agents'}=\@listAgents;
			};
	
		# jumping to next line
		$irASR++;
		if ($irASR>$#lines) {die "$script: format error \($nameFile line ".($irASR+1)."\)! \n";};
	
		#reading targets, finding the correspondant word and taking the canonical form
		my @listTargets;
		if (not ($lines[$irASR] =~ /^targets/)) {die "$script: format error \($nameFile line ".($irASR+1)."\) ! \n";}
			else {
			while ($lines[$irASR] =~ s/^targets\t([^\t|\r|\n]+)[\t|\r|\n]/targets\t/) {
				if (not ($1=~ /^target\(([0-9]+)\)/)) {die "$script: format error \($nameFile line ".($irASR+1)."\) ! \n";}
					else {  my $canonicalForm=canonicalForm($wordsDeclared[$1]);
						if ($canonicalForm eq "") {die "Error : in $nameFile, for ID ".$response{'ID'}.", target ".$wordsDeclared[$1]." is not referenced in the dictionary\n"} else {
							push (@listTargets,canonicalForm($wordsDeclared[$1]));
							};
						}
				};
			$response{'targets'}=\@listTargets;
			};
	
		# jumping to next line
		$irASR++;
		if ($irASR>$#lines) {die "$script: format error \($nameFile line ".($irASR+1)."\)! \n";};
	
		#reading interactions, checking if agents and target are declared
		my @listInteractions;
		if (not ($lines[$irASR] =~ /^genic_interactions/)) {die "$script: format error \($nameFile line ".($irASR+1)."\) ! \n";}
			else {
			while ($lines[$irASR] =~ s/^genic_interactions\t([^\t|\r|\n]+)[\t|\r|\n]/genic_interactions\t/) {
				if (not ($1=~ /^genic_interaction\(([0-9]+),([0-9]+)\)$/)) {die "$script: format error \($nameFile line ".($irASR+1)."\) ! \n";}
				else {
					my %interaction;
					$interaction{'agent'}=canonicalForm($wordsDeclared[$1]);
					if ($interaction{'agent'} eq "") {die "Error : in $nameFile, for ID ".$response{'ID'}.", the ID $1 of an agent in an interaction is wrong or not declared\n"};
					$interaction{'target'}=canonicalForm($wordsDeclared[$2]);
					if ($interaction{'target'} eq "") {die "Error : in $nameFile, for ID ".$response{'ID'}.", the ID $2 of a target in an interaction is wrong ot not declared\n"};
					push(@listInteractions,\%interaction);
					};
				};
			$response{'interactions'}=\@listInteractions;
			};
	
		#saving response in OutputLines
		my $line1="ID\t".$response{'ID'}."\n";
		push (@OutputLines,$line1);

		my $line2="agents";
		foreach my $ilA (0..$#listAgents) {$line2=$line2."\t"."agent\(\'".$listAgents[$ilA]."\'\)";};
		$line2=$line2."\n";
		push (@OutputLines,$line2);

		my $line3="targets";
		foreach my $ilT (0..$#listTargets) {$line3=$line3."\t"."target\(\'".$listTargets[$ilT]."\'\)";};
		$line3=$line3."\n";
		push (@OutputLines,$line3);

		my $line4="genic_interactions";
		foreach my $ilGI (0..$#listInteractions) {
			$line4=$line4."\t"."genic_interaction\(\'".$listInteractions[$ilGI]{'agent'}."\',";
			$line4=$line4."\'$listInteractions[$ilGI]{'target'}\'\)";
			};
		$line4=$line4."\n";
		push (@OutputLines,$line4);

		#going to next ID or end
		$irASR++;
	
		};

	};


##########################################################################################
#											 #
#		 		     Little scripts					 #
#											 #
##########################################################################################

sub canonicalForm{
#return the canonical form of the entity (1st parameter) if it exists
	my $entitycF=$_[0];
	my $returnedEntity="";

	#trying if it's is a canonical form
	if (myInclude($entitycF,\@ListOfCanonicalForms)) {$returnedEntity=$entitycF}

		else{

	#trying if it's a synonym
		foreach my $icF (0..$#ListOfCanonicalForms) {
			my @listOfOneFormAndSyns=@{$ListOfFormsAndSyns[$icF]};
			foreach my $jcF (1..$#listOfOneFormAndSyns) {
				if ($entitycF eq $listOfOneFormAndSyns[$jcF]) {
					$returnedEntity=$listOfOneFormAndSyns[0];last;
					};
				};
			if (not ($returnedEntity eq "")) {last;};
			};
		};

	return $returnedEntity;
	};

sub myInclude{
#return 1 if the element (1st parameter) is an element of the array (2nd parameter), 0 if not

	my $element=$_[0];
	my @array=@{$_[1]};
	
	my $found=0;
	my $imI=0;
	while ((not $found) and ($imI<=$#array)) {
		if ($element eq $array[$imI]) {$found=1};
		$imI++;
		};

	return $found;
	}
