package Gui::MapDrawingWindow;

use strict;

use Gtk;

use FeatureWidget;

use Gui::MapListWindow;

use PaintDevice;
use PaintDevice::GtkPaintDevice;
use PaintDevice::PNGPaintDevice;
use PaintDevice::PSPaintDevice;
use PaintDevice::XFigPaintDevice;

use PhysicalMapWidget;

my $MINBPL=50;

sub new {
     my $class=shift;
     my $manager=shift;
     my ($width,$height)=@_;

     $width=800
	 if (!defined $width || $width<200);
     $height=600
	 if (!defined $height || $height<150);
     my $self={};
     bless $self,$class;

     $self->{manager}=$manager;

     $self->{mapids}=();
     $self->{strands_expanded}=0;
     $self->{fwstartposition}=1;
     $self->{revstartposition}=1;
     $self->{nlines}=4;
     $self->{bpl}=5000;
     $self->{width}=$width;
     $self->{height}=$height;

     my $window=new Gtk::Window -toplevel;
     $window->set_title('Multi-Genome Navigator');
     $self->{window}=$window;
     $window->set_usize(800,600);
     $window->signal_connect('delete_event'=>sub { Gtk::main_quit($self->{window}); });
     $window->signal_connect('destroy_event'=>sub { Gtk::main_quit($self->{window}); });
     

     my $vbox=new Gtk::VBox(0,1);
     $vbox->show_all();
     $window->add($vbox);
     
     my $menubar=new Gtk::MenuBar;
     $vbox->pack_start($menubar,0,0,1);
     $self->{menubar}=$menubar;

     my $menuitem=new Gtk::MenuItem('File');
     $menubar->append($menuitem);
     my $submenu=new Gtk::Menu;
     $menuitem->set_submenu($submenu);
     $menuitem= new Gtk::MenuItem("Export Map");
     $submenu->append($menuitem);
     my $subsubmenu=new Gtk::Menu;
     $menuitem->set_submenu($subsubmenu);
     $menuitem = new Gtk::MenuItem("XFig");
     $menuitem->signal_connect("activate"=>sub{$self->_export_map('fig')});
     $subsubmenu->append($menuitem);
     $menuitem = new Gtk::MenuItem("PostScript");
     $menuitem->signal_connect("activate"=>sub{$self->_export_map('ps')});
     $subsubmenu->append($menuitem);     
     $menuitem = new Gtk::MenuItem("Encapsulated PostScript");
     $menuitem->signal_connect("activate"=>sub{$self->_export_map('eps')});
     $subsubmenu->append($menuitem);     
     $menuitem = new Gtk::MenuItem("PNG");
     $menuitem->signal_connect("activate"=>sub{$self->_export_map('png')});
     $subsubmenu->append($menuitem);     
     $menuitem=new Gtk::MenuItem('Quit');
     $menuitem->signal_connect('activate'=>sub{Gtk::main_quit($self->{window});});
     $submenu->append($menuitem);

     $menuitem=new Gtk::MenuItem('Preferences');
     $menubar->append($menuitem);
     $submenu=new Gtk::Menu;
     $menuitem->set_submenu($submenu);
     $menuitem=new Gtk::CheckMenuItem('Expand Strands');
     if (!ScalableFeatureWidget::are_strands_collapsed) {
	 $menuitem->set_active(1);
	 $self->{strands_expanded}=1;
     }
     $menuitem->signal_connect('toggled'=>sub{$self->_change_strand_state;});
     $submenu->append($menuitem);
     $menuitem=new Gtk::CheckMenuItem('Show Frames');
     if (ScalableFeatureWidget::are_frames_visible) {
	 $menuitem->set_active(1);
	 $self->{frames_visible}=1;
     }
     $menuitem->signal_connect('toggled'=>sub{$self->_change_frame_state;});
     $submenu->append($menuitem);
     $menuitem=new Gtk::MenuItem('Visible Features');
     $submenu->append($menuitem);
     $self->{featureitem}=$menuitem;

     $menuitem=new Gtk::MenuItem('Map Area Width');
     $submenu->append($menuitem);
     $subsubmenu=new Gtk::Menu();
     $menuitem->set_submenu($subsubmenu);
     my $previtem=undef;
     my %sizes=(800=>1,
		1000=>1,
		1200=>1,
		1600=>1,
		2000=>1,
		2400=>1,
		3000=>1,
		3600=>1,
		4000=>1,
		4800=>1,
		5600=>1,
		6400=>1,
		7200=>1,
		8000=>1,
		9000=>1,
		10000=>1);
     $sizes{$width}=1;
     foreach my $size (sort { $a <=> $b } keys %sizes) {
	 if (defined $previtem) {
	     $menuitem=new Gtk::RadioMenuItem("$size",$previtem);
	 } else {
	     $menuitem=new Gtk::RadioMenuItem("$size");

	 }
	 $menuitem->set_active(1)
	     if ($width == $size);
	 $menuitem->signal_connect('activate'=>sub{$self->_set_width($size);});
	 $subsubmenu->append($menuitem);
	 $previtem=$menuitem;
     }
     
     $menuitem=new Gtk::MenuItem('Color Schemes');
     $submenu->append($menuitem);
     $subsubmenu=new Gtk::Menu();
     $menuitem->set_submenu($subsubmenu);
     $previtem=undef;
     foreach my $color_scheme (sort keys %ColorHandlerFactory::Factory) {
	 $menuitem=new Gtk::RadioMenuItem($color_scheme,$previtem);
	 $menuitem->signal_connect('activate'=>sub{$self->_set_color_scheme($color_scheme)});
	 if ($FeatureWidget::ColorHandler->name eq $color_scheme) {
	     $menuitem->set_active(1);
	 } else {
	     $menuitem->set_active(0);
	 }
	 $subsubmenu->append($menuitem);
	 $previtem=$menuitem;
     }
     
     $menuitem=new Gtk::MenuItem('Save Preferences');
     $menuitem->signal_connect('activate'=>sub{$self->_save_preferences;});
     $submenu->append($menuitem);

     my $frame=new Gtk::Frame;
     $self->{frame}=$frame;
     $vbox->pack_start($frame,1,1,1);

     my $paintdevice=new PaintDevice::GtkPaintDevice($width,$height,$frame);
     $self->{paintdevice}=$paintdevice;

     my $mapwidget=new PhysicalMapWidget($paintdevice);
     $self->{mapwidget}=$mapwidget;

     my $hbox=new Gtk::HBox(0,1);
     $hbox->show();
     $vbox->pack_start($hbox,0,0,1);

     my $arrowbox=new Gtk::HBox(0,0);
     my $alignment=new Gtk::Alignment(1.0,0.5,0.0,1.0);
     $arrowbox->add($alignment);
     my $arrow=new Gtk::Arrow('left','out');
     $alignment->add($arrow);
     $alignment=new Gtk::Alignment(0.0,0.5,0.0,1.0);
     $arrowbox->add($alignment);
     $arrow=new Gtk::Arrow('left','out');
     $alignment->add($arrow);

     my $prevbtn=new Gtk::Button;
     $prevbtn->add($arrowbox);
     $prevbtn->show();
     $prevbtn->signal_connect('clicked',sub {$self->_move_btn_clicked(-1);});
     $hbox->pack_start($prevbtn,1,1,0);

     $arrow=new Gtk::Arrow('left','out');
     $prevbtn=new Gtk::Button;
     $prevbtn->add($arrow);
     $prevbtn->show();
     $prevbtn->signal_connect('clicked',sub {$self->_move_btn_clicked(-0.5);});
     $hbox->pack_start($prevbtn,1,1,2);

     $arrow=new Gtk::Arrow('right','out');
     my  $nextbtn=new Gtk::Button;
     $nextbtn->add($arrow);
     $nextbtn->show();
     $nextbtn->signal_connect('clicked',sub {$self->_move_btn_clicked(0.5);});
     $hbox->pack_start($nextbtn,1,1,1);

     $arrowbox=new Gtk::HBox(0,0);
     $alignment=new Gtk::Alignment(1.0,0.5,0.0,1.0);
     $arrowbox->add($alignment);
     $arrow=new Gtk::Arrow('right','out');
     $alignment->add($arrow);
     $alignment=new Gtk::Alignment(0.0,0.5,0.0,1.0);
     $arrowbox->add($alignment);
     $arrow=new Gtk::Arrow('right','out');
     $alignment->add($arrow);

     $nextbtn=new Gtk::Button;
     $nextbtn->add($arrowbox);
     $nextbtn->show();
     $nextbtn->signal_connect('clicked',sub {$self->_move_btn_clicked(1);});
     $hbox->pack_start($nextbtn,1,1,1);

     $hbox=new Gtk::HBox(0,4);
     $hbox->show();
     $vbox->pack_start($hbox,0,0,1);

     my $label=new Gtk::Label('Start base');
     $hbox->pack_start($label,0,1,1);
     my $entry=new Gtk::Entry(9);
     $hbox->pack_start($entry,1,1,1);
     $entry->signal_connect('activate',sub {$self->_map_entry_changed();});
     $entry->set_text($self->{fwstartposition});
     $self->{startentry}=$entry;
     $label=new Gtk::Label('Lines');
     $hbox->pack_start($label,0,1,1);
     $entry=new Gtk::Entry(9);
     $entry->signal_connect('activate',sub {$self->_map_entry_changed();});
     $hbox->pack_start($entry,1,1,1);
     $entry->set_text($self->{nlines});
     $self->{linesentry}=$entry;
     $label=new Gtk::Label('BPL');
     $hbox->pack_start($label,0,1,1);
     $entry=new Gtk::Entry(9);
     $entry->signal_connect('activate',sub {$self->_map_entry_changed();});
     $hbox->pack_start($entry,1,1,1);
     $entry->set_text($self->{bpl});
     $self->{bplentry}=$entry;

     $vbox=new Gtk::VBox(0,2);
     $hbox->pack_start($vbox,0,1,1);

     $hbox=new Gtk::HBox(0,2);
     $vbox->pack_start($hbox,0,1,1);
     $label=new Gtk::Label('Name display thresh.');
     $hbox->pack_start($label,0,1,1);
     my $namethreshadj=new Gtk::Adjustment($ScalableFeatureWidget::display_threshold,0.0,0.2,0.005,0.005,0.00000000001);
     $namethreshadj->signal_connect('value_changed',sub {$self->_map_entry_changed();});
     $self->{namethreshadj}=$namethreshadj;
     my $scale=new Gtk::HScale($namethreshadj);
     $scale->set_update_policy('delayed');
     $scale->set_draw_value(1);
     $scale->set_digits(3);
     $hbox->pack_start($scale,1,1,1);

     $hbox=new Gtk::HBox(0,2);
     $vbox->pack_start($hbox,0,1,1);
     $label=new Gtk::Label('Seq. display thresh.');
     $hbox->pack_start($label,0,1,1);
     my $seqthreshadj=new Gtk::Adjustment(100,1,1001,100,10,1);
     $seqthreshadj->signal_connect('value_changed',sub {$self->_map_entry_changed();});
     $self->{seqthreshadj}=$seqthreshadj;
     $scale=new Gtk::HScale($seqthreshadj);
     $scale->set_update_policy('delayed');
     $scale->set_draw_value(1);
     $scale->set_digits(2);
     $hbox->pack_start($scale,1,1,1);
     

     return $self;
}

sub append_map {
    my $self=shift;
    my $map=shift;

    my $start=$self->{fwstartposition};
    my $nlines=$self->{nlines};
    my $bpl=$self->{bpl};
    my $end=$start+$nlines*$bpl-1;

    my $mapid=$self->{mapwidget}->append_map($map,$start,$end,$bpl);

    push @{$self->{mapids}},$mapid;

    $self->render();

    return $mapid;
}

sub replace_map {
    my $self=shift;
    my $repmapid=shift;
    my $newmap=shift;

    $self->{mapwidget}->replace_map($newmap,$repmapid);
    $self->render();
}

sub remove_map {
    my $self=shift;
    my $delmapid=shift;

    my $newmapids=[];
    foreach my $mapid (@{$self->{mapids}}) {
	push @{$newmapids},$mapid
	    if ($mapid != $delmapid);
    }

    $self->{mapids}=$newmapids;
    $self->{mapwidget}->remove_map($delmapid);
    $self->render();
}

sub render {
    my $self=shift;
    $self->{mapwidget}->render();
    $self->_rebuild_feature_menu();
}


sub add_listener {
    my $self=shift;
    my $listener=shift;
    my $type=shift;

    $self->{paintdevice}->add_listener($listener,$type);
}

sub show_all {
    my $self=shift;
    $self->{window}->show_all();
}

sub get_start_position {
    my $self=shift;
    return $self->{fwstartposition};
}

sub set_start_position {
    my $self=shift;
    my $start=shift;

    my $deltastart=$start-$self->{fwstartposition};
    $self->{fwstartposition}=$start;
    $self->{revstartposition}=$self->{revstartposition}-$deltastart;
    $self->{startentry}->set_text($start);

    $self->update_map_display;
}

sub get_bpl {
    my $self=shift;
    return $self->{bpl};
}

sub set_bpl {
    my $self=shift;
    my $bpl=shift;
    
    $self->{bpl}=$bpl;
    $self->update_map_display;
}


sub get_display_parameters {
    my $self=shift;
    
    my $endposition=$self->{fwstartposition}+$self->{nlines}*$self->{bpl}-1;
    return ($self->{fwstartposition},$endposition,$self->{bpl});
}

sub set_display_parameters {
    my $self=shift;
    my ($start,$end,$bpl)=@_;
    
    $self->{fwstartposition}=$start;
    $self->{startentry}->set_text($start);
    $self->{bpl}=$bpl;
    $self->{bplentry}->set_text($bpl);
    $self->{nlines}=($end-$start+1)/$bpl;
    $self->{linesentry}->set_text($self->{nlines});

    $self->{update_map_display};
}


sub update_map_display {
    my $self=shift;

    my $fwstart=$self->{fwstartposition};
    my $revstart=$self->{revstartposition};
    my $nlines=$self->{nlines};
    my $bpl=$self->{bpl};

    if ($fwstart==0) {
	$fwstart++;
	$self->{fwstartposition}=$fwstart;
	$self->{startentry}->set_text($fwstart);
    }
    if ($fwstart+$nlines*$bpl-1==0) {
	$fwstart--;
	$self->{fwstartposition}=$fwstart;
	$self->{startentry}->set_text($fwstart);
	
    }
    
    if ($revstart==0) {
	$revstart++;
	$self->{revstartposition}=$revstart;
    }
    if ($revstart+$nlines*$bpl-1==0) {
	$revstart--;
	$self->{revstartposition}=$revstart;
	
    }
    
    foreach my $mapid (@{$self->{mapids}}) {
	my $anchor=$self->{manager}->get_anchor_position('ID'=>$mapid);
	my $mapstart;
	my $mapend;
	if (! $self->{manager}->is_reverse_complemented($mapid)) {
	    $mapstart=$fwstart+$anchor-1;
	    $mapend=$mapstart+$nlines*$bpl-1;
	} else {
	    $mapstart=$revstart+$anchor-1;
	    $mapend=$mapstart-$nlines*$bpl+1;
	}

	$self->{mapwidget}->set_view_range($mapid,$mapstart,$mapend,$bpl);
    }
    $self->render();
}

sub _set_width {
    my $self=shift;

    return
	if (!defined $self->{frame});
    my $width=shift;
    $self->{width}=$width;
    my $height=$self->{height};
    my $frame=$self->{frame};
    my $oldpaintdevice=$self->{mapwidget}->get_paintdevice();
    my $newpaintdevice=new PaintDevice::GtkPaintDevice($width,$height,$frame);
    $oldpaintdevice->copy_listeners($newpaintdevice);
    my $oldgtkwidget=$self->{frame}->child();
    $oldgtkwidget->destroy();
    $self->{paintdevice}=$newpaintdevice;
    $self->{mapwidget}->set_paintdevice($newpaintdevice);
    $self->render();

}

sub _load_color_scheme {
    my $self=shift;

    my $dialog = new Gtk::FileSelection('Load Color Scheme');
    $dialog->ok_button->signal_connect('clicked',sub {$self->_color_scheme_selected($dialog)});
    $dialog->cancel_button->signal_connect('clicked',sub {$dialog->destroy});
    
    $dialog->show();
}

sub _color_scheme_selected {
    my $self=shift;
    my $dialog=shift;

    my $filename=$dialog->get_filename();
    $dialog->destroy();
    $FeatureWidget::ColorHandler->load_colors($filename);
    $self->render();
    
}
sub _set_color_scheme {
    my $self=shift;
    my $new_scheme=shift;

    $FeatureWidget::ColorHandler=&{$ColorHandlerFactory::Factory{$new_scheme}};
    if ($FeatureWidget::ColorHandler->name() eq 'File based') {
	$self->_load_color_scheme();
    } else {
	$self->render();
    }
    
}

sub _move_btn_clicked {
    my $self=shift;
    my $direction=shift;

    my $fwstart=$self->{fwstartposition};
    my $revstart=$self->{revstartposition};
    my $nlines=$self->{nlines};
    my $bpl=$self->{bpl};
    my $newfwstart=$fwstart+$direction*$bpl*$nlines;
    my $newrevstart=$revstart-$direction*$bpl*$nlines;
    $self->{fwstartposition}=$newfwstart;
    $self->{revstartposition}=$newrevstart;
    $self->update_map_display;
    $newfwstart=$self->{fwstartposition};
    $self->{startentry}->set_text($newfwstart);
}

sub _rebuild_feature_menu {
    my $self=shift;

    my @featurenames=FeatureWidgetFactory::get_feature_names();

    if (defined $self->{featuremenu}) {
	$self->{featureitem}->remove_submenu();
	$self->{featuremenu}->destroy();
    }

    $self->{featuremenu}=new Gtk::Menu;

    foreach my $name (@featurenames) {
	my $item= new Gtk::CheckMenuItem($name);
	if (FeatureWidgetFactory::is_masked($name)) {
	    $item->set_active(0);
	} else {
	    $item->set_active(1);
	}
	$item->signal_connect("toggled"=>sub{$self->_toggle_feature_mask($name);});
	$self->{featuremenu}->append($item);
    }
    
    $self->{featureitem}->set_submenu($self->{featuremenu});
    $self->{menubar}->show_all();
}


sub _toggle_feature_mask {
    my $self=shift;
    my $name=shift;
    
    if (FeatureWidgetFactory::is_masked($name)) {
      FeatureWidgetFactory::unmask_feature($name);
    } else {
      FeatureWidgetFactory::mask_feature($name);
    }
    $self->render();
}

sub _change_strand_state {
    my $self=shift;

    if ($self->{strands_expanded}) {
      ScalableFeatureWidget::collapse_strands();
	$self->{strands_expanded}=0;
    } else {
      ScalableFeatureWidget::expand_strands();
	$self->{strands_expanded}=1;
    }
    $self->render();
}

sub _change_frame_state {
    my $self=shift;

    if ($self->{frames_visible}) {
      ScalableFeatureWidget::hide_frames();
	$self->{frames_visible}=0;
    } else {
      ScalableFeatureWidget::show_frames();
	$self->{frames_visible}=1;
    }
    $self->render();
}

sub _export_map {
    my $self=shift;
    my $format = shift;

    my $dialog = new Gtk::FileSelection("Export Map");
    $dialog->set_filename("map.$format");
    $dialog->ok_button->signal_connect("clicked",sub {$self->_write_map_file($dialog,$format)});
    $dialog->ok_button->signal_connect("clicked",sub {$dialog->destroy});
    $dialog->cancel_button->signal_connect("clicked",sub {$dialog->destroy});
    
    $dialog->show();
}

sub _write_map_file {
    my $self=shift;
    my $dialog = shift;
    my $format = shift;

    my $filename=$dialog->get_filename;
    my $filepaintdevice;
    my $oldpaintdevice=$self->{mapwidget}->get_paintdevice;
    my $screenresolution=$PaintDevice::GtkPaintDevice::SCREEN_RESOLUTION;
    
    if ($format eq 'ps') {
	my $media='a4';
	my $pagewidth=$PaintDevice::PAGE_WIDTHS_CM{$media};
	my $pageheight=$PaintDevice::PAGE_HEIGHTS_CM{$media};
	$filepaintdevice=new PaintDevice::PSPaintDevice($filename,$pagewidth,
							$pageheight,
							type=>'ps');
    }
    if ($format eq 'eps') {
	my $width_cm=$self->{width}/$screenresolution*$PaintDevice::PSPaintDevice::CM_PER_INCH;
	my $height_cm=$width_cm*$self->{height}/$self->{width};
	$filepaintdevice=new PaintDevice::PSPaintDevice($filename,$width_cm,
							$height_cm,
							type=>'eps');
    }


    if ($format eq 'fig') {
	my $width_inch=$self->{width}/$screenresolution;
	my $height_inch=$width_inch*$self->{height}/$self->{width};
	$filepaintdevice=new PaintDevice::XFigPaintDevice($filename,
							  $width_inch,
							  $height_inch);
    }
	

    my $factor=PaintDevice::PNGPaintDevice->text_height/$oldpaintdevice->text_height;
    $filepaintdevice=new PaintDevice::PNGPaintDevice($filename,$oldpaintdevice->width,int($oldpaintdevice->height*$factor))
	if ($format eq 'png');

    $self->{mapwidget}->set_paintdevice($filepaintdevice);
    $self->render();

    $self->{mapwidget}->set_paintdevice($oldpaintdevice);
}

sub _map_entry_changed {
    my $self=shift;

    my $do_update=0;

    my $newstart=$self->{startentry}->get_text();
    if ( !($newstart =~ /-?\d+/) || $newstart==0) {
	
	$self->{startentry}->set_text($self->{fwstartposition});
    } else {
	my $deltastart=$newstart-$self->{fwstartposition};
        $self->{revstartposition}=$self->{revstartposition}-$deltastart;
 	$self->{fwstartposition}=$newstart;
	$do_update=1;
    }
    my $newnlines=$self->{linesentry}->get_text();
    if ($newnlines =~ /\D/ || $newnlines < 1) {
	$self->{linesentry}->set_text($self->{nlines});
    } else {
	$self->{nlines}=$newnlines;
	$do_update=1;
    }
    my $newbpl=$self->{bplentry}->get_text();
    if ($newbpl =~ /\D/ || $newbpl < $MINBPL) {
	$self->{bplentry}->set_text($self->{bpl});
    } else {
	$self->{bpl}=$newbpl;
	$do_update=1;
    }

    my $oldthresh=$ScalableFeatureWidget::display_threshold;
    my $newthresh=$self->{namethreshadj}->value;
    if ($newthresh != $oldthresh) {
	$ScalableFeatureWidget::display_threshold=$newthresh;
	$do_update=1;
    }

    my $oldseqthresh=$self->{mapwidget}->get_sequence_display_threshold();
    my $newseqthresh=$self->{seqthreshadj}->value();
    if ($newseqthresh != $oldseqthresh) {
	$self->{mapwidget}->set_sequence_display_threshold($newseqthresh);
	$do_update=1;
    }

    $self->update_map_display
	if ($do_update);
}

sub _save_preferences {
    my $self=shift;

    PreferencesHandler::save_preferences;

}

1
