package PhysicalMapStrip;

use Bio::Seq;
use Bio::Tools::CodonTable;

use FeatureWidget;
use PaintDevice;

use strict;

sub new {
    my $class = shift;
    my $self = {};
    
    $self->{startbase}=shift;
    $self->{endbase}=shift; 

    $self->{x_start}=shift;
    if (!defined($self->{x_start})) {
	$self->{x_start}=0;
    }

    $self->{y_start}=shift;
    if (!defined($self->{y_start})) {
	$self->{y_start}=0;
    }

    $self->{x_scale}=shift;
    if (!defined($self->{x_scale})) {
	$self->{x_scale}=1;
    }

    $self->{y_scale}=shift;
    if (!defined($self->{y_scale})) {
	$self->{y_scale}=1;
    }

    $self->{featurewidgets}= [];

    return $self;
}

sub add_feature_widget {
    my $self=shift;
    my $widget=shift;
    push @{$self->{featurewidgets}},$widget;
}

sub get_bounds {
    my $self=shift;
    return ($self->{startbase},$self->{endbase});
}

sub draw {
    my $self=shift;
    my $paintdevice=shift;
    for (my $i=0;$i<@{$self->{featurewidgets}};$i++) {
	$self->{featurewidgets}->[$i]->draw($self,$paintdevice);
    }
}

sub height_in_lines {
    return 1;
}

sub offset {
    my $self=shift;
    my $x_start=shift;
    if (defined $x_start) {
	$self->{x_start}=$x_start;
	$self->{y_start}=shift;
    } else {
	return ($self->{x_start},$self->{y_start});
    }
}
sub scale {
    my $self=shift;
    my $x_scale=shift;
    if (defined $x_scale) {
	$self->{x_scale}=$x_scale;
	$self->{y_scale}=shift;
    } else {
	return ($self->{x_scale},$self->{y_scale});
    }
}



package AxisStrip;
@AxisStrip::ISA=qw( PhysicalMapStrip );

sub new {
    my $class = shift;
    my $map=shift;
    my $self = new PhysicalMapStrip(@_);

    $self->{tick_interval}=compute_tick_interval($self->{startbase},
						 $self->{endbase});

    bless $self,$class;

    return $self;
}

sub compute_tick_interval {
    my $start=shift;
    my $end=shift;
    my $range=abs($end-$start)+1;
    my $logrange=int((log($range)/log(10))+0.5);
    
    return 10**($logrange-1);

}

sub draw {
    my $self = shift;
    my $paintdevice = shift;
    my $x_scale=$self->{x_scale};
    my $y_scale=$self->{y_scale};
    my $x_start=$self->{x_start};
    my $y_start=$self->{y_start};

    my $leftbound=$self->{startbase};
    my $rightbound=$self->{endbase};
    if ($self->{startbase}>$self->{endbase}) {
	$leftbound=$self->{endbase};
	$rightbound=$self->{startbase};
    }

    my $leftstring=reverse $self->{startbase};
    $leftstring =~ s/(\d{3})/$1 /g;
    $leftstring=reverse $leftstring;
    $leftstring =~ s/^\s+//;

    my $rightstring=reverse $self->{endbase};
    $rightstring =~ s/(\d{3})/$1 /g;
    $rightstring=reverse $rightstring;
    $rightstring =~ s/^\s+//;

    
    $paintdevice->add_line($x_start,$y_start+0.5*$y_scale,
			   $x_start+1*$x_scale,$y_start+0.5*$y_scale);
    $paintdevice->add_text($x_start,$y_start+0.47*$y_scale,
			   $leftstring,'halign'=>'left','valign'=>'bottom');
    $paintdevice->add_text($x_start+1.0*$x_scale,$y_start+0.47*$y_scale,
			   $rightstring,'halign'=>'right','valign'=>'bottom');
    $paintdevice->add_line($x_start,$y_start+0.48*$y_scale,
			   $x_start,$y_start+0.52*$y_scale);
    $paintdevice->add_line($x_start+1.0*$x_scale,$y_start+0.48*$y_scale,
			   $x_start+1.0*$x_scale,$y_start+0.52*$y_scale);
    
    my $tick_start=int($leftbound/$self->{tick_interval})+1;
    $tick_start*=$self->{tick_interval};
    my $tick_end=int($rightbound/$self->{tick_interval});
    $tick_end*=$self->{tick_interval};

    my $x_position;
    for (my $i=$tick_start;$i<=$tick_end;$i+=$self->{tick_interval}) {
	$x_position=($i-$leftbound)/
	    ($rightbound-$leftbound);
	$paintdevice->add_line($x_start+$x_position*$x_scale,
			       $y_start+0.48*$y_scale,
			       $x_start+$x_position*$x_scale,
			       $y_start+0.52*$y_scale);
    }

    $self->PhysicalMapStrip::draw($paintdevice);
}

sub height_in_lines {
    return 2;
}

package SequenceStrip;
@SequenceStrip::ISA=qw( PhysicalMapStrip );

%SequenceStrip::REVCOMS=('A'=>'T',
			 'T'=> 'A',
			 'G'=>'C',
			 'C'=>'G');
sub new {
    my $class = shift;
    my $map=shift;
    my $self = new PhysicalMapStrip(@_);

    $self->{map}=$map;
    $self->{seqobj}=$map->get_seqobj;

    bless $self,$class;

    return $self;
}

sub draw {
    my $self = shift;
    my $paintdevice = shift;
    my $x_scale=$self->{x_scale};
    my $y_scale=$self->{y_scale};
    my $x_start=$self->{x_start};
    my $y_start=$self->{y_start};


    my $x_position;
    my $y_position_fwd=$y_start+0.10*$y_scale;
    my $y_position_rev=$y_start+0.50*$y_scale;

    my $startbase=$self->{startbase};
    my $endbase=$self->{endbase};
    ($startbase,$endbase)=($endbase,$startbase)
	if ($self->{startbase}>$self->{endbase});

    my $leftbound=$startbase;
    my ($minbase,$maxbase)=$self->{map}->get_bounds();
    $startbase=$minbase
	if ($startbase<$minbase);
    $endbase=$maxbase
	if ($endbase>$maxbase);

    if ($startbase>0 && $endbase>0 && $endbase>$startbase) {
	my $subseq=$self->{seqobj}->subseq($startbase,$endbase);
	$subseq = reverse $subseq
	    if ($self->{startbase}>$self->{endbase});
	for (my $i=$startbase;$i<=$endbase;$i++) {
	    my $letter_fwd=uc substr $subseq,($i-$startbase),1;
	    my $letter_rev=$SequenceStrip::REVCOMS{$letter_fwd};

	    $x_position=($i-$leftbound)/
		abs($self->{endbase}-$self->{startbase});

	    $paintdevice->add_text($x_start+$x_position*$x_scale,
				   $y_position_fwd,
				   $letter_fwd,
				   'halign'=>'middle',
				   'valign'=>'top');

	    $paintdevice->add_text($x_start+$x_position*$x_scale,
				   $y_position_rev,
				   $letter_rev,
				   'halign'=>'middle',
				   'valign'=>'top');
	}
	$self->PhysicalMapStrip::draw($paintdevice);
    }
}

sub height_in_lines {
    return 3;
}

package ScalableStrip;

@ScalableStrip::ISA = qw( PhysicalMapStrip );

sub new {
    my $class = shift;
    my $self = new PhysicalMapStrip(@_);

    bless $self,$class;

    return $self;

}

sub height_in_lines {
    return 4;
}

package FeatureStrip;

use strict;

@FeatureStrip::ISA=qw(ScalableStrip);

sub _find_ctable_id {
    my $self=shift;
    my $ctableid=1;

    foreach my $feat ($self->{seqobj}->all_SeqFeatures()) {
	if ($feat->primary_tag() eq 'CDS' && $feat->has_tag('transl_table')) {
	    ($ctableid, my @dummy)=$feat->each_tag_value('transl_table');
	    last
	}
    }
    return $ctableid;
}

sub new {
    my $class=shift;
    my $map=shift;
    my $self=new ScalableStrip(@_);

    bless $self,$class;

    $self->{map}=$map;
    $self->{seqobj}=$map->get_seqobj();
    $self->{residues_visisble}=0;
    my $ctableid=$self->_find_ctable_id();
    $self->{codontable}=new Bio::Tools::CodonTable(-id=>$ctableid);


    return $self;

}

sub draw {
    my $self=shift;
    my $paintdevice=shift;

    my $x_start=$self->{x_start};
    my $y_start=$self->{y_start};
    my $x_scale=$self->{x_scale};
    my $y_scale=$self->{y_scale};
    my $startbase=$self->{startbase};
    my $endbase=$self->{endbase};
    ($startbase,$endbase)=($endbase,$startbase)
	if ($self->{startbase}>$self->{endbase});
    my $basespan=$endbase-$startbase+1;
    my $leftbound=$startbase;

    PaintDevice::color_alloc('grey80',0.8,0.8,0.8)
	if (!PaintDevice::color_defined('grey80'));
    
    if (ScalableFeatureWidget::are_frames_visible() && 
	$basespan>$PhysicalMapWidget::sequence_display_threshold) {
	foreach my $frame (0..5) {
	    my $y_line=$ScalableFeatureWidget::CDSPositions[$frame];
	    $paintdevice->add_line($x_start,
				   $y_start+$y_line*$y_scale,
				   $x_start+$x_scale,
				   $y_start+$y_line*$y_scale,
				   'color'=>'grey80');
	}
    }

    $self->ScalableStrip::draw($paintdevice);

    if (ScalableFeatureWidget::are_frames_visible() && 
	$basespan <= $PhysicalMapWidget::sequence_display_threshold) {
	my ($minbase,$maxbase)=$self->{map}->get_bounds();
	$startbase=$minbase
	    if ($startbase<$minbase);
	$endbase=$maxbase
	    if ($endbase>$maxbase);
	if ($startbase>0 && $endbase>0 && $endbase>$startbase) {
	    my $subseq=$self->{seqobj}->subseq($startbase,$endbase);
	    for (my $i=$startbase;$i<=$endbase;$i++) {
		if ($i<=$endbase-2) {
		    my $x_position=($i-$leftbound+1)/
			abs($self->{endbase}-$self->{startbase});
		    $x_position=1-$x_position
			if ($self->{endbase}<$self->{startbase});
		    my $frame=($i-1)%3;
		    my $codon=uc substr $subseq,($i-$startbase),3;
		    my $y_pos=ScalableFeatureWidget::get_vertical_position(1,$frame+1);
		    my $residue=$self->{codontable}->translate($codon);
		    my $color='black';
		    $color='red'
			if ($residue eq '*');
		    $paintdevice->add_text($x_start+$x_position*$x_scale,
					   $y_start+$y_pos*$y_scale,
					   $residue,
					   'color'=>$color,
					   'halign'=>'middle',
					   'valign'=>'middle');
		}
		if ($i-2>=0) {
		    my $x_position=($i-$leftbound-1)/
			abs($self->{endbase}-$self->{startbase});
		    $x_position=1-$x_position
			if ($self->{endbase}<$self->{startbase});
		    my $revframe=2-($i-1)%3;
		    my $revcodon=reverse uc substr $subseq,($i-2-$startbase),3;
		    $revcodon=~ tr/ACGT/TGCA/;
		    my $y_pos=ScalableFeatureWidget::get_vertical_position(-1,$revframe+1);
		    my $residue=$self->{codontable}->translate($revcodon);
		    my $color='black';
		    $color='red'
			if ($residue eq '*');
		    $paintdevice->add_text($x_start+$x_position*$x_scale,
					   $y_start+$y_pos*$y_scale,
					   $residue,
					   'color'=>$color,
					   'halign'=>'middle',
					   'valign'=>'middle');
		}

	    }
	}
    }

}

sub height_in_lines {

    return 12.5
	if (ScalableFeatureWidget::are_frames_visible() &&
	    !ScalableFeatureWidget::are_strands_collapsed());

    return 6.5
	if (ScalableFeatureWidget::are_frames_visible() &&
	    ScalableFeatureWidget::are_strands_collapsed());

    return 6.5
	if (!ScalableFeatureWidget::are_frames_visible() &&
	    !ScalableFeatureWidget::are_strands_collapsed());

    return 3.5;
}

1

__END__

=head1


=cut
