package Gui::MapDrawingWindow;

use strict;

use Gtk;

use FeatureWidget;

use Gui::MapListWindow;

use PaintDevice;
use PaintDevice::GtkPaintDevice;
use PaintDevice::PNGPaintDevice;
use PaintDevice::PSPaintDevice;
use PaintDevice::XFigPaintDevice;

use PhysicalMapWidget;

use CompAnalResultWidget;

my $MINBPL=50;

sub new {
     my $class=shift;
     my $manager=shift;
     my ($width,$height)=@_;

     $width=800
	 if (!defined $width || $width<200);
     $height=600
	 if (!defined $height || $height<150);
     my $self={};
     bless $self,$class;

     $self->{manager}=$manager;

     $self->{mapids}=();
     $self->{strands_expanded}=0;
     $self->{fwstartposition}=1;
     $self->{revstartposition}=1;
     $self->{nlines}=4;
     $self->{bpl}=5000;
     $self->{width}=$width;
     $self->{height}=$height;

     my $window=new Gtk::Window -toplevel;
     $window->set_title('Multi-Genome Navigator (release 20030103)');
     $self->{window}=$window;
     $window->set_usize(820,600);
     $window->signal_connect('delete_event'=>sub { $manager->quit(); });
     $window->signal_connect('destroy_event'=>sub { $manager->quit(); });
     

     my $vbox=new Gtk::VBox(0,1);
     $vbox->show_all();
     $window->add($vbox);
     
     my $menubar=new Gtk::MenuBar;
     $vbox->pack_start($menubar,0,0,1);
     $self->{menubar}=$menubar;

     my $menuitem=new Gtk::MenuItem('File');
     $menubar->append($menuitem);
     my $submenu=new Gtk::Menu;
     $menuitem->set_submenu($submenu);
     $menuitem= new Gtk::MenuItem("Export Map");
     $submenu->append($menuitem);
     my $subsubmenu=new Gtk::Menu;
     $menuitem->set_submenu($subsubmenu);
     $menuitem = new Gtk::MenuItem("XFig");
     $menuitem->signal_connect("activate"=>sub{$self->_export_map('fig')});
     $subsubmenu->append($menuitem);
     $menuitem = new Gtk::MenuItem("PostScript");
     $subsubmenu->append($menuitem);
     my $pssubmenu=new Gtk::Menu;
     $menuitem->set_submenu($pssubmenu);
     foreach my $media (sort keys %PaintDevice::PAGE_WIDTHS_CM) {
	 if ($media ne 'custom') {
	     $menuitem=new Gtk::MenuItem($media);	 
	     $menuitem->signal_connect("activate"=>sub{$self->_export_map('ps',$media)});
	     $pssubmenu->append($menuitem);
	 }
     }
     $menuitem = new Gtk::MenuItem("Encapsulated PostScript");
     $menuitem->signal_connect("activate"=>sub{$self->_export_map('eps')});
     $subsubmenu->append($menuitem);     
     $menuitem = new Gtk::MenuItem("PNG");
     $menuitem->signal_connect("activate"=>sub{$self->_export_map('png')});
     $subsubmenu->append($menuitem);     
     $menuitem=new Gtk::MenuItem('Quit');
     $menuitem->signal_connect('activate'=>sub{$self->{manager}->quit();});
     $submenu->append($menuitem);

     $menuitem=new Gtk::MenuItem('Preferences');
     $menubar->append($menuitem);
     $submenu=new Gtk::Menu;
     $menuitem->set_submenu($submenu);
     $menuitem=new Gtk::CheckMenuItem('Expand Strands');
     if (!ScalableFeatureWidget::are_strands_collapsed) {
	 $menuitem->set_active(1);
	 $self->{strands_expanded}=1;
     }
     $menuitem->signal_connect('toggled'=>sub{$self->_change_strand_state;});
     $submenu->append($menuitem);
     $menuitem=new Gtk::CheckMenuItem('Show Frames');
     if (ScalableFeatureWidget::are_frames_visible) {
	 $menuitem->set_active(1);
	 $self->{frames_visible}=1;
     }
     $menuitem->signal_connect('toggled'=>sub{$self->_change_frame_state;});
     $submenu->append($menuitem);
     $menuitem=new Gtk::CheckMenuItem('Show Result Highlights');
     if ($CompAnalResultWidget::show_highlights) {
	 $menuitem->set_active(1);
	 $self->{highlights_visible}=1;
     }
     $menuitem->signal_connect('toggled'=>sub{$self->_change_highlight_state;});     $submenu->append($menuitem);
     $menuitem=new Gtk::MenuItem('Visible Features');
     $submenu->append($menuitem);
     $self->{featureitem}=$menuitem;

     $menuitem=new Gtk::MenuItem('Map Area Width');
     $submenu->append($menuitem);
     $subsubmenu=new Gtk::Menu();
     $menuitem->set_submenu($subsubmenu);
     my $previtem=undef;
     my %sizes=(800=>1,
		1000=>1,
		1200=>1,
		1600=>1,
		2000=>1,
		2400=>1,
		3000=>1,
		3600=>1,
		4000=>1,
		4800=>1,
		5600=>1,
		6400=>1,
		7200=>1,
		8000=>1,
		9000=>1,
		10000=>1);
     $sizes{$width}=1;
     foreach my $size (sort { $a <=> $b } keys %sizes) {
	 if (defined $previtem) {
	     $menuitem=new Gtk::RadioMenuItem("$size",$previtem);
	 } else {
	     $menuitem=new Gtk::RadioMenuItem("$size");

	 }
	 $menuitem->set_active(1)
	     if ($width == $size);
	 $menuitem->signal_connect('activate'=>sub{$self->_set_width($size);});
	 $subsubmenu->append($menuitem);
	 $previtem=$menuitem;
     }
     


     $menuitem=new Gtk::MenuItem('Save Preferences');
     $menuitem->signal_connect('activate'=>sub{$self->_save_preferences;});
     $submenu->append($menuitem);

     my $frame=new Gtk::Frame;
     $self->{frame}=$frame;
     $vbox->pack_start($frame,1,1,1);

     my $paintdevice=new PaintDevice::GtkPaintDevice($width,$height,$frame);
     $self->{paintdevice}=$paintdevice;

     my $mapwidget=new PhysicalMapWidget($paintdevice);
     $self->{mapwidget}=$mapwidget;

     my $hbox=new Gtk::HBox(0,1);
     $hbox->show();
     $vbox->pack_start($hbox,0,0,1);

     my $arrowbox=new Gtk::HBox(0,0);
     my $alignment=new Gtk::Alignment(1.0,0.5,0.0,1.0);
     $arrowbox->add($alignment);
     my $arrow=new Gtk::Arrow('left','out');
     $alignment->add($arrow);
     $alignment=new Gtk::Alignment(0.0,0.5,0.0,1.0);
     $arrowbox->add($alignment);
     $arrow=new Gtk::Arrow('left','out');
     $alignment->add($arrow);

     my $prevbtn=new Gtk::Button;
     $prevbtn->add($arrowbox);
     $prevbtn->show();
     $prevbtn->signal_connect('clicked',sub {$self->_move_btn_clicked(-1);});
     $hbox->pack_start($prevbtn,1,1,0);

     $arrow=new Gtk::Arrow('left','out');
     $prevbtn=new Gtk::Button;
     $prevbtn->add($arrow);
     $prevbtn->show();
     $prevbtn->signal_connect('clicked',sub {$self->_move_btn_clicked(-0.5);});
     $hbox->pack_start($prevbtn,1,1,2);

     $arrow=new Gtk::Arrow('right','out');
     my  $nextbtn=new Gtk::Button;
     $nextbtn->add($arrow);
     $nextbtn->show();
     $nextbtn->signal_connect('clicked',sub {$self->_move_btn_clicked(0.5);});
     $hbox->pack_start($nextbtn,1,1,1);

     $arrowbox=new Gtk::HBox(0,0);
     $alignment=new Gtk::Alignment(1.0,0.5,0.0,1.0);
     $arrowbox->add($alignment);
     $arrow=new Gtk::Arrow('right','out');
     $alignment->add($arrow);
     $alignment=new Gtk::Alignment(0.0,0.5,0.0,1.0);
     $arrowbox->add($alignment);
     $arrow=new Gtk::Arrow('right','out');
     $alignment->add($arrow);

     $nextbtn=new Gtk::Button;
     $nextbtn->add($arrowbox);
     $nextbtn->show();
     $nextbtn->signal_connect('clicked',sub {$self->_move_btn_clicked(1);});
     $hbox->pack_start($nextbtn,1,1,1);

     $hbox=new Gtk::HBox(0,1);
     $hbox->show();
     $vbox->pack_start($hbox,0,0,1);

     my $zoomoutoutbtn=new Gtk::Button '--';
     $zoomoutoutbtn->show();
     $zoomoutoutbtn->signal_connect('clicked',
				    sub {$self->_zoom_btn_clicked(4);});
     $hbox->pack_start($zoomoutoutbtn,1,1,0);

     my $zoomoutbtn=new Gtk::Button '-';
     $zoomoutbtn->show();
     $zoomoutbtn->signal_connect('clicked',
				 sub {$self->_zoom_btn_clicked(2);});
     $hbox->pack_start($zoomoutbtn,1,1,0);

     my $zoominbtn=new Gtk::Button '+';
     $zoominbtn->show();
     $zoominbtn->signal_connect('clicked',
				sub {$self->_zoom_btn_clicked(0.5);});
     $hbox->pack_start($zoominbtn,1,1,0);

     my $zoomininbtn=new Gtk::Button '++';
     $zoomininbtn->show();
     $zoomininbtn->signal_connect('clicked',
				  sub {$self->_zoom_btn_clicked(0.25);});
     $hbox->pack_start($zoomininbtn,1,1,0);

     $hbox=new Gtk::HBox(0,4);
     $hbox->show();
     $vbox->pack_start($hbox,0,0,1);

     my $label=new Gtk::Label('Start base');
     $hbox->pack_start($label,0,1,1);
     my $entry=new Gtk::Entry(9);
     $hbox->pack_start($entry,1,1,1);
     $entry->signal_connect('activate',sub {$self->_map_entry_changed();});
     $entry->set_text($self->{fwstartposition});
     $self->{startentry}=$entry;
     $label=new Gtk::Label('Lines');
     $hbox->pack_start($label,0,1,1);
     $entry=new Gtk::Entry(9);
     $entry->signal_connect('activate',sub {$self->_map_entry_changed();});
     $hbox->pack_start($entry,1,1,1);
     $entry->set_text($self->{nlines});
     $self->{linesentry}=$entry;
     $label=new Gtk::Label('BPL');
     $hbox->pack_start($label,0,1,1);
     $entry=new Gtk::Entry(9);
     $entry->signal_connect('activate',sub {$self->_map_entry_changed();});
     $hbox->pack_start($entry,1,1,1);
     $entry->set_text($self->{bpl});
     $self->{bplentry}=$entry;


     $hbox=new Gtk::HBox(0,4);
     $hbox->show();
     $vbox->pack_start($hbox,0,0,1);

     $label=new Gtk::Label('Birds eye view thresh. (kbp) :');
     $hbox->pack_start($label,0,1,1);
     my $viewthreshadj=new Gtk::Adjustment($mapwidget->get_birdseye_display_threshold/1000,10,1010,10,100,100);
     $viewthreshadj->signal_connect('value_changed',sub {$self->_map_entry_changed();});
     $self->{viewthreshadj}=$viewthreshadj;
     my $scale=new Gtk::HScale($viewthreshadj);
     $scale->set_update_policy('delayed');
     $scale->set_draw_value(1);
     $scale->set_digits(0);
     $hbox->pack_start($scale,1,1,1);
     
     $label=new Gtk::Label('Seq. display thresh. (bp) :');
     $hbox->pack_start($label,0,1,1);
     my $seqthreshadj=new Gtk::Adjustment($mapwidget->get_sequence_display_threshold,1,1011,1,10,10);
     $seqthreshadj->signal_connect('value_changed',sub {$self->_map_entry_changed();});
     $self->{seqthreshadj}=$seqthreshadj;
     $scale=new Gtk::HScale($seqthreshadj);
     $scale->set_update_policy('delayed');
     $scale->set_draw_value(1);
     $scale->set_digits(0);
     $hbox->pack_start($scale,1,1,1);
     
     $label=new Gtk::Label('Name display thresh.');
     $hbox->pack_start($label,0,1,1);
     my $namethreshadj=new Gtk::Adjustment($ScalableFeatureWidget::display_threshold,0.0,1.1,0.01,0.01,0.1);
     $namethreshadj->signal_connect('value_changed',sub {$self->_map_entry_changed();});
     $self->{namethreshadj}=$namethreshadj;
     $scale=new Gtk::HScale($namethreshadj);
     $scale->set_update_policy('delayed');
     $scale->set_draw_value(1);
     $scale->set_digits(3);
     $hbox->pack_start($scale,1,1,1);


     return $self;
}

sub append_map {
    my $self=shift;
    my $map=shift;

    my $start=$self->{fwstartposition};
    my $nlines=$self->{nlines};
    my $bpl=$self->{bpl};
    my $end=$start+$nlines*$bpl-1;

    my $mapid=$self->{mapwidget}->append_map($map,$start,$end,$bpl);

    push @{$self->{mapids}},$mapid;

    $self->{mapwidget}->render();
    $self->_rebuild_feature_menu;

    return $mapid;
}

sub shift_map_up {
    my $self=shift;
    my $mapid=shift;

    $self->{mapwidget}->shift_map_up($mapid);
    $self->{mapwidget}->render();
}

sub shift_map_down {
    my $self=shift;
    my $mapid=shift;

    $self->{mapwidget}->shift_map_down($mapid);
    $self->{mapwidget}->render();
}


sub hide_map {
    my $self=shift;
    my $mapid=shift;

    $self->{mapwidget}->hide_map($mapid);
    $self->{mapwidget}->render();
}

sub show_map {
    my $self=shift;
    my $mapid=shift;

    $self->{mapwidget}->show_map($mapid);
    $self->{mapwidget}->render();
}



sub replace_map {
    my $self=shift;
    my $repmapid=shift;
    my $newmap=shift;

    $self->{mapwidget}->replace_map($newmap,$repmapid);
    $self->{mapwidget}->render();
}

sub remove_map {
    my $self=shift;
    my $delmapid=shift;

    my $newmapids=[];
    foreach my $mapid (@{$self->{mapids}}) {
	push @{$newmapids},$mapid
	    if ($mapid != $delmapid);
    }

    $self->{mapids}=$newmapids;
    $self->{mapwidget}->remove_map($delmapid);
    $self->{mapwidget}->render();
}

sub render {
    my $self=shift;
    $self->{mapwidget}->render();
}


sub add_listener {
    my $self=shift;
    my $listener=shift;
    my $type=shift;

    $self->{paintdevice}->add_listener($listener,$type);
}

sub show_all {
    my $self=shift;
    $self->{window}->show_all();
}

sub get_start_position {
    my $self=shift;
    return $self->{fwstartposition};
}

sub set_start_position {
    my $self=shift;
    my $start=shift;

    my $deltastart=$start-$self->{fwstartposition};
    $self->{fwstartposition}=$start;
    $self->{revstartposition}=$self->{revstartposition}-$deltastart;
    $self->{startentry}->set_text($start);

    $self->update_map_display;
}

sub get_bpl {
    my $self=shift;
    return $self->{bpl};
}

sub set_bpl {
    my $self=shift;
    my $bpl=shift;
    
    $self->{bpl}=$bpl;
    $self->{bplentry}->set_text($bpl);
    $self->update_map_display;
}


sub get_display_parameters {
    my $self=shift;
    
    my $endposition=$self->{fwstartposition}+$self->{nlines}*$self->{bpl}-1;
    return ($self->{fwstartposition},$endposition,$self->{bpl});
}

sub set_display_parameters {
    my $self=shift;
    my ($start,$end,$bpl)=@_;
    
    my $deltastart=$start-$self->{fwstartposition};
    $self->{fwstartposition}=$start;
    $self->{bpl}=$bpl;
    $self->{nlines}=($end-$start+1)/$bpl;
    $self->{nlines}=1
	if ($self->{nlines}<1);
    $self->{revstartposition}=$self->{revstartposition}-$deltastart;

    $self->{startentry}->set_text($start);
    $self->{linesentry}->set_text($self->{nlines});
    $self->{bplentry}->set_text($bpl);
    $self->update_map_display();
}


sub update_map_display {
    my $self=shift;

    my $fwstart=$self->{fwstartposition};
    my $revstart=$self->{revstartposition};
    my $nlines=$self->{nlines};
    my $bpl=$self->{bpl};

    if ($fwstart==0) {
	$fwstart++;
	$self->{fwstartposition}=$fwstart;
	$self->{startentry}->set_text($fwstart);
    }
    if ($fwstart+$nlines*$bpl-1==0) {
	$fwstart--;
	$self->{fwstartposition}=$fwstart;
	$self->{startentry}->set_text($fwstart);
	
    }
    
    if ($revstart==0) {
	$revstart++;
	$self->{revstartposition}=$revstart;
    }
    if ($revstart+$nlines*$bpl-1==0) {
	$revstart--;
	$self->{revstartposition}=$revstart;
	
    }
    
    foreach my $mapid (@{$self->{mapids}}) {
	my $anchor=$self->{manager}->get_anchor_position('ID'=>$mapid);
	my $mapstart;
	my $mapend;
	if (! $self->{manager}->is_reverse_complemented($mapid)) {
	    $mapstart=$fwstart+$anchor-1;
	    $mapend=$mapstart+$nlines*$bpl-1;
	} else {
	    $mapstart=$revstart+$anchor-1;
	    $mapend=$mapstart-$nlines*$bpl+1;
	}

	$self->{mapwidget}->set_view_range($mapid,$mapstart,$mapend,$bpl);
    }
    $self->{mapwidget}->render();
    $self->_rebuild_feature_menu;
}

sub _set_width {
    my $self=shift;

    return 1
	if (!defined $self->{frame});
    my $width=shift;
    $self->{width}=$width;
    my $height=$self->{height};
    my $paintdevice=$self->{mapwidget}->get_paintdevice();
    $paintdevice->resize($width,$height);
    $self->{mapwidget}->render();

    return 1;
}

sub _move_btn_clicked {
    my $self=shift;
    my $direction=shift;

    my $fwstart=$self->{fwstartposition};
    my $revstart=$self->{revstartposition};
    my $nlines=$self->{nlines};
    my $bpl=$self->{bpl};
    my $newfwstart=$fwstart+$direction*$bpl*$nlines;
    my $newrevstart=$revstart-$direction*$bpl*$nlines;
    $self->{fwstartposition}=$newfwstart;
    $self->{revstartposition}=$newrevstart;
    $self->update_map_display;
    $newfwstart=$self->{fwstartposition};
    $self->{startentry}->set_text($newfwstart);

    return 1;
}

sub _zoom_btn_clicked {
    my $self=shift;
    my $factor=shift;

    my $fwstart=$self->{fwstartposition};
    my $revstart=$self->{revstartposition};
    my $nlines=$self->{nlines};
    my $bpl=$self->{bpl};
    my $range=$bpl*$nlines;    
    $range=$range*$factor;
    my $newbpl=int($range/$nlines);
    $self->{bpl}=$newbpl;
    $self->{bplentry}->set_text($newbpl);
    $self->update_map_display;

    return 1;
}

sub _rebuild_feature_menu {
    my $self=shift;

    my @featurenames=FeatureWidgetFactory::get_feature_names();

    if (defined $self->{featuremenu}) {
	$self->{featureitem}->remove_submenu();
	$self->{featuremenu}->destroy();
    }

    $self->{featuremenu}=new Gtk::Menu;

    foreach my $name (@featurenames) {
	my $item= new Gtk::CheckMenuItem($name);
	if (FeatureWidgetFactory::is_masked($name)) {
	    $item->set_active(0);
	} else {
	    $item->set_active(1);
	}
	$item->signal_connect("toggled"=>sub{$self->_toggle_feature_mask($name);});
	$self->{featuremenu}->append($item);
    }
    
    $self->{featureitem}->set_submenu($self->{featuremenu});
    $self->{menubar}->show_all();
}


sub _toggle_feature_mask {
    my $self=shift;
    my $name=shift;
    
    if (FeatureWidgetFactory::is_masked($name)) {
      FeatureWidgetFactory::unmask_feature($name);
    } else {
      FeatureWidgetFactory::mask_feature($name);
    }
    $self->{mapwidget}->render;

    return 1;
}

sub _change_strand_state {
    my $self=shift;

    if ($self->{strands_expanded}) {
      ScalableFeatureWidget::collapse_strands();
	$self->{strands_expanded}=0;
    } else {
      ScalableFeatureWidget::expand_strands();
	$self->{strands_expanded}=1;
    }
    $self->{mapwidget}->render;

    return 1;
}

sub _change_frame_state {
    my $self=shift;

    if ($self->{frames_visible}) {
      ScalableFeatureWidget::hide_frames();
	$self->{frames_visible}=0;
    } else {
      ScalableFeatureWidget::show_frames();
	$self->{frames_visible}=1;
    }
    $self->{mapwidget}->render;

    return 1;
}


sub _change_highlight_state {
    my $self=shift;

    if ($self->{highlights_visible}) {
	$CompAnalResultWidget::show_highlights=0;
	$self->{highlights_visible}=0;
    } else {
	$CompAnalResultWidget::show_highlights=1;
	$self->{highlights_visible}=1;
    }
    $self->{mapwidget}->render;

    return 1;
}

sub _export_map {
    my $self=shift;
    my $format = shift;
    my $media=shift;

    my $dialog = new Gtk::FileSelection("Export Map");
    $dialog->set_filename("map.$format");
    $dialog->ok_button->signal_connect("clicked",sub {$self->_write_map_file($dialog,$format,$media)});
    $dialog->ok_button->signal_connect("clicked",sub {$dialog->destroy(); 
						      return 1});
    $dialog->cancel_button->signal_connect("clicked",sub {$dialog->destroy();
						      return 1});
    
    $dialog->show();

    return 1;
}

sub _write_map_file {
    my $self=shift;
    my $dialog = shift;
    my $format = shift;

    my $filename=$dialog->get_filename;
    my $filepaintdevice;
    my $oldpaintdevice=$self->{mapwidget}->get_paintdevice;
    my $screenresolution=$PaintDevice::GtkPaintDevice::SCREEN_RESOLUTION;
    
    if ($format eq 'ps') {
	my $media=shift;
	my $pagewidth=$PaintDevice::PAGE_WIDTHS_CM{$media};
	my $pageheight=$PaintDevice::PAGE_HEIGHTS_CM{$media};
	$filepaintdevice=new PaintDevice::PSPaintDevice($filename,$pagewidth,
							$pageheight,
							type=>'ps',
							media=>$media);
    }
    if ($format eq 'eps') {
	my $width_cm=$self->{width}/$screenresolution*$PaintDevice::PSPaintDevice::CM_PER_INCH;
	my $height_cm=$width_cm*$self->{height}/$self->{width};
	$filepaintdevice=new PaintDevice::PSPaintDevice($filename,$width_cm,
							$height_cm,
							type=>'eps');
    }


    if ($format eq 'fig') {
	my $width_inch=$self->{width}/$screenresolution;
	my $height_inch=$width_inch*$self->{height}/$self->{width};
	$filepaintdevice=new PaintDevice::XFigPaintDevice($filename,
							  $width_inch,
							  $height_inch);
    }
	

    my $factor=PaintDevice::PNGPaintDevice->text_height/$oldpaintdevice->text_height;
    $filepaintdevice=new PaintDevice::PNGPaintDevice($filename,$oldpaintdevice->width,int($oldpaintdevice->height*$factor))
	if ($format eq 'png');

    $self->{mapwidget}->set_paintdevice($filepaintdevice);
    $self->{mapwidget}->render;

    $self->{mapwidget}->set_paintdevice($oldpaintdevice);

    return 1;
}

sub _map_entry_changed {
    my $self=shift;

    my $do_update=0;

    my $newstart=$self->{startentry}->get_text();
    if ( !($newstart =~ /-?\d+/) || $newstart==0) {
	
	$self->{startentry}->set_text($self->{fwstartposition});
    } else {
	my $deltastart=$newstart-$self->{fwstartposition};
        $self->{revstartposition}=$self->{revstartposition}-$deltastart;
 	$self->{fwstartposition}=$newstart;
	$do_update=1;
    }
    my $newnlines=$self->{linesentry}->get_text();
    if ($newnlines =~ /\D/ || $newnlines < 1) {
	$self->{linesentry}->set_text($self->{nlines});
    } else {
	$self->{nlines}=$newnlines;
	$do_update=1;
    }
    my $newbpl=$self->{bplentry}->get_text();
    if ($newbpl =~ /\D/ || $newbpl < $MINBPL) {
	$self->{bplentry}->set_text($self->{bpl});
    } else {
	$self->{bpl}=$newbpl;
	$do_update=1;
    }

    my $oldthresh=$ScalableFeatureWidget::display_threshold;
    my $newthresh=$self->{namethreshadj}->value;
    if ($newthresh != $oldthresh) {
	$ScalableFeatureWidget::display_threshold=$newthresh;
	$do_update=1;
    }

    my $oldseqthresh=$self->{mapwidget}->get_sequence_display_threshold();
    my $newseqthresh=$self->{seqthreshadj}->value();
    if ($newseqthresh != $oldseqthresh) {
	$self->{mapwidget}->set_sequence_display_threshold($newseqthresh);
	$do_update=1;
    }

    my $oldviewthresh=$self->{mapwidget}->get_birdseye_display_threshold();
    my $newviewthresh=$self->{viewthreshadj}->value();
    if ($newviewthresh != $oldviewthresh) {
	$self->{mapwidget}->set_birdseye_display_threshold($newviewthresh*1000);
	$do_update=1;
    }


    $self->update_map_display
	if ($do_update);

    return 1;
}

sub _save_preferences {
    my $self=shift;

    PreferencesHandler::save_preferences;

    return 1;

}

sub destroy {
    my $self=shift;
    
    $self->{window}->destroy();
}


1
