package PhysicalMapStrip;

use Bio::Seq;
use Bio::Tools::CodonTable;

use FeatureWidget;
use PaintDevice;

use strict;

sub new {
    my $class = shift;
    my $self = {};
    
    $self->{startbase}=shift;
    $self->{endbase}=shift; 

    $self->{x_start}=shift;
    if (!defined($self->{x_start})) {
	$self->{x_start}=0;
    }

    $self->{y_start}=shift;
    if (!defined($self->{y_start})) {
	$self->{y_start}=0;
    }

    $self->{x_scale}=shift;
    if (!defined($self->{x_scale})) {
	$self->{x_scale}=1;
    }

    $self->{y_scale}=shift;
    if (!defined($self->{y_scale})) {
	$self->{y_scale}=1;
    }

    $self->{featurewidgets}= [];

    return $self;
}

sub add_feature_widget {
    my $self=shift;
    my $widget=shift;
    push @{$self->{featurewidgets}},$widget;
}

sub get_bounds {
    my $self=shift;
    return ($self->{startbase},$self->{endbase});
}

sub draw {
    my $self=shift;
    my $paintdevice=shift;
    for (my $i=0;$i<@{$self->{featurewidgets}};$i++) {
	$self->{featurewidgets}->[$i]->draw($self,$paintdevice);
    }
}

sub height_in_lines {
    return 1;
}

sub offset {
    my $self=shift;
    my $x_start=shift;
    if (defined $x_start) {
	$self->{x_start}=$x_start;
	$self->{y_start}=shift;
    } else {
	return ($self->{x_start},$self->{y_start});
    }
}
sub scale {
    my $self=shift;
    my $x_scale=shift;
    if (defined $x_scale) {
	$self->{x_scale}=$x_scale;
	$self->{y_scale}=shift;
    } else {
	return ($self->{x_scale},$self->{y_scale});
    }
}



package AxisStrip;
@AxisStrip::ISA=qw( PhysicalMapStrip );

$AxisStrip::AXIS_POS=0.67;
$AxisStrip::TICK_LEN=0.05;
$AxisStrip::Y_SEQ_FWD=0.65;
$AxisStrip::Y_SEQ_REV=0.95;

%AxisStrip::REVCOMS=('A'=>'T',
		     'T'=> 'A',
		     'G'=>'C',
		     'C'=>'G');
sub new {
    my $class = shift;
    my $map=shift;
    my $self = new PhysicalMapStrip(@_);

    $self->{map}=$map;
    $self->{seqobj}=$map->get_seqobj;
    $self->{tick_interval}=compute_tick_interval($self->{startbase},
						 $self->{endbase});

    bless $self,$class;

    return $self;
}

sub compute_tick_interval {
    my $start=shift;
    my $end=shift;
    my $range=abs($end-$start)+1;
    my $logrange=int((log($range)/log(10))+0.5);
    
    return 10**($logrange-1);

}

sub draw {
    my $self = shift;
    my $paintdevice = shift;
    my $x_scale=$self->{x_scale};
    my $y_scale=$self->{y_scale};
    my $x_start=$self->{x_start};
    my $y_start=$self->{y_start};

    my $leftbound=$self->{startbase};
    my $rightbound=$self->{endbase};
    if ($self->{startbase}>$self->{endbase}) {
	$leftbound=$self->{endbase};
	$rightbound=$self->{startbase};
    }

    my $leftstring=reverse $self->{startbase};
    $leftstring =~ s/(\d{3})/$1 /g;
    $leftstring=reverse $leftstring;
    $leftstring =~ s/^\s+//;

    my $rightstring=reverse $self->{endbase};
    $rightstring =~ s/(\d{3})/$1 /g;
    $rightstring=reverse $rightstring;
    $rightstring =~ s/^\s+//;

    
    if ($PhysicalMapStripSet::view_mode ==
	$PhysicalMapStripSet::FEATURE_VIEW ||
	$PhysicalMapStripSet::view_mode ==
	$PhysicalMapStripSet::BIRDSEYE_VIEW) {

	$AxisStrip::AXIS_POS=0.67;
	$AxisStrip::TICK_LEN=0.05;

    }

    if ($PhysicalMapStripSet::view_mode ==
	$PhysicalMapStripSet::SEQUENCE_VIEW) {
	$AxisStrip::AXIS_POS=0.3;
	$AxisStrip::TICK_LEN=0.025;
    }

    $paintdevice->add_line($x_start,
			   $y_start+$AxisStrip::AXIS_POS*$y_scale,
			   $x_start+$x_scale,
			   $y_start+$AxisStrip::AXIS_POS*$y_scale);
    $paintdevice->add_text($x_start,$y_start+0.01*$y_scale,
			   $leftstring,'halign'=>'left','valign'=>'top');
    $paintdevice->add_text($x_start+$x_scale,$y_start+0.01*$y_scale,
			   $rightstring,'halign'=>'right','valign'=>'top');
    $paintdevice->add_line($x_start,
			   $y_start+
			   ($AxisStrip::AXIS_POS-$AxisStrip::TICK_LEN)*
			   $y_scale,
			   $x_start,
			   $y_start+
			   ($AxisStrip::AXIS_POS+$AxisStrip::TICK_LEN)*
			   $y_scale);
    $paintdevice->add_line($x_start+$x_scale,
			   $y_start+
			   ($AxisStrip::AXIS_POS-$AxisStrip::TICK_LEN)*
			   $y_scale,
			   $x_start+$x_scale,
			   $y_start+
			   ($AxisStrip::AXIS_POS+$AxisStrip::TICK_LEN)*
			   $y_scale);
    
    my $tick_start=int($leftbound/$self->{tick_interval})+1;
    $tick_start*=$self->{tick_interval};
    my $tick_end=int($rightbound/$self->{tick_interval});
    $tick_end*=$self->{tick_interval};

    my $x_position;
    for (my $i=$tick_start;$i<=$tick_end;$i+=$self->{tick_interval}) {
	$x_position=($i-$leftbound)/
	    ($rightbound-$leftbound);
	$paintdevice->add_line($x_start+$x_position*$x_scale,
			       $y_start+
			       ($AxisStrip::AXIS_POS-$AxisStrip::TICK_LEN)*
			       $y_scale,
			       $x_start+$x_position*$x_scale,
			       $y_start+
			       ($AxisStrip::AXIS_POS+$AxisStrip::TICK_LEN)*
			       $y_scale);

    }


    if ($PhysicalMapStripSet::view_mode ==
	$PhysicalMapStripSet::SEQUENCE_VIEW) {
	my $x_position;
	my $y_position_fwd=$y_start+$AxisStrip::Y_SEQ_FWD*$y_scale;
	my $y_position_rev=$y_start+$AxisStrip::Y_SEQ_REV*$y_scale;

	my $startbase=$self->{startbase};
	my $endbase=$self->{endbase};
	($startbase,$endbase)=($endbase,$startbase)
	    if ($self->{startbase}>$self->{endbase});
	
	my $leftbound=$startbase;
	my ($minbase,$maxbase)=$self->{map}->get_bounds();
	$startbase=$minbase
	    if ($startbase<$minbase);
	$endbase=$maxbase
	    if ($endbase>$maxbase);
	
	if ($startbase>0 && $endbase>0 && $endbase>$startbase) {
	    my $subseq=$self->{seqobj}->subseq($startbase,$endbase);
	    for (my $i=$startbase;$i<=$endbase;$i++) {
		my $letter_fwd=uc substr $subseq,($i-$startbase),1;
		my $letter_rev=$AxisStrip::REVCOMS{$letter_fwd};
		
		$x_position=($i-$leftbound)/
		    abs($self->{endbase}-$self->{startbase});
		$x_position=1-$x_position
		    if ($self->{startbase}>$self->{endbase});
		$paintdevice->add_text($x_start+$x_position*$x_scale,
				       $y_position_fwd,
				       $letter_fwd,
				       'halign'=>'middle',
				       'valign'=>'bottom',
				       'depth' => $FeatureWidget::TEXT_DEPTH);
		
		$paintdevice->add_text($x_start+$x_position*$x_scale,
				       $y_position_rev,
				       $letter_rev,
				       'halign'=>'middle',
				       'valign'=>'bottom',
				       'depth' => $FeatureWidget::TEXT_DEPTH);
	    }
	}
    }
    $self->PhysicalMapStrip::draw($paintdevice);
}

sub height_in_lines {
    my $self=shift;
    my $lines=-1;

    if ($PhysicalMapStripSet::view_mode ==
	$PhysicalMapStripSet::FEATURE_VIEW ||
	$PhysicalMapStripSet::view_mode ==
	$PhysicalMapStripSet::BIRDSEYE_VIEW) {

	$lines=3;
    }

    if ($PhysicalMapStripSet::view_mode ==
	$PhysicalMapStripSet::SEQUENCE_VIEW) {
	$lines=6;
    }

    return $lines;

}

package ScalableStrip;

@ScalableStrip::ISA = qw( PhysicalMapStrip );

sub new {
    my $class = shift;
    my $self = new PhysicalMapStrip(@_);

    bless $self,$class;

    return $self;

}

sub height_in_lines {
    my $lines=4;

    $lines=$lines/2.0
    if ($PhysicalMapStripSet::view_mode ==
	$PhysicalMapStripSet::BIRDSEYE_VIEW);

    return $lines;
}


package ResultStrip;

use strict;

@ResultStrip::ISA=qw(ScalableStrip);

sub new {
    my $class=shift;
    my $widget=shift;
    my $self=new ScalableStrip(@_);
    
    $self->{widget}=$widget;
    bless $self,$class;

    return $self;
}

sub height_in_lines {
    my $self=shift;

    return $self->{widget}->height_in_lines();
}

package FeatureStrip;

use strict;

@FeatureStrip::ISA=qw(ScalableStrip);

sub _find_ctable_id {
    my $self=shift;
    my $ctableid=1;

    foreach my $feat ($self->{seqobj}->all_SeqFeatures()) {
	if ($feat->primary_tag() eq 'CDS' && $feat->has_tag('transl_table')) {
	    ($ctableid, my @dummy)=$feat->each_tag_value('transl_table');
	    last
	}
    }
    return $ctableid;
}

sub new {
    my $class=shift;
    my $map=shift;
    my $self=new ScalableStrip(@_);

    bless $self,$class;

    $self->{map}=$map;
    $self->{seqobj}=$map->get_seqobj();
    $self->{residues_visisble}=0;
    my $ctableid=$self->_find_ctable_id();
    $self->{codontable}=new Bio::Tools::CodonTable(-id=>$ctableid);


    return $self;

}

sub draw {
    my $self=shift;
    my $paintdevice=shift;

    my $x_start=$self->{x_start};
    my $y_start=$self->{y_start};
    my $x_scale=$self->{x_scale};
    my $y_scale=$self->{y_scale};
    my $startbase=$self->{startbase};
    my $endbase=$self->{endbase};
    ($startbase,$endbase)=($endbase,$startbase)
	if ($self->{startbase}>$self->{endbase});
    my $basespan=$endbase-$startbase+1;
    my $leftbound=$startbase;

    PaintDevice::color_alloc('lightgrey',0.8,0.8,0.8)
	if (!PaintDevice::color_defined('lightgrey'));
    
    if (ScalableFeatureWidget::are_frames_visible() && 
	$PhysicalMapStripSet::view_mode !=
	$PhysicalMapStripSet::SEQUENCE_VIEW) {
	foreach my $frame (0..5) {
	    my $y_line=$ScalableFeatureWidget::CDSPositions[$frame];
	    $paintdevice->add_line($x_start,
				   $y_start+$y_line*$y_scale,
				   $x_start+$x_scale,
				   $y_start+$y_line*$y_scale,
				   'color'=>'lightgrey');
	}
    }

    $self->ScalableStrip::draw($paintdevice);

    if (ScalableFeatureWidget::are_frames_visible() && 
	$PhysicalMapStripSet::view_mode ==
	$PhysicalMapStripSet::SEQUENCE_VIEW) {
	my ($minbase,$maxbase)=$self->{map}->get_bounds();
	$startbase=$minbase
	    if ($startbase<$minbase);
	$endbase=$maxbase
	    if ($endbase>$maxbase);
	if ($startbase>0 && $endbase>0 && $endbase>$startbase) {
	    my $subseq=$self->{seqobj}->subseq($startbase,$endbase);
	    for (my $i=$startbase;$i<=$endbase;$i++) {
		if ($i<=$endbase-2) {
		    my $x_position=($i-$leftbound+1)/
			abs($self->{endbase}-$self->{startbase});
		    $x_position=1-$x_position
			if ($self->{endbase}<$self->{startbase});
		    my $frame=($i-1)%3;
		    my $codon=uc substr $subseq,($i-$startbase),3;
		    my $y_pos=ScalableFeatureWidget::get_vertical_position(1,$frame+1);
		    my $residue=$self->{codontable}->translate($codon);
		    my $color='black';
		    $color='red'
			if ($residue eq '*');
		    $paintdevice->add_text($x_start+$x_position*$x_scale,
					   $y_start+$y_pos*$y_scale,
					   $residue,
					   'color'=>$color,
					   'halign'=>'middle',
					   'valign'=>'middle',
					   'depth' => $FeatureWidget::TEXT_DEPTH);
		}
		if ($i-2>=0) {
		    my $x_position=($i-$leftbound-1)/
			abs($self->{endbase}-$self->{startbase});
		    $x_position=1-$x_position
			if ($self->{endbase}<$self->{startbase});
		    my $revframe=2-($i-1)%3;
		    my $revcodon=reverse uc substr $subseq,($i-2-$startbase),3;
		    $revcodon=~ tr/ACGT/TGCA/;
		    my $y_pos=ScalableFeatureWidget::get_vertical_position(-1,$revframe+1);
		    my $residue=$self->{codontable}->translate($revcodon);
		    my $color='black';
		    $color='red'
			if ($residue eq '*');
		    $paintdevice->add_text($x_start+$x_position*$x_scale,
					   $y_start+$y_pos*$y_scale,
					   $residue,
					   'color'=>$color,
					   'halign'=>'middle',
					   'valign'=>'middle',
					   'depth' => $FeatureWidget::TEXT_DEPTH);
		}

	    }
	}
    }

}

sub height_in_lines {

    my $lines=3.5;

    if (ScalableFeatureWidget::are_frames_visible() &&
	!ScalableFeatureWidget::are_strands_collapsed()) {
	$lines=12.5;
    } elsif (ScalableFeatureWidget::are_frames_visible() &&
	     ScalableFeatureWidget::are_strands_collapsed()) {
	$lines=6.5;
    } elsif (!ScalableFeatureWidget::are_frames_visible() &&
	     !ScalableFeatureWidget::are_strands_collapsed()) {
	$lines=6.5;
    }

    $lines/=2.0
	if ($PhysicalMapStripSet::view_mode ==
	    $PhysicalMapStripSet::BIRDSEYE_VIEW);

    return $lines;
}

1

__END__

=head1


=cut
