package PreferencesHandler;

use strict;

use IO::File;

use XML::Parser::PerlSAX;
use XML::Writer;

use PhysicalMapStripSet;
use FeatureWidget;
use ExternalViewerLinker;

$PreferencesHandler::PrefsFile=undef;

sub new {
    my $class=shift;

    my $self={};

    bless $self,$class;
}


sub start_micado {
    my $self=shift;
    my $tag=shift;

    if ( eval {require FeatureDataSource::MicadoFeatureDataSource}) {
	my $attributes=$tag->{Attributes};
	$FeatureDataSource::MicadoFeatureDataSource::DBHOST=$attributes->{host};
	$FeatureDataSource::MicadoFeatureDataSource::DBNAME=$attributes->{dbname};
	$FeatureDataSource::MicadoFeatureDataSource::DBUSER=$attributes->{user};
	$FeatureDataSource::MicadoFeatureDataSource::DBPASSWD=$attributes->{passwd};
    }

}

sub start_embl {
    my $self=shift;
    my $tag=shift;

    if ( eval {require FeatureDataSource::EMBLFeatureDataSource}) {
	my $attributes=$tag->{Attributes};
	$FeatureDataSource::EMBLFeatureDataSource::IORURL=$attributes->{iorurl};
    }

}

sub start_strands {
    my $self=shift;
    my $tag=shift;

    my $attributes=$tag->{Attributes};
    ScalableFeatureWidget::collapse_strands()
	if ($attributes->{layout} eq 'collapsed');
    ScalableFeatureWidget::expand_strands()
	if ($attributes->{layout} eq 'expanded');
    
}

sub start_frames {
    my $self=shift;
    my $tag=shift;

    my $attributes=$tag->{Attributes};
    ScalableFeatureWidget::show_frames()
	if ($attributes->{visible} eq 'true');
    ScalableFeatureWidget::hide_frames()
	if ($attributes->{visible} eq 'false');
    
}


sub start_feature {
    my $self=shift;
    my $tag=shift;

    my $attributes=$tag->{Attributes};

    my $feature=$attributes->{type};
    my $visible=$attributes->{visible};

    FeatureWidgetFactory::unmask_feature($feature)
      if ($visible eq 'true');

    FeatureWidgetFactory::mask_feature($feature)
      if ($visible eq 'false');
}

sub start_namethresh {
    my $self=shift;
    my $tag=shift;
    my $attributes=$tag->{Attributes};
    
    $ScalableFeatureWidget::display_threshold=$attributes->{percent}/100;
}

sub start_seqthresh {
    my $self=shift;
    my $tag=shift;

    my $attributes=$tag->{Attributes};

    $PhysicalMapStripSet::FEATURE_SEQUENCE_THRESH=$attributes->{bp};
}

sub start_birdseyethresh {
    my $self=shift;
    my $tag=shift;

    my $attributes=$tag->{Attributes};

    $PhysicalMapStripSet::BIRD_FEATURE_THRESH=$attributes->{kbp}*1000;
}

sub start_screen {
    my $self=shift;
    my $tag=shift;

    if ( eval {require PaintDevice::GtkPaintDevice}) {
	my $attributes=$tag->{Attributes};
	my $resolution=$attributes->{resolution};
	$PaintDevice::GtkPaintDevice::SCREEN_RESOLUTION=$resolution;
    }
}

sub start_browser {
    my $self=shift;
    my $tag=shift;

    my $attributes=$tag->{Attributes};

    my $command=$attributes->{command};

    $ExternalViewerLinker::browser_command=$command;
}

sub start_link {
    my $self=shift;
    my $tag=shift;

    my $attributes=$tag->{Attributes};

    my $name=$attributes->{name};
    my $id=$attributes->{id};
    my $url=$attributes->{url};

    $ExternalViewerLinker::dbLinks{$id}={name => $name, url => $url}
    if (defined $name && defined $id && defined $url);
}

my %subs = (
	    strands => { start => sub {start_strands @_}},
	    frames =>  { start => sub {start_frames @_}},
	    feature => { start => sub {start_feature @_}},
	    namethresh => { start => sub {start_namethresh @_}},
	    seqthresh => { start => sub {start_seqthresh @_}},
	    birdseyethresh => { start => sub {start_birdseyethresh @_}},
	    browser => { start => sub {start_browser @_}},
	    micado => { start => sub {start_micado @_}},
	    embl => { start => sub {start_embl @_}},
	    screen => { start => sub {start_screen @_}},
	    link => { start => sub {start_link @_}}
	    );

sub start_element {
	my $self=shift;
	my $tag=shift;

	my $tagname=$tag->{Name};
	&{$subs{$tagname}->{start}}($self,$tag)
		if (defined $subs{$tagname});
}


sub set_preferences_file {
    my $prefsfile=shift;

    $PreferencesHandler::PrefsFile=$prefsfile;
}

sub load_preferences {


    my $parser=new XML::Parser::PerlSAX();

    if (-r $PreferencesHandler::PrefsFile) {
	$parser->parse(Source => { SystemId => $PreferencesHandler::PrefsFile},
		       Handler => new PreferencesHandler);
    } else {
	warn "Unable to load preferences from: $PreferencesHandler::PrefsFile.\n";
    }
}

sub save_preferences {

    my $xmlfh=new IO::File(">$PreferencesHandler::PrefsFile");
    my $writer=new XML::Writer(OUTPUT=>$xmlfh,DATA_MODE=>1,DATA_INDENT=>1);
    $writer->xmlDecl('ISO-8859-1',1);
    
    print $xmlfh <<EODOCTYPE;

<!DOCTYPE mugen [
<!ELEMENT mugen (featuredatasources*,mapdisplay*,links*,browser*)>
<!ELEMENT featuredatasources (micado|embl)*>
<!ELEMENT micado EMPTY>
<!ATTLIST micado
          host CDATA #REQUIRED
	  dbname CDATA #REQUIRED
          user CDATA #REQUIRED
          passwd CDATA #REQUIRED>
<!ELEMENT embl EMPTY>
<!ATTLIST embl
          iorurl CDATA #REQUIRED>
<!ELEMENT mapdisplay (strands*,feature*,namethresh*,seqthresh*,
		      birdseyethresh*)>
<!ELEMENT strands EMPTY>
<!ATTLIST strands 
        layout (collapsed|expanded) "expanded">
<!ELEMENT frames EMPTY>
<!ATTLIST frames
	visible (true|false) "false">
<!ELEMENT feature EMPTY>
<!ATTLIST feature 
          type CDATA #REQUIRED
	  visible (true|false) "true">
<!ELEMENT namethresh EMPTY>
<!ATTLIST namethresh
	  percent CDATA #REQUIRED>
<!ELEMENT seqthresh EMPTY>
<!ATTLIST seqthresh
          bp CDATA #REQUIRED>
<!ELEMENT birdseyethresh EMPTY>
<!ATTLIST birdseyethresh
          kbp CDATA #REQUIRED>
<!ELEMENT browser EMPTY>
<!ATTLIST browser
	  command CDATA #REQUIRED>

<!ELEMENT links (link)*>
<!ELEMENT link EMPTY>
<!ATTLIST link
          name CDATA #REQUIRED
          id CDATA #REQUIRED
          url CDATA #REQUIRED>
]>

EODOCTYPE

    $writer->startTag('mugen');

    $writer->startTag('featuredatasources');

    if ( eval {require FeatureDataSource::MicadoFeatureDataSource}) {
	if (FeatureDataSource::MicadoFeatureDataSource::init()) {
	    $writer->emptyTag('micado',
			      'host' => $FeatureDataSource::MicadoFeatureDataSource::DBHOST,
			      'dbname' => $FeatureDataSource::MicadoFeatureDataSource::DBNAME,
			      'user' => $FeatureDataSource::MicadoFeatureDataSource::DBUSER,
			      'passwd' => $FeatureDataSource::MicadoFeatureDataSource::DBPASSWD);
	}
    }


    if ( eval {require FeatureDataSource::EMBLFeatureDataSource}) {
	$writer->emptyTag('embl',
			  'iorurl' => $FeatureDataSource::EMBLFeatureDataSource::IORURL);
    }

    $writer->endTag('featuredatasources');

    $writer->startTag('devices');
    $writer->emptyTag('screen',
		      'resolution' => $PaintDevice::GtkPaintDevice::SCREEN_RESOLUTION);
    $writer->endTag('devices');
    $writer->startTag('mapdisplay');

    my $layout='expanded';
    $layout='collapsed'
	if (ScalableFeatureWidget::are_strands_collapsed());
    $writer->emptyTag('strands','layout'=>$layout);


    my $frames_visible='false';
    $frames_visible='true'
	if (ScalableFeatureWidget::are_frames_visible());
    $writer->emptyTag('frames','visible'=>$frames_visible);

    foreach my $featurename (FeatureWidgetFactory::get_feature_names()) {
	$writer->emptyTag('feature',
			  'type'=>$featurename,
			  'visible'=>'false')
	    if (FeatureWidgetFactory::is_masked($featurename));
    }

    $writer->emptyTag('namethresh',
		      'percent' =>  
		      int($ScalableFeatureWidget::display_threshold*100));

    $writer->emptyTag('seqthresh',
		      'bp' => 
		      int($PhysicalMapStripSet::FEATURE_SEQUENCE_THRESH));
    $writer->emptyTag('birdseyethresh',
		      'kbp' => int($PhysicalMapStripSet::BIRD_FEATURE_THRESH/
				   1000));

    $writer->endTag('mapdisplay');

    $writer->startTag('links');
    foreach my $id (keys %ExternalViewerLinker::dbLinks) {
	$writer->emptyTag('link',
			  'name' =>
			  $ExternalViewerLinker::dbLinks{$id}->{name},
			  'id' => $id,
			  'url' => 
			  $ExternalViewerLinker::dbLinks{$id}->{url});
    }
    $writer->endTag('links');

    $writer->emptyTag('browser',
		    'command' => $ExternalViewerLinker::browser_command);

    $writer->endTag('mugen');


    $xmlfh->close();
}

1

