package FeatureDataSource::SeqDBFeatureDataSource;

use strict;

use DBI;
use DBD::Pg;
use Bio::Seq;
use Bio::SeqFeature::Generic;

use FeatureDataSource;

@FeatureDataSource::SeqDBFeatureDataSource::ISA=qw(FeatureDataSource);

sub set_preferences {
    my $mugen=$PreferencesHandler::Preferences->getDocumentElement();
    my @nodes=$mugen->getElementsByTagName("featuredatasources");
    my $fds=shift @nodes;
    my @nodes=$fds->getElementsByTagName("seqdb",0);
    if (scalar(@nodes)==0) {
	my $doctype=$PreferencesHandler::Preferences->getDoctype();
	$doctype->addElementDecl("seqdb","EMPTY");
	$doctype->addAttDef("seqdb","host","CDATA","#REQUIRED");
	$doctype->addAttDef("seqdb","dbname","CDATA","#REQUIRED");
	$doctype->addAttDef("seqdb","user","CDATA","#REQUIRED");
	$doctype->addAttDef("seqdb","passwd","CDATA","#REQUIRED");
	$doctype->addAttDef("seqdb","enabled","(yes|no)",'"no"');
	my $seqdb=$PreferencesHandler::Preferences->createElement("seqdb");
	$seqdb->setAttribute("host","127.0.0.1");
	$seqdb->setAttribute("dbname","seqdb");
	$seqdb->setAttribute("user","unknown");
	$seqdb->setAttribute("passwd","unknown");
	$seqdb->setAttribute("enabled","no");
	$fds->appendChild($seqdb);
    }
}

BEGIN {
    PreferencesHandler::register_listener(\&set_preferences);
}

my $dbh=undef;

my %descriptions=();
my %accessnumbers=();
my $entries_retrieved=0;

sub init {

    my $res=0;
    if (!$dbh) {
	my @nodes=$PreferencesHandler::Preferences->getElementsByTagName("seqdb");
	my $seqdb=shift @nodes;
	if ($seqdb) {
	    my $host=$seqdb->getAttribute("host");
	    my $dbname=$seqdb->getAttribute("dbname");
	    my $user=$seqdb->getAttribute("user");
	    my $passwd=$seqdb->getAttribute("passwd");
	    my $enabled=$seqdb->getAttribute("enabled");
	    $dbh=DBI->connect("dbi:Pg:host=$host;dbname=$dbname",$user,$passwd)
		if ($enabled eq 'yes');
	}
    }
    $res=1
	if (defined $dbh);
    
    return $res;
}


sub _retrieve_by_entry_id {
    my $self=shift;
    my $entry_id=shift;
    my $start=shift;
    my $end=shift;
    my $qualifiers=shift;

    my %features=();
    my $sth=undef;
    my $rv=undef;
    my $rowhash=undef;


    my $location_restriction='';
    $location_restriction="and (location_end>=$start and location_begin<=$end)" 
	if ($start>0 && $end>$start);

    my $sth=$dbh->prepare("select features.feature_id as feature_id, feature_types.name as name, location_begin, location_end, location_strand from features join feature_types using(feature_type_id) where entry_id = $entry_id $location_restriction");
    my $rv=$sth->execute;
    while ($rowhash=$sth->fetchrow_hashref()) {
	$features{$rowhash->{feature_id}}={
	    primary=>$rowhash->{name},
	    start=>$rowhash->{location_begin},
	    end=>$rowhash->{location_end},
	    strand=>$rowhash->{location_strand},
	    tags => {}
	};
    }


    $sth=$dbh->prepare("select features.feature_id as feature_id, qualifier_types.name as name, value from qualifiers join features using (feature_id) join qualifier_types using (qualifier_type_id) where entry_id = $entry_id $location_restriction");
    $rv=$sth->execute;
    while ($rowhash=$sth->fetchrow_hashref()) {
	$features{$rowhash->{feature_id}}->{tags}->{$rowhash->{name}}=$rowhash->{value};
    }

    my $seq='';
    my $sequences='' ;
    $sth=$dbh->prepare("select sequence from sequences where entry_id = $entry_id");
    $rv=$sth->execute;
    $rowhash=$sth->fetchrow_hashref();
    $seq=$rowhash->{sequence} ;


    my $accessnumber='';
    my $rows=$dbh->selectall_arrayref("SELECT accession FROM entries WHERE entry_id=$entry_id");
    $accessnumber=$rows->[0]->[0];
    $seq=new Bio::Seq(-seq => $seq,
		      -id => 'seqdb_retrieved_sequence',
		      -accession_number => $accessnumber);
    
    foreach my $feature_id (sort keys %features) {
	my $start=$features{$feature_id}->{start};
	my $end=$features{$feature_id}->{end};
	my $strand=$features{$feature_id}->{strand};
	my $primary=$features{$feature_id}->{primary};

	my $seqfeature=new Bio::SeqFeature::Generic(
						    -start => $start,
						    -end => $end,
						    -strand => $strand,
						    -primary => $primary,
						    -tag => $features{$feature_id}->{tags});
	$seq->add_SeqFeature($seqfeature);
    }

    return $seq;

}

sub _retrieve_by_description {
    my $self=shift;
    my $description=shift;
    my $start=shift;
    my $end=shift;
    my $qualifiers=shift;

    my $entry_id=$descriptions{$description};
    return undef
	if (!defined $entry_id);

    return $self->_retrieve_by_entry_id($entry_id,$start,$end,$qualifiers);
}

sub _retrieve_by_accessnumber {
    my $self=shift;
    my $accessnumber=shift;
    my $start=shift;
    my $end=shift;
    my $qualifiers=shift;

    my $entry_id=$accessnumbers{$accessnumber};
    return undef
	if (!defined $entry_id);

    return $self->_retrieve_by_entry_id($entry_id,$start,$end,$qualifiers);
}


sub _retrieve_entry_descriptions {

    my $sth=$dbh->prepare("SELECT entry_id,accession,description FROM entries");
    my $rv=$sth->execute();
    while (my $rowhash=$sth->fetchrow_hashref()) {
	my $entry_id=$rowhash->{entry_id};
	my $description=$rowhash->{description};
	my $accessnumber=$rowhash->{accession};
	$descriptions{$description}=$entry_id;
	$accessnumbers{$accessnumber}=$entry_id;
    }
    $entries_retrieved=1;
}

sub get_entry_descriptions {

    _retrieve_entry_descriptions
	if (!$entries_retrieved);

    return sort keys %descriptions;

}

sub get_access_number {
    my $description=shift;

    _retrieve_entry_descriptions
	if (!$entries_retrieved);

    my $entry_id=$descriptions{$description};
    my $accessnumber=undef;
    foreach my $tmp_accessnumber (keys %accessnumbers) {
	$accessnumber=$tmp_accessnumber
	if ($accessnumbers{$tmp_accessnumber} == $entry_id);
    }
    return $accessnumber;
}

sub new {
    my $class=shift;
    my %params=();

    $params{start}=-1;
    $params{end}=-1;
    $params{qualifiers}=0;
    while (my $paramname=shift) {
	$params{lc $paramname}=shift;
    }

    my $self=_new FeatureDataSource;

    bless $self,$class;

    my $seqobj=undef;

    if (!defined $dbh) {
	warn "Trying to use SeqDB as data source without valid database connection.\n";
	return undef;
    }

    _retrieve_entry_descriptions
	if (!$entries_retrieved);

    if (defined $params{description}) {
	$seqobj=$self->_retrieve_by_description($params{description},
						$params{start},
						$params{end},
						$params{qualifiers});
    } elsif (defined $params{accessnumber}){
	$seqobj=$self->_retrieve_by_accessnumber($params{accessnumber},
						 $params{start},
						 $params{end},
						 $params{qualifiers});
    }

    $self->FeatureDataSource::_load_features($seqobj)
	if (defined $seqobj);

    return $self;
}

END {
    $dbh->disconnect()
	if (defined $dbh);
}

1
