package FeatureDataSource::MicadoFeatureDataSource;

use strict;

use DBI;
use DBD::Pg;
use Bio::Seq;
use Bio::SeqFeature::Generic;

use FeatureDataSource;
use PreferencesHandler;

@FeatureDataSource::MicadoFeatureDataSource::ISA=qw(FeatureDataSource);

sub set_preferences {
    my $mugen=$PreferencesHandler::Preferences->getDocumentElement();
    my @nodes=$mugen->getElementsByTagName("featuredatasources");
    my $fds=shift @nodes;
    my @nodes=$fds->getElementsByTagName("micado",0);
    if (scalar(@nodes)==0) {
	my $doctype=$PreferencesHandler::Preferences->getDoctype();
	$doctype->addElementDecl("micado","EMPTY");
	$doctype->addAttDef("micado","host","CDATA","#REQUIRED");
	$doctype->addAttDef("micado","dbname","CDATA","#REQUIRED");
	$doctype->addAttDef("micado","user","CDATA","#REQUIRED");
	$doctype->addAttDef("micado","passwd","CDATA","#REQUIRED");
	$doctype->addAttDef("micado","enabled","(yes|no)",'"no"');
	my $micado=$PreferencesHandler::Preferences->createElement("micado");
	$micado->setAttribute("host","127.0.0.1");
	$micado->setAttribute("dbname","micado");
	$micado->setAttribute("user","unknown");
	$micado->setAttribute("passwd","unknown");
	$micado->setAttribute("enabled","no");
	$fds->appendChild($micado);
    }
}

BEGIN {
    PreferencesHandler::register_listener(\&set_preferences);
}

my $BUFFSIZE=4096;

my $dbh=undef;

my %complete_genomes=();

sub init {
    my $res=0;
    if (!$dbh) {
	my @nodes=$PreferencesHandler::Preferences->getElementsByTagName("micado");
	my $micado=shift @nodes;
	if ($micado) {
	    my $host=$micado->getAttribute("host");
	    my $dbname=$micado->getAttribute("dbname");
	    my $user=$micado->getAttribute("user");
	    my $passwd=$micado->getAttribute("passwd");
	    my $enabled=$micado->getAttribute("enabled");
	    $dbh=DBI->connect("dbi:Pg:host=$host;dbname=$dbname",$user,$passwd)
		if ($enabled eq 'yes');
	}
    }
    $res=1
	if (defined $dbh);

    return $res;

}


sub _retrieve_by_access_number {
    my $self=shift;
    my $accessnumber=shift;
    my $start=shift;
    my $end=shift;
    my $qualifiers=shift;

    my %features=();
    my $sth=undef;
    my $rv=undef;
    my $rowhash=undef;


    #retrieve all locations for given access number
    my $location_restriction='';
    $location_restriction="and ((startpos_begin>=$start and startpos_begin<=$end) or (stoppos_end>=$start and stoppos_end<=$end) or (startpos_begin<$start and stoppos_end>$end))" 
	if ($start>0 && $end>$start);
    $sth=$dbh->prepare("select code_feat, startpos_begin, stoppos_end, strand from locations where accession = '$accessnumber' $location_restriction");
    $rv=$sth->execute;
    while ($rowhash=$sth->fetchrow_hashref()) {
	my $startpos_begin=$rowhash->{startpos_begin};
	my $stoppos_end=$rowhash->{stoppos_end};
	my $strand=$rowhash->{strand};
	$features{$rowhash->{code_feat}}={
	    start=>$startpos_begin,
	    end=>$stoppos_end,
	    strand=>$strand,
	    tags => {}
	};
    }


    #retrieve all feature types for given access number
    $sth=$dbh->prepare("select code_feat, type_feat from features where accession = '$accessnumber'");
    $rv=$sth->execute;
    while ($rowhash=$sth->fetchrow_hashref()) {
	$features{$rowhash->{code_feat}}->{primary}=$rowhash->{type_feat}
	if (defined $features{$rowhash->{code_feat}});
    }

    #retrieve qualifiers with values for given access number.
    if (!$qualifiers) {
	# Don't retrieve all qualifiers : only 'gene' and 'product' are
	# needed for correct map display.
	# Queries are split to improve speed : one 'OR' query seems
	# slower than two separate queries...
	$sth=$dbh->prepare("select code_feat, type_qual, qualifier from qualifiers  where accession = '$accessnumber' and type_qual='gene'");
	$rv=$sth->execute;
	while ($rowhash=$sth->fetchrow_hashref()) {
	    $features{$rowhash->{code_feat}}->{tags}->{$rowhash->{type_qual}}=$rowhash->{qualifier}
	    if (defined $features{$rowhash->{code_feat}});
	}
	$sth=$dbh->prepare("select code_feat, type_qual, qualifier from qualifiers  where accession = '$accessnumber' and type_qual='product'");
	$rv=$sth->execute;
	while ($rowhash=$sth->fetchrow_hashref()) {
	    $features{$rowhash->{code_feat}}->{tags}->{$rowhash->{type_qual}}=$rowhash->{qualifier}
	    if (defined $features{$rowhash->{code_feat}});
	}
    } else {
	my $query_string="select code_feat, type_qual, qualifier from qualifiers  where accession =  '$accessnumber'";
	$sth=$dbh->prepare($query_string);
	$rv=$sth->execute;
	while ($rowhash=$sth->fetchrow_hashref()) {
	    $features{$rowhash->{code_feat}}->{tags}->{$rowhash->{type_qual}}=$rowhash->{qualifier}
	    if (defined $features{$rowhash->{code_feat}});
	}
    }

    #retrieve sequence for given access number.
    my $seq='';
    my $sequences='' ;
    $sth=$dbh->prepare("select sequences from dna_seq where accession = '$accessnumber'");
    $rv=$sth->execute;
    
   ## Modif Helene 19/12/02 => suppression lob de micado
   $sth->bind_columns(\$sequences);
   $sth->fetch;
   $sth->finish();
   $seq=$sequences ;

    $seq=new Bio::Seq(-seq => $seq,
		      -id => 'micado_retrieved_sequence',
		      -accession_number => $accessnumber);
    
    foreach my $feature (sort keys %features) {
	my $start=$features{$feature}->{start};
	my $end=$features{$feature}->{end};
	my $strand=$features{$feature}->{strand};
	my $primary=$features{$feature}->{primary};

	my $seqfeature=new Bio::SeqFeature::Generic(
						    -start => $start,
						    -end => $end,
						    -strand => $strand,
						    -primary => $primary,
						    -tag => $features{$feature}->{tags});
	$seq->add_SeqFeature($seqfeature);
    }

    return $seq;

}

sub _retrieve_by_organism {
    my $self=shift;
    my $organism=shift;
    my $start=shift;
    my $end=shift;
    my $qualifiers=shift;

    return undef
	if (!defined $complete_genomes{$organism});

    my $accession=$complete_genomes{$organism};
    
    
    return $self->_retrieve_by_access_number($accession,$start,$end,$qualifiers);
}

sub _retrieve_complete_genome_names {

    my $sth=$dbh->prepare("select species,accession,type from complete_genomes");
    my $rv=$sth->execute();
    while (my $rowhash=$sth->fetchrow_hashref()) {
	my $species=$rowhash->{species};
	my $accession=$rowhash->{accession};
	my $type=$rowhash->{type};
	my $species_type="$species, $type";
	$complete_genomes{$species_type}=$accession;
    }
}

sub get_complete_genome_names {

    _retrieve_complete_genome_names
	if (!scalar(keys(%complete_genomes)));

    my @complete_genome_names=();
    foreach my $species_type (sort keys %complete_genomes) {
	push @complete_genome_names,$species_type;
    }

    return @complete_genome_names;

}

sub get_access_number {
    my $organism=shift;

    _retrieve_complete_genome_names
	if (!scalar(keys(%complete_genomes)));

    return $complete_genomes{$organism};
}

sub new {
    my $class=shift;
    my %params=();

    $params{start}=-1;
    $params{end}=-1;
    $params{qualifiers}=0;
    while (my $paramname=shift) {
	$params{lc $paramname}=shift;
    }

    my $self=_new FeatureDataSource;

    bless $self,$class;

    my $seqobj=undef;

    if (!defined $dbh) {
	warn "Trying to use Micado as data source without valid database connection.\n";
	return undef;
    }

    if (defined $params{organism} && defined $params{gene}) {
	$seqobj=$self->_retrieve_by_organism_gene($params{organism},
						  $params{gene},
						  $params{start},
						  $params{end},
						  $params{qualifiers});
    } elsif (defined $params{organism} && !defined $params{gene}) {
	$seqobj=$self->_retrieve_by_organism($params{organism},
					     $params{start},
					     $params{end},
					     $params{qualifiers});
    } elsif (defined $params{accessnumber}){
	$seqobj=$self->_retrieve_by_access_number($params{accessnumber},
						  $params{start},
						  $params{end},
						  $params{qualifiers});
    }

    $self->FeatureDataSource::_load_features($seqobj)
	if (defined $seqobj);
    
    return $self;
}

END {
    $dbh->disconnect()
	if (defined $dbh);
}

1
