package FeatureDataSource::SmallgenFeatureDataSource;

use strict;

use DBI;
use DBD::Pg;
use Bio::Seq;
use Bio::SeqFeature::Generic;

use FeatureDataSource;
use PreferencesHandler;

@FeatureDataSource::SmallgenFeatureDataSource::ISA=qw(FeatureDataSource);


sub set_preferences {
    my $mugen=$PreferencesHandler::Preferences->getDocumentElement();
    my @nodes=$mugen->getElementsByTagName("featuredatasources");
    my $fds=shift @nodes;
    my @nodes=$fds->getElementsByTagName("smallgen",0);
    if (scalar(@nodes)==0) {
	my $doctype=$PreferencesHandler::Preferences->getDoctype();
	$doctype->addElementDecl("smallgen","EMPTY");
	$doctype->addAttDef("smallgen","host","CDATA","#REQUIRED");
	$doctype->addAttDef("smallgen","dbname","CDATA","#REQUIRED");
	$doctype->addAttDef("smallgen","user","CDATA","#REQUIRED");
	$doctype->addAttDef("smallgen","passwd","CDATA","#REQUIRED");
	my $smallgen=$PreferencesHandler::Preferences->createElement("smallgen");
	$smallgen->setAttribute("host","127.0.0.1");
	$smallgen->setAttribute("dbname","smallgenes");
	$smallgen->setAttribute("user","unknown");
	$smallgen->setAttribute("passwd","unknown");
	$fds->appendChild($smallgen);
    }
}

BEGIN {
    PreferencesHandler::register_listener(\&set_preferences);
}

my $dbh=undef;

my %descriptions=();
my %accessnumbers=();
my $entries_retrieved=0;

sub init {

    my $res=0;
    if (!$dbh) {
	my @nodes=$PreferencesHandler::Preferences->getElementsByTagName("smallgen");
	my $smallgen=shift @nodes;
	if ($smallgen) {
	    my $host=$smallgen->getAttribute("host");
	    my $dbname=$smallgen->getAttribute("dbname");
	    my $user=$smallgen->getAttribute("user");
	    my $passwd=$smallgen->getAttribute("passwd");
	    $dbh=DBI->connect("dbi:Pg:host=$host;dbname=$dbname",$user,$passwd);
	}
    }
    $res=1
	if (defined $dbh);

    return $res;
}


sub _retrieve_by_entry_id {
    my $self=shift;
    my $entry_id=shift;
    my $start=shift;
    my $end=shift;
    my $qualifiers=shift;               # non renseign

    my %features=();
    my %features_cds=();
    my $sth=undef;
    my $sth2=undef;
    my $rv=undef;
    my $rowhash=undef;
    my $ref_hash=undef;
    my $compteur_gb=0;
    my $compteur_show =0;

    my $location_restriction='';
    $location_restriction="and (posmin>=$start and posmax<=$end)" 
	if ($start>0 && $end>$start);

    # Entres autres que CDS (Genbank)
    my $sth=$dbh->prepare("select gb_id, type_feat, posmin, posmax, strand 
                             from gb_feature
                             where contig_id = $entry_id $location_restriction
                               and type_feat <> 'CDS'
                              order by posmin");

    my $rv=$sth->execute;
    while ($rowhash=$sth->fetchrow_hashref()) {
	$compteur_gb++;
	$features{$compteur_gb}={
	    id=>$rowhash->{gb_id},
	    primary=>$rowhash->{type_feat},
	    start=>$rowhash->{posmin},
	    end=>$rowhash->{posmax},
	    strand=>$rowhash->{strand},
	    tags => {}
	};

        my $sth2=$dbh->prepare("select type_qual, value_qual
                             from gb_qualifier
                             where gb_id = $rowhash->{gb_id}");
	
	$sth2->execute();
	while($ref_hash = $sth2->fetchrow_hashref){ 	
	    $features{$compteur_gb}->{tags}->{$ref_hash->{type_qual}}=$ref_hash->{value_qual};
	}

	$sth2->finish();
    }

    # Entres CDS (Genbank + show)
    my $sth=$dbh->prepare("select putcds_id, posmin, posmax, strand
                             from putative_cds
                             where putative_cds.contig_id = $entry_id $location_restriction
                               order by posmin");

    my $rv=$sth->execute;
    while ($rowhash=$sth->fetchrow_hashref()) {
	$compteur_show++;
	$features_cds{$compteur_show}={
	    id=>$rowhash->{putcds_id},
	    primary=>'CDS',
	    start=>$rowhash->{posmin},
	    end=>$rowhash->{posmax},
	    strand=>$rowhash->{strand},
	    tags => {}
	};

	
	my $sth2=$dbh->prepare("select type_qual, value_qual
                             from putative_cds_ref, gb_qualifier
                             where putative_cds_ref.putcds_id = $rowhash->{putcds_id}
                               and putative_cds_ref.gb_id <> 0
                               and gb_qualifier.gb_id = putative_cds_ref.gb_id");

	$sth2->execute();

	if ($sth2->rows==0){
           # Entres uniquement dans show (pas de reference GenBank trouvee)
	    	my $sth2=$dbh->prepare("select type_qual, value_qual
                             from putative_cds_ref, show_res_qual
                             where putative_cds_ref.putcds_id = $rowhash->{putcds_id}
                               and putative_cds_ref.show_res_id <> 0
                               and show_res_qual.show_res_id = putative_cds_ref.show_res_id");
		
		$sth2->execute();
		while($ref_hash = $sth2->fetchrow_hashref){ 	
		    $features_cds{$compteur_show}->{tags}->{$ref_hash->{type_qual}}=$ref_hash->{value_qual};
		}

		$sth2->finish();	
	}
	else {
           # Entres dans Genbank

    	   while($ref_hash = $sth2->fetchrow_hashref){ 	
	    $features_cds{$compteur_show}->{tags}->{$ref_hash->{type_qual}}=$ref_hash->{value_qual};
	    }

 	    $sth2->finish();
        }
	
    }

    #print "compteur 1", $compteur_gb, " compteur CDS ", $compteur_show;   

    my $seq='';
    my $sequences='' ;
    $sth=$dbh->prepare("select dna_sequence from contig where contig_id = $entry_id");
    $rv=$sth->execute;
    $rowhash=$sth->fetchrow_hashref();
    $seq=$rowhash->{dna_sequence} ;


    my $accessnumber='';
    my $rows=$dbh->selectall_arrayref("SELECT contig_ac FROM contig WHERE contig_id=$entry_id");
    $accessnumber=$rows->[0]->[0];
    $seq=new Bio::Seq(-seq => $seq,
		      -id => 'smallgen_retrieved_sequence',
		      -accession_number => $accessnumber);
    
    my $indice_cds=1;
    foreach my $feature_id (%features) {

	my $start=$features{$feature_id}->{start};
	my $end=$features{$feature_id}->{end};
	my $strand=$features{$feature_id}->{strand};
	my $primary=$features{$feature_id}->{primary};

	while ($start > $features_cds{$indice_cds}->{start} && $indice_cds <= $compteur_show) {
	    my $seqfeature=new Bio::SeqFeature::Generic(
						    -start => $features_cds{$indice_cds}->{start},
						    -end => $features_cds{$indice_cds}->{end},
						    -strand => $features_cds{$indice_cds}->{strand} ,
						    -primary => $features_cds{$indice_cds}->{primary},
						    -tag => $features_cds{$indice_cds}->{tags});
	    $seq->add_SeqFeature($seqfeature);  
	    $indice_cds++;
	}

	my $seqfeature=new Bio::SeqFeature::Generic(
						    -start => $start,
						    -end => $end,
						    -strand => $strand,
						    -primary => $primary,
						    -tag => $features{$feature_id}->{tags});
	$seq->add_SeqFeature($seqfeature);
    }

    return $seq;

}

sub _retrieve_by_description {
    my $self=shift;
    my $description=shift;
    my $start=shift;
    my $end=shift;
    my $qualifiers=shift;

    my $entry_id=$descriptions{$description};
    return undef
	if (!defined $entry_id);

    return $self->_retrieve_by_entry_id($entry_id,$start,$end,$qualifiers);
}

sub _retrieve_by_accessnumber {
    my $self=shift;
    my $accessnumber=shift;
    my $start=shift;
    my $end=shift;
    my $qualifiers=shift;

    my $entry_id=$accessnumbers{$accessnumber};

    return undef
	if (!defined $entry_id);

    return $self->_retrieve_by_entry_id($entry_id,$start,$end,$qualifiers);
}


sub _retrieve_entry_descriptions {

    my $sth=$dbh->prepare("SELECT contig_id,contig_ac,organism 
                             FROM contig, genome
                             WHERE genome.genome_id = contig.genome_id");
    my $rv=$sth->execute();
    while (my $rowhash=$sth->fetchrow_hashref()) {
	my $entry_id=$rowhash->{contig_id};
	my $description=$rowhash->{organism};
	my $accessnumber=$rowhash->{contig_ac};
	$descriptions{$description}=$entry_id;
	$accessnumbers{$accessnumber}=$entry_id;
    }
    $entries_retrieved=1;
}

sub get_entry_descriptions {

    _retrieve_entry_descriptions
	if (!$entries_retrieved);

    return sort keys %descriptions;

}

sub get_access_number {
    my $description=shift;

    _retrieve_entry_descriptions
	if (!$entries_retrieved);

    my $entry_id=$descriptions{$description};
    my $accessnumber=undef;
    foreach my $tmp_accessnumber (keys %accessnumbers) {
	$accessnumber=$tmp_accessnumber
	if ($accessnumbers{$tmp_accessnumber} == $entry_id);
    }
    return $accessnumber;
}

sub new {
    my $class=shift;
    my %params=();

    $params{start}=-1;
    $params{end}=-1;
    $params{qualifiers}=0;
    while (my $paramname=shift) {
	$params{lc $paramname}=shift;
    }

    my $self=_new FeatureDataSource;

    bless $self,$class;

    my $seqobj=undef;

    if (!defined $dbh) {
	warn "Trying to use Smallgenes as data source without valid database connection.\n";
	return undef;
    }

    _retrieve_entry_descriptions
	if (!$entries_retrieved);


    if (defined $params{description}) {
	$seqobj=$self->_retrieve_by_description($params{description},
						$params{start},
						$params{end},
						$params{qualifiers});
    } elsif (defined $params{accessnumber}){

	$seqobj=$self->_retrieve_by_accessnumber($params{accessnumber},
						 $params{start},
						 $params{end},
						 $params{qualifiers});
    }

    $self->FeatureDataSource::_load_features($seqobj)
	if (defined $seqobj);

    return $self;
}

END {
    $dbh->disconnect()
	if (defined $dbh);
}

1
