# : ### FICHIER calib.nls2.s ###
calib <- function(x, ...)
  UseMethod("calib")

calib.default <- function(x,...)
  {
  if (!inherits(x,"nls2"))
    stop("This function is valid only on a 'nls2.object'")
  }

# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
#	Specifications:
# - Extraire les resultats du nls2.object:
#      valide que si OK,
# - Verifier l'object:	 en particulier, 1 seule va explicative,
#      pas de sedo, pas d'alterne, et vari.type=CST pour le calcul de S
#      et CST ou method=MLT ou MLTB pour celui de R
# - Initialiser GNLControle et GNLTrace
# - Zmoy=mean(Zj), n=length des donnees origine
# - Creer les arbres de calcul de f et f-1 si besoin,
# - Calculer les xbounds par defaut, bornes de l'espace 
#      de recherche xbounds= (l1,l2)= (min(x), max(x))
# - Calculer fl1=f(l1,thetachapeau), fl2(l2,thetachapeau)
#      et flmin et flmax = min et max de ces valeurs
# - Verifier monotonie si demande  dans tout l'intervalle donnees standard
#   + espace de recherche	, i.e
#      verifier que les reponses sont croissantes ou decroissantes
#      sur l'axe des x
# - Zchapeau=f-1(Zmoy, thetachapeau) ou bien l1 ou bien l2
#      si Zchapeau en-dehors de [l1,l2]
# - Schapeau=sum(residus**2) + sum((Zj-Zmoy)**2)
# - Calcul de df.zchapeau=df(Zchapeau, thetachapeau)

	
# --------------------	
#  Calcul de S:
# --------------------	
#  - D=sqrt(Schapeau) * 
#     sqrt(1/m + (df.zchapeau * as.varchapeau/sigma2 * df.zchapeau))
#  - A=sqrt(n+m)/D
#  - S.xbounds=[A*(Zmoy-fl1), A*(Zmoy-fl2)] en verifiant si ca appartient
#  aux bornes de l'intervalle de confiance [b1,b2]
#  - S.conf.int=[f-1(Zmoy-b1/A, thetachapeau),
#                f-1(Zmoy-b2/A, thetachapeau)]
#  en verifiant si ca appartient aux bornes de l'intervalle 
#  de recherche [l1,l2]
	
	
# --------------------	
#  Calcul de R:
# --------------------			
# - Rechercher les data si on ne les a pas deja cherchees lors du
#      calcul de x.bounds 
# - Interdire les courbes 
# - Maj du nls2.object$stat.ctx: theta.start, beta.start,
#     si les mu sont connus leur rajouter 0 pour l'observation inconnue 
#    qu'on va rajouter aux donnees, et si sigma2 est connu, repeter sa
#    valeur.  
#	- Si la variance est non CST:
#       ---------------------------- 
# - Preparer l'estimation:	
#	data2=data + (Zchapeau, Zj) ,
#       le modele est le modele de f avec x0 en dernier parametre
#          theta, parametre initialise a Zchapeau, et avec les
#          contraintes appartient a [l1, l2] 
#       la methode est MYOWN, avec  2 stat exhau, matrice W sym, 
#       stat.crit=-2logV 
# - On estime: ca donne un nouveau thetachapeau, Lchapeau=-2logV, 
#  et un xchapeau (le dernier theta estime)
#        - Fin variance est non CST

#       Boucle sur les points de la grille, xi:
#        ----------------------------------
#  - estimation en rajoutant le point (xi,Zj), renls2=T
#  Le model est celui de f, la methode est celle initiale.
#  On obtient 	residus et  Ltilde=-2logV/(n+m)
#  - Si la variance est CST:
#        IR=sqrt( (m+n)*log(sum(residus**2)/Schapeau))
#  Sinon:
#        IR= sqrt(-(n+m)*(Lchapeau-Ltilde))
#  - IR=sign(Zmoy-f(thetachapeau, xi))*IR
#  - si IR dans [b1,b2], xi appartient a R.

#  FIN Specifications
	
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# examCalibnls2: 
# Role:
#  renvoyer n, k, theta ,beta, gamf, gamv,response, asvar,residuals , etc...
#  theta et as.var en dimension estimes (et non actifs)
# PS:	 pour la calib, il ne peut y avoir qu'1 etape	
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
examCalibnls2 <- function(nls2.object)
{
if (is.null(nls2.object) )
  {
  stop("Error when running nls2.")
  }
	
# Verification qu'on n'est pas en alterne
# (en effet, il faut variance constante ou Maximum Likekihood
# ce qui n'est pas possible en alterne
# ------------------------------------------------------------
# On fait la verif ici et non dans VerifCalib, car ici, on extrait des
# resultats qu'on ne peut trouver que dans les composants step en alterne
if (nls2.object$nb.steps != 1)
  stop("Alternated method not possible for the estimation of the 'nls2.object'\n")
	
if( is.null(nls2.object$code) || (nls2.object$code !=0))
  {
  # on ne tolere que OK
  stop( paste("Execution of nls2 ends with message:", 
              nls2.object$message))
  }

if (is.null(nls2.object$sigma2) || all(is.na(nls2.object$sigma2)))
  stop("The nls2.object does not include valid sigma2")
if (is.null(nls2.object$as.var) || all(is.na(nls2.object$as.var)))
  stop("The nls2.object does not include a valid as.var")
if (is.null(nls2.object$residuals) || all(is.na(nls2.object$residuals)))
  stop("The nls2.object does not include valid residuals")

if (is.null(nls2.object$beta) || all(is.na(nls2.object$beta)))
  {
  beta <- 0
  qbase <- 0
  }	
else
  {
  beta <- nls2.object$beta
  qbase <- length(beta)
  }	
	
if (is.null(nls2.object$model$gamf) || all(is.na(nls2.object$model$gamf)))
  {
  gamf <- 0
  nbgf <- 0
  }
else
  {
  gamf <- nls2.object$model$gamf
  nbgf <- length(gamf)
  }		
if (is.null(nls2.object$model$gamv) ||  all(is.na(nls2.object$model$gamv)))
  {
  gamv <- 0
  nbgv <- 0
  }
else
  {
  gamv <- nls2.object$model$gamv
  nbgv <- length(gamv)
  }		


n <- length(nls2.object$residuals)
if (is.null(nls2.object$data.stat$Y1))
  stop("The nls2.object should include the component 'data.stat$Y1'")
k <- length(nls2.object$data.stat$Y1)
if (is.null(nls2.object$theta))
  stop("The nls2.object should include the component 'theta'")
pbase <- length(nls2.object$theta)

if (!is.numeric(nls2.object$response))
  stop("The nls2.object should include the component 'response'")
if (!is.numeric(nls2.object$replications))
  stop("The nls2.object should include the component 'replications'")

return(list(n=n,k=k,pbase=pbase,qbase=qbase,nbgf=nbgf,nbgv=nbgv,
	 theta=nls2.object$theta, beta=beta, gamf=gamf, gamv=gamv,
	response=nls2.object$response, as.var=nls2.object$as.var, 
	residuals=nls2.object$residuals, 
	sigma2=nls2.object$sigma2))

}
# ----- end of function examCalibnls2 ---------------------------------


# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# verifCalibnls2:
# Role:
# faire les verif de calib.nls2
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
verifCalibnls2 <- function(nls2.object, file, ord,
                theta,
                conf.level,
                conf.bounds,
                x.bounds, check.points,
	        R.grid, 
	        R.max.size,
	        R.nsplit,
	        R.nx)

{
# Verification de la structure des arguments:
# -------------------------------------------
if (!is.character(file) || (length(file) >1))
  stop("Argument `file' should be a character string\n")
if (!is.numeric(ord) || !is.vector(ord))
  stop("Argument `ord' should be a numerical vector\n")
if (!is.numeric(conf.level) || (length(conf.level) >1) ||
     (conf.level <=0) || (conf.level >=1))
  stop("The argument 'conf.level' must be a value between 0 and 1\n")
if (!is.numeric(conf.bounds) || !is.vector(conf.bounds) || (length(conf.bounds) !=2))
  stop("Argument `conf.bounds' should be a numerical vector of length 2\n")
if (conf.bounds[1]>=conf.bounds[2])
  stop(" `conf.bounds[1]' must be greater than `conf.bounds[2]' \n")
if (!all(is.na(x.bounds)) && (!is.numeric(x.bounds) || (length(x.bounds) !=2)))
  stop("Argument `x.bounds' should be a numerical vector of length 2\n")
if (!all(is.na(x.bounds)) && (x.bounds[1]>=x.bounds[2]))
  stop(" `x.bounds[1]' must be greater than `x.bounds[2]' \n")
if (!is.numeric(check.points) || ((check.points<3) && (check.points!=0)))
  stop(paste("Argument `check.points':", check.points,
"should be greater or equal to 3 or a null integer\n"))

# Verif des arguments relatifs a R:	
# --------------------------------		
if (!is.numeric(R.grid) || ((R.grid<=1) && (R.grid!=0)))
  stop(paste("Argument `R.grid':", R.grid,
"should be an integer null or greater than 1\n"))
if (!is.numeric(R.max.size) || ((R.grid>1) && (R.max.size<2)))
  stop(paste("Argument `R.max.size':", R.max.size,
"should be an integer greater than 1 if R.grid>1\n"))
if ( (!is.numeric(R.nsplit) || (R.nsplit < 0)))
    stop(paste("Argument `R.nsplit':", R.nsplit,
"should be positive or null\n"))
if ( (!is.numeric(R.nx) || (R.nx < 0)))
    stop(paste("Argument `R.nx':", R.nx,
"should be positive or null\n"))

# Verification que x est univariate:
# ---------------------------------
if (length(nls2.object$X.names) >1)
  stop("In this current version, there must be only one dependent variable\n")
						
# Verification que la variance est constante:
# Obligatoire pour le calcul de S mais pas pour R	
# -------------------------------------------
if (nls2.object$model$vari.type!="CST")
  {	
  warning("The variance of y (model$vari.type) must be constant for calculating the S confidence interval\n")
  Scalc <- F
  }
else
  Scalc <- T
	
# Verification que si, la variance n'est pas constante,
# l'estimateur est Maximum Likelihood pour le calcul de R:
# ---------------------------------------------------------	
if ( (R.grid !=0) && (nls2.object$model$vari.type!="CST") && 
	(nls2.object$method != "MLT") && (	nls2.object$method != "MLTB"))
  {
  warning(paste(
"For calculating the R confidence interval, the variance of y (model$vari.type) must be constant\n",
  "or the method a Maximum Likelihood method ('MLT' or 'MLTB')\n"))
  R.grid  <- 0
  }		
			
# Verification qu'on n'est pas dans un cas sedo:
# un sedo n'est pas possible car on a besoin de f-1	
# -------------------------------------------
if (is.character(nls2.object$model))
  {		
  RetModel <-  analFilenls2(nls2.object$model)
  }
else
  {		
  RetModel <-  analFilenls2(nls2.object$model$file)
  }	
if (RetModel$CasSedo ==1) 
   stop("Model described by differential equations not accepted in this current version\n")

# Analyse du fichier formel de l'inverse:
RetInv <-  analInvnls2(file)

# si le vecteur des parametres est labelle, 
# on ne garde que les elements dont les
# labels correspondent a ceux du fichier formel:
# (on peut avoir plus de parametres estimes qu'il n'y a de parametres
# decrits dans le fichier formel de l'inverse dans le cas de contraintes)
Ret <- verifNamePnls2(theta, RetInv$NomParact, RetInv$NbParact, "theta")
if (is.null(Ret)) return(NULL)
RetInv$theta <- Ret$Param
# idem pour les pbis:
Ret <- verifNamePnls2(nls2.object$model$gamf, RetInv$NomPbisabs, RetInv$NbPbisabs, "gamf")
if (is.null(Ret)) return(NULL)		
RetInv$pbisabs <-  Ret$Param
Ret <- verifNamePnls2(nls2.object$model$gamv, RetInv$NomPbisvar, RetInv$NbPbisvar, "gamv")
if (is.null(Ret)) return(NULL)
RetInv$pbisvar <-  Ret$Param

return(list(RetModel=RetModel,RetInv=RetInv, Scalc=Scalc, R.grid=R.grid))
}
# -------------- fin de verifCalibnls2 ---------------------------

# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# calibffnls2: call the program calcf <-  that evaluates the model
# on given values of the parameters and of x
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
calibffnls2 <- function(x, theta, gamf)
{
x <- as.matrix(x)
nblig <- nrow(x)
nbcol <- ncol(x)
pbase <- length(theta)
nbgf <- length(gamf)
f <- vector(mode="double",length=nblig)
df <- vector(mode="double", length=(nblig*pbase))
le <- ie <- 0

Ret <- .C("calibffnls2",
   x=as.double(x),
   nblig=as.integer(nblig), nbcol=as.integer(nbcol),
   pbase=as.integer(pbase),
   nbgf=as.integer(nbgf),
   Theta=as.double(theta),
   GamF=as.double(gamf),
   f=as.double(f), df=as.double(df),
   le=as.integer(le), ie=as.integer(ie))

if(Ret$le !=0)
  {
  # Treatment of error:
  coderr <- c(
   "the function f",
   "the function v",
   "the derivatives of f",
   "the derivatives of v against the theta parameters",
   "the derivatives of v against the beta parameters",
   "auxiliary variables",
   "the derivatives of auxiliary variables against the theta parameters",
   "the derivatives of auxiliary variables against the beta parameters",
   "auxiliary variables of the model of f",
   "auxiliary variables of the model of v")
  if (Ret$le <= length(coderr))
    lieuerr <- coderr[Ret$le]
  else
    lieuerr <- "the model"
  stop(paste("\nError when calculating \n",
              lieuerr,"\n on observation",Ret$ie,
              "\n No valid returned value\n"))
  }
# Transform df into a matrix
# --------------------------

if(all(Ret$df ==0)) Ret$df  <- NA
else Ret$df <- matrix(Ret$df, ncol=pbase, byrow=T,
          dimnames=list(names(x), names(theta) ))

if(all(Ret$f==0)) Ret$f  <- NA
else names(Ret$f) <- names(x)

return(list(response=Ret$f, d.resp=Ret$df))
}
# --------------- End of calibffnls2 ---------------------------------------

# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# calibf1nls2: call the program calcinv <-  that evaluates the inverse model
# on given values of the parameters and of ord
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
calibf1nls2 <- function(ord, theta, pbisabs, pbisvar)
{
nbl <- length(ord)
nbtheta <- length(theta)
nbgamf <- length(pbisabs)
nbgamv <- length(pbisvar)

# les sorties de calcinv:
abs <- vector(mode="double", length=nbl)
dabsdp <- vector(mode="double", length=(nbl * nbtheta))
dabsdo <- vector(mode="double", length=nbl) # derivees de l'abs par rapport a l'ord
varord <- vector(mode="double", length=nbl) # variance de l'ordonnee
le <- 0
ie <- 0

Ret <- .C("calcinvnls2",
   nbp=as.integer(nbtheta),
   nbga=as.integer(nbgamf),
   nbl=as.integer(nbl),
   nbgv=as.integer(nbgamv),
   p=as.double(theta),
   ga=as.double(pbisabs),
   ord=as.double(ord),
   gv=as.double(pbisvar),
   abs=as.double(abs),
   dabsdp=as.double(dabsdp),
   dabsdo=as.double(dabsdo),
   varord=as.double(varord),
   le=as.integer(le), ie=as.integer(ie))
	
if(Ret$le !=0)
  {
  # Treatment of error:
  coderr <- c(
   "the function inverse",
   "the function v",
   "the derivatives of the inverse",
   "the derivatives of v with respect to the theta parameters",
   "the derivatives of v with respect to the beta parameters",
   "auxiliary variables",
   "the derivatives of the auxiliary variables with respect to the theta parameters",
   "the derivatives of the auxiliary variables with respect to the beta parameters",
   "auxiliary variables the model inverse",
   "auxiliary variables of the model of v")
  if (Ret$le <= length(coderr))
    lieuerr <- coderr[Ret$le]
  else
    lieuerr <- "the inverse model"
  stop(paste("\nError when calculating \n",
              lieuerr,"\n on observation",Ret$ie,
              "\n No valid returned value\n"))
  }
# Transform dinv into a matrix
# --------------------------
labelrow <- names(ord)

if(is.null(Ret$dabsdp) || all(is.na(Ret$dabsdp)) || all(Ret$dabsdp ==0)) Ret$dabsdp  <- NA
else Ret$dabsdp <- matrix(Ret$dabsdp, ncol=nbtheta, byrow=T,
          dimnames=list(labelrow, names(theta)))

if(is.null(Ret$abs) || all(is.na(Ret$abs)) || all(Ret$abs==0)) Ret$abs  <- NA
else names(Ret$abs) <- labelrow

if(is.null(Ret$dabsdo) || all(is.na(Ret$dabsdo)) || all(Ret$dabsdo==0)) Ret$dabsdo  <- NA
else names(Ret$dabsdo) <- labelrow
if(is.null(Ret$varord) || all(is.na(Ret$varord)) || all(Ret$varord==0)) Ret$varord  <- NA
else names(Ret$varord) <- labelrow

return(list(inv=Ret$abs, d.inv.p=Ret$dabsdp, d.inv.o=Ret$dabsdo, variance=Ret$varord))
}
# --------------fin de calibf1nls2 --------------------------

	
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# cxChapnls2:			
# Quand la variance n'est pas constante, il faut calculer xchapeau
# i.e rappeler nls2 en rajoutant les points ord, en mettant x parmi
# les parametres et en modifiant le calcul des equN
# et recuperer -2logVchapeau	
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
cxChapnls2<-function
	(nls2.object, data, n, m, ord, x.bounds, zchapeau, 
	 R.filex)
  {
  # --------------------------------------------------------------------
  # On rajoute (zchapeau, ord) en fin des donnees:	
  # Construction du nouveau data frame:	
  #----------------------------------
  # Attention: si le 1ier point qu'on rajoute est egal au 
  # dernier du nls2.object$data,
  # il sera considere comme une repet et ca fera une erreur

   if (zchapeau == data[[nls2.object$X.names]][n])
      xval <- zchapeau + .Machine$double.eps
   else
     xval <- zchapeau
   		
   x <- c(data[[nls2.object$X.names]],  rep(xval,m))
	   
   data2  <- data.frame(matrix(c(
       x , # la variable explicative
      data[[nls2.object$response.name]],ord),         # la reponse          
    ncol=2, 
    dimnames=list(NULL,c(nls2.object$X.names, nls2.object$response.name))))

  # S'il y a des poids, on rajoute des poids=1 aux nouveaux points:
   if (!is.null(data$weights)) data2$weights <- c(data$weights, rep(1,m))
  # --------------------------------------------------------------------
  # On rajoute dans le model, un parametre:	 c'est le xchapeau a estimer
  # qu'on met en fin des theta.
  # Il est initialise a zchapeau.
  # --------------------------------------------------------------------
  model2 <- nls2.object$model
  model2$file  <- R.filex
  stat.ctx2 <-	nls2.object$stat.ctx
  nparam <-  length(stat.ctx2$theta.start)	
  stat.ctx2$theta.start <- c(  stat.ctx2$theta.start, zchapeau)
  #  S'il y a des contraintes sur les theta, il faut en rajouter une
  if (!is.null(model2$eqp.theta))
    model2$eqp.theta <- c(model2$eqp.theta, (max(model2$eqp.theta)+1))
  if (!is.null(model2$eq.theta))
    model2$eq.theta <- c(model2$eq.theta, NaN)
# rajouter une contrainte sur le x0:
# il doit etre compris dans x.bounds	
  if (!is.null(model2$inf.theta))
    model2$inf.theta <- c(model2$inf.theta,x.bounds[1]) 
  else
    model2$inf.theta <- c(rep(NaN,nparam),x.bounds[1])		
  if (!is.null(model2$sup.theta))
    model2$sup.theta <- c(model2$sup.theta, x.bounds[2]) 
  else
    model2$sup.theta <- c(rep(NaN,nparam),x.bounds[2])

  # --------------------------------------------------------------------
  #  	Appel de nls2:
  # --------------------------------------------------------------------

  # 	stat.crit.code="MYOWN" bien que ce soit en fait CLv mais
  # mettre ceci pour que les equn soient calculees a chaque iter	
 Ret  <- 
        nls2(data2, model2, stat.ctx2,
       method="MYOWN",
       	num.ctx=list(nh=2,effic=T,W.type="SYM",stat.crit.code="MYOWN"),
 	control=list(freq=0))
# on ne tolere que OK
  if (is.null(Ret$code) || (Ret$code !=0))
    {
    cat("Problem code", Ret$code,
	"in nls2 when calculating `loghat': See the warnings\n")
	return(NULL)
    }
		
  # On retourne les nouveaux theta+ x estimes:	
  return(list(theta=Ret$theta, logchapeau=Ret$loglik))
  }
# ------------ fin cxChapnls2 ------------------------------------------

# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# treeBnls2
# Cree eventuellement les arbres de calcul de f et f-1 
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
treeBnls2 <- function(nls2.object, RetVerif, file, R.grid)
{

pload <-  as.integer(0)
inv.pload <-  .C("invloadnls2",pload=pload)$pload
model.pload <- .C("ploadnls2",pload=pload)$pload

if ((inv.pload ==0) && (RetVerif$RetInv$YaSubr==0))
  stop("\nThe inverse model description file doesn't include expressions: you must provide the inverse model evaluation programs\n")
if ((model.pload ==0) && (RetVerif$RetModel$YaSubr==0))
  stop("\nThe model description file doesn't include expressions: you must provide the model evaluation programs\n")
if ((model.pload ==0) && 
	(R.grid>0) && (nls2.object$model$vari.type !="CST"))
  stop("\nThe program that calculates the model must be previously loaded by 'loadnls2' when the variance is not constant\n")
	
	
# Eventuellement, creer les arbres de calcul de f de f-1
# ------------------------------------------------------
if (model.pload==0)
  {
  #modele non charge: evaluation par arbre de f
  if (is.character(nls2.object$model))
    {		
    analDernls2(nls2.object$model) # creation des arbres de f
    }
  else
    {		
    analDernls2(nls2.object$model$file) # creation des arbres de f
    }	
  }

if (inv.pload==0)
  {
  # evaluation par arbre de f-1:
  # creer les arbres de calcul de f-1
  crInvnls2(file)
  }
invisible()
}
# ------------------fin de treeBnls2 -----------------------------------------------		  

# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# xBoundsnls2
# initialise x.bounds si besoin
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
"xBoundsnls2"<-
function(nls2.object, data, ord, x.bounds)
{
# Determination des valeurs par defaut de x.bounds:
# -------------------------------------------------
# on determine les absices telles que toutes les reponses sur les repet
# soient immediatement  inf et 	immediatement sup a chacune des ordonnes
# a calibrer
# On trie les X et les Y selon l'ordre des X
# comme la courbe est monotone, les moyennes des Y par repet
# seront croissantes ou decroissantes
ordre <- order(data[[nls2.object$X.names]])
yy <- data[[nls2.object$response.name]][ordre]
xx <- data[[nls2.object$X.names]][ordre]
ni <- rep(nls2.object$replications, nls2.object$replications)[ordre]
n <- length(ordre)
if (yy[1] < yy[n]) croit <- T else croit <- F
x1 <- xx[1];  x2 <- xx[n];
ipred <- i <- 1	
while (i<=n)
  {
  if(all(yy[ipred:(ipred+ni[i]-1)] <= rep(ord, length=ni[i])))
    {
    if (croit)
      {
      x1 <- xx[ipred]
      }
    else
      {
      x2 <- xx[ipred]
      break
      }
    }

  if(all(yy[ipred:(ipred+ni[i]-1)] >= rep(ord, length=ni[i])))
    {
    if (croit)
      {
      x2 <- xx[ipred]
      break
      }
    else
      {
      x1 <- xx[ipred]
      }
    }
  ipred <-  ipred+ni[ipred]
  i <- i+ni[i]		
  } # fin du for
    
if(is.na(x.bounds[1])) x.bounds[1] <- x1
if(is.na(x.bounds[2])) x.bounds[2] <- x2
warning(paste("The research interval is set to [", 
  x.bounds[1], ",", x.bounds[2], "]\n", sep=""))
return(x.bounds)
}
# ------------------fin de  xBoundsnls2-----------------------------------------------		
	
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# calcIsnls2:		
# Calcul de S confidence interval:	
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
calcIsnls2 <- function(nls2.object, RetVerif, x.bounds, conf.bounds,
                     cc, m, Schapeau,Zmoy, fl1, fl2, df.zchapeau)
{
  retour <- list(S.message=NULL, S.conf.int=c(NA,NA), 
	S.x=c(NA,NA),S.y=c(NA,NA))
  names(retour$S.conf.int) <- c("lower", "upper")
  # Calcul du denominateur D de S:
  # -----------------------------
  # df.zchapeau:  derivees de f par rapport aux theta en zchapeau
  # Remarque: gamma-1= (n * as.var)/sigma2 
  # D= sqrt(Schapeau) * sqrt( 1/m + (df.zchapeau * as.var * t(df.zchapeau)))
  # Les nombres dont on prend les sqrt sont  toujours >0
  # +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  E <-  	cc$as.var / nls2.object$sigma2
  D <-  sqrt(Schapeau) * sqrt( 1/m + (df.zchapeau %*% E %*% t(df.zchapeau)))
  
  # calcul du terme constant dans l'expression de S:
  # -----------------------------------------------
  A <-  sqrt(cc$n + m)/D
  
  # calcul de l'intervalle de confiance Is:
  # ---------------------------------------
  # Smin, Smax: mini et maxi des valeurs de S sur l'intervalle x.bounds
	
  retour$S.x.bounds <- c( ( A * (Zmoy - fl1)), (A * (Zmoy - fl2)))
  if (retour$S.x.bounds[1] > retour$S.x.bounds[2])
    {
    # S est croissante
    S.increasing <- T		
    Smax <- retour$S.x.bounds[1]
    xmax <- x.bounds[1]
    Smin <- retour$S.x.bounds[2]
    xmin <- x.bounds[2]
    }
  else
    {
    S.increasing <- F		
    Smin <- retour$S.x.bounds[1]
    xmin <- x.bounds[1]
    Smax <- retour$S.x.bounds[2]
    xmax <- x.bounds[2]
    }
  
  
  # Verification que [conf.bounds[1], conf.bounds[2]]
  #  est inclus dans [Smin, Smax]
  # --------------------------------------------
  if ((Smax < conf.bounds[1]) || (Smin > conf.bounds[2]))
    {
    warning(paste(
  "The S-function in the interval x.bounds varies from\n",
   Smin, "to", Smax, 
  "\nThese values are not compatible with the conf.bounds values:\n",
   conf.bounds[1], conf.bounds[2],"\n"))
   retour$S.conf.int <- NULL
   retour$S.message <- "The values of the S-function are not included in conf.bounds."
    }
  else
    {
    # Cas ou conf.bounds[2] est > Smax ou conf.bounds[1] est <Smin:
    # ------------------------------------------------------------
    # pas besoin de calculer les valeurs des bornes de Is
    if (conf.bounds[2] >= Smax)
      {	
      retour$S.conf.int[2] <-   xmax
      retour$S.message <- paste(retour$S.message,
	"The upper bound of S.conf.int is set to",
	   xmax,"but it may be greater than this value.")
      }	
		
    if (conf.bounds[1] <= Smin)
      {	
      retour$S.conf.int[1] <-   xmin
      retour$S.message <- paste(retour$S.message,
	"The lower bound of S.conf.int is set to",
	   xmin,"but it may be less than this value.")
      warning(retour$S.message)
      }		
  
    if (is.na(retour$S.conf.int[2]))
      {
      retour$S.y[2] <-  	Zmoy -(conf.bounds[2]/A)
      retour$S.conf.int[2] <- retour$S.x[2] <- 
                   calibf1nls2(ord=retour$S.y[2],
                              theta=RetVerif$RetInv$theta, 
                              pbisabs=RetVerif$RetInv$pbisabs, 
                              pbisvar=RetVerif$RetInv$pbisvar)$inv
      }
    if (is.na(retour$S.conf.int[1]))
      {
      retour$S.y[1] <- 	Zmoy -(conf.bounds[1]/A)
      retour$S.conf.int[1] <- retour$S.x[1] <- 
                  calibf1nls2(ord=retour$S.y[1],
                              theta=RetVerif$RetInv$theta, 
                              pbisabs=RetVerif$RetInv$pbisabs, 
                              pbisvar=RetVerif$RetInv$pbisvar)$inv 
       }
	
    # verif que les bornes de iS sont dans l'intervalle x.bounds
    if ( (retour$S.conf.int[1] < x.bounds[1]) ||
	 (retour$S.conf.int[2] < x.bounds[1]))
	{
	# l'intervalle de confiance est inf a celui de recherche
	retour$S.conf.int <- NULL
        retour$S.message <- paste(retour$S.message,
	  "No S interval found in [",
	    x.bounds[1], ",", x.bounds[2],
	    "]. Probably, it is at the left of x.bounds", sep="")
	}
    if ( (retour$S.conf.int[1] > x.bounds[2]) ||
	 (retour$S.conf.int[2] > x.bounds[2]))
	{
	# l'intervalle de confiance est sup a celui de recherche
	retour$S.conf.int <- NULL
        retour$S.message <- paste(retour$S.message,
	"No S interval found in [",
	    x.bounds[1], ",", x.bounds[2],
	    "]. Probably, it is at the right of x.bounds.", sep="")
	}
	
			
    retour$S.conf.int <- sort(retour$S.conf.int)
    }
  if (!is.null(retour$S.message))	
	warning(retour$S.message)
  return(retour)
}		  
# ------------------ fin de calcIsnls2 ----------------------------------
	
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# calcIrnls2
# Calcul de l'intervalle R
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
calcIrnls2 <- function(nls2.object, data, ord, x.bounds, conf.bounds,
                     R.grid, R.max.size,
	             R.nsplit, R.nx,
                     cc, m, Schapeau,zchapeau, 
	              Zmoy)
{
# Initialiser la structure retournee:	
# ----------------------------------
retour <- list(R.conf.set=NULL, R.conf.int=NULL, R.x=NULL, R.xhat=NULL,
	     R.values=NULL, R.message=NULL)


# Initialiser les entrees de nls2:	
# On va refaire des estimations a partir des dernieres valeurs estimees des param:
# -------------------------------------------------------------------------------
nls2.object$stat.ctx$theta.start <- cc$theta
if (cc$qbase >0) nls2.object$stat.ctx$beta.start <- cc$beta

# quand sigma2 connu, le remettre dans stat.ctx, car nls2 ne le
# remet pas	
if (nls2.object$stat.ctx$sigma2.type=="KNOWN")
  nls2.object$stat.ctx$sigma2 <- nls2.object$sigma2
	
if (nls2.object$stat.ctx$mu.type=="KNOWN")
	{
	# on fixe les mu du point rajoute a 0
	nls2.object$stat.ctx$mu3  <-  c(nls2.object$mu3,0)
	nls2.object$stat.ctx$mu4  <-  c(nls2.object$mu4,0)
	}

logchapeau <- 0		
# Quand la variance est non constante, il faut calculer logchapeau:	
# ----------------------------------------------------------------		
if (nls2.object$model$vari.type != "CST")
  {
  # creer un fichier temporaire pour contenir la
  # description du modele de f avec x en paresp supplementaire
  R.filex <- tempfile()
  on.exit(unlink(R.filex))	
  code <- 0	
  if (.C("genxnls2", as.character(nls2.object$model$file),
	             as.character(R.filex),
	             code=as.integer(code))$code !=0)
    stop("Problem for generating a new description model file\n")
  		
  Ret <- cxChapnls2(nls2.object, data, cc$n, m, ord, x.bounds, zchapeau,
	 R.filex)
  if (is.null(Ret)) return(Ret)
  retour$R.xhat <- Ret$theta[length(Ret$theta)]
  # maj des theta estimes origine:	
  cc$theta <- Ret$theta[1:length(cc$theta)]
  logchapeau <- Ret$logchapeau
  } # fin variance non cst		

# On va refaire l'estimation en rajoutant chaque point de la grille
# avec la methode d'estimation initiale.
# Construction du nouveau data frame:	
#----------------------------------			
xval <- x.bounds[1]
increment <-  (x.bounds[2]-x.bounds[1]) /(R.grid	-1)
		
# Attention: si le 1ier point qu'on rajoute est egal au dernier du nls2.object$data,
# il sera considere comme une repet et ca fera une erreur
if (xval == data[[nls2.object$X.names]][cc$n])
	{
	 xval <- xval + .Machine$double.eps
	 increment <-  (x.bounds[2]-xval) /(R.grid-1)
	}
x <- c(data[[nls2.object$X.names]],  rep(xval,m))

data2 <- data.frame(matrix(c(
    x , # la variable explicative
   data[[nls2.object$response.name]],ord),         # la reponse          
 ncol=2, 
 dimnames=list(NULL,c(nls2.object$X.names, nls2.object$response.name))))
# S'il y a des poids, on rajoute des poids=1 aux nouveaux points:
if (!is.null(data$weights)) data2$weights <- c(data$weights, rep(1,m))
	
# On calcule au 1ier point pour initier le puss:			
#----------------------------------			
Ret <-  nls2(data2, nls2.object$model, nls2.object$stat.ctx,
      method=nls2.object$method, control=list(freq=0),renls2=T)

		
if ( is.null(Ret) )
  stop(paste(
"Fatal error when estimating the parameters at the first point of ",
" the research grid of R.conf.set.\n",
"See the warnings\n"))

if (is.null(Ret$residuals) || all(is.na(Ret$residuals)))
  stop(paste(
"Problem to calculate the residuals when adding the first point of the research grid for R:\n",
"See the warnings\n"))

# Erreur non fatale si Ret$code !=0 
if (is.null(Ret$code) || ( Ret$code !=0))
  TousCodes  <- 1
else 
  TousCodes <- 0	
	
# Appel du programme qui fait les calculs de R:
# --------------------------------------------	
travt <-  rep(0,cc$pbase)
code <- cestint <- nsol <- nx <- 0
points <- rep(0,R.max.size)
R.x  <-  R.values <-  rep(0, R.nx)
RetCalc <- .C("calcRnls2",
	as.double(xval), as.double(x.bounds[2]),
        as.integer(cc$n), as.integer(m),
        as.integer(cc$pbase),
	as.integer(cc$nbgf),
        as.integer(R.grid), as.integer(R.max.size), as.integer(R.nsplit),
	as.integer(R.nx), as.double(increment),
        as.double(Schapeau), as.double(logchapeau),
        as.double(Zmoy),
        as.double(conf.bounds), as.double(ord), 
        as.double(Ret$residuals),as.double(cc$theta), 
	as.double(cc$gamf),
	as.double(travt),
        nsol= as.integer(nsol), 
	nx=as.integer(nx),
	points=as.double(points),
	R.x=as.double(R.x), R.values=as.double(R.values),
	cestint=as.integer(cestint),
	code=as.integer(code),
	TousCodes=as.integer(TousCodes)
	)
# nsol:	 nbre de points solution
# nx:	         nbre de points de calcul

	
if(RetCalc$code !=0)
  {
  retour$R.message <- 
"Fatal error when estimating the parameters at a point of the research grid of R.conf.set. See the warnings."	
  }
	
# recuperer le code des val. manquantes
codeNa  <-  as.double(0)
codeNa  <-  .C("recupCodeNa", as.double(codeNa))[[1]]
retour$R.values[retour$R.values == codeNa]  <- NA
	
	
if(RetCalc$TousCodes !=0)
  {
  retour$R.message <- 
"Problem for estimating the parameters for at least one point of the research grid of R.conf.set. See the warnings."	
  }
	
if(RetCalc$nsol==0)
  {
  retour$R.message <- paste(retour$R.message,"No points in R.conf.set")
  }
else
  {
  if ((R.max.size < RetCalc$nsol) &&  (RetCalc$cestint!=1))
    {
    retour$R.message <- paste(retour$R.message,
     "R.conf.set does not contain all the",
      RetCalc$nsol,"solutions (R.max.size is too small).",
     "It contains the", ( R.max.size-1), "first ones and the last.")	
    }
  retour$R.conf.set  <- RetCalc$points[seq(1, min(R.max.size, RetCalc$nsol))]

  if (RetCalc$cestint==1)
    {	 # c'est un intervalle
    retour$R.conf.int <- c(min(retour$R.conf.set), max(retour$R.conf.set))
    retour$R.conf.set <- NULL
    if ( (retour$R.conf.int[1]-.Machine$double.eps) <= as.double(xval))
      {
      retour$R.message <- paste(retour$R.message,
         "The lower bound of R.conf.int is set to", xval,
         "but it may be less than this value.")
      }							
    if ( (retour$R.conf.int[2]+.Machine$double.eps) >= x.bounds[2])
      {
      retour$R.message <- paste(retour$R.message,
         "The upper bound of R.conf.int is set to", retour$R.conf.int[2]   ,
         "but it may be greater than this value.")
      }							
    }		
  }	# fin du else  (RetCalc$nsol = 0)

if (R.nx>0)
  {
  if (R.nx < RetCalc$nx)
    {
    retour$R.message <- paste(retour$R.message,
	"R.x and R.values do not contain all the",
	RetCalc$nx,
	"points where R has been calculated (R.nx is too small).",
	"They contain only the first ones.")
    }
  retour$R.x  <-  RetCalc$R.x[seq(1 , min(R.nx, RetCalc$nx))]
  retour$R.values  <-  RetCalc$R.values[seq(1 , min(R.nx, RetCalc$nx))]
  # Trier les valeurs en cas de redecoupe:
  if (R.nsplit >0)
    {		
    ord <- order(retour$R.x)
    retour$R.x <- retour$R.x[ord]
    retour$R.values <- retour$R.values[ord]
    }
  # Remplacer les points ou on n'a pas pu calculer
  retour$R.values[retour$R.values==1.e+5] <- NaN		
  retour$R.values[retour$R.values==-1.e+5] <- NaN		
  } # fin R.nx >0


	
		
if (!is.null(retour$R.message))
	    warning(retour$R.message)
return(retour)
}

# --------------- fin de  calcIrnls2 ----------------------------------------


# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# calib.nls2
# calibration function
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
calib.nls2 <- function(nls2.object,  file, ord,
                conf.level=0.95,
                conf.bounds=c(qnorm((1-conf.level)/2, 0,1), 
                              qnorm((1+conf.level)/2, 0,1)),
                x.bounds=c(NA,NA), check.points=20,
	        R.grid=20, 
	        R.max.size=20,
	        R.nsplit=0,
	        R.nx=(R.grid+1)**(R.nsplit+1))

{
# R.grid:	nbre de points dans chaque decoupe
# R.max.size:	nbre maxi de points solution retournes
# R.nsplit:	nbre de fois qu'on decoupe un intervalle
#	        quand un point appartient a R et pas le suivant ou l'inverse
# R.nx:	nbre maxi de points retournes dans R.x et R.values
		
# Extraire les derniers resultats obtenus dans le nls2.object:
# -----------------------------------------------------------
cc <- examCalibnls2(nls2.object)

# Verifications:
# --------------
# On les fait apres examCalib car on a besoin de cc$theta dans les verif
# Remarque: Les parametres bis sont toujours les memes dans le modele de f-1 et 
# dans le modele de f:
RetVerif <-  verifCalibnls2(nls2.object, file, ord,
                cc$theta,
                conf.level,
                conf.bounds,
                x.bounds, check.points,
	        R.grid, 
	        R.max.size,
	        R.nsplit,
	        R.nx)

if (is.null(RetVerif)||
	( (!RetVerif$Scalc) && (RetVerif$R.grid==0))) return(NULL)

# Initialisation de la structure retournee:
# ----------------------------------------
retour <- list(call=match.call(), ord=ord)
retour$conf.bounds <- conf.bounds
	
# Initialisation de GNLControle et de la trace
# ---------------------------------------------
check <-  options()$check
warn <-  options()$warn
.C("initcrolenls2",  as.integer(check), as.integer(warn))
.C("crTracenls2")

# Calcul de Zmoy=moyenne des Z:
# ----------------------------
Zmoy  <-  mean(ord) 

# Determination de m:
# -------------------
m <- length(ord)

# Creer si besoin les arbres de calcul
# ------------------------------------
treeBnls2(nls2.object, RetVerif, file, R.grid)
	
# Rechercher les data:
# --------------------
data  <- nls2.object$data.sv
if (is.null(data))
  data <- plsplitoutnls2(nls2.object)


# Calculer les x.bounds par defaut
# ------------------------------
if (is.na(x.bounds[1]) || is.na(x.bounds[2]))
  {
  Retbounds <-  	xBoundsnls2(nls2.object, data, ord, x.bounds )
  retour$x.bounds <- Retbounds
  }
else
  retour$x.bounds <- x.bounds
		
	

# Calcul des bornes admises pour zchapeau:
# ---------------------------------------
fl1 <- calibffnls2(retour$x.bounds[1], cc$theta, cc$gamf)$response 
fl2 <-  calibffnls2(retour$x.bounds[2], cc$theta, cc$gamf)$response 

if (check.points >0)
  {
  # Verification que la fonction de regression est monotone:
  # -------------------------------------------------------
  pasmonot <- code <- 0
  trav <-  rep(0,cc$pbase)
  RetMonot <- .C("verifmonotnls2",
	as.integer(cc$pbase),
	as.integer(cc$nbgf),
	as.integer(check.points),
	as.double(min(retour$x.bounds[1], data[[nls2.object$X.names]])),
	as.double(max(retour$x.bounds[2], data[[nls2.object$X.names]])),
	as.double(cc$theta), as.double(cc$gamf),
	as.double(trav),
	pasmonot=as.integer(pasmonot), code=as.integer(code))
  if (RetMonot$code !=0)
	{
	stop("Problem for calculating the model when checking the monotony\n")
	}
  if (RetMonot$pasmonot==1)
	{
	stop("The regression function is not monotonous\n")
	}
  } # fin verif monot

	
flmin <-  min(fl1,fl2)
flmax <-  max(fl1,fl2)
 

# Calcul de zchapeau:
# ------------------
if (Zmoy < flmin)
  {
  if(fl1<fl2)
    zchapeau <- retour$x.bounds[1]
  else
    zchapeau <- retour$x.bounds[2]
  }
		
if (Zmoy > flmax)
  {
  if(fl1<fl2)
    zchapeau <- retour$x.bounds[2]
  else
    zchapeau <- retour$x.bounds[1]
  }	

if ((Zmoy >= flmin) && (Zmoy <= flmax))
  {
  #  Zmoy est dans l'intervalle [fl1, fl2]
  # Calcul de finv= f-1 en ZMoy:
  zchapeau <- calibf1nls2(ord=Zmoy, theta=RetVerif$RetInv$theta, 
                            pbisabs=RetVerif$RetInv$pbisabs, 
                            pbisvar=RetVerif$RetInv$pbisvar)$inv
  if( (zchapeau < retour$x.bounds[1]) || (zchapeau > retour$x.bounds[2]))
    stop("Probably there is a mistake in the inverse model calculation: Verify the formal description or the program\n")		
	
  }
	
retour$x <- zchapeau
	
# Calcul de Schapeau(m,n):
# ----------------------
Schapeau <-  sum(cc$residuals*cc$residuals) + sum((ord-Zmoy)*(ord-Zmoy))

# Calcul de f pour les theta estimes en zchapeau:
# -----------------------------------------------	
f.zchapeau <-  calibffnls2(zchapeau, cc$theta, cc$gamf)
		
# Calcul de S confidence interval:	
# ---------------------------------
if (RetVerif$Scalc)
  {			
  Ret <- calcIsnls2(nls2.object, RetVerif, retour$x.bounds, conf.bounds,
                     cc, m, Schapeau,Zmoy, fl1, fl2, f.zchapeau$d.resp )
  retour$S.conf.int <-  Ret$S.conf.int
  retour$S.message <- Ret$S.message
  if (!all(is.na(Ret$S.x)))  retour$S.x <-  Ret$S.x[!is.na(Ret$S.x)]
  if (!all(is.na(Ret$S.y)))  retour$S.y <- Ret$S.y[!is.na(Ret$S.y)]
  retour$S.x.bounds <- Ret$S.x.bounds
		
  } # fin calcul de S interval
	  
# Calcul d'IR:
# ------------
if (RetVerif$R.grid >0)
  {
  if (is.null(data))
    {		
    # Rechercher les data:
    data  <- nls2.object$data.sv
    if (is.null(data))
      data <- plsplitoutnls2(nls2.object)
    }	
  #S'il y a des courbes, erreur:
  if (!is.null(data$curves)) 
    {
    warning("For calculating the R interval, the nls2.object must not include curves   No value returned for the R interval\n")
    retour$R.message <- paste(retour$message,
     "For calculating the R interval, the nls2.object must not include curves.")
    }
  else
    {
    Ret <- calcIrnls2(nls2.object, data, ord, retour$x.bounds, conf.bounds,
                       R.grid,  R.max.size,
	               R.nsplit, R.nx,
                       cc, m, Schapeau,
	               zchapeau, Zmoy)
	if (is.null(Ret)) return(NULL)
	
    retour$R.conf.set <-  Ret$R.conf.set
    retour$R.conf.int <-  Ret$R.conf.int
    retour$R.x  <-  Ret$R.x
    if (!is.null(Ret$R.xhat))
      retour$x  <-  Ret$R.xhat
#    names(retour$x) <-  "estimated.abscissa"
# R.xhat est l'estimation de x quand la vari !=CST
			
    retour$R.values  <-  Ret$R.values
    retour$R.message  <-  Ret$R.message
    on.exit(
      {
      # On detruit les structures internes a NL:
      delnls2()
      })

	
    } # fin de pas de courbes
  } # fin de R.grid >0


# J'enleve des sorties:
# -`"S.x.bounds"': values of the statistic S at points `x.bounds'.
# - `"S.y"' and `"S.x"': 
# values of the ordinate and corresponding values of "f-1"
# calculated to determine `S.conf.int'.
retour$S.x.bounds <- S.y <- S.x <- NULL
	
class(retour) <- "calibnls2"
print(retour)
return(retour)	
}
# ---------------------end calib.nls2 ----------------------------
# ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++



	
