/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- MODULE IDENTIFICATION ------------
| Name                 : ToMyOwn                   |
| Role                 : the programmes the user   |
| must provide when the method is MYOWN            |
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <stdio.h>

#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlglobal.h"
#include "nlmacros.h"
#include "errcodes.h"

#include "nldcl.h"
/*------------- Some explanations --------------------
To acces global variables:
It may be necessary to access variables in addition
to those accessible through the argument-lists.
Some are global variables:
see include/nldcl.h and include/nltypes.h
For example, Theta[0] contains the values of the
regression parameters for the first step.
Its "Act" component contains the values of the
active parameters and "Estim", the corresponding
values of the multiple parameters.
ResStat[0] contains the current results of the
first step.
For example, ResStat[0].FctSensib.DValf contains
the derivatives of f with respect to the multiple
parameters and ResStat[0].FctSensib.DVarYTheta,
ResStat[0].FctSensib.DVarYBeta contain the derivatives
of the variance of Y, with respect to the multiple
parameters Theta and Beta , respectively.
----------------------------------------------------*/

/* --------------------------------------------------------
  function: CalcTtV 
  called by: calcb
  computes the inverse of the variance of Y and its derivatives 
  with respect to the parameters phi and sigma2.
  The outputs are not matrices because all their values are identical,
  except for the first and last observations.
  The output arguments whose name ends with 00 are related to
  the first observation and those whose name ends with Ni the last.
--------------------------------------------------------------- */
TShortInt    CalcTtV(  
         Vinv0i,Vinv1i,Vinv00,Vinv0Ni,
         DVinvPhi0i,DVinvPhi1i,DVinvPhi00, DVinvPhi0Ni, DVinvPhi1Ni,
         DVinvSig0i, DVinvSig1i, DVinvSig00, DVinvSig0Ni, DVinvSig1Ni)

/* output arguments */
TDouble  *Vinv0i,*Vinv1i,*Vinv00,*Vinv0Ni;
TDouble  *DVinvPhi0i,*DVinvPhi1i,*DVinvPhi00, *DVinvPhi0Ni, *DVinvPhi1Ni;
TDouble  *DVinvSig0i, *DVinvSig1i, *DVinvSig00, *DVinvSig0Ni, *DVinvSig1Ni;

{
TDouble phi, sig, sig2, tpd;


phi=Beta[0].Estim.donnees[0];
sig=Beta[0].Estim.donnees[1];
if (sig ==0)
  {
  printf("Sig nul\n");
  exit(1);
  }

tpd=1+phi*phi;
sig2=sig*sig;

*Vinv0i=tpd/sig;
*DVinvPhi0i=(TDouble)2*phi/sig;
*Vinv1i=-phi/sig;
*DVinvPhi1i=-1/sig;
*DVinvSig0i=-tpd/sig2;
*DVinvSig1i=phi/sig2;

/* cas particulier des premiers et derniers elements */
*Vinv00=*Vinv0Ni=1/sig;
*DVinvPhi00=*DVinvPhi0Ni=*DVinvPhi1Ni=*DVinvSig1Ni=0;
*DVinvSig00=*DVinvSig0Ni=-1/sig2;

return(OK);
}

/*--------------- Function identification ----------
| Function name         : calcb                    |
| Role                  : Calculate B              |
| Input arguments       :                          |
|  NbRepet: number of replications of each         |
|          observation                             |
|  Valf: values of the regression function f       |
|  VarY: values of the variance of Y               |
|  S2: sum of squares of each observation          |
|  DValf: derivates of the function f with respect |
|     to the active parameters Theta               |
|  DVarYTheta: derivates of the variance of Y      |
|     with respect to the active parameters Theta  |
|     or matrix of dimensions equal to zero        |
|  DVarYBeta: derivates of the variance of Y       |
|     with respect to the active parameters Beta   |
|     or matrix of dimensions equal to zero        |
| Output arguments      :                          |
|  ValB: matrix B                                  |
|        dimension: NbZ*NbObs, (NbTheta+NbBeta)    | 
|    Already allocated                             |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :  CalcTtV                  |
| Calling functions    : CCovNu2, CCovNu3, PussIter|
|       Retablir                                   |
|  through the function pointer PCValB             |
--------------------------------------------------*/


/*--------------- Function definition ------------*/

TShortInt calcb(NbRepet, Valf, VarY, S2, DValf, DVarYTheta, DVarYBeta,
                 ValB)

/* input arguments */
TVectLong *NbRepet; /* dimension NbObs */
TVect *Valf, *VarY, *S2; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta */
TMat *DVarYBeta;  /* dimension NbObs, NbBeta */

/* output arguments */
TMat *ValB; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- End function definition ----*/

{
/* locals */
TShortInt i, a, NbObs, NbTheta, NbBeta;
/* pointeurs pour ameliorer la performance */
TDouble *valf ;
TDouble **valb, **Dvalf;

/* outputs of CalcTtV */
TDouble  Vinv0i,Vinv1i,Vinv00,Vinv0Ni;
TDouble  DVinvPhi0i,DVinvPhi1i,DVinvPhi00, DVinvPhi0Ni, DVinvPhi1Ni;
TDouble  DVinvSig0i, DVinvSig1i, DVinvSig00, DVinvSig0Ni, DVinvSig1Ni;

/* Write into the trace of the called programs */
ECRTRACE("calcb");

/* determination des dimensions */
NbObs = NbRepet->nbele;
NbTheta = DValf->nbcol;
NbBeta = DVarYBeta->nbcol;

/* affectation des pointeurs */
valf = Valf->donnees;
valb = ValB->donnees;
Dvalf =  DValf->donnees;

/* calcul de l'inverse de V et de ses derivees */
CalcTtV(
         &Vinv0i, &Vinv1i, &Vinv00, &Vinv0Ni,
         &DVinvPhi0i, &DVinvPhi1i, &DVinvPhi00, &DVinvPhi0Ni, &DVinvPhi1Ni,
         &DVinvSig0i, &DVinvSig1i, &DVinvSig00, &DVinvSig0Ni, &DVinvSig1Ni);

 
for(i = 0; i<NbObs; i++)
  {
  for(a  =  0; a < NbTheta; a++)
    {
    if((i>0)&&(i<(NbObs-1)))
       {
       valb[i][a] =-(TDouble)2*(Vinv0i*Dvalf[i][a]
           +Vinv1i*(Dvalf[i-1][a]+Dvalf[i+1][a]));
       } 
    if(i==0)
     {
      valb[0][a]=-(TDouble)2*(Vinv00*Dvalf[0][a]
         +Vinv1i*Dvalf[1][a]);
      }
    if(i==(NbObs-1))
     {
     valb[NbObs-1][a]=-(TDouble)2*(Vinv0Ni*Dvalf[NbObs-1][a]
           +Vinv1i*Dvalf[NbObs-2][a]);
     }
    } /* fin for(a  =  0; a < NbTheta; a++) */

/* calcul partie Beta */

    if((i>0)&&(i<(NbObs-1))) 
       {
       valb[i][NbTheta]=-(TDouble)2*(DVinvPhi0i*valf[i]
           +DVinvPhi1i*(valf[i-1]+valf[i+1]));
       valb[i][NbTheta+1]=-(TDouble)2*(DVinvSig0i*valf[i]
           +DVinvSig1i*(valf[i-1]+valf[i+1]));

       valb[i+NbObs][NbTheta]=DVinvPhi0i;
       valb[i+NbObs][NbTheta+1]=DVinvSig0i;
       valb[i+2*NbObs][NbTheta]=(TDouble)2*DVinvPhi1i;
       valb[i+2*NbObs][NbTheta+1]=(TDouble)2*DVinvSig1i;
       }
    if(i==0)
       {
       valb[i][NbTheta]=-(TDouble)2*DVinvPhi1i*valf[i+1];
       valb[i][NbTheta+1]=-(TDouble)2*(DVinvSig00*valf[i]
                   +DVinvSig1i*valf[i+1]);
       valb[i+NbObs][NbTheta]=DVinvPhi00;
       valb[i+NbObs][NbTheta+1]=DVinvSig00;
       valb[i+2*NbObs][NbTheta]=(TDouble)2*DVinvPhi1i;
       valb[i+2*NbObs][NbTheta+1]=(TDouble)2*DVinvSig1i;
       }
     if(i==(NbObs-1))
       {
       valb[i][NbTheta]=-(TDouble)2*(DVinvPhi1i*valf[i-1]);
       valb[i][NbTheta+1]=-(TDouble)2*(DVinvSig0Ni*valf[i]
                   +DVinvSig1i*valf[i-1]);
       valb[i+NbObs][NbTheta]=DVinvPhi0Ni;
       valb[i+NbObs][NbTheta+1]=DVinvSig0Ni;
       valb[i+2*NbObs][NbTheta]=(TDouble)2*DVinvPhi1Ni;
       valb[i+2*NbObs][NbTheta+1]=(TDouble)2*DVinvSig1Ni;
       }

  } /* fin boucle sur i */
for(i=0;i<3*NbObs-1;i++)
   {
   for(a=0;a<NbTheta+NbBeta;a++)
     {
     valb[i][a]=-valb[i][a];
     }
   }

return(OK);
}



/*--------------- Function identification ----------
| Function name         : calcd                    |
| Role                  : Calculate D              |
| Input arguments       :                          |
|  Valf: values of the regression function f       |
|  DValf: derivates of the function f with respect |
|     to the active parameters Theta               |
|  DVarYTheta: derivates of the variance of Y      |
|     with respect to the active parameters Theta  |
|     or matrix of dimensions equal to zero        |
|  DVarYBeta: derivates of the variance of Y       |
|     with respect to the active parameters Beta   |
|     or matrix of dimensions equal to zero        |
| Output arguments      :                          |
|  ValD: matrix D                                  |
|        dimension: NbZ*NbObs,(NbTheta+NbBeta)     | 
|    Already allocated                             |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :                           |
| Calling functions    : CCovNu2, CCovNu3, PussIter|
|       Retablir                                   |
|  through the function pointer PCValD             |
--------------------------------------------------*/

/*--------------- Function definition ------------*/

TShortInt calcd( Valf, DValf, DVarYTheta, DVarYBeta,
                 ValD)

/* input arguments */
TVect *Valf; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta */
TMat *DVarYBeta;  /* dimension NbObs, NbBeta */


/* output arguments */
TMat *ValD; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- End function definition ----*/

{
/* locals */
TShortInt NbObs, NbTheta, i, j;


/* pointeurs pour ameliorer la performance */
TDouble  *valf;
TDouble **vald, **Dvalf, **dvarybeta;

TDouble phi, factphi;

/* Write into the trace of the called programs */
ECRTRACE("calcd");


/* determination des dimensions */
NbObs = DValf->nblig;
NbTheta = DValf->nbcol;

/* affectation des pointeurs */
valf = Valf->donnees;
vald = ValD->donnees;
Dvalf =  DValf->donnees;
dvarybeta = ResStat[0].FctSensib.DVarYBeta.donnees;

phi = Beta[0].Estim.donnees[0];
if (phi ==0) {
  printf("phi nul\n");
  exit(1);
  }
factphi = (1.0 + phi*phi) / (2 * phi);

for (i = 0; i<NbObs; i++)
  {
  for (j = 0 ; j < NbTheta; j++)
    {
    vald[i][j] = Dvalf[i][j];
    vald[i+NbObs][j] = (TDouble)2 * valf[i] * Dvalf[i][j];
    if(i<(NbObs-1))
      vald[i+2*NbObs][j]=valf[i+1]*Dvalf[i][j]+
           valf[i]*Dvalf[i+1][j];
    }
    vald[i+NbObs][NbTheta] = dvarybeta[i][0];
    vald[i+NbObs][NbTheta+1] = dvarybeta[i][1];
    if(i<(NbObs-1)){
       vald[i+2*NbObs][NbTheta]= factphi * dvarybeta[i][0] ;
       vald[i+2*NbObs][NbTheta+1]= phi * dvarybeta[i][1] ;
     }
  }

return(OK);
}


/*--------------- Function identification ----------
| Function name         : calceta                  |
| Role                  : Calculate Eta            |
| Input arguments       :                          |
|  Valf: values of the regression function f       |
|  VarY: values of the variance of Y               |
| Output arguments      :                          |
|  ValEta: vector Eta                              |
|        dimension: NbZ*NbObs                      | 
|    Already allocated                             |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :                           |
| Calling functions    :  PussIter, Retablir       |
|  through the function pointer PCValEta           |
--------------------------------------------------*/

/*--------------- Function definition ------------*/

TShortInt calceta( Valf, VarY,
                 ValEta)

/* input arguments */
TVect *Valf; /* dimension NbObs */
TVect *VarY; /* dimension NbObs */

/* output arguments */
TVect *ValEta; /* dim: NbZ*NbObs */

/*--------------- End function definition ----*/

{
/* locals */
TShortInt NbObs, i;

/* pointeurs pour ameliorer la performance */
TDouble *valf,*vary, *valeta ;
TDouble phi;

/* Write into the trace of the called programs */
ECRTRACE("calceta");

/* determination des dimensions */
NbObs = Valf->nbele;

/* affectation des pointeurs */
valf = Valf->donnees;
vary = VarY->donnees;
valeta = ValEta->donnees;

phi = Beta[0].Estim.donnees[0];

for (i = 0; i<NbObs; i++)
  {
  valeta[i] = valf[i];
  valeta[i+NbObs] = (valf[i] * valf[i]) + vary[i];
  if(i<(NbObs-1))
    valeta[i+2*NbObs]=(valf[i]*valf[i+1])+ (phi * vary[i]);
  }

return(OK);
}



/*--------------- Function identification ----------
| Function name         : calcz                    |
| Role                  : Calculate the vector of  |
|  the sufficient statistics Z and calculate its   |
|  values                                          |
| Input arguments       :                          |
|  Y1 and Y2: pointers on the components Y1 and Y2 |
|    of StatDon                                    |
| Output arguments      :                          |
|  ValZ: the vector Z                              |
|    Must be allocated by this function            |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :                           |
| Functions called     :                           |
| Calling functions    : NLEtape                   |
|  through the function pointer PCValZ             |
--------------------------------------------------*/

/*--------------- Function definition ------------*/
TShortInt calcz(Y1,Y2,ValZ)

  /* input arguments */
  TVect *Y1,*Y2;

  /* output argument */
  TVect *ValZ;


/*--------------- End function definition ----*/
{
  /* locals */
  TShortInt i, NbObs;
  /* pointeurs pour ameliorer la performance */
  TDouble *valz, *y1, *y2;



/* Write into the trace of the called programs */
 ECRTRACE("calcz");


/* insert here the function body */

  /* Affectation des pointeurs */
  valz = ValZ->donnees;
  y1 = Y1->donnees;
  y2 = Y2->donnees;

  /* Affectation des valeurs */
  NbObs = Y1->nbele;
  for(i = 0; i < NbObs; i++)
    {
    valz[i] = y1[i];
    valz[i+NbObs] = y2[i];
    if ( i!=(NbObs-1) ) valz[i+2*NbObs]= y1[i]*y1[i+1];
    }


return(OK);
}


/*--------------- Function identification ----------
| Function name         : calcvarz                 |
| Role                  : Create the vector of the |
|  variance of Z and calculate its value           |
| Input arguments       :                          |
|  NbObsT: number of observations, replications    |
|   included                                       |
|  NbRepet: number of replications of each         |
|          observation                             |
|  PoidsT: weighting values affected on the        |
|   observations, replications included            |
|  ValY: observed values of the response           |
|  Ajustes: a structure that contains:             |
|      Valf: values of the regression function f   |
|      VarY: values of the variance of Y           |
| Input-output arguments :                         |
|  TypeMu: code for the way of calculating the     |
|    moments                                       |
|  Residus : the non-standardized residuals        |
|  Mu3 : moments of order 3                        |
|  Mu4 : moments of order 4                        |
| Output arguments      :                          |
|  VarZ: matrix that contains the variance of Z    |
|    Must be allocated by this function            |
|  Code:  OK (=0) or error code                    |
| Return value          : OK (=0) or error code    |
|  when the error is fatal for the remaining of the|
|  execution                                       |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :  <insert here the names   |
|    of the functions called>                      |
| Functions called     :                           |
| Calling functions    :  CBVarZBP                 |
|   through the function pointer PCValZ            |
--------------------------------------------------*/

/*--------------- Function definition ------------*/


TShortInt calcvarz( NbObsT, NbRepet, PoidsT, ValY, Ajustes,
              TypeMu, Residus, Mu3, Mu4,
              VarZ, Code)

/* input arguments */
TLongInt NbObsT;
TVectLong *NbRepet;
TVect *PoidsT, *ValY;
TAjustes *Ajustes;

/* input-output arguments */
TShortInt *TypeMu;
TVect *Residus;
TVect *Mu3, *Mu4;

/* output arguments*/
TMat *VarZ; /* dimension NbObs*NbZ, NbObs*NbZ  */
TShortInt *Code;

/*--------------- End function definition ----*/
{
  /* locals */
  TShortInt NbObs, Dim, i, j, ii;
  TChar Mess1[5], Mess2[50];


/* pointeurs sur des elements de structure pour ameliorer la performance */
  TLongInt *nbrepet;
  TDouble **varz;
  TDouble *vary, *valf, *mu3, *mu4;

/* Write into the trace of the called programs */
ECRTRACE("calcvarz");

/* insert here the function body */
  *Code = OK;
  NbObs = NbRepet->nbele;
  Dim = NbObs * 2;

  /* calculer les Mu */
  if (GNLControle.Voulu.Mu==FAUX)
    {
    /* ils n'ont pas encore ete calcules */
   APPEL(CMu( NbObsT, NbRepet, PoidsT, ValY, &(Ajustes->Valf), &(Ajustes->VarY),
              TypeMu, Residus,
              Mu3, Mu4));
    }
 /* affectation des pointeurs pour ameliorer la performance */
  varz = VarZ->donnees;
  vary = Ajustes->VarY.donnees;
  valf = Ajustes->Valf.donnees;
  nbrepet = NbRepet->donnees;
  mu3 = Mu3->donnees;
  mu4 = Mu4->donnees;

  /* initialisation a zero */
  for (i=0; i<Dim; i++)
    {
    for(j=0; j<Dim; j++)
      {
      varz[i][j] = (TDouble)ZERO;
      }
    }

  for (i=0; i< NbObs; i++)
    {
    ii = NbObs+i;
    varz[i][i] = vary[i] / nbrepet[i];
    varz[i][ii] =
       (mu3[i] + ((TDouble)2 * valf[i]  * vary[i]))
       / nbrepet[i];
    varz[ii][i] = varz[i][ii];
    varz[ii][ii] =
       (mu4[i] + ((TDouble)4 * valf[i] * valf[i] * vary[i])
       - (vary[i] * vary[i])
       + ((TDouble)4 * valf[i] * mu3[i]))
       / nbrepet[i];
    if (varz[ii][ii] <= (TDouble)ZERO)
      {
      /* theoriquement ca n'est pas possible,
      mais ca peut le devenir, suite a des arrondis de calcul */
      sprintf(Mess1,"%d", (ii+1));
      sprintf(Mess2, "%10.20e", varz[ii][ii]);
/*      printf("CVarZmv: le %s-ieme terme de la diagonale de VarZ est <=0 (= %!
        Mess1, Mess2);
*/
      NLWARNING((WARVARZ1,3,"CVarZmv",Mess1, Mess2,WNUMER));
      /* dans ce cas, les calculs qui incluent la variance
      n'ont pas de sens */
      *Code = WARVARZ1;
      return(OK);
      } /* fin du cas <=0 */

    }

return(OK);
}

/*--------------- Function identification ----------
| Function name         : myinv                    |
| Role                  : invert the matrix W      |
|  when it is no SYM neither SYMB                  |
| Input arguments       :                          |
|  ValW: matrix W                                  |
| Output arguments      :                          |
|  ValWInv: matrix of same dimension than ValW     |
|    Already allocated                             |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references -----------------
| Functions called     :  <insert here the names   |
|    of the functions called>                      |
| Calling functions    :  CArret                   |
--------------------------------------------------*/


/*--------------- Function definition ------------*/

TShortInt myinv(ValW, ValWInv)

/* input arguments */
TMat *ValW;

/* output arguments*/
TMat *ValWInv;

/*--------------- End function definition ----*/

{

/* insert here the function body */

return(OK);
}

/*--------------- Function identification ----------
| Function name         : calcc                    |
| Role                  : Calculate the fitting    |
|  criterion: to provide only if the way of        |
|  calculating it is your own                      |
|  You can inspire from the programs of file       |
|  Critere.c in the source-directory of NL         |
| Input arguments       :                          |
|  Sigma2:  the estimated value of square sigma    |
|  NbObsT:  number of observations, replications   |
|           included                               |
|  NbRepet: number of replications of each         |
|          observation                             |
|  Y1: the sum of the response values on the       |
|     replications weighted by the number of       |
|      replications                                |
|  Y2: the sum of the squared values of the        |
|      response on the replications weighted by the|
|      number of replications                      |
|  S2: the intra-replications variance.            |
|  Valf: values of the regression function f       |
|  VarY: values of the variance of Y               |
| Output arguments      :                          |
|  Crit: the fitting criterion                     |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :  <insert here the names   |
|    of the functions called>                      |
| Calling functions    :  PussIter                 |
--------------------------------------------------*/


/*--------------- Function definition ------------*/

TShortInt calcc(Sigma2, NbObsT, NbRepet, Y1, Y2, S2, Valf, VarY,
                Crit)

/* input arguments */
TDouble Sigma2;
TLongInt NbObsT;
TVectLong *NbRepet; /* dimension NbObs */
TVect *Y1, *Y2, *S2, *Valf, *VarY; /* dimension NbObs */

/* output arguments */
TDouble *Crit;

/*--------------- End function definition ----*/

{
extern TDouble log();

/* locals */
TShortInt i, n;


/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *valf, *y1;
TDouble phi,sig;

/* If wanted, write into the trace of the called programs */
ECRTRACE("calcc");

/* insert here the function body */
n = NbRepet->nbele;
valf = Valf->donnees;
y1 = Y1->donnees;
phi= Beta[0].Estim.donnees[0];
sig= Beta[0].Estim.donnees[1];


*Crit=(TDouble)ZERO;
*Crit = (1-phi*phi) * (y1[0]-valf[0])*(y1[0]-valf[0]);

for (i = 1; i < n ; i++ )
  {
  *Crit = *Crit + (y1[i]-phi*y1[i-1]-valf[i]+phi*valf[i-1])
      * (y1[i]-phi*y1[i-1]-valf[i]+phi*valf[i-1]);
  }

  *Crit = log(((TDouble)2 * (TDouble)M_PI)) + log(sig)
    - (log(1-phi*phi) / (TDouble)NbObsT) + (*Crit / ((TDouble)NbObsT*sig));


/* retour */

return(OK);
}
