
/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- MODULE IDENTIFICATION ------------
| Name                 : EquNCalib                 |
| Role                 : the programs              |
| for calibration analysis                         |
| On suppose que x est un parametre supp de la     |
| variance rajoute au cas MVTB                     |
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <stdio.h>
#include <errno.h>

#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlglobal.h"
#include "nlmacros.h"
#include "errcodes.h"

#include "nldcl.h"

/*--------------- FONCTIONS EXTERNES -------------*/
TShortInt CreerMatC( TShortInt nblig, TShortInt nbcol, TMat *pmat);
void DetruMat(TMat *pmat);
TShortInt GerMessage( );
TShortInt CMu( TLongInt NbObsT, TVectLong *NbRepet,
	       TVect *PoidsT, TVect *ValY, TVect *Valf, TVect *VarY,
              TShortInt *TypeMu, TVect *Residus,
              TVect *Mu3, TVect *Mu4);


/*--------------- Function identification ----------
| Function name         : calcb                    |
| inspire de CBmvtb
| Role                  : Calculate B              |
| Input arguments       :                          |
|  NbRepet: number of replications of each         |
|          observation                             |
|  Valf: values of the regression function f       |
|  VarY: values of the variance of Y               |
|  S2: sum of squares of each observation          |
|  DValf: derivates of the function f with respect |
|     to the active parameters Theta               |
|  DVarYTheta: derivates of the variance of Y      |
|     with respect to the active parameters Theta  |
|     or matrix of dimensions equal to zero        |
|  DVarYBeta: derivates of the variance of Y       |
|     with respect to the active parameters Beta   |
|     or matrix of dimensions equal to zero        |
| Output arguments      :                          |
|  ValB: matrix B                                  |
|        dimension: NbZ*NbObs, (NbTheta+NbBeta)    | 
|    Already allocated                             |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :  calcfx_, calcvx_         |
| Calling functions    : CCovNu2, CCovNu3, PussIter|
|       Retablir                                   |
|  through the function pointer PCValB             |
--------------------------------------------------*/


/*--------------- Function definition ------------*/

TShortInt calcb(NbRepet, Valf, VarY, S2, DValf, DVarYTheta, DVarYBeta,
                 ValB)

/* input arguments */
TVectLong *NbRepet; /* dimension NbObs */
TVect *Valf, *VarY, *S2; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta */
TMat *DVarYBeta;  /* dimension NbObs, NbBeta */

/* output arguments */
TMat *ValB; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- End function definition ----*/

{
/* locals */
TShortInt a, IObs, iparam, iparama,
  m, NbObs, NbObsT, NbTheta, NbBeta,NbThetaA, NbBetaA ;
TDouble X,U,T,M, parama;

/* pointeurs pour ameliorer la performance */
TLongInt *nbrepet;
TDouble *vary, *valf ;
TDouble **valb, **Dvalf, **dvarytheta, **dvarybeta;

/* outputs of calcfx_, calcvx_ */
TShortInt Code, IndErr;
TMat  dvbdx;
TDouble fx, vx, *ptrfx, *ptrvx,dfdx, dvtdx,  *ptr1dfdx, **ptrdfdx,
   *ptr1dvtdx, **ptrdvtdx;
TDouble **ptrobs;

extern short int calcfx_(), calcvx_();
 TChar Mess4[5];
  TChar Message[5], Mess1[20], Mess2[20], Mess3[20]; 

/* Write into the trace of the called programs */
ECRTRACE("calcb");

/* determination des dimensions */
/* Attention: les argu sont en dimension actifs */
NbObs = Donnees.NbObs;
NbObsT = Donnees.NbObsT;
NbTheta = Modele.NomTheta.nbele;
NbBeta = Modele.NomBeta.nbele;
m= NbRepet->donnees[NbObs-1];


NbThetaA = DValf->nbcol;
NbBetaA = DVarYBeta->nbcol;


/* Allocation de dvbdx */
APPEL(CreerMatC((TShortInt)1,  NbBeta, &(dvbdx)));

/* affectation des pointeurs */
nbrepet = NbRepet->donnees;
vary = VarY->donnees;
valf = Valf->donnees;
valb = ValB->donnees;
Dvalf =  DValf->donnees;
dvarytheta = DVarYTheta->donnees;
dvarybeta = DVarYBeta->donnees;


ptrfx=&fx;
ptr1dfdx=&dfdx;
ptrdfdx=&ptr1dfdx;
ptrvx=&vx;
ptr1dvtdx=&dvtdx;
ptrdvtdx=&ptr1dvtdx;
ptrobs=&(Theta[0].Estim.donnees);


iparama =NbThetaA-1; /* param rajoute */
iparam =NbTheta-1; /* param rajoute */
/* parama: valeur en dim actif du param rajoute.
Celle-ci n'est pas dans Theta[0].Act
(celui-ci contient la valeur au pas precedent): il est dans ThetaCour
structure interne de PussIter: donc on la recalcule a partir de Estim
qui lui, est a jour .
*/
parama= asin(sqrt((Theta[0].Estim.donnees[iparam]-
                     Modele.CTheta0.BInf.donnees[iparam]) /
                   (Modele.CTheta0.BSup.donnees[iparam]- 
                   Modele.CTheta0.BInf.donnees[iparam])));
if ((errno==EDOM) || (errno== ERANGE))
  {
  /* overflow */
  sprintf(Message, "%hd", (iparam+1));
  sprintf(Mess4, "%hd", (iparama+1));
  sprintf(Mess1, "%g", parama);
  sprintf(Mess2, "%g", Modele.CTheta0.BInf.donnees[iparam]);
  sprintf(Mess3, "%g", Modele.CTheta0.BSup.donnees[iparam]);

  /* Actions effectuees par la macro suivante NLERREUR:
  fprintf(stderr,
 "ModAAct: Overflow dans la transformation du %s ieme parametre different en le %s ieme param. actif\n", 
 Mess4, Message);
  fprintf(stderr," La valeur du parametre actif est: %s\n", Mess1);
  fprintf(stderr," La valeur de sa borne inferieure est: %s\n", Mess2);
  fprintf(stderr," La valeur de sa borne superieure est: %s\n", Mess3);
  return(ERROVFL4);
  fin actions de NLERREUR */
  NLERREUR((ERROVFL4,6,"ModAAct",Mess4,Message,Mess1,Mess2,Mess3, ERR));
  } /* fin cas d'erreur */


/* maj de XObs, XObsT:
  la valeur du  xchapeau est dans le dernier Theta[0] pour l'obs rajoutee */
Donnees.XObs.donnees[NbObs-1][0]=  Theta[0].Estim.donnees[iparam];
for (IObs=(NbObsT-m); IObs<NbObsT; IObs++)
  Donnees.XObsT.donnees[IObs][0]=  Theta[0].Estim.donnees[iparam];


/* calcul des derivees de f et, si besoin, de v par rapport a la variable explicative */
/* 1 param theta (le x), 1 obs (celle composee des valeurs
 courantes des Theta),
  le nombre de va explicatives est NbTheta-1 puisque on a inverse
  dans le fichier formel qui a genere calcvx, les varind et parresp,
*/
/*
for (IObs=0; IObs<NbObs; IObs++)
  {
*/
IObs = NbObs-1;

  Code = calcfx_(    1,
                   Modele.GamF.nbele, 
                   1,
                   (NbTheta-1),
                   &(Donnees.XObs.donnees[IObs][0]),
                   &(Modele.GamF.donnees[0]),
                   ptrobs, 
                   ptrfx, ptrdfdx,
                   &(ResNum[0].InfoModErr.LieuErr), &IndErr);

  if (Code != OK)
    {
    if (GNLControle.SortWE !=NULL)
      fprintf(GNLControle.SortWE,
        "Problem when calculating f or df/dx in calcb\n");
  
    return(Code);
    }
  ResStat[0].FctSensib.DValf.donnees[IObs][iparam]= dfdx;

  /* Transfo en actif pour tenir des contraintes */
  dfdx = ((Modele.CTheta0.BSup.donnees[iparam] - Modele.CTheta0.BInf.donnees[iparam]) *
            sin((TDouble)2 * parama)) * dfdx;
  Dvalf[IObs][iparama]= dfdx;

  if (Modele.YaCalcV== VRAI)
    {
    Code = calcvx_(1, NbBeta, Modele.GamV.nbele,
               1,
               (NbTheta-1),
                   &(Donnees.XObs.donnees[IObs][0]),
               &(Beta[0].Estim.donnees[0]), 
               &(Modele.GamV.donnees[0]),
               ptrfx, ptrdfdx,
               ptrobs,
               ptrvx, ptrdvtdx,
               &(dvbdx.donnees[0]),
                   &(ResNum[0].InfoModErr.LieuErr), &IndErr);

    if (Code != OK)
      {
      if (GNLControle.SortWE !=NULL)
        fprintf(GNLControle.SortWE,
        "Problem when calculating v or dv/dx in calcb\n");
      return(Code);
      }
    /* pas de ponderation  et Multiplication par Sigma, on a appele CModele avant,
    donc le sigma2 est toujours calcule */
    dvtdx = dvtdx * ResNum[0].Sigma;
    } /* fin de YaCalcV  */
  else
    {
    dvtdx =0;
    }
    
  /* maj des derivees dim modele */
  ResStat[0].FctSensib.DVarYTheta.donnees[IObs][iparam]= dvtdx;

  /* Transfo en actif pour tenir des contraintes */
  dvtdx = ((Modele.CTheta0.BSup.donnees[iparam] - Modele.CTheta0.BInf.donnees[iparam]) *
          sin((TDouble)2 * parama)) * dvtdx;
  dvarytheta[IObs][iparama]= dvtdx;

for(IObs=0; IObs<NbObs; IObs++)
  {
  if (IObs != (NbObs-1))
    {
    Dvalf[IObs][iparama]= dvarytheta[IObs][iparama]=(TDouble)0;
    ResStat[0].FctSensib.DValf.donnees[IObs][iparam]= (TDouble)0;
    ResStat[0].FctSensib.DVarYTheta.donnees[IObs][iparam]=(TDouble)0;
    }


  /* calcul de B */
  X = (TDouble)nbrepet[IObs] / vary[IObs];
  U = X / vary[IObs];
  T =  U * valf[IObs];
  M =   (U / (TDouble)2);
  for(a  =  0; a < NbThetaA; a++)
    {
    valb[IObs][a] =   (X * Dvalf[IObs][a]) - (T * dvarytheta[IObs][a]);
    valb[IObs+NbObs][a] =  M * dvarytheta[IObs][a];
    }
  if (NbBetaA>0)
    {
    for(a = 0; a < NbBetaA; a++)
      {
      valb[IObs][NbThetaA+a] = - (T * dvarybeta[IObs][a]);
      valb[IObs+NbObs][NbThetaA+a] =  M * dvarybeta[IObs][a];
      }
    }
  }  /* fin boucle IObs */


DetruMat(&(dvbdx));

return(OK);
}



/*--------------- Function identification ----------
| Function name         : calcd                    |
| Identique a CDmv
| Role                  : Calculate D              |
| Input arguments       :                          |
|  Valf: values of the regression function f       |
|  DValf: derivates of the function f with respect |
|     to the active parameters Theta               |
|  DVarYTheta: derivates of the variance of Y      |
|     with respect to the active parameters Theta  |
|     or matrix of dimensions equal to zero        |
|  DVarYBeta: derivates of the variance of Y       |
|     with respect to the active parameters Beta   |
|     or matrix of dimensions equal to zero        |
| Output arguments      :                          |
|  ValD: matrix D                                  |
|        dimension: NbZ*NbObs,(NbTheta+NbBeta)     | 
|    Already allocated                             |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :                           |
| Calling functions    : CCovNu2, CCovNu3, PussIter|
|       Retablir                                   |
|  through the function pointer PCValD             |
--------------------------------------------------*/


/*--------------- Function definition ------------*/

TShortInt calcd( Valf, DValf, DVarYTheta, DVarYBeta,
                 ValD)

/* input arguments */
TVect *Valf; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta */
TMat *DVarYBeta;  /* dimension NbObs, NbBeta */


/* output arguments */
TMat *ValD; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- End function definition ----*/

{
/* locals */
TShortInt NbObs,NbThetaA, NbBetaA, i, j;

/* pointeurs pour ameliorer la performance */
TDouble  *valf;
TDouble **vald, **Dvalf, **dvarytheta, **dvarybeta;



/* Write into the trace of the called programs */
ECRTRACE("calcd");


/* determination des dimensions */
NbObs = DValf->nblig;
NbThetaA = DValf->nbcol;
NbBetaA = DVarYBeta->nbcol;



/*  affectation des pointeurs*/
valf = Valf->donnees;
vald = ValD->donnees;
Dvalf =  DValf->donnees;
dvarytheta = DVarYTheta->donnees;
dvarybeta = DVarYBeta->donnees;


for (i = 0; i<NbObs; i++)
  {
  for (j = 0 ; j < NbThetaA; j++)
    {
    vald[i][j] = Dvalf[i][j];
    vald[i+NbObs][j] = dvarytheta[i][j] + 
               ((TDouble)2 * valf[i] * Dvalf[i][j]);
    }
  if (NbBetaA>0)
    {
    for (j = 0 ; j < NbBetaA; j++)
      {
      vald[i][j+NbThetaA] = (TDouble)ZERO;
      vald[i+NbObs][j+NbThetaA] = dvarybeta[i][j];
      }
    } /* fin de NbBetaA>0 */
  }


return(OK);
}


/*--------------- Function identification ----------
| Function name         : calceta                  |
| Role                  : Calculate Eta            |
| Identique a CEtamv
| Input arguments       :                          |
|  Valf: values of the regression function f       |
|  VarY: values of the variance of Y               |
| Output arguments      :                          |
|  ValEta: vector Eta                              |
|        dimension: NbZ*NbObs                      | 
|    Already allocated                             |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :                           |
| Calling functions    :  PussIter, Retablir       |
|  through the function pointer PCValEta           |
--------------------------------------------------*/

/*--------------- Function definition ------------*/

TShortInt calceta( Valf, VarY,
                 ValEta)

/* input arguments */
TVect *Valf; /* dimension NbObs */
TVect *VarY; /* dimension NbObs */

/* output arguments */
TVect *ValEta; /* dim: NbZ*NbObs */

/*--------------- End function definition ----*/

{
/* locals */
TShortInt NbObs, i;

/* pointeurs pour ameliorer la performance */
TDouble *vary, *valf, *valeta ;


/* Ecriture de la trace */
ECRTRACE("CEtamv");

/* determination des dimensions */
NbObs = Valf->nbele;

/* affectation des pointeurs */
vary = VarY->donnees;
valf = Valf->donnees;
valeta = ValEta->donnees;

for (i = 0; i<NbObs; i++)
  {
  valeta[i] = valf[i];
  valeta[i+NbObs] = (valf[i] * valf[i]) + vary[i];
  }

return(OK);
}



/*--------------- Function identification ----------
| Function name         : calcz                    |
| Identique a CZmv
| Role                  : Calculate the vector of  |
|  the sufficient statistics Z and calculate its   |
|  values                                          |
| Input arguments       :                          |
|  Y1 and Y2: pointers on the components Y1 and Y2 |
|    of StatDon                                    |
| Output arguments      :                          |
|  ValZ: the vector Z                              |
|    Must be allocated by this function            |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :                           |
| Functions called     :                           |
| Calling functions    : NLEtape                   |
|  through the function pointer PCValZ             |
--------------------------------------------------*/

/*--------------- Function definition ------------*/
TShortInt calcz(Y1,Y2,ValZ)

  /* input arguments */
  TVect *Y1,*Y2;

  /* output argument */
  TVect *ValZ;


/*--------------- End function definition ----*/
{
  /* locals */
  TShortInt i, NbObs;
  /* pointeurs pour ameliorer la performance */
  TDouble *valz, *y1, *y2;

  /* mettre son nom dans la trace */
  ECRTRACE("CZmv");

  /* Affectation des pointeurs */
  valz = ValZ->donnees;
  y1 = Y1->donnees;
  y2 = Y2->donnees;

  /* Affectation des valeurs */
  NbObs = Y1->nbele;
  for(i = 0; i < NbObs; i++)
    {
    valz[i] = y1[i];
    valz[i+NbObs] = y2[i];
    }
  /* Retour */
  return(OK);
  }


/*--------------- Function identification ----------
| Function name         : calcvarz                 |
| Identique a CVarZmv
| Role                  : Create the vector of the |
|  variance of Z and calculate its value           |
| Input arguments       :                          |
|  NbObsT: number of observations, replications    |
|   included                                       |
|  NbRepet: number of replications of each         |
|          observation                             |
|  PoidsT: weighting values affected on the        |
|   observations, replications included            |
|  ValY: observed values of the response           |
|  Ajustes: a structure that contains:             |
|      Valf: values of the regression function f   |
|      VarY: values of the variance of Y           |
| Input-output arguments :                         |
|  TypeMu: code for the way of calculating the     |
|    moments                                       |
|  Residus : the non-standardized residuals        |
|  Mu3 : moments of order 3                        |
|  Mu4 : moments of order 4                        |
| Output arguments      :                          |
|  VarZ: matrix that contains the variance of Z    |
|    Must be allocated by this function            |
|  Code:  OK (=0) or error code                    |
| Return value          : OK (=0) or error code    |
|  when the error is fatal for the remaining of the|
|  execution                                       |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :  <insert here the names   |
|    of the functions called>                      |
| Functions called     :                           |
| Calling functions    :  CBVarZBP                 |
|   through the function pointer PCValZ            |
--------------------------------------------------*/

/*--------------- Function definition ------------*/


TShortInt calcvarz( NbObsT, NbRepet, PoidsT, ValY, Ajustes,
              TypeMu, Residus, Mu3, Mu4,
              VarZ, Code)

/* input arguments */
TLongInt NbObsT;
TVectLong *NbRepet;
TVect *PoidsT, *ValY;
TAjustes *Ajustes;

/* input-output arguments */
TShortInt *TypeMu;
TVect *Residus;
TVect *Mu3, *Mu4;

/* output arguments*/
TMat *VarZ; /* dimension NbObs*NbZ, NbObs*NbZ  */
TShortInt *Code;

/*--------------- End function definition ----*/
{
  /* locals */
  TShortInt NbObs, Dim, i, j, ii;
  TChar Mess1[5], Mess2[50];


/* pointeurs sur des elements de structure pour ameliorer la performance */
  TLongInt *nbrepet;
  TDouble **varz;
  TDouble *vary, *valf, *mu3, *mu4;

  /* mettre son nom dans la trace */
  ECRTRACE("CVarZmv");

  *Code = OK;
  NbObs = NbRepet->nbele;
  Dim = NbObs * 2;

  /* calculer les Mu */
  if (GNLControle.Voulu.Mu==FAUX)
    {
    /* ils n'ont pas encore ete calcules */
    APPEL(CMu( NbObsT, NbRepet, PoidsT, ValY, &(Ajustes->Valf), &(Ajustes->VarY),
              TypeMu, Residus,
              Mu3, Mu4));

    }

  /* affectation des pointeurs pour ameliorer la performance */
  varz = VarZ->donnees;
  vary = Ajustes->VarY.donnees;
  valf = Ajustes->Valf.donnees;
  nbrepet = NbRepet->donnees;
  mu3 = Mu3->donnees;
  mu4 = Mu4->donnees;

  /* initialisation a zero */
  for (i=0; i<Dim; i++)
    {
    for(j=0; j<Dim; j++)
      {
      varz[i][j] = (TDouble)ZERO;
      }
    }

  for (i=0; i< NbObs; i++)
    {
    ii = NbObs+i;
    varz[i][i] = vary[i] / nbrepet[i];
    varz[i][ii] = 
       (mu3[i] + ((TDouble)2 * valf[i]  * vary[i]))
       / nbrepet[i];
    varz[ii][i] = varz[i][ii];
    varz[ii][ii] = 
       (mu4[i] + ((TDouble)4 * valf[i] * valf[i] * vary[i])
       - (vary[i] * vary[i]) 
       + ((TDouble)4 * valf[i] * mu3[i]))
       / nbrepet[i];
    if (varz[ii][ii] <= (TDouble)ZERO)
      {
      /* theoriquement ca n'est pas possible,
      mais ca peut le devenir, suite a des arrondis de calcul */
      sprintf(Mess1,"%d", (ii+1));
      sprintf(Mess2, "%10.20e", varz[ii][ii]);
/*      printf("CVarZmv: le %s-ieme terme de la diagonale de VarZ est <=0 (= %s)\n",
        Mess1, Mess2);
*/
      NLWARNING((WARVARZ1,3,"CVarZmv",Mess1, Mess2,WNUMER));
      /* dans ce cas, les calculs qui incluent la variance
      n'ont pas de sens */
      *Code = WARVARZ1;
      return(OK);
      } /* fin du cas <=0 */
    }

return(OK);
}

/*--------------- Function identification ----------
| Function name         : myinv                    |
| Role                  : invert the matrix W      |
|  when it is no SYM neither SYMB                  |
| Input arguments       :                          |
|  ValW: matrix W                                  |
| Output arguments      :                          |
|  ValWInv: matrix of same dimension than ValW     |
|    Already allocated                             |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references -----------------
| Functions called     :  <insert here the names   |
|    of the functions called>                      |
| Calling functions    :  CArret                   |
--------------------------------------------------*/


/*--------------- Function definition ------------*/

TShortInt myinv(ValW, ValWInv)

/* input arguments */
TMat *ValW;

/* output arguments*/
TMat *ValWInv;

/*--------------- End function definition ----*/

{

return(OK);
}

/*--------------- Function identification ----------
| Function name         : calcc                    |
| Role                  : Calculate the fitting    |
|  criterion: to provide only if the way of        |
|  calculating it is your own                      |
|  You can inspire from the programs of file       |
|  Critere.c in the source-directory of NL         |
| Input arguments       :                          |
|  Sigma2:  the estimated value of square sigma    |
|  NbObsT:  number of observations, replications   |
|           included                               |
|  NbRepet: number of replications of each         |
|          observation                             |
|  Y1: the sum of the response values on the       |
|     replications weighted by the number of       |
|      replications                                |
|  Y2: the sum of the squared values of the        |
|      response on the replications weighted by the|
|      number of replications                      |
|  S2: the intra-replications variance.            |
|  Valf: values of the regression function f       |
|  VarY: values of the variance of Y               |
| Output arguments      :                          |
|  Crit: the fitting criterion                     |
| Return value          : OK (=0) or error code    |
--------------------------------------------------*/

/*--------------- Cross references ----------------
| Functions called     :  <insert here the names   |
|    of the functions called>                      |
| Calling functions    :  PussIter                 |
--------------------------------------------------*/


/*--------------- Function definition ------------*/


TShortInt calcc(Sigma2, NbObsT, NbRepet, Y1, Y2, S2, Valf, VarY,
                Lv)

/* input arguments */
TDouble Sigma2;
TLongInt NbObsT;
TVectLong *NbRepet; /* dimension NbObs */
TVect *Y1, *Y2, *S2, *Valf, *VarY; /* dimension NbObs */

/* output arguments */
TDouble *Lv;

/*--------------- End function definition ----*/

{
/* locals */
TShortInt i, n;


/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *vary, *valf, *y1, *y2;
TLongInt *nbrepet;


/* If wanted, write into the trace of the called programs */
ECRTRACE("calcc");

n = NbRepet->nbele;
nbrepet = NbRepet->donnees;
vary = VarY->donnees;
valf = Valf->donnees;
y1 = Y1->donnees;
y2 = Y2->donnees;

*Lv=(TDouble)ZERO;

for (i = 0; i < n; i++)
  {
  *Lv = *Lv + (nbrepet[i] * (log(vary[i])
    + (( (TDouble)1 / vary[i])
               * (y2[i] + (valf[i] * valf[i] )
               - ((TDouble)2 * valf[i] * y1[i])))));
  }

  *Lv = log(((TDouble)2 * (TDouble)M_PI)) + (*Lv / (TDouble)NbObsT);

/* retour */
return(OK);
}
