/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : geneSubr.c                |
| Date                 : 13 dec. 1991              |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : O. Nicole                 |
| Role                 : stocke les arbres dans des subroutines C
| Reference conception : DCP module 22             |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include "nltypes.h"
#include "nlcodes.h"
#include "dftypes.h"
#include "dfcodes.h"
#include <stdlib.h>

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/
extern char *strcat(char *dest, const char *src);
extern void *malloc(size_t size);
extern TShortInt erreAnal(TChaine message, TChaine param);

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/

/*--------------- Identification fonction ----------
| Nom de la fonction    : ecrArb                   |
| Role                  : stocke un arbre dans le fichier de subroutine
| Parametres d'entree   : arbre fout decl modSpecif theta
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 22            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : ecrTest                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TLogic ecrArb(arbre, fout, decl, modSpecif, theta)
     TArbre * arbre;
     FILE * fout;
     TShortInt *decl, theta;
     TModSpecif modSpecif;

/*--------------- Fin identification fonction ----*/

{
  TShortInt ident, j;
  TLogic indicder;

  if (arbre==NULL)
    {
      return(VRAI);
    }
  if (*decl>60)
    {
      fprintf(fout, "\n      ");
      *decl=7;
    }
  switch (arbre->fonction)
    {
    case PLUS:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") + (");
      *decl=*decl+5;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout,")");
      *decl=*decl+1;
      break;
    case MOINS:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") - (");
      *decl=*decl+5;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout,")");
      *decl=*decl+1;
      break;
    case MULT:
      if (arbre->filsG->fonction==CONSTANTE &&
	  arbre->filsG->valeur==(double)-1)
	{
	  fprintf(fout, "-(");
	  *decl=*decl+2;
	  ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
	  fprintf(fout, ")");
	  *decl=*decl+1;
	}
      else if (arbre->filsD->fonction==CONSTANTE &&
	  arbre->filsD->valeur==(double)-1)
	{
	  fprintf(fout, "-(");
	  *decl=*decl+2;
	  ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
	  fprintf(fout, ")");
	  *decl=*decl+1;
	}
      else
	{
	  fprintf(fout, "(");
	  *decl=*decl+1;
	  ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
	  fprintf(fout, ") * (");
	  *decl=*decl+5;
	  ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
	  fprintf(fout,")");
	  *decl=*decl+1;
	}
      break;
    case DIV:
      fprintf(fout, "mydiv(");
      *decl=*decl+5;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ", ");
      *decl=*decl+2;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout,")");
      *decl=*decl+1;
      break;
    case LOG:
      fprintf(fout, "mylog(");
      *decl=*decl+6;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case TG:
      fprintf(fout, "(double)tan(");
      *decl=*decl+11;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case LN:
      fprintf(fout, "myln(");
      *decl=*decl+7;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case EXP:
      fprintf(fout, "(double)myexp(");
      *decl=*decl+14;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case SIN:
      fprintf(fout, "(double)sin(");
      *decl=*decl+12;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case COS:
      fprintf(fout, "(double)cos(");
      *decl=*decl+12;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case IF:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") ? (");
      *decl=*decl+5;
      ecrArb(arbre->filsD->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") : (");
      *decl=*decl+5;
      ecrArb(arbre->filsD->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case OR:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") || (");
      *decl=*decl+6;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case AND:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") && (");
      *decl=*decl+6;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case EQ:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") == (");
      *decl=*decl+6;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case NEQ:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") != (");
      *decl=*decl+6;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case LE:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") < (");
      *decl=*decl+5;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case LEQ:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") <= (");
      *decl=*decl+6;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case GE:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") > (");
      *decl=*decl+5;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case GEQ:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") >= (");
      *decl=*decl+6;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case NOT:
      fprintf(fout, "!(");
      *decl=*decl+2;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case PUISSANCE:
      fprintf(fout, "mypow(");
      *decl=*decl+8;
      ecrArb(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ", ");
      *decl=*decl+2;
      ecrArb(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case CONSTANTE:
      if (*decl>40)
	{
	  fprintf(fout, "\n      ");
	  *decl=7;
	}
      fprintf(fout, "(double)");
      *decl=*decl+20;
      if (arbre->valeur==(double)((long)arbre->valeur))
	{
	  fprintf(fout, "%ld", (long)arbre->valeur);
	}
      else
	{
	  fprintf(fout, "%.8e", arbre->valeur);
	}
      break;
    case VALINT:
      ident=-arbre->filsG->fonction;
      indicder=FAUX;
      /* si le numero codant la fonction est compris entre 0 et -PPident.nbele
	 c'est un identificateur simple, s'il est compris entre
	 -PPident.nbele et -2*PPident.nbele, il s'agit d'une derivee
	 (typiquement la derivee de dF[T]) */
      if (ident>=modSpecif.PPident.nbele)
	{
	  ident=ident-modSpecif.PPident.nbele;
	  indicder=VRAI;
	}
      if (indicder==VRAI)
	{
	  /* c'est une derivee, on regarde dFsedo */
	  if (theta>=modSpecif.NbThetaSedo + modSpecif.IdeCondInit.nbele)
	    {
	      /* c'est la derivee d'une equation du sedo par rapport a un
		 parametre n'intervenant pas dans le sedo, ca vaut 0 */
	      fprintf(fout, "(double)0");
	      *decl=*decl+9;
	      break;
	    }
	  fprintf(fout, "d");
	  *decl=*decl+1;
	}
      ident=-arbre->filsD->fonction;
      /* il faut chercher dans valint */
      j=0;
      while (ident!=modSpecif.IdeValInt.donnees[j])
	{
	  j++;
	}
      fprintf(fout, "Fsedo[i][%d][", j);
      ident=-arbre->filsG->fonction;
      if (indicder==VRAI)
	{
	  ident=ident-modSpecif.PPident.nbele;
	}
      j=0;
      while (ident!=modSpecif.IdeLesF.donnees[j])
	{
	  j++;
	}
      if (indicder==VRAI)
	{
	  j=j*(modSpecif.NbThetaSedo+modSpecif.IdeCondInit.nbele)+theta;
	}
      fprintf(fout, "%d]", j);
      *decl=*decl+16;
      break;
    default:
      /* cas d'un identificateur */
      ident=-arbre->fonction;
      indicder=FAUX;
      /* si le numero codant la fonction est compris entre 0 et -PPident.nbele
	 c'est un identificateur simple, s'il est compris entre
	 -PPident.nbele et -2*PPident.nbele, il s'agit d'une derivee
	 (typiquement la derivee de f ou d'un aux) */
      if (ident>=modSpecif.PPident.nbele)
	{
	  ident=ident-modSpecif.PPident.nbele;
	  indicder=VRAI;
	}
      
      /* selon le type de l'identificateur */
      switch (modSpecif.PPident.ident[ident].type)
	{
	case IDECONST:
	  /* on remplace la contante par sa valeur */
	  j=0;
	  while (modSpecif.Constantes.UneCst[j].ident!=ident)
	    {
	      j++;
	    }
	  if (*decl>40)
	    {
	      fprintf(fout, "\n      ");
	      *decl=7;
	    }
	  fprintf(fout, "(double)");
	  *decl=*decl+20;
	  if (modSpecif.Constantes.UneCst[j].valeur==
	      (double)((long)modSpecif.Constantes.UneCst[j].valeur))
	    {
	      fprintf(fout, "%ld",
		      (long)modSpecif.Constantes.UneCst[j].valeur);
	    }
	  else
	    {
	      fprintf(fout, "%.8e", modSpecif.Constantes.UneCst[j].valeur);
	    }
	  break;
	case IDEAUX:
	  /* on met l'appel a l'aux correspondant */
	  j=0;
	  while (modSpecif.Auxiliaire.UnAux[j].ident!=ident)
	    {
	      j++;
	    }
	  if (indicder==VRAI)
	    {
	      /* on appel la derivee de l'aux par rapport a theta*/
	      fprintf(fout, "da[%d][%d]", j, theta);
	      *decl=*decl+9;
	    }
	  else
	    {
	      /* on appel l'aux */
	      fprintf(fout, "a[%d]", j);
	      *decl=*decl+5;
	    }
	  break;
	case IDEF:
	  if (indicder==VRAI)
	    {
	      /* derive de f (typiquement par rapport a theta */
	      fprintf(fout, "df[i][%d]", theta);
	      *decl=*decl+5;
	    }
	  else
	    {
	      fprintf(fout, "f[i]");
	      *decl=*decl+4;
	    }
	  break;
	case IDEX:
	  /* on remplace par l'appel de la variable correspondante */
	  j=0;
	  while (modSpecif.IdeX.donnees[j]!=ident)
	    {
	      j++;
	    }
	  fprintf(fout, "x[i][%d]", j);
	  *decl=*decl+8;
	  break;
	case IDETHETA:
	  j=0;
	  while (modSpecif.IdeTheta.donnees[j]!=ident)
	    {
	      j++;
	    }
	  fprintf(fout, "t[%d]", j);
	  *decl=*decl+5;
	  break;
	case IDEBETA:
	  j=0;
	  while (modSpecif.IdeBeta.donnees[j]!=ident)
	    {
	      j++;
	    }
	  fprintf(fout, "b[%d]", j);
	  *decl=*decl+5;
	  break;
	case IDEGAMF:
	  j=0;
	  while (modSpecif.IdeGamF.donnees[j]!=ident)
	    {
	      j++;
	    }
	  fprintf(fout, "g[%d]", j);
	  *decl=*decl+5;
	  break;
	case IDEGAMV:
	  j=0;
	  while (modSpecif.IdeGamV.donnees[j]!=ident)
	    {
	      j++;
	    }
	  fprintf(fout, "g[%d]", j);
	  *decl=*decl+5;
	  break;
	case IDELESF:
	  /* on ne passera jamais ici normalement */
	  return(FAUX);
	  break;
	case IDEVALINT:
	  /* on ne passera jamais ici normalement */
	  return(FAUX);
	  break;	  
	}
      break;
    }
  return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : ecrArbDF                   |
| Role                  : stocke un arbre dans le fichier de subroutine pour
                          un arbre de type LesDF
| Parametres d'entree   : arbre fout decl modSpecif theta
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 22            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : ecrTestDF                 |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TLogic ecrArbDF(arbre, fout, decl, modSpecif, theta)
     TArbre * arbre;
     FILE * fout;
     TShortInt *decl, theta;
     TModSpecif modSpecif;

/*--------------- Fin identification fonction ----*/

{
  TShortInt ident, j, dbt, dbg, dbobs, nbt, nbTot;
  TLogic indicder;


  nbt=modSpecif.IdeCondInit.nbele+modSpecif.NbThetaSedo;
  dbt=(nbt+1)*modSpecif.IdeLesDF.nbele;

  /* AB, 26/05/98 rajoute: */
  nbTot= modSpecif.IdeTheta.nbele;
  /* nbTot=nbre total de parametres tandis que 
     nbt=nbre de param dans le sedo +condinit*/
  dbg=nbTot+dbt;
  dbobs=dbg+modSpecif.IdeGamF.nbele;

  if (arbre==NULL)
    {
      return(VRAI);
    }
  if (*decl>60)
    {
      fprintf(fout, "\n      ");
      *decl=7;
    }
  switch (arbre->fonction)
    {
    case PLUS:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") + (");
      *decl=*decl+5;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout,")");
      *decl=*decl+1;
      break;
    case MOINS:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") - (");
      *decl=*decl+5;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout,")");
      *decl=*decl+1;
      break;
    case MULT:
      if (arbre->filsG->fonction==CONSTANTE &&
	  arbre->filsG->valeur==(double)-1)
	{
	  fprintf(fout, "-(");
	  *decl=*decl+2;
	  ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
	  fprintf(fout, ")");
	  *decl=*decl+1;
	}
      else if (arbre->filsD->fonction==CONSTANTE &&
	  arbre->filsD->valeur==(double)-1)
	{
	  fprintf(fout, "-(");
	  *decl=*decl+2;
	  ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
	  fprintf(fout, ")");
	  *decl=*decl+1;
	}
      else
	{
	  fprintf(fout, "(");
	  *decl=*decl+1;
	  ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
	  fprintf(fout, ") * (");
	  *decl=*decl+5;
	  ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
	  fprintf(fout,")");
	  *decl=*decl+1;
	}
      break;
    case DIV:
      fprintf(fout, "mydiv(");
      *decl=*decl+5;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ", ");
      *decl=*decl+2;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout,")");
      *decl=*decl+1;
      break;
    case LOG:
      fprintf(fout, "mylog(");
      *decl=*decl+6;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case TG:
      fprintf(fout, "(double)tan(");
      *decl=*decl+11;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case LN:
      fprintf(fout, "myln(");
      *decl=*decl+7;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case EXP:
      fprintf(fout, "(double)myexp(");
      *decl=*decl+14;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case SIN:
      fprintf(fout, "(double)sin(");
      *decl=*decl+12;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case COS:
      fprintf(fout, "(double)cos(");
      *decl=*decl+12;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case IF:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") ? (");
      *decl=*decl+5;
      ecrArbDF(arbre->filsD->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") : (");
      *decl=*decl+5;
      ecrArbDF(arbre->filsD->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case OR:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") || (");
      *decl=*decl+6;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case AND:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") && (");
      *decl=*decl+6;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case EQ:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") == (");
      *decl=*decl+6;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case NEQ:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") != (");
      *decl=*decl+6;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case LE:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") < (");
      *decl=*decl+5;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case LEQ:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") <= (");
      *decl=*decl+6;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case GE:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") > (");
      *decl=*decl+5;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case GEQ:
      fprintf(fout, "(");
      *decl=*decl+1;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ") >= (");
      *decl=*decl+6;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case NOT:
      fprintf(fout, "!(");
      *decl=*decl+2;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case PUISSANCE:
      fprintf(fout, "mypow(");
      *decl=*decl+8;
      ecrArbDF(arbre->filsG, fout, decl, modSpecif, theta);
      fprintf(fout, ", ");
      *decl=*decl+2;
      ecrArbDF(arbre->filsD, fout, decl, modSpecif, theta);
      fprintf(fout, ")");
      *decl=*decl+1;
      break;
    case CONSTANTE:
      if (*decl>40)
	{
	  fprintf(fout, "\n      ");
	  *decl=7;
	}
      fprintf(fout, "(double)");
      *decl=*decl+20;
      if (arbre->valeur==(double)((long)arbre->valeur))
	{
	  fprintf(fout, "%ld", (long)arbre->valeur);
	}
      else
	{
	  fprintf(fout, "%.8e", arbre->valeur);
	}
      break;
    case VALINT:
      /* on ne passera jamais ici normalement */
      return(FAUX);
      break;
    default:
      /* cas d'un identificateur */
      ident=-arbre->fonction;
      indicder=FAUX;
      /* si le numero codant la fonction est compris entre 0 et -PPident.nbele
	 c'est un identificateur simple, s'il est compris entre
	 -PPident.nbele et -2*PPident.nbele, il s'agit d'une derivee
	 (typiquement la derivee de f ou d'un aux) */
      if (ident>=modSpecif.PPident.nbele)
	{
	  ident=ident-modSpecif.PPident.nbele;
	  indicder=VRAI;
	}
      
      /* selon le type de l'identificateur */
      switch (modSpecif.PPident.ident[ident].type)
	{
	case IDECONST:
	  /* on remplace la contante par sa valeur */
	  j=0;
	  while (modSpecif.Constantes.UneCst[j].ident!=ident)
	    {
	      j++;
	    }
	  if (*decl>40)
	    {
	      fprintf(fout, "\n      ");
	      *decl=7;
	    }
	  fprintf(fout, "(double)");
	  *decl=*decl+20;
	  if (modSpecif.Constantes.UneCst[j].valeur==
	      (double)((long)modSpecif.Constantes.UneCst[j].valeur))
	    {
	      fprintf(fout, "%ld",
		      (long)modSpecif.Constantes.UneCst[j].valeur);
	    }
	  else
	    {
	      fprintf(fout, "%.8e", modSpecif.Constantes.UneCst[j].valeur);
	    }
	  break;
	case IDEAUX:
	  /* on met l'appel a l'aux correspondant */
	  j=0;
	  while (modSpecif.Auxiliaire.UnAux[j].ident!=ident)
	    {
	      j++;
	    }
	  if (indicder==VRAI)
	    {
	      /* on appel la derivee de l'aux par rapport a theta*/
	      fprintf(fout, "da[%d][%d]", j, theta);
	      *decl=*decl+9;
	    }
	  else
	    {
	      /* on appel l'aux */
	      fprintf(fout, "a[%d]", j);
	      *decl=*decl+5;
	    }
	  break;
	case IDEF:
	  /* on ne passera jamais ici normalement */
	  return(FAUX);
	  break;
	case IDEX:
	  /* on remplace par l'appel de la variable correspondante */
	  j=0;
	  while (modSpecif.IdeX.donnees[j]!=ident)
	    {
	      j++;
	    }
	  fprintf(fout, "y[%d+nbobs*%d+idobs]", dbobs, j);
	  *decl=*decl+20;
	  break;
	case IDETHETA:
	  j=0;
	  while (modSpecif.IdeTheta.donnees[j]!=ident)
	    {
	      j++;
	    }
	  fprintf(fout, "y[%d]", dbt+j);
	  *decl=*decl+5;
	  break;
	case IDEBETA:
	  /* on ne passera jamais ici normalement */
	  return(FAUX);
	  break;
	case IDEGAMF:
	  j=0;
	  while (modSpecif.IdeGamF.donnees[j]!=ident)
	    {
	      j++;
	    }
	  fprintf(fout, "y[%d]", dbg+j);
	  *decl=*decl+5;
	  break;
	case IDEGAMV:
	  /* on ne passera jamais ici normalement */
	  return(FAUX);
	  break;
	case IDELESF:
	  j=0;
	  while (modSpecif.IdeLesF.donnees[j]!=ident)
	    {
	      j++;
	    }
	  if (indicder==NON)
	    {
	      fprintf(fout, "y[%d]", j);
	      *decl=*decl+5;
	    }
	  else
	    {
	      fprintf(fout, "y[%d]", modSpecif.IdeLesDF.nbele+j*nbt+theta);
	      *decl=*decl+5;
	    }
	  break;
	case IDEVALINT:
	  /* on ne passera jamais ici normalement */
	  return(FAUX);
	  break;
	case IDEVARINT:
	  fprintf(fout, "*t");
	  *decl=*decl+2;
	  break;
	}
      break;
    }
  return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : ecrTest                  |
| Role                  : ecrit le test systematique de la valeur de la
                          fonction
| Parametres d'entree   : arbre fout lieuerr modSpecif
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 22            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : ecrArb                    |
| Fonctions appelantes : stokVari stokResp stokPhi |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TLogic ecrTest(arbre, fout, lieuerr, modSpecif, theta)
     TArbre * arbre;
     FILE * fout;
     TChaine lieuerr;
     TModSpecif modSpecif;
     TShortInt theta;
     
/*--------------- Fin identification fonction ----*/
     
{
  TShortInt posit;
  
  fprintf(fout, "    res= ");
  posit=10;
  if (ecrArb(arbre, fout, &posit, modSpecif, theta)==FAUX)
    {
      return(FAUX);
    }
  fprintf(fout, ";\n");
  if (arbre!=NULL)
    {
      if (arbre->fonction>CONSTANTE)
	{
	  /* pour un identificateur ou une constante, pas la peine de
	     tester la valeur de calcul */
	  fprintf(fout, "    if (finite(res)!=1)\n      {\n        *ie=i+1;\n");
	 fprintf(fout, "        *le=%s;\n        return(ERRMATH);\n      }\n",
		  lieuerr);
	}
    }
  else
    {
      return(FAUX);
    }
return(VRAI);

}

/*--------------- Identification fonction ----------
| Nom de la fonction    : ecrTestDF                |
| Role                  : ecrit le test systematique de la valeur de la
                          fonction
| Parametres d'entree   : arbre fout modSpecif     
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 22            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : ecrArbDF                  |
| Fonctions appelantes : stokSedo
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TLogic ecrTestDF(arbre, fout, modSpecif, theta)
     TArbre * arbre;
     FILE * fout;
     TModSpecif modSpecif;
     TShortInt theta;
     
/*--------------- Fin identification fonction ----*/
     
{
  TShortInt posit;
  
  fprintf(fout, "res= ");
  posit=5;
  if (ecrArbDF(arbre, fout, &posit, modSpecif, theta)==FAUX)
    {
      return(FAUX);
    }
  fprintf(fout, ";\n");
  if (arbre!=NULL)
    {
      if (arbre->fonction>CONSTANTE)
	{
	  /* pour un identificateur ou une constante, pas la peine de
	     tester la valeur de calcul */
	  fprintf(fout, "if (finite(res)!=1)\n  {\n");
	  fprintf(fout, "  neq[6]=ERRMATH;\n");
	  fprintf(fout, "  return(ERRMATH);\n  }\n");
	}
    }
  else
    {
      return(FAUX);
    }
return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : stokResp                 |
| Role                  : stocke la fonction calcf |
| Parametres d'entree   : fout modSpecif derivees  |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 22            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : erreAnal ecrTest          |
| Fonctions appelantes : geneSubr                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TLogic stokResp(fout, modSpecif, derivees)
     FILE * fout;
     TModSpecif modSpecif;
     TDerivees derivees;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i, j, k;
  TLogic test;
  TChaine lieuerr;
  
  /* cree l'entete de la fonction calcf */
  fprintf(fout,
	  "\nshort int calcf_(nbt, nbg, nbl, nbc, t, g, x, f, df, le, ie)\n");
  fprintf(fout, "      short int nbt, nbg, nbl, nbc;\n");
  fprintf(fout, "      double *t, *g, **x, *f, **df;\n");
  fprintf(fout, "      short int *le, *ie;\n");
  fprintf(fout, "{\nshort int i;\ndouble res;\n");
  if (modSpecif.Auxiliaire.nbele>0)
    {
      fprintf(fout, "double a[%d], da[%d][%d];\n", modSpecif.Auxiliaire.nbele,
	      modSpecif.Auxiliaire.nbele, modSpecif.IdeTheta.nbele);
    }

  fprintf(fout, "ECRTRACE(\"calcf\");\n");

  fprintf(fout, "for (i=0; i<nbl; i++)\n  {\n");
  
  /* stocke les auxiliaires */
  lieuerr=LA;
  for (i=0; i<modSpecif.Auxiliaire.nbele; i++)
    {
      test=FAUX;
      k=0;
      while (test==FAUX && k<modSpecif.AuxResp.nbele)
	{
	  if (modSpecif.Auxiliaire.UnAux[i].ident==
	      modSpecif.AuxResp.donnees[k])
	    {
	      test=VRAI;
	    }
	  k++;
	}
      if (test==VRAI)
	{
	  if (ecrTest(modSpecif.Auxiliaire.UnAux[i].arbre, fout,
		      lieuerr, modSpecif, -1)==FAUX)
	    {
	      return(FAUX);
	    }
	  fprintf(fout, "    a[%d]= res;\n", i);
	}
    }

  /* stocke f */
  lieuerr=LF;
  if (ecrTest(modSpecif.IdeF.arbre, fout, lieuerr, modSpecif, -1)==FAUX)
    {
      return(FAUX);
    }
  fprintf(fout, "    f[i]= res;\n");

  /* stocke les derivees des auxiliaires */
  lieuerr=LDADT;
  for (i=0; i<modSpecif.Auxiliaire.nbele; i++)
    {
      test=FAUX;
      k=0;
      while (test==FAUX && k<modSpecif.AuxResp.nbele)
	{
	  if (modSpecif.Auxiliaire.UnAux[i].ident==
	      modSpecif.AuxResp.donnees[k])
	    {
	      test=VRAI;
	    }
	  k++;
	}
      if (test==VRAI)
	{
	  for (j=0; j<modSpecif.IdeTheta.nbele; j++)
	    {
	      if (ecrTest(derivees.DerivAux.arbre[j*modSpecif.Auxiliaire.
						  nbele+i],
			  fout, lieuerr, modSpecif, j)==FAUX)
		{
		  return(FAUX);
		}
	      fprintf(fout, "    da[%d][%d]= res;\n", i, j);
	    }
	}
    }

  lieuerr=LDFDT;
  for (i=0; i<modSpecif.IdeTheta.nbele; i++)
    {
      /* stocke la derivee de f pour theta[i] */
      if (ecrTest(derivees.DerivF.arbre[i], fout, lieuerr, modSpecif, i)
	==FAUX)
	{
	  return(FAUX);
	}
      fprintf(fout, "    df[i][%d]= res;\n", i);
    }  

  fprintf(fout, "  }\nreturn(OK);\n}\n");
  return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : stokPhi                  |
| Role                  : stocke la fonction calcphi
| Parametres d'entree   : fout modSpecif derivees  |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 22            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : erreAnal ecrTest          |
| Fonctions appelantes : geneSubr                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TLogic stokPhi(fout, modSpecif, derivees)
     FILE * fout;
     TModSpecif modSpecif;
     TDerivees derivees;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i, j, k;
  TLogic test;
  TChaine lieuerr;
  
  /* cree l'entete de la fonction calcphi */
  fprintf(fout,
	  "\nshort int calcphi_(nbt, nbg, nbl, nbc, t, g, x, Fsedo, dFsedo, f, df, le, ie)\n");
  fprintf(fout, "      short int nbt, nbg, nbl, nbc;\n");
  fprintf(fout, "      double *t, *g, **x, *f, **df, ***Fsedo, ***dFsedo;\n");
  fprintf(fout, "      short int *le, *ie;\n");
  fprintf(fout, "{\nshort int i;\ndouble res;\n");
  if (modSpecif.Auxiliaire.nbele>0)
    {
      fprintf(fout, "double a[%d], da[%d][%d];\n", modSpecif.Auxiliaire.nbele,
	      modSpecif.Auxiliaire.nbele, modSpecif.IdeTheta.nbele);
    }

  fprintf(fout, "ECRTRACE(\"calcphi\");\n");

  fprintf(fout, "for (i=0; i<nbl; i++)\n  {\n");
  
  /* stocke les auxiliaires */
  lieuerr=LA;
  for (i=0; i<modSpecif.Auxiliaire.nbele; i++)
    {
      test=FAUX;
      k=0;
      while (test==FAUX && k<modSpecif.AuxResp.nbele)
	{
	  if (modSpecif.Auxiliaire.UnAux[i].ident==
	      modSpecif.AuxResp.donnees[k])
	    {
	      test=VRAI;
	    }
	  k++;
	}
      if (test==VRAI)
	{
	  if (ecrTest(modSpecif.Auxiliaire.UnAux[i].arbre, fout,
		      lieuerr, modSpecif, -1)==FAUX)
	    {
	      return(FAUX);
	    }
	  fprintf(fout, "    a[%d]= res;\n", i);
	}
    }

  /* stocke f */
  lieuerr=LF;
  if (ecrTest(modSpecif.IdeF.arbre, fout, lieuerr, modSpecif, -1)==FAUX)
    {
      return(FAUX);
    }
  fprintf(fout, "    f[i]= res;\n");

  /* stocke les derivees des auxiliaires */
  lieuerr=LDADT;
  for (i=0; i<modSpecif.Auxiliaire.nbele; i++)
    {
      test=FAUX;
      k=0;
      while (test==FAUX && k<modSpecif.AuxResp.nbele)
	{
	  if (modSpecif.Auxiliaire.UnAux[i].ident==
	      modSpecif.AuxResp.donnees[k])
	    {
	      test=VRAI;
	    }
	  k++;
	}
      if (test==VRAI)
	{
	  for (j=0; j<modSpecif.IdeTheta.nbele; j++)
	    {
	      if (ecrTest(derivees.DerivAux.arbre[j*modSpecif.Auxiliaire.
						  nbele+i],
			  fout, lieuerr, modSpecif, j)==FAUX)
		{
		  return(FAUX);
		}
	      fprintf(fout, "    da[%d][%d]= res;\n", i, j);
	    }
	}
    }

  lieuerr=LDFDT;
  for (i=0; i<modSpecif.IdeTheta.nbele; i++)
    {
      /* stocke la derivee de f pour theta[i] */
      if (ecrTest(derivees.DerivF.arbre[i], fout, lieuerr, modSpecif, i)
	==FAUX)
	{
	  return(FAUX);
	}
      fprintf(fout, "    df[i][%d]= res;\n", i);
    }  

  fprintf(fout, "  }\nreturn(OK);\n}\n");
  return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : stokSedo                 |
| Role                  : stocke la fonction calcodes
| Parametres d'entree   : fout modSpecif derivees  |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 22            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : erreAnal ecrTestDF        |
| Fonctions appelantes : geneSubr                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TLogic stokSedo(fout, modSpecif, derivees)
     FILE * fout;
     TModSpecif modSpecif;
     TDerivees derivees;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i, j, k, nbt;
  TLogic test;
  
  nbt=modSpecif.IdeCondInit.nbele+modSpecif.NbThetaSedo;


  /* cree l'entete de la fonction calcphi */
  fprintf(fout, "\nshort int calcodes_(neq, t, y, ydot)\n");
  fprintf(fout, "      int *neq;\n");
  fprintf(fout, "      double *t, *y, *ydot;\n");
  fprintf(fout, "{\nshort int nbobs, idobs;\ndouble res;\n");
  if (modSpecif.Auxiliaire.nbele>0)
    {
      fprintf(fout, "double a[%d], da[%d][%d];\n", modSpecif.Auxiliaire.nbele,
	      modSpecif.Auxiliaire.nbele, modSpecif.IdeTheta.nbele);
    }

  fprintf(fout, "ECRTRACE(\"calcodes\");\n");

  fprintf(fout, "nbobs=neq[4];\nidobs=neq[3];\n");
  
  /* stocke les auxiliaires */
  for (i=0; i<modSpecif.Auxiliaire.nbele; i++)
    {
      test=FAUX;
      k=0;
      while (test==FAUX && k<modSpecif.AuxSedo.nbele)
	{
	  if (modSpecif.Auxiliaire.UnAux[i].ident==
	      modSpecif.AuxSedo.donnees[k])
	    {
	      test=VRAI;
	    }
	  k++;
	}
      if (test==VRAI)
	{
	  if (ecrTestDF(modSpecif.Auxiliaire.UnAux[i].arbre, fout,
			modSpecif, -1)==FAUX)
	    {
	      return(FAUX);
	    }
	  fprintf(fout, "a[%d]= res;\n", i);
	}
    }

  /* stocke dF */
  for (i=0; i<modSpecif.IdeLesDF.nbele; i++)
    {
      if (ecrTestDF(modSpecif.IdeLesDF.UnDF[i].arbre, fout,
		    modSpecif, -1)==FAUX)
	{
	  return(FAUX);
	}
      fprintf(fout, "ydot[%d]= res;\n", i);
    }

  /* stocke les derivees des auxiliaires */
  for (i=0; i<modSpecif.Auxiliaire.nbele; i++)
    {
      test=FAUX;
      k=0;
      while (test==FAUX && k<modSpecif.AuxSedo.nbele)
	{
	  if (modSpecif.Auxiliaire.UnAux[i].ident==
	      modSpecif.AuxSedo.donnees[k])
	    {
	      test=VRAI;
	    }
	  k++;
	}
      if (test==VRAI)
	{
	  for (j=0; j<nbt; j++)
	    {
	      if (ecrTestDF(derivees.DerivAux.arbre[j*modSpecif.Auxiliaire.
						  nbele+i],
			  fout, modSpecif, j)==FAUX)
		{
		  return(FAUX);
		}
	      fprintf(fout, "da[%d][%d]= res;\n", i, j);
	    }
	}
    }

  k=modSpecif.IdeLesDF.nbele;
  for (j=0; j<modSpecif.IdeLesDF.nbele; j++)
    {
      for (i=0; i<nbt; i++)
	{
	  /* stocke la derivee de dF pour theta[i] */
	  if (ecrTestDF(derivees.DerivSedo.
			arbre[i*modSpecif.IdeLesDF.nbele+j],
			fout, modSpecif, i)
	      ==FAUX)
	    {
	      return(FAUX);
	    }
	  fprintf(fout, "ydot[%d]= res;\n", k);
	  k++;
	}
    }

  fprintf(fout, "return(OK);\n}\n");
  return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : stokVari                 |
| Role                  : stocke la fonction calcv |
| Parametres d'entree   : fout modSpecif derivees  |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 22            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : erreAnal                  |
| Fonctions appelantes : geneSubr                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TLogic stokVari(fout, modSpecif, derivees)
     FILE * fout;
     TModSpecif modSpecif;
     TDerivees derivees;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i, j, k;
  TLogic test;
  TChaine lieuerr;
  
  /* cree l'entete de la fonction calcv */
  fprintf(fout, "\nshort int calcv_(nbt, nbb, nbg, nbl, nbc, t, b, g,\n");
  fprintf(fout, "                f, df, x, v, dtv, dbv, le, ie)\n");
  fprintf(fout, "      short int nbt, nbb, nbg, nbl, nbc;\n");
  fprintf(fout,
	  "      double *t, *b, *g, **x, *v, **dtv, **dbv, *f, **df;\n");
  fprintf(fout, "      short int *le, *ie;\n");
  fprintf(fout, "{\nshort int i;\ndouble res;\n");
  if (modSpecif.Auxiliaire.nbele>0)
    {
      fprintf(fout, "double a[%d], da[%d][%d];\n", modSpecif.Auxiliaire.nbele,
	      modSpecif.Auxiliaire.nbele, (modSpecif.IdeTheta.nbele +
					   modSpecif.IdeBeta.nbele));
    }

  fprintf(fout, "ECRTRACE(\"calcv\");\n");

  fprintf(fout, "for (i=0; i<nbl; i++)\n  {\n");
  
  /* stocke les auxiliaires */
  lieuerr=LA;
  for (i=0; i<modSpecif.Auxiliaire.nbele; i++)
    {
      test=FAUX;
      k=0;
      while (test==FAUX && k<modSpecif.AuxVar.nbele)
	{
	  if (modSpecif.Auxiliaire.UnAux[i].ident==
	      modSpecif.AuxVar.donnees[k])
	    {
	      test=VRAI;
	    }
	  k++;
	}
      if (test==VRAI)
	{
	  if (ecrTest(modSpecif.Auxiliaire.UnAux[i].arbre, fout, lieuerr,
		      modSpecif, -1)==FAUX)
	    {
	      return(FAUX);
	    }
	  fprintf(fout, "    a[%d]= res;\n", i);
	}
    }

  /* stocke V */
  lieuerr=LERRV;
  if (ecrTest(modSpecif.IdeV.arbre, fout, lieuerr, modSpecif, -1)==FAUX)
    {
      return(FAUX);
    }
  fprintf(fout, "    v[i]= res;\n");

  /* stocke les derivees des auxiliaires */
  for (i=0; i<modSpecif.Auxiliaire.nbele; i++)
    {
      lieuerr=LDADT;
      test=FAUX;
      k=0;
      while (test==FAUX && k<modSpecif.AuxVar.nbele)
	{
	  if (modSpecif.Auxiliaire.UnAux[i].ident==
	      modSpecif.AuxVar.donnees[k])
	    {
	      test=VRAI;
	    }
	  k++;
	}
      if (test==VRAI)
	{
	  for (j=0; j<modSpecif.IdeTheta.nbele; j++)
	    {
	      if (ecrTest(derivees.DerivAux.arbre[j*modSpecif.Auxiliaire.
						  nbele+i],
			  fout, lieuerr, modSpecif, j)==FAUX)
		{
		  return(FAUX);
		}
	      fprintf(fout, "    da[%d][%d]= res;\n", i, j);
	    }
	  lieuerr=LDADB;
	  for (j=modSpecif.IdeTheta.nbele;
	       j<modSpecif.IdeTheta.nbele+modSpecif.IdeBeta.nbele; j++)
	    {
	      if (ecrTest(derivees.DerivAux.arbre[j*modSpecif.Auxiliaire.
						  nbele+i],
			  fout, lieuerr, modSpecif, j)==FAUX)
		{
		  return(FAUX);
		}
	      fprintf(fout, "    da[%d][%d]= res;\n", i, j);
	    }
	}
    }

  for (i=0; i<modSpecif.IdeTheta.nbele; i++)
    {
      lieuerr=LDVDT;
      /* stocke la derivee de v pour theta[i] */
      if (ecrTest(derivees.DerivV.arbre[i], fout, lieuerr, modSpecif, i)
	==FAUX)
	{
	  return(FAUX);
	}
      fprintf(fout, "    dtv[i][%d]= res;\n", i);
    }  

  for (i=0; i<modSpecif.IdeBeta.nbele; i++)
    {
      /* stocke la derivee de v pour beta[i] */
      lieuerr=LDVDB;
      if (ecrTest(derivees.DerivV.arbre[i+modSpecif.IdeTheta.nbele],
	fout, lieuerr, modSpecif, i+modSpecif.IdeTheta.nbele)==FAUX)
	{
	  return(FAUX);
	}
      fprintf(fout, "    dbv[i][%d]= res;\n", i);
    }  

  fprintf(fout, "  }\nreturn(OK);\n}\n");
  return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : copiDescr                |
| Role                  : copie le fichier de description en tete de la
                          subroutine               |
| Parametres d'entree   : fin fout NomDescr       |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 22            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : erreAnal                  |
| Fonctions appelantes : geneSubr                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TLogic copiDescr(fin, fout, NomDescr)
     FILE *fout, *fin;
     TChaine NomDescr;

/*--------------- Fin identification fonction ----*/

{
  int c;

  fprintf(fout, "/*----------------------------------------\n");
  fprintf(fout, "description file: %s\n\n", NomDescr);
  while((c=getc(fin))!=EOF)
    {
      putc(c, fout);
    }
  fprintf(fout, "----------------------------------------*/\n");
/*  fclose(fin); */
return(VRAI);

}

/*--------------- Identification fonction ----------
| Nom de la fonction    : geneSubr                 |
| Role                  : stocke les arbres dans des subroutines C
| Parametres d'entree   : GModSpecif Derivees NomDescr
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 22            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : stokResp stokVari stokCS erreAnal copiDescr
| Fonctions appelantes : analDer                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt geneSubr( TModSpecif GModSpecif, 
		   TDerivees Derivees, TChaine NomDescr)

/*--------------- Fin identification fonction ----*/

{
  FILE *fin, *fout;

  int c;

  /* AB: Pour que ca marche sous linux, - et aussi pour que le fichier
en sortie de analder ne soit pas automatiquement pathname-du-fichier-en-entree.c
ce qui oblige  avoir les droits w sur le directory de celui-ci,
on fait la gnration du programme c sur la sortie standard  via un fichier dans /tmp
On remplace:
  TChaine nom;
  nom=(TChaine)
    malloc((strlen(GModSpecif.PPident.ident[GModSpecif.NomFich].libelle)
	    +2) * sizeof(TChar));
ce qui , pour que ca marche sur linux, avait t remplac par
  nom=(TChaine)
    malloc((strlen(GModSpecif.PPident.ident[GModSpecif.NomFich].libelle)
	    +3 ) * sizeof(TChar));

  if (nom==NULL)
    {
      erreAnal(EMEM, NULL);
      return(FAUX);
    }
  strcpy(nom,NomDescr);
  strcat(nom, ".c");
  fout=fopen(nom, "w");
  */

  fout=tmpfile();
  if (fout==NULL)
    {
      erreAnal(EFOUT, "/tmp/...");
      return(FAUX);
    }

  fin=fopen(NomDescr, "r");
  if (fin==NULL)
    {
      erreAnal(EFIN, NomDescr);
      return(FAUX);
    }

  /* recopie du fichier de description en tete du fichier */
  if (copiDescr(fin, fout, NomDescr)==FAUX)
    {
      return(FAUX);
    }

  /* entete general a toutes les fonctions calc */
  fprintf(fout, "#include <stdio.h>\n#include <math.h>\n");
  fprintf(fout, "#include \"nltypes.h\"\n#include \"nlcodes.h\"\n");
  fprintf(fout, "#include \"nlmacros.h\"\n#include \"nlfonc.h\"\n");
  
  if (GModSpecif.Resp==OUI && GModSpecif.CasSedo==NON)
    {
      if (stokResp(fout, GModSpecif, Derivees)==FAUX)
	{
	  return(FAUX);
	}
    }
  else
    {
      fprintf(fout, "\nshort int calcf_()\n{}\n");
    }
  if (GModSpecif.Vari==OUI)
    {
      if (stokVari(fout, GModSpecif, Derivees)==FAUX)
	{
	  return(FAUX);
	}
    }
  else
    {
      fprintf(fout, "\nshort int calcv_()\n{}\n");
    }
  if (GModSpecif.CasSedo==OUI)
    {
      if (stokPhi(fout, GModSpecif, Derivees)==FAUX)
	{
	  return(FAUX);
	}
      if (stokSedo(fout, GModSpecif, Derivees)==FAUX)
	{
	  return(FAUX);
	}
      fprintf(fout, "\ncalcjac_()\n{}\n");
    }
  else
    {
      fprintf(fout, "\nshort int calcodes_()\n{}\n");
      fprintf(fout, "\nshort int calcjac_()\n{}\n");
      fprintf(fout, "\nshort int calcphi_()\n{}\n");
    }
  fclose(fin);

  /* AB: */
  rewind(fout);

  while ((c = getc(fout)) != EOF) {
       putc(c, stdout);
  }

  return(VRAI);
}
