/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : prepEval.c                |
| Date                 : 15 decembre 1992          |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : O. Nicole                 |
| Role                 : remplace dans les arbres tous les appels a des
                         identificateurs par un lien sur l'arbre d'expression
                         dans le cas des aux, f ou derivees, sur un arbre
                         constante dans le cas des X, theta, beta, gamma...
| Reference conception : DCP module 31             |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/

#include "nltypes.h"
#include "nlcodes.h"
#include "dftypes.h"
#include "dfcodes.h"
#include "nlmacros.h"

/*--------------- VARIABLES EXTERNES -------------*/

extern TModSpecif GModSpecif;
extern TDerivees Derivees;

/*--------------- FONCTIONS EXTERNES -------------*/
extern void *malloc(size_t size);
extern TShortInt erreAnal(TChaine message, TChaine param);

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/

/*--------------- Identification fonction ----------
| Nom de la fonction    : bclCop                   |
| Role                  : fonction recursive de copie d'un arbre
| Parametres d'entree   : arbre                    |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 31            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : copieAux                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TArbre * bclCop(arbre)
     TArbre * arbre;

/*--------------- Fin identification fonction ----*/

{
  /*  TArbre * noeud; */

/**  if (arbre==NULL)
    {
      return(NULL);
    }
  noeud=(TArbre*) malloc(sizeof(TArbre));
  if (noeud==NULL)
    {
      erreAnal(EMEM, NULL);
      return(NULL);
    }
  noeud->fonction=arbre->fonction;
  noeud->valeur=arbre->valeur;
  noeud->filsG=bclCop(arbre->filsG);
  noeud->filsD=bclCop(arbre->filsD);
  return(noeud);
  */
  return(arbre);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : copieAux                 |
| Role                  : fait une copie des expressions auxiliaires et
                          de leurs derivees        |
| Parametres d'entree   :                          |
| Parametres de sortie  :                          |
|    - copie : tableau contenant les copies des expressions auxiliaires
|              et de leurs derivees
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 31            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : bclCop                    |
| Fonctions appelantes : prepEval                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt copieAux(copie)
     TArbre ** copie;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i;
  for (i=0; i<GModSpecif.Auxiliaire.nbele; i++)
    {
      if ((copie[i]=bclCop(GModSpecif.Auxiliaire.UnAux[i].arbre)) == NULL)
	{
	  return(FAUX);
	}
    }
  for (i=0; i<Derivees.DerivAux.nbele; i++)
    {
      if ((copie[i+GModSpecif.Auxiliaire.nbele]=
	  bclCop(Derivees.DerivAux.arbre[i])) == NULL)
	{
	  return(FAUX);
	}
    }
  return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : FchoixArb                |
| Role                  : retourne un pointeur sur un arbre correspondant
                          a l'identificateur pour le modele simple
| Parametres d'entree   :                          |
|    - ident : numero de l'identificateur a remplacer, ce numero est
|              negatif. Si PPident.nbele<ident<=0 c'est un identificateur
|              "simple", si ident<=PPident.nbele c'est la derivee d'un
|              identificateur
|    - param : numero du parametre par rapport auquel on derive, sert a
|              retourver la bonne expression dans le cas ou
|              ident<=PPident.nbele, dans les autres cas il n'est pas
|              utilise
|    - aux   : tableau contenant une copie des expressions auxiliaires et
|              de leurs derivees (on est oblige de faire une copie car
|              tout n'est pas traite pareil dans tous les cas)
|    - type  : permet de definir les cas, CLF, CLV...
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 31            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : FremplIde prepEval        |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TArbre * FchoixArb(ident, param, aux, type)
     TShortInt ident;
     TShortInt param;
     TArbre ** aux;
     TShortInt type;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i;
  TArbre * Noeud;
  TLogic indicder;

  TShortInt nbX, nbG, nbT, nbF, dbX, dbG, dbT;
  TShortInt nbTot;

  nbX=GModSpecif.IdeX.nbele;
  nbG=GModSpecif.IdeGamF.nbele;
  nbT=GModSpecif.NbThetaSedo+GModSpecif.IdeCondInit.nbele;
  nbF=GModSpecif.IdeLesF.nbele;
  dbT=1+(1+nbT)*nbF;

  /* AB, 26/05/98 remplace: 
  dbG=dbT+nbT;
      par: */

  nbTot = GModSpecif.IdeTheta.nbele;
/* nbTot=nbre total de parametres y compris ceux qui ne sont pas dans le sedo
ni condinit */
  dbG=dbT+ nbTot ;

  dbX=dbG+nbG;

  /* on prend un numero positif */
  ident=-ident;
  indicder=FAUX;
  if (ident >= GModSpecif.PPident.nbele)
    {
      ident=ident-GModSpecif.PPident.nbele;
      indicder=VRAI;
    }
  switch(GModSpecif.PPident.ident[ident].type)
    {
    case IDECONST:
      /* on construit un bout d'arbre ne contenant que la valeur de la
	 constante */
      Noeud=(TArbre *) malloc(sizeof(TArbre));
      if (Noeud==NULL)
	{
	  erreAnal(EMEM, NULL);
	  return(NULL);
	}
      Noeud->fonction=CONSTANTE;
      Noeud->filsG=NULL;
      Noeud->filsD=NULL;
      i=0;
      while (GModSpecif.Constantes.UneCst[i].ident!=ident)
	{
	  i++;
	}
      Noeud->valeur=GModSpecif.Constantes.UneCst[i].valeur;
      return(Noeud);
      break;
    case IDEAUX:
      i=0;
      while (GModSpecif.Auxiliaire.UnAux[i].ident!=ident)
	{
	  i++;
	}
      if (indicder==VRAI)
	{
	  return(aux[(param+1) * GModSpecif.Auxiliaire.nbele+i]);
	}
      else
	{
	  return(aux[i]);
	}
      break;
    case IDEF:
      if (type==CLDF)
	{
	  /* on ne passera jamais ici normalement */
	  erreAnal(EARBRE, NULL);
	  break;
	}
      if (indicder==VRAI)
	{
	  switch(type)
	    {
	    case CLF:
	      return(Derivees.DerivF.arbre[param]);
	      break;
	    case CLV:
	      return(GModSpecif.ParamEntre[GModSpecif.IdeX.nbele +
					   GModSpecif.IdeTheta.nbele +
					   GModSpecif.IdeBeta.nbele +
					   GModSpecif.IdeGamV.nbele + 1 +
					   param]);
	      break;
	    }
	}
      else
	{
	  switch(type)
	    {
	    case CLF:
	      return(GModSpecif.IdeF.arbre);
	      break;
	    case CLV:
	      return(GModSpecif.ParamEntre[GModSpecif.IdeX.nbele +
					   GModSpecif.IdeTheta.nbele +
					   GModSpecif.IdeBeta.nbele +
					   GModSpecif.IdeGamV.nbele]);
	      break;
	    }
	}
      break;
    case IDEX:
      i=0;
      while (GModSpecif.IdeX.donnees[i]!=ident)
	{
	  i++;
	}
      if (type==CLDF)
	{
	  return(GModSpecif.ParamEntre[dbX+i]);
	}
      else
	{
	  return(GModSpecif.ParamEntre[i]);
	}
      break;
    case IDETHETA:
      i=0;
      while (GModSpecif.IdeTheta.donnees[i]!=ident)
	{
	  i++;
	}
      if (type==CLDF)
	{
	  return(GModSpecif.ParamEntre[dbT+i]);
	}
      else
	{
	  return(GModSpecif.ParamEntre[i+GModSpecif.IdeX.nbele]);
	}
      break;
    case IDEBETA:
      if (type==CLDF)
	{
	  /* on ne passera jamais ici normalement */
	  erreAnal(EARBRE, NULL);
	  break;
	}
      i=0;
      while (GModSpecif.IdeBeta.donnees[i]!=ident)
	{
	  i++;
	}
      return(GModSpecif.ParamEntre[i + GModSpecif.IdeX.nbele +
				   GModSpecif.IdeTheta.nbele]);
      break;
    case IDEGAMF:
      i=0;
      while (GModSpecif.IdeGamF.donnees[i]!=ident)
	{
	  i++;
	}
      if (type==CLDF)
	{
	  return(GModSpecif.ParamEntre[dbG+i]);
	}
      else
	{
	  return(GModSpecif.ParamEntre[i + GModSpecif.IdeX.nbele +
				       GModSpecif.IdeTheta.nbele]);
	}
      break;
    case IDEGAMV:
      if (type==CLDF)
	{
	  /* on ne passera jamais ici normalement */
	  erreAnal(EARBRE, NULL);
	  break;
	}
      i=0;
      while (GModSpecif.IdeGamV.donnees[i]!=ident)
	{
	  i++;
	}
      return(GModSpecif.ParamEntre[i + GModSpecif.IdeX.nbele +
				   GModSpecif.IdeTheta.nbele +
				   GModSpecif.IdeBeta.nbele]);
      break;
    case IDEVARINT:
      if (type!=CLDF)
	{
	  /* on ne passera jamais ici normalement */
	  erreAnal(EARBRE, NULL);
	  break;
	}
      return(GModSpecif.ParamEntre[0]);
      break;
    case IDELESF:
      if (type!=CLDF)
	{
	  /* on ne passera jamais ici normalement */
	  erreAnal(EARBRE, NULL);
	  break;
	}
      i=0;
      while(GModSpecif.IdeLesF.donnees[i]!=ident)
	{
	  i++;
	}
      if (indicder!=VRAI)
	{
	  /* cas d'un dF */
	  return(GModSpecif.ParamEntre[1+i]);
	}
      else
	{
	  /* case d'un ddF/dT */
	  return(GModSpecif.ParamEntre[1+nbF+i*nbT+param]);
	}
      break;
    case IDEVALINT:
      /* on ne devrait pas trouver ici d'autres identificateurs */
      erreAnal(EARBRE, NULL);
      break;
    }
  return(NULL);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : FremplIde                |
| Role                  : remplace un identificateur par un lien vers 
                          l'arbre correspondant pour un modele simple
| Parametres d'entree   :                          |
|    - arbre : arbre sur lequel est fait le remplacement des identificateurs
|    - param : numero du parametre par rapport auquel on derive si l'arbre
|              traite est un arbre de derivee
|    - aux   : tableau contenant une copue des auxiliaires et de 
|              leurs derivees
|    - type  : type de l'arbre traite : CLF, CLV...
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 31            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : FchoixArb                 |
| Fonctions appelantes : prepF prepV prepAux prepSedo erreAnal
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TArbre * FremplIde(arbre, param, aux, type)
     TArbre * arbre;
     TShortInt param;
     TArbre ** aux;
     TShortInt type;

/*--------------- Fin identification fonction ----*/

{
  TShortInt ident, i, j, nbF;
  TShortInt indicder, offset;
  TArbre * noeud;
  
  if (arbre!=NULL)
    {
      if (arbre->fonction <=0)
	{
	  /* cas ou l'expression est reduite a un identificateur, par exemple:
	     f=aux1;
	     aux1=aux2;... */
	  return(FchoixArb(arbre->fonction, param, aux, type));
	}
      else if (arbre->fonction==VALINT)
	    {
	      if (type==CLDF)
		{
		  /* on ne passera jamais ici normalement */
		  return(NULL);
		}
	      /* cas a traiter a part, on a reconnu F[T] ou dF[T] */
	      ident=-arbre->filsG->fonction;
	      if (ident >= GModSpecif.PPident.nbele)
		{
		  /* cas de dF[T] */
		  indicder=VRAI;
		  ident=ident-GModSpecif.PPident.nbele;
		}
	      else
		{
		  /* cas de F[T] */
		  indicder=FAUX;
		}
	      /* il faut determiner quel F est concerne */
	      i=0;
	      while (GModSpecif.IdeLesF.donnees[i]!=ident)
		{
		  i++;
		}
	      /* il faut determiner quel T est concerne */
	      j=0;
	      ident=-arbre->filsD->fonction;
	      while (GModSpecif.IdeValInt.donnees[j]!=ident)
		{
		  j++;
		}
	      /* on calcule un decalage par raport au debut de ParamEntre */
	      if (indicder==VRAI)
		{
		  /* cas de dF[T] */
		  if (param>= GModSpecif.NbThetaSedo +
		      GModSpecif.IdeCondInit.nbele)
		    {
		      /* c'est la derivee d'une equation du sedo par rapport
			 a un parametre qui n'intervient pas dans le sedo,
			 ca vaut 0 (ie un arbre CONSTANTE de valeur 0) */
		      noeud=(TArbre *)malloc(sizeof(TArbre));
		      if (noeud==NULL)
			{
			  erreAnal(EMEM, NULL);
			  return(NULL);
			}
		      noeud->filsG=NULL;
		      noeud->filsD=NULL;
		      noeud->fonction=CONSTANTE;
		      noeud->valeur=(TDouble)0;
		      return(noeud);
		    }
		  offset=GModSpecif.IdeX.nbele + GModSpecif.IdeTheta.nbele +
		    GModSpecif.IdeGamF.nbele + GModSpecif.IdeValInt.nbele *
		    GModSpecif.IdeLesF.nbele;

		  /* nbF= nbF*(nbThetaSedo+nbCondInit) */
		  nbF=GModSpecif.IdeLesF.nbele *
		    (GModSpecif.NbThetaSedo + GModSpecif.IdeCondInit.nbele);

		  /* calcule du decalage dans la j-eme ligne de
		     dF[][j][i]/dTheta_param*/
		  i=i*(GModSpecif.NbThetaSedo+GModSpecif.IdeCondInit.nbele) +
		    param;
		  return(GModSpecif.ParamEntre[offset + j*nbF + i]);
		}
	      else
		{
		  /* cas de F[T] */
		  offset=GModSpecif.IdeX.nbele+GModSpecif.IdeTheta.nbele+
		    GModSpecif.IdeGamF.nbele;
		  return(GModSpecif.ParamEntre[offset +
					       j*GModSpecif.IdeValInt.nbele +
					       i]);
		}
	    }
      else
	{
	  arbre->filsG=FremplIde(arbre->filsG, param, aux, type);
	  arbre->filsD=FremplIde(arbre->filsD, param, aux, type);
	  return(arbre);
	}
    }
  /* AB, 28/06/2001 on ne passera jamais ici normalement */
		  return(NULL);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : prepAux                  |
| Role                  : fait le rattachement des arbres sur une copie des
|                         auxiliaires
| Parametres d'entree   :                          |
|    - liste : liste des auxiliaires a traiter effectivement
|    - type : type de l'expression traitee (CLF, CLV...)
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
|    - aux : copie des auxiliaires a traiter       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : FremplIde                 |
| Fonctions appelantes : prepF prepV prepSedo      |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt prepAux(aux, liste, type)
     TArbre **aux;
     TVectShort liste;
     TShortInt type;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i, j, ii, k;
  TLogic test;
  
  for (i=0; i<GModSpecif.Auxiliaire.nbele; i++)
    {
      if (aux[i] == NULL)
	{
	  erreAnal(EARBRE, NULL);
	  return(FAUX);
	}
      k=0;
      test=FAUX;
      while (test==FAUX && k<liste.nbele)
	{
	  if (GModSpecif.Auxiliaire.UnAux[i].ident==liste.donnees[k])
	    {
	      test=VRAI;
	    }
	  k++;
	}
      if (test==VRAI)
	{
	  aux[i]=FremplIde(aux[i], 0, aux, type);
	}
    }
  /* derivees des auxiliaires */
  for (i=0; i<GModSpecif.Auxiliaire.nbele; i++)
    {
      k=0;
      test=FAUX;
      while (test==FAUX && k<liste.nbele)
	{
	  if (GModSpecif.Auxiliaire.UnAux[i].ident==liste.donnees[k])
	    {
	      test=VRAI;
	    }
	  k++;
	}
      if (test==VRAI)
	{
	  for (j=0; j<GModSpecif.IdeTheta.nbele+GModSpecif.IdeBeta.nbele;
	       j++)
	    {
	      ii=(j+1)*GModSpecif.Auxiliaire.nbele+i;
	         /* (j+1) car les NbAux premieres equations sont les
		    expressions auxiliaires simples */
	      if (aux[ii]==NULL)
		{
		  erreAnal(EARBRE, NULL);
		  return(FAUX);
		}
	      aux[ii]=FremplIde(aux[ii], j, aux, type);
	    }
	}
      }
  return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : prepF                    |
| Role                  : prepare le modele simple |
| Parametres d'entree   :                          |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : FremplIde prepAux         |
| Fonctions appelantes : prepEval                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt prepF()

/*--------------- Fin identification fonction ----*/

{
  TArbre ** aux;
  TShortInt i;
  
  /* cas des auxiliaires */
  if (GModSpecif.Auxiliaire.nbele > 0)
    {
      /* on cree et copie les expressions auxiliaires */
      aux=(TArbre**)malloc((GModSpecif.Auxiliaire.nbele +
			    Derivees.DerivAux.nbele) *
			   sizeof(TArbre*));
      if (aux==NULL)
	{
	  erreAnal(EMEM, NULL);
	  return(FAUX);
	}
      if(copieAux(aux)==FAUX)
	{
	  return(FAUX);
	}
      
      /* on prepare les expressions auxiliaires */
      if (prepAux(aux, GModSpecif.AuxResp, CLF) == FAUX)
	{
	  return(FAUX);
	}
    }

  /* F */
  if (GModSpecif.IdeF.arbre==NULL)
    {
      erreAnal(EARBRE, NULL);
      return(FAUX);
    }
  GModSpecif.IdeF.arbre=FremplIde(GModSpecif.IdeF.arbre, 0, aux, CLF);
  
  /* derivees de F */
  for (i=0; i<GModSpecif.IdeTheta.nbele; i++)
    {
      if (Derivees.DerivF.arbre[i]==NULL)
	{
	  erreAnal(EARBRE, NULL);
	  return(FAUX);
 	}
      Derivees.DerivF.arbre[i]=
	FremplIde(Derivees.DerivF.arbre[i], i, aux, CLF);
    }
  return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : prepV                    |
| Role                  : prepare les expressions de V et dV
| Parametres d'entree   :                          |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : prepAux FremplIde         |
| Fonctions appelantes : prepEval                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt prepV()

/*--------------- Fin identification fonction ----*/

{
  TShortInt i;
  TArbre **aux;

  /* cas des auxiliaires */
  if (GModSpecif.Auxiliaire.nbele > 0)
    {
      /* copie des expressions auxiliaires */
      aux=(TArbre**)malloc((GModSpecif.Auxiliaire.nbele +
				  Derivees.DerivAux.nbele) *
				 sizeof(TArbre*));
      if (aux==NULL)
	{
	  erreAnal(EMEM, NULL);
	  return(FAUX);
	}
      if(copieAux(aux)==FAUX)
	{
	  return(FAUX);
	}
      
      /* on prepare les expressions auxiliaires */
      if (prepAux(aux, GModSpecif.AuxVar, CLV) == FAUX)
	{
	  return(FAUX);
	}
    }
  
  /* V */
  if (GModSpecif.IdeV.arbre==NULL)
    {
      erreAnal(EARBRE, NULL);
      return(FAUX);
    }
  GModSpecif.IdeV.arbre=FremplIde(GModSpecif.IdeV.arbre, 0, aux, CLV);
  
  /* derivees de V */
  for (i=0; i<GModSpecif.IdeTheta.nbele+GModSpecif.IdeBeta.nbele; i++)
    {
      if (Derivees.DerivV.arbre[i]==NULL)
	{
	  erreAnal(EARBRE, NULL);
	  return(FAUX);
	}
      Derivees.DerivV.arbre[i]=
	FremplIde(Derivees.DerivV.arbre[i], i, aux, CLV);
      }
  return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : prepSedo                 |
| Role                  : prepare les expressions du sedo
| Parametres d'entree   :                          |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : prepAux FremplIde         |
| Fonctions appelantes : prepEval                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt prepSedo()

/*--------------- Fin identification fonction ----*/

{
  TArbre ** aux;
  TShortInt i, j, nbF, nbT;
  
  /* cas des auxiliaires */
  if (GModSpecif.Auxiliaire.nbele > 0)
    {
      /* on cree et copie les expressions auxiliaires */
      aux=(TArbre**)malloc((GModSpecif.Auxiliaire.nbele +
			    Derivees.DerivAux.nbele) *
			   sizeof(TArbre*));
      if (aux==NULL)
	{
	  erreAnal(EMEM, NULL);
	  return(FAUX);
	}
      if(copieAux(aux)==FAUX)
	{
	  return(FAUX);
	}
      
      /* on prepare les expressions auxiliaires */
      if (prepAux(aux, GModSpecif.AuxSedo, CLDF) == FAUX)
	{
	  return(FAUX);
	}
    }

  /* dF */
  nbF=GModSpecif.IdeLesF.nbele;
  for (i=0; i<nbF; i++)
    {
      if (GModSpecif.IdeLesDF.UnDF[i].arbre==NULL)
	{
	  erreAnal(EARBRE, NULL);
	  return(FAUX);
	}
      GModSpecif.IdeLesDF.UnDF[i].arbre=
	FremplIde(GModSpecif.IdeLesDF.UnDF[i].arbre, 0, aux, CLDF);
    }

  /* derivees de ddF/dT */
  nbT=GModSpecif.NbThetaSedo+GModSpecif.IdeCondInit.nbele;
  for (i=0; i<nbT; i++)
    {
      for (j=0; j<nbF; j++)
	{
	  if (Derivees.DerivSedo.arbre[i*nbF+j]==NULL)
	    {
	      erreAnal(EARBRE, NULL);
	      return(FAUX);
	    }
	  Derivees.DerivSedo.arbre[i*nbF+j]=
	    FremplIde(Derivees.DerivSedo.arbre[i*nbF+j], i, aux, CLDF);
	}
    }
  return(VRAI);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : prepEval                 |
| Role                  : supprime des arbres tous les appels a des
                          identificateurs          |
| Parametres d'entree   :                          |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 31            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : prepF                     |
| Fonctions appelantes : analDer                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt prepEval()

/*--------------- Fin identification fonction ----*/

{
  /*  TShortInt i, j, longueur, ii, nbTsedo, nbF, nbTheta, nbBeta, nbGf, nbX; */
 TShortInt i, longueur, nbTsedo, nbF, nbTheta, nbGf, nbX;
 /*  TArbre **copieS1Aux, **copieS2Aux; */

  /* cree un tableau d'arbres reduits a une feuille ou seront stockees
     les valeurs des differents parametres du modele a chaque appel
     aux fonctions calcf, calcv... passant par l'evaluation d'un arbre */
  nbTsedo=GModSpecif.NbThetaSedo+GModSpecif.IdeCondInit.nbele;
  nbF=GModSpecif.IdeLesF.nbele;
  nbTheta=GModSpecif.IdeTheta.nbele;
  nbX=GModSpecif.IdeX.nbele;
  nbGf=GModSpecif.IdeGamF.nbele;
  longueur=nbX + nbTheta + MAX(nbGf, GModSpecif.IdeGamV.nbele) +
    MAX(GModSpecif.IdeBeta.nbele + 1 + nbTheta,
	nbF * GModSpecif.IdeValInt.nbele * (1 + nbTsedo));

  /* AB, 26/05/98: remplace:
  longueur=MAX(longueur, 1+(1+nbTsedo)*nbF+nbTsedo+nbGf+nbX);
   par : */

  longueur=MAX(longueur, 1+(1+nbTsedo)*nbF+MAX(nbTheta,nbTsedo)+nbGf+nbX);

  /* car, il faut que la longueur soit suffisante pour stocker
  tous les parametres, y compris ceux qui ne sont ni dans le sedo
  ni condinit */

  GModSpecif.ParamEntre=(TArbre **) malloc(longueur*sizeof(TArbre *));
  if (GModSpecif.ParamEntre==NULL)
    {
      erreAnal(EMEM, NULL);
      return(FAUX);
    }
  for (i=0; i<longueur; i++)
    {
      /* on initialise les arbres comme etant des feuilles */
      GModSpecif.ParamEntre[i]=(TArbre *) malloc(sizeof(TArbre));
      if (GModSpecif.ParamEntre[i]==NULL)
	{
	  erreAnal(EMEM, NULL);
	  return(FAUX);
	}
      GModSpecif.ParamEntre[i]->filsG=NULL;
      GModSpecif.ParamEntre[i]->filsD=NULL;
      GModSpecif.ParamEntre[i]->fonction=CONSTANTE;
    }
  
  /* cas du modele simple */
  if (GModSpecif.IdeF.arbre!=NULL && GModSpecif.CasSedo==FAUX)
    {
      if (prepF() == FAUX)
	{
	  return(FAUX);
	}
     }

  /* cas de la variance */
  if (GModSpecif.Vari==VRAI)
    {
      if (prepV() == FAUX)
	{
	  return(FAUX);
	}
     }

  /* cas du sedo */
  if (GModSpecif.CasSedo==VRAI)
    {
      /* on traire calcphi */
      if (prepF() == FAUX)
	{
	  return(FAUX);
	}
      /* on traite calcodes */
      if (prepSedo() == FAUX)
	{
	  return(FAUX);
	}
    }
  
  return(VRAI);
}
