/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : evalArb.c                 |
| Date                 : 16 decembre 1992          |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : O. Nicole                 |
| Role                 : evaluation formelle du modele par parcours des
                         arbres                    |
| Reference conception : DCP modules 24-29         |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <float.h>

#include "nltypes.h"
#include "nlcodes.h"
#include "errcodes.h"
#include "dftypes.h"
#include "dfcodes.h"
#include "nlmacros.h"

/*--------------- VARIABLES EXTERNES -------------*/

extern TModSpecif GModSpecif;
extern TDerivees Derivees;

/*--------------- FONCTIONS EXTERNES -------------*/
TShortInt GerMessage( );
/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/

/*--------------- Identification fonction ----------
| Nom de la fonction    : imprErr                  |
| Role                  : sert d'appel a NLWARNING et retourne la valeur NaN
| Parametres d'entree   : fonction x y             |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : evalArb                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

double imprErr(fonction, x, y)
     TShortInt fonction;
     TDouble x, y;

/*--------------- Fin identification fonction ----*/

{
  double r;
  char Mess1[30];
  char Mess2[30];
  
    switch(fonction)
      {
      case LOG:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "log10", Mess1, " ", WNUMER));
	break;
      case LN:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "log", Mess1, " ", WNUMER));
	break;
      case DIV:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "div", Mess1, " ", WNUMER));
	break;
      case EXP:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "exp", Mess1, " ", WNUMER));
	break;
      case PUISSANCE:
	sprintf(Mess1, "%g", x);
	sprintf(Mess2, "%g", y);
	NLWARNING((WARNUM, 3, "pow", Mess1, Mess2,  WNUMER));
	break;
      }

  r=mysignaling_nan(n);
  return(r);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : evalArb                  |
| Role                  : calcule formellement la valeur d'un arbre
| Parametres d'entree   : arbre                    |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 29            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : calcf_ calcv_ calcphi_ calsedo_
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TDouble evalArb(arbre)
     TArbre * arbre;

/*--------------- Fin identification fonction ----*/

{
  TDouble x, y;

  switch(arbre->fonction)
    {
    case CONSTANTE:
      return(arbre->valeur);
      break;
    case PLUS:
      return(evalArb(arbre->filsG) + evalArb(arbre->filsD));
      break;
    case MOINS:
      return(evalArb(arbre->filsG) - evalArb(arbre->filsD));
      break;
    case MULT:
      return(evalArb(arbre->filsG) * evalArb(arbre->filsD));
      break;
    case DIV:
      x=evalArb(arbre->filsD);
      return( (x==(TDouble)0) ?
	     imprErr(DIV, x, 0) :
	     evalArb(arbre->filsG) / x);
      break;
    case LOG:
      x=evalArb(arbre->filsG);
      return( (x<=(TDouble)0) ?
	     imprErr(LOG, x, 0) :
	     log(x)/log((double)10));
      break;
    case TG:
      return(tan(evalArb(arbre->filsG)));
      break;
    case LN:
      x=evalArb(arbre->filsG);
      return( (x<=(TDouble)0) ?
	     imprErr(LN, x, 0) :
	     log(x));
      break;
    case EXP:
      x=evalArb(arbre->filsG);
      return((x>(double)709) ?
	     imprErr(EXP, x, 0) :
	     ((x<(double)-745) ?
	       (double)0 :
		exp(x)));
      break;
    case SIN:
      return(sin(evalArb(arbre->filsG)));
      break;
    case COS:
      return(cos(evalArb(arbre->filsG)));
      break;
    case IF:
      if (arbre->filsD==NULL)
	{
	  return((double)0);
	}
      if (evalArb(arbre->filsG)!=0)
	{
	  return(evalArb(arbre->filsD->filsG));
	}
      else
	{
	  return(evalArb(arbre->filsD->filsD));
	}
      break;
    case OR:
      return(evalArb(arbre->filsG)+evalArb(arbre->filsD));
      break;
    case AND:
      return(evalArb(arbre->filsG)*evalArb(arbre->filsD));
      break;
    case EQ:
      return( (evalArb(arbre->filsG)==evalArb(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case NEQ:
      return( (evalArb(arbre->filsG)!=evalArb(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case LE:
      return( (evalArb(arbre->filsG)<evalArb(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case LEQ:
      return( (evalArb(arbre->filsG)<=evalArb(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case GE:
      return( (evalArb(arbre->filsG)>evalArb(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case GEQ:
      return( (evalArb(arbre->filsG)>=evalArb(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case NOT:
      return( (evalArb(arbre->filsG)==(TDouble)0) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case PUISSANCE:
      x=evalArb(arbre->filsG);
      y=evalArb(arbre->filsD);
      return(((x==0 && y<=0) || (x<0 && (int)y!=y)) ?
	     imprErr(PUISSANCE, x, y) :
	     pow(x, y));
      break;
    case VALINT:
      break;
    }
return (TDouble)0.0;
}




/*--------------- Identification fonction ----------
| Nom de la fonction    : calcf_                   |
| Role                  : calcule la reponse pour un modele simple
| Parametres d'entree   : nbt nbg nbl nbc t g x    |
| Parametres de sortie  : f df ie le               |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 24            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : evalArb                   |
| Fonctions appelantes :                           |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt calcf_(nbt, nbg, nbl, nbc, t, g, x, f, df, le, ie)
      TShortInt nbt, nbg, nbl, nbc;
      TDouble *t, *g, **x, *f, **df;
      TShortInt *le, *ie;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i, xt, j;

  ECRTRACE("calcf");


  /* recopie des parametres d'entree */
  for (i=0; i<nbt; i++)
    {
      GModSpecif.ParamEntre[i+nbc]->valeur=t[i];
    }
  xt=nbc+nbt;
  for (i=0; i<nbg; i++)
    {
      GModSpecif.ParamEntre[i+xt]->valeur=g[i];
    }

  /* boucle sur les nbl observations */
  for (i=0; i<nbl; i++)
    {
      /* recopie des parametres d'entree */
      for (j=0; j<nbc; j++)
	{
	  GModSpecif.ParamEntre[j]->valeur=x[i][j];
	}
      /* evaluation de l'arbre de F */
      f[i]=evalArb(GModSpecif.IdeF.arbre);
      if (finite(f[i])!=1)
	  {
	    *ie=i+1;
	    *le=F;
	    return(ERRMATH);
	  }

      /* evaluation des arbres des dF */
      for (j=0; j<nbt; j++)
	{
	  df[i][j]=evalArb(Derivees.DerivF.arbre[j]);
	  if (finite(df[i][j])!=1)
	    {
	      *ie=i+1;
	      *le=DFDT;
	      return(ERRMATH);
	    }
	}
    }
  return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : calcv_                   |
| Role                  : calcule la reponse pour la variance
| Parametres d'entree   : nbt nbg nbl nbc t g x f df
| Parametres de sortie  : v dtv dbv ie le          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 25            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : evalArb                   |
| Fonctions appelantes :                           |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt calcv_(nbt, nbb, nbg, nbl, nbc, t, b, g, f, df, x, v,
		 dtv, dbv, le, ie)
      TShortInt nbt, nbb, nbg, nbl, nbc;
      TDouble *t, *b, *g, **x, *f, **df, *v, **dtv, **dbv;
      TShortInt *le, *ie;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i, xt, j;

  ECRTRACE("calcv");

  /* recopie des parametres d'entree */
  for (i=0; i<nbt; i++)
    {
      GModSpecif.ParamEntre[i+nbc]->valeur=t[i];
    }

  for (i=0; i<nbb; i++)
    {
      GModSpecif.ParamEntre[i+nbc+nbt]->valeur=b[i];
    }
  
  xt=nbc+nbt+nbb;
  for (i=0; i<nbg; i++)
    {
      GModSpecif.ParamEntre[i+xt]->valeur=g[i];
    }

  /* boucle sur les nbl observations */
  for (i=0; i<nbl; i++)
    {
      /* recopie des parametres d'entree */
      for (j=0; j<nbc; j++)
	{
	  GModSpecif.ParamEntre[j]->valeur=x[i][j];
	}
      GModSpecif.ParamEntre[xt+nbg]->valeur=f[i];
      for (j=0; j<nbt; j++)
	{
	  GModSpecif.ParamEntre[xt+nbg+1+j]->valeur=df[i][j];
	}
      
      /* evaluation de l'arbre de V */
      v[i]=evalArb(GModSpecif.IdeV.arbre);
      if (finite(v[i])!=1)
	  {
	    *ie=i+1;
	    *le=V;
	    return(ERRMATH);
	  }

      /* evaluation des arbres des dV/dt */
      for (j=0; j<nbt; j++)
	{
	  dtv[i][j]=evalArb(Derivees.DerivV.arbre[j]);
	  if (finite(dtv[i][j])!=1)
	    {
	      *ie=i+1;
	      *le=DVDT;
	      return(ERRMATH);
	    }
	}
      /* evaluation des arbres des dV/db */
      for (j=0; j<nbb; j++)
	{
	  dbv[i][j]=evalArb(Derivees.DerivV.arbre[nbt+j]);
	  if (finite(dbv[i][j])!=1)
	    {
	      *ie=i+1;
	      *le=DVDB;
	      return(ERRMATH);
	    }
	}
    }
  return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : calcphi_                 |
| Role                  : calcule la reponse d'un modele defini par un sedo
| Parametres d'entree   : nbt nbg nbl nbc t g x Fsedo dFsedo
| Parametres de sortie  : f df ie le               |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 26            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : evalArb                   |
| Fonctions appelantes :                           |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt calcphi_(nbt, nbg, nbl, nbc, t, g, x, Fsedo, dFsedo, f, df, le, ie)
      TShortInt nbt, nbg, nbl, nbc;
      TDouble *t, *g, **x, *f, **df, ***Fsedo, ***dFsedo;
      TShortInt *le, *ie;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i, xt, xt2, j, ii, jj, nbT, nbF, nl, nbParam;
  /* TShortInt i, xt, xt2, j, ii, jj, nbT, nbF, ni, param, nl, nbParam;*/

  ECRTRACE("calcphi");

  /* recopie des parametres d'entree */
  for (i=0; i<nbt; i++)
    {
      GModSpecif.ParamEntre[i+nbc]->valeur=t[i];
    }
  xt=nbc+nbt;
  for (i=0; i<nbg; i++)
    {
      GModSpecif.ParamEntre[i+xt]->valeur=g[i];
    }

  /* decalage pour la copie de Fsedo */
  xt=xt+nbg;
  /* decalage pour la copie de dFsedo */
  xt2=xt+GModSpecif.IdeValInt.nbele*GModSpecif.IdeLesF.nbele;
  /* nombre de valeur d'integration */
  nbT=GModSpecif.IdeValInt.nbele;
  /* nombre d'equations du sedo */
  nbF=GModSpecif.IdeLesF.nbele;
  /* nombre de parametres du sedo */
  nbParam=GModSpecif.NbThetaSedo+GModSpecif.IdeCondInit.nbele;
  /* longueur d'une ligne de dFsedo=nbF*(nbThetaSedo+NbCondInit) */
  nl=nbF*nbParam;
  /* boucle sur les nbl observations */
  for (i=0; i<nbl; i++)
    {
      /* recopie des X */
      for (j=0; j<nbc; j++)
	{
	  GModSpecif.ParamEntre[j]->valeur=x[i][j];
	}
      /* on recopie les Fsedo */
      for (jj=0; jj<nbT; jj++)
	{
	  for (ii=0; ii<nbF; ii++)
	    {
	      GModSpecif.ParamEntre[xt+jj*nbF+ii]->valeur=Fsedo[i][jj][ii];
	    }
	}
      /* on recopie les dFsedo */
      for (jj=0; jj<nbT; jj++)
	{
	  for (ii=0; ii<nl; ii++)
	    {
	      GModSpecif.ParamEntre[xt2+jj*nl+ii]->valeur=dFsedo[i][jj][ii];
	    }
	}

      /* evaluation de l'arbre de F */
      f[i]=evalArb(GModSpecif.IdeF.arbre);
      if (finite(f[i])!=1)
	  {
	    *ie=i+1;
	    *le=F;
	    return(ERRMATH);
	  }

      /* evaluation des arbres des dF */
      for (j=0; j<nbt; j++)
	{
	  df[i][j]=evalArb(Derivees.DerivF.arbre[j]);
	  if (finite(df[i][j])!=1)
	    {
	      *ie=i+1;
	      *le=DFDT;
	      return(ERRMATH);
	    }
	}
    }
  return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : calcodes_                |
| Role                  : defini le modele a integrer
| Parametres d'entree   :                          |
|    t : variable d'integration                    |
|    neq : differents parametres                   |
|    y : valeurs en entree                         |
| Parametres de sortie  :                          |
|    ydot : valeurs du systeme integre             |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 27            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : evalArb                   |
| Fonctions appelantes :                           |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

short int calcodes_(neq, t, y, ydot)
     int *neq;
     double *t, *y, *ydot;

/*--------------- Fin identification fonction ----*/

{
  TShortInt nbobs, iobs, i, j, k, nbX, nbG, nbT, nbF, dbX;

  ECRTRACE("calcodes");
  nbobs=neq[4];  /* nombre d'observations */
  iobs=neq[3];   /* idice de l'observation */
  nbX=neq[5];    /* nombre de variables observees */
  nbG=neq[2];    /* nombre de GammaF */
  nbT=GModSpecif.NbThetaSedo+GModSpecif.IdeCondInit.nbele; /* nombre de theta
							      dans le sedo */
  nbF=GModSpecif.IdeLesDF.nbele;        /* nombre d'equations dans le sedo */

  /* AB: 26/05/98 remplacement de:
  dbX=nbT+(1+nbT)*nbF+nbG;    debut de la copie des observations dans y 
 par: */

 dbX=neq[1]+(1+nbT)*nbF+nbG;   /* debut de la copie des observations dans y */
 /* neq[1]= nbre total de param, y compris ceux qui ne sont pas dans le sedo,
ni condinit */

  /* copie des parametres en entree, il suffit de faire une copie de y,
   t est copie tout au debut */
  GModSpecif.ParamEntre[0]->valeur=*t;
  for (i=0; i< dbX; i++)
    {
      GModSpecif.ParamEntre[1+i]->valeur=y[i];
    }

  /* copie de l'observation iobs */

  for (i=0; i<nbX; i++)
    {
      GModSpecif.ParamEntre[1+dbX+i]->valeur=y[dbX+nbobs*i+iobs];
    }

  /* evaluation des arbres des dF */
  for (i=0; i<nbF; i++)
    {
      ydot[i]=evalArb(GModSpecif.IdeLesDF.UnDF[i].arbre);
      if (finite(ydot[i])!=1)
	{
	  neq[6]=ERRMATH;
	  return(ERRMATH);
	}
    }
  
  /* evaluation des arbres des ddF/dT */
  k=nbF;
  for (i=0; i<nbF; i++)
    {
      for (j=0; j<nbT; j++)
	{
	  ydot[k]=evalArb(Derivees.DerivSedo.arbre[j*nbF+i]);
	  if (finite(ydot[k])!=1)
	    {
	      neq[6]=ERRMATH;
	      return(ERRMATH);
	    }
	  k++;
	}
    }

  return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : cacljac_                 |
| Role                  : defini la matrice du Jacobien
| Parametres d'entree   :                          |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 27            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes :                           |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

short int calcjac_()

/*--------------- Fin identification fonction ----*/

{
return 0;
}
