      subroutine lsodar (f, neq, y, t, tout, itol, rtol, atol, itask,
     1            istate, iopt, rwork, lrw, iwork, liw, jac, jt,
     2            g, ng, jroot)
      external f, jac, g
      integer neq, itol, itask, istate, iopt, lrw, iwork, liw, jt,
     1   ng, jroot
      double precision y, t, tout, rtol, atol, rwork
      dimension neq(1), y(1), rtol(1), atol(1), rwork(lrw), iwork(liw),
     1   jroot(ng)
c-----------------------------------------------------------------------
c this is the march 30, 1987 version of
c lsodar.. livermore solver for ordinary differential equations, with
c          automatic method switching for stiff and nonstiff problems,
c          and with root-finding.
c
c this version is in double precision.
c
c lsodar solves the initial value problem for stiff or nonstiff
c systems of first order ode-s,
c     dy/dt = f(t,y) ,  or, in component form,
c     dy(i)/dt = f(i) = f(i,t,y(1),y(2),...,y(neq)) (i = 1,...,neq).
c at the same time, it locates the roots of any of a set of functions
c     g(i) = g(i,t,y(1),...,y(neq))  (i = 1,...,ng).
c
c this a variant version of the lsode package.  it differs from lsode
c in two ways..
c (a) it switches automatically between stiff and nonstiff methods.
c this means that the user does not have to determine whether the
c problem is stiff or not, and the solver will automatically choose the
c appropriate method.  it always starts with the nonstiff method.
c (b) it finds the root of at least one of a set of constraint
c functions g(i) of the independent and dependent variables.
c it finds only those roots for which some g(i), as a function
c of t, changes sign in the interval of integration.
c it then returns the solution at the root, if that occurs
c sooner than the specified stop condition, and otherwise returns
c the solution according the specified stop condition.
c
c authors..
c                linda r. petzold  and  alan c. hindmarsh,
c                computing and mathematics research division, l-316
c                lawrence livermore national laboratory
c                livermore, ca 94550.
c
c references..
c 1.  alan c. hindmarsh,  odepack, a systematized collection of ode
c     solvers, in scientific computing, r. s. stepleman et al. (eds.),
c     north-holland, amsterdam, 1983, pp. 55-64.
c 2.  linda r. petzold, automatic selection of methods for solving
c     stiff and nonstiff systems of ordinary differential equations,
c     siam j. sci. stat. comput. 4 (1983), pp. 136-148.
c 3.  kathie l. hiebert and lawrence f. shampine, implicitly defined
c     output points for solutions of ode-s, sandia report sand80-0180,
c     february, 1980.
c-----------------------------------------------------------------------
c summary of usage.
c
c communication between the user and the lsodar package, for normal
c situations, is summarized here.  this summary describes only a subset
c of the full set of options available.  see the full description for
c details, including alternative treatment of the jacobian matrix,
c optional inputs and outputs, nonstandard options, and
c instructions for special situations.  see also the example
c problem (with program and output) following this summary.
c
c a. first provide a subroutine of the form..
c               subroutine f (neq, t, y, ydot)
c               dimension y(neq), ydot(neq)
c which supplies the vector function f by loading ydot(i) with f(i).
c
c b. provide a subroutine of the form..
c               subroutine g (neq, t, y, ng, gout)
c               dimension y(neq), gout(ng)
c which supplies the vector function g by loading gout(i) with
c g(i), the i-th constraint function whose root is sought.
c
c c. write a main program which calls subroutine lsodar once for
c each point at which answers are desired.  this should also provide
c for possible use of logical unit 6 for output of error messages by
c lsodar.  on the first call to lsodar, supply arguments as follows..
c f      = name of subroutine for right-hand side vector f.
c          this name must be declared external in calling program.
c neq    = number of first order ode-s.
c y      = array of initial values, of length neq.
c t      = the initial value of the independent variable.
c tout   = first point where output is desired (.ne. t).
c itol   = 1 or 2 according as atol (below) is a scalar or array.
c rtol   = relative tolerance parameter (scalar).
c atol   = absolute tolerance parameter (scalar or array).
c          the estimated local error in y(i) will be controlled so as
c          to be less than
c             ewt(i) = rtol*abs(y(i)) + atol     if itol = 1, or
c             ewt(i) = rtol*abs(y(i)) + atol(i)  if itol = 2.
c          thus the local error test passes if, in each component,
c          either the absolute error is less than atol (or atol(i)),
c          or the relative error is less than rtol.
c          use rtol = 0.0 for pure absolute error control, and
c          use atol = 0.0 (or atol(i) = 0.0) for pure relative error
c          control.  caution.. actual (global) errors may exceed these
c          local tolerances, so choose them conservatively.
c itask  = 1 for normal computation of output values of y at t = tout.
c istate = integer flag (input and output).  set istate = 1.
c iopt   = 0 to indicate no optional inputs used.
c rwork  = real work array of length at least..
c             22 + neq * max(16, neq + 9) + 3*ng.
c          see also paragraph f below.
c lrw    = declared length of rwork (in user-s dimension).
c iwork  = integer work array of length at least  20 + neq.
c liw    = declared length of iwork (in user-s dimension).
c jac    = name of subroutine for jacobian matrix.
c          use a dummy name.  see also paragraph f below.
c jt     = jacobian type indicator.  set jt = 2.
c          see also paragraph f below.
c g      = name of subroutine for constraint functions, whose
c          roots are desired during the integration.
c          this name must be declared external in calling program.
c ng     = number of constraint functions g(i).  if there are none,
c          set ng = 0, and pass a dummy name for g.
c jroot  = integer array of length ng for output of root information.
c          see next paragraph.
c note that the main program must declare arrays y, rwork, iwork,
c jroot, and possibly atol.
c
c d. the output from the first call (or any call) is..
c      y = array of computed values of y(t) vector.
c      t = corresponding value of independent variable.  this is
c          tout if istate = 2, or the root location if istate = 3,
c          or the farthest point reached if lsodar was unsuccessful.
c istate = 2 or 3  if lsodar was successful, negative otherwise.
c           2 means no root was found, and tout was reached as desired.
c           3 means a root was found prior to reaching tout.
c          -1 means excess work done on this call (perhaps wrong jt).
c          -2 means excess accuracy requested (tolerances too small).
c          -3 means illegal input detected (see printed message).
c          -4 means repeated error test failures (check all inputs).
c          -5 means repeated convergence failures (perhaps bad jacobian
c             supplied or wrong choice of jt or tolerances).
c          -6 means error weight became zero during problem. (solution
c             component i vanished, and atol or atol(i) = 0.)
c          -7 means work space insufficient to finish (see messages).
c jroot  = array showing roots found if istate = 3 on return.
c          jroot(i) = 1 if g(i) has a root at t, or 0 otherwise.
c
c e. to continue the integration after a successful return, proceed
c as follows..
c  (a) if istate = 2 on return, reset tout and call lsodar again.
c  (b) if istate = 3 on return, reset istate to 2 and call lsodar again.
c in either case, no other parameters need be reset.
c
c f. note.. if and when lsodar regards the problem as stiff, and
c switches methods accordingly, it must make use of the neq by neq
c jacobian matrix, j = df/dy.  for the sake of simplicity, the
c inputs to lsodar recommended in paragraph c above cause lsodar to
c treat j as a full matrix, and to approximate it internally by
c difference quotients.  alternatively, j can be treated as a band
c matrix (with great potential reduction in the size of the rwork
c array).  also, in either the full or banded case, the user can supply
c j in closed form, with a routine whose name is passed as the jac
c argument.  these alternatives are described in the paragraphs on
c rwork, jac, and jt in the full description of the call sequence below.
c
c-----------------------------------------------------------------------
c example problem.
c
c the following is a simple example problem, with the coding
c needed for its solution by lsodar.  the problem is from chemical
c kinetics, and consists of the following three rate equations..
c     dy1/dt = -.04*y1 + 1.e4*y2*y3
c     dy2/dt = .04*y1 - 1.e4*y2*y3 - 3.e7*y2**2
c     dy3/dt = 3.e7*y2**2
c on the interval from t = 0.0 to t = 4.e10, with initial conditions
c y1 = 1.0, y2 = y3 = 0.  the problem is stiff.
c in addition, we want to find the values of t, y1, y2, and y3 at which
c   (1) y1 reaches the value 1.e-4, and
c   (2) y3 reaches the value 1.e-2.
c
c the following coding solves this problem with lsodar,
c printing results at t = .4, 4., ..., 4.e10, and at the computed
c roots.  it uses itol = 2 and atol much smaller for y2 than y1 or y3
c because y2 has much smaller values.
c at the end of the run, statistical quantities of interest are
c printed (see optional outputs in the full description below).
c
c     external fex, gex
c     double precision atol, rtol, rwork, t, tout, y
c     dimension y(3), atol(3), rwork(76), iwork(23), jroot(2)
c     neq = 3
c     y(1) = 1.0d0
c     y(2) = 0.0d0
c     y(3) = 0.0d0
c     t = 0.0d0
c     tout = 0.4d0
c     itol = 2
c     rtol = 1.0d-4
c     atol(1) = 1.0d-6
c     atol(2) = 1.0d-10
c     atol(3) = 1.0d-6
c     itask = 1
c     istate = 1
c     iopt = 0
c     lrw = 76
c     liw = 23
c     jt = 2
c     ng = 2
c     do 40 iout = 1,12
c 10    call lsodar(fex,neq,y,t,tout,itol,rtol,atol,itask,istate,
c    1     iopt,rwork,lrw,iwork,liw,jdum,jt,gex,ng,jroot)
c       write(6,20)t,y(1),y(2),y(3)
c 20    format(7h at t =,e12.4,6h   y =,3e14.6)
c       if (istate .lt. 0) go to 80
c       if (istate .eq. 2) go to 40
c       write(6,30)jroot(1),jroot(2)
c 30    format(5x,35h the above line is a root,  jroot =,2i5)
c       istate = 2
c       go to 10
c 40    tout = tout*10.0d0
c     write(6,60)iwork(11),iwork(12),iwork(13),iwork(10),
c    1   iwork(19),rwork(15)
c 60  format(/12h no. steps =,i4,11h  no. f-s =,i4,11h  no. j-s =,i4,
c    1   11h  no. g-s =,i4/
c    2   19h method last used =,i2,25h   last switch was at t =,e12.4)
c     stop
c 80  write(6,90)istate
c 90  format(///22h error halt.. istate =,i3)
c     stop
c     end
c
c     subroutine fex (neq, t, y, ydot)
c     double precision t, y, ydot
c     dimension y(3), ydot(3)
c     ydot(1) = -0.04d0*y(1) + 1.0d4*y(2)*y(3)
c     ydot(3) = 3.0d7*y(2)*y(2)
c     ydot(2) = -ydot(1) - ydot(3)
c     return
c     end
c
c     subroutine gex (neq, t, y, ng, gout)
c     double precision t, y, gout
c     dimension y(3), gout(2)
c     gout(1) = y(1) - 1.0d-4
c     gout(2) = y(3) - 1.0d-2
c     return
c     end
c
c the output of this program (on a cdc-7600 in single precision)
c is as follows..
c
c   at t =  2.6400e-01   y =  9.899653e-01  3.470563e-05  1.000000e-02
c        the above line is a root,  jroot =    0    1
c   at t =  4.0000e-01   y =  9.851712e-01  3.386380e-05  1.479493e-02
c   at t =  4.0000e+00   y =  9.055333e-01  2.240655e-05  9.444430e-02
c   at t =  4.0000e+01   y =  7.158403e-01  9.186334e-06  2.841505e-01
c   at t =  4.0000e+02   y =  4.505250e-01  3.222964e-06  5.494717e-01
c   at t =  4.0000e+03   y =  1.831975e-01  8.941774e-07  8.168016e-01
c   at t =  4.0000e+04   y =  3.898730e-02  1.621940e-07  9.610125e-01
c   at t =  4.0000e+05   y =  4.936363e-03  1.984221e-08  9.950636e-01
c   at t =  4.0000e+06   y =  5.161831e-04  2.065786e-09  9.994838e-01
c   at t =  2.0745e+07   y =  1.000000e-04  4.000395e-10  9.999000e-01
c        the above line is a root,  jroot =    1    0
c   at t =  4.0000e+07   y =  5.179817e-05  2.072032e-10  9.999482e-01
c   at t =  4.0000e+08   y =  5.283401e-06  2.113371e-11  9.999947e-01
c   at t =  4.0000e+09   y =  4.659031e-07  1.863613e-12  9.999995e-01
c   at t =  4.0000e+10   y =  1.404280e-08  5.617126e-14  1.000000e+00
c
c   no. steps = 361  no. f-s = 693  no. j-s =  64  no. g-s = 390
c   method last used = 2   last switch was at t =  6.0092e-03
c-----------------------------------------------------------------------
c full description of user interface to lsodar.
c
c the user interface to lsodar consists of the following parts.
c
c i.   the call sequence to subroutine lsodar, which is a driver
c      routine for the solver.  this includes descriptions of both
c      the call sequence arguments and of user-supplied routines.
c      following these descriptions is a description of
c      optional inputs available through the call sequence, and then
c      a description of optional outputs (in the work arrays).
c
c ii.  descriptions of other routines in the lsodar package that may be
c      (optionally) called by the user.  these provide the ability to
c      alter error message handling, save and restore the internal
c      common, and obtain specified derivatives of the solution y(t).
c
c iii. descriptions of common blocks to be declared in overlay
c      or similar environments, or to be saved when doing an interrupt
c      of the problem and continued solution later.
c
c iv.  description of a subroutine in the lsodar package,
c      which the user may replace with his own version, if desired.
c      this relates to the measurement of errors.
c
c-----------------------------------------------------------------------
c part i.  call sequence.
c
c the call sequence parameters used for input only are
c     f, neq, tout, itol, rtol, atol, itask, iopt, lrw, liw, jac,
c     jt, g, and ng,
c that used only for output is  jroot,
c and those used for both input and output are
c     y, t, istate.
c the work arrays rwork and iwork are also used for conditional and
c optional inputs and optional outputs.  (the term output here refers
c to the return from subroutine lsodar to the user-s calling program.)
c
c the legality of input parameters will be thoroughly checked on the
c initial call for the problem, but not checked thereafter unless a
c change in input parameters is flagged by istate = 3 on input.
c
c the descriptions of the call arguments are as follows.
c
c f      = the name of the user-supplied subroutine defining the
c          ode system.  the system must be put in the first-order
c          form dy/dt = f(t,y), where f is a vector-valued function
c          of the scalar t and the vector y.  subroutine f is to
c          compute the function f.  it is to have the form
c               subroutine f (neq, t, y, ydot)
c               dimension y(1), ydot(1)
c          where neq, t, and y are input, and the array ydot = f(t,y)
c          is output.  y and ydot are arrays of length neq.
c          (in the dimension statement above, 1 is a dummy
c          dimension.. it can be replaced by any value.)
c          subroutine f should not alter y(1),...,y(neq).
c          f must be declared external in the calling program.
c
c          subroutine f may access user-defined quantities in
c          neq(2),... and/or in y(neq(1)+1),... if neq is an array
c          (dimensioned in f) and/or y has length exceeding neq(1).
c          see the descriptions of neq and y below.
c
c          if quantities computed in the f routine are needed
c          externally to lsodar, an extra call to f should be made
c          for this purpose, for consistent and accurate results.
c          if only the derivative dy/dt is needed, use intdy instead.
c
c neq    = the size of the ode system (number of first order
c          ordinary differential equations).  used only for input.
c          neq may be decreased, but not increased, during the problem.
c          if neq is decreased (with istate = 3 on input), the
c          remaining components of y should be left undisturbed, if
c          these are to be accessed in f and/or jac.
c
c          normally, neq is a scalar, and it is generally referred to
c          as a scalar in this user interface description.  however,
c          neq may be an array, with neq(1) set to the system size.
c          (the lsodar package accesses only neq(1).)  in either case,
c          this parameter is passed as the neq argument in all calls
c          to f, jac, and g.  hence, if it is an array, locations
c          neq(2),... may be used to store other integer data and pass
c          it to f, jac, and g.  each such subroutine must include
c          neq in a dimension statement in that case.
c
c y      = a real array for the vector of dependent variables, of
c          length neq or more.  used for both input and output on the
c          first call (istate = 1), and only for output on other calls.
c          on the first call, y must contain the vector of initial
c          values.  on output, y contains the computed solution vector,
c          evaluated at t.  if desired, the y array may be used
c          for other purposes between calls to the solver.
c
c          this array is passed as the y argument in all calls to f,
c          jac, and g.  hence its length may exceed neq, and locations
c          y(neq+1),... may be used to store other real data and
c          pass it to f, jac, and g.  (the lsodar package accesses only
c          y(1),...,y(neq).)
c
c t      = the independent variable.  on input, t is used only on the
c          first call, as the initial point of the integration.
c          on output, after each call, t is the value at which a
c          computed solution y is evaluated (usually the same as tout).
c          if a root was found, t is the computed location of the
c          root reached first, on output.
c          on an error return, t is the farthest point reached.
c
c tout   = the next value of t at which a computed solution is desired.
c          used only for input.
c
c          when starting the problem (istate = 1), tout may be equal
c          to t for one call, then should .ne. t for the next call.
c          for the initial t, an input value of tout .ne. t is used
c          in order to determine the direction of the integration
c          (i.e. the algebraic sign of the step sizes) and the rough
c          scale of the problem.  integration in either direction
c          (forward or backward in t) is permitted.
c
c          if itask = 2 or 5 (one-step modes), tout is ignored after
c          the first call (i.e. the first call with tout .ne. t).
c          otherwise, tout is required on every call.
c
c          if itask = 1, 3, or 4, the values of tout need not be
c          monotone, but a value of tout which backs up is limited
c          to the current internal t interval, whose endpoints are
c          tcur - hu and tcur (see optional outputs, below, for
c          tcur and hu).
c
c itol   = an indicator for the type of error control.  see
c          description below under atol.  used only for input.
c
c rtol   = a relative error tolerance parameter, either a scalar or
c          an array of length neq.  see description below under atol.
c          input only.
c
c atol   = an absolute error tolerance parameter, either a scalar or
c          an array of length neq.  input only.
c
c             the input parameters itol, rtol, and atol determine
c          the error control performed by the solver.  the solver will
c          control the vector e = (e(i)) of estimated local errors
c          in y, according to an inequality of the form
c                      max-norm of ( e(i)/ewt(i) )   .le.   1,
c          where ewt = (ewt(i)) is a vector of positive error weights.
c          the values of rtol and atol should all be non-negative.
c          the following table gives the types (scalar/array) of
c          rtol and atol, and the corresponding form of ewt(i).
c
c             itol    rtol       atol          ewt(i)
c              1     scalar     scalar     rtol*abs(y(i)) + atol
c              2     scalar     array      rtol*abs(y(i)) + atol(i)
c              3     array      scalar     rtol(i)*abs(y(i)) + atol
c              4     array      array      rtol(i)*abs(y(i)) + atol(i)
c
c          when either of these parameters is a scalar, it need not
c          be dimensioned in the user-s calling program.
c
c          if none of the above choices (with itol, rtol, and atol
c          fixed throughout the problem) is suitable, more general
c          error controls can be obtained by substituting a
c          user-supplied routine for the setting of ewt.
c          see part iv below.
c
c          if global errors are to be estimated by making a repeated
c          run on the same problem with smaller tolerances, then all
c          components of rtol and atol (i.e. of ewt) should be scaled
c          down uniformly.
c
c itask  = an index specifying the task to be performed.
c          input only.  itask has the following values and meanings.
c          1  means normal computation of output values of y(t) at
c             t = tout (by overshooting and interpolating).
c          2  means take one step only and return.
c          3  means stop at the first internal mesh point at or
c             beyond t = tout and return.
c          4  means normal computation of output values of y(t) at
c             t = tout but without overshooting t = tcrit.
c             tcrit must be input as rwork(1).  tcrit may be equal to
c             or beyond tout, but not behind it in the direction of
c             integration.  this option is useful if the problem
c             has a singularity at or beyond t = tcrit.
c          5  means take one step, without passing tcrit, and return.
c             tcrit must be input as rwork(1).
c
c          note..  if itask = 4 or 5 and the solver reaches tcrit
c          (within roundoff), it will return t = tcrit (exactly) to
c          indicate this (unless itask = 4 and tout comes before tcrit,
c          in which case answers at t = tout are returned first).
c
c istate = an index used for input and output to specify the
c          the state of the calculation.
c
c          on input, the values of istate are as follows.
c          1  means this is the first call for the problem
c             (initializations will be done).  see note below.
c          2  means this is not the first call, and the calculation
c             is to continue normally, with no change in any input
c             parameters except possibly tout and itask.
c             (if itol, rtol, and/or atol are changed between calls
c             with istate = 2, the new values will be used but not
c             tested for legality.)
c          3  means this is not the first call, and the
c             calculation is to continue normally, but with
c             a change in input parameters other than
c             tout and itask.  changes are allowed in
c             neq, itol, rtol, atol, iopt, lrw, liw, jt, ml, mu,
c             and any optional inputs except h0, mxordn, and mxords.
c             (see iwork description for ml and mu.)
c             in addition, immediately following a return with
c             istate = 3 (root found), ng and g may be changed.
c             (but changing ng from 0 to .gt. 0 is not allowed.)
c          note..  a preliminary call with tout = t is not counted
c          as a first call here, as no initialization or checking of
c          input is done.  (such a call is sometimes useful for the
c          purpose of outputting the initial conditions.)
c          thus the first call for which tout .ne. t requires
c          istate = 1 on input.
c
c          on output, istate has the following values and meanings.
c           1  means nothing was done, as tout was equal to t with
c              istate = 1 on input.  (however, an internal counter was
c              set to detect and prevent repeated calls of this type.)
c           2  means the integration was performed successfully, and
c              no roots were found.
c           3  means the integration was successful, and one or more
c              roots were found before satisfying the stop condition
c              specified by itask.  see jroot.
c          -1  means an excessive amount of work (more than mxstep
c              steps) was done on this call, before completing the
c              requested task, but the integration was otherwise
c              successful as far as t.  (mxstep is an optional input
c              and is normally 500.)  to continue, the user may
c              simply reset istate to a value .gt. 1 and call again
c              (the excess work step counter will be reset to 0).
c              in addition, the user may increase mxstep to avoid
c              this error return (see below on optional inputs).
c          -2  means too much accuracy was requested for the precision
c              of the machine being used.  this was detected before
c              completing the requested task, but the integration
c              was successful as far as t.  to continue, the tolerance
c              parameters must be reset, and istate must be set
c              to 3.  the optional output tolsf may be used for this
c              purpose.  (note.. if this condition is detected before
c              taking any steps, then an illegal input return
c              (istate = -3) occurs instead.)
c          -3  means illegal input was detected, before taking any
c              integration steps.  see written message for details.
c              note..  if the solver detects an infinite loop of calls
c              to the solver with illegal input, it will cause
c              the run to stop.
c          -4  means there were repeated error test failures on
c              one attempted step, before completing the requested
c              task, but the integration was successful as far as t.
c              the problem may have a singularity, or the input
c              may be inappropriate.
c          -5  means there were repeated convergence test failures on
c              one attempted step, before completing the requested
c              task, but the integration was successful as far as t.
c              this may be caused by an inaccurate jacobian matrix,
c              if one is being used.
c          -6  means ewt(i) became zero for some i during the
c              integration.  pure relative error control (atol(i)=0.0)
c              was requested on a variable which has now vanished.
c              the integration was successful as far as t.
c          -7  means the length of rwork and/or iwork was too small to
c              proceed, but the integration was successful as far as t.
c              this happens when lsodar chooses to switch methods
c              but lrw and/or liw is too small for the new method.
c
c          note..  since the normal output value of istate is 2,
c          it does not need to be reset for normal continuation.
c          also, since a negative input value of istate will be
c          regarded as illegal, a negative output value requires the
c          user to change it, and possibly other inputs, before
c          calling the solver again.
c
c iopt   = an integer flag to specify whether or not any optional
c          inputs are being used on this call.  input only.
c          the optional inputs are listed separately below.
c          iopt = 0 means no optional inputs are being used.
c                   default values will be used in all cases.
c          iopt = 1 means one or more optional inputs are being used.
c
c rwork  = a real array (double precision) for work space, and (in the
c          first 20 words) for conditional and optional inputs and
c          optional outputs.
c          as lsodar switches automatically between stiff and nonstiff
c          methods, the required length of rwork can change during the
c          problem.  thus the rwork array passed to lsodar can either
c          have a static (fixed) length large enough for both methods,
c          or have a dynamic (changing) length altered by the calling
c          program in response to output from lsodar.
c
c                       --- fixed length case ---
c          if the rwork length is to be fixed, it should be at least
c               max (lrn, lrs),
c          where lrn and lrs are the rwork lengths required when the
c          current method is nonstiff or stiff, respectively.
c
c          the separate rwork length requirements lrn and lrs are
c          as follows..
c          if neq is constant and the maximum method orders have
c          their default values, then
c             lrn = 20 + 16*neq + 3*ng,
c             lrs = 22 + 9*neq + neq**2 + 3*ng           (jt = 1 or 2),
c             lrs = 22 + 10*neq + (2*ml+mu)*neq + 3*ng   (jt = 4 or 5).
c          under any other conditions, lrn and lrs are given by..
c             lrn = 20 + nyh*(mxordn+1) + 3*neq + 3*ng,
c             lrs = 20 + nyh*(mxords+1) + 3*neq + lmat + 3*ng,
c          where
c             nyh    = the initial value of neq,
c             mxordn = 12, unless a smaller value is given as an
c                      optional input,
c             mxords = 5, unless a smaller value is given as an
c                      optional input,
c             lmat   = length of matrix work space..
c             lmat   = neq**2 + 2              if jt = 1 or 2,
c             lmat   = (2*ml + mu + 1)*neq + 2 if jt = 4 or 5.
c
c                       --- dynamic length case ---
c          if the length of rwork is to be dynamic, then it should
c          be at least lrn or lrs, as defined above, depending on the
c          current method.  initially, it must be at least lrn (since
c          lsodar starts with the nonstiff method).  on any return
c          from lsodar, the optional output mcur indicates the current
c          method.  if mcur differs from the value it had on the
c          previous return, or if there has only been one call to
c          lsodar and mcur is now 2, then lsodar has switched
c          methods during the last call, and the length of rwork
c          should be reset (to lrn if mcur = 1, or to lrs if
c          mcur = 2).  (an increase in the rwork length is required
c          if lsodar returned istate = -7, but not otherwise.)
c          after resetting the length, call lsodar with istate = 3
c          to signal that change.
c
c lrw    = the length of the array rwork, as declared by the user.
c          (this will be checked by the solver.)
c
c iwork  = an integer array for work space.
c          as lsodar switches automatically between stiff and nonstiff
c          methods, the required length of iwork can change during
c          problem, between
c             lis = 20 + neq   and   lin = 20,
c          respectively.  thus the iwork array passed to lsodar can
c          either have a fixed length of at least 20 + neq, or have a
c          dynamic length of at least lin or lis, depending on the
c          current method.  the comments on dynamic length under
c          rwork above apply here.  initially, this length need
c          only be at least lin = 20.
c
c          the first few words of iwork are used for conditional and
c          optional inputs and optional outputs.
c
c          the following 2 words in iwork are conditional inputs..
c            iwork(1) = ml     these are the lower and upper
c            iwork(2) = mu     half-bandwidths, respectively, of the
c                       banded jacobian, excluding the main diagonal.
c                       the band is defined by the matrix locations
c                       (i,j) with i-ml .le. j .le. i+mu.  ml and mu
c                       must satisfy  0 .le.  ml,mu  .le. neq-1.
c                       these are required if jt is 4 or 5, and
c                       ignored otherwise.  ml and mu may in fact be
c                       the band parameters for a matrix to which
c                       df/dy is only approximately equal.
c
c liw    = the length of the array iwork, as declared by the user.
c          (this will be checked by the solver.)
c
c note.. the base addresses of the work arrays must not be
c altered between calls to lsodar for the same problem.
c the contents of the work arrays must not be altered
c between calls, except possibly for the conditional and
c optional inputs, and except for the last 3*neq words of rwork.
c the latter space is used for internal scratch space, and so is
c available for use by the user outside lsodar between calls, if
c desired (but not for use by f, jac, or g).
c
c jac    = the name of the user-supplied routine to compute the
c          jacobian matrix, df/dy, if jt = 1 or 4.  the jac routine
c          is optional, but if the problem is expected to be stiff much
c          of the time, you are encouraged to supply jac, for the sake
c          of efficiency.  (alternatively, set jt = 2 or 5 to have
c          lsodar compute df/dy internally by difference quotients.)
c          if and when lsodar uses df/dy, if treats this neq by neq
c          matrix either as full (jt = 1 or 2), or as banded (jt =
c          4 or 5) with half-bandwidths ml and mu (discussed under
c          iwork above).  in either case, if jt = 1 or 4, the jac
c          routine must compute df/dy as a function of the scalar t
c          and the vector y.  it is to have the form
c               subroutine jac (neq, t, y, ml, mu, pd, nrowpd)
c               dimension y(1), pd(nrowpd,1)
c          where neq, t, y, ml, mu, and nrowpd are input and the array
c          pd is to be loaded with partial derivatives (elements of
c          the jacobian matrix) on output.  pd must be given a first
c          dimension of nrowpd.  t and y have the same meaning as in
c          subroutine f.  (in the dimension statement above, 1 is a
c          dummy dimension.. it can be replaced by any value.)
c               in the full matrix case (jt = 1), ml and mu are
c          ignored, and the jacobian is to be loaded into pd in
c          columnwise manner, with df(i)/dy(j) loaded into pd(i,j).
c               in the band matrix case (jt = 4), the elements
c          within the band are to be loaded into pd in columnwise
c          manner, with diagonal lines of df/dy loaded into the rows
c          of pd.  thus df(i)/dy(j) is to be loaded into pd(i-j+mu+1,j).
c          ml and mu are the half-bandwidth parameters (see iwork).
c          the locations in pd in the two triangular areas which
c          correspond to nonexistent matrix elements can be ignored
c          or loaded arbitrarily, as they are overwritten by lsodar.
c               jac need not provide df/dy exactly.  a crude
c          approximation (possibly with a smaller bandwidth) will do.
c               in either case, pd is preset to zero by the solver,
c          so that only the nonzero elements need be loaded by jac.
c          each call to jac is preceded by a call to f with the same
c          arguments neq, t, and y.  thus to gain some efficiency,
c          intermediate quantities shared by both calculations may be
c          saved in a user common block by f and not recomputed by jac,
c          if desired.  also, jac may alter the y array, if desired.
c          jac must be declared external in the calling program.
c               subroutine jac may access user-defined quantities in
c          neq(2),... and/or in y(neq(1)+1),... if neq is an array
c          (dimensioned in jac) and/or y has length exceeding neq(1).
c          see the descriptions of neq and y above.
c
c jt     = jacobian type indicator.  used only for input.
c          jt specifies how the jacobian matrix df/dy will be
c          treated, if and when lsodar requires this matrix.
c          jt has the following values and meanings..
c           1 means a user-supplied full (neq by neq) jacobian.
c           2 means an internally generated (difference quotient) full
c             jacobian (using neq extra calls to f per df/dy value).
c           4 means a user-supplied banded jacobian.
c           5 means an internally generated banded jacobian (using
c             ml+mu+1 extra calls to f per df/dy evaluation).
c          if jt = 1 or 4, the user must supply a subroutine jac
c          (the name is arbitrary) as described above under jac.
c          if jt = 2 or 5, a dummy argument can be used.
c
c g      = the name of subroutine for constraint functions, whose
c          roots are desired during the integration.  it is to have
c          the form
c               subroutine g (neq, t, y, ng, gout)
c               dimension y(neq), gout(ng)
c          where neq, t, y, and ng are input, and the array gout
c          is output.  neq, t, and y have the same meaning as in
c          the f routine, and gout is an array of length ng.
c          for i = 1,...,ng, this routine is to load into gout(i)
c          the value at (t,y) of the i-th constraint function g(i).
c          lsodar will find roots of the g(i) of odd multiplicity
c          (i.e. sign changes) as they occur during the integration.
c          g must be declared external in the calling program.
c
c          caution.. because of numerical errors in the functions
c          g(i) due to roundoff and integration error, lsodar may
c          return false roots, or return the same root at two or more
c          nearly equal values of t.  if such false roots are
c          suspected, the user should consider smaller error tolerances
c          and/or higher precision in the evaluation of the g(i).
c
c          if a root of some g(i) defines the end of the problem,
c          the input to lsodar should nevertheless allow integration
c          to a point slightly past that root, so that lsodar can
c          locate the root by interpolation.
c
c          subroutine g may access user-defined quantities in
c          neq(2),... and y(neq(1)+1),... if neq is an array
c          (dimensioned in g) and y has length exceeding neq(1).
c          see the descriptions of neq and y above.
c
c ng     = number of constraint functions g(i).  if there are none,
c          set ng = 0, and pass a dummy name for g.
c
c jroot  = integer array of length ng.  used only for output.
c          on a return with istate = 3 (one or more roots found),
c          jroot(i) = 1 if g(i) has a root at t, or jroot(i) = 0 if not.
c-----------------------------------------------------------------------
c optional inputs.
c
c the following is a list of the optional inputs provided for in the
c call sequence.  (see also part ii.)  for each such input variable,
c this table lists its name as used in this documentation, its
c location in the call sequence, its meaning, and the default value.
c the use of any of these inputs requires iopt = 1, and in that
c case all of these inputs are examined.  a value of zero for any
c of these optional inputs will cause the default value to be used.
c thus to use a subset of the optional inputs, simply preload
c locations 5 to 10 in rwork and iwork to 0.0 and 0 respectively, and
c then set those of interest to nonzero values.
c
c name    location      meaning and default value
c
c h0      rwork(5)  the step size to be attempted on the first step.
c                   the default value is determined by the solver.
c
c hmax    rwork(6)  the maximum absolute step size allowed.
c                   the default value is infinite.
c
c hmin    rwork(7)  the minimum absolute step size allowed.
c                   the default value is 0.  (this lower bound is not
c                   enforced on the final step before reaching tcrit
c                   when itask = 4 or 5.)
c
c ixpr    iwork(5)  flag to generate extra printing at method switches.
c                   ixpr = 0 means no extra printing (the default).
c                   ixpr = 1 means print data on each switch.
c                   t, h, and nst will be printed on the same logical
c                   unit as used for error messages.
c
c mxstep  iwork(6)  maximum number of (internally defined) steps
c                   allowed during one call to the solver.
c                   the default value is 500.
c
c mxhnil  iwork(7)  maximum number of messages printed (per problem)
c                   warning that t + h = t on a step (h = step size).
c                   this must be positive to result in a non-default
c                   value.  the default value is 10.
c
c mxordn  iwork(8)  the maximum order to be allowed for the nonstiff
c                   (adams) method.  the default value is 12.
c                   if mxordn exceeds the default value, it will
c                   be reduced to the default value.
c                   mxordn is held constant during the problem.
c
c mxords  iwork(9)  the maximum order to be allowed for the stiff
c                   (bdf) method.  the default value is 5.
c                   if mxords exceeds the default value, it will
c                   be reduced to the default value.
c                   mxords is held constant during the problem.
c-----------------------------------------------------------------------
c optional outputs.
c
c as optional additional output from lsodar, the variables listed
c below are quantities related to the performance of lsodar
c which are available to the user.  these are communicated by way of
c the work arrays, but also have internal mnemonic names as shown.
c except where stated otherwise, all of these outputs are defined
c on any successful return from lsodar, and on any return with
c istate = -1, -2, -4, -5, or -6.  on an illegal input return
c (istate = -3), they will be unchanged from their existing values
c (if any), except possibly for tolsf, lenrw, and leniw.
c on any error return, outputs relevant to the error will be defined,
c as noted below.
c
c name    location      meaning
c
c hu      rwork(11) the step size in t last used (successfully).
c
c hcur    rwork(12) the step size to be attempted on the next step.
c
c tcur    rwork(13) the current value of the independent variable
c                   which the solver has actually reached, i.e. the
c                   current internal mesh point in t.  on output, tcur
c                   will always be at least as far as the argument
c                   t, but may be farther (if interpolation was done).
c
c tolsf   rwork(14) a tolerance scale factor, greater than 1.0,
c                   computed when a request for too much accuracy was
c                   detected (istate = -3 if detected at the start of
c                   the problem, istate = -2 otherwise).  if itol is
c                   left unaltered but rtol and atol are uniformly
c                   scaled up by a factor of tolsf for the next call,
c                   then the solver is deemed likely to succeed.
c                   (the user may also ignore tolsf and alter the
c                   tolerance parameters in any other way appropriate.)
c
c tsw     rwork(15) the value of t at the time of the last method
c                   switch, if any.
c
c nge     iwork(10) the number of g evaluations for the problem so far.
c
c nst     iwork(11) the number of steps taken for the problem so far.
c
c nfe     iwork(12) the number of f evaluations for the problem so far.
c
c nje     iwork(13) the number of jacobian evaluations (and of matrix
c                   lu decompositions) for the problem so far.
c
c nqu     iwork(14) the method order last used (successfully).
c
c nqcur   iwork(15) the order to be attempted on the next step.
c
c imxer   iwork(16) the index of the component of largest magnitude in
c                   the weighted local error vector ( e(i)/ewt(i) ),
c                   on an error return with istate = -4 or -5.
c
c lenrw   iwork(17) the length of rwork actually required, assuming
c                   that the length of rwork is to be fixed for the
c                   rest of the problem, and that switching may occur.
c                   this is defined on normal returns and on an illegal
c                   input return for insufficient storage.
c
c leniw   iwork(18) the length of iwork actually required, assuming
c                   that the length of iwork is to be fixed for the
c                   rest of the problem, and that switching may occur.
c                   this is defined on normal returns and on an illegal
c                   input return for insufficient storage.
c
c mused   iwork(19) the method indicator for the last successful step..
c                   1 means adams (nonstiff), 2 means bdf (stiff).
c
c mcur    iwork(20) the current method indicator..
c                   1 means adams (nonstiff), 2 means bdf (stiff).
c                   this is the method to be attempted
c                   on the next step.  thus it differs from mused
c                   only if a method switch has just been made.
c
c the following two arrays are segments of the rwork array which
c may also be of interest to the user as optional outputs.
c for each array, the table below gives its internal name,
c its base address in rwork, and its description.
c
c name    base address      description
c
c yh      21 + 3*ng      the nordsieck history array, of size nyh by
c                        (nqcur + 1), where nyh is the initial value
c                        of neq.  for j = 0,1,...,nqcur, column j+1
c                        of yh contains hcur**j/factorial(j) times
c                        the j-th derivative of the interpolating
c                        polynomial currently representing the solution,
c                        evaluated at t = tcur.
c
c acor     lacor         array of size neq used for the accumulated
c         (from common   corrections on each step, scaled on output
c           as noted)    to represent the estimated local error in y
c                        on the last step.  this is the vector e in
c                        the description of the error control.  it is
c                        defined only on a successful return from
c                        lsodar.  the base address lacor is obtained by
c                        including in the user-s program the
c                        following 3 lines..
c                           double precision rls
c                           common /ls0001/ rls(218), ils(39)
c                           lacor = ils(5)
c
c-----------------------------------------------------------------------
c part ii.  other routines callable.
c
c the following are optional calls which the user may make to
c gain additional capabilities in conjunction with lsodar.
c (the routines xsetun and xsetf are designed to conform to the
c slatec error handling package.)
c
c     form of call                  function
c   call xsetun(lun)          set the logical unit number, lun, for
c                             output of messages from lsodar, if
c                             the default is not desired.
c                             the default value of lun is 6.
c
c   call xsetf(mflag)         set a flag to control the printing of
c                             messages by lsodar.
c                             mflag = 0 means do not print. (danger..
c                             this risks losing valuable information.)
c                             mflag = 1 means print (the default).
c
c                             either of the above calls may be made at
c                             any time and will take effect immediately.
c
c   call srcar(rsav,isav,job) saves and restores the contents of
c                             the internal common blocks used by
c                             lsodar (see part iii below).
c                             rsav must be a real array of length 245
c                             or more, and isav must be an integer
c                             array of length 59 or more.
c                             job=1 means save common into rsav/isav.
c                             job=2 means restore common from rsav/isav.
c                                srcar is useful if one is
c                             interrupting a run and restarting
c                             later, or alternating between two or
c                             more problems solved with lsodar.
c
c   call intdy(,,,,,)         provide derivatives of y, of various
c        (see below)          orders, at a specified point t, if
c                             desired.  it may be called only after
c                             a successful return from lsodar.
c
c the detailed instructions for using intdy are as follows.
c the form of the call is..
c
c   call intdy (t, k, rwork(lyh), nyh, dky, iflag)
c
c the input parameters are..
c
c t         = value of independent variable where answers are desired
c             (normally the same as the t last returned by lsodar).
c             for valid results, t must lie between tcur - hu and tcur.
c             (see optional outputs for tcur and hu.)
c k         = integer order of the derivative desired.  k must satisfy
c             0 .le. k .le. nqcur, where nqcur is the current order
c             (see optional outputs).  the capability corresponding
c             to k = 0, i.e. computing y(t), is already provided
c             by lsodar directly.  since nqcur .ge. 1, the first
c             derivative dy/dt is always available with intdy.
c lyh       = 21 + 3*ng = base address in rwork of the history array yh.
c nyh       = column length of yh, equal to the initial value of neq.
c
c the output parameters are..
c
c dky       = a real array of length neq containing the computed value
c             of the k-th derivative of y(t).
c iflag     = integer flag, returned as 0 if k and t were legal,
c             -1 if k was illegal, and -2 if t was illegal.
c             on an error return, a message is also written.
c-----------------------------------------------------------------------
c part iii.  common blocks.
c
c if lsodar is to be used in an overlay situation, the user
c must declare, in the primary overlay, the variables in..
c   (1) the call sequence to lsodar,
c   (2) the four internal common blocks
c         /ls0001/  of length  257  (218 double precision words
c                         followed by 39 integer words),
c         /lsa001/  of length  31    (22 double precision words
c                         followed by  9 integer words),
c         /lsr001/  of length  14     (5 double precision words
c                         followed by  9 integer words),
c         /eh0001/  of length  2 (integer words).
c
c if lsodar is used on a system in which the contents of internal
c common blocks are not preserved between calls, the user should
c declare the above common blocks in his main program to insure
c that their contents are preserved.
c
c if the solution of a given problem by lsodar is to be interrupted
c and then later continued, such as when restarting an interrupted run
c or alternating between two or more problems, the user should save,
c following the return from the last lsodar call prior to the
c interruption, the contents of the call sequence variables and the
c internal common blocks, and later restore these values before the
c next lsodar call for that problem.  to save and restore the common
c blocks, use subroutine srcar (see part ii above).
c
c-----------------------------------------------------------------------
c part iv.  optionally replaceable solver routines.
c
c below is a description of a routine in the lsodar package which
c relates to the measurement of errors, and can be
c replaced by a user-supplied version, if desired.  however, since such
c a replacement may have a major impact on performance, it should be
c done only when absolutely necessary, and only with great caution.
c (note.. the means by which the package version of a routine is
c superseded by the user-s version may be system-dependent.)
c
c (a) ewset.
c the following subroutine is called just before each internal
c integration step, and sets the array of error weights, ewt, as
c described under itol/rtol/atol above..
c     subroutine ewset (neq, itol, rtol, atol, ycur, ewt)
c where neq, itol, rtol, and atol are as in the lsodar call sequence,
c ycur contains the current dependent variable vector, and
c ewt is the array of weights set by ewset.
c
c if the user supplies this subroutine, it must return in ewt(i)
c (i = 1,...,neq) a positive quantity suitable for comparing errors
c in y(i) to.  the ewt array returned by ewset is passed to the
c vmnorm routine, and also used by lsodar in the computation
c of the optional output imxer, and the increments for difference
c quotient jacobians.
c
c in the user-supplied version of ewset, it may be desirable to use
c the current values of derivatives of y.  derivatives up to order nq
c are available from the history array yh, described above under
c optional outputs.  in ewset, yh is identical to the ycur array,
c extended to nq + 1 columns with a column length of nyh and scale
c factors of h**j/factorial(j).  on the first call for the problem,
c given by nst = 0, nq is 1 and h is temporarily set to 1.0.
c the quantities nq, nyh, h, and nst can be obtained by including
c in ewset the statements..
c     double precision h, rls
c     common /ls0001/ rls(218),ils(39)
c     nq = ils(35)
c     nyh = ils(14)
c     nst = ils(36)
c     h = rls(212)
c thus, for example, the current value of dy/dt can be obtained as
c ycur(nyh+i)/h  (i=1,...,neq)  (and the division by h is
c unnecessary when nst = 0).
c-----------------------------------------------------------------------
c-----------------------------------------------------------------------
c other routines in the lsodar package.
c
c in addition to subroutine lsodar, the lsodar package includes the
c following subroutines and function routines..
c  rchek    does preliminary checking for roots, and serves as an
c           interface between subroutine lsodar and subroutine roots.
c  roots    finds the leftmost root of a set of functions.
c  intdy    computes an interpolated value of the y vector at t = tout.
c  stoda    is the core integrator, which does one step of the
c           integration and the associated error control.
c  cfode    sets all method coefficients and test constants.
c  prja     computes and preprocesses the jacobian matrix j = df/dy
c           and the newton iteration matrix p = i - h*l0*j.
c  solsy    manages solution of linear system in chord iteration.
c  ewset    sets the error weight vector ewt before each step.
c  vmnorm   computes the weighted max-norm of a vector.
c  fnorm    computes the norm of a full matrix consistent with the
c           weighted max-norm on vectors.
c  bnorm    computes the norm of a band matrix consistent with the
c           weighted max-norm on vectors.
c  srcar    is a user-callable routine to save and restore
c           the contents of the internal common blocks.
c  mondgefa and mondgesl   are routines from linpack for solving full
c           systems of linear algebraic equations.
c  dgbfa and dgbsl   are routines from linpack for solving banded
c           linear systems.
c  mondaxpy, mondscal, monidamax, monddot, and mondcopy   are basic linear algebra
c           modules (blas) used by the above linpack routines.
c  mond1mach   computes the unit roundoff in a machine-independent manner.
c  monwv, xsetun, and xsetf   handle the printing of all error
c           messages and warnings.  monwv is machine-dependent.
c note..  vmnorm, fnorm, bnorm, monidamax, monddot, and mond1mach are function
c routines.  all the others are subroutines.
c
c the intrinsic and external routines used by lsodar are..
c dabs, dmax1, dmin1, dfloat, max0, min0, mod, dsign, dsqrt, and write.
c
c a block data subprogram is also included with the package,
c for loading some of the variables in internal common.
c
c-----------------------------------------------------------------------
c the following card is for optimized compilation on lll compilers.
clll. optimize
c-----------------------------------------------------------------------
      external prja, solsy
      integer illin, init, lyh, lewt, lacor, lsavf, lwm, liwm,
     1   mxstep, mxhnil, nhnil, ntrep, nslast, nyh, iowns
      integer icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     1   maxord, maxcor, msbp, mxncf, n, nq, nst, nfe, nje, nqu
      integer insufr, insufi, ixpr, iowns2, jtyp, mused, mxordn, mxords
      integer lg0, lg1, lgx, iownr3, irfnd, itaskc, ngc, nge
      integer i, i1, i2, iflag, imxer, kgo, lf0,
     1   leniw, lenrw, lenwm, ml, mord, mu, mxhnl0, mxstp0
      integer len1, len1c, len1n, len1s, len2, leniwc,
     1   lenrwc, lenrwn, lenrws
      integer irfp, irt, lenyh, lyhnew
      double precision rowns,
     1   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround
      double precision tsw, rowns2, pdnorm
      double precision rownr3, t0, tlast, toutc
      double precision atoli, ayi, big, ewti, h0, hmax, hmx, rh, rtoli,
     1   tcrit, tdist, tnext, tol, tolsf, tp, size, sum, w0,
     2   mond1mach, vmnorm
      dimension mord(2)
      logical ihit
c-----------------------------------------------------------------------
c the following three internal common blocks contain
c (a) variables which are local to any subroutine but whose values must
c     be preserved between calls to the routine (own variables), and
c (b) variables which are communicated between subroutines.
c the structure of each block is as follows..  all real variables are
c listed first, followed by all integers.  within each type, the
c variables are grouped with those local to subroutine lsodar first,
c then those local to subroutine roots or subroutine stoda
c (no other routines have own variables), and finally those used
c for communication.  the block ls0001 is declared in subroutines
c lsodar, intdy, stoda, prja, and solsy.  the block lsa001 is declared
c in subroutines lsodar, stoda, and prja.  the block lsr001 is declared
c in subroutines lsodar, rchek, and roots.  groups of variables are
c replaced by dummy arrays in the common declarations in routines
c where those variables are not used.
c-----------------------------------------------------------------------
      common /ls0001/ rowns(209),
     1   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround,
     2   illin, init, lyh, lewt, lacor, lsavf, lwm, liwm,
     3   mxstep, mxhnil, nhnil, ntrep, nslast, nyh, iowns(6),
     4   icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     5   maxord, maxcor, msbp, mxncf, n, nq, nst, nfe, nje, nqu
      common /lsa001/ tsw, rowns2(20), pdnorm,
     1   insufr, insufi, ixpr, iowns2(2), jtyp, mused, mxordn, mxords
      common /lsr001/ rownr3(2), t0, tlast, toutc,
     1   lg0, lg1, lgx, iownr3(2), irfnd, itaskc, ngc, nge
c
      data mord(1),mord(2)/12,5/, mxstp0/500/, mxhnl0/10/
c-----------------------------------------------------------------------
c block a.
c this code block is executed on every call.
c it tests istate and itask for legality and branches appropriately.
c if istate .gt. 1 but the flag init shows that initialization has
c not yet been done, an error return occurs.
c if istate = 1 and tout = t, jump to block g and return immediately.
c-----------------------------------------------------------------------
      if (istate .lt. 1 .or. istate .gt. 3) go to 601
      if (itask .lt. 1 .or. itask .gt. 5) go to 602
      itaskc = itask
      if (istate .eq. 1) go to 10
      if (init .eq. 0) go to 603
      if (istate .eq. 2) go to 200
      go to 20
 10   init = 0
      if (tout .eq. t) go to 430
 20   ntrep = 0
c-----------------------------------------------------------------------
c block b.
c the next code block is executed for the initial call (istate = 1),
c or for a continuation call with parameter changes (istate = 3).
c it contains checking of all inputs and various initializations.
c
c first check legality of the non-optional inputs neq, itol, iopt,
c jt, ml, mu, and ng.
c-----------------------------------------------------------------------
      if (neq(1) .le. 0) go to 604
      if (istate .eq. 1) go to 25
      if (neq(1) .gt. n) go to 605
 25   n = neq(1)
      if (itol .lt. 1 .or. itol .gt. 4) go to 606
      if (iopt .lt. 0 .or. iopt .gt. 1) go to 607
      if (jt .eq. 3 .or. jt .lt. 1 .or. jt .gt. 5) go to 608
      jtyp = jt
      if (jt .le. 2) go to 30
      ml = iwork(1)
      mu = iwork(2)
      if (ml .lt. 0 .or. ml .ge. n) go to 609
      if (mu .lt. 0 .or. mu .ge. n) go to 610
 30   continue
      if (ng .lt. 0) go to 630
      if (istate .eq. 1) go to 35
      if (irfnd .eq. 0 .and. ng .ne. ngc) go to 631
 35   ngc = ng
c next process and check the optional inputs. --------------------------
      if (iopt .eq. 1) go to 40
      ixpr = 0
      mxstep = mxstp0
      mxhnil = mxhnl0
      hmxi = 0.0d0
      hmin = 0.0d0
      if (istate .ne. 1) go to 60
      h0 = 0.0d0
      mxordn = mord(1)
      mxords = mord(2)
      go to 60
 40   ixpr = iwork(5)
      if (ixpr .lt. 0 .or. ixpr .gt. 1) go to 611
      mxstep = iwork(6)
      if (mxstep .lt. 0) go to 612
      if (mxstep .eq. 0) mxstep = mxstp0
      mxhnil = iwork(7)
      if (mxhnil .lt. 0) go to 613
      if (mxhnil .eq. 0) mxhnil = mxhnl0
      if (istate .ne. 1) go to 50
      h0 = rwork(5)
      mxordn = iwork(8)
      if (mxordn .lt. 0) go to 628
      if (mxordn .eq. 0) mxordn = 100
      mxordn = min0(mxordn,mord(1))
      mxords = iwork(9)
      if (mxords .lt. 0) go to 629
      if (mxords .eq. 0) mxords = 100
      mxords = min0(mxords,mord(2))
      if ((tout - t)*h0 .lt. 0.0d0) go to 614
 50   hmax = rwork(6)
      if (hmax .lt. 0.0d0) go to 615
      hmxi = 0.0d0
      if (hmax .gt. 0.0d0) hmxi = 1.0d0/hmax
      hmin = rwork(7)
      if (hmin .lt. 0.0d0) go to 616
c-----------------------------------------------------------------------
c set work array pointers and check lengths lrw and liw.
c if istate = 1, meth is initialized to 1 here to facilitate the
c checking of work space lengths.
c pointers to segments of rwork and iwork are named by prefixing l to
c the name of the segment.  e.g., the segment yh starts at rwork(lyh).
c segments of rwork (in order) are denoted  g0, g1, gx, yh, wm,
c ewt, savf, acor.
c if the lengths provided are insufficient for the current method,
c an error return occurs.  this is treated as illegal input on the
c first call, but as a problem interruption with istate = -7 on a
c continuation call.  if the lengths are sufficient for the current
c method but not for both methods, a warning message is sent.
c-----------------------------------------------------------------------
 60   if (istate .eq. 1) meth = 1
      if (istate .eq. 1) nyh = n
      lg0 = 21
      lg1 = lg0 + ng
      lgx = lg1 + ng
      lyhnew = lgx + ng
      if (istate .eq. 1) lyh = lyhnew
      if (lyhnew .eq. lyh) go to 62
c if istate = 3 and ng was changed, shift yh to its new location. ------
      lenyh = l*nyh
      if (lrw .lt. lyhnew-1+lenyh) go to 62
      i1 = 1
      if (lyhnew .gt. lyh) i1 = -1
      call mondcopy (lenyh, rwork(lyh), i1, rwork(lyhnew), i1)
      lyh = lyhnew
 62   continue
      len1n = lyhnew - 1 + (mxordn + 1)*nyh
      len1s = lyhnew - 1 + (mxords + 1)*nyh
      lwm = len1s + 1
      if (jt .le. 2) lenwm = n*n + 2
      if (jt .ge. 4) lenwm = (2*ml + mu + 1)*n + 2
      len1s = len1s + lenwm
      len1c = len1n
      if (meth .eq. 2) len1c = len1s
      len1 = max0(len1n,len1s)
      len2 = 3*n
      lenrw = len1 + len2
      lenrwn = len1n + len2
      lenrws = len1s + len2
      lenrwc = len1c + len2
      iwork(17) = lenrw
      liwm = 1
      leniw = 20 + n
      leniwc = 20
      if (meth .eq. 2) leniwc = leniw
      iwork(18) = leniw
      if (istate .eq. 1 .and. lrw .lt. lenrwc) go to 617
      if (istate .eq. 1 .and. liw .lt. leniwc) go to 618
      if (istate .eq. 3 .and. lrw .lt. lenrwc) go to 550
      if (istate .eq. 3 .and. liw .lt. leniwc) go to 555
      lewt = len1 + 1
      insufr = 0
      if (lrw .ge. lenrw) go to 65
      insufr = 2
      lewt = len1c + 1
      call monwv(
     1  60hlsodar-  warning.. rwork length is sufficient for now, but  ,
     1   60, 103, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  60h      may not be later.  integration will proceed anyway.   ,
     1   60, 103, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  50h      length needed is lenrw = i1, while lrw = i2.,
     1   50, 103, 0, 2, lenrw, lrw, 0, 0.0d0, 0.0d0)
 65   lsavf = lewt + n
      lacor = lsavf + n
      insufi = 0
      if (liw .ge. leniw) go to 70
      insufi = 2
      call monwv(
     1  60hlsodar-  warning.. iwork length is sufficient for now, but  ,
     1   60, 104, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  60h      may not be later.  integration will proceed anyway.   ,
     1   60, 104, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  50h      length needed is leniw = i1, while liw = i2.,
     1   50, 104, 0, 2, leniw, liw, 0, 0.0d0, 0.0d0)
 70   continue
c check rtol and atol for legality. ------------------------------------
      rtoli = rtol(1)
      atoli = atol(1)
      do 75 i = 1,n
        if (itol .ge. 3) rtoli = rtol(i)
        if (itol .eq. 2 .or. itol .eq. 4) atoli = atol(i)
        if (rtoli .lt. 0.0d0) go to 619
        if (atoli .lt. 0.0d0) go to 620
 75     continue
      if (istate .eq. 1) go to 100
c if istate = 3, set flag to signal parameter changes to stoda. --------
      jstart = -1
      if (n .eq. nyh) go to 200
c neq was reduced.  zero part of yh to avoid undefined references. -----
      i1 = lyh + l*nyh
      i2 = lyh + (maxord + 1)*nyh - 1
      if (i1 .gt. i2) go to 200
      do 95 i = i1,i2
 95     rwork(i) = 0.0d0
      go to 200
c-----------------------------------------------------------------------
c block c.
c the next block is for the initial call only (istate = 1).
c it contains all remaining initializations, the initial call to f,
c and the calculation of the initial step size.
c the error weights in ewt are inverted after being loaded.
c-----------------------------------------------------------------------
 100  uround = mond1mach(4)
      tn = t
      tsw = t
      maxord = mxordn
      if (itask .ne. 4 .and. itask .ne. 5) go to 110
      tcrit = rwork(1)
      if ((tcrit - tout)*(tout - t) .lt. 0.0d0) go to 625
      if (h0 .ne. 0.0d0 .and. (t + h0 - tcrit)*h0 .gt. 0.0d0)
     1   h0 = tcrit - t
 110  jstart = 0
      nhnil = 0
      nst = 0
      nje = 0
      nslast = 0
      hu = 0.0d0
      nqu = 0
      mused = 0
      miter = 0
      ccmax = 0.3d0
      maxcor = 3
      msbp = 20
      mxncf = 10
c initial call to f.  (lf0 points to yh(*,2).) -------------------------
      lf0 = lyh + nyh
      call f (neq, t, y, rwork(lf0))
      nfe = 1
c load the initial value vector in yh. ---------------------------------
      do 115 i = 1,n
 115    rwork(i+lyh-1) = y(i)
c load and invert the ewt array.  (h is temporarily set to 1.0.) -------
      nq = 1
      h = 1.0d0
      call ewset (n, itol, rtol, atol, rwork(lyh), rwork(lewt))
      do 120 i = 1,n
        if (rwork(i+lewt-1) .le. 0.0d0) go to 621
 120    rwork(i+lewt-1) = 1.0d0/rwork(i+lewt-1)
c-----------------------------------------------------------------------
c the coding below computes the step size, h0, to be attempted on the
c first step, unless the user has supplied a value for this.
c first check that tout - t differs significantly from zero.
c a scalar tolerance quantity tol is computed, as max(rtol(i))
c if this is positive, or max(atol(i)/abs(y(i))) otherwise, adjusted
c so as to be between 100*uround and 1.0e-3.
c then the computed value h0 is given by..
c
c   h0**(-2)  =  1./(tol * w0**2)  +  tol * (norm(f))**2
c
c where   w0     = max ( abs(t), abs(tout) ),
c         f      = the initial value of the vector f(t,y), and
c         norm() = the weighted vector norm used throughout, given by
c                  the vmnorm function routine, and weighted by the
c                  tolerances initially loaded into the ewt array.
c the sign of h0 is inferred from the initial values of tout and t.
c abs(h0) is made .le. abs(tout-t) in any case.
c-----------------------------------------------------------------------
      if (h0 .ne. 0.0d0) go to 180
      tdist = dabs(tout - t)
      w0 = dmax1(dabs(t),dabs(tout))
      if (tdist .lt. 2.0d0*uround*w0) go to 622
      tol = rtol(1)
      if (itol .le. 2) go to 140
      do 130 i = 1,n
 130    tol = dmax1(tol,rtol(i))
 140  if (tol .gt. 0.0d0) go to 160
      atoli = atol(1)
      do 150 i = 1,n
        if (itol .eq. 2 .or. itol .eq. 4) atoli = atol(i)
        ayi = dabs(y(i))
        if (ayi .ne. 0.0d0) tol = dmax1(tol,atoli/ayi)
 150    continue
 160  tol = dmax1(tol,100.0d0*uround)
      tol = dmin1(tol,0.001d0)
      sum = vmnorm (n, rwork(lf0), rwork(lewt))
      sum = 1.0d0/(tol*w0*w0) + tol*sum**2
      h0 = 1.0d0/dsqrt(sum)
      h0 = dmin1(h0,tdist)
      h0 = dsign(h0,tout-t)
c adjust h0 if necessary to meet hmax bound. ---------------------------
 180  rh = dabs(h0)*hmxi
      if (rh .gt. 1.0d0) h0 = h0/rh
c load h with h0 and scale yh(*,2) by h0. ------------------------------
      h = h0
      do 190 i = 1,n
 190    rwork(i+lf0-1) = h0*rwork(i+lf0-1)
c
c check for a zero of g at t. ------------------------------------------
      irfnd = 0
      toutc = tout
      if (ngc .eq. 0) go to 270
      call rchek (1, g, neq, y, rwork(lyh), nyh,
     1   rwork(lg0), rwork(lg1), rwork(lgx), jroot, irt)
      if (irt .eq. 0) go to 270
      go to 632
c-----------------------------------------------------------------------
c block d.
c the next code block is for continuation calls only (istate = 2 or 3)
c and is to check stop conditions before taking a step.
c first, rchek is called to check for a root within the last step
c taken, other than the last root found there, if any.
c if itask = 2 or 5, and y(tn) has not yet been returned to the user
c because of an intervening root, return through block g.
c-----------------------------------------------------------------------
 200  nslast = nst
c
      irfp = irfnd
      if (ngc .eq. 0) go to 205
      if (itask .eq. 1 .or. itask .eq. 4) toutc = tout
      call rchek (2, g, neq, y, rwork(lyh), nyh,
     1   rwork(lg0), rwork(lg1), rwork(lgx), jroot, irt)
      if (irt .ne. 1) go to 205
      irfnd = 1
      istate = 3
      t = t0
      go to 425
 205  continue
      irfnd = 0
      if (irfp .eq. 1 .and. tlast .ne. tn .and. itask .eq. 2) go to 400
c
      go to (210, 250, 220, 230, 240), itask
 210  if ((tn - tout)*h .lt. 0.0d0) go to 250
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      if (iflag .ne. 0) go to 627
      t = tout
      go to 420
 220  tp = tn - hu*(1.0d0 + 100.0d0*uround)
      if ((tp - tout)*h .gt. 0.0d0) go to 623
      if ((tn - tout)*h .lt. 0.0d0) go to 250
      t = tn
      go to 400
 230  tcrit = rwork(1)
      if ((tn - tcrit)*h .gt. 0.0d0) go to 624
      if ((tcrit - tout)*h .lt. 0.0d0) go to 625
      if ((tn - tout)*h .lt. 0.0d0) go to 245
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      if (iflag .ne. 0) go to 627
      t = tout
      go to 420
 240  tcrit = rwork(1)
      if ((tn - tcrit)*h .gt. 0.0d0) go to 624
 245  hmx = dabs(tn) + dabs(h)
      ihit = dabs(tn - tcrit) .le. 100.0d0*uround*hmx
      if (ihit) t = tcrit
      if (irfp .eq. 1 .and. tlast .ne. tn .and. itask .eq. 5) go to 400
      if (ihit) go to 400
      tnext = tn + h*(1.0d0 + 4.0d0*uround)
      if ((tnext - tcrit)*h .le. 0.0d0) go to 250
      h = (tcrit - tn)*(1.0d0 - 4.0d0*uround)
      if (istate .eq. 2) jstart = -2
c-----------------------------------------------------------------------
c block e.
c the next block is normally executed for all calls and contains
c the call to the one-step core integrator stoda.
c
c this is a looping point for the integration steps.
c
c first check for too many steps being taken, update ewt (if not at
c start of problem), check for too much accuracy being requested, and
c check for h below the roundoff level in t.
c-----------------------------------------------------------------------
 250  continue
      if (meth .eq. mused) go to 255
      if (insufr .eq. 1) go to 550
      if (insufi .eq. 1) go to 555
 255  if ((nst-nslast) .ge. mxstep) go to 500
      call ewset (n, itol, rtol, atol, rwork(lyh), rwork(lewt))
      do 260 i = 1,n
        if (rwork(i+lewt-1) .le. 0.0d0) go to 510
 260    rwork(i+lewt-1) = 1.0d0/rwork(i+lewt-1)
 270  tolsf = uround*vmnorm (n, rwork(lyh), rwork(lewt))
      if (tolsf .le. 0.01d0) go to 280
      tolsf = tolsf*200.0d0
      if (nst .eq. 0) go to 626
      go to 520
 280  if ((tn + h) .ne. tn) go to 290
      nhnil = nhnil + 1
      if (nhnil .gt. mxhnil) go to 290
      call monwv(50hlsodar-  warning..internal t (=r1) and h (=r2) are,
     1   50, 101, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  60h      such that in the machine, t + h = t on the next step  ,
     1   60, 101, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      (h = step size). solver will continue anyway,
     1   50, 101, 0, 0, 0, 0, 2, tn, h)
      if (nhnil .lt. mxhnil) go to 290
      call monwv(50hlsodar-  above warning has been issued i1 times.  ,
     1   50, 102, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      it will not be issued again for this problem,
     1   50, 102, 0, 1, mxhnil, 0, 0, 0.0d0, 0.0d0)
 290  continue
c-----------------------------------------------------------------------
c     call stoda(neq,y,yh,nyh,yh,ewt,savf,acor,wm,iwm,f,jac,prja,solsy)
c-----------------------------------------------------------------------
      call stoda (neq, y, rwork(lyh), nyh, rwork(lyh), rwork(lewt),
     1   rwork(lsavf), rwork(lacor), rwork(lwm), iwork(liwm),
     2   f, jac, prja, solsy)
      kgo = 1 - kflag
      go to (300, 530, 540), kgo
c-----------------------------------------------------------------------
c block f.
c the following block handles the case of a successful return from the
c core integrator (kflag = 0).
c if a method switch was just made, record tsw, reset maxord,
c set jstart to -1 to signal stoda to complete the switch,
c and do extra printing of data if ixpr = 1.
c then call rchek to check for a root within the last step.
c then, if no root was found, check for stop conditions.
c-----------------------------------------------------------------------
 300  init = 1
      if (meth .eq. mused) go to 310
      tsw = tn
      maxord = mxordn
      if (meth .eq. 2) maxord = mxords
      if (meth .eq. 2) rwork(lwm) = dsqrt(uround)
      insufr = min0(insufr,1)
      insufi = min0(insufi,1)
      jstart = -1
      if (ixpr .eq. 0) go to 310
      if (meth .eq. 2) call monwv(
     1  60hlsodar- a switch to the bdf (stiff) method has occurred     ,
     1   60, 105, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      if (meth .eq. 1) call monwv(
     1  60hlsodar- a switch to the adams (nonstiff) method has occurred,
     1   60, 106, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  60h     at t = r1,  tentative step size h = r2,  step nst = i1 ,
     1   60, 107, 0, 1, nst, 0, 2, tn, h)
 310  continue
c
      if (ngc .eq. 0) go to 315
      call rchek (3, g, neq, y, rwork(lyh), nyh,
     1   rwork(lg0), rwork(lg1), rwork(lgx), jroot, irt)
      if (irt .ne. 1) go to 315
      irfnd = 1
      istate = 3
      t = t0
      go to 425
 315  continue
c
      go to (320, 400, 330, 340, 350), itask
c itask = 1.  if tout has been reached, interpolate. -------------------
 320  if ((tn - tout)*h .lt. 0.0d0) go to 250
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      t = tout
      go to 420
c itask = 3.  jump to exit if tout was reached. ------------------------
 330  if ((tn - tout)*h .ge. 0.0d0) go to 400
      go to 250
c itask = 4.  see if tout or tcrit was reached.  adjust h if necessary.
 340  if ((tn - tout)*h .lt. 0.0d0) go to 345
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      t = tout
      go to 420
 345  hmx = dabs(tn) + dabs(h)
      ihit = dabs(tn - tcrit) .le. 100.0d0*uround*hmx
      if (ihit) go to 400
      tnext = tn + h*(1.0d0 + 4.0d0*uround)
      if ((tnext - tcrit)*h .le. 0.0d0) go to 250
      h = (tcrit - tn)*(1.0d0 - 4.0d0*uround)
      jstart = -2
      go to 250
c itask = 5.  see if tcrit was reached and jump to exit. ---------------
 350  hmx = dabs(tn) + dabs(h)
      ihit = dabs(tn - tcrit) .le. 100.0d0*uround*hmx
c-----------------------------------------------------------------------
c block g.
c the following block handles all successful returns from lsodar.
c if itask .ne. 1, y is loaded from yh and t is set accordingly.
c istate is set to 2, the illegal input counter is zeroed, and the
c optional outputs are loaded into the work arrays before returning.
c if istate = 1 and tout = t, there is a return with no action taken,
c except that if this has happened repeatedly, the run is terminated.
c-----------------------------------------------------------------------
 400  do 410 i = 1,n
 410    y(i) = rwork(i+lyh-1)
      t = tn
      if (itask .ne. 4 .and. itask .ne. 5) go to 420
      if (ihit) t = tcrit
 420  istate = 2
 425  continue
      illin = 0
      rwork(11) = hu
      rwork(12) = h
      rwork(13) = tn
      rwork(15) = tsw
      iwork(11) = nst
      iwork(12) = nfe
      iwork(13) = nje
      iwork(14) = nqu
      iwork(15) = nq
      iwork(19) = mused
      iwork(20) = meth
      iwork(10) = nge
      tlast = t
      return
c
 430  ntrep = ntrep + 1
      if (ntrep .lt. 5) return
      call monwv(
     1  60hlsodar-  repeated calls with istate = 1 and tout = t (=r1)  ,
     1   60, 301, 0, 0, 0, 0, 1, t, 0.0d0)
      go to 800
c-----------------------------------------------------------------------
c block h.
c the following block handles all unsuccessful returns other than
c those for illegal input.  first the error message routine is called.
c if there was an error test or convergence test failure, imxer is set.
c then y is loaded from yh, t is set to tn, and the illegal input
c counter illin is set to 0.  the optional outputs are loaded into
c the work arrays before returning.
c-----------------------------------------------------------------------
c the maximum number of steps was taken before reaching tout. ----------
 500  call monwv(50hlsodar-  at current t (=r1), mxstep (=i1) steps   ,
     1   50, 201, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      taken on this call before reaching tout     ,
     1   50, 201, 0, 1, mxstep, 0, 1, tn, 0.0d0)
      istate = -1
      go to 580
c ewt(i) .le. 0.0 for some i (not at start of problem). ----------------
 510  ewti = rwork(lewt+i-1)
      call monwv(50hlsodar-  at t (=r1), ewt(i1) has become r2 .le. 0.,
     1   50, 202, 0, 1, i, 0, 2, tn, ewti)
      istate = -6
      go to 580
c too much accuracy requested for machine precision. -------------------
 520  call monwv(50hlsodar-  at t (=r1), too much accuracy requested  ,
     1   50, 203, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      for precision of machine..  see tolsf (=r2) ,
     1   50, 203, 0, 0, 0, 0, 2, tn, tolsf)
      rwork(14) = tolsf
      istate = -2
      go to 580
c kflag = -1.  error test failed repeatedly or with abs(h) = hmin. -----
 530  call monwv(50hlsodar-  at t(=r1) and step size h(=r2), the error,
     1   50, 204, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      test failed repeatedly or with abs(h) = hmin,
     1   50, 204, 0, 0, 0, 0, 2, tn, h)
      istate = -4
      go to 560
c kflag = -2.  convergence failed repeatedly or with abs(h) = hmin. ----
 540  call monwv(50hlsodar-  at t (=r1) and step size h (=r2), the    ,
     1   50, 205, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      corrector convergence failed repeatedly     ,
     1   50, 205, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(30h      or with abs(h) = hmin   ,
     1   30, 205, 0, 0, 0, 0, 2, tn, h)
      istate = -5
      go to 560
c rwork length too small to proceed. -----------------------------------
 550  call monwv(50hlsodar-  at current t(=r1), rwork length too small,
     1   50, 206, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  60h      to proceed.  the integration was otherwise successful.,
     1   60, 206, 0, 0, 0, 0, 1, tn, 0.0d0)
      istate = -7
      go to 580
c iwork length too small to proceed. -----------------------------------
 555  call monwv(50hlsodar-  at current t(=r1), iwork length too small,
     1   50, 207, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  60h      to proceed.  the integration was otherwise successful.,
     1   60, 207, 0, 0, 0, 0, 1, tn, 0.0d0)
      istate = -7
      go to 580
c compute imxer if relevant. -------------------------------------------
 560  big = 0.0d0
      imxer = 1
      do 570 i = 1,n
        size = dabs(rwork(i+lacor-1)*rwork(i+lewt-1))
        if (big .ge. size) go to 570
        big = size
        imxer = i
 570    continue
      iwork(16) = imxer
c set y vector, t, illin, and optional outputs. ------------------------
 580  do 590 i = 1,n
 590    y(i) = rwork(i+lyh-1)
      t = tn
      illin = 0
      rwork(11) = hu
      rwork(12) = h
      rwork(13) = tn
      rwork(15) = tsw
      iwork(11) = nst
      iwork(12) = nfe
      iwork(13) = nje
      iwork(14) = nqu
      iwork(15) = nq
      iwork(19) = mused
      iwork(20) = meth
      iwork(10) = nge
      tlast = t
      return
c-----------------------------------------------------------------------
c block i.
c the following block handles all error returns due to illegal input
c (istate = -3), as detected before calling the core integrator.
c first the error message routine is called.  then if there have been
c 5 consecutive such returns just before this call to the solver,
c the run is halted.
c-----------------------------------------------------------------------
 601  call monwv(30hlsodar-  istate (=i1) illegal ,
     1   30, 1, 0, 1, istate, 0, 0, 0.0d0, 0.0d0)
      go to 700
 602  call monwv(30hlsodar-  itask (=i1) illegal  ,
     1   30, 2, 0, 1, itask, 0, 0, 0.0d0, 0.0d0)
      go to 700
 603  call monwv(50hlsodar-  istate .gt. 1 but lsodar not initialized ,
     1   50, 3, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      go to 700
 604  call monwv(30hlsodar-  neq (=i1) .lt. 1     ,
     1   30, 4, 0, 1, neq(1), 0, 0, 0.0d0, 0.0d0)
      go to 700
 605  call monwv(50hlsodar-  istate = 3 and neq increased (i1 to i2)  ,
     1   50, 5, 0, 2, n, neq(1), 0, 0.0d0, 0.0d0)
      go to 700
 606  call monwv(30hlsodar-  itol (=i1) illegal   ,
     1   30, 6, 0, 1, itol, 0, 0, 0.0d0, 0.0d0)
      go to 700
 607  call monwv(30hlsodar-  iopt (=i1) illegal   ,
     1   30, 7, 0, 1, iopt, 0, 0, 0.0d0, 0.0d0)
      go to 700
 608  call monwv(30hlsodar-  jt (=i1) illegal     ,
     1   30, 8, 0, 1, jt, 0, 0, 0.0d0, 0.0d0)
      go to 700
 609  call monwv(50hlsodar-  ml (=i1) illegal.. .lt.0 or .ge.neq (=i2),
     1   50, 9, 0, 2, ml, neq(1), 0, 0.0d0, 0.0d0)
      go to 700
 610  call monwv(50hlsodar-  mu (=i1) illegal.. .lt.0 or .ge.neq (=i2),
     1   50, 10, 0, 2, mu, neq(1), 0, 0.0d0, 0.0d0)
      go to 700
 611  call monwv(30hlsodar-  ixpr (=i1) illegal   ,
     1   30, 11, 0, 1, ixpr, 0, 0, 0.0d0, 0.0d0)
      go to 700
 612  call monwv(30hlsodar-  mxstep (=i1) .lt. 0  ,
     1   30, 12, 0, 1, mxstep, 0, 0, 0.0d0, 0.0d0)
      go to 700
 613  call monwv(30hlsodar-  mxhnil (=i1) .lt. 0  ,
     1   30, 13, 0, 1, mxhnil, 0, 0, 0.0d0, 0.0d0)
      go to 700
 614  call monwv(40hlsodar-  tout (=r1) behind t (=r2)      ,
     1   40, 14, 0, 0, 0, 0, 2, tout, t)
      call monwv(50h      integration direction is given by h0 (=r1)  ,
     1   50, 14, 0, 0, 0, 0, 1, h0, 0.0d0)
      go to 700
 615  call monwv(30hlsodar-  hmax (=r1) .lt. 0.0  ,
     1   30, 15, 0, 0, 0, 0, 1, hmax, 0.0d0)
      go to 700
 616  call monwv(30hlsodar-  hmin (=r1) .lt. 0.0  ,
     1   30, 16, 0, 0, 0, 0, 1, hmin, 0.0d0)
      go to 700
 617  call monwv(
     1  60hlsodar-  rwork length needed, lenrw (=i1), exceeds lrw (=i2),
     1   60, 17, 0, 2, lenrw, lrw, 0, 0.0d0, 0.0d0)
      go to 700
 618  call monwv(
     1  60hlsodar-  iwork length needed, leniw (=i1), exceeds liw (=i2),
     1   60, 18, 0, 2, leniw, liw, 0, 0.0d0, 0.0d0)
      go to 700
 619  call monwv(40hlsodar-  rtol(i1) is r1 .lt. 0.0        ,
     1   40, 19, 0, 1, i, 0, 1, rtoli, 0.0d0)
      go to 700
 620  call monwv(40hlsodar-  atol(i1) is r1 .lt. 0.0        ,
     1   40, 20, 0, 1, i, 0, 1, atoli, 0.0d0)
      go to 700
 621  ewti = rwork(lewt+i-1)
      call monwv(40hlsodar-  ewt(i1) is r1 .le. 0.0         ,
     1   40, 21, 0, 1, i, 0, 1, ewti, 0.0d0)
      go to 700
 622  call monwv(
     1  60hlsodar-  tout (=r1) too close to t(=r2) to start integration,
     1   60, 22, 0, 0, 0, 0, 2, tout, t)
      go to 700
 623  call monwv(
     1  60hlsodar-  itask = i1 and tout (=r1) behind tcur - hu (= r2)  ,
     1   60, 23, 0, 1, itask, 0, 2, tout, tp)
      go to 700
 624  call monwv(
     1  60hlsodar-  itask = 4 or 5 and tcrit (=r1) behind tcur (=r2)   ,
     1   60, 24, 0, 0, 0, 0, 2, tcrit, tn)
      go to 700
 625  call monwv(
     1  60hlsodar-  itask = 4 or 5 and tcrit (=r1) behind tout (=r2)   ,
     1   60, 25, 0, 0, 0, 0, 2, tcrit, tout)
      go to 700
 626  call monwv(50hlsodar-  at start of problem, too much accuracy   ,
     1   50, 26, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  60h      requested for precision of machine..  see tolsf (=r1) ,
     1   60, 26, 0, 0, 0, 0, 1, tolsf, 0.0d0)
      rwork(14) = tolsf
      go to 700
 627  call monwv(50hlsodar-  trouble from intdy. itask = i1, tout = r1,
     1   50, 27, 0, 1, itask, 0, 1, tout, 0.0d0)
      go to 700
 628  call monwv(30hlsodar-  mxordn (=i1) .lt. 0  ,
     1   30, 28, 0, 1, mxordn, 0, 0, 0.0d0, 0.0d0)
      go to 700
 629  call monwv(30hlsodar-  mxords (=i1) .lt. 0  ,
     1   30, 29, 0, 1, mxords, 0, 0, 0.0d0, 0.0d0)
      go to 700
 630  call monwv(30hlsodar-  ng (=i1) .lt. 0      ,
     1   30, 30, 0, 1, ng, 0, 0, 0.0d0, 0.0d0)
      go to 700
 631  call monwv(50hlsodar-  ng changed (from i1 to i2) illegally,    ,
     1   50, 31, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      i.e. not immediately after a root was found ,
     1   50, 31, 0, 2, ngc, ng, 0, 0.0d0, 0.0d0)
      go to 700
 632  call monwv(50hlsodar-  one or more components of g has a root   ,
     1   50, 32, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(40h      too near to the initial point     ,
     1   40, 32, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
c
 700  if (illin .eq. 5) go to 710
      illin = illin + 1
      tlast = t
      istate = -3
      return
 710  call monwv(50hlsodar-  repeated occurrences of illegal input    ,
     1   50, 302, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
c
 800  call monwv(50hlsodar-  run aborted.. apparent infinite loop     ,
     1   50, 303, 2, 0, 0, 0, 0, 0.0d0, 0.0d0)
      return
c----------------------- end of subroutine lsodar ----------------------
      end
      subroutine rchek (job, g, neq, y, yh, nyh, g0, g1, gx, jroot, irt)
clll. optimize
      external g
      integer job, neq, nyh, jroot, irt
      double precision y, yh, g0, g1, gx
      dimension neq(1), y(1), yh(nyh,1), g0(1), g1(1), gx(1), jroot(1)
      integer iownd, iowns,
     1   icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     2   maxord, maxcor, msbp, mxncf, n, nq, nst, nfe, nje, nqu
      integer iownd3, iownr3, irfnd, itaskc, ngc, nge
      integer i, iflag, jflag
      double precision rowns,
     1   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround
      double precision rownr3, t0, tlast, toutc
      double precision hming, t1, temp1, temp2, x
      logical zroot
      common /ls0001/ rowns(209),
     2   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround,
     3   iownd(14), iowns(6),
     4   icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     5   maxord, maxcor, msbp, mxncf, n, nq, nst, nfe, nje, nqu
      common /lsr001/ rownr3(2), t0, tlast, toutc,
     1   iownd3(3), iownr3(2), irfnd, itaskc, ngc, nge
c-----------------------------------------------------------------------
c this routine checks for the presence of a root in the
c vicinity of the current t, in a manner depending on the
c input flag job.  it calls subroutine roots to locate the root
c as precisely as possible.
c
c in addition to variables described previously, rchek
c uses the following for communication..
c job    = integer flag indicating type of call..
c          job = 1 means the problem is being initialized, and rchek
c                  is to look for a root at or very near the initial t.
c          job = 2 means a continuation call to the solver was just
c                  made, and rchek is to check for a root in the
c                  relevant part of the step last taken.
c          job = 3 means a successful step was just taken, and rchek
c                  is to look for a root in the interval of the step.
c g0     = array of length ng, containing the value of g at t = t0.
c          g0 is input for job .ge. 2 and on output in all cases.
c g1,gx  = arrays of length ng for work space.
c irt    = completion flag..
c          irt = 0  means no root was found.
c          irt = -1 means job = 1 and a root was found too near to t.
c          irt = 1  means a legitimate root was found (job = 2 or 3).
c                   on return, t0 is the root location, and y is the
c                   corresponding solution vector.
c t0     = value of t at one endpoint of interval of interest.  only
c          roots beyond t0 in the direction of integration are sought.
c          t0 is input if job .ge. 2, and output in all cases.
c          t0 is updated by rchek, whether a root is found or not.
c tlast  = last value of t returned by the solver (input only).
c toutc  = copy of tout (input only).
c irfnd  = input flag showing whether the last step taken had a root.
c          irfnd = 1 if it did, = 0 if not.
c itaskc = copy of itask (input only).
c ngc    = copy of ng (input only).
c-----------------------------------------------------------------------
c
      irt = 0
      do 10 i = 1,ngc
 10     jroot(i) = 0
      hming = (dabs(tn) + dabs(h))*uround*100.0d0
c
      go to (100, 200, 300), job
c
c evaluate g at initial t, and check for zero values. ------------------
 100  continue
      t0 = tn
      call g (neq, t0, y, ngc, g0)
      nge = 1
      zroot = .false.
      do 110 i = 1,ngc
 110    if (dabs(g0(i)) .le. 0.0d0) zroot = .true.
      if (.not. zroot) go to 190
c g has a zero at t.  look at g at t + (small increment). --------------
      temp1 = dsign(hming,h)
      t0 = t0 + temp1
      temp2 = temp1/h
      do 120 i = 1,n
 120    y(i) = y(i) + temp2*yh(i,2)
      call g (neq, t0, y, ngc, g0)
      nge = nge + 1
      zroot = .false.
      do 130 i = 1,ngc
 130    if (dabs(g0(i)) .le. 0.0d0) zroot = .true.
      if (.not. zroot) go to 190
c g has a zero at t and also close to t.  take error return. -----------
      irt = -1
      return
c
 190  continue
      return
c
c
 200  continue
      if (irfnd .eq. 0) go to 260
c if a root was found on the previous step, evaluate g0 = g(t0). -------
      call intdy (t0, 0, yh, nyh, y, iflag)
      call g (neq, t0, y, ngc, g0)
      nge = nge + 1
      zroot = .false.
      do 210 i = 1,ngc
 210    if (dabs(g0(i)) .le. 0.0d0) zroot = .true.
      if (.not. zroot) go to 260
c g has a zero at t0.  look at g at t + (small increment). -------------
      temp1 = dsign(hming,h)
      t0 = t0 + temp1
      if ((t0 - tn)*h .lt. 0.0d0) go to 230
      temp2 = temp1/h
      do 220 i = 1,n
 220    y(i) = y(i) + temp2*yh(i,2)
      go to 240
 230  call intdy (t0, 0, yh, nyh, y, iflag)
 240  call g (neq, t0, y, ngc, g0)
      nge = nge + 1
      zroot = .false.
      do 250 i = 1,ngc
        if (dabs(g0(i)) .gt. 0.0d0) go to 250
        jroot(i) = 1
        zroot = .true.
 250    continue
      if (.not. zroot) go to 260
c g has a zero at t0 and also close to t0.  return root. ---------------
      irt = 1
      return
c     here, g0 does not have a root
c g0 has no zero components.  proceed to check relevant interval. ------
 260  if (tn .eq. tlast) go to 390
c
 300  continue
c set t1 to tn or toutc, whichever comes first, and get g at t1. -------
      if (itaskc.eq.2 .or. itaskc.eq.3 .or. itaskc.eq.5) go to 310
      if ((toutc - tn)*h .ge. 0.0d0) go to 310
      t1 = toutc
      if ((t1 - t0)*h .le. 0.0d0) go to 390
      call intdy (t1, 0, yh, nyh, y, iflag)
      go to 330
 310  t1 = tn
      do 320 i = 1,n
 320    y(i) = yh(i,1)
 330  call g (neq, t1, y, ngc, g1)
      nge = nge + 1
c call roots to search for root in interval from t0 to t1. -------------
      jflag = 0
 350  continue
      call roots (ngc, hming, jflag, t0, t1, g0, g1, gx, x, jroot)
      if (jflag .gt. 1) go to 360
      call intdy (x, 0, yh, nyh, y, iflag)
      call g (neq, x, y, ngc, gx)
      nge = nge + 1
      go to 350
 360  t0 = x
      call mondcopy (ngc, gx, 1, g0, 1)
      if (jflag .eq. 4) go to 390
c found a root.  interpolate to x and return. --------------------------
      call intdy (x, 0, yh, nyh, y, iflag)
      irt = 1
      return
c
 390  continue
      return
c----------------------- end of subroutine rchek -----------------------
      end
      subroutine roots (ng, hmin, jflag, x0, x1, g0, g1, gx, x, jroot)
clll. optimize
      integer ng, jflag, jroot
      double precision hmin, x0, x1, g0, g1, gx, x
      dimension g0(ng), g1(ng), gx(ng), jroot(ng)
      integer iownd3, imax, last, idum3
      double precision alpha, x2, rdum3
      common /lsr001/ alpha, x2, rdum3(3),
     1   iownd3(3), imax, last, idum3(4)
c-----------------------------------------------------------------------
c this subroutine finds the leftmost root of a set of arbitrary
c functions gi(x) (i = 1,...,ng) in an interval (x0,x1).  only roots
c of odd multiplicity (i.e. changes of sign of the gi) are found.
c here the sign of x1 - x0 is arbitrary, but is constant for a given
c problem, and -leftmost- means nearest to x0.
c the values of the vector-valued function g(x) = (gi, i=1...ng)
c are communicated through the call sequence of roots.
c the method used is the illinois algorithm.
c
c reference..
c kathie l. hiebert and lawrence f. shampine, implicitly defined
c output points for solutions of ode-s, sandia report sand80-0180,
c february, 1980.
c
c description of parameters.
c
c ng     = number of functions gi, or the number of components of
c          the vector valued function g(x).  input only.
c
c hmin   = resolution parameter in x.  input only.  when a root is
c          found, it is located only to within an error of hmin in x.
c          typically, hmin should be set to something on the order of
c               100 * uround * max(abs(x0),abs(x1)),
c          where uround is the unit roundoff of the machine.
c
c jflag  = integer flag for input and output communication.
c
c          on input, set jflag = 0 on the first call for the problem,
c          and leave it unchanged until the problem is completed.
c          (the problem is completed when jflag .ge. 2 on return.)
c
c          on output, jflag has the following values and meanings..
c          jflag = 1 means roots needs a value of g(x).  set gx = g(x)
c                    and call roots again.
c          jflag = 2 means a root has been found.  the root is
c                    at x, and gx contains g(x).  (actually, x is the
c                    rightmost approximation to the root on an interval
c                    (x0,x1) of size hmin or less.)
c          jflag = 3 means x = x1 is a root, with one or more of the gi
c                    being zero at x1 and no sign changes in (x0,x1).
c                    gx contains g(x) on output.
c          jflag = 4 means no roots (of odd multiplicity) were
c                    found in (x0,x1) (no sign changes).
c
c x0,x1  = endpoints of the interval where roots are sought.
c          x1 and x0 are input when jflag = 0 (first call), and
c          must be left unchanged between calls until the problem is
c          completed.  x0 and x1 must be distinct, but x1 - x0 may be
c          of either sign.  however, the notion of -left- and -right-
c          will be used to mean nearer to x0 or x1, respectively.
c          when jflag .ge. 2 on return, x0 and x1 are output, and
c          are the endpoints of the relevant interval.
c
c g0,g1  = arrays of length ng containing the vectors g(x0) and g(x1),
c          respectively.  when jflag = 0, g0 and g1 are input and
c          none of the g0(i) should be be zero.
c          when jflag .ge. 2 on return, g0 and g1 are output.
c
c gx     = array of length ng containing g(x).  gx is input
c          when jflag = 1, and output when jflag .ge. 2.
c
c x      = independent variable value.  output only.
c          when jflag = 1 on output, x is the point at which g(x)
c          is to be evaluated and loaded into gx.
c          when jflag = 2 or 3, x is the root.
c          when jflag = 4, x is the right endpoint of the interval, x1.
c
c jroot  = integer array of length ng.  output only.
c          when jflag = 2 or 3, jroot indicates which components
c          of g(x) have a root at x.  jroot(i) is 1 if the i-th
c          component has a root, and jroot(i) = 0 otherwise.
c
c note.. this routine uses the common block /lsr001/ to save
c the values of certain variables between calls (own variables).
c-----------------------------------------------------------------------
      integer i, imxold, nxlast
      double precision t2, tmax, zero
      logical zroot, sgnchg, xroot
      data zero/0.0d0/
c
      if (jflag .eq. 1) go to 200
c jflag .ne. 1.  check for change in sign of g or zero at x1. ----------
      imax = 0
      tmax = zero
      zroot = .false.
      do 120 i = 1,ng
        if (dabs(g1(i)) .gt. zero) go to 110
        zroot = .true.
        go to 120
c at this point, g0(i) has been checked and cannot be zero. ------------
 110    if (dsign(1.0d0,g0(i)) .eq. dsign(1.0d0,g1(i))) go to 120
          t2 = dabs(g1(i)/(g1(i)-g0(i)))
          if (t2 .le. tmax) go to 120
            tmax = t2
            imax = i
 120    continue
      if (imax .gt. 0) go to 130
      sgnchg = .false.
      go to 140
 130  sgnchg = .true.
 140  if (.not. sgnchg) go to 400
c there is a sign change.  find the first root in the interval. --------
      xroot = .false.
      nxlast = 0
      last = 1
c
c repeat until the first root in the interval is found.  loop point. ---
 150  continue
      if (xroot) go to 300
      if (nxlast .eq. last) go to 160
      alpha = 1.0d0
      go to 180
 160  if (last .eq. 0) go to 170
      alpha = 0.5d0*alpha
      go to 180
 170  alpha = 2.0d0*alpha
 180  x2 = x1 - (x1-x0)*g1(imax)/(g1(imax) - alpha*g0(imax))
      if ((dabs(x2-x0) .lt. hmin) .and.
     1   (dabs(x1-x0) .gt. 10.0d0*hmin)) x2 = x0 + 0.1d0*(x1-x0)
      jflag = 1
      x = x2
c return to the calling routine to get a value of gx = g(x). -----------
      return
c check to see in which interval g changes sign. -----------------------
 200  imxold = imax
      imax = 0
      tmax = zero
      zroot = .false.
      do 220 i = 1,ng
        if (dabs(gx(i)) .gt. zero) go to 210
        zroot = .true.
        go to 220
c neither g0(i) nor gx(i) can be zero at this point. -------------------
 210    if (dsign(1.0d0,g0(i)) .eq. dsign(1.0d0,gx(i))) go to 220
          t2 = dabs(gx(i)/(gx(i) - g0(i)))
          if (t2 .le. tmax) go to 220
            tmax = t2
            imax = i
 220    continue
      if (imax .gt. 0) go to 230
      sgnchg = .false.
      imax = imxold
      go to 240
 230  sgnchg = .true.
 240  nxlast = last
      if (.not. sgnchg) go to 250
c sign change between x0 and x2, so replace x1 with x2. ----------------
      x1 = x2
      call mondcopy (ng, gx, 1, g1, 1)
      last = 1
      xroot = .false.
      go to 270
 250  if (.not. zroot) go to 260
c zero value at x2 and no sign change in (x0,x2), so x2 is a root. -----
      x1 = x2
      call mondcopy (ng, gx, 1, g1, 1)
      xroot = .true.
      go to 270
c no sign change between x0 and x2.  replace x0 with x2. ---------------
 260  continue
      call mondcopy (ng, gx, 1, g0, 1)
      x0 = x2
      last = 0
      xroot = .false.
 270  if (dabs(x1-x0) .le. hmin) xroot = .true.
      go to 150
c
c return with x1 as the root.  set jroot.  set x = x1 and gx = g1. -----
 300  jflag = 2
      x = x1
      call mondcopy (ng, g1, 1, gx, 1)
      do 320 i = 1,ng
        jroot(i) = 0
        if (dabs(g1(i)) .gt. zero) go to 310
          jroot(i) = 1
          go to 320
 310    if (dsign(1.0d0,g0(i)) .ne. dsign(1.0d0,g1(i))) jroot(i) = 1
 320    continue
      return
c
c no sign change in the interval.  check for zero at right endpoint. ---
 400  if (.not. zroot) go to 420
c
c zero value at x1 and no sign change in (x0,x1).  return jflag = 3. ---
      x = x1
      call mondcopy (ng, g1, 1, gx, 1)
      do 410 i = 1,ng
        jroot(i) = 0
        if (dabs(g1(i)) .le. zero) jroot (i) = 1
 410  continue
      jflag = 3
      return
c
c no sign changes in this interval.  set x = x1, return jflag = 4. -----
 420  call mondcopy (ng, g1, 1, gx, 1)
      x = x1
      jflag = 4
      return
c----------------------- end of subroutine roots -----------------------
      end
      subroutine srcar (rsav, isav, job)
c-----------------------------------------------------------------------
c this routine saves or restores (depending on job) the contents of
c the common blocks ls0001, lsa001, lar001, and eh0001, which are used
c internally by one or more odepack solvers.
c
c rsav = real array of length 245 or more.
c isav = integer array of length 59 or more.
c job  = flag indicating to save or restore the common blocks..
c        job  = 1 if common is to be saved (written to rsav/isav)
c        job  = 2 if common is to be restored (read from rsav/isav)
c        a call with job = 2 presumes a prior call with job = 1.
c-----------------------------------------------------------------------
      integer isav, job
      integer ieh, ils, ilsa, ilsr
      integer i, ioff, lenrls, lenils, lenrla, lenila, lenrlr, lenilr
      double precision rsav
      double precision rls, rlsa, rlsr
      dimension rsav(1), isav(1)
      common /ls0001/ rls(218), ils(39)
      common /lsa001/ rlsa(22), ilsa(9)
      common /lsr001/ rlsr(5), ilsr(9)
      common /eh0001/ ieh(2)
      data lenrls/218/, lenils/39/, lenrla/22/, lenila/9/
      data lenrlr/5/, lenilr/9/
c
      if (job .eq. 2) go to 100
      do 10 i = 1,lenrls
 10     rsav(i) = rls(i)
      do 15 i = 1,lenrla
 15     rsav(lenrls+i) = rlsa(i)
      ioff = lenrls + lenrla
      do 20 i = 1,lenrlr
 20     rsav(ioff+i) = rlsr(i)
c
      do 30 i = 1,lenils
 30     isav(i) = ils(i)
      do 35 i = 1,lenila
 35     isav(lenils+i) = ilsa(i)
      ioff = lenils + lenila
      do 40 i = 1,lenilr
 40     isav(ioff+i) = ilsr(i)
c
      ioff = ioff + lenilr
      isav(ioff+1) = ieh(1)
      isav(ioff+2) = ieh(2)
      return
c
 100  continue
      do 110 i = 1,lenrls
 110     rls(i) = rsav(i)
      do 115 i = 1,lenrla
 115     rlsa(i) = rsav(lenrls+i)
      ioff = lenrls + lenrla
      do 120 i = 1,lenrlr
 120     rlsr(i) = rsav(ioff+i)
c
      do 130 i = 1,lenils
 130     ils(i) = isav(i)
      do 135 i = 1,lenila
 135     ilsa(i) = isav(lenils+i)
      ioff = lenils + lenila
      do 140 i = 1,lenilr
 140     ilsr(i) = isav(ioff+i)
c
      ioff = ioff + lenilr
      ieh(1) = isav(ioff+1)
      ieh(2) = isav(ioff+2)
      return
c----------------------- end of subroutine srcar -----------------------
      end
      subroutine lsodi (res, adda, jac, neq, y, ydoti, t, tout, itol,
     1  rtol, atol, itask, istate, iopt, rwork, lrw, iwork, liw, mf )
      external res, adda, jac
      integer neq, itol, itask, istate, iopt, lrw, iwork, liw, mf
      double precision y, ydoti, t, tout, rtol, atol, rwork
      dimension neq(1), y(1), ydoti(1), rtol(1), atol(1), rwork(lrw),
     1          iwork(liw)
c-----------------------------------------------------------------------
c this is the march 30, 1987 version of lsodi..
c  livermore solver for ordinary differential equations (implicit form).
c this version is in double precision.
c
c lsodi solves the initial value problem for linearly implicit
c systems of first order ode-s,
c     a(t,y) * dy/dt = g(t,y) ,  where a(t,y) is a square matrix,
c or, in component form,
c     ( a   * ( dy / dt ))  + ... +  ( a     * ( dy   / dt ))  =
c        i,1      1                     i,neq      neq
c
c      =   g ( t, y , y ,..., y    )   ( i = 1,...,neq )
c           i      1   2       neq
c
c if a is singular, this is a differential-algebraic system.
c
c lsodi is a variant version of the lsode package.
c-----------------------------------------------------------------------
c reference..
c     alan c. hindmarsh,  odepack, a systematized collection of ode
c     solvers, in scientific computing, r. s. stepleman et al. (eds.),
c     north-holland, amsterdam, 1983, pp. 55-64.
c-----------------------------------------------------------------------
c authors...  jeffrey f. painter  and
c             alan c. hindmarsh
c             computing and mathematics research division, l-316
c             lawrence livermore national laboratory
c             livermore, ca 94550.
c
c-----------------------------------------------------------------------
c summary of usage.
c
c communication between the user and the lsodi package, for normal
c situations, is summarized here.  this summary describes only a subset
c of the full set of options available.  see the full description for
c details, including optional communication, nonstandard options,
c and instructions for special situations.  see also the example
c problem (with program and output) following this summary.
c
c a. first, provide a subroutine of the form..
c               subroutine res (neq, t, y, s, r, ires)
c               dimension y(neq), s(neq), r(neq)
c which computes the residual function
c     r = g(t,y)  -  a(t,y) * s ,
c as a function of t and the vectors y and s.  (s is an internally
c generated approximation to dy/dt.)  the arrays y and s are inputs
c to the res routine and should not be altered.  the residual
c vector is to be stored in the array r.  the argument ires should be
c ignored for casual use of lsodi.  (for uses of ires, see the
c paragraph on res in the full description below.)
c
c b. next, decide whether full or banded form is more economical
c for the storage of matrices.  lsodi must deal internally with the
c matrices a and dr/dy, where r is the residual function defined above.
c lsodi generates a linear combination of these two matrices, and
c this is treated in either full or banded form.
c     the matrix structure is communicated by a method flag mf,
c which is 21 or 22 for the full case, and 24 or 25 in the band case.
c     in the banded case, lsodi requires two half-bandwidth
c parameters ml and mu.  these are, respectively, the widths of the
c lower and upper parts of the band, excluding the main diagonal.
c thus the band consists of the locations (i,j) with
c i-ml .le. j .le. i+mu, and the full bandwidth is ml+mu+1.
c note that the band must accommodate the nonzero elements of
c a(t,y), dg/dy, and d(a*s)/dy (s fixed).  alternatively, one
c can define a band that encloses only the elements that are relatively
c large in magnitude, and gain some economy in storage and possibly
c also efficiency, although the appropriate threshhold for
c retaining matrix elements is highly problem-dependent.
c
c c. you must also provide a subroutine of the form..
c               subroutine adda (neq, t, y, ml, mu, p, nrowp)
c               dimension y(neq), p(nrowp,neq)
c which adds the matrix a = a(t,y) to the contents of the array p.
c t and the y array are input and should not be altered.
c     in the full matrix case, this routine should add elements of
c to p in the usual order.  i.e., add a(i,j) to p(i,j).  (ignore the
c ml and mu arguments in this case.)
c     in the band matrix case, this routine should add element a(i,j)
c to p(i-j+mu+1,j).  i.e., add the diagonal lines of a to the rows of
c p from the top down (the top line of a added to the first row of p).
c
c d. for the sake of efficiency, you are encouraged to supply the
c jacobian matrix dr/dy in closed form, where r = g(t,y) - a(t,y)*s
c (s = a fixed vector) as above.  if dr/dy is being supplied,
c use mf = 21 or 24, and provide a subroutine of the form..
c               subroutine jac (neq, t, y, s, ml, mu, p, nrowp)
c               dimension y(neq), s(neq), p(nrowp,neq)
c which computes dr/dy as a function of t, y, and s.  here t, y, and
c s are inputs, and the routine is to load dr/dy into p as follows..
c     in the full matrix case (mf = 21), load p(i,j) with dr(i)/dy(j),
c the partial derivative of r(i) with respect to y(j).  (ignore the
c ml and mu arguments in this case.)
c     in the band matrix case (mf = 24), load p(i-j+mu+1,j) with
c dr(i)/dy(j), i.e. load the diagonal lines of dr/dy into the rows of
c p from the top down.
c     in either case, only nonzero elements need be loaded, and the
c indexing of p is the same as in the adda routine.
c     note that if a is independent of y (or this dependence
c is weak enough to be ignored) then jac is to compute dg/dy.
c     if it is not feasible to provide a jac routine, use
c mf = 22 or 25, and lsodi will compute an approximate jacobian
c internally by difference quotients.
c
c e. next decide whether or not to provide the initial value of the
c derivative vector dy/dt.  if the initial value of a(t,y) is
c nonsingular (and not too ill-conditioned), you may let lsodi compute
c this vector (istate = 0).  (lsodi will solve the system a*s = g for
c s, with initial values of a and g.)  if a(t,y) is initially
c singular, then the system is a differential-algebraic system, and
c you must make use of the particular form of the system to compute the
c initial values of y and dy/dt.  in that case, use istate = 1 and
c load the initial value of dy/dt into the array ydoti.
c the input array ydoti and the initial y array must be consistent with
c the equations a*dy/dt = g.  this implies that the initial residual
c r = g(t,y) - a(t,y)*ydoti   must be approximately zero.
c
c f. write a main program which calls subroutine lsodi once for
c each point at which answers are desired.  this should also provide
c for possible use of logical unit 6 for output of error messages
c by lsodi.  on the first call to lsodi, supply arguments as follows..
c res    = name of user subroutine for residual function r.
c adda   = name of user subroutine for computing and adding a(t,y).
c jac    = name of user subroutine for jacobian matrix dr/dy
c          (mf = 21 or 24).  if not used, pass a dummy name.
c note.. the names for the res and adda routines and (if used) the
c        jac routine must be declared external in the calling program.
c neq    = number of scalar equations in the system.
c y      = array of initial values, of length neq.
c ydoti  = array of length neq (containing initial dy/dt if istate = 1).
c t      = the initial value of the independent variable.
c tout   = first point where output is desired (.ne. t).
c itol   = 1 or 2 according as atol (below) is a scalar or array.
c rtol   = relative tolerance parameter (scalar).
c atol   = absolute tolerance parameter (scalar or array).
c          the estimated local error in y(i) will be controlled so as
c          to be roughly less (in magnitude) than
c             ewt(i) = rtol*abs(y(i)) + atol     if itol = 1, or
c             ewt(i) = rtol*abs(y(i)) + atol(i)  if itol = 2.
c          thus the local error test passes if, in each component,
c          either the absolute error is less than atol (or atol(i)),
c          or the relative error is less than rtol.
c          use rtol = 0.0 for pure absolute error control, and
c          use atol = 0.0 (or atol(i) = 0.0) for pure relative error
c          control.  caution.. actual (global) errors may exceed these
c          local tolerances, so choose them conservatively.
c itask  = 1 for normal computation of output values of y at t = tout.
c istate = integer flag (input and output).  set istate = 1 if the
c          initial dy/dt is supplied, and 0 otherwise.
c iopt   = 0 to indicate no optional inputs used.
c rwork  = real work array of length at least..
c             22 +  9*neq + neq**2           for mf = 21 or 22,
c             22 + 10*neq + (2*ml + mu)*neq  for mf = 24 or 25.
c lrw    = declared length of rwork (in user-s dimension).
c iwork  = integer work array of length at least 20 + neq.
c          if mf = 24 or 25, input in iwork(1),iwork(2) the lower
c          and upper half-bandwidths ml,mu.
c liw    = declared length of iwork (in user-s dimension).
c mf     = method flag.  standard values are..
c          21 for a user-supplied full jacobian.
c          22 for an internally generated full jacobian.
c          24 for a user-supplied banded jacobian.
c          25 for an internally generated banded jacobian.
c          for other choices of mf, see the paragraph on mf in
c          the full description below.
c note that the main program must declare arrays y, ydoti, rwork, iwork,
c and possibly atol.
c
c g. the output from the first call (or any call) is..
c      y = array of computed values of y(t) vector.
c      t = corresponding value of independent variable (normally tout).
c istate = 2  if lsodi was successful, negative otherwise.
c          -1 means excess work done on this call (check all inputs).
c          -2 means excess accuracy requested (tolerances too small).
c          -3 means illegal input detected (see printed message).
c          -4 means repeated error test failures (check all inputs).
c          -5 means repeated convergence failures (perhaps bad jacobian
c             supplied or wrong choice of tolerances).
c          -6 means error weight became zero during problem. (solution
c             component i vanished, and atol or atol(i) = 0.)
c          -7 cannot occur in casual use.
c          -8 means lsodi was unable to compute the initial dy/dt.
c             in casual use, this means a(t,y) is initially singular.
c             supply ydoti and use istate = 1 on the first call.
c
c  if lsodi returns istate = -1, -4, or -5, then the output of
c  lsodi also includes ydoti = array containing residual vector
c  r = g - a * dy/dt  evaluated at the current t, y, and dy/dt.
c
c h. to continue the integration after a successful return, simply
c reset tout and call lsodi again.  no other parameters need be reset.
c
c
c-----------------------------------------------------------------------
c example problem.
c
c the following is a simple example problem, with the coding
c needed for its solution by lsodi.  the problem is from chemical
c kinetics, and consists of the following three equations..
c     dy1/dt = -.04*y1 + 1.e4*y2*y3
c     dy2/dt = .04*y1 - 1.e4*y2*y3 - 3.e7*y2**2
c       0.   = y1 + y2 + y3 - 1.
c on the interval from t = 0.0 to t = 4.e10, with initial conditions
c y1 = 1.0, y2 = y3 = 0.
c
c the following coding solves this problem with lsodi, using mf = 21
c and printing results at t = .4, 4., ..., 4.e10.  it uses
c itol = 2 and atol much smaller for y2 than y1 or y3 because
c y2 has much smaller values.  dy/dt is supplied in ydoti. we had
c obtained the initial value of dy3/dt by differentiating the
c third equation and evaluating the first two at t=0.
c at the end of the run, statistical quantities of interest are
c printed (see optional outputs in the full description below).
c
c     external resid, aplusp, dgbydy
c     double precision atol, rtol, rwork, t, tout, y, ydoti
c     dimension y(3), ydoti(3), atol(3), rwork(58), iwork(23)
c     neq = 3
c     y(1) = 1.d0
c     y(2) = 0.d0
c     y(3) = 0.d0
c     ydoti(1) = -.04d0
c     ydoti(2) =  .04d0
c     ydoti(3) =  0.d0
c     t = 0.d0
c     tout = .4d0
c     itol = 2
c     rtol = 1.d-4
c     atol(1) = 1.d-6
c     atol(2) = 1.d-10
c     atol(3) = 1.d-6
c     itask = 1
c     istate = 1
c     iopt = 0
c     lrw = 58
c     liw = 23
c     mf = 21
c     do 40  iout = 1,12
c        call lsodi(resid, aplusp, dgbydy, neq, y, ydoti, t, tout, itol,
c    1      rtol, atol, itask, istate, iopt, rwork, lrw, iwork, liw, mf)
c        write (6,20)  t, y(1), y(2), y(3)
c  20    format(7h at t =,e12.4,6h   y =,3e14.6)
c        if (istate .lt. 0 )  go to 80
c  40    tout = tout*10.d0
c     write (6,60)  iwork(11), iwork(12), iwork(13)
c  60 format(/12h no. steps =,i4,11h  no. r-s =,i4,
c    1         11h  no. j-s =,i4)
c     stop
c  80 write (6,90)  istate
c  90 format(///22h error halt.. istate =,i3)
c     stop
c     end
c
c     subroutine resid(neq, t, y, s, r, ires)
c     double precision r, s, t, y
c     dimension y(3), s(3), r(3)
c     r(1) = -.04d0*y(1) + 1.d4*y(2)*y(3) - s(1)
c     r(2) = .04d0*y(1) - 1.d4*y(2)*y(3) - 3.d7*y(2)*y(2) - s(2)
c     r(3) = y(1) + y(2) + y(3) - 1.d0
c     return
c     end
c
c     subroutine aplusp(neq, t, y, ml, mu, p, nrowp)
c     double precision p, t, y
c     dimension y(3), p(nrowp,3)
c     p(1,1) = p(1,1) + 1.d0
c     p(2,2) = p(2,2) + 1.d0
c     return
c     end
c
c     subroutine dgbydy(neq, t, y, s, ml, mu, p, nrowp)
c     double precision s, t, p, y
c     dimension y(3), s(3), p(nrowp,3)
c     p(1,1) = -.04d0
c     p(1,2) = 1.d4*y(3)
c     p(1,3) = 1.d4*y(2)
c     p(2,1) = .04d0
c     p(2,2) = -1.d4*y(3) - 6.d7*y(2)
c     p(2,3) = -1.d4*y(2)
c     p(3,1) = 1.d0
c     p(3,2) = 1.d0
c     p(3,3) = 1.d0
c     return
c     end
c
c the output of this program (on a cdc-7600 in single precision)
c is as follows..
c
c   at t =  4.0000e-01   y =  9.851726e-01  3.386406e-05  1.479357e-02
c   at t =  4.0000e+00   y =  9.055142e-01  2.240418e-05  9.446344e-02
c   at t =  4.0000e+01   y =  7.158050e-01  9.184616e-06  2.841858e-01
c   at t =  4.0000e+02   y =  4.504846e-01  3.222434e-06  5.495122e-01
c   at t =  4.0000e+03   y =  1.831701e-01  8.940379e-07  8.168290e-01
c   at t =  4.0000e+04   y =  3.897016e-02  1.621193e-07  9.610297e-01
c   at t =  4.0000e+05   y =  4.935213e-03  1.983756e-08  9.950648e-01
c   at t =  4.0000e+06   y =  5.159269e-04  2.064759e-09  9.994841e-01
c   at t =  4.0000e+07   y =  5.306413e-05  2.122677e-10  9.999469e-01
c   at t =  4.0000e+08   y =  5.494532e-06  2.197826e-11  9.999945e-01
c   at t =  4.0000e+09   y =  5.129457e-07  2.051784e-12  9.999995e-01
c   at t =  4.0000e+10   y = -7.170472e-08 -2.868188e-13  1.000000e+00
c
c   no. steps = 330  no. r-s = 404  no. j-s =  69
c-----------------------------------------------------------------------
c full description of user interface to lsodi.
c
c the user interface to lsodi consists of the following parts.
c
c i.   the call sequence to subroutine lsodi, which is a driver
c      routine for the solver.  this includes descriptions of both
c      the call sequence arguments and of user-supplied routines.
c      following these descriptions is a description of
c      optional inputs available through the call sequence, and then
c      a description of optional outputs (in the work arrays).
c
c ii.  descriptions of other routines in the lsodi package that may be
c      (optionally) called by the user.  these provide the ability to
c      alter error message handling, save and restore the internal
c      common, and obtain specified derivatives of the solution y(t).
c
c iii. descriptions of common blocks to be declared in overlay
c      or similar environments, or to be saved when doing an interrupt
c      of the problem and continued solution later.
c
c iv.  description of two routines in the lsodi package, either of
c      which the user may replace with his own version, if desired.
c      these relate to the measurement of errors.
c
c-----------------------------------------------------------------------
c part i.  call sequence.
c
c the call sequence parameters used for input only are
c     res, adda, jac, neq, tout, itol, rtol, atol, itask,
c     iopt, lrw, liw, mf,
c and those used for both input and output are
c     y, t, istate, ydoti.
c the work arrays rwork and iwork are also used for conditional and
c optional inputs and optional outputs.  (the term output here refers
c to the return from subroutine lsodi to the user-s calling program.)
c
c the legality of input parameters will be thoroughly checked on the
c initial call for the problem, but not checked thereafter unless a
c change in input parameters is flagged by istate = 3 on input.
c
c the descriptions of the call arguments are as follows.
c
c res    = the name of the user-supplied subroutine which supplies
c          the residual vector for the ode system, defined by
c            r = g(t,y) - a(t,y) * s
c          as a function of the scalar t and the vectors
c          s and y ( s approximates dy/dt ). this
c          subroutine is to have the form
c              subroutine res ( neq, t, y, s, r, ires )
c              dimension y(1), s(1), r(1)
c          where neq, t, y, s, and ires are input, and r and
c          ires are output. y, s, and r are arrays of length neq.
c          in dimension statements such as that above, 1 is a
c          dummy dimension. it can be replaced by any value.
c             on input, ires indicates how lsodi will use the
c          returned array r, as follows..
c             ires = 1  means that lsodi needs the full residual,
c                       r = g - a*s, exactly.
c             ires = -1 means that lsodi is using r only to compute
c                       the jacobian dr/dy by difference quotients.
c          the res routine can ignore ires, or it can omit some terms
c          if ires = -1.  if a does not depend on y, then res can
c          just return r = g when ires = -1.  if g - a*s contains other
c          additive terms that are independent of y, these can also be
c          dropped, if done consistently, when ires = -1.
c             the subroutine should set the flag ires if it
c          encounters a halt condition or illegal input.
c          otherwise, it should not reset ires.  on output,
c             ires = 1 or -1 represents a normal return, and
c          lsodi continues integrating the ode.  leave ires
c          unchanged from its input value.
c             ires = 2 tells lsodi to immediately return control
c          to the calling program, with istate = 3.  this lets
c          the calling program change parameters of the prob-
c          lem if necessary.
c             ires = 3 represents an error condition (for example, an
c          illegal value of y). lsodi tries to integrate the ode without
c          getting ires = 3 from res.  if it cannot, lsodi returns
c          with istate = -7 or -1.
c             on an lsodi return with istate = 3, -1, or -7, the values
c          of t and y returned correspond to the last point reached
c          successfully without getting the flag ires = 2 or 3.
c             the flag values ires = 2 and 3 should not be used to
c          handle switches or root-stop conditions.  this is better
c          done by calling lsodi in a one-step mode and checking the
c          stopping function for a sign change at each step.
c             if quantities computed in the res routine are needed
c          externally to lsodi, an extra call to res should be made
c          for this purpose, for consistent and accurate results.
c          to get the current dy/dt for the s argument, use intdy.
c             res must be declared external in the calling
c          program.  see note below for more about res.
c
c adda   = the name of the user-supplied subroutine which adds
c          the matrix a = a(t,y) to another matrix stored in the same
c          form as a. the storage form is determined by miter (see
c          mf).  this subroutine is to have the form
c               subroutine adda ( neq, t, y, ml, mu, p, nrowp )
c               dimension y(1), p(nrowp,1)
c          where neq, t, y, ml, mu, and nrowp are input and p is
c          output. y is an array of length neq, and the matrix p is
c          stored in an nrowp by neq array.
c             in the full matrix case ( miter =  1 or 2 ) adda should
c          add  a    to p(i,j). ml and mu are ignored.
c                i,j
c             in the band matrix case ( miter = 4 or 5 ) adda should
c          add  a    to  p(i-j+mu+1,j).
c                i,j
c          see jac for details on this band storage form.
c             adda must be declared external in the calling program.
c          see note below for more information about adda.
c
c jac    = the name of the user-supplied subroutine which supplies
c          the jacobian matrix, dr/dy, where r = g-a*s. the form of the
c          jacobian matrix is determined by miter. jac is required
c          if miter = 1 or 4 -- otherwise a dummy name can be
c          passed. this subroutine is to have the form
c               subroutine jac ( neq, t, y, s, ml, mu, p, nrowp )
c               dimension y(1), s(1), p(nrowp,1)
c          where neq, t, y, s, ml, mu, and nrowp are input and p
c          is output. y and s are arrays of length neq, and the
c          matrix p is stored in an nrowp by neq array.
c          p is to be loaded with partial derivatives ( elements
c          of the jacobian matrix ) on output.
c             in the full matrix case ( miter = 1 ), ml and mu
c          are ignored and the jacobian is to be loaded into p
c          by columns- i.e., dr(i)/dy(j) is loaded into p(i,j).
c             in the band matrix case ( miter = 4 ), the ele-
c          ments within the band are to be loaded into p by
c          by columns, with diagonal lines of dr/dy loaded into
c          the rows of p.  thus dr(i)/dy(j) is to be loaded
c          into p(i-j+mu+1,j). the locations in p in the two
c          triangular areas which correspond to nonexistent matrix
c          elements can be ignored or loaded arbitrarily, as they
c          they are overwritten by lsodi. ml and mu are the half-
c          bandwidth parameters ( see iwork ).
c               in either case, p is preset to zero by the solver,
c          so that only the nonzero elements need be loaded by jac.
c          each call to jac is preceded by a call to res with the same
c          arguments neq, t, y, and s.  thus to gain some efficiency,
c          intermediate quantities shared by both calculations may be
c          saved in a user common block by res and not recomputed by jac
c          if desired.  also, jac may alter the y array, if desired.
c               jac need not provide dr/dy exactly.  a crude
c          approximation (possibly with a smaller bandwidth) will do.
c               jac must be declared external in the calling program.
c               see note below for more about jac.
c
c       note on res, adda, and jac--  these
c          subroutines may access user-defined quantities in
c          neq(2),... and/or in y(neq(1)+1),... if neq is an array
c          (dimensioned in the subroutines) and/or y has length
c          exceeding neq(1).  however, these routines should not alter
c          neq(1), y(1),...,y(neq) or any other input variables.
c          see the descriptions of neq and y below.
c
c neq    = the size of the system (number of first order ordinary
c          differential equations or scalar algebraic equations).
c          used only for input.
c          neq may be decreased, but not increased, during the problem.
c          if neq is decreased (with istate = 3 on input), the
c          remaining components of y should be left undisturbed, if
c          these are to be accessed in res, adda, or jac.
c
c          normally, neq is a scalar, and it is generally referred to
c          as a scalar in this user interface description.  however,
c          neq may be an array, with neq(1) set to the system size.
c          (the lsodi package accesses only neq(1).)  in either case,
c          this parameter is passed as the neq argument in all calls
c          to res, adda, and jac.  hence, if it is an array,
c          locations neq(2),... may be used to store other integer data
c          and pass it to  res, adda, or jac.  each such subroutine
c          must include neq in a dimension statement in that case.
c
c y      = a real array for the vector of dependent variables, of
c          length neq or more.  used for both input and output on the
c          first call (istate = 0 or 1), and only for output on other
c          calls.  on the first call, y must contain the vector of
c          initial values.  on output, y contains the computed solution
c          vector, evaluated at t.  if desired, the y array may be used
c          for other purposes between calls to the solver.
c
c          this array is passed as the y argument in all calls to res,
c          adda, and jac.  hence its length may exceed neq,
c          and locations y(neq+1),... may be used to store other real
c          data and pass it to res, adda, or jac.  (the lsodi
c          package accesses only y(1),...,y(neq). )
c
c ydoti  = a real array for the initial value of the vector
c          dy/dt and for work space, of dimension at least neq.
c
c          on input...
c            if istate = 0 then lsodi will compute the initial value
c          of dy/dt, if a is nonsingular.  thus ydoti will
c          serve only as work space and may have any value.
c            if istate = 1 then ydoti must contain the initial value
c          of dy/dt.
c            if istate = 2 or 3 (continuation calls) then ydoti
c          may have any value.
c            n.b.- if the initial value of a is singular, then
c          lsodi cannot compute the initial value of dy/dt, so
c          it must be provided in ydoti, with istate=1.
c
c          on output, when lsodi terminates abnormally with istate =
c          -1, -4, or -5, ydoti will contain the residual
c          r = g(t,y) - a(t,y)*(dy/dt).  if r is large, t is near
c          its initial value, and ydoti is supplied with istate=1,
c          there may have been an incorrect input value of
c          ydoti = dy/dt or the problem ( as given to lsodi )
c          may not have a solution.
c
c          if desired, the ydoti array may be used for other
c          purposes between calls to the solver.
c
c t      = the independent variable.  on input, t is used only on the
c          first call, as the initial point of the integration.
c          on output, after each call, t is the value at which a
c          computed solution y is evaluated (usually the same as tout).
c          on an error return, t is the farthest point reached.
c
c tout   = the next value of t at which a computed solution is desired.
c          used only for input.
c
c          when starting the problem (istate = 0 or 1), tout may be
c          equal to t for one call, then should .ne. t for the next
c          call.  for the initial t, an input value of tout .ne. t is
c          used in order to determine the direction of the integration
c          (i.e. the algebraic sign of the step sizes) and the rough
c          scale of the problem.  integration in either direction
c          (forward or backward in t) is permitted.
c
c          if itask = 2 or 5 (one-step modes), tout is ignored after
c          the first call (i.e. the first call with tout .ne. t).
c          otherwise, tout is required on every call.
c
c          if itask = 1, 3, or 4, the values of tout need not be
c          monotone, but a value of tout which backs up is limited
c          to the current internal t interval, whose endpoints are
c          tcur - hu and tcur (see optional outputs, below, for
c          tcur and hu).
c
c itol   = an indicator for the type of error control.  see
c          description below under atol.  used only for input.
c
c rtol   = a relative error tolerance parameter, either a scalar or
c          an array of length neq.  see description below under atol.
c          input only.
c
c atol   = an absolute error tolerance parameter, either a scalar or
c          an array of length neq.  input only.
c
c             the input parameters itol, rtol, and atol determine
c          the error control performed by the solver.  the solver will
c          control the vector e = (e(i)) of estimated local errors
c          in y, according to an inequality of the form
c                      rms-norm of ( e(i)/ewt(i) )   .le.   1,
c          where       ewt(i) = rtol(i)*abs(y(i)) + atol(i),
c          and the rms-norm (root-mean-square norm) here is
c          rms-norm(v) = sqrt(sum v(i)**2 / neq).  here ewt = (ewt(i))
c          is a vector of weights which must always be positive, and
c          the values of rtol and atol should all be non-negative.
c          the following table gives the types (scalar/array) of
c          rtol and atol, and the corresponding form of ewt(i).
c
c             itol    rtol       atol          ewt(i)
c              1     scalar     scalar     rtol*abs(y(i)) + atol
c              2     scalar     array      rtol*abs(y(i)) + atol(i)
c              3     array      scalar     rtol(i)*abs(y(i)) + atol
c              4     array      scalar     rtol(i)*abs(y(i)) + atol(i)
c
c          when either of these parameters is a scalar, it need not
c          be dimensioned in the user-s calling program.
c
c          if none of the above choices (with itol, rtol, and atol
c          fixed throughout the problem) is suitable, more general
c          error controls can be obtained by substituting
c          user-supplied routines for the setting of ewt and/or for
c          the norm calculation.  see part iv below.
c
c          if global errors are to be estimated by making a repeated
c          run on the same problem with smaller tolerances, then all
c          components of rtol and atol (i.e. of ewt) should be scaled
c          down uniformly
c
c itask  = an index specifying the task to be performed.
c          input only.  itask has the following values and meanings.
c          1  means normal computation of output values of y(t) at
c             t = tout (by overshooting and interpolating).
c          2  means take one step only and return.
c          3  means stop at the first internal mesh point at or
c             beyond t = tout and return.
c          4  means normal computation of output values of y(t) at
c             t = tout but without overshooting t = tcrit.
c             tcrit must be input as rwork(1).  tcrit may be equal to
c             or beyond tout, but not behind it in the direction of
c             integration.  this option is useful if the problem
c             has a singularity at or beyond t = tcrit.
c          5  means take one step, without passing tcrit, and return.
c             tcrit must be input as rwork(1).
c
c          note..  if itask = 4 or 5 and the solver reaches tcrit
c          (within roundoff), it will return t = tcrit (exactly) to
c          indicate this (unless itask = 4 and tout comes before tcrit,
c          in which case answers at t = tout are returned first).
c
c istate = an index used for input and output to specify the
c          state of the calculation.
c
c          on input, the values of istate are as follows.
c          0  means this is the first call for the problem, and
c             lsodi is to compute the initial value of dy/dt
c             (while doing other initializations).  see note below.
c          1  means this is the first call for the problem, and
c             the initial value of dy/dt has been supplied in
c             ydoti (lsodi will do other initializations). see note
c             below.
c          2  means this is not the first call, and the calculation
c             is to continue normally, with no change in any input
c             parameters except possibly tout and itask.
c             (if itol, rtol, and/or atol are changed between calls
c             with istate = 2, the new values will be used but not
c             tested for legality.)
c          3  means this is not the first call, and the
c             calculation is to continue normally, but with
c             a change in input parameters other than
c             tout and itask.  changes are allowed in
c             neq, itol, rtol, atol, iopt, lrw, liw, mf, ml, mu,
c             and any of the optional inputs except h0.
c             (see iwork description for ml and mu.)
c          note..  a preliminary call with tout = t is not counted
c          as a first call here, as no initialization or checking of
c          input is done.  (such a call is sometimes useful for the
c          purpose of outputting the initial conditions.)
c          thus the first call for which tout .ne. t requires
c          istate = 0 or 1 on input.
c
c          on output, istate has the following values and meanings.
c           0 or 1  means nothing was done, as tout was equal to t with
c              istate = 0 or 1 on input.  (however, an internal counter
c              was set to detect and prevent repeated calls of this
c              type. )
c           2  means that the integration was performed successfully.
c           3  means that the user-supplied subroutine res signalled
c              lsodi to halt the integration and return (ires=2).
c              integration as far as t was achieved with no occurrence
c              of ires=2, but this flag was set on attempting the next
c              step.
c          -1  means an excessive amount of work (more than mxstep
c              steps) was done on this call, before completing the
c              requested task, but the integration was otherwise
c              successful as far as t.  (mxstep is an optional input
c              and is normally 500.)  to continue, the user may
c              simply reset istate to a value .gt. 1 and call again
c              (the excess work step counter will be reset to 0).
c              in addition, the user may increase mxstep to avoid
c              this error return (see below on optional inputs).
c          -2  means too much accuracy was requested for the precision
c              of the machine being used.  this was detected before
c              completing the requested task, but the integration
c              was successful as far as t.  to continue, the tolerance
c              parameters must be reset, and istate must be set
c              to 3.  the optional output tolsf may be used for this
c              purpose.  (note.. if this condition is detected before
c              taking any steps, then an illegal input return
c              (istate = -3) occurs instead.)
c          -3  means illegal input was detected, before taking any
c              integration steps.  see written message for details.
c              note..  if the solver detects an infinite loop of calls
c              to the solver with illegal input, it will cause
c              the run to stop.
c          -4  means there were repeated error test failures on
c              one attempted step, before completing the requested
c              task, but the integration was successful as far as t.
c              the problem may have a singularity, or the input
c              may be inappropriate.
c          -5  means there were repeated convergence test failures on
c              one attempted step, before completing the requested
c              task, but the integration was successful as far as t.
c              this may be caused by an inaccurate jacobian matrix.
c          -6  means ewt(i) became zero for some i during the
c              integration.  pure relative error control (atol(i)=0.0)
c              was requested on a variable which has now vanished.
c              the integration was successful as far as t.
c          -7  means that the user-supplied subroutine res set
c              its error flag (ires = 3) despite repeated tries by
c              lsodi to avoid that condition.
c          -8  means that istate was 0 on input but lsodi was unable
c              to compute the initial value of dy/dt.  see the
c              printed message for details.
c
c          note..  since the normal output value of istate is 2,
c          it does not need to be reset for normal continuation.
c          similarly, istate need not be reset if res told lsodi
c          to return because the calling program must change
c          the parameters of the problem.
c          also, since a negative input value of istate will be
c          regarded as illegal, a negative output value requires the
c          user to change it, and possibly other inputs, before
c          calling the solver again.
c
c iopt   = an integer flag to specify whether or not any optional
c          inputs are being used on this call.  input only.
c          the optional inputs are listed separately below.
c          iopt = 0 means no optional inputs are being used.
c                   default values will be used in all cases.
c          iopt = 1 means one or more optional inputs are being used.
c
c rwork  = a real working array (double precision).
c          the length of rwork must be at least
c             20 + nyh*(maxord + 1) + 3*neq + lenwm    where
c          nyh    = the initial value of neq,
c          maxord = 12 (if meth = 1) or 5 (if meth = 2) (unless a
c                   smaller value is given as an optional input),
c          lenwm   = neq**2 + 2    if miter is 1 or 2, and
c          lenwm   = (2*ml+mu+1)*neq + 2 if miter is 4 or 5.
c          (see mf description for the definition of meth and miter.)
c          thus if maxord has its default value and neq is constant,
c          this length is
c             22 + 16*neq + neq**2         for mf = 11 or 12,
c             22 + 17*neq + (2*ml+mu)*neq  for mf = 14 or 15,
c             22 +  9*neq + neq**2         for mf = 21 or 22,
c             22 + 10*neq + (2*ml+mu)*neq  for mf = 24 or 25.
c          the first 20 words of rwork are reserved for conditional
c          and optional inputs and optional outputs.
c
c          the following word in rwork is a conditional input..
c            rwork(1) = tcrit = critical value of t which the solver
c                       is not to overshoot.  required if itask is
c                       4 or 5, and ignored otherwise.  (see itask.)
c
c lrw    = the length of the array rwork, as declared by the user.
c          (this will be checked by the solver.)
c
c iwork  = an integer work array.  the length of iwork must be at least
c          20 + neq .  the first few words of iwork are used for
c          conditional and optional inputs and optional outputs.
c
c          the following 2 words in iwork are conditional inputs..
c            iwork(1) = ml     these are the lower and upper
c            iwork(2) = mu     half-bandwidths, respectively, of the
c                       matrices in the problem-- the jacobian dr/dy
c                       and the left-hand side matrix a. these half-
c                       bandwidths exclude the main diagonal, so
c                       the total bandwidth is ml + mu + 1 .
c                       the band is defined by the matrix locations
c                       (i,j) with i-ml .le. j .le. i+mu.  ml and mu
c                       must satisfy  0 .le.  ml,mu  .le. neq-1.
c                       these are required if miter is 4 or 5, and
c                       ignored otherwise.
c                       ml and mu may in fact be the band parameters for
c                       matrices to which dr/dy and a are only
c                       approximately equal.
c
c liw    = the length of the array iwork, as declared by the user.
c          (this will be checked by the solver.)
c
c note..  the work arrays must not be altered between calls to lsodi
c for the same problem, except possibly for the conditional and
c optional inputs, and except for the last 3*neq words of rwork.
c the latter space is used for internal scratch space, and so is
c available for use by the user outside lsodi between calls, if
c desired (but not for use by res, adda, or jac).
c
c mf     = the method flag.  used only for input.  the legal values of
c          mf are 11, 12, 14, 15, 21, 22, 24, and 25.
c          mf has decimal digits meth and miter.. mf = 10*meth + miter.
c            meth indicates the basic linear multistep method..
c              meth = 1 means the implicit adams method.
c              meth = 2 means the method based on backward
c                       differentiation formulas (bdf-s).
c                the bdf method is strongly preferred for stiff prob-
c              lems, while the adams method is preferred when the prob-
c              lem is not stiff. if the matrix a(t,y) is nonsingular,
c              stiffness here can be taken to mean that of the explicit
c              ode system dy/dt = a**(-1) * g.  if a is singular, the
c              concept of stiffness is not well defined.
c                if you do not know whether the problem is stiff, we
c              recommend using meth = 2.  if it is stiff, the advan-
c              tage of meth = 2 over 1 will be great, while if it is
c              not stiff, the advantage of meth = 1 will be slight.
c              if maximum efficiency is important, some experimentation
c              with meth may be necessary.
c            miter indicates the corrector iteration method..
c              miter = 1 means chord iteration with a user-supplied
c                        full (neq by neq) jacobian.
c              miter = 2 means chord iteration with an internally
c                        generated (difference quotient) full jacobian.
c                        this uses neq+1 extra calls to res per dr/dy
c                        evaluation.
c              miter = 4 means chord iteration with a user-supplied
c                        banded jacobian.
c              miter = 5 means chord iteration with an internally
c                        generated banded jacobian (using ml+mu+2
c                        extra calls to res per dr/dy evaluation).
c              if miter = 1 or 4, the user must supply a subroutine jac
c              (the name is arbitrary) as described above under jac.
c              for other values of miter, a dummy argument can be used.
c-----------------------------------------------------------------------
c optional inputs.
c
c the following is a list of the optional inputs provided for in the
c call sequence.  (see also part ii.)  for each such input variable,
c this table lists its name as used in this documentation, its
c location in the call sequence, its meaning, and the default value.
c the use of any of these inputs requires iopt = 1, and in that
c case all of these inputs are examined.  a value of zero for any
c of these optional inputs will cause the default value to be used.
c thus to use a subset of the optional inputs, simply preload
c locations 5 to 10 in rwork and iwork to 0.0 and 0 respectively, and
c then set those of interest to nonzero values.
c
c name    location      meaning and default value
c
c h0      rwork(5)  the step size to be attempted on the first step.
c                   the default value is determined by the solver.
c
c hmax    rwork(6)  the maximum absolute step size allowed.
c                   the default value is infinite.
c
c hmin    rwork(7)  the minimum absolute step size allowed.
c                   the default value is 0.  (this lower bound is not
c                   enforced on the final step before reaching tcrit
c                   when itask = 4 or 5.)
c
c maxord  iwork(5)  the maximum order to be allowed.  the default
c                   value is 12 if meth = 1, and 5 if meth = 2.
c                   if maxord exceeds the default value, it will
c                   be reduced to the default value.
c                   if maxord is changed during the problem, it may
c                   cause the current order to be reduced.
c
c mxstep  iwork(6)  maximum number of (internally defined) steps
c                   allowed during one call to the solver.
c                   the default value is 500.
c
c mxhnil  iwork(7)  maximum number of messages printed (per problem)
c                   warning that t + h = t on a step (h = step size).
c                   this must be positive to result in a non-default
c                   value.  the default value is 10.
c-----------------------------------------------------------------------
c optional outputs.
c
c as optional additional output from lsodi, the variables listed
c below are quantities related to the performance of lsodi
c which are available to the user.  these are communicated by way of
c the work arrays, but also have internal mnemonic names as shown.
c except where stated otherwise, all of these outputs are defined
c on any successful return from lsodi, and on any return with
c istate = -1, -2, -4, -5, -6, or -7. on a return with -3 (illegal
c input) or -8, they will be unchanged from their existing values
c (if any), except possibly for tolsf, lenrw, and leniw.
c on any error return, outputs relevant to the error will be defined,
c as noted below.
c
c name    location      meaning
c
c hu      rwork(11) the step size in t last used (successfully).
c
c hcur    rwork(12) the step size to be attempted on the next step.
c
c tcur    rwork(13) the current value of the independent variable
c                   which the solver has actually reached, i.e. the
c                   current internal mesh point in t.  on output, tcur
c                   will always be at least as far as the argument
c                   t, but may be farther (if interpolation was done).
c
c tolsf   rwork(14) a tolerance scale factor, greater than 1.0,
c                   computed when a request for too much accuracy was
c                   detected (istate = -3 if detected at the start of
c                   the problem, istate = -2 otherwise).  if itol is
c                   left unaltered but rtol and atol are uniformly
c                   scaled up by a factor of tolsf for the next call,
c                   then the solver is deemed likely to succeed.
c                   (the user may also ignore tolsf and alter the
c                   tolerance parameters in any other way appropriate.)
c
c nst     iwork(11) the number of steps taken for the problem so far.
c
c nre     iwork(12) the number of residual evaluations (res calls)
c                   for the problem so far.
c
c nje     iwork(13) the number of jacobian evaluations (each involving
c                   an evaluation of a and dr/dy) for the problem so
c                   far.  this equals the number of calls to adda and
c                   (if miter = 1 or 4) jac, and the number of matrix
c                   l-u decompositions.
c
c nqu     iwork(14) the method order last used (successfully).
c
c nqcur   iwork(15) the order to be attempted on the next step.
c
c imxer   iwork(16) the index of the component of largest magnitude in
c                   the weighted local error vector ( e(i)/ewt(i) ),
c                   on an error return with istate = -4 or -5.
c
c lenrw   iwork(17) the length of rwork actually required.
c                   this is defined on normal returns and on an illegal
c                   input return for insufficient storage.
c
c leniw   iwork(18) the length of iwork actually required.
c                   this is defined on normal returns and on an illegal
c                   input return for insufficient storage.
c
c
c the following two arrays are segments of the rwork array which
c may also be of interest to the user as optional outputs.
c for each array, the table below gives its internal name,
c its base address in rwork, and its description.
c
c name    base address      description
c
c yh      21             the nordsieck history array, of size nyh by
c                        (nqcur + 1), where nyh is the initial value
c                        of neq.  for j = 0,1,...,nqcur, column j+1
c                        of yh contains hcur**j/factorial(j) times
c                        the j-th derivative of the interpolating
c                        polynomial currently representing the solution,
c                        evaluated at t = tcur.
c
c acor     lenrw-neq+1   array of size neq used for the accumulated
c                        corrections on each step, scaled on output to
c                        represent the estimated local error in y on the
c                        last step. this is the vector e in the descrip-
c                        tion of the error control.  it is defined only
c                        on a return from lsodi with istate = 2.
c
c-----------------------------------------------------------------------
c part ii.  other routines callable.
c
c the following are optional calls which the user may make to
c gain additional capabilities in conjunction with lsodi.
c (the routines xsetun and xsetf are designed to conform to the
c slatec error handling package.)
c
c     form of call                  function
c   call xsetun(lun)          set the logical unit number, lun, for
c                             output of messages from lsodi, if
c                             the default is not desired.
c                             the default value of lun is 6.
c
c   call xsetf(mflag)         set a flag to control the printing of
c                             messages by lsodi.
c                             mflag = 0 means do not print. (danger..
c                             this risks losing valuable information.)
c                             mflag = 1 means print (the default).
c
c                             either of the above calls may be made at
c                             any time and will take effect immediately.
c
c   call srcom(rsav,isav,job) saves and restores the contents of
c                             the internal common blocks used by
c                             lsodi (see part iii below).
c                             rsav must be a real array of length 218
c                             or more, and isav must be an integer
c                             array of length 41 or more.
c                             job=1 means save common into rsav/isav.
c                             job=2 means restore common from rsav/isav.
c                                srcom is useful if one is
c                             interrupting a run and restarting
c                             later, or alternating between two or
c                             more problems solved with lsodi.
c
c   call intdy(,,,,,)         provide derivatives of y, of various
c        (see below)          orders, at a specified point t, if
c                             desired.  it may be called only after
c                             a successful return from lsodi.
c
c the detailed instructions for using intdy are as follows.
c the form of the call is..
c
c   call intdy (t, k, rwork(21), nyh, dky, iflag)
c
c the input parameters are..
c
c t         = value of independent variable where answers are desired
c             (normally the same as the t last returned by lsodi).
c             for valid results, t must lie between tcur - hu and tcur.
c             (see optional outputs for tcur and hu.)
c k         = integer order of the derivative desired.  k must satisfy
c             0 .le. k .le. nqcur, where nqcur is the current order
c             (see optional outputs).  the capability corresponding
c             to k = 0, i.e. computing y(t), is already provided
c             by lsodi directly.  since nqcur .ge. 1, the first
c             derivative dy/dt is always available with intdy.
c rwork(21) = the base address of the history array yh.
c nyh       = column length of yh, equal to the initial value of neq.
c
c the output parameters are..
c
c dky       = a real array of length neq containing the computed value
c             of the k-th derivative of y(t).
c iflag     = integer flag, returned as 0 if k and t were legal,
c             -1 if k was illegal, and -2 if t was illegal.
c             on an error return, a message is also written.
c-----------------------------------------------------------------------
c part iii.  common blocks.
c
c if lsodi is to be used in an overlay situation, the user
c must declare, in the primary overlay, the variables in..
c   (1) the call sequence to lsodi,
c   (2) the two internal common blocks
c         /ls0001/  of length  257  (218 double precision words
c                         followed by 39 integer words),
c         /eh0001/  of length  2 (integer words).
c
c if lsodi is used on a system in which the contents of internal
c common blocks are not preserved between calls, the user should
c declare the above two common blocks in his main program to insure
c that their contents are preserved.
c
c if the solution of a given problem by lsodi is to be interrupted
c and then later continued, such as when restarting an interrupted run
c or alternating between two or more problems, the user should save,
c following the return from the last lsodi call prior to the
c interruption, the contents of the call sequence variables and the
c internal common blocks, and later restore these values before the
c next lsodi call for that problem.  to save and restore the common
c blocks, use subroutine srcom (see part ii above).
c
c-----------------------------------------------------------------------
c part iv.  optionally replaceable solver routines.
c
c below are descriptions of two routines in the lsodi package which
c relate to the measurement of errors.  either routine can be
c replaced by a user-supplied version, if desired.  however, since such
c a replacement may have a major impact on performance, it should be
c done only when absolutely necessary, and only with great caution.
c (note.. the means by which the package version of a routine is
c superseded by the user-s version may be system-dependent.)
c
c (a) ewset.
c the following subroutine is called just before each internal
c integration step, and sets the array of error weights, ewt, as
c described under itol/rtol/atol above..
c     subroutine ewset (neq, itol, rtol, atol, ycur, ewt)
c where neq, itol, rtol, and atol are as in the lsodi call sequence,
c ycur contains the current dependent variable vector, and
c ewt is the array of weights set by ewset.
c
c if the user supplies this subroutine, it must return in ewt(i)
c (i = 1,...,neq) a positive quantity suitable for comparing errors
c in y(i) to.  the ewt array returned by ewset is passed to the
c vnorm routine (see below), and also used by lsodi in the computation
c of the optional output imxer, the diagonal jacobian approximation,
c and the increments for difference quotient jacobians.
c
c in the user-supplied version of ewset, it may be desirable to use
c the current values of derivatives of y.  derivatives up to order nq
c are available from the history array yh, described above under
c optional outputs.  in ewset, yh is identical to the ycur array,
c extended to nq + 1 columns with a column length of nyh and scale
c factors of h**j/factorial(j).  on the first call for the problem,
c given by nst = 0, nq is 1 and h is temporarily set to 1.0.
c the quantities nq, nyh, h, and nst can be obtained by including
c in ewset the statements..
c     double precision h, rls
c     common /ls0001/ rls(218),ils(39)
c     nq = ils(35)
c     nyh = ils(14)
c     nst = ils(36)
c     h = rls(212)
c thus, for example, the current value of dy/dt can be obtained as
c ycur(nyh+i)/h  (i=1,...,neq)  (and the division by h is
c unnecessary when nst = 0).
c
c (b) vnorm.
c the following is a real function routine which computes the weighted
c root-mean-square norm of a vector v..
c     d = vnorm (n, v, w)
c where..
c   n = the length of the vector,
c   v = real array of length n containing the vector,
c   w = real array of length n containing weights,
c   d = sqrt( (1/n) * sum(v(i)*w(i))**2 ).
c vnorm is called with n = neq and with w(i) = 1.0/ewt(i), where
c ewt is as set by subroutine ewset.
c
c if the user supplies this function, it should return a non-negative
c value of vnorm suitable for use in the error control in lsodi.
c none of the arguments should be altered by vnorm.
c for example, a user-supplied vnorm routine might..
c   -substitute a max-norm of (v(i)*w(i)) for the rms-norm, or
c   -ignore some components of v in the norm, with the effect of
c    suppressing the error control on those components of y.
c-----------------------------------------------------------------------
c-----------------------------------------------------------------------
c other routines in the lsodi package.
c
c in addition to subroutine lsodi, the lsodi package includes the
c following subroutines and function routines..
c  ainvg    computes the initial value of the vector
c             dy/dt = inverse(a) * g
c  intdy    computes an interpolated value of the y vector at t = tout.
c  stodi    is the core integrator, which does one step of the
c           integration and the associated error control.
c  cfode    sets all method coefficients and test constants.
c  prepji   computes and preprocesses the jacobian matrix
c           and the newton iteration matrix p.
c  solsy    manages solution of linear system in chord iteration.
c  ewset    sets the error weight vector ewt before each step.
c  vnorm    computes the weighted r.m.s. norm of a vector.
c  srcom    is a user-callable routine to save and restore
c           the contents of the internal common blocks.
c  mondgefa and mondgesl   are routines from linpack for solving full
c           systems of linear algebraic equations.
c  dgbfa and dgbsl   are routines from linpack for solving banded
c           linear systems.
c  mondaxpy, mondscal, monidamax, and monddot   are basic linear algebra modules
c           (blas) used by the above linpack routines.
c  mond1mach   computes the unit roundoff in a machine-independent manner.
c  monwv, xsetun, and xsetf   handle the printing of all error
c           messages and warnings.  monwv is machine-dependent.
c note..  vnorm, monidamax, monddot, and mond1mach are function routines.
c all the others are subroutines.
c
c the intrinsic and external routines used by lsodi are..  dabs,
c dmax1, dmin1, dfloat, iabs, max0, min0, mod, dsign, dsqrt, and write.
c
c a block data subprogram is also included with the package,
c for loading some of the variables in internal common.
c
c-----------------------------------------------------------------------
c the following card is for optimized compilation on llnl compilers.
clll. optimize
c-----------------------------------------------------------------------
      external prepji, solsy
      integer illin, init, lyh, lewt, lacor, lsavr, lwm, liwm,
     1   mxstep, mxhnil, nhnil, ntrep, nslast, nyh, iowns
      integer icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     1   maxord, maxcor, msbp, mxncf, n, nq, nst, nre, nje, nqu
      integer i, i1, i2, ier, iflag, imxer, ires, kgo,
     1   leniw, lenrw, lenwm, lp, lyd0, ml, mord, mu, mxhnl0, mxstp0
      double precision rowns,
     1   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround
      double precision atoli, ayi, big, ewti, h0, hmax, hmx, rh, rtoli,
     1   tcrit, tdist, tnext, tol, tolsf, tp, size, sum, w0,
     2   mond1mach, vnorm
      dimension mord(2)
      logical ihit
c-----------------------------------------------------------------------
c the following internal common block contains
c (a) variables which are local to any subroutine but whose values must
c     be preserved between calls to the routine (own variables), and
c (b) variables which are communicated between subroutines.
c common block ls0001 is shared by the lsodi and lsode packages.
c the structure of ls0001 is as follows..  all real variables are
c listed first, followed by all integers.  within each type, the
c variables are grouped with those local to subroutine lsodi first,
c then those local to subroutine stodi, and finally those used
c for communication.  the block is declared in subroutines
c lsodi, intdy, stodi, prepji, and solsy.  groups of variables are
c replaced by dummy arrays in the common declarations in routines
c where those variables are not used.
c-----------------------------------------------------------------------
      common /ls0001/ rowns(209),
     1   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround,
     2   illin, init, lyh, lewt, lacor, lsavr, lwm, liwm,
     3   mxstep, mxhnil, nhnil, ntrep, nslast, nyh, iowns(6),
     4   icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     5   maxord, maxcor, msbp, mxncf, n, nq, nst, nre, nje, nqu
c
      data  mord(1),mord(2)/12,5/, mxstp0/500/, mxhnl0/10/
c-----------------------------------------------------------------------
c block a.
c this code block is executed on every call.
c it tests istate and itask for legality and branches appropriately.
c if istate .gt. 1 but the flag init shows that initialization has
c not yet been done, an error return occurs.
c if istate = 0 or 1 and tout = t, jump to block g and return
c immediately.
c-----------------------------------------------------------------------
      if (istate .lt. 0 .or. istate .gt. 3) go to 601
      if (itask .lt. 1 .or. itask .gt. 5) go to 602
      if (istate .le. 1) go to 10
      if (init .eq. 0) go to 603
      if (istate .eq. 2) go to 200
      go to 20
 10   init = 0
      if (tout .eq. t) go to 430
 20   ntrep = 0
c-----------------------------------------------------------------------
c block b.
c the next code block is executed for the initial call (istate = 0 or 1)
c or for a continuation call with parameter changes (istate = 3).
c it contains checking of all inputs and various initializations.
c
c first check legality of the non-optional inputs neq, itol, iopt,
c mf, ml, and mu.
c-----------------------------------------------------------------------
      if (neq(1) .le. 0) go to 604
      if (istate .le. 1) go to 25
      if (neq(1) .gt. n) go to 605
 25   n = neq(1)
      if (itol .lt. 1 .or. itol .gt. 4) go to 606
      if (iopt .lt. 0 .or. iopt .gt. 1) go to 607
      meth = mf/10
      miter = mf - 10*meth
      if (meth .lt. 1 .or. meth .gt. 2) go to 608
      if (miter .le. 0 .or. miter .gt. 5) go to 608
      if (miter .eq. 3)  go to 608
      if (miter .lt. 3) go to 30
      ml = iwork(1)
      mu = iwork(2)
      if (ml .lt. 0 .or. ml .ge. n) go to 609
      if (mu .lt. 0 .or. mu .ge. n) go to 610
 30   continue
c next process and check the optional inputs. --------------------------
      if (iopt .eq. 1) go to 40
      maxord = mord(meth)
      mxstep = mxstp0
      mxhnil = mxhnl0
      if (istate .le. 1) h0 = 0.0d0
      hmxi = 0.0d0
      hmin = 0.0d0
      go to 60
 40   maxord = iwork(5)
      if (maxord .lt. 0) go to 611
      if (maxord .eq. 0) maxord = 100
      maxord = min0(maxord,mord(meth))
      mxstep = iwork(6)
      if (mxstep .lt. 0) go to 612
      if (mxstep .eq. 0) mxstep = mxstp0
      mxhnil = iwork(7)
      if (mxhnil .lt. 0) go to 613
      if (mxhnil .eq. 0) mxhnil = mxhnl0
      if (istate .gt. 1) go to 50
      h0 = rwork(5)
      if ((tout - t)*h0 .lt. 0.0d0) go to 614
 50   hmax = rwork(6)
      if (hmax .lt. 0.0d0) go to 615
      hmxi = 0.0d0
      if (hmax .gt. 0.0d0) hmxi = 1.0d0/hmax
      hmin = rwork(7)
      if (hmin .lt. 0.0d0) go to 616
c-----------------------------------------------------------------------
c set work array pointers and check lengths lrw and liw.
c pointers to segments of rwork and iwork are named by prefixing l to
c the name of the segment.  e.g., the segment yh starts at rwork(lyh).
c segments of rwork (in order) are denoted yh, wm, ewt, savr, acor.
c-----------------------------------------------------------------------
 60   lyh = 21
      if (istate .le. 1) nyh = n
      lwm = lyh + (maxord + 1)*nyh
      if (miter .le. 2) lenwm = n*n + 2
      if (miter .ge. 4) lenwm = (2*ml + mu + 1)*n + 2
      lewt = lwm + lenwm
      lsavr = lewt + n
      lacor = lsavr + n
      lenrw = lacor + n - 1
      iwork(17) = lenrw
      liwm = 1
      leniw = 20 + n
      iwork(18) = leniw
      if (lenrw .gt. lrw) go to 617
      if (leniw .gt. liw) go to 618
c check rtol and atol for legality. ------------------------------------
      rtoli = rtol(1)
      atoli = atol(1)
      do 70 i = 1,n
        if (itol .ge. 3) rtoli = rtol(i)
        if (itol .eq. 2 .or. itol .eq. 4) atoli = atol(i)
        if (rtoli .lt. 0.0d0) go to 619
        if (atoli .lt. 0.0d0) go to 620
 70     continue
      if (istate .le. 1) go to 100
c if istate = 3, set flag to signal parameter changes to stodi. --------
      jstart = -1
      if (nq .le. maxord) go to 90
c maxord was reduced below nq.  copy yh(*,maxord+2) into ydoti.---------
      do 80 i = 1,n
 80     ydoti(i) = rwork(i+lwm-1)
c reload wm(1) = rwork(lwm), since lwm may have changed. ---------------
 90   rwork(lwm) = dsqrt(uround)
      if (n .eq. nyh) go to 200
c neq was reduced.  zero part of yh to avoid undefined references. -----
      i1 = lyh + l*nyh
      i2 = lyh + (maxord + 1)*nyh - 1
      if (i1 .gt. i2) go to 200
      do 95 i = i1,i2
 95     rwork(i) = 0.0d0
      go to 200
c-----------------------------------------------------------------------
c block c.
c the next block is for the initial call only (istate = 0 or 1).
c it contains all remaining initializations, the call to ainvg
c (if istate = 1), and the calculation of the initial step size.
c the error weights in ewt are inverted after being loaded.
c-----------------------------------------------------------------------
 100  uround = mond1mach(4)
      tn = t
      if (itask .ne. 4 .and. itask .ne. 5) go to 105
      tcrit = rwork(1)
      if ((tcrit - tout)*(tout - t) .lt. 0.0d0) go to 625
      if (h0 .ne. 0.0d0 .and. (t + h0 - tcrit)*h0 .gt. 0.0d0)
     1   h0 = tcrit - t
 105  jstart = 0
      rwork(lwm) = dsqrt(uround)
      nhnil = 0
      nst = 0
      nre = 0
      nje = 0
      nslast = 0
      hu = 0.0d0
      nqu = 0
      ccmax = 0.3d0
      maxcor = 3
      msbp = 20
      mxncf = 10
c compute initial dy/dt, if necessary, and load it and initial y into yh
      lyd0 = lyh + nyh
      lp = lwm + 1
      if (istate .eq. 1) go to 120
c lsodi must compute initial dy/dt (lyd0 points to yh(*,2)). -----------
         call ainvg( res, adda, neq, t, y, rwork(lyd0), miter,
     1               ml, mu, rwork(lp), iwork(21), ier )
         nre = nre + 1
         if (ier)  560,110,565
 110     continue
         do 115 i = 1,n
 115        rwork(i+lyh-1) = y(i)
         go to 130
c initial dy/dt has been supplied. -------------------------------------
 120     do 125 i = 1,n
            rwork(i+lyh-1) = y(i)
 125        rwork(i+lyd0-1) = ydoti(i)
c load and invert the ewt array.  (h is temporarily set to 1.0.) -------
 130  continue
      nq = 1
      h = 1.0d0
      call ewset (n, itol, rtol, atol, rwork(lyh), rwork(lewt))
      do 135 i = 1,n
        if (rwork(i+lewt-1) .le. 0.0d0) go to 621
 135    rwork(i+lewt-1) = 1.0d0/rwork(i+lewt-1)
c-----------------------------------------------------------------------
c the coding below computes the step size, h0, to be attempted on the
c first step, unless the user has supplied a value for this.
c first check that tout - t differs significantly from zero.
c a scalar tolerance quantity tol is computed, as max(rtol(i))
c if this is positive, or max(atol(i)/abs(y(i))) otherwise, adjusted
c so as to be between 100*uround and 1.0e-3.
c then the computed value h0 is given by..
c                                      neq
c   h0**2 = tol / ( w0**-2 + (1/neq) * sum ( ydot(i)/ywt(i) )**2  )
c                                       1
c where   w0      = max ( abs(t), abs(tout) ),
c         ydot(i) = i-th component of initial value of dy/dt,
c         ywt(i)  = ewt(i)/tol  (a weight for y(i)).
c the sign of h0 is inferred from the initial values of tout and t.
c-----------------------------------------------------------------------
      if (h0 .ne. 0.0d0) go to 180
      tdist = dabs(tout - t)
      w0 = dmax1(dabs(t),dabs(tout))
      if (tdist .lt. 2.0d0*uround*w0) go to 622
      tol = rtol(1)
      if (itol .le. 2) go to 145
      do 140 i = 1,n
 140    tol = dmax1(tol,rtol(i))
 145  if (tol .gt. 0.0d0) go to 160
      atoli = atol(1)
      do 150 i = 1,n
        if (itol .eq. 2 .or. itol .eq. 4) atoli = atol(i)
        ayi = dabs(y(i))
        if (ayi .ne. 0.0d0) tol = dmax1(tol,atoli/ayi)
 150    continue
 160  tol = dmax1(tol,100.0d0*uround)
      tol = dmin1(tol,0.001d0)
      sum = vnorm (n, rwork(lyd0), rwork(lewt))
      sum = 1.0d0/(tol*w0*w0) + tol*sum**2
      h0 = 1.0d0/dsqrt(sum)
      h0 = dmin1(h0,tdist)
      h0 = dsign(h0,tout-t)
c adjust h0 if necessary to meet hmax bound. ---------------------------
 180  rh = dabs(h0)*hmxi
      if (rh .gt. 1.0d0) h0 = h0/rh
c load h with h0 and scale yh(*,2) by h0. ------------------------------
      h = h0
      do 190 i = 1,n
 190    rwork(i+lyd0-1) = h0*rwork(i+lyd0-1)
      go to 270
c-----------------------------------------------------------------------
c block d.
c the next code block is for continuation calls only (istate = 2 or 3)
c and is to check stop conditions before taking a step.
c-----------------------------------------------------------------------
 200  nslast = nst
      go to (210, 250, 220, 230, 240), itask
 210  if ((tn - tout)*h .lt. 0.0d0) go to 250
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      if (iflag .ne. 0) go to 627
      t = tout
      go to 420
 220  tp = tn - hu*(1.0d0 + 100.0d0*uround)
      if ((tp - tout)*h .gt. 0.0d0) go to 623
      if ((tn - tout)*h .lt. 0.0d0) go to 250
      go to 400
 230  tcrit = rwork(1)
      if ((tn - tcrit)*h .gt. 0.0d0) go to 624
      if ((tcrit - tout)*h .lt. 0.0d0) go to 625
      if ((tn - tout)*h .lt. 0.0d0) go to 245
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      if (iflag .ne. 0) go to 627
      t = tout
      go to 420
 240  tcrit = rwork(1)
      if ((tn - tcrit)*h .gt. 0.0d0) go to 624
 245  hmx = dabs(tn) + dabs(h)
      ihit = dabs(tn - tcrit) .le. 100.0d0*uround*hmx
      if (ihit) go to 400
      tnext = tn + h*(1.0d0 + 4.0d0*uround)
      if ((tnext - tcrit)*h .le. 0.0d0) go to 250
      h = (tcrit - tn)*(1.0d0 - 4.0d0*uround)
      if (istate .eq. 2) jstart = -2
c-----------------------------------------------------------------------
c block e.
c the next block is normally executed for all calls and contains
c the call to the one-step core integrator stodi.
c
c this is a looping point for the integration steps.
c
c first check for too many steps being taken, update ewt (if not at
c start of problem), check for too much accuracy being requested, and
c check for h below the roundoff level in t.
c-----------------------------------------------------------------------
 250  continue
      if ((nst-nslast) .ge. mxstep) go to 500
      call ewset (n, itol, rtol, atol, rwork(lyh), rwork(lewt))
      do 260 i = 1,n
        if (rwork(i+lewt-1) .le. 0.0d0) go to 510
 260    rwork(i+lewt-1) = 1.0d0/rwork(i+lewt-1)
 270  tolsf = uround*vnorm (n, rwork(lyh), rwork(lewt))
      if (tolsf .le. 1.0d0) go to 280
      tolsf = tolsf*2.0d0
      if (nst .eq. 0) go to 626
      go to 520
 280  if ((tn + h) .ne. tn) go to 290
      nhnil = nhnil + 1
      if (nhnil .gt. mxhnil) go to 290
      call monwv(50hlsodi--  warning..internal t (=r1) and h (=r2) are,
     1   50, 101, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  60h      such that in the machine, t + h = t on the next step  ,
     1   60, 101, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      (h = step size). solver will continue anyway,
     1   50, 101, 0, 0, 0, 0, 2, tn, h)
      if (nhnil .lt. mxhnil) go to 290
      call monwv(50hlsodi--  above warning has been issued i1 times.  ,
     1   50, 102, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      it will not be issued again for this problem,
     1   50, 102, 0, 1, mxhnil, 0, 0, 0.0d0, 0.0d0)
 290  continue
c-----------------------------------------------------------------------
c     call stodi(neq,y,yh,nyh,yh1,ewt,savf,savr,acor,wm,iwm,res,
c                adda,jac,prepji,solsy)
c note... savf in stodi occupies the same space as ydoti in lsodi.
c-----------------------------------------------------------------------
      call stodi (neq, y, rwork(lyh), nyh, rwork(lyh), rwork(lewt),
     1   ydoti, rwork(lsavr), rwork(lacor), rwork(lwm),
     2   iwork(liwm), res, adda, jac, prepji, solsy )
      kgo = 1 - kflag
      go to (300, 530, 540, 400, 550), kgo
c
c kgo = 1,success. 2,error test failure. 3,convergence failure.
c       4,res ordered return. 5,res returned error.
c-----------------------------------------------------------------------
c block f.
c the following block handles the case of a successful return from the
c core integrator (kflag = 0).  test for stop conditions.
c-----------------------------------------------------------------------
 300  init = 1
      go to (310, 400, 330, 340, 350), itask
c itask = 1.  if tout has been reached, interpolate. -------------------
 310  if ((tn - tout)*h .lt. 0.0d0) go to 250
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      t = tout
      go to 420
c itask = 3.  jump to exit if tout was reached. ------------------------
 330  if ((tn - tout)*h .ge. 0.0d0) go to 400
      go to 250
c itask = 4.  see if tout or tcrit was reached.  adjust h if necessary.
 340  if ((tn - tout)*h .lt. 0.0d0) go to 345
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      t = tout
      go to 420
 345  hmx = dabs(tn) + dabs(h)
      ihit = dabs(tn - tcrit) .le. 100.0d0*uround*hmx
      if (ihit) go to 400
      tnext = tn + h*(1.0d0 + 4.0d0*uround)
      if ((tnext - tcrit)*h .le. 0.0d0) go to 250
      h = (tcrit - tn)*(1.0d0 - 4.0d0*uround)
      jstart = -2
      go to 250
c itask = 5.  see if tcrit was reached and jump to exit. ---------------
 350  hmx = dabs(tn) + dabs(h)
      ihit = dabs(tn - tcrit) .le. 100.0d0*uround*hmx
c-----------------------------------------------------------------------
c block g.
c the following block handles all successful returns from lsodi.
c if itask .ne. 1, y is loaded from yh and t is set accordingly.
c istate is set to 2, the illegal input counter is zeroed, and the
c optional outputs are loaded into the work arrays before returning.  if
c istate = 0 or 1 and tout = t, there is a return with no action taken,
c except that if this has happened repeatedly, the run is terminated.
c-----------------------------------------------------------------------
 400  do 410 i = 1,n
 410    y(i) = rwork(i+lyh-1)
      t = tn
      if (itask .ne. 4 .and. itask .ne. 5) go to 420
      if (ihit) t = tcrit
 420  istate = 2
      if (kflag .eq. -3) istate = 3
      illin = 0
      rwork(11) = hu
      rwork(12) = h
      rwork(13) = tn
      iwork(11) = nst
      iwork(12) = nre
      iwork(13) = nje
      iwork(14) = nqu
      iwork(15) = nq
      return
c
 430  ntrep = ntrep + 1
      if (ntrep .lt. 5) return
      call monwv(
     1  60hlsodi--  repeated calls with istate= 0 or 1 and tout= t(=r1),
     1   60, 301, 0, 0, 0, 0, 1, t, 0.0d0)
      go to 800
c-----------------------------------------------------------------------
c block h.
c the following block handles all unsuccessful returns other than
c those for illegal input.  first the error message routine is called.
c if there was an error test or convergence test failure, imxer is set.
c then y is loaded from yh, t is set to tn, and the illegal input
c counter illin is set to 0.  the optional outputs are loaded into
c the work arrays before returning.
c-----------------------------------------------------------------------
c the maximum number of steps was taken before reaching tout. ----------
 500  call monwv(50hlsodi--  at current t (=r1), mxstep (=i1) steps   ,
     1   50, 201, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      taken on this call before reaching tout     ,
     1   50, 201, 0, 1, mxstep, 0, 1, tn, 0.0d0)
      istate = -1
      go to 580
c ewt(i) .le. 0.0 for some i (not at start of problem). ----------------
 510  ewti = rwork(lewt+i-1)
      call monwv(50hlsodi--  at t (=r1), ewt(i1) has become r2 .le. 0.,
     1   50, 202, 0, 1, i, 0, 2, tn, ewti)
      istate = -6
      go to 590
c too much accuracy requested for machine precision. -------------------
 520  call monwv(50hlsodi--  at t (=r1), too much accuracy requested  ,
     1   50, 203, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      for precision of machine..  see tolsf (=r2) ,
     1   50, 203, 0, 0, 0, 0, 2, tn, tolsf)
      rwork(14) = tolsf
      istate = -2
      go to 590
c kflag = -1.  error test failed repeatedly or with abs(h) = hmin. -----
 530  call monwv(50hlsodi--  at t(=r1) and step size h(=r2), the error,
     1   50, 204, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      test failed repeatedly or with abs(h) = hmin,
     1   50, 204, 0, 0, 0, 0, 2, tn, h)
      istate = -4
      go to 570
c kflag = -2.  convergence failed repeatedly or with abs(h) = hmin. ----
 540  call monwv(50hlsodi--  at t (=r1) and step size h (=r2), the    ,
     1   50, 205, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      corrector convergence failed repeatedly     ,
     1   50, 205, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(30h      or with abs(h) = hmin   ,
     1   30, 205, 0, 0, 0, 0, 2, tn, h)
      istate = -5
      go to 570
c ires = 3 returned by res, despite retries by stodi. ------------------
 550  call monwv(50hlsodi--  at t (=r1) residual routine returned     ,
     1   50, 206, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(40h      error ires = 3 repeatedly         ,
     1   40, 206, 0, 0, 0, 0, 1, tn, 0.0d0)
      istate = -7
      go to 590
c ainvg failed because a-matrix was singular. --------------------------
 560  ier = -ier
      call monwv(
     1  60hlsodi-- attempt to initialize dy/dt failed.. matrix a is    ,
     1   60, 207, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      singular.  sgefa or sgbfa returned info=(i1),
     2   50, 207, 0, 1, ier, 0, 0, 0.0d0, 0.0d0)
      istate = -8
      return
c ainvg failed because res set ires to 2 or 3. -------------------------
 565  call monwv(50hlsodi--  attempt to initialize dy/dt failed       ,
     1   50, 208, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      because residual routine set its error flag ,
     1   50, 208, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(20h      to ires = (i1),
     1   20, 208, 0, 1, ier, 0, 0, 0.0d0, 0.0d0)
      istate = -8
      return
c compute imxer if relevant. -------------------------------------------
 570  big = 0.0d0
      imxer = 1
      do 575 i = 1,n
        size = dabs(rwork(i+lacor-1)*rwork(i+lewt-1))
        if (big .ge. size) go to 575
        big = size
        imxer = i
 575    continue
      iwork(16) = imxer
c compute residual if relevant. ----------------------------------------
 580  lyd0 = lyh + nyh
      do 585 i = 1,n
         rwork(i+lsavr-1) = rwork(i+lyd0-1)/h
 585     y(i) = rwork(i+lyh-1)
      ires = 1
      call res ( neq, tn, y, rwork(lsavr), ydoti, ires )
      nre = nre + 1
      if (ires .le. 1) go to 595
      call monwv(50hlsodi--  residual routine set its flag ires       ,
     1   50, 210, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      to (i1) when called for final output.       ,
     1   50, 210, 0, 1, ires, 0, 0, 0.0d0, 0.0d0)
      go to 595
c set y vector, t, illin, and optional outputs. ------------------------
 590  do 592 i = 1,n
 592    y(i) = rwork(i+lyh-1)
 595  t = tn
      illin = 0
      rwork(11) = hu
      rwork(12) = h
      rwork(13) = tn
      iwork(11) = nst
      iwork(12) = nre
      iwork(13) = nje
      iwork(14) = nqu
      iwork(15) = nq
      return
c-----------------------------------------------------------------------
c block i.
c the following block handles all error returns due to illegal input
c (istate = -3), as detected before calling the core integrator.
c first the error message routine is called.  then if there have been
c 5 consecutive such returns just before this call to the solver,
c the run is halted.
c-----------------------------------------------------------------------
 601  call monwv(30hlsodi--  istate (=i1) illegal ,
     1   30, 1, 0, 1, istate, 0, 0, 0.0d0, 0.0d0)
      go to 700
 602  call monwv(30hlsodi--  itask (=i1) illegal  ,
     1   30, 2, 0, 1, itask, 0, 0, 0.0d0, 0.0d0)
      go to 700
 603  call monwv(50hlsodi--  istate .gt. 1 but lsodi not initialized  ,
     1   50, 3, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      go to 700
 604  call monwv(30hlsodi--  neq (=i1) .lt. 1     ,
     1   30, 4, 0, 1, neq(1), 0, 0, 0.0d0, 0.0d0)
      go to 700
 605  call monwv(50hlsodi--  istate = 3 and neq increased (i1 to i2)  ,
     1   50, 5, 0, 2, n, neq(1), 0, 0.0d0, 0.0d0)
      go to 700
 606  call monwv(30hlsodi--  itol (=i1) illegal   ,
     1   30, 6, 0, 1, itol, 0, 0, 0.0d0, 0.0d0)
      go to 700
 607  call monwv(30hlsodi--  iopt (=i1) illegal   ,
     1   30, 7, 0, 1, iopt, 0, 0, 0.0d0, 0.0d0)
      go to 700
 608  call monwv(30hlsodi--  mf (=i1) illegal     ,
     1   30, 8, 0, 1, mf, 0, 0, 0.0d0, 0.0d0)
      go to 700
 609  call monwv(50hlsodi--  ml(=i1) illegal.. .lt. 0 or .ge. neq(=i2),
     1   50, 9, 0, 2, ml, neq(1), 0, 0.0d0, 0.0d0)
      go to 700
 610  call monwv(50hlsodi--  mu(=i1) illegal.. .lt. 0 or .ge. neq(=i2),
     1   50, 10, 0, 2, mu, neq(1), 0, 0.0d0, 0.0d0)
      go to 700
 611  call monwv(30hlsodi--  maxord (=i1) .lt. 0  ,
     1   30, 11, 0, 1, maxord, 0, 0, 0.0d0, 0.0d0)
      go to 700
 612  call monwv(30hlsodi--  mxstep (=i1) .lt. 0  ,
     1   30, 12, 0, 1, mxstep, 0, 0, 0.0d0, 0.0d0)
      go to 700
 613  call monwv(30hlsodi--  mxhnil (=i1) .lt. 0  ,
     1   30, 13, 0, 1, mxhnil, 0, 0, 0.0d0, 0.0d0)
      go to 700
 614  call monwv(40hlsodi--  tout (=r1) behind t (=r2)      ,
     1   40, 14, 0, 0, 0, 0, 2, tout, t)
      call monwv(50h      integration direction is given by h0 (=r1)  ,
     1   50, 14, 0, 0, 0, 0, 1, h0, 0.0d0)
      go to 700
 615  call monwv(30hlsodi--  hmax (=r1) .lt. 0.0  ,
     1   30, 15, 0, 0, 0, 0, 1, hmax, 0.0d0)
      go to 700
 616  call monwv(30hlsodi--  hmin (=r1) .lt. 0.0  ,
     1   30, 16, 0, 0, 0, 0, 1, hmin, 0.0d0)
      go to 700
 617  call monwv(
     1  60hlsodi--  rwork length needed, lenrw (=i1), exceeds lrw (=i2),
     1   60, 17, 0, 2, lenrw, lrw, 0, 0.0d0, 0.0d0)
      go to 700
 618  call monwv(
     1  60hlsodi--  iwork length needed, leniw (=i1), exceeds liw (=i2),
     1   60, 18, 0, 2, leniw, liw, 0, 0.0d0, 0.0d0)
      go to 700
 619  call monwv(40hlsodi--  rtol(=i1) is r1 .lt. 0.0       ,
     1   40, 19, 0, 1, i, 0, 1, rtoli, 0.0d0)
      go to 700
 620  call monwv(40hlsodi--  atol(=i1) is r1 .lt. 0.0       ,
     1   40, 20, 0, 1, i, 0, 1, atoli, 0.0d0)
      go to 700
 621  ewti = rwork(lewt+i-1)
      call monwv(40hlsodi--  ewt(=i1) is r1 .le. 0.0        ,
     1   40, 21, 0, 1, i, 0, 1, ewti, 0.0d0)
      go to 700
 622  call monwv(
     1  60hlsodi--  tout (=r1) too close to t(=r2) to start integration,
     1   60, 22, 0, 0, 0, 0, 2, tout, t)
      go to 700
 623  call monwv(
     1  60hlsodi--  itask = i1 and tout (=r1) behind tcur - hu (= r2)  ,
     1   60, 23, 0, 1, itask, 0, 2, tout, tp)
      go to 700
 624  call monwv(
     1  60hlsodi--  itask = 4 or 5 and tcrit (=r1) behind tcur (=r2)   ,
     1   60, 24, 0, 0, 0, 0, 2, tcrit, tn)
      go to 700
 625  call monwv(
     1  60hlsodi--  itask = 4 or 5 and tcrit (=r1) behind tout (=r2)   ,
     1   60, 25, 0, 0, 0, 0, 2, tcrit, tout)
      go to 700
 626  call monwv(50hlsodi--  at start of problem, too much accuracy   ,
     1   50, 26, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  60h      requested for precision of machine..  see tolsf (=r1) ,
     1   60, 26, 0, 0, 0, 0, 1, tolsf, 0.0d0)
      rwork(14) = tolsf
      go to 700
 627  call monwv(50hlsodi--  trouble from intdy. itask = i1, tout = r1,
     1   50, 27, 0, 1, itask, 0, 1, tout, 0.0d0)
c
 700  if (illin .eq. 5) go to 710
      illin = illin + 1
      istate = -3
      return
 710  call monwv(50hlsodi--  repeated occurrences of illegal input    ,
     1   50, 302, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
c
 800  call monwv(50hlsodi--  run aborted.. apparent infinite loop     ,
     1   50, 303, 2, 0, 0, 0, 0, 0.0d0, 0.0d0)
      return
c----------------------- end of subroutine lsodi -----------------------
      end
      subroutine ainvg (res, adda, neq, t, y, ydot, miter,
     1                   ml, mu, pw, ipvt, ier )
clll. optimize
      external res, adda
      integer neq, miter, ml, mu, ipvt, ier
      integer i, lenpw, mlp1, nrowpw
      double precision t, y, ydot, pw
      dimension y(1), ydot(1), pw(1), ipvt(1)
c-----------------------------------------------------------------------
c this subroutine computes the initial value
c of the vector ydot satisfying
c     a * ydot = g(t,y)
c when a is nonsingular.  it is called by lsodi for
c initialization only, when istate = 0 .
c ainvg returns an error flag ier..
c   ier  =  0  means ainvg was successful.
c   ier .ge. 2 means res returned an error flag ires = ier.
c   ier .lt. 0 means the a-matrix was found to be singular.
c-----------------------------------------------------------------------
c
      if (miter .ge. 4)  go to 100
c
c full matrix case -----------------------------------------------------
c
      lenpw = neq*neq
      do 10  i = 1, lenpw
   10    pw(i) = 0.0d0
c
      ier = 1
      call res ( neq, t, y, pw, ydot, ier )
      if (ier .gt. 1) return
c
      call adda ( neq, t, y, 0, 0, pw, neq )
      call mondgefa ( pw, neq, neq, ipvt, ier )
      if (ier .eq. 0) go to 20
         ier = -ier
         return
   20 call mondgesl ( pw, neq, neq, ipvt, ydot, 0 )
      return
c
c band matrix case -----------------------------------------------------
c
  100 continue
      nrowpw = 2*ml + mu + 1
      lenpw = neq * nrowpw
      do 110  i = 1, lenpw
  110    pw(i) = 0.0d0
c
      ier = 1
      call res ( neq, t, y, pw, ydot, ier )
      if (ier .gt. 1) return
c
      mlp1 = ml + 1
      call adda ( neq, t, y, ml, mu, pw(mlp1), nrowpw )
      call dgbfa ( pw, nrowpw, neq, ml, mu, ipvt, ier )
      if (ier .eq. 0) go to 120
         ier = -ier
         return
  120 call dgbsl ( pw, nrowpw, neq, ml, mu, ipvt, ydot, 0 )
      return
c-------------------- end of subroutine ainvg --------------------------
      end
      subroutine stodi (neq, y, yh, nyh, yh1, ewt, savf, savr,
     1   acor, wm, iwm, res, adda, jac, pjac, slvs )
clll. optimize
      external res, adda, jac, pjac, slvs
      integer neq, nyh, iwm
      integer iownd, ialth, ipup, lmax, meo, nqnyh, nslp,
     1   icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     2   maxord, maxcor, msbp, mxncf, n, nq, nst, nre, nje, nqu
      integer i, i1, iredo, ires, iret, j, jb, kgo, m, ncf, newq
      double precision y, yh, yh1, ewt, savf, savr, acor, wm
      double precision conit, crate, el, elco, hold, rmax, tesco,
     2   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround
      double precision dcon, ddn, del, delp, dsm, dup,
     1   eljh, el1h, exdn, exsm, exup,
     2   r, rh, rhdn, rhsm, rhup, told, vnorm
      dimension neq(1), y(1), yh(nyh,1), yh1(1), ewt(1), savf(1),
     1   savr(1), acor(1), wm(1), iwm(1)
      common /ls0001/ conit, crate, el(13), elco(13,12),
     1   hold, rmax, tesco(3,12),
     2   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround, iownd(14),
     3   ialth, ipup, lmax, meo, nqnyh, nslp,
     4   icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     5   maxord, maxcor, msbp, mxncf, n, nq, nst, nre, nje, nqu
c-----------------------------------------------------------------------
c stodi performs one step of the integration of an initial value
c problem for a system of ordinary differential equations.
c note.. stodi is independent of the value of the iteration method
c indicator miter, and hence is independent
c of the type of chord method used, or the jacobian structure.
c communication with stodi is done with the following variables..
c
c neq    = integer array containing problem size in neq(1), and
c          passed as the neq argument in all calls to res, adda,
c          and jac.
c y      = an array of length .ge. n used as the y argument in
c          all calls to res, jac, and adda.
c neq    = integer array containing problem size in neq(1), and
c          passed as the neq argument in all calls to res, g, adda,
c          and jac
c yh     = an nyh by lmax array containing the dependent variables
c          and their approximate scaled derivatives, where
c          lmax = maxord + 1.  yh(i,j+1) contains the approximate
c          j-th derivative of y(i), scaled by h**j/factorial(j)
c          (j = 0,1,...,nq).  on entry for the first step, the first
c          two columns of yh must be set from the initial values.
c nyh    = a constant integer .ge. n, the first dimension of yh.
c yh1    = a one-dimensional array occupying the same space as yh.
c ewt    = an array of length n containing multiplicative weights
c          for local error measurements.  local errors in y(i) are
c          compared to 1.0/ewt(i) in various error tests.
c savf   = an array of working storage, of length n. also used for
c          input of yh(*,maxord+2) when jstart = -1 and maxord is less
c          than the current order nq.
c          same as ydoti in driver.
c savr   = an array of working storage, of length n.
c acor   = a work array of length n used for the accumulated
c          corrections. on a succesful return, acor(i) contains
c          the estimated one-step local error in y(i).
c wm,iwm = real and integer work arrays associated with matrix
c          operations in chord iteration.
c pjac   = name of routine to evaluate and preprocess jacobian matrix.
c slvs   = name of routine to solve linear system in chord iteration.
c ccmax  = maximum relative change in h*el0 before pjac is called.
c h      = the step size to be attempted on the next step.
c          h is altered by the error control algorithm during the
c          problem.  h can be either positive or negative, but its
c          sign must remain constant throughout the problem.
c hmin   = the minimum absolute value of the step size h to be used.
c hmxi   = inverse of the maximum absolute value of h to be used.
c          hmxi = 0.0 is allowed and corresponds to an infinite hmax.
c          hmin and hmxi may be changed at any time, but will not
c          take effect until the next change of h is considered.
c tn     = the independent variable. tn is updated on each step taken.
c jstart = an integer used for input only, with the following
c          values and meanings..
c               0  perform the first step.
c           .gt.0  take a new step continuing from the last.
c              -1  take the next step with a new value of h, maxord,
c                    n, meth, miter, and/or matrix parameters.
c              -2  take the next step with a new value of h,
c                    but with other inputs unchanged.
c          on return, jstart is set to 1 to facilitate continuation.
c kflag  = a completion code with the following meanings..
c               0  the step was succesful.
c              -1  the requested error could not be achieved.
c              -2  corrector convergence could not be achieved.
c              -3  res ordered immediate return.
c              -4  error condition from res could not be avoided.
c              -5  fatal error in pjac or slvs.
c          a return with kflag = -1, -2, or -4 means either
c          abs(h) = hmin or 10 consecutive failures occurred.
c          on a return with kflag negative, the values of tn and
c          the yh array are as of the beginning of the last
c          step, and h is the last step size attempted.
c maxord = the maximum order of integration method to be allowed.
c maxcor = the maximum number of corrector iterations allowed.
c msbp   = maximum number of steps between pjac calls.
c mxncf  = maximum number of convergence failures allowed.
c meth/miter = the method flags.  see description in driver.
c n      = the number of first-order differential equations.
c-----------------------------------------------------------------------
      kflag = 0
      told = tn
      ncf = 0
      ierpj = 0
      iersl = 0
      jcur = 0
      icf = 0
      delp = 0.0d0
      if (jstart .gt. 0) go to 200
      if (jstart .eq. -1) go to 100
      if (jstart .eq. -2) go to 160
c-----------------------------------------------------------------------
c on the first call, the order is set to 1, and other variables are
c initialized.  rmax is the maximum ratio by which h can be increased
c in a single step.  it is initially 1.e4 to compensate for the small
c initial h, but then is normally equal to 10.  if a failure
c occurs (in corrector convergence or error test), rmax is set at 2
c for the next increase.
c-----------------------------------------------------------------------
      lmax = maxord + 1
      nq = 1
      l = 2
      ialth = 2
      rmax = 10000.0d0
      rc = 0.0d0
      el0 = 1.0d0
      crate = 0.7d0
      hold = h
      meo = meth
      nslp = 0
      ipup = miter
      iret = 3
      go to 140
c-----------------------------------------------------------------------
c the following block handles preliminaries needed when jstart = -1.
c ipup is set to miter to force a matrix update.
c if an order increase is about to be considered (ialth = 1),
c ialth is reset to 2 to postpone consideration one more step.
c if the caller has changed meth, cfode is called to reset
c the coefficients of the method.
c if the caller has changed maxord to a value less than the current
c order nq, nq is reduced to maxord, and a new h chosen accordingly.
c if h is to be changed, yh must be rescaled.
c if h or meth is being changed, ialth is reset to l = nq + 1
c to prevent further changes in h for that many steps.
c-----------------------------------------------------------------------
 100  ipup = miter
      lmax = maxord + 1
      if (ialth .eq. 1) ialth = 2
      if (meth .eq. meo) go to 110
      call cfode (meth, elco, tesco)
      meo = meth
      if (nq .gt. maxord) go to 120
      ialth = l
      iret = 1
      go to 150
 110  if (nq .le. maxord) go to 160
 120  nq = maxord
      l = lmax
      do 125 i = 1,l
 125    el(i) = elco(i,nq)
      nqnyh = nq*nyh
      rc = rc*el(1)/el0
      el0 = el(1)
      conit = 0.5d0/dfloat(nq+2)
      ddn = vnorm (n, savf, ewt)/tesco(1,l)
      exdn = 1.0d0/dfloat(l)
      rhdn = 1.0d0/(1.3d0*ddn**exdn + 0.0000013d0)
      rh = dmin1(rhdn,1.0d0)
      iredo = 3
      if (h .eq. hold) go to 170
      rh = dmin1(rh,dabs(h/hold))
      h = hold
      go to 175
c-----------------------------------------------------------------------
c cfode is called to get all the integration coefficients for the
c current meth.  then the el vector and related constants are reset
c whenever the order nq is changed, or at the start of the problem.
c-----------------------------------------------------------------------
 140  call cfode (meth, elco, tesco)
 150  do 155 i = 1,l
 155    el(i) = elco(i,nq)
      nqnyh = nq*nyh
      rc = rc*el(1)/el0
      el0 = el(1)
      conit = 0.5d0/dfloat(nq+2)
      go to (160, 170, 200), iret
c-----------------------------------------------------------------------
c if h is being changed, the h ratio rh is checked against
c rmax, hmin, and hmxi, and the yh array rescaled.  ialth is set to
c l = nq + 1 to prevent a change of h for that many steps, unless
c forced by a convergence or error test failure.
c-----------------------------------------------------------------------
 160  if (h .eq. hold) go to 200
      rh = h/hold
      h = hold
      iredo = 3
      go to 175
 170  rh = dmax1(rh,hmin/dabs(h))
 175  rh = dmin1(rh,rmax)
      rh = rh/dmax1(1.0d0,dabs(h)*hmxi*rh)
      r = 1.0d0
      do 180 j = 2,l
        r = r*rh
        do 180 i = 1,n
 180      yh(i,j) = yh(i,j)*r
      h = h*rh
      rc = rc*rh
      ialth = l
      if (iredo .eq. 0) go to 690
c-----------------------------------------------------------------------
c this section computes the predicted values by effectively
c multiplying the yh array by the pascal triangle matrix.
c rc is the ratio of new to old values of the coefficient  h*el(1).
c when rc differs from 1 by more than ccmax, ipup is set to miter
c to force pjac to be called.
c in any case, pjac is called at least every msbp steps.
c-----------------------------------------------------------------------
 200  if (dabs(rc-1.0d0) .gt. ccmax) ipup = miter
      if (nst .ge. nslp+msbp) ipup = miter
      tn = tn + h
      i1 = nqnyh + 1
      do 215 jb = 1,nq
        i1 = i1 - nyh
cdir$ ivdep
        do 210 i = i1,nqnyh
 210      yh1(i) = yh1(i) + yh1(i+nyh)
 215    continue
c-----------------------------------------------------------------------
c up to maxcor corrector iterations are taken.  a convergence test is
c made on the r.m.s. norm of each correction, weighted by h and the
c error weight vector ewt.  the sum of the corrections is accumulated
c in acor(i).  the yh array is not altered in the corrector loop.
c-----------------------------------------------------------------------
 220  m = 0
      do 230 i = 1,n
        savf(i) = yh(i,2) / h
 230    y(i) = yh(i,1)
      if (ipup .le. 0) go to 240
c-----------------------------------------------------------------------
c if indicated, the matrix p = a - h*el(1)*dr/dy is reevaluated and
c preprocessed before starting the corrector iteration.  ipup is set
c to 0 as an indicator that this has been done.
c-----------------------------------------------------------------------
      call pjac (neq, y, yh, nyh, ewt, acor, savr, savf, wm, iwm,
     1   res, jac, adda )
      ipup = 0
      rc = 1.0d0
      nslp = nst
      crate = 0.7d0
      if (ierpj .eq. 0) go to 250
      ires = ierpj
      go to (430, 435, 430), ires
c get residual at predicted values, if not already done in pjac. -------
 240  ires = 1
      call res ( neq, tn, y, savf, savr, ires )
      nre = nre + 1
      kgo = iabs(ires)
      go to ( 250, 435, 430 ) , kgo
 250  do 260 i = 1,n
 260    acor(i) = 0.0d0
c-----------------------------------------------------------------------
c solve the linear system with the current residual as
c right-hand side and p as coefficient matrix.
c-----------------------------------------------------------------------
 270  continue
      call slvs (wm, iwm, savr, savf)
      if (iersl .lt. 0) go to 430
      if (iersl .gt. 0) go to 410
      el1h = el(1) * h
      del = vnorm (n, savr, ewt) * dabs(h)
      do 380 i = 1,n
        acor(i) = acor(i) + savr(i)
        savf(i) = acor(i) + yh(i,2)/h
 380    y(i) = yh(i,1) + el1h*acor(i)
c-----------------------------------------------------------------------
c test for convergence.  if m.gt.0, an estimate of the convergence
c rate constant is stored in crate, and this is used in the test.
c-----------------------------------------------------------------------
      if (m .ne. 0) crate = dmax1(0.2d0*crate,del/delp)
      dcon = del*dmin1(1.0d0,1.5d0*crate)/(tesco(2,nq)*conit)
      if (dcon .le. 1.0d0) go to 460
      m = m + 1
      if (m .eq. maxcor) go to 410
      if (m .ge. 2 .and. del .gt. 2.0d0*delp) go to 410
      delp = del
      ires = 1
      call res ( neq, tn, y, savf, savr, ires )
      nre = nre + 1
      kgo = iabs(ires)
      go to ( 270, 435, 410 ) , kgo
c-----------------------------------------------------------------------
c the correctors failed to converge, or res has returned abnormally.
c on a convergence failure, if the jacobian is out of date, pjac is
c called for the next try.  otherwise the yh array is retracted to its
c values before prediction, and h is reduced, if possible.
c take an error exit if ires = 2, or h cannot be reduced, or mxncf
c failures have occurred, or a fatal error occurred in pjac or slvs.
c-----------------------------------------------------------------------
 410  icf = 1
      if (jcur .eq. 1) go to 430
      ipup = miter
      go to 220
 430  icf = 2
      ncf = ncf + 1
      rmax = 2.0d0
 435  tn = told
      i1 = nqnyh + 1
      do 445 jb = 1,nq
        i1 = i1 - nyh
cdir$ ivdep
        do 440 i = i1,nqnyh
 440      yh1(i) = yh1(i) - yh1(i+nyh)
 445    continue
      if (ires .eq. 2) go to 680
      if (ierpj .lt. 0 .or. iersl .lt. 0) go to 685
      if (dabs(h) .le. hmin*1.00001d0) go to 450
      if (ncf .eq. mxncf) go to 450
      rh = 0.25d0
      ipup = miter
      iredo = 1
      go to 170
 450  if (ires .eq. 3) go to 680
      go to 670
c-----------------------------------------------------------------------
c the corrector has converged.  jcur is set to 0
c to signal that the jacobian involved may need updating later.
c the local error test is made and control passes to statement 500
c if it fails.
c-----------------------------------------------------------------------
 460  jcur = 0
      if (m .eq. 0) dsm = del/tesco(2,nq)
      if (m .gt. 0) dsm = dabs(h) * vnorm (n, acor, ewt)/tesco(2,nq)
      if (dsm .gt. 1.0d0) go to 500
c-----------------------------------------------------------------------
c after a successful step, update the yh array.
c consider changing h if ialth = 1.  otherwise decrease ialth by 1.
c if ialth is then 1 and nq .lt. maxord, then acor is saved for
c use in a possible order increase on the next step.
c if a change in h is considered, an increase or decrease in order
c by one is considered also.  a change in h is made only if it is by a
c factor of at least 1.1.  if not, ialth is set to 3 to prevent
c testing for that many steps.
c-----------------------------------------------------------------------
      kflag = 0
      iredo = 0
      nst = nst + 1
      hu = h
      nqu = nq
      do 470 j = 1,l
        eljh = el(j)*h
        do 470 i = 1,n
 470      yh(i,j) = yh(i,j) + eljh*acor(i)
      ialth = ialth - 1
      if (ialth .eq. 0) go to 520
      if (ialth .gt. 1) go to 700
      if (l .eq. lmax) go to 700
      do 490 i = 1,n
 490    yh(i,lmax) = acor(i)
      go to 700
c-----------------------------------------------------------------------
c the error test failed.  kflag keeps track of multiple failures.
c restore tn and the yh array to their previous values, and prepare
c to try the step again.  compute the optimum step size for this or
c one lower order.  after 2 or more failures, h is forced to decrease
c by a factor of 0.1 or less.
c-----------------------------------------------------------------------
 500  kflag = kflag - 1
      tn = told
      i1 = nqnyh + 1
      do 515 jb = 1,nq
        i1 = i1 - nyh
cdir$ ivdep
        do 510 i = i1,nqnyh
 510      yh1(i) = yh1(i) - yh1(i+nyh)
 515    continue
      rmax = 2.0d0
      if (dabs(h) .le. hmin*1.00001d0) go to 660
      if (kflag .le. -7) go to 660
      iredo = 2
      rhup = 0.0d0
      go to 540
c-----------------------------------------------------------------------
c regardless of the success or failure of the step, factors
c rhdn, rhsm, and rhup are computed, by which h could be multiplied
c at order nq - 1, order nq, or order nq + 1, respectively.
c in the case of failure, rhup = 0.0 to avoid an order increase.
c the largest of these is determined and the new order chosen
c accordingly.  if the order is to be increased, we compute one
c additional scaled derivative.
c-----------------------------------------------------------------------
 520  rhup = 0.0d0
      if (l .eq. lmax) go to 540
      do 530 i = 1,n
 530    savf(i) = acor(i) - yh(i,lmax)
      dup = dabs(h) * vnorm (n, savf, ewt)/tesco(3,nq)
      exup = 1.0d0/dfloat(l+1)
      rhup = 1.0d0/(1.4d0*dup**exup + 0.0000014d0)
 540  exsm = 1.0d0/dfloat(l)
      rhsm = 1.0d0/(1.2d0*dsm**exsm + 0.0000012d0)
      rhdn = 0.0d0
      if (nq .eq. 1) go to 560
      ddn = vnorm (n, yh(1,l), ewt)/tesco(1,nq)
      exdn = 1.0d0/dfloat(nq)
      rhdn = 1.0d0/(1.3d0*ddn**exdn + 0.0000013d0)
 560  if (rhsm .ge. rhup) go to 570
      if (rhup .gt. rhdn) go to 590
      go to 580
 570  if (rhsm .lt. rhdn) go to 580
      newq = nq
      rh = rhsm
      go to 620
 580  newq = nq - 1
      rh = rhdn
      if (kflag .lt. 0 .and. rh .gt. 1.0d0) rh = 1.0d0
      go to 620
 590  newq = l
      rh = rhup
      if (rh .lt. 1.1d0) go to 610
      r = h*el(l)/dfloat(l)
      do 600 i = 1,n
 600    yh(i,newq+1) = acor(i)*r
      go to 630
 610  ialth = 3
      go to 700
 620  if ((kflag .eq. 0) .and. (rh .lt. 1.1d0)) go to 610
      if (kflag .le. -2) rh = dmin1(rh,0.1d0)
c-----------------------------------------------------------------------
c if there is a change of order, reset nq, l, and the coefficients.
c in any case h is reset according to rh and the yh array is rescaled.
c then exit from 690 if the step was ok, or redo the step otherwise.
c-----------------------------------------------------------------------
      if (newq .eq. nq) go to 170
 630  nq = newq
      l = nq + 1
      iret = 2
      go to 150
c-----------------------------------------------------------------------
c all returns are made through this section.  h is saved in hold
c to allow the caller to change h on the next step.
c-----------------------------------------------------------------------
 660  kflag = -1
      go to 720
 670  kflag = -2
      go to 720
 680  kflag = -1 - ires
      go to 720
 685  kflag = -5
      go to 720
 690  rmax = 10.0d0
 700  r = h/tesco(2,nqu)
      do 710 i = 1,n
 710    acor(i) = acor(i)*r
 720  hold = h
      jstart = 1
      return
c----------------------- end of subroutine stodi -----------------------
      end
      subroutine prepji (neq, y, yh, nyh, ewt, rtem, savr, s, wm, iwm,
     1   res, jac, adda)
clll. optimize
      external res, jac, adda
      integer neq, nyh, iwm
      integer iownd, iowns,
     1   icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     2   maxord, maxcor, msbp, mxncf, n, nq, nst, nre, nje, nqu
      integer i, i1, i2, ier, ii, ires, j, j1, jj, lenp,
     1   mba, mband, meb1, meband, ml, ml3, mu
      double precision y, yh, ewt, rtem, savr, s, wm
      double precision rowns,
     1   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround
      double precision con, fac, hl0, r, srur, yi, yj, yjj
      dimension neq(1), y(1), yh(nyh,1), ewt(1), rtem(1),
     1   s(1), savr(1), wm(1), iwm(1)
      common /ls0001/ rowns(209),
     2   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround,
     3   iownd(14), iowns(6),
     4   icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     5   maxord, maxcor, msbp, mxncf, n, nq, nst, nre, nje, nqu
c-----------------------------------------------------------------------
c prepji is called by stodi to compute and process the matrix
c p = a - h*el(1)*j , where j is an approximation to the jacobian dr/dy,
c where r = g(t,y) - a(t,y)*s. here j is computed by the user-supplied
c routine jac if miter = 1 or 4, or by finite differencing if miter =
c 2 or 5. j is stored in wm, rescaled, and adda is called to generate
c p. p is then subjected to lu decomposition in preparation
c for later solution of linear systems with p as coefficient
c matrix.  this is done by mondgefa if miter = 1 or 2, and by
c dgbfa if miter = 4 or 5.
c
c in addition to variables described previously, communication
c with prepji uses the following..
c y     = array containing predicted values on entry.
c rtem  = work array of length n (acor in stodi).
c savr  = array used for output only.  on output it contains the
c         residual evaluated at current values of t and y.
c s     = array containing predicted values of dy/dt (savf in stodi).
c wm    = real work space for matrices.  on output it contains the
c         lu decomposition of p.
c         storage of matrix elements starts at wm(3).
c         wm also contains the following matrix-related data..
c         wm(1) = sqrt(uround), used in numerical jacobian increments.
c iwm   = integer work space containing pivot information, starting at
c         iwm(21).  iwm also contains the band parameters
c         ml = iwm(1) and mu = iwm(2) if miter is 4 or 5.
c el0   = el(1) (input).
c ierpj = output error flag.
c         = 0 if no trouble occurred,
c         = 1 if the p matrix was found to be singular,
c         = ires (= 2 or 3) if res returned ires = 2 or 3.
c jcur  = output flag = 1 to indicate that the jacobian matrix
c         (or approximation) is now current.
c this routine also uses the common variables el0, h, tn, uround,
c miter, n, nre, and nje.
c-----------------------------------------------------------------------
      nje = nje + 1
      hl0 = h*el0
      ierpj = 0
      jcur = 1
      go to (100, 200, 300, 400, 500), miter
c if miter = 1, call res, then jac, and multiply by scalar. ------------
 100  ires = 1
      call res (neq, tn, y, s, savr, ires)
      nre = nre + 1
      if (ires .gt. 1) go to 600
      lenp = n*n
      do 110 i = 1,lenp
 110    wm(i+2) = 0.0d0
      call jac ( neq, tn, y, s, 0, 0, wm(3), n )
      con = -hl0
      do 120 i = 1,lenp
 120    wm(i+2) = wm(i+2)*con
      go to 240
c if miter = 2, make n + 1 calls to res to approximate j. --------------
 200  continue
      ires = -1
      call res (neq, tn, y, s, savr, ires)
      nre = nre + 1
      if (ires .gt. 1) go to 600
      srur = wm(1)
      j1 = 2
      do 230 j = 1,n
        yj = y(j)
        r = dmax1(srur*dabs(yj),0.01d0/ewt(j))
        y(j) = y(j) + r
        fac = -hl0/r
        call res ( neq, tn, y, s, rtem, ires )
        nre = nre + 1
        if (ires .gt. 1) go to 600
        do 220 i = 1,n
 220      wm(i+j1) = (rtem(i) - savr(i))*fac
        y(j) = yj
        j1 = j1 + n
 230    continue
      ires = 1
      call res (neq, tn, y, s, savr, ires)
      nre = nre + 1
      if (ires .gt. 1) go to 600
c add matrix a. --------------------------------------------------------
 240  continue
      call adda(neq, tn, y, 0, 0, wm(3), n)
c do lu decomposition on p. --------------------------------------------
      call mondgefa (wm(3), n, n, iwm(21), ier)
      if (ier .ne. 0) ierpj = 1
      return
c dummy section for miter = 3
 300  return
c if miter = 4, call res, then jac, and multiply by scalar. ------------
 400  ires = 1
      call res (neq, tn, y, s, savr, ires)
      nre = nre + 1
      if (ires .gt. 1) go to 600
      ml = iwm(1)
      mu = iwm(2)
      ml3 = ml + 3
      mband = ml + mu + 1
      meband = mband + ml
      lenp = meband*n
      do 410 i = 1,lenp
 410    wm(i+2) = 0.0d0
      call jac ( neq, tn, y, s, ml, mu, wm(ml3), meband)
      con = -hl0
      do 420 i = 1,lenp
 420    wm(i+2) = wm(i+2)*con
      go to 570
c if miter = 5, make ml + mu + 2 calls to res to approximate j. --------
 500  continue
      ires = -1
      call res (neq, tn, y, s, savr, ires)
      nre = nre + 1
      if (ires .gt. 1) go to 600
      ml = iwm(1)
      mu = iwm(2)
      ml3 = ml + 3
      mband = ml + mu + 1
      mba = min0(mband,n)
      meband = mband + ml
      meb1 = meband - 1
      srur = wm(1)
      do 560 j = 1,mba
        do 530 i = j,n,mband
          yi = y(i)
          r = dmax1(srur*dabs(yi),0.01d0/ewt(i))
 530      y(i) = y(i) + r
        call res ( neq, tn, y, s, rtem, ires)
        nre = nre + 1
        if (ires .gt. 1) go to 600
        do 550 jj = j,n,mband
          y(jj) = yh(jj,1)
          yjj = y(jj)
          r = dmax1(srur*dabs(yjj),0.01d0/ewt(jj))
          fac = -hl0/r
          i1 = max0(jj-mu,1)
          i2 = min0(jj+ml,n)
          ii = jj*meb1 - ml + 2
          do 540 i = i1,i2
 540        wm(ii+i) = (rtem(i) - savr(i))*fac
 550      continue
 560    continue
      ires = 1
      call res (neq, tn, y, s, savr, ires)
      nre = nre + 1
      if (ires .gt. 1) go to 600
c add matrix a. --------------------------------------------------------
 570  continue
      call adda(neq, tn, y, ml, mu, wm(ml3), meband)
c do lu decomposition of p. --------------------------------------------
      call dgbfa (wm(3), meband, n, ml, mu, iwm(21), ier)
      if (ier .ne. 0) ierpj = 1
      return
c error return for ires = 2 or ires = 3 return from res. ---------------
 600  ierpj = ires
      return
c----------------------- end of subroutine prepji ----------------------
      end
      subroutine lsoibt (res, adda, jac, neq, y, ydoti, t, tout, itol,
     1  rtol, atol, itask, istate, iopt, rwork, lrw, iwork, liw, mf )
      external res, adda, jac
      integer neq, itol, itask, istate, iopt, lrw, iwork, liw, mf
      double precision y, ydoti, t, tout, rtol, atol, rwork
      dimension neq(1), y(1), ydoti(1), rtol(1), atol(1), rwork(lrw),
     1          iwork(liw)
c-----------------------------------------------------------------------
c this is the march 30, 1987 version of lsoibt..
c  livermore solver for ordinary differential equations given in
c  implicit form, with block-tridiagonal jacobian treatment.
c this version is in double precision.
c
c lsoibt solves the initial value problem for linearly implicit
c systems of first order ode-s,
c     a(t,y) * dy/dt = g(t,y) ,  where a(t,y) is a square matrix,
c or, in component form,
c     ( a   * ( dy / dt ))  + ... +  ( a     * ( dy   / dt ))  =
c        i,1      1                     i,neq      neq
c
c      =   g ( t, y , y ,..., y    )   ( i = 1,...,neq )
c           i      1   2       neq
c
c if a is singular, this is a differential-algebraic system.
c
c lsoibt is a variant version of the lsodi package, for the case where
c the matrices a, dg/dy, and d(a*s)/dy are all block-tridiagonal.
c-----------------------------------------------------------------------
c reference..
c     alan c. hindmarsh,  odepack, a systematized collection of ode
c     solvers, in scientific computing,  r. s. stepleman et al. (eds.),
c     north-holland, amsterdam, 1983, pp. 55-64.
c-----------------------------------------------------------------------
c authors..   charles s. kenney
c                formerly at..
c             naval weapons center
c             china lake, ca 93555
c    and
c             jeffrey f. painter  and
c             alan c. hindmarsh
c             computing and mathematics research division, l-316
c             lawrence livermore national laboratory
c             livermore, ca 94550.
c-----------------------------------------------------------------------
c summary of usage.
c
c communication between the user and the lsoibt package, for normal
c situations, is summarized here.  this summary describes only a subset
c of the full set of options available.  see the full description for
c details, including optional communication, nonstandard options,
c and instructions for special situations.  see also the example
c problem (with program and output) following this summary.
c
c a. first, provide a subroutine of the form..
c               subroutine res (neq, t, y, s, r, ires)
c               dimension y(neq), s(neq), r(neq)
c which computes the residual function
c     r = g(t,y)  -  a(t,y) * s ,
c as a function of t and the vectors y and s.  (s is an internally
c generated approximation to dy/dt.)  the arrays y and s are inputs
c to the res routine and should not be altered.  the residual
c vector is to be stored in the array r.  the argument ires should be
c ignored for casual use of lsoibt.  (for uses of ires, see the
c paragraph on res in the full description below.)
c
c b. next, identify the block structure of the matrices a = a(t,y) and
c dr/dy.  lsoibt must deal internally with a linear combination, p, of
c these two matrices.  the matrix p (hence both a and dr/dy) must have
c a block-tridiagonal form with fixed structure parameters
c     mb = block size, mb .ge. 1, and
c     nb = number of blocks in each direction, nb .ge. 4,
c with mb*nb = neq.  in each of the nb block-rows of the matrix p
c (each consisting of mb consecutive rows), the nonzero elements are
c to lie in three consecutive mb by mb blocks.  in block-rows
c 2 through nb - 1, these are centered about the main diagonal.
c in block-rows 1 and nb, they are the diagonal blocks and the two
c blocks adjacent to the diagonal block.  (thus block positions (1,3)
c and (nb,nb-2) can be nonzero.)
c alternatively, p (hence a and dr/dy) may be only approximately
c equal to matrices with this form, and lsoibt should still succeed.
c the block-tridiagonal matrix p is described by three arrays,
c each of size mb by mb by nb..
c     pa = array of diagonal blocks,
c     pb = array of superdiagonal (and one subdiagonal) blocks, and
c     pc = array of subdiagonal (and one superdiagonal) blocks.
c specifically, the three mb by mb blocks in the k-th block-row of p
c are stored in (reading across)..
c     pc(*,*,k) = block to the left of the diagonal block,
c     pa(*,*,k) = diagonal block, and
c     pb(*,*,k) = block to the right of the diagonal block,
c except for k = 1, where the three blocks (reading across) are
c     pa(*,*,1) (= diagonal block), pb(*,*,1), and pc(*,*,1),
c and k = nb, where they are
c     pb(*,*,nb), pc(*,*,nb), and pa(*,*,nb) (= diagonal block).
c (each asterisk * stands for an index that ranges from 1 to mb.)
c
c c. you must also provide a subroutine of the form..
c     subroutine adda (neq, t, y, mb, nb, pa, pb, pc)
c     dimension y(neq), pa(mb,mb,nb), pb(mb,mb,nb), pc(mb,mb,nb)
c which adds the nonzero blocks of the matrix a = a(t,y) to the
c contents of the arrays pa, pb, and pc, following the structure
c description in paragraph b above.
c t and the y array are input and should not be altered.
c thus the affect of adda should be the following..
c     do 30 k = 1,nb
c       do 20 j = 1,mb
c         do 10 i = 1,mb
c           pa(i,j,k) = pa(i,j,k) +
c             ( (i,j) element of k-th diagonal block of a)
c           pb(i,j,k) = pb(i,j,k) +
c             ( (i,j) element of block in block position (k,k+1) of a,
c             or in block position (nb,nb-2) if k = nb)
c           pc(i,j,k) = pc(i,j,k) +
c             ( (i,j) element of block in block position (k,k-1) of a,
c             or in block position (1,3) if k = 1)
c 10        continue
c 20      continue
c 30    continue
c
c d. for the sake of efficiency, you are encouraged to supply the
c jacobian matrix dr/dy in closed form, where r = g(t,y) - a(t,y)*s
c (s = a fixed vector) as above.  if dr/dy is being supplied,
c use mf = 21, and provide a subroutine of the form..
c     subroutine jac (neq, t, y, s, mb, nb, pa, pb, pc)
c     dimension y(neq), s(neq), pa(mb,mb,nb), pb(mb,mb,nb), pc(mb,mb,nb)
c which computes dr/dy as a function of t, y, and s.  here t, y, and
c s are inputs, and the routine is to load dr/dy into pa, pb, pc,
c according to the structure description in paragraph b above.
c that is, load the diagonal blocks into pa, the superdiagonal blocks
c (and block (nb,nb-2) ) into pb, and the subdiagonal blocks (and
c block (1,3) ) into pc.  the blocks in block-row k of dr/dy are to
c be loaded into pa(*,*,k), pb(*,*,k), and pc(*,*,k).
c     only nonzero elements need be loaded, and the indexing
c of pa, pb, and pc is the same as in the adda routine.
c     note that if a is independent of y (or this dependence
c is weak enough to be ignored) then jac is to compute dg/dy.
c     if it is not feasible to provide a jac routine, use
c mf = 22, and lsoibt will compute an approximate jacobian
c internally by difference quotients.
c
c e. next decide whether or not to provide the initial value of the
c derivative vector dy/dt.  if the initial value of a(t,y) is
c nonsingular (and not too ill-conditioned), you may let lsoibt compute
c this vector (istate = 0).  (lsoibt will solve the system a*s = g for
c s, with initial values of a and g.)  if a(t,y) is initially
c singular, then the system is a differential-algebraic system, and
c you must make use of the particular form of the system to compute the
c initial values of y and dy/dt.  in that case, use istate = 1 and
c load the initial value of dy/dt into the array ydoti.
c the input array ydoti and the initial y array must be consistent with
c the equations a*dy/dt = g.  this implies that the initial residual
c r = g(t,y) - a(t,y)*ydoti   must be approximately zero.
c
c f. write a main program which calls subroutine lsoibt once for
c each point at which answers are desired.  this should also provide
c for possible use of logical unit 6 for output of error messages
c by lsoibt.  on the first call to lsoibt, supply arguments as follows..
c res    = name of user subroutine for residual function r.
c adda   = name of user subroutine for computing and adding a(t,y).
c jac    = name of user subroutine for jacobian matrix dr/dy
c          (mf = 21).  if not used, pass a dummy name.
c note.. the names for the res and adda routines and (if used) the
c        jac routine must be declared external in the calling program.
c neq    = number of scalar equations in the system.
c y      = array of initial values, of length neq.
c ydoti  = array of length neq (containing initial dy/dt if istate = 1).
c t      = the initial value of the independent variable.
c tout   = first point where output is desired (.ne. t).
c itol   = 1 or 2 according as atol (below) is a scalar or array.
c rtol   = relative tolerance parameter (scalar).
c atol   = absolute tolerance parameter (scalar or array).
c          the estimated local error in y(i) will be controlled so as
c          to be roughly less (in magnitude) than
c             ewt(i) = rtol*abs(y(i)) + atol     if itol = 1, or
c             ewt(i) = rtol*abs(y(i)) + atol(i)  if itol = 2.
c          thus the local error test passes if, in each component,
c          either the absolute error is less than atol (or atol(i)),
c          or the relative error is less than rtol.
c          use rtol = 0.0 for pure absolute error control, and
c          use atol = 0.0 (or atol(i) = 0.0) for pure relative error
c          control.  caution.. actual (global) errors may exceed these
c          local tolerances, so choose them conservatively.
c itask  = 1 for normal computation of output values of y at t = tout.
c istate = integer flag (input and output).  set istate = 1 if the
c          initial dy/dt is supplied, and 0 otherwise.
c iopt   = 0 to indicate no optional inputs used.
c rwork  = real work array of length at least..
c             22 + 9*neq + 3*mb*mb*nb        for mf = 21 or 22.
c lrw    = declared length of rwork (in user-s dimension).
c iwork  = integer work array of length at least 20 + neq.
c          input in iwork(1) the block size mb and in iwork(2) the
c          number nb of blocks in each direction along the matrix a.
c          these must satisfy  mb .ge. 1, nb .ge. 4, and mb*nb = neq.
c liw    = declared length of iwork (in user-s dimension).
c mf     = method flag.  standard values are..
c          21 for a user-supplied jacobian.
c          22 for an internally generated jacobian.
c          for other choices of mf, see the paragraph on mf in
c          the full description below.
c note that the main program must declare arrays y, ydoti, rwork, iwork,
c and possibly atol.
c
c g. the output from the first call (or any call) is..
c      y = array of computed values of y(t) vector.
c      t = corresponding value of independent variable (normally tout).
c istate = 2  if lsoibt was successful, negative otherwise.
c          -1 means excess work done on this call (check all inputs).
c          -2 means excess accuracy requested (tolerances too small).
c          -3 means illegal input detected (see printed message).
c          -4 means repeated error test failures (check all inputs).
c          -5 means repeated convergence failures (perhaps bad jacobian
c             supplied or wrong choice of tolerances).
c          -6 means error weight became zero during problem. (solution
c             component i vanished, and atol or atol(i) = 0.)
c          -7 cannot occur in casual use.
c          -8 means lsoibt was unable to compute the initial dy/dt.
c             in casual use, this means a(t,y) is initially singular.
c             supply ydoti and use istate = 1 on the first call.
c
c  if lsoibt returns istate = -1, -4, or -5, then the output of
c  lsoibt also includes ydoti = array containing residual vector
c  r = g - a * dy/dt  evaluated at the current t, y, and dy/dt.
c
c h. to continue the integration after a successful return, simply
c reset tout and call lsoibt again.  no other parameters need be reset.
c
c
c-----------------------------------------------------------------------
c example problem.
c
c the following is an example problem, with the coding needed
c for its solution by lsoibt.  the problem comes from the partial
c differential equation (the burgers equation)
c   du/dt  =  - u * du/dx  +  eta * d**2 u/dx**2,   eta = .05,
c on -1 .le. x .le. 1.  the boundary conditions are
c   du/dx = 0  at x = -1 and at x = 1.
c the initial profile is a square wave,
c   u = 1 in abs(x) .lt. .5,  u = .5 at abs(x) = .5,  u = 0 elsewhere.
c the p.d.e. is discretized in x by a simplified galerkin method,
c using piecewise linear basis functions, on a grid of 40 intervals.
c the equations at x = -1 and 1 use a 3-point difference approximation
c for the right-hand side.  the result is a system a * dy/dt = g(y),
c of size neq = 41, where y(i) is the approximation to u at x = x(i),
c with x(i) = -1 + (i-1)*delx, delx = 2/(neq-1) = .05.  the individual
c equations in the system are
c   dy(1)/dt = ( y(3) - 2*y(2) + y(1) ) * eta / delx**2,
c   dy(neq)/dt = ( y(neq-2) - 2*y(neq-1) + y(neq) ) * eta / delx**2,
c and for i = 2, 3, ..., neq-1,
c   (1/6) dy(i-1)/dt + (4/6) dy(i)/dt + (1/6) dy(i+1)/dt
c       = ( y(i-1)**2 - y(i+1)**2 ) / (4*delx)
c         + ( y(i+1) - 2*y(i) + y(i-1) ) * eta / delx**2.
c the following coding solves the problem with mf = 21, with output
c of solution statistics at t = .1, .2, .3, and .4, and of the
c solution vector at t = .4.  here the block size is just mb = 1.
c
c     external resid, addabt, jacbt
c     double precision atol, rtol, rwork, t, tout, y, ydoti
c     dimension y(41), ydoti(41), rwork(514), iwork(61)
c     neq = 41
c     do 10 i = 1,neq
c  10   y(i) = 0.0d0
c     y(11) = 0.5d0
c     do 20 i = 12,30
c  20   y(i) = 1.0d0
c     y(31) = 0.5d0
c     t = 0.0d0
c     tout = 0.1d0
c     itol = 1
c     rtol = 1.0d-4
c     atol = 1.0d-5
c     itask = 1
c     istate = 0
c     iopt = 0
c     lrw = 514
c     liw = 61
c     iwork(1) = 1
c     iwork(2) = neq
c     mf = 21
c     do 40 io = 1,4
c       call lsoibt (resid, addabt, jacbt, neq, y, ydoti, t, tout,
c    1   itol,rtol,atol, itask, istate, iopt, rwork,lrw,iwork,liw, mf)
c       write (6,30) t, iwork(11), iwork(12), iwork(13)
c  30   format(7h at t =,f5.2,14h   no. steps =,i4,11h  no. r-s =,i4,
c    1   11h  no. j-s =,i3)
c       if (istate .ne. 2)  go to 90
c       tout = tout + 0.1d0
c  40   continue
c     write(6,50) (y(i),i=1,neq)
c  50 format(/24h final solution values../9(5e12.4/))
c     stop
c  90 write(6,95) istate
c  95 format(///22h error halt.. istate =,i3)
c     stop
c     end
c
c     subroutine resid (n, t, y, s, r, ires)
c     double precision delx, eta, eodsq, r, s, t, y
c     dimension y(n), s(n), r(n)
c     data eta/0.05d0/, delx/0.05d0/
c     eodsq = eta/delx**2
c     r(1) = eodsq*(y(3) - 2.0d0*y(2) + y(1)) - s(1)
c     nm1 = n - 1
c     do 10 i = 2,nm1
c       r(i) = (y(i-1)**2 - y(i+1)**2)/(4.0d0*delx)
c    1        + eodsq*(y(i+1) - 2.0d0*y(i) + y(i-1))
c    2        - (s(i-1) + 4.0d0*s(i) + s(i+1))/6.0d0
c  10   continue
c     r(n) = eodsq*(y(n-2) - 2.0d0*y(nm1) + y(n)) - s(n)
c     return
c     end
c
c     subroutine addabt (n, t, y, mb, nb, pa, pb, pc)
c     double precision pa, pb, pc, t, y
c     dimension y(n), pa(mb,mb,nb), pb(mb,mb,nb), pc(mb,mb,nb)
c     pa(1,1,1) = pa(1,1,1) + 1.0d0
c     nm1 = n - 1
c     do 10 k = 2,nm1
c       pa(1,1,k) = pa(1,1,k) + (4.0d0/6.0d0)
c       pb(1,1,k) = pb(1,1,k) + (1.0d0/6.0d0)
c       pc(1,1,k) = pc(1,1,k) + (1.0d0/6.0d0)
c  10   continue
c     pa(1,1,n) = pa(1,1,n) + 1.0d0
c     return
c     end
c
c     subroutine jacbt (n, t, y, s, mb, nb, pa, pb, pc)
c     double precision delx, eta, eodsq, pa, pb, pc, s, t, y
c     dimension y(n), s(n), pa(mb,mb,nb),pb(mb,mb,nb),pc(mb,mb,nb)
c     data eta/0.05d0/, delx/0.05d0/
c     eodsq = eta/delx**2
c     pa(1,1,1) = eodsq
c     pb(1,1,1) = -2.0d0*eodsq
c     pc(1,1,1) = eodsq
c     do 10 k = 2,n
c       pa(1,1,k) = -2.0d0*eodsq
c       pb(1,1,k) = -y(k+1)*(0.5d0/delx) + eodsq
c       pc(1,1,k) = y(k-1)*(0.5d0/delx) + eodsq
c  10   continue
c     pb(1,1,n) = eodsq
c     pc(1,1,n) = -2.0d0*eodsq
c     pa(1,1,n) = eodsq
c     return
c     end
c
c the output of this program (on a cdc-7600 in single precision)
c is as follows..
c
c at t = 0.10   no. steps =  35  no. r-s =  45  no. j-s =  9
c at t = 0.20   no. steps =  43  no. r-s =  54  no. j-s = 10
c at t = 0.30   no. steps =  48  no. r-s =  60  no. j-s = 11
c at t = 0.40   no. steps =  51  no. r-s =  64  no. j-s = 12
c
c final solution values..
c  1.2747e-02  1.1997e-02  1.5560e-02  2.3767e-02  3.7224e-02
c  5.6646e-02  8.2645e-02  1.1557e-01  1.5541e-01  2.0177e-01
c  2.5397e-01  3.1104e-01  3.7189e-01  4.3530e-01  5.0000e-01
c  5.6472e-01  6.2816e-01  6.8903e-01  7.4612e-01  7.9829e-01
c  8.4460e-01  8.8438e-01  9.1727e-01  9.4330e-01  9.6281e-01
c  9.7632e-01  9.8426e-01  9.8648e-01  9.8162e-01  9.6617e-01
c  9.3374e-01  8.7535e-01  7.8236e-01  6.5321e-01  5.0003e-01
c  3.4709e-01  2.1876e-01  1.2771e-01  7.3671e-02  5.0642e-02
c  5.4496e-02
c-----------------------------------------------------------------------
c full description of user interface to lsoibt.
c
c the user interface to lsoibt consists of the following parts.
c
c i.   the call sequence to subroutine lsoibt, which is a driver
c      routine for the solver.  this includes descriptions of both
c      the call sequence arguments and of user-supplied routines.
c      following these descriptions is a description of
c      optional inputs available through the call sequence, and then
c      a description of optional outputs (in the work arrays).
c
c ii.  descriptions of other routines in the lsoibt package that may be
c      (optionally) called by the user.  these provide the ability to
c      alter error message handling, save and restore the internal
c      common, and obtain specified derivatives of the solution y(t).
c
c iii. descriptions of common blocks to be declared in overlay
c      or similar environments, or to be saved when doing an interrupt
c      of the problem and continued solution later.
c
c iv.  description of two routines in the lsoibt package, either of
c      which the user may replace with his own version, if desired.
c      these relate to the measurement of errors.
c
c-----------------------------------------------------------------------
c part i.  call sequence.
c
c the call sequence parameters used for input only are
c     res, adda, jac, neq, tout, itol, rtol, atol, itask,
c     iopt, lrw, liw, mf,
c and those used for both input and output are
c     y, t, istate, ydoti.
c the work arrays rwork and iwork are also used for additional and
c optional inputs and optional outputs.  (the term output here refers
c to the return from subroutine lsoibt to the user-s calling program.)
c
c the legality of input parameters will be thoroughly checked on the
c initial call for the problem, but not checked thereafter unless a
c change in input parameters is flagged by istate = 3 on input.
c
c the descriptions of the call arguments are as follows.
c
c res    = the name of the user-supplied subroutine which supplies
c          the residual vector for the ode system, defined by
c            r = g(t,y) - a(t,y) * s
c          as a function of the scalar t and the vectors
c          s and y ( s approximates dy/dt ). this
c          subroutine is to have the form
c              subroutine res ( neq, t, y, s, r, ires )
c              dimension y(1), s(1), r(1)
c          where neq, t, y, s, and ires are input, and r and
c          ires are output. y, s, and r are arrays of length neq.
c          in dimension statements such as that above, 1 is a
c          dummy dimension. it can be replaced by any value.
c             on input, ires indicates how lsoibt will use the
c          returned array r, as follows..
c             ires = 1  means that lsoibt needs the full residual,
c                       r = g - a*s, exactly.
c             ires = -1 means that lsoibt is using r only to compute
c                       the jacobian dr/dy by difference quotients.
c          the res routine can ignore ires, or it can omit some terms
c          if ires = -1.  if a does not depend on y, then res can
c          just return r = g when ires = -1.  if g - a*s contains other
c          additive terms that are independent of y, these can also be
c          dropped, if done consistently, when ires = -1.
c             the subroutine should set the flag ires if it
c          encounters a halt condition or illegal input.
c          otherwise, it should not reset ires.  on output,
c             ires = 1 or -1 represents a normal return, and
c          lsoibt continues integrating the ode.  leave ires
c          unchanged from its input value.
c             ires = 2 tells lsoibt to immediately return control
c          to the calling program, with istate = 3.  this lets
c          the calling program change parameters of the prob-
c          lem if necessary.
c             ires = 3 represents an error condition (for example, an
c          illegal value of y). lsoibt tries to integrate the ode
c          without getting ires = 3 from res.  if it cannot, lsoibt
c          returns with istate = -7 or -1.
c             on an lsoibt return with istate = 3, -1, or -7, the values
c          of t and y returned correspond to the last point reached
c          successfully without getting the flag ires = 2 or 3.
c             the flag values ires = 2 and 3 should not be used to
c          handle switches or root-stop conditions.  this is better
c          done by calling lsoibt in a one-step mode and checking the
c          stopping function for a sign change at each step.
c             if quantities computed in the res routine are needed
c          externally to lsoibt, an extra call to res should be made
c          for this purpose, for consistent and accurate results.
c          to get the current dy/dt for the s argument, use intdy.
c             res must be declared external in the calling
c          program.  see note below for more about res.
c
c adda   = the name of the user-supplied subroutine which adds the
c          matrix a = a(t,y) to another matrix, p, stored in
c          block-tridiagonal form.  this routine is to have the form
c               subroutine adda (neq, t, y, mb, nb, pa, pb, pc)
c               dimension y(neq),
c              1    pa(mb,mb,nb), pb(mb,mb,nb), pc(mb,mb,nb)
c          where neq, t, y, mb, nb, and the arrays pa, pb, and pc
c          are input, and the arrays pa, pb, and pc are output.
c          y is an array of length neq, and the arrays pa, pb, pc
c          are all mb by mb by nb.
c             here a block-tridiagonal structure is assumed for a(t,y),
c          and also for the matrix p to which a is added here,
c          as described in paragraph b of the summary of usage above.
c          thus the affect of adda should be the following..
c               do 30 k = 1,nb
c                 do 20 j = 1,mb
c                   do 10 i = 1,mb
c                     pa(i,j,k) = pa(i,j,k) +
c                       ( (i,j) element of k-th diagonal block of a)
c                     pb(i,j,k) = pb(i,j,k) +
c                       ( (i,j) element of block (k,k+1) of a,
c                       or block (nb,nb-2) if k = nb)
c                     pc(i,j,k) = pc(i,j,k) +
c                       ( (i,j) element of block (k,k-1) of a,
c                       or block (1,3) if k = 1)
c           10        continue
c           20      continue
c           30    continue
c             adda must be declared external in the calling program.
c          see note below for more information about adda.
c
c jac    = the name of the user-supplied subroutine which supplies
c          the jacobian matrix, dr/dy, where r = g-a*s.  jac is
c          required if miter = 1 -- otherwise a dummy name can be
c          passed. this subroutine is to have the form
c               subroutine jac (neq, t, y, s, mb, nb, pa, pb, pc)
c               dimension y(neq), s(neq),
c              1    pa(mb,mb,nb), pb(mb,mb,nb), pc(mb,mb,nb)
c          where neq, t, y, s, mb, nb, and the arrays pa, pb, and pc
c          are input, and the arrays pa, pb, and pc are output.
c          y and s are arrays of length neq, and the arrays pa, pb, pc
c          are all mb by mb by nb.
c          pa, pb, and pc are to be loaded with partial derivatives
c          (elements of the jacobian matrix) on output, in terms of the
c          block-tridiagonal structure assumed, as described
c          in paragraph b of the summary of usage above.
c          that is, load the diagonal blocks into pa, the
c          superdiagonal blocks (and block (nb,nb-2) ) intp pb, and
c          the subdiagonal blocks (and block (1,3) ) into pc.
c          the blocks in block-row k of dr/dy are to be loaded into
c          pa(*,*,k), pb(*,*,k), and pc(*,*,k).
c          thus the affect of jac should be the following..
c               do 30 k = 1,nb
c                 do 20 j = 1,mb
c                   do 10 i = 1,mb
c                     pa(i,j,k) = ( (i,j) element of
c                       k-th diagonal block of dr/dy)
c                     pb(i,j,k) = ( (i,j) element of block (k,k+1)
c                       of dr/dy, or block (nb,nb-2) if k = nb)
c                     pc(i,j,k) = ( (i,j) element of block (k,k-1)
c                       of dr/dy, or block (1,3) if k = 1)
c           10        continue
c           20      continue
c           30    continue
c               pa, pb, and pc are preset to zero by the solver,
c          so that only the nonzero elements need be loaded by jac.
c          each call to jac is preceded by a call to res with the same
c          arguments neq, t, y, and s.  thus to gain some efficiency,
c          intermediate quantities shared by both calculations may be
c          saved in a user common block by res and not recomputed by jac
c          if desired.  also, jac may alter the y array, if desired.
c               jac need not provide dr/dy exactly.  a crude
c          approximation will do, so that lsoibt may be used when
c          a and dr/dy are not really block-tridiagonal, but are close
c          to matrices that are.
c               jac must be declared external in the calling program.
c               see note below for more about jac.
c
c       note on res, adda, and jac--  these
c          subroutines may access user-defined quantities in
c          neq(2),... and/or in y(neq(1)+1),... if neq is an array
c          (dimensioned in the subroutines) and/or y has length
c          exceeding neq(1).  however, these routines should not alter
c          neq(1), y(1),...,y(neq) or any other input variables.
c          see the descriptions of neq and y below.
c
c neq    = the size of the system (number of first order ordinary
c          differential equations or scalar algebraic equations).
c          used only for input.
c          neq may be decreased, but not increased, during the problem.
c          if neq is decreased (with istate = 3 on input), the
c          remaining components of y should be left undisturbed, if
c          these are to be accessed in res, adda, or jac.
c
c          normally, neq is a scalar, and it is generally referred to
c          as a scalar in this user interface description.  however,
c          neq may be an array, with neq(1) set to the system size.
c          (the lsoibt package accesses only neq(1).)  in either case,
c          this parameter is passed as the neq argument in all calls
c          to res, adda, and jac.  hence, if it is an array,
c          locations neq(2),... may be used to store other integer data
c          and pass it to  res, adda, or jac.  each such subroutine
c          must include neq in a dimension statement in that case.
c
c y      = a real array for the vector of dependent variables, of
c          length neq or more.  used for both input and output on the
c          first call (istate = 0 or 1), and only for output on other
c          calls.  on the first call, y must contain the vector of
c          initial values.  on output, y contains the computed solution
c          vector, evaluated at t.  if desired, the y array may be used
c          for other purposes between calls to the solver.
c
c          this array is passed as the y argument in all calls to res,
c          adda, and jac.  hence its length may exceed neq,
c          and locations y(neq+1),... may be used to store other real
c          data and pass it to res, adda, or jac.  (the lsoibt
c          package accesses only y(1),...,y(neq). )
c
c ydoti  = a real array for the initial value of the vector
c          dy/dt and for work space, of dimension at least neq.
c
c          on input...
c            if istate = 0 then lsoibt will compute the initial value
c          of dy/dt, if a is nonsingular.  thus ydoti will
c          serve only as work space and may have any value.
c            if istate = 1 then ydoti must contain the initial value
c          of dy/dt.
c            if istate = 2 or 3 (continuation calls) then ydoti
c          may have any value.
c            n.b.- if the initial value of a is singular, then
c          lsoibt cannot compute the initial value of dy/dt, so
c          it must be provided in ydoti, with istate=1.
c
c          on output, when lsoibt terminates abnormally with istate =
c          -1, -4, or -5, ydoti will contain the residual
c          r = g(t,y) - a(t,y)*(dy/dt).  if r is large, t is near
c          its initial value, and ydoti is supplied with istate=1,
c          there may have been an incorrect input value of
c          ydoti = dy/dt or the problem ( as given to lsoibt )
c          may not have a solution.
c
c          if desired, the ydoti array may be used for other
c          purposes between calls to the solver.
c
c t      = the independent variable.  on input, t is used only on the
c          first call, as the initial point of the integration.
c          on output, after each call, t is the value at which a
c          computed solution y is evaluated (usually the same as tout).
c          on an error return, t is the farthest point reached.
c
c tout   = the next value of t at which a computed solution is desired.
c          used only for input.
c
c          when starting the problem (istate = 0 or 1), tout may be
c          equal to t for one call, then should .ne. t for the next
c          call.  for the initial t, an input value of tout .ne. t is
c          used in order to determine the direction of the integration
c          (i.e. the algebraic sign of the step sizes) and the rough
c          scale of the problem.  integration in either direction
c          (forward or backward in t) is permitted.
c
c          if itask = 2 or 5 (one-step modes), tout is ignored after
c          the first call (i.e. the first call with tout .ne. t).
c          otherwise, tout is required on every call.
c
c          if itask = 1, 3, or 4, the values of tout need not be
c          monotone, but a value of tout which backs up is limited
c          to the current internal t interval, whose endpoints are
c          tcur - hu and tcur (see optional outputs, below, for
c          tcur and hu).
c
c itol   = an indicator for the type of error control.  see
c          description below under atol.  used only for input.
c
c rtol   = a relative error tolerance parameter, either a scalar or
c          an array of length neq.  see description below under atol.
c          input only.
c
c atol   = an absolute error tolerance parameter, either a scalar or
c          an array of length neq.  input only.
c
c             the input parameters itol, rtol, and atol determine
c          the error control performed by the solver.  the solver will
c          control the vector e = (e(i)) of estimated local errors
c          in y, according to an inequality of the form
c                      rms-norm of ( e(i)/ewt(i) )   .le.   1,
c          where       ewt(i) = rtol(i)*abs(y(i)) + atol(i),
c          and the rms-norm (root-mean-square norm) here is
c          rms-norm(v) = sqrt(sum v(i)**2 / neq).  here ewt = (ewt(i))
c          is a vector of weights which must always be positive, and
c          the values of rtol and atol should all be non-negative.
c          the following table gives the types (scalar/array) of
c          rtol and atol, and the corresponding form of ewt(i).
c
c             itol    rtol       atol          ewt(i)
c              1     scalar     scalar     rtol*abs(y(i)) + atol
c              2     scalar     array      rtol*abs(y(i)) + atol(i)
c              3     array      scalar     rtol(i)*abs(y(i)) + atol
c              4     array      scalar     rtol(i)*abs(y(i)) + atol(i)
c
c          when either of these parameters is a scalar, it need not
c          be dimensioned in the user-s calling program.
c
c          if none of the above choices (with itol, rtol, and atol
c          fixed throughout the problem) is suitable, more general
c          error controls can be obtained by substituting
c          user-supplied routines for the setting of ewt and/or for
c          the norm calculation.  see part iv below.
c
c          if global errors are to be estimated by making a repeated
c          run on the same problem with smaller tolerances, then all
c          components of rtol and atol (i.e. of ewt) should be scaled
c          down uniformly
c
c itask  = an index specifying the task to be performed.
c          input only.  itask has the following values and meanings.
c          1  means normal computation of output values of y(t) at
c             t = tout (by overshooting and interpolating).
c          2  means take one step only and return.
c          3  means stop at the first internal mesh point at or
c             beyond t = tout and return.
c          4  means normal computation of output values of y(t) at
c             t = tout but without overshooting t = tcrit.
c             tcrit must be input as rwork(1).  tcrit may be equal to
c             or beyond tout, but not behind it in the direction of
c             integration.  this option is useful if the problem
c             has a singularity at or beyond t = tcrit.
c          5  means take one step, without passing tcrit, and return.
c             tcrit must be input as rwork(1).
c
c          note..  if itask = 4 or 5 and the solver reaches tcrit
c          (within roundoff), it will return t = tcrit (exactly) to
c          indicate this (unless itask = 4 and tout comes before tcrit,
c          in which case answers at t = tout are returned first).
c
c istate = an index used for input and output to specify the
c          state of the calculation.
c
c          on input, the values of istate are as follows.
c          0  means this is the first call for the problem, and
c             lsoibt is to compute the initial value of dy/dt
c             (while doing other initializations).  see note below.
c          1  means this is the first call for the problem, and
c             the initial value of dy/dt has been supplied in
c             ydoti (lsoibt will do other initializations). see note
c             below.
c          2  means this is not the first call, and the calculation
c             is to continue normally, with no change in any input
c             parameters except possibly tout and itask.
c             (if itol, rtol, and/or atol are changed between calls
c             with istate = 2, the new values will be used but not
c             tested for legality.)
c          3  means this is not the first call, and the
c             calculation is to continue normally, but with
c             a change in input parameters other than
c             tout and itask.  changes are allowed in
c             neq, itol, rtol, atol, iopt, lrw, liw, mf, mb, nb,
c             and any of the optional inputs except h0.
c             (see iwork description for mb and nb.)
c          note..  a preliminary call with tout = t is not counted
c          as a first call here, as no initialization or checking of
c          input is done.  (such a call is sometimes useful for the
c          purpose of outputting the initial conditions.)
c          thus the first call for which tout .ne. t requires
c          istate = 0 or 1 on input.
c
c          on output, istate has the following values and meanings.
c           0 or 1  means nothing was done, as tout was equal to t with
c              istate = 0 or 1 on input.  (however, an internal counter
c              was set to detect and prevent repeated calls of this
c              type. )
c           2  means that the integration was performed successfully.
c           3  means that the user-supplied subroutine res signalled
c              lsoibt to halt the integration and return (ires=2).
c              integration as far as t was achieved with no occurrence
c              of ires=2, but this flag was set on attempting the next
c              step.
c          -1  means an excessive amount of work (more than mxstep
c              steps) was done on this call, before completing the
c              requested task, but the integration was otherwise
c              successful as far as t.  (mxstep is an optional input
c              and is normally 500.)  to continue, the user may
c              simply reset istate to a value .gt. 1 and call again
c              (the excess work step counter will be reset to 0).
c              in addition, the user may increase mxstep to avoid
c              this error return (see below on optional inputs).
c          -2  means too much accuracy was requested for the precision
c              of the machine being used.  this was detected before
c              completing the requested task, but the integration
c              was successful as far as t.  to continue, the tolerance
c              parameters must be reset, and istate must be set
c              to 3.  the optional output tolsf may be used for this
c              purpose.  (note.. if this condition is detected before
c              taking any steps, then an illegal input return
c              (istate = -3) occurs instead.)
c          -3  means illegal input was detected, before taking any
c              integration steps.  see written message for details.
c              note..  if the solver detects an infinite loop of calls
c              to the solver with illegal input, it will cause
c              the run to stop.
c          -4  means there were repeated error test failures on
c              one attempted step, before completing the requested
c              task, but the integration was successful as far as t.
c              the problem may have a singularity, or the input
c              may be inappropriate.
c          -5  means there were repeated convergence test failures on
c              one attempted step, before completing the requested
c              task, but the integration was successful as far as t.
c              this may be caused by an inaccurate jacobian matrix.
c          -6  means ewt(i) became zero for some i during the
c              integration.  pure relative error control (atol(i)=0.0)
c              was requested on a variable which has now vanished.
c              the integration was successful as far as t.
c          -7  means that the user-supplied subroutine res set
c              its error flag (ires=3) despite repeated tries by lsoibt
c              to avoid that condition.
c          -8  means that istate was 0 on input but lsoibt was unable
c              to compute the initial value of dy/dt.  see the
c              printed message for details.
c
c          note..  since the normal output value of istate is 2,
c          it does not need to be reset for normal continuation.
c          similarly, istate need not be reset if res told lsoibt
c          to return because the calling program must change
c          the parameters of the problem.
c          also, since a negative input value of istate will be
c          regarded as illegal, a negative output value requires the
c          user to change it, and possibly other inputs, before
c          calling the solver again.
c
c iopt   = an integer flag to specify whether or not any optional
c          inputs are being used on this call.  input only.
c          the optional inputs are listed separately below.
c          iopt = 0 means no optional inputs are being used.
c                   default values will be used in all cases.
c          iopt = 1 means one or more optional inputs are being used.
c
c rwork  = a real working array (double precision).
c          the length of rwork must be at least
c             20 + nyh*(maxord + 1) + 3*neq + lenwm    where
c          nyh    = the initial value of neq,
c          maxord = 12 (if meth = 1) or 5 (if meth = 2) (unless a
c                   smaller value is given as an optional input),
c          lenwm  = 3*mb*mb*nb + 2.
c          (see mf description for the definition of meth.)
c          thus if maxord has its default value and neq is constant,
c          this length is
c             22 + 16*neq + 3*mb*mb*nb     for mf = 11 or 12,
c             22 + 9*neq + 3*mb*mb*nb      for mf = 21 or 22.
c          the first 20 words of rwork are reserved for conditional
c          and optional inputs and optional outputs.
c
c          the following word in rwork is a conditional input..
c            rwork(1) = tcrit = critical value of t which the solver
c                       is not to overshoot.  required if itask is
c                       4 or 5, and ignored otherwise.  (see itask.)
c
c lrw    = the length of the array rwork, as declared by the user.
c          (this will be checked by the solver.)
c
c iwork  = an integer work array.  the length of iwork must be at least
c          20 + neq .  the first few words of iwork are used for
c          additional and optional inputs and optional outputs.
c
c          the following 2 words in iwork are additional required
c          inputs to lsoibt..
c            iwork(1) = mb = block size
c            iwork(2) = nb = number of blocks in the main diagonal
c          these must satisfy  mb .ge. 1, nb .ge. 4, and mb*nb = neq.
c
c liw    = the length of the array iwork, as declared by the user.
c          (this will be checked by the solver.)
c
c note..  the work arrays must not be altered between calls to lsoibt
c for the same problem, except possibly for the additional and
c optional inputs, and except for the last 3*neq words of rwork.
c the latter space is used for internal scratch space, and so is
c available for use by the user outside lsoibt between calls, if
c desired (but not for use by res, adda, or jac).
c
c mf     = the method flag.  used only for input.  the legal values of
c          mf are 11, 12, 21, and 22.
c          mf has decimal digits meth and miter.. mf = 10*meth + miter.
c            meth indicates the basic linear multistep method..
c              meth = 1 means the implicit adams method.
c              meth = 2 means the method based on backward
c                       differentiation formulas (bdf-s).
c                the bdf method is strongly preferred for stiff prob-
c              lems, while the adams method is preferred when the prob-
c              lem is not stiff. if the matrix a(t,y) is nonsingular,
c              stiffness here can be taken to mean that of the explicit
c              ode system dy/dt = a**(-1) * g.  if a is singular, the
c              concept of stiffness is not well defined.
c                if you do not know whether the problem is stiff, we
c              recommend using meth = 2.  if it is stiff, the advan-
c              tage of meth = 2 over 1 will be great, while if it is
c              not stiff, the advantage of meth = 1 will be slight.
c              if maximum efficiency is important, some experimentation
c              with meth may be necessary.
c            miter indicates the corrector iteration method..
c              miter = 1 means chord iteration with a user-supplied
c                        block-tridiagonal jacobian.
c              miter = 2 means chord iteration with an internally
c                        generated (difference quotient) block-
c                        tridiagonal jacobian approximation, using
c                        3*mb+1 extra calls to res per dr/dy evaluation.
c              if miter = 1, the user must supply a subroutine
c              jac (the name is arbitrary) as described above under jac.
c              for miter = 2, a dummy argument can be used.
c-----------------------------------------------------------------------
c optional inputs.
c
c the following is a list of the optional inputs provided for in the
c call sequence.  (see also part ii.)  for each such input variable,
c this table lists its name as used in this documentation, its
c location in the call sequence, its meaning, and the default value.
c the use of any of these inputs requires iopt = 1, and in that
c case all of these inputs are examined.  a value of zero for any
c of these optional inputs will cause the default value to be used.
c thus to use a subset of the optional inputs, simply preload
c locations 5 to 10 in rwork and iwork to 0.0 and 0 respectively, and
c then set those of interest to nonzero values.
c
c name    location      meaning and default value
c
c h0      rwork(5)  the step size to be attempted on the first step.
c                   the default value is determined by the solver.
c
c hmax    rwork(6)  the maximum absolute step size allowed.
c                   the default value is infinite.
c
c hmin    rwork(7)  the minimum absolute step size allowed.
c                   the default value is 0.  (this lower bound is not
c                   enforced on the final step before reaching tcrit
c                   when itask = 4 or 5.)
c
c maxord  iwork(5)  the maximum order to be allowed.  the default
c                   value is 12 if meth = 1, and 5 if meth = 2.
c                   if maxord exceeds the default value, it will
c                   be reduced to the default value.
c                   if maxord is changed during the problem, it may
c                   cause the current order to be reduced.
c
c mxstep  iwork(6)  maximum number of (internally defined) steps
c                   allowed during one call to the solver.
c                   the default value is 500.
c
c mxhnil  iwork(7)  maximum number of messages printed (per problem)
c                   warning that t + h = t on a step (h = step size).
c                   this must be positive to result in a non-default
c                   value.  the default value is 10.
c-----------------------------------------------------------------------
c optional outputs.
c
c as optional additional output from lsoibt, the variables listed
c below are quantities related to the performance of lsoibt
c which are available to the user.  these are communicated by way of
c the work arrays, but also have internal mnemonic names as shown.
c except where stated otherwise, all of these outputs are defined
c on any successful return from lsoibt, and on any return with
c istate = -1, -2, -4, -5, -6, or -7. on a return with -3 (illegal
c input) or -8, they will be unchanged from their existing values
c (if any), except possibly for tolsf, lenrw, and leniw.
c on any error return, outputs relevant to the error will be defined,
c as noted below.
c
c name    location      meaning
c
c hu      rwork(11) the step size in t last used (successfully).
c
c hcur    rwork(12) the step size to be attempted on the next step.
c
c tcur    rwork(13) the current value of the independent variable
c                   which the solver has actually reached, i.e. the
c                   current internal mesh point in t.  on output, tcur
c                   will always be at least as far as the argument
c                   t, but may be farther (if interpolation was done).
c
c tolsf   rwork(14) a tolerance scale factor, greater than 1.0,
c                   computed when a request for too much accuracy was
c                   detected (istate = -3 if detected at the start of
c                   the problem, istate = -2 otherwise).  if itol is
c                   left unaltered but rtol and atol are uniformly
c                   scaled up by a factor of tolsf for the next call,
c                   then the solver is deemed likely to succeed.
c                   (the user may also ignore tolsf and alter the
c                   tolerance parameters in any other way appropriate.)
c
c nst     iwork(11) the number of steps taken for the problem so far.
c
c nre     iwork(12) the number of residual evaluations (res calls)
c                   for the problem so far.
c
c nje     iwork(13) the number of jacobian evaluations (each involving
c                   an evaluation of a and dr/dy) for the problem so
c                   far.  this equals the number of calls to adda and
c                   (if miter = 1) to jac, and the number of matrix
c                   l-u decompositions.
c
c nqu     iwork(14) the method order last used (successfully).
c
c nqcur   iwork(15) the order to be attempted on the next step.
c
c imxer   iwork(16) the index of the component of largest magnitude in
c                   the weighted local error vector ( e(i)/ewt(i) ),
c                   on an error return with istate = -4 or -5.
c
c lenrw   iwork(17) the length of rwork actually required.
c                   this is defined on normal returns and on an illegal
c                   input return for insufficient storage.
c
c leniw   iwork(18) the length of iwork actually required.
c                   this is defined on normal returns and on an illegal
c                   input return for insufficient storage.
c
c
c the following two arrays are segments of the rwork array which
c may also be of interest to the user as optional outputs.
c for each array, the table below gives its internal name,
c its base address in rwork, and its description.
c
c name    base address      description
c
c yh      21             the nordsieck history array, of size nyh by
c                        (nqcur + 1), where nyh is the initial value
c                        of neq.  for j = 0,1,...,nqcur, column j+1
c                        of yh contains hcur**j/factorial(j) times
c                        the j-th derivative of the interpolating
c                        polynomial currently representing the solution,
c                        evaluated at t = tcur.
c
c acor     lenrw-neq+1   array of size neq used for the accumulated
c                        corrections on each step, scaled on output to
c                        represent the estimated local error in y on the
c                        last step. this is the vector e in the descrip-
c                        tion of the error control.  it is defined only
c                        on a return from lsoibt with istate = 2.
c
c-----------------------------------------------------------------------
c part ii.  other routines callable.
c
c the following are optional calls which the user may make to
c gain additional capabilities in conjunction with lsoibt.
c (the routines xsetun and xsetf are designed to conform to the
c slatec error handling package.)
c
c     form of call                  function
c   call xsetun(lun)          set the logical unit number, lun, for
c                             output of messages from lsoibt, if
c                             the default is not desired.
c                             the default value of lun is 6.
c
c   call xsetf(mflag)         set a flag to control the printing of
c                             messages by lsoibt.
c                             mflag = 0 means do not print. (danger..
c                             this risks losing valuable information.)
c                             mflag = 1 means print (the default).
c
c                             either of the above calls may be made at
c                             any time and will take effect immediately.
c
c   call srcom(rsav,isav,job) saves and restores the contents of
c                             the internal common blocks used by
c                             lsoibt (see part iii below).
c                             rsav must be a real array of length 218
c                             or more, and isav must be an integer
c                             array of length 41 or more.
c                             job=1 means save common into rsav/isav.
c                             job=2 means restore common from rsav/isav.
c                                srcom is useful if one is
c                             interrupting a run and restarting
c                             later, or alternating between two or
c                             more problems solved with lsoibt.
c
c   call intdy(,,,,,)         provide derivatives of y, of various
c        (see below)          orders, at a specified point t, if
c                             desired.  it may be called only after
c                             a successful return from lsoibt.
c
c the detailed instructions for using intdy are as follows.
c the form of the call is..
c
c   call intdy (t, k, rwork(21), nyh, dky, iflag)
c
c the input parameters are..
c
c t         = value of independent variable where answers are desired
c             (normally the same as the t last returned by lsoibt).
c             for valid results, t must lie between tcur - hu and tcur.
c             (see optional outputs for tcur and hu.)
c k         = integer order of the derivative desired.  k must satisfy
c             0 .le. k .le. nqcur, where nqcur is the current order
c             (see optional outputs).  the capability corresponding
c             to k = 0, i.e. computing y(t), is already provided
c             by lsoibt directly.  since nqcur .ge. 1, the first
c             derivative dy/dt is always available with intdy.
c rwork(21) = the base address of the history array yh.
c nyh       = column length of yh, equal to the initial value of neq.
c
c the output parameters are..
c
c dky       = a real array of length neq containing the computed value
c             of the k-th derivative of y(t).
c iflag     = integer flag, returned as 0 if k and t were legal,
c             -1 if k was illegal, and -2 if t was illegal.
c             on an error return, a message is also written.
c-----------------------------------------------------------------------
c part iii.  common blocks.
c
c if lsoibt is to be used in an overlay situation, the user
c must declare, in the primary overlay, the variables in..
c   (1) the call sequence to lsoibt,
c   (2) the two internal common blocks
c         /ls0001/  of length  257  (218 double precision words
c                         followed by 39 integer words),
c         /eh0001/  of length  2 (integer words).
c
c if lsoibt is used on a system in which the contents of internal
c common blocks are not preserved between calls, the user should
c declare the above two common blocks in his main program to insure
c that their contents are preserved.
c
c if the solution of a given problem by lsoibt is to be interrupted
c and then later continued, such as when restarting an interrupted run
c or alternating between two or more problems, the user should save,
c following the return from the last lsoibt call prior to the
c interruption, the contents of the call sequence variables and the
c internal common blocks, and later restore these values before the
c next lsoibt call for that problem.  to save and restore the common
c blocks, use subroutine srcom (see part ii above).
c
c-----------------------------------------------------------------------
c part iv.  optionally replaceable solver routines.
c
c below are descriptions of two routines in the lsoibt package which
c relate to the measurement of errors.  either routine can be
c replaced by a user-supplied version, if desired.  however, since such
c a replacement may have a major impact on performance, it should be
c done only when absolutely necessary, and only with great caution.
c (note.. the means by which the package version of a routine is
c superseded by the user-s version may be system-dependent.)
c
c (a) ewset.
c the following subroutine is called just before each internal
c integration step, and sets the array of error weights, ewt, as
c described under itol/rtol/atol above..
c     subroutine ewset (neq, itol, rtol, atol, ycur, ewt)
c where neq, itol, rtol, and atol are as in the lsoibt call sequence,
c ycur contains the current dependent variable vector, and
c ewt is the array of weights set by ewset.
c
c if the user supplies this subroutine, it must return in ewt(i)
c (i = 1,...,neq) a positive quantity suitable for comparing errors
c in y(i) to.  the ewt array returned by ewset is passed to the
c vnorm routine (see below), and also used by lsoibt in the computation
c of the optional output imxer, the diagonal jacobian approximation,
c and the increments for difference quotient jacobians.
c
c in the user-supplied version of ewset, it may be desirable to use
c the current values of derivatives of y.  derivatives up to order nq
c are available from the history array yh, described above under
c optional outputs.  in ewset, yh is identical to the ycur array,
c extended to nq + 1 columns with a column length of nyh and scale
c factors of h**j/factorial(j).  on the first call for the problem,
c given by nst = 0, nq is 1 and h is temporarily set to 1.0.
c the quantities nq, nyh, h, and nst can be obtained by including
c in ewset the statements..
c     double precision h, rls
c     common /ls0001/ rls(218),ils(39)
c     nq = ils(35)
c     nyh = ils(14)
c     nst = ils(36)
c     h = rls(212)
c thus, for example, the current value of dy/dt can be obtained as
c ycur(nyh+i)/h  (i=1,...,neq)  (and the division by h is
c unnecessary when nst = 0).
c
c (b) vnorm.
c the following is a real function routine which computes the weighted
c root-mean-square norm of a vector v..
c     d = vnorm (n, v, w)
c where..
c   n = the length of the vector,
c   v = real array of length n containing the vector,
c   w = real array of length n containing weights,
c   d = sqrt( (1/n) * sum(v(i)*w(i))**2 ).
c vnorm is called with n = neq and with w(i) = 1.0/ewt(i), where
c ewt is as set by subroutine ewset.
c
c if the user supplies this function, it should return a non-negative
c value of vnorm suitable for use in the error control in lsoibt.
c none of the arguments should be altered by vnorm.
c for example, a user-supplied vnorm routine might..
c   -substitute a max-norm of (v(i)*w(i)) for the rms-norm, or
c   -ignore some components of v in the norm, with the effect of
c    suppressing the error control on those components of y.
c-----------------------------------------------------------------------
c-----------------------------------------------------------------------
c other routines in the lsoibt package.
c
c in addition to subroutine lsoibt, the lsoibt package includes the
c following subroutines and function routines..
c  aigbt    computes the initial value of the vector
c             dy/dt = inverse(a) * g
c  intdy    computes an interpolated value of the y vector at t = tout.
c  stodi    is the core integrator, which does one step of the
c           integration and the associated error control.
c  cfode    sets all method coefficients and test constants.
c  ewset    sets the error weight vector ewt before each step.
c  vnorm    computes the weighted r.m.s. norm of a vector.
c  srcom    is a user-callable routine to save and restore
c           the contents of the internal common blocks.
c  pjibt    computes and preprocesses the jacobian matrix
c           and the newton iteration matrix p.
c  slsbt    manages solution of linear system in chord iteration.
c  decbt and solbt   are routines for solving block-tridiagonal
c           systems of linear algebraic equations.
c  mondgefa and mondgesl   are routines from linpack for solving full
c           systems of linear algebraic equations.
c  mondaxpy, mondscal, monidamax, and monddot   are basic linear algebra modules
c           (blas) used by the above linpack routines.
c  mond1mach   computes the unit roundoff in a machine-independent manner.
c  monwv, xsetun, and xsetf   handle the printing of all error
c           messages and warnings.  monwv is machine-dependent.
c note..  vnorm, monidamax, monddot, and mond1mach are function routines.
c all the others are subroutines.
c
c the intrinsic and external routines used by lsoibt are.. dabs,
c dmax1, dmin1, dfloat, iabs, max0, min0, mod, dsign, dsqrt, and write.
c
c a block data subprogram is also included with the package,
c for loading some of the variables in internal common.
c
c-----------------------------------------------------------------------
c the following card is for optimized compilation on llnl compilers.
clll. optimize
c-----------------------------------------------------------------------
      external pjibt, slsbt
      integer illin, init, lyh, lewt, lacor, lsavr, lwm, liwm,
     1   mxstep, mxhnil, nhnil, ntrep, nslast, nyh, iowns
      integer icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     1   maxord, maxcor, msbp, mxncf, n, nq, nst, nre, nje, nqu
      integer i, i1, i2, ier, iflag, imxer, ires, kgo,
     1   leniw, lenrw, lenwm, lp, lyd0, mb, mord, mxhnl0, mxstp0, nb
      double precision rowns,
     1   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround
      double precision atoli, ayi, big, ewti, h0, hmax, hmx, rh, rtoli,
     1   tcrit, tdist, tnext, tol, tolsf, tp, size, sum, w0,
     2   mond1mach, vnorm
      dimension mord(2)
      logical ihit
c-----------------------------------------------------------------------
c the following internal common block contains
c (a) variables which are local to any subroutine but whose values must
c     be preserved between calls to the routine (own variables), and
c (b) variables which are communicated between subroutines.
c block ls0001 is shared by the lsoibt, lsodi, and lsode packages.
c the structure of ls0001 is as follows..  all real variables are
c listed first, followed by all integers.  within each type, the
c variables are grouped with those local to subroutine lsoibt first,
c then those local to subroutine stodi, and finally those used
c for communication.  the block is declared in subroutines
c lsoibt, intdy, stodi, pjibt, and slsbt.  groups of variables are
c replaced by dummy arrays in the common declarations in routines
c where those variables are not used.
c-----------------------------------------------------------------------
      common /ls0001/ rowns(209),
     1   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround,
     2   illin, init, lyh, lewt, lacor, lsavr, lwm, liwm,
     3   mxstep, mxhnil, nhnil, ntrep, nslast, nyh, iowns(6),
     4   icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     5   maxord, maxcor, msbp, mxncf, n, nq, nst, nre, nje, nqu
c
      data  mord(1),mord(2)/12,5/, mxstp0/500/, mxhnl0/10/
c-----------------------------------------------------------------------
c block a.
c this code block is executed on every call.
c it tests istate and itask for legality and branches appropriately.
c if istate .gt. 1 but the flag init shows that initialization has
c not yet been done, an error return occurs.
c if istate = 0 or 1 and tout = t, jump to block g and return
c immediately.
c-----------------------------------------------------------------------
      if (istate .lt. 0 .or. istate .gt. 3) go to 601
      if (itask .lt. 1 .or. itask .gt. 5) go to 602
      if (istate .le. 1) go to 10
      if (init .eq. 0) go to 603
      if (istate .eq. 2) go to 200
      go to 20
 10   init = 0
      if (tout .eq. t) go to 430
 20   ntrep = 0
c-----------------------------------------------------------------------
c block b.
c the next code block is executed for the initial call (istate = 0 or 1)
c or for a continuation call with parameter changes (istate = 3).
c it contains checking of all inputs and various initializations.
c
c first check legality of the non-optional inputs neq, itol, iopt,
c mf, mb, and nb.
c-----------------------------------------------------------------------
      if (neq(1) .le. 0) go to 604
      if (istate .le. 1) go to 25
      if (neq(1) .gt. n) go to 605
 25   n = neq(1)
      if (itol .lt. 1 .or. itol .gt. 4) go to 606
      if (iopt .lt. 0 .or. iopt .gt. 1) go to 607
      meth = mf/10
      miter = mf - 10*meth
      if (meth .lt. 1 .or. meth .gt. 2) go to 608
      if (miter .lt. 1 .or. miter .gt. 2) go to 608
      mb = iwork(1)
      nb = iwork(2)
      if (mb .lt. 1 .or. mb .gt. n) go to 609
      if (nb .lt. 4) go to 610
      if (mb*nb .ne. n) go to 609
c next process and check the optional inputs. --------------------------
      if (iopt .eq. 1) go to 40
      maxord = mord(meth)
      mxstep = mxstp0
      mxhnil = mxhnl0
      if (istate .le. 1) h0 = 0.0d0
      hmxi = 0.0d0
      hmin = 0.0d0
      go to 60
 40   maxord = iwork(5)
      if (maxord .lt. 0) go to 611
      if (maxord .eq. 0) maxord = 100
      maxord = min0(maxord,mord(meth))
      mxstep = iwork(6)
      if (mxstep .lt. 0) go to 612
      if (mxstep .eq. 0) mxstep = mxstp0
      mxhnil = iwork(7)
      if (mxhnil .lt. 0) go to 613
      if (mxhnil .eq. 0) mxhnil = mxhnl0
      if (istate .gt. 1) go to 50
      h0 = rwork(5)
      if ((tout - t)*h0 .lt. 0.0d0) go to 614
 50   hmax = rwork(6)
      if (hmax .lt. 0.0d0) go to 615
      hmxi = 0.0d0
      if (hmax .gt. 0.0d0) hmxi = 1.0d0/hmax
      hmin = rwork(7)
      if (hmin .lt. 0.0d0) go to 616
c-----------------------------------------------------------------------
c set work array pointers and check lengths lrw and liw.
c pointers to segments of rwork and iwork are named by prefixing l to
c the name of the segment.  e.g., the segment yh starts at rwork(lyh).
c segments of rwork (in order) are denoted yh, wm, ewt, savr, acor.
c-----------------------------------------------------------------------
 60   lyh = 21
      if (istate .le. 1) nyh = n
      lwm = lyh + (maxord + 1)*nyh
      lenwm = 3*mb*mb*nb + 2
      lewt = lwm + lenwm
      lsavr = lewt + n
      lacor = lsavr + n
      lenrw = lacor + n - 1
      iwork(17) = lenrw
      liwm = 1
      leniw = 20 + n
      iwork(18) = leniw
      if (lenrw .gt. lrw) go to 617
      if (leniw .gt. liw) go to 618
c check rtol and atol for legality. ------------------------------------
      rtoli = rtol(1)
      atoli = atol(1)
      do 70 i = 1,n
        if (itol .ge. 3) rtoli = rtol(i)
        if (itol .eq. 2 .or. itol .eq. 4) atoli = atol(i)
        if (rtoli .lt. 0.0d0) go to 619
        if (atoli .lt. 0.0d0) go to 620
 70     continue
      if (istate .le. 1) go to 100
c if istate = 3, set flag to signal parameter changes to stodi. --------
      jstart = -1
      if (nq .le. maxord) go to 90
c maxord was reduced below nq.  copy yh(*,maxord+2) into ydoti.---------
      do 80 i = 1,n
 80     ydoti(i) = rwork(i+lwm-1)
c reload wm(1) = rwork(lwm), since lwm may have changed. ---------------
 90   rwork(lwm) = dsqrt(uround)
      if (n .eq. nyh) go to 200
c neq was reduced.  zero part of yh to avoid undefined references. -----
      i1 = lyh + l*nyh
      i2 = lyh + (maxord + 1)*nyh - 1
      if (i1 .gt. i2) go to 200
      do 95 i = i1,i2
 95     rwork(i) = 0.0d0
      go to 200
c-----------------------------------------------------------------------
c block c.
c the next block is for the initial call only (istate = 0 or 1).
c it contains all remaining initializations, the call to aigbt
c (if istate = 1), and the calculation of the initial step size.
c the error weights in ewt are inverted after being loaded.
c-----------------------------------------------------------------------
 100  uround = mond1mach(4)
      tn = t
      if (itask .ne. 4 .and. itask .ne. 5) go to 105
      tcrit = rwork(1)
      if ((tcrit - tout)*(tout - t) .lt. 0.0d0) go to 625
      if (h0 .ne. 0.0d0 .and. (t + h0 - tcrit)*h0 .gt. 0.0d0)
     1   h0 = tcrit - t
 105  jstart = 0
      rwork(lwm) = dsqrt(uround)
      nhnil = 0
      nst = 0
      nre = 0
      nje = 0
      nslast = 0
      hu = 0.0d0
      nqu = 0
      ccmax = 0.3d0
      maxcor = 3
      msbp = 20
      mxncf = 10
c compute initial dy/dt, if necessary, and load it and initial y into yh
      lyd0 = lyh + nyh
      lp = lwm + 1
      if ( istate .eq. 1 )  go to 120
c lsoibt must compute initial dy/dt (lyd0 points to yh(*,2)). ----------
         call aigbt( res, adda, neq, t, y, rwork(lyd0),
     1               mb, nb, rwork(lp), iwork(21), ier )
         nre = nre + 1
         if (ier)  560,110,565
  110    continue
         do 115 i = 1,n
  115       rwork(i+lyh-1) = y(i)
         go to 130
c initial dy/dt has been supplied. -------------------------------------
  120    do 125 i = 1,n
            rwork(i+lyh-1) = y(i)
  125       rwork(i+lyd0-1) = ydoti(i)
c load and invert the ewt array.  (h is temporarily set to 1.0.) -------
  130 continue
      nq = 1
      h = 1.0d0
      call ewset (n, itol, rtol, atol, rwork(lyh), rwork(lewt))
      do 135 i = 1,n
        if (rwork(i+lewt-1) .le. 0.0d0) go to 621
 135    rwork(i+lewt-1) = 1.0d0/rwork(i+lewt-1)
c-----------------------------------------------------------------------
c the coding below computes the step size, h0, to be attempted on the
c first step, unless the user has supplied a value for this.
c first check that tout - t differs significantly from zero.
c a scalar tolerance quantity tol is computed, as max(rtol(i))
c if this is positive, or max(atol(i)/abs(y(i))) otherwise, adjusted
c so as to be between 100*uround and 1.0e-3.
c then the computed value h0 is given by..
c                                      neq
c   h0**2 = tol / ( w0**-2 + (1/neq) * sum ( ydot(i)/ywt(i) )**2  )
c                                       1
c where   w0      = max ( abs(t), abs(tout) ),
c         ydot(i) = i-th component of initial value of dy/dt,
c         ywt(i)  = ewt(i)/tol  (a weight for y(i)).
c the sign of h0 is inferred from the initial values of tout and t.
c-----------------------------------------------------------------------
      if (h0 .ne. 0.0d0) go to 180
      tdist = dabs(tout - t)
      w0 = dmax1(dabs(t),dabs(tout))
      if (tdist .lt. 2.0d0*uround*w0) go to 622
      tol = rtol(1)
      if (itol .le. 2) go to 145
      do 140 i = 1,n
 140    tol = dmax1(tol,rtol(i))
 145  if (tol .gt. 0.0d0) go to 160
      atoli = atol(1)
      do 150 i = 1,n
        if (itol .eq. 2 .or. itol .eq. 4) atoli = atol(i)
        ayi = dabs(y(i))
        if (ayi .ne. 0.0d0) tol = dmax1(tol,atoli/ayi)
 150    continue
 160  tol = dmax1(tol,100.0d0*uround)
      tol = dmin1(tol,0.001d0)
      sum = vnorm (n, rwork(lyd0), rwork(lewt))
      sum = 1.0d0/(tol*w0*w0) + tol*sum**2
      h0 = 1.0d0/dsqrt(sum)
      h0 = dmin1(h0,tdist)
      h0 = dsign(h0,tout-t)
c adjust h0 if necessary to meet hmax bound. ---------------------------
 180  rh = dabs(h0)*hmxi
      if (rh .gt. 1.0d0) h0 = h0/rh
c load h with h0 and scale yh(*,2) by h0. ------------------------------
      h = h0
      do 190 i = 1,n
 190    rwork(i+lyd0-1) = h0*rwork(i+lyd0-1)
      go to 270
c-----------------------------------------------------------------------
c block d.
c the next code block is for continuation calls only (istate = 2 or 3)
c and is to check stop conditions before taking a step.
c-----------------------------------------------------------------------
 200  nslast = nst
      go to (210, 250, 220, 230, 240), itask
 210  if ((tn - tout)*h .lt. 0.0d0) go to 250
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      if (iflag .ne. 0) go to 627
      t = tout
      go to 420
 220  tp = tn - hu*(1.0d0 + 100.0d0*uround)
      if ((tp - tout)*h .gt. 0.0d0) go to 623
      if ((tn - tout)*h .lt. 0.0d0) go to 250
      go to 400
 230  tcrit = rwork(1)
      if ((tn - tcrit)*h .gt. 0.0d0) go to 624
      if ((tcrit - tout)*h .lt. 0.0d0) go to 625
      if ((tn - tout)*h .lt. 0.0d0) go to 245
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      if (iflag .ne. 0) go to 627
      t = tout
      go to 420
 240  tcrit = rwork(1)
      if ((tn - tcrit)*h .gt. 0.0d0) go to 624
 245  hmx = dabs(tn) + dabs(h)
      ihit = dabs(tn - tcrit) .le. 100.0d0*uround*hmx
      if (ihit) go to 400
      tnext = tn + h*(1.0d0 + 4.0d0*uround)
      if ((tnext - tcrit)*h .le. 0.0d0) go to 250
      h = (tcrit - tn)*(1.0d0 - 4.0d0*uround)
      if (istate .eq. 2) jstart = -2
c-----------------------------------------------------------------------
c block e.
c the next block is normally executed for all calls and contains
c the call to the one-step core integrator stodi.
c
c this is a looping point for the integration steps.
c
c first check for too many steps being taken, update ewt (if not at
c start of problem), check for too much accuracy being requested, and
c check for h below the roundoff level in t.
c-----------------------------------------------------------------------
 250  continue
      if ((nst-nslast) .ge. mxstep) go to 500
      call ewset (n, itol, rtol, atol, rwork(lyh), rwork(lewt))
      do 260 i = 1,n
        if (rwork(i+lewt-1) .le. 0.0d0) go to 510
 260    rwork(i+lewt-1) = 1.0d0/rwork(i+lewt-1)
 270  tolsf = uround*vnorm (n, rwork(lyh), rwork(lewt))
      if (tolsf .le. 1.0d0) go to 280
      tolsf = tolsf*2.0d0
      if (nst .eq. 0) go to 626
      go to 520
 280  if ((tn + h) .ne. tn) go to 290
      nhnil = nhnil + 1
      if (nhnil .gt. mxhnil) go to 290
      call monwv(50hlsoibt-- warning..internal t (=r1) and h (=r2) are,
     1   50, 101, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  60h      such that in the machine, t + h = t on the next step  ,
     1   60, 101, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      (h = step size). solver will continue anyway,
     1   50, 101, 0, 0, 0, 0, 2, tn, h)
      if (nhnil .lt. mxhnil) go to 290
      call monwv(50hlsoibt-- above warning has been issued i1 times.  ,
     1   50, 102, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      it will not be issued again for this problem,
     1   50, 102, 0, 1, mxhnil, 0, 0, 0.0d0, 0.0d0)
 290  continue
c-----------------------------------------------------------------------
c     call stodi(neq,y,yh,nyh,yh1,ewt,savf,savr,acor,wm,iwm,res,
c                adda,jac,pjibt,slsbt)
c note... savf in stodi occupies the same space as ydoti in lsoibt.
c-----------------------------------------------------------------------
      call stodi (neq, y, rwork(lyh), nyh, rwork(lyh), rwork(lewt),
     1   ydoti, rwork(lsavr), rwork(lacor), rwork(lwm),
     2   iwork(liwm), res, adda, jac, pjibt, slsbt )
      kgo = 1 - kflag
      go to (300, 530, 540, 400, 550), kgo
c
c kgo = 1,success. 2,error test failure. 3,convergence failure.
c       4,res ordered return. 5,res returned error.
c-----------------------------------------------------------------------
c block f.
c the following block handles the case of a successful return from the
c core integrator (kflag = 0).  test for stop conditions.
c-----------------------------------------------------------------------
 300  init = 1
      go to (310, 400, 330, 340, 350), itask
c itask = 1.  if tout has been reached, interpolate. -------------------
 310  if ((tn - tout)*h .lt. 0.0d0) go to 250
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      t = tout
      go to 420
c itask = 3.  jump to exit if tout was reached. ------------------------
 330  if ((tn - tout)*h .ge. 0.0d0) go to 400
      go to 250
c itask = 4.  see if tout or tcrit was reached.  adjust h if necessary.
 340  if ((tn - tout)*h .lt. 0.0d0) go to 345
      call intdy (tout, 0, rwork(lyh), nyh, y, iflag)
      t = tout
      go to 420
 345  hmx = dabs(tn) + dabs(h)
      ihit = dabs(tn - tcrit) .le. 100.0d0*uround*hmx
      if (ihit) go to 400
      tnext = tn + h*(1.0d0 + 4.0d0*uround)
      if ((tnext - tcrit)*h .le. 0.0d0) go to 250
      h = (tcrit - tn)*(1.0d0 - 4.0d0*uround)
      jstart = -2
      go to 250
c itask = 5.  see if tcrit was reached and jump to exit. ---------------
 350  hmx = dabs(tn) + dabs(h)
      ihit = dabs(tn - tcrit) .le. 100.0d0*uround*hmx
c-----------------------------------------------------------------------
c block g.
c the following block handles all successful returns from lsoibt.
c if itask .ne. 1, y is loaded from yh and t is set accordingly.
c istate is set to 2, the illegal input counter is zeroed, and the
c optional outputs are loaded into the work arrays before returning.  if
c istate = 0 or 1 and tout = t, there is a return with no action taken,
c except that if this has happened repeatedly, the run is terminated.
c-----------------------------------------------------------------------
 400  do 410 i = 1,n
 410    y(i) = rwork(i+lyh-1)
      t = tn
      if (itask .ne. 4 .and. itask .ne. 5) go to 420
      if (ihit) t = tcrit
  420 istate = 2
      if ( kflag .eq. -3 )  istate = 3
      illin = 0
      rwork(11) = hu
      rwork(12) = h
      rwork(13) = tn
      iwork(11) = nst
      iwork(12) = nre
      iwork(13) = nje
      iwork(14) = nqu
      iwork(15) = nq
      return
c
 430  ntrep = ntrep + 1
      if (ntrep .lt. 5) return
      call monwv(
     1  60hlsoibt-- repeated calls with istate= 0 or 1 and tout= t(=r1),
     1   60, 301, 0, 0, 0, 0, 1, t, 0.0d0)
      go to 800
c-----------------------------------------------------------------------
c block h.
c the following block handles all unsuccessful returns other than
c those for illegal input.  first the error message routine is called.
c if there was an error test or convergence test failure, imxer is set.
c then y is loaded from yh, t is set to tn, and the illegal input
c counter illin is set to 0.  the optional outputs are loaded into
c the work arrays before returning.
c-----------------------------------------------------------------------
c the maximum number of steps was taken before reaching tout. ----------
 500  call monwv(50hlsoibt-- at current t (=r1), mxstep (=i1) steps   ,
     1   50, 201, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      taken on this call before reaching tout     ,
     1   50, 201, 0, 1, mxstep, 0, 1, tn, 0.0d0)
      istate = -1
      go to 580
c ewt(i) .le. 0.0 for some i (not at start of problem). ----------------
 510  ewti = rwork(lewt+i-1)
      call monwv(50hlsoibt-- at t (=r1), ewt(i1) has become r2 .le. 0.,
     1   50, 202, 0, 1, i, 0, 2, tn, ewti)
      istate = -6
      go to 590
c too much accuracy requested for machine precision. -------------------
 520  call monwv(50hlsoibt-- at t (=r1), too much accuracy requested  ,
     1   50, 203, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      for precision of machine..  see tolsf (=r2) ,
     1   50, 203, 0, 0, 0, 0, 2, tn, tolsf)
      rwork(14) = tolsf
      istate = -2
      go to 590
c kflag = -1.  error test failed repeatedly or with abs(h) = hmin. -----
 530  call monwv(50hlsoibt-- at t(=r1) and step size h(=r2), the error,
     1   50, 204, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      test failed repeatedly or with abs(h) = hmin,
     1   50, 204, 0, 0, 0, 0, 2, tn, h)
      istate = -4
      go to 570
c kflag = -2.  convergence failed repeatedly or with abs(h) = hmin. ----
 540  call monwv(50hlsoibt-- at t (=r1) and step size h (=r2), the    ,
     1   50, 205, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      corrector convergence failed repeatedly     ,
     1   50, 205, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(30h      or with abs(h) = hmin   ,
     1   30, 205, 0, 0, 0, 0, 2, tn, h)
      istate = -5
      go to 570
c ires = 3 returned by res, despite retries by stodi. ------------------
 550  call monwv(50hlsoibt-- at t (=r1) residual routine returned     ,
     1   50, 206, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(40h      error ires = 3 repeatedly         ,
     1   40, 206, 0, 0, 0, 0, 1, tn, 0.0d0)
      istate = -7
      go to 590
c aigbt failed because a diagonal block of a-matrix was singular. ------
 560  ier = -ier
      call monwv(
     1  60hlsoibt-- attempt to initialize dy/dt failed.. matrix a has a,
     1   60, 207, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      singular diagonal block, block no. = (i1)   ,
     2   50, 207, 0, 1, ier, 0, 0, 0.0d0, 0.0d0)
      istate = -8
      return
c aigbt failed because res set ires to 2 or 3. -------------------------
 565  call monwv(50hlsoibt-- attempt to initialize dy/dt failed       ,
     1   50, 208, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      because residual routine set its error flag ,
     1   50, 208, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(20h      to ires = (i1),
     1   20, 208, 0, 1, ier, 0, 0, 0.0d0, 0.0d0)
      istate = -8
      return
c compute imxer if relevant. -------------------------------------------
 570  big = 0.0d0
      imxer = 1
      do 575 i = 1,n
        size = dabs(rwork(i+lacor-1)*rwork(i+lewt-1))
        if (big .ge. size) go to 575
        big = size
        imxer = i
 575    continue
      iwork(16) = imxer
c compute residual if relevant. ----------------------------------------
 580  lyd0 = lyh + nyh
      do 585 i = 1,n
         rwork(i+lsavr-1) = rwork(i+lyd0-1)/h
 585     y(i) = rwork(i+lyh-1)
      ires = 1
      call res ( neq, tn, y, rwork(lsavr), ydoti, ires )
      nre = nre + 1
      if ( ires .le. 1 )  go to 595
      call monwv(50hlsoibt-- residual routine set its flag ires       ,
     1   50, 210, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(50h      to (i1) when called for final output.       ,
     1   50, 210, 0, 1, ires, 0, 0, 0.0d0, 0.0d0)
      go to 595
c set y vector, t, illin, and optional outputs. ------------------------
 590  do 592 i = 1,n
 592    y(i) = rwork(i+lyh-1)
 595  t = tn
      illin = 0
      rwork(11) = hu
      rwork(12) = h
      rwork(13) = tn
      iwork(11) = nst
      iwork(12) = nre
      iwork(13) = nje
      iwork(14) = nqu
      iwork(15) = nq
      return
c-----------------------------------------------------------------------
c block i.
c the following block handles all error returns due to illegal input
c (istate = -3), as detected before calling the core integrator.
c first the error message routine is called.  then if there have been
c 5 consecutive such returns just before this call to the solver,
c the run is halted.
c-----------------------------------------------------------------------
 601  call monwv(30hlsoibt-- istate (=i1) illegal ,
     1   30, 1, 0, 1, istate, 0, 0, 0.0d0, 0.0d0)
      go to 700
 602  call monwv(30hlsoibt-- itask (=i1) illegal  ,
     1   30, 2, 0, 1, itask, 0, 0, 0.0d0, 0.0d0)
      go to 700
 603  call monwv(50hlsoibt-- istate .gt. 1 but lsoibt not initialized ,
     1   50, 3, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      go to 700
 604  call monwv(30hlsoibt-- neq (=i1) .lt. 1     ,
     1   30, 4, 0, 1, neq(1), 0, 0, 0.0d0, 0.0d0)
      go to 700
 605  call monwv(50hlsoibt-- istate = 3 and neq increased (i1 to i2)  ,
     1   50, 5, 0, 2, n, neq(1), 0, 0.0d0, 0.0d0)
      go to 700
 606  call monwv(30hlsoibt-- itol (=i1) illegal   ,
     1   30, 6, 0, 1, itol, 0, 0, 0.0d0, 0.0d0)
      go to 700
 607  call monwv(30hlsoibt-- iopt (=i1) illegal   ,
     1   30, 7, 0, 1, iopt, 0, 0, 0.0d0, 0.0d0)
      go to 700
 608  call monwv(30hlsoibt-- mf (=i1) illegal     ,
     1   30, 8, 0, 1, mf, 0, 0, 0.0d0, 0.0d0)
      go to 700
 609  call monwv(40hlsoibt-- mb (=i1) or nb (=i2) illegal   ,
     1   50, 9, 0, 2, mb, nb, 0, 0.0d0, 0.0d0)
      go to 700
 610  call monwv(40hlsoibt-- nb(=i1) illegal.. .lt. 4       ,
     1   50, 10, 0, 1, nb, 0, 0, 0.0d0, 0.0d0)
      go to 700
 611  call monwv(30hlsoibt-- maxord (=i1) .lt. 0  ,
     1   30, 11, 0, 1, maxord, 0, 0, 0.0d0, 0.0d0)
      go to 700
 612  call monwv(30hlsoibt-- mxstep (=i1) .lt. 0  ,
     1   30, 12, 0, 1, mxstep, 0, 0, 0.0d0, 0.0d0)
      go to 700
 613  call monwv(30hlsoibt-- mxhnil (=i1) .lt. 0  ,
     1   30, 13, 0, 1, mxhnil, 0, 0, 0.0d0, 0.0d0)
      go to 700
 614  call monwv(40hlsoibt-- tout (=r1) behind t (=r2)      ,
     1   40, 14, 0, 0, 0, 0, 2, tout, t)
      call monwv(50h      integration direction is given by h0 (=r1)  ,
     1   50, 14, 0, 0, 0, 0, 1, h0, 0.0d0)
      go to 700
 615  call monwv(30hlsoibt-- hmax (=r1) .lt. 0.0  ,
     1   30, 15, 0, 0, 0, 0, 1, hmax, 0.0d0)
      go to 700
 616  call monwv(30hlsoibt-- hmin (=r1) .lt. 0.0  ,
     1   30, 16, 0, 0, 0, 0, 1, hmin, 0.0d0)
      go to 700
 617  call monwv(
     1  60hlsoibt-- rwork length needed, lenrw (=i1), exceeds lrw (=i2),
     1   60, 17, 0, 2, lenrw, lrw, 0, 0.0d0, 0.0d0)
      go to 700
 618  call monwv(
     1  60hlsoibt-- iwork length needed, leniw (=i1), exceeds liw (=i2),
     1   60, 18, 0, 2, leniw, liw, 0, 0.0d0, 0.0d0)
      go to 700
 619  call monwv(40hlsoibt-- rtol(=i1) is r1 .lt. 0.0       ,
     1   40, 19, 0, 1, i, 0, 1, rtoli, 0.0d0)
      go to 700
 620  call monwv(40hlsoibt-- atol(=i1) is r1 .lt. 0.0       ,
     1   40, 20, 0, 1, i, 0, 1, atoli, 0.0d0)
      go to 700
 621  ewti = rwork(lewt+i-1)
      call monwv(40hlsoibt-- ewt(=i1) is r1 .le. 0.0        ,
     1   40, 21, 0, 1, i, 0, 1, ewti, 0.0d0)
      go to 700
 622  call monwv(
     1  60hlsoibt-- tout (=r1) too close to t(=r2) to start integration,
     1   60, 22, 0, 0, 0, 0, 2, tout, t)
      go to 700
 623  call monwv(
     1  60hlsoibt-- itask = i1 and tout (=r1) behind tcur - hu (= r2)  ,
     1   60, 23, 0, 1, itask, 0, 2, tout, tp)
      go to 700
 624  call monwv(
     1  60hlsoibt-- itask = 4 or 5 and tcrit (=r1) behind tcur (=r2)   ,
     1   60, 24, 0, 0, 0, 0, 2, tcrit, tn)
      go to 700
 625  call monwv(
     1  60hlsoibt-- itask = 4 or 5 and tcrit (=r1) behind tout (=r2)   ,
     1   60, 25, 0, 0, 0, 0, 2, tcrit, tout)
      go to 700
 626  call monwv(50hlsoibt-- at start of problem, too much accuracy   ,
     1   50, 26, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
      call monwv(
     1  60h      requested for precision of machine..  see tolsf (=r1) ,
     1   60, 26, 0, 0, 0, 0, 1, tolsf, 0.0d0)
      rwork(14) = tolsf
      go to 700
 627  call monwv(50hlsoibt-- trouble from intdy. itask = i1, tout = r1,
     1   50, 27, 0, 1, itask, 0, 1, tout, 0.0d0)
c
 700  if (illin .eq. 5) go to 710
      illin = illin + 1
      istate = -3
      return
 710  call monwv(50hlsoibt-- repeated occurrences of illegal input    ,
     1   50, 302, 0, 0, 0, 0, 0, 0.0d0, 0.0d0)
c
 800  call monwv(50hlsoibt-- run aborted.. apparent infinite loop     ,
     1   50, 303, 2, 0, 0, 0, 0, 0.0d0, 0.0d0)
      return
c----------------------- end of subroutine lsoibt ----------------------
      end
      subroutine aigbt (res, adda, neq, t, y, ydot,
     1                   mb, nb, pw, ipvt, ier )
clll. optimize
      external res, adda
      integer neq, mb, nb, ipvt, ier
      integer i, lenpw, lblox, lpb, lpc
      double precision t, y, ydot, pw
      dimension y(1), ydot(1), pw(1), ipvt(1), neq(1)
c-----------------------------------------------------------------------
c this subroutine computes the initial value
c of the vector ydot satisfying
c     a * ydot = g(t,y)
c when a is nonsingular.  it is called by lsoibt for
c initialization only, when istate = 0 .
c aigbt returns an error flag ier..
c   ier  =  0  means aigbt was successful.
c   ier .ge. 2 means res returned an error flag ires = ier.
c   ier .lt. 0 means the a matrix was found to have a singular
c              diagonal block (hence ydot could not be solved for).
c-----------------------------------------------------------------------
      lblox = mb*mb*nb
      lpb = 1 + lblox
      lpc = lpb + lblox
      lenpw = 3*lblox
      do 10 i = 1,lenpw
 10     pw(i) = 0.0d0
      ier = 1
      call res (neq, t, y, pw, ydot, ier)
      if (ier .gt. 1) return
      call adda (neq, t, y, mb, nb, pw(1), pw(lpb), pw(lpc) )
      call decbt (mb, nb, pw, pw(lpb), pw(lpc), ipvt, ier)
      if (ier .eq. 0) go to 20
      ier = -ier
      return
 20   call solbt (mb, nb, pw, pw(lpb), pw(lpc), ydot, ipvt)
      return
c-------------------- end of subroutine aigbt --------------------------
      end
      subroutine pjibt (neq, y, yh, nyh, ewt, rtem, savr, s, wm, iwm,
     1   res, jac, adda)
clll. optimize
      external res, jac, adda
      integer neq, nyh, iwm
      integer iownd, iowns,
     1   icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     2   maxord, maxcor, msbp, mxncf, n, nq, nst, nre, nje, nqu
      integer i, ier, iia, iib, iic, ipa, ipb, ipc, ires, j, j1, j2,
     1   k, k1, lenp, lblox, lpb, lpc, mb, mbsq, mwid, nb
      double precision y, yh, ewt, rtem, savr, s, wm
      double precision rowns,
     1   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround
      double precision con, fac, hl0, r, srur
      dimension neq(1), y(1), yh(nyh,1), ewt(1), rtem(1),
     1   s(1), savr(1), wm(1), iwm(1)
      common /ls0001/ rowns(209),
     2   ccmax, el0, h, hmin, hmxi, hu, rc, tn, uround,
     3   iownd(14), iowns(6),
     4   icf, ierpj, iersl, jcur, jstart, kflag, l, meth, miter,
     5   maxord, maxcor, msbp, mxncf, n, nq, nst, nre, nje, nqu
c-----------------------------------------------------------------------
c pjibt is called by stodi to compute and process the matrix
c p = a - h*el(1)*j , where j is an approximation to the jacobian dr/dy,
c and r = g(t,y) - a(t,y)*s.  here j is computed by the user-supplied
c routine jac if miter = 1, or by finite differencing if miter = 2.
c j is stored in wm, rescaled, and adda is called to generate p.
c p is then subjected to lu decomposition by decbt in preparation
c for later solution of linear systems with p as coefficient matrix.
c
c in addition to variables described previously, communication
c with pjibt uses the following..
c y     = array containing predicted values on entry.
c rtem  = work array of length n (acor in stodi).
c savr  = array used for output only.  on output it contains the
c         residual evaluated at current values of t and y.
c s     = array containing predicted values of dy/dt (savf in stodi).
c wm    = real work space for matrices.  on output it contains the
c         lu decomposition of p.
c         storage of matrix elements starts at wm(3).
c         wm also contains the following matrix-related data..
c         wm(1) = dsqrt(uround), used in numerical jacobian increments.
c iwm   = integer work space containing pivot information, starting at
c         iwm(21).  iwm also contains block structure parameters
c         mb = iwm(1) and nb = iwm(2).
c el0   = el(1) (input).
c ierpj = output error flag.
c         = 0 if no trouble occurred,
c         = 1 if the p matrix was found to be unfactorable,
c         = ires (= 2 or 3) if res returned ires = 2 or 3.
c jcur  = output flag = 1 to indicate that the jacobian matrix
c         (or approximation) is now current.
c this routine also uses the common variables el0, h, tn, uround,
c miter, n, nre, and nje.
c-----------------------------------------------------------------------
      nje = nje + 1
      hl0 = h*el0
      ierpj = 0
      jcur = 1
      mb = iwm(1)
      nb = iwm(2)
      mbsq = mb*mb
      lblox = mbsq*nb
      lpb = 3 + lblox
      lpc = lpb + lblox
      lenp = 3*lblox
      go to (100, 200), miter
c if miter = 1, call res, then jac, and multiply by scalar. ------------
 100  ires = 1
      call res (neq, tn, y, s, savr, ires)
      nre = nre + 1
      if (ires .gt. 1) go to 600
      do 110 i = 1,lenp
 110    wm(i+2) = 0.0d0
      call jac (neq, tn, y, s, mb, nb, wm(3), wm(lpb), wm(lpc))
      con = -hl0
      do 120 i = 1,lenp
 120    wm(i+2) = wm(i+2)*con
      go to 260
c
c if miter = 2, make 3*mb + 1 calls to res to approximate j. -----------
 200  continue
      ires = -1
      call res (neq, tn, y, s, savr, ires)
      nre = nre + 1
      if (ires .gt. 1) go to 600
      mwid = 3*mb
      srur = wm(1)
      do 205 i = 1,lenp
 205    wm(2+i) = 0.0d0
      do 250 k = 1,3
        do 240 j = 1,mb
c         increment y(i) for group of column indices, and call res. ----
          j1 = j+(k-1)*mb
          do 210 i = j1,n,mwid
            r = dmax1(srur*dabs(y(i)),0.01d0/ewt(i))
            y(i) = y(i) + r
 210      continue
          call res (neq, tn, y, s, rtem, ires)
          nre = nre + 1
          if (ires .gt. 1) go to 600
          do 215 i = 1,n
 215        rtem(i) = rtem(i) - savr(i)
          k1 = k
          do 230 i = j1,n,mwid
c           get jacobian elements in column i (block-column k1). -------
            y(i) = yh(i,1)
            r = dmax1(srur*dabs(y(i)),0.01d0/ewt(i))
            fac = -hl0/r
c           compute and load elements pa(*,j,k1). ----------------------
            iia = i - j
            ipa = 2 + (j-1)*mb + (k1-1)*mbsq
            do 221 j2 = 1,mb
 221          wm(ipa+j2) = rtem(iia+j2)*fac
            if (k1 .le. 1) go to 223
c           compute and load elements pb(*,j,k1-1). --------------------
            iib = iia - mb
            ipb = ipa + lblox - mbsq
            do 222 j2 = 1,mb
 222          wm(ipb+j2) = rtem(iib+j2)*fac
 223        continue
            if (k1 .ge. nb) go to 225
c           compute and load elements pc(*,j,k1+1). --------------------
            iic = iia + mb
            ipc = ipa + 2*lblox + mbsq
            do 224 j2 = 1,mb
 224          wm(ipc+j2) = rtem(iic+j2)*fac
 225        continue
            if (k1 .ne. 3) go to 227
c           compute and load elements pc(*,j,1). -----------------------
            ipc = ipa - 2*mbsq + 2*lblox
            do 226 j2 = 1,mb
 226          wm(ipc+j2) = rtem(j2)*fac
 227        continue
            if (k1 .ne. nb-2) go to 229
c           compute and load elements pb(*,j,nb). ----------------------
            iib = n - mb
            ipb = ipa + 2*mbsq + lblox
            do 228 j2 = 1,mb
 228          wm(ipb+j2) = rtem(iib+j2)*fac
 229      k1 = k1 + 3
 230      continue
 240    continue
 250  continue
c res call for first corrector iteration. ------------------------------
      ires = 1
      call res (neq, tn, y, s, savr, ires)
      nre = nre + 1
      if (ires .gt. 1) go to 600
c add matrix a. --------------------------------------------------------
 260  continue
      call adda (neq, tn, y, mb, nb, wm(3), wm(lpb), wm(lpc))
c do lu decomposition on p. --------------------------------------------
      call decbt (mb, nb, wm(3), wm(lpb), wm(lpc), iwm(21), ier)
      if (ier .ne. 0) ierpj = 1
      return
c error return for ires = 2 or ires = 3 return from res. ---------------
 600  ierpj = ires
      return
c----------------------- end of subroutine pjibt -----------------------
      end
      subroutine slsbt (wm, iwm, x, tem)
clll. optimize
      integer iwm
      integer lblox, lpb, lpc, mb, nb
      double precision wm, x, tem
      dimension wm(1), iwm(1), x(1), tem(1)
c-----------------------------------------------------------------------
c this routine acts as an interface between the core integrator
c routine and the solbt routine for the solution of the linear system
c arising from chord iteration.
c communication with slsbt uses the following variables..
c wm    = real work space containing the lu decomposition,
c         starting at wm(3).
c iwm   = integer work space containing pivot information, starting at
c         iwm(21).  iwm also contains block structure parameters
c         mb = iwm(1) and nb = iwm(2).
c x     = the right-hand side vector on input, and the solution vector
c         on output, of length n.
c tem   = vector of work space of length n, not used in this version.
c-----------------------------------------------------------------------
      mb = iwm(1)
      nb = iwm(2)
      lblox = mb*mb*nb
      lpb = 3 + lblox
      lpc = lpb + lblox
      call solbt (mb, nb, wm(3), wm(lpb), wm(lpc), x, iwm(21))
      return
c----------------------- end of subroutine slsbt -----------------------
      end
      subroutine decbt (m, n, a, b, c, ip, ier)
      integer m, n, ip(m,n), ier
      double precision a(m,m,n), b(m,m,n), c(m,m,n)
c-----------------------------------------------------------------------
c the following line is for optimized compilation on llnl compilers.
clll. optimize
c-----------------------------------------------------------------------
c block-tridiagonal matrix decomposition routine.
c written by a. c. hindmarsh.
c latest revision.. november 10, 1983 (ach)
c reference.. ucid-30150
c             solution of block-tridiagonal systems of linear
c             algebraic equations
c             a.c. hindmarsh
c             february 1977
c the input matrix contains three blocks of elements in each block-row,
c including blocks in the (1,3) and (n,n-2) block positions.
c decbt uses block gauss elimination and subroutines mondgefa and mondgesl
c for solution of blocks.  partial pivoting is done within
c block-rows only.
c
c note.. this version uses linpack routines mondgefa/mondgesl instead of
c of dec/sol for solution of blocks, and it uses the bla routine monddot
c for dot product calculations.
c
c input..
c     m = order of each block.
c     n = number of blocks in each direction of the matrix.
c         n must be 4 or more.  the complete matrix has order m*n.
c     a = m by m by n array containing diagonal blocks.
c         a(i,j,k) contains the (i,j) element of the k-th block.
c     b = m by m by n array containing the super-diagonal blocks
c         (in b(*,*,k) for k = 1,...,n-1) and the block in the (n,n-2)
c         block position (in b(*,*,n)).
c     c = m by m by n array containing the subdiagonal blocks
c         (in c(*,*,k) for k = 2,3,...,n) and the block in the
c         (1,3) block position (in c(*,*,1)).
c    ip = integer array of length m*n for working storage.
c output..
c a,b,c = m by m by n arrays containing the block lu decomposition
c         of the input matrix.
c    ip = m by n array of pivot information.  ip(*,k) contains
c         information for the k-th digonal block.
c   ier = 0  if no trouble occurred, or
c       = -1 if the input value of m or n was illegal, or
c       = k  if a singular matrix was found in the k-th diagonal block.
c use solbt to solve the associated linear system.
c
c external routines required.. mondgefa and mondgesl (from linpack) and
c monddot (from the blas, or basic linear algebra package).
c-----------------------------------------------------------------------
      integer nm1, nm2, km1, i, j, k
      double precision dp, monddot
      if (m .lt. 1 .or. n .lt. 4) go to 210
      nm1 = n - 1
      nm2 = n - 2
c process the first block-row. -----------------------------------------
      call mondgefa (a, m, m, ip, ier)
      k = 1
      if (ier .ne. 0) go to 200
      do 10 j = 1,m
        call mondgesl (a, m, m, ip, b(1,j,1), 0)
        call mondgesl (a, m, m, ip, c(1,j,1), 0)
 10     continue
c adjust b(*,*,2). -----------------------------------------------------
      do 40 j = 1,m
        do 30 i = 1,m
          dp = monddot (m, c(i,1,2), m, c(1,j,1), 1)
          b(i,j,2) = b(i,j,2) - dp
 30       continue
 40     continue
c main loop.  process block-rows 2 to n-1. -----------------------------
      do 100 k = 2,nm1
        km1 = k - 1
        do 70 j = 1,m
          do 60 i = 1,m
            dp = monddot (m, c(i,1,k), m, b(1,j,km1), 1)
            a(i,j,k) = a(i,j,k) - dp
 60         continue
 70       continue
        call mondgefa (a(1,1,k), m, m, ip(1,k), ier)
        if (ier .ne. 0) go to 200
        do 80 j = 1,m
 80       call mondgesl (a(1,1,k), m, m, ip(1,k), b(1,j,k), 0)
 100    continue
c process last block-row and return. -----------------------------------
      do 130 j = 1,m
        do 120 i = 1,m
          dp = monddot (m, b(i,1,n), m, b(1,j,nm2), 1)
          c(i,j,n) = c(i,j,n) - dp
 120      continue
 130    continue
      do 160 j = 1,m
        do 150 i = 1,m
          dp = monddot (m, c(i,1,n), m, b(1,j,nm1), 1)
          a(i,j,n) = a(i,j,n) - dp
 150      continue
 160    continue
      call mondgefa (a(1,1,n), m, m, ip(1,n), ier)
      k = n
      if (ier .ne. 0) go to 200
      return
c error returns. -------------------------------------------------------
 200  ier = k
      return
 210  ier = -1
      return
c-----------------------  end of subroutine decbt  ---------------------
      end
      subroutine solbt (m, n, a, b, c, y, ip)
      integer m, n, ip(m,n)
      double precision a(m,m,n), b(m,m,n), c(m,m,n), y(m,n)
clll. optimize
c-----------------------------------------------------------------------
c solution of block-tridiagonal linear system.
c coefficient matrix must have been previously processed by decbt.
c m, n, a, b, c, and ip  must not have been changed since call to decbt.
c written by a. c. hindmarsh.
c input..
c     m = order of each block.
c     n = number of blocks in each direction of matrix.
c a,b,c = m by m by n arrays containing block lu decomposition
c         of coefficient matrix from decbt.
c    ip = m by n integer array of pivot information from decbt.
c     y = array of length m*n containg the right-hand side vector
c         (treated as an m by n array here).
c output..
c     y = solution vector, of length m*n.
c
c external routines required.. mondgesl (linpack) and monddot (blas).
c-----------------------------------------------------------------------
c
      integer nm1, nm2, i, k, kb, km1, kp1
      double precision dp, monddot
      nm1 = n - 1
      nm2 = n - 2
c forward solution sweep. ----------------------------------------------
      call mondgesl (a, m, m, ip, y, 0)
      do 30 k = 2,nm1
        km1 = k - 1
        do 20 i = 1,m
          dp = monddot (m, c(i,1,k), m, y(1,km1), 1)
          y(i,k) = y(i,k) - dp
 20       continue
        call mondgesl (a(1,1,k), m, m, ip(1,k), y(1,k), 0)
 30     continue
      do 50 i = 1,m
        dp = monddot (m, c(i,1,n), m, y(1,nm1), 1)
     1     + monddot (m, b(i,1,n), m, y(1,nm2), 1)
        y(i,n) = y(i,n) - dp
 50     continue
      call mondgesl (a(1,1,n), m, m, ip(1,n), y(1,n), 0)
c backward solution sweep. ---------------------------------------------
      do 80 kb = 1,nm1
        k = n - kb
        kp1 = k + 1
        do 70 i = 1,m
          dp = monddot (m, b(i,1,k), m, y(1,kp1), 1)
          y(i,k) = y(i,k) - dp
 70       continue
 80     continue
      do 100 i = 1,m
        dp = monddot (m, c(i,1,1), m, y(1,3), 1)
        y(i,1) = y(i,1) - dp
 100    continue
      return
c-----------------------  end of subroutine solbt  ---------------------
      end
      double precision function mond1mach (idum)
      integer idum
c-----------------------------------------------------------------------
c this routine computes the unit roundoff of the machine in double
c precision.  this is defined as the smallest positive machine number
c u such that  1.0d0 + u .ne. 1.0d0 (in double precision).
c-----------------------------------------------------------------------
      double precision u, comp
      u = 1.0d0
 10   u = u*0.5d0
      comp = 1.0d0 + u
      if (comp .ne. 1.0d0) go to 10
      mond1mach = u*2.0d0
      return
c----------------------- end of function mond1mach ------------------------
      end
      subroutine monwv (msg, nmes, nerr, level, ni, i1, i2, nr, r1, r2)
      integer msg, nmes, nerr, level, ni, i1, i2, nr,
     1   i, lun, lunit, mesflg, ncpw, nch, nwds
      double precision r1, r2
      dimension msg(nmes)
c-----------------------------------------------------------------------
c subroutines monwv, xsetf, and xsetun, as given here, constitute
c a simplified version of the slatec error handling package.
c written by a. c. hindmarsh at llnl.  version of march 30, 1987.
c this version is in double precision.
c
c all arguments are input arguments.
c
c msg    = the message (hollerith literal or integer array).
c nmes   = the length of msg (number of characters).
c nerr   = the error number (not used).
c level  = the error level..
c          0 or 1 means recoverable (control returns to caller).
c          2 means fatal (run is aborted--see note below).
c ni     = number of integers (0, 1, or 2) to be printed with message.
c i1,i2  = integers to be printed, depending on ni.
c nr     = number of reals (0, 1, or 2) to be printed with message.
c r1,r2  = reals to be printed, depending on nr.
c
c note..  this routine is machine-dependent and specialized for use
c in limited context, in the following ways..
c 1. the number of hollerith characters stored per word, denoted
c    by ncpw below, is a data-loaded constant.
c 2. the value of nmes is assumed to be at most 60.
c    (multi-line messages are generated by repeated calls.)
c 3. if level = 2, control passes to the statement   stop
c    to abort the run.  this statement may be machine-dependent.
c 4. r1 and r2 are assumed to be in double precision and are printed
c    in d21.13 format.
c 5. the common block /eh0001/ below is data-loaded (a machine-
c    dependent feature) with default values.
c    this block is needed for proper retention of parameters used by
c    this routine which the user can reset by calling xsetf or xsetun.
c    the variables in this block are as follows..
c       mesflg = print control flag..
c                1 means print all messages (the default).
c                0 means no printing.
c       lunit  = logical unit number for messages.
c                the default is 6 (machine-dependent).
c-----------------------------------------------------------------------
c the following are instructions for installing this routine
c in different machine environments.
c
c to change the default output unit, change the data statement
c in the block data subprogram below.
c
c for a different number of characters per word, change the
c data statement setting ncpw below, and format 10.  alternatives for
c various computers are shown in comment cards.
c
c for a different run-abort command, change the statement following
c statement 100 at the end.
c-----------------------------------------------------------------------
      common /eh0001/ mesflg, lunit
c-----------------------------------------------------------------------
c the following data-loaded value of ncpw is valid for the cdc-6600
c and cdc-7600 computers.
c     data ncpw/10/
c the following is valid for the cray-1 computer.
c     data ncpw/8/
c the following is valid for the burroughs 6700 and 7800 computers.
c     data ncpw/6/
c the following is valid for the pdp-10 computer.
c     data ncpw/5/
c the following is valid for the vax computer with 4 bytes per integer,
c and for the ibm-360, ibm-370, ibm-303x, and ibm-43xx computers.
      data ncpw/4/
c the following is valid for the pdp-11, or vax with 2-byte integers.
c     data ncpw/2/
c-----------------------------------------------------------------------
      if (mesflg .eq. 0) go to 100
c get logical unit number. ---------------------------------------------
      lun = lunit
c get number of words in message. --------------------------------------
      nch = min0(nmes,60)
      nwds = nch/ncpw
      if (nch .ne. nwds*ncpw) nwds = nwds + 1
c write the message. ---------------------------------------------------
      write (lun, 10) (msg(i),i=1,nwds)
c-----------------------------------------------------------------------
c the following format statement is to have the form
c 10  format(1x,mmann)
c where nn = ncpw and mm is the smallest integer .ge. 60/ncpw.
c the following is valid for ncpw = 10.
c 10  format(1x,6a10)
c the following is valid for ncpw = 8.
c 10  format(1x,8a8)
c the following is valid for ncpw = 6.
c 10  format(1x,10a6)
c the following is valid for ncpw = 5.
c 10  format(1x,12a5)
c the following is valid for ncpw = 4.
  10  format(1x,15a4)
c the following is valid for ncpw = 2.
c 10  format(1x,30a2)
c-----------------------------------------------------------------------
      if (ni .eq. 1) write (lun, 20) i1
 20   format(6x,23hin above message,  i1 =,i10)
      if (ni .eq. 2) write (lun, 30) i1,i2
 30   format(6x,23hin above message,  i1 =,i10,3x,4hi2 =,i10)
      if (nr .eq. 1) write (lun, 40) r1
 40   format(6x,23hin above message,  r1 =,d21.13)
      if (nr .eq. 2) write (lun, 50) r1,r2
 50   format(6x,15hin above,  r1 =,d21.13,3x,4hr2 =,d21.13)
c abort the run if level = 2. ------------------------------------------
 100  if (level .ne. 2) return
      stop
c----------------------- end of subroutine monwv ----------------------
      end
      subroutine xsetf (mflag)
c
c this routine resets the print control flag mflag.
c
      integer mflag, mesflg, lunit
      common /eh0001/ mesflg, lunit
c
      if (mflag .eq. 0 .or. mflag .eq. 1) mesflg = mflag
      return
c----------------------- end of subroutine xsetf -----------------------
      end
      subroutine xsetun (lun)
c
c this routine resets the logical unit number for messages.
c
      integer lun, mesflg, lunit
      common /eh0001/ mesflg, lunit
c
      if (lun .gt. 0) lunit = lun
      return
c----------------------- end of subroutine xsetun ----------------------
      end
      subroutine mondgefa(a,lda,n,ipvt,info)
      integer lda,n,ipvt(1),info
      double precision a(lda,1)
c
c     mondgefa factors a double precision matrix by gaussian elimination.
c
c     mondgefa is usually called by dgeco, but it can be called
c     directly with a saving in time if  rcond  is not needed.
c     (time for dgeco) = (1 + 9/n)*(time for mondgefa) .
c
c     on entry
c
c        a       double precision(lda, n)
c                the matrix to be factored.
c
c        lda     integer
c                the leading dimension of the array  a .
c
c        n       integer
c                the order of the matrix  a .
c
c     on return
c
c        a       an upper triangular matrix and the multipliers
c                which were used to obtain it.
c                the factorization can be written  a = l*u  where
c                l  is a product of permutation and unit lower
c                triangular matrices and  u  is upper triangular.
c
c        ipvt    integer(n)
c                an integer vector of pivot indices.
c
c        info    integer
c                = 0  normal value.
c                = k  if  u(k,k) .eq. 0.0 .  this is not an error
c                     condition for this subroutine, but it does
c                     indicate that mondgesl or dgedi will divide by zero
c                     if called.  use  rcond  in dgeco for a reliable
c                     indication of singularity.
c
c     linpack. this version dated 08/14/78 .
c     cleve moler, university of new mexico, argonne national lab.
c
c     subroutines and functions
c
c     blas mondaxpy,mondscal,monidamax
c
c     internal variables
c
      double precision t
      integer monidamax,j,k,kp1,l,nm1
c
c
c     gaussian elimination with partial pivoting
c
      info = 0
      nm1 = n - 1
      if (nm1 .lt. 1) go to 70
      do 60 k = 1, nm1
         kp1 = k + 1
c
c        find l = pivot index
c
         l = monidamax(n-k+1,a(k,k),1) + k - 1
         ipvt(k) = l
c
c        zero pivot implies this column already triangularized
c
         if (a(l,k) .eq. 0.0d0) go to 40
c
c           interchange if necessary
c
            if (l .eq. k) go to 10
               t = a(l,k)
               a(l,k) = a(k,k)
               a(k,k) = t
   10       continue
c
c           compute multipliers
c
            t = -1.0d0/a(k,k)
            call mondscal(n-k,t,a(k+1,k),1)
c
c           row elimination with column indexing
c
            do 30 j = kp1, n
               t = a(l,j)
               if (l .eq. k) go to 20
                  a(l,j) = a(k,j)
                  a(k,j) = t
   20          continue
               call mondaxpy(n-k,t,a(k+1,k),1,a(k+1,j),1)
   30       continue
         go to 50
   40    continue
            info = k
   50    continue
   60 continue
   70 continue
      ipvt(n) = n
      if (a(n,n) .eq. 0.0d0) info = n
      return
      end
      subroutine mondgesl(a,lda,n,ipvt,b,job)
      integer lda,n,ipvt(1),job
      double precision a(lda,1),b(1)
c
c     mondgesl solves the double precision system
c     a * x = b  or  trans(a) * x = b
c     using the factors computed by dgeco or mondgefa.
c
c     on entry
c
c        a       double precision(lda, n)
c                the output from dgeco or mondgefa.
c
c        lda     integer
c                the leading dimension of the array  a .
c
c        n       integer
c                the order of the matrix  a .
c
c        ipvt    integer(n)
c                the pivot vector from dgeco or mondgefa.
c
c        b       double precision(n)
c                the right hand side vector.
c
c        job     integer
c                = 0         to solve  a*x = b ,
c                = nonzero   to solve  trans(a)*x = b  where
c                            trans(a)  is the transpose.
c
c     on return
c
c        b       the solution vector  x .
c
c     error condition
c
c        a division by zero will occur if the input factor contains a
c        zero on the diagonal.  technically this indicates singularity
c        but it is often caused by improper arguments or improper
c        setting of lda .  it will not occur if the subroutines are
c        called correctly and if dgeco has set rcond .gt. 0.0
c        or mondgefa has set info .eq. 0 .
c
c     to compute  inverse(a) * c  where  c  is a matrix
c     with  p  columns
c           call dgeco(a,lda,n,ipvt,rcond,z)
c           if (rcond is too small) go to ...
c           do 10 j = 1, p
c              call mondgesl(a,lda,n,ipvt,c(1,j),0)
c        10 continue
c
c     linpack. this version dated 08/14/78 .
c     cleve moler, university of new mexico, argonne national lab.
c
c     subroutines and functions
c
c     blas mondaxpy,monddot
c
c     internal variables
c
      double precision monddot,t
      integer k,kb,l,nm1
c
      nm1 = n - 1
      if (job .ne. 0) go to 50
c
c        job = 0 , solve  a * x = b
c        first solve  l*y = b
c
         if (nm1 .lt. 1) go to 30
         do 20 k = 1, nm1
            l = ipvt(k)
            t = b(l)
            if (l .eq. k) go to 10
               b(l) = b(k)
               b(k) = t
   10       continue
            call mondaxpy(n-k,t,a(k+1,k),1,b(k+1),1)
   20    continue
   30    continue
c
c        now solve  u*x = y
c
         do 40 kb = 1, n
            k = n + 1 - kb
            b(k) = b(k)/a(k,k)
            t = -b(k)
            call mondaxpy(k-1,t,a(1,k),1,b(1),1)
   40    continue
      go to 100
   50 continue
c
c        job = nonzero, solve  trans(a) * x = b
c        first solve  trans(u)*y = b
c
         do 60 k = 1, n
            t = monddot(k-1,a(1,k),1,b(1),1)
            b(k) = (b(k) - t)/a(k,k)
   60    continue
c
c        now solve trans(l)*x = y
c
         if (nm1 .lt. 1) go to 90
         do 80 kb = 1, nm1
            k = n - kb
            b(k) = b(k) + monddot(n-k,a(k+1,k),1,b(k+1),1)
            l = ipvt(k)
            if (l .eq. k) go to 70
               t = b(l)
               b(l) = b(k)
               b(k) = t
   70       continue
   80    continue
   90    continue
  100 continue
      return
      end
      subroutine dgbfa(abd,lda,n,ml,mu,ipvt,info)
      integer lda,n,ml,mu,ipvt(1),info
      double precision abd(lda,1)
c
c     dgbfa factors a double precision band matrix by elimination.
c
c     dgbfa is usually called by dgbco, but it can be called
c     directly with a saving in time if  rcond  is not needed.
c
c     on entry
c
c        abd     double precision(lda, n)
c                contains the matrix in band storage.  the columns
c                of the matrix are stored in the columns of  abd  and
c                the diagonals of the matrix are stored in rows
c                ml+1 through 2*ml+mu+1 of  abd .
c                see the comments below for details.
c
c        lda     integer
c                the leading dimension of the array  abd .
c                lda must be .ge. 2*ml + mu + 1 .
c
c        n       integer
c                the order of the original matrix.
c
c        ml      integer
c                number of diagonals below the main diagonal.
c                0 .le. ml .lt. n .
c
c        mu      integer
c                number of diagonals above the main diagonal.
c                0 .le. mu .lt. n .
c                more efficient if  ml .le. mu .
c     on return
c
c        abd     an upper triangular matrix in band storage and
c                the multipliers which were used to obtain it.
c                the factorization can be written  a = l*u  where
c                l  is a product of permutation and unit lower
c                triangular matrices and  u  is upper triangular.
c
c        ipvt    integer(n)
c                an integer vector of pivot indices.
c
c        info    integer
c                = 0  normal value.
c                = k  if  u(k,k) .eq. 0.0 .  this is not an error
c                     condition for this subroutine, but it does
c                     indicate that dgbsl will divide by zero if
c                     called.  use  rcond  in dgbco for a reliable
c                     indication of singularity.
c
c     band storage
c
c           if  a  is a band matrix, the following program segment
c           will set up the input.
c
c                   ml = (band width below the diagonal)
c                   mu = (band width above the diagonal)
c                   m = ml + mu + 1
c                   do 20 j = 1, n
c                      i1 = max0(1, j-mu)
c                      i2 = min0(n, j+ml)
c                      do 10 i = i1, i2
c                         k = i - j + m
c                         abd(k,j) = a(i,j)
c                10    continue
c                20 continue
c
c           this uses rows  ml+1  through  2*ml+mu+1  of  abd .
c           in addition, the first  ml  rows in  abd  are used for
c           elements generated during the triangularization.
c           the total number of rows needed in  abd  is  2*ml+mu+1 .
c           the  ml+mu by ml+mu  upper left triangle and the
c           ml by ml  lower right triangle are not referenced.
c
c     linpack. this version dated 08/14/78 .
c     cleve moler, university of new mexico, argonne national lab.
c
c     subroutines and functions
c
c     blas mondaxpy,mondscal,monidamax
c     fortran max0,min0
c
c     internal variables
c
      double precision t
      integer i,monidamax,i0,j,ju,jz,j0,j1,k,kp1,l,lm,m,mm,nm1
c
c
      m = ml + mu + 1
      info = 0
c
c     zero initial fill-in columns
c
      j0 = mu + 2
      j1 = min0(n,m) - 1
      if (j1 .lt. j0) go to 30
      do 20 jz = j0, j1
         i0 = m + 1 - jz
         do 10 i = i0, ml
            abd(i,jz) = 0.0d0
   10    continue
   20 continue
   30 continue
      jz = j1
      ju = 0
c
c     gaussian elimination with partial pivoting
c
      nm1 = n - 1
      if (nm1 .lt. 1) go to 130
      do 120 k = 1, nm1
         kp1 = k + 1
c
c        zero next fill-in column
c
         jz = jz + 1
         if (jz .gt. n) go to 50
         if (ml .lt. 1) go to 50
            do 40 i = 1, ml
               abd(i,jz) = 0.0d0
   40       continue
   50    continue
c
c        find l = pivot index
c
         lm = min0(ml,n-k)
         l = monidamax(lm+1,abd(m,k),1) + m - 1
         ipvt(k) = l + k - m
c
c        zero pivot implies this column already triangularized
c
         if (abd(l,k) .eq. 0.0d0) go to 100
c
c           interchange if necessary
c
            if (l .eq. m) go to 60
               t = abd(l,k)
               abd(l,k) = abd(m,k)
               abd(m,k) = t
   60       continue
c
c           compute multipliers
c
            t = -1.0d0/abd(m,k)
            call mondscal(lm,t,abd(m+1,k),1)
c
c           row elimination with column indexing
c
            ju = min0(max0(ju,mu+ipvt(k)),n)
            mm = m
            if (ju .lt. kp1) go to 90
            do 80 j = kp1, ju
               l = l - 1
               mm = mm - 1
               t = abd(l,j)
               if (l .eq. mm) go to 70
                  abd(l,j) = abd(mm,j)
                  abd(mm,j) = t
   70          continue
               call mondaxpy(lm,t,abd(m+1,k),1,abd(mm+1,j),1)
   80       continue
   90       continue
         go to 110
  100    continue
            info = k
  110    continue
  120 continue
  130 continue
      ipvt(n) = n
      if (abd(m,n) .eq. 0.0d0) info = n
      return
      end
      subroutine dgbsl(abd,lda,n,ml,mu,ipvt,b,job)
      integer lda,n,ml,mu,ipvt(1),job
      double precision abd(lda,1),b(1)
c
c     dgbsl solves the double precision band system
c     a * x = b  or  trans(a) * x = b
c     using the factors computed by dgbco or dgbfa.
c
c     on entry
c
c        abd     double precision(lda, n)
c                the output from dgbco or dgbfa.
c
c        lda     integer
c                the leading dimension of the array  abd .
c
c        n       integer
c                the order of the original matrix.
c
c        ml      integer
c                number of diagonals below the main diagonal.
c
c        mu      integer
c                number of diagonals above the main diagonal.
c
c        ipvt    integer(n)
c                the pivot vector from dgbco or dgbfa.
c
c        b       double precision(n)
c                the right hand side vector.
c
c        job     integer
c                = 0         to solve  a*x = b ,
c                = nonzero   to solve  trans(a)*x = b , where
c                            trans(a)  is the transpose.
c
c     on return
c
c        b       the solution vector  x .
c
c     error condition
c
c        a division by zero will occur if the input factor contains a
c        zero on the diagonal.  technically this indicates singularity
c        but it is often caused by improper arguments or improper
c        setting of lda .  it will not occur if the subroutines are
c        called correctly and if dgbco has set rcond .gt. 0.0
c        or dgbfa has set info .eq. 0 .
c
c     to compute  inverse(a) * c  where  c  is a matrix
c     with  p  columns
c           call dgbco(abd,lda,n,ml,mu,ipvt,rcond,z)
c           if (rcond is too small) go to ...
c           do 10 j = 1, p
c              call dgbsl(abd,lda,n,ml,mu,ipvt,c(1,j),0)
c        10 continue
c
c     linpack. this version dated 08/14/78 .
c     cleve moler, university of new mexico, argonne national lab.
c
c     subroutines and functions
c
c     blas mondaxpy,monddot
c     fortran min0
c
c     internal variables
c
      double precision monddot,t
      integer k,kb,l,la,lb,lm,m,nm1
c
      m = mu + ml + 1
      nm1 = n - 1
      if (job .ne. 0) go to 50
c
c        job = 0 , solve  a * x = b
c        first solve l*y = b
c
         if (ml .eq. 0) go to 30
         if (nm1 .lt. 1) go to 30
            do 20 k = 1, nm1
               lm = min0(ml,n-k)
               l = ipvt(k)
               t = b(l)
               if (l .eq. k) go to 10
                  b(l) = b(k)
                  b(k) = t
   10          continue
               call mondaxpy(lm,t,abd(m+1,k),1,b(k+1),1)
   20       continue
   30    continue
c
c        now solve  u*x = y
c
         do 40 kb = 1, n
            k = n + 1 - kb
            b(k) = b(k)/abd(m,k)
            lm = min0(k,m) - 1
            la = m - lm
            lb = k - lm
            t = -b(k)
            call mondaxpy(lm,t,abd(la,k),1,b(lb),1)
   40    continue
      go to 100
   50 continue
c
c        job = nonzero, solve  trans(a) * x = b
c        first solve  trans(u)*y = b
c
         do 60 k = 1, n
            lm = min0(k,m) - 1
            la = m - lm
            lb = k - lm
            t = monddot(lm,abd(la,k),1,b(lb),1)
            b(k) = (b(k) - t)/abd(m,k)
   60    continue
c
c        now solve trans(l)*x = y
c
         if (ml .eq. 0) go to 90
         if (nm1 .lt. 1) go to 90
            do 80 kb = 1, nm1
               k = n - kb
               lm = min0(ml,n-k)
               b(k) = b(k) + monddot(lm,abd(m+1,k),1,b(k+1),1)
               l = ipvt(k)
               if (l .eq. k) go to 70
                  t = b(l)
                  b(l) = b(k)
                  b(k) = t
   70          continue
   80       continue
   90    continue
  100 continue
      return
      end
      subroutine mondaxpy(n,da,dx,incx,dy,incy)
c
c     constant times a vector plus a vector.
c     uses unrolled loops for increments equal to one.
c     jack dongarra, linpack, 3/11/78.
c
      double precision dx(1),dy(1),da
      integer i,incx,incy,ix,iy,m,mp1,n
c
      if(n.le.0)return
      if (da .eq. 0.0d0) return
      if(incx.eq.1.and.incy.eq.1)go to 20
c
c        code for unequal increments or equal increments
c          not equal to 1
c
      ix = 1
      iy = 1
      if(incx.lt.0)ix = (-n+1)*incx + 1
      if(incy.lt.0)iy = (-n+1)*incy + 1
      do 10 i = 1,n
        dy(iy) = dy(iy) + da*dx(ix)
        ix = ix + incx
        iy = iy + incy
   10 continue
      return
c
c        code for both increments equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,4)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        dy(i) = dy(i) + da*dx(i)
   30 continue
      if( n .lt. 4 ) return
   40 mp1 = m + 1
      do 50 i = mp1,n,4
        dy(i) = dy(i) + da*dx(i)
        dy(i + 1) = dy(i + 1) + da*dx(i + 1)
        dy(i + 2) = dy(i + 2) + da*dx(i + 2)
        dy(i + 3) = dy(i + 3) + da*dx(i + 3)
   50 continue
      return
      end
      subroutine mondscal(n,da,dx,incx)
c
c     scales a vector by a constant.
c     uses unrolled loops for increment equal to one.
c     jack dongarra, linpack, 3/11/78.
c
      double precision da,dx(1)
      integer i,incx,m,mp1,n,nincx
c
      if(n.le.0)return
      if(incx.eq.1)go to 20
c
c        code for increment not equal to 1
c
      nincx = n*incx
      do 10 i = 1,nincx,incx
        dx(i) = da*dx(i)
   10 continue
      return
c
c        code for increment equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,5)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        dx(i) = da*dx(i)
   30 continue
      if( n .lt. 5 ) return
   40 mp1 = m + 1
      do 50 i = mp1,n,5
        dx(i) = da*dx(i)
        dx(i + 1) = da*dx(i + 1)
        dx(i + 2) = da*dx(i + 2)
        dx(i + 3) = da*dx(i + 3)
        dx(i + 4) = da*dx(i + 4)
   50 continue
      return
      end
      double precision function monddot(n,dx,incx,dy,incy)
c
c     forms the dot product of two vectors.
c     uses unrolled loops for increments equal to one.
c     jack dongarra, linpack, 3/11/78.
c
      double precision dx(1),dy(1),dtemp
      integer i,incx,incy,ix,iy,m,mp1,n
c
      monddot = 0.0d0
      dtemp = 0.0d0
      if(n.le.0)return
      if(incx.eq.1.and.incy.eq.1)go to 20
c
c        code for unequal increments or equal increments
c          not equal to 1
c
      ix = 1
      iy = 1
      if(incx.lt.0)ix = (-n+1)*incx + 1
      if(incy.lt.0)iy = (-n+1)*incy + 1
      do 10 i = 1,n
        dtemp = dtemp + dx(ix)*dy(iy)
        ix = ix + incx
        iy = iy + incy
   10 continue
      monddot = dtemp
      return
c
c        code for both increments equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,5)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        dtemp = dtemp + dx(i)*dy(i)
   30 continue
      if( n .lt. 5 ) go to 60
   40 mp1 = m + 1
      do 50 i = mp1,n,5
        dtemp = dtemp + dx(i)*dy(i) + dx(i + 1)*dy(i + 1) +
     *   dx(i + 2)*dy(i + 2) + dx(i + 3)*dy(i + 3) + dx(i + 4)*dy(i + 4)
   50 continue
   60 monddot = dtemp
      return
      end
      integer function monidamax(n,dx,incx)
c
c     finds the index of element having max. absolute value.
c     jack dongarra, linpack, 3/11/78.
c
      double precision dx(1),dmax
      integer i,incx,ix,n
c
      monidamax = 0
      if( n .lt. 1 ) return
      monidamax = 1
      if(n.eq.1)return
      if(incx.eq.1)go to 20
c
c        code for increment not equal to 1
c
      ix = 1
      dmax = dabs(dx(1))
      ix = ix + incx
      do 10 i = 2,n
         if(dabs(dx(ix)).le.dmax) go to 5
         monidamax = i
         dmax = dabs(dx(ix))
    5    ix = ix + incx
   10 continue
      return
c
c        code for increment equal to 1
c
   20 dmax = dabs(dx(1))
      do 30 i = 2,n
         if(dabs(dx(i)).le.dmax) go to 30
         monidamax = i
         dmax = dabs(dx(i))
   30 continue
      return
      end
      subroutine mondcopy(n,sx,incx,sy,incy)
c
c     copies a vector, x, to a vector, y.
c     uses unrolled loops for increments equal to 1.
c     jack dongarra, linpack, 3/11/78.
c
      double precision sx(1),sy(1)
      integer i,incx,incy,ix,iy,m,mp1,n
c
      if(n.le.0)return
      if(incx.eq.1.and.incy.eq.1)go to 20
c
c        code for unequal increments or equal increments
c          not equal to 1
c
      ix = 1
      iy = 1
      if(incx.lt.0)ix = (-n+1)*incx + 1
      if(incy.lt.0)iy = (-n+1)*incy + 1
      do 10 i = 1,n
        sy(iy) = sx(ix)
        ix = ix + incx
        iy = iy + incy
   10 continue
      return
c
c        code for both increments equal to 1
c
c
c        clean-up loop
c
   20 m = mod(n,7)
      if( m .eq. 0 ) go to 40
      do 30 i = 1,m
        sy(i) = sx(i)
   30 continue
      if( n .lt. 7 ) return
   40 mp1 = m + 1
      do 50 i = mp1,n,7
        sy(i) = sx(i)
        sy(i + 1) = sx(i + 1)
        sy(i + 2) = sx(i + 2)
        sy(i + 3) = sx(i + 3)
        sy(i + 4) = sx(i + 4)
        sy(i + 5) = sx(i + 5)
        sy(i + 6) = sx(i + 6)
   50 continue
      return
      end
c      block data
c-----------------------------------------------------------------------
c this data subprogram loads variables into the internal common
c blocks used by the odepack solvers.  the variables are
c defined as follows..
c   illin  = counter for the number of consecutive times the package
c            was called with illegal input.  the run is stopped when
c            illin reaches 5.
c   ntrep  = counter for the number of consecutive times the package
c            was called with istate = 1 and tout = t.  the run is
c            stopped when ntrep reaches 5.
c   mesflg = flag to control printing of error messages.  1 means print,
c            0 means no printing.
c   lunit  = default value of logical unit number for printing of error
c            messages.
c-----------------------------------------------------------------------
c      integer illin, iduma, ntrep, idumb, iowns, icomm, mesflg, lunit
c      double precision rowns, rcomm
c      common /ls0001/ rowns(209), rcomm(9),
c     1   illin, iduma(10), ntrep, idumb(2), iowns(6), icomm(19)
c      common /eh0001/ mesflg, lunit
c      data illin/0/, ntrep/0/
c      data mesflg/1/, lunit/6/
c
c----------------------- end of block data -----------------------------
c      end

