/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : CCov3                     |
| Date                 : 1992                      |
| Derniere mise a jour :                           |
| Concepteur           : A. Bouvier                |
| Role                 : les programmes de calcul  |
|   la variance asymptotique a l'etape 3           |
| Reference conception :                           |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/


/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <stdio.h>
#include <errno.h>


#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlglobal.h"
#include "nlmacros.h"
#include "errcodes.h"


/*--------------- FONCTIONS EXTERNES -------------*/
/* fonctions des autres modules */
TShortInt CCovNu(), CEtamco(), CEtamv(), 
          CopyMat(), CreerMat(), CreerVect(), 
          CDmvt(), CDmcb(), CVarZmco(), CVarZmv(), CValW(), CZmco(),
          DivMatVal(), DModAAct(),
          EcrMat(), EcrVect(),
          InvMat(),
          ModAAct(), MultMat(), MultMatT(),
          TrValW(),
          VNuAVMod();

void DetruMat(TMat *pmat);
void DetruVect(TVect *pvect);
TShortInt GerMessage( );

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/

/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueNu3D               |
| Role                  : allouer les structures de|
|  travail relatives aux derivees par rapport aux  |
|  parametres necessaires au programme CCovNu3     |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         CreerMat,                |
| Fonctions appelantes :  CCovNu3                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt AlloueNu3D( NbObs, NbTEff1, NbBEff1,  NbTEff2, NbBEff2, NbTAct1, NbBAct1, NbTAct2, NbBAct2,
                    DEffT1t, DEffB1t, DEffT2t, DEffB2t,
                    DValf1t, DValf2t,
                    DVarYT1t,  DVarYB1t, DVarYT2t, DVarYB2t)

/* arguments d'entree */
TShortInt NbObs, NbTEff1, NbBEff1,  NbTEff2, NbBEff2, NbTAct1, NbBAct1, NbTAct2, NbBAct2;

/* arguments de sortie */
TMat *DEffT1t, *DEffB1t, *DEffT2t, *DEffB2t,
     *DValf1t, * DValf2t,
     *DVarYT1t, *DVarYB1t, *DVarYT2t, *DVarYB2t;

/*--------------- Fin identification fonction ----*/

{

APPEL(CreerMat(NbObs, NbTEff1, DEffT1t));
APPEL(CreerMat(NbObs, NbBEff1, DEffB1t));
APPEL(CreerMat(NbObs, NbTEff2, DEffT2t));
APPEL(CreerMat(NbObs, NbBEff2, DEffB2t));
APPEL(CreerMat(NbObs, NbTAct1, DValf1t));
APPEL(CreerMat(NbObs, NbTAct2, DValf2t));
APPEL(CreerMat(NbObs, NbTAct1, DVarYT1t));
APPEL(CreerMat(NbObs, NbBAct1, DVarYB1t));
APPEL(CreerMat(NbObs, NbTAct2, DVarYT2t));
APPEL(CreerMat(NbObs, NbBAct2, DVarYB2t));

return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueNu3N               |
| Role                  : allouer les structures de|
|  travail relatives aux equations normales        |
|  necessaires au programme CCovNu3                |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         CreerMat,                |
| Fonctions appelantes :  CCovNu3                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt AlloueNu3N( NbZ1, NbZ2, NbZ3, NbAct1, NbAct2,
                     ValB1t, ValD1t, 
                     ValB2t, ValD2t,
                     ValDB3t,
                     ValDT2t) 

/* arguments d'entree */
TShortInt NbZ1, NbZ2, NbZ3, NbAct1, NbAct2;

/* arguments de sortie */
TMat *ValB1t, *ValD1t, *ValB2t, *ValD2t, *ValDB3t, *ValDT2t;

/*--------------- Fin identification fonction ----*/

{

APPEL(CreerMat(NbZ1, NbAct1,  ValB1t));
APPEL(CreerMat(NbZ1, NbAct1,  ValD1t));
APPEL(CreerMat(NbZ2, NbAct2,  ValB2t));
APPEL(CreerMat(NbZ2, NbAct2,  ValD2t));
APPEL(CreerMat(NbZ3, NbAct2,  ValDB3t));
APPEL(CreerMat(NbZ3,  NbAct1, ValDT2t));

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueNu3P               |
| Role                  : allouer les structures de|
|  travail relatives aux parametres                |
|  necessaires au programme CCovNu3                |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         CreerVect                |
| Fonctions appelantes :  CCovNu3                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt AlloueNu3P(NbTEff1, NbBEff1, NbTEff2,  NbBEff2,NbTAct1, NbBAct1, NbTAct2, NbBAct2,
                    TAct1t, BAct1t, TAct2t, BAct2t, TEff1t, BEff1t, TEff2t, BEff2t)

/* arguments d'entree */
TShortInt NbTEff1, NbBEff1,NbTEff2,  NbBEff2, NbTAct1, NbBAct1, NbTAct2, NbBAct2;

/* arguments de sortie */
TVect *TAct1t, *BAct1t, *TAct2t, *BAct2t, *TEff1t, *BEff1t, *TEff2t, *BEff2t;

/*--------------- Fin identification fonction ----*/

{

APPEL(CreerVect(NbTAct1, TAct1t));
APPEL(CreerVect(NbBAct1, BAct1t));
APPEL(CreerVect(NbTAct2, TAct2t));
APPEL(CreerVect(NbBAct2, BAct2t));
APPEL(CreerVect(NbTEff1, TEff1t));
APPEL(CreerVect(NbBEff1, BEff1t));
APPEL(CreerVect(NbTEff2, TEff2t));
APPEL(CreerVect(NbBEff2, BEff2t));

return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueNu3T               |
| Role                  : allouer les structures de|
|  travail non allouees par les autres programmes  |
|  ALloue  et necessaires au programme CCovNu3     |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         CreerMat,                |
| Fonctions appelantes :  CCovNu3                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt AlloueNu3T(NbAct1, NbAct2, NbAct3, NbZ1, NbZ2, NbZ3,
                     AsVar13, AsVarNu23, AsVarNuB2t, AsVarNuT3t,
                     BVarZBP2t,
                     CovZ31t, CovZ32t, G13, G123, Trav)


/* arguments d'entree */
TShortInt NbAct1, NbAct2, NbAct3, NbZ1, NbZ2, NbZ3;

/* arguments de sortie */
TMat *AsVar13, *AsVarNu23, *AsVarNuB2t, *AsVarNuT3t,
    *BVarZBP2t, *CovZ31t, *CovZ32t, *G13, *G123, *Trav;           

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt NbCol, MaxNbZ13, MaxNbZ23;

MaxNbZ13 = MAX(NbZ1, NbZ3);
MaxNbZ23 = MAX(NbZ2, NbZ3);


APPEL(CreerMat(NbAct3, NbAct1, AsVar13));
APPEL(CreerMat(NbAct3, NbAct2, AsVarNu23));
APPEL(CreerMat(NbAct2, NbAct2, AsVarNuB2t));
APPEL(CreerMat(NbAct3, NbAct3, AsVarNuT3t));
APPEL(CreerMat(NbAct2, NbAct2, BVarZBP2t));
APPEL(CreerMat(MaxNbZ13, MaxNbZ13, CovZ31t));
APPEL(CreerMat(MaxNbZ23, MaxNbZ23, CovZ32t));
APPEL(CreerMat(NbAct3, NbAct2, G13));
APPEL(CreerMat(NbAct3, NbAct3, G123));

NbCol = MAX(NbAct1, NbAct2);
NbCol = MAX(NbCol, NbAct3);

APPEL(CreerMat(NbAct3, NbCol, Trav));
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueNu3W               |
| Role                  : allouer les structures de|
|  travail relatives aux matrices ValW             |
|  necessaires au programme CCovNu3                |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         CreerMat,                |
| Fonctions appelantes :  CCovNu3                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt AlloueNu3W(NbAct1, NbAct2, NbAct3,
                     ValWB2t, ValWB3t, ValWT1t, ValWT2t, ValWInv)

/* arguments d'entree */
TShortInt NbAct1, NbAct2, NbAct3;

/* arguments de sortie */
TMat *ValWB2t, *ValWB3t, *ValWT1t, *ValWT2t, *ValWInv;


/*--------------- Fin identification fonction ----*/

{

APPEL(CreerMat(NbAct2, NbAct2,  ValWB2t));
APPEL(CreerMat(NbAct3, NbAct2,  ValWB3t));
APPEL(CreerMat(NbAct1, NbAct1, ValWT1t));
APPEL(CreerMat(NbAct2, NbAct1, ValWT2t));
APPEL(CreerMat(NbAct3, NbAct3, ValWInv));

return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : DesAlloueNu3D            |
| Role                  : allouer les structures de|
|  travail allouees par AlloueNu3D                 |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         DetruMat                 |
| Fonctions appelantes :  CCovNu3                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt DesAlloueNu3D(
                    DEffT1t, DEffB1t, DEffT2t, DEffB2t,
                    DValf1t, DValf2t,
                    DVarYT1t,  DVarYB1t, DVarYT2t, DVarYB2t)

/* arguments de sortie */
TMat *DEffT1t, *DEffB1t, *DEffT2t, *DEffB2t,
     *DValf1t, * DValf2t,
     *DVarYT1t, *DVarYB1t, *DVarYT2t, *DVarYB2t;

/*--------------- Fin identification fonction ----*/

{
DetruMat(DVarYB2t);
DetruMat(DVarYT2t);
DetruMat(DVarYB1t);
DetruMat(DVarYT1t);
DetruMat(DValf2t);
DetruMat(DValf1t);
DetruMat(DEffB2t);
DetruMat(DEffT2t);
DetruMat(DEffB1t);
DetruMat(DEffT1t);
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : DesAlloueNu3N            |
| Role                  : desallouer les structures|
|  de travail allouees par AlloueNu3N dans l'ordre |
|  inverse d'allocation                            |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         DetruMat,                |
| Fonctions appelantes :  CCovNu3                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt DesAlloueNu3N(
                     ValB1t, ValD1t, 
                     ValB2t, ValD2t,
                     ValDB3t,
                     ValDT2t) 


/* arguments de sortie */
TMat *ValB1t, *ValD1t, *ValB2t, *ValD2t, *ValDB3t, *ValDT2t;

/*--------------- Fin identification fonction ----*/

{

DetruMat( ValDT2t);
DetruMat(  ValDB3t);
DetruMat(  ValD2t);
DetruMat(  ValB2t);
DetruMat(  ValD1t);
DetruMat(  ValB1t);

return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : DesAlloueNu3P            |
| Role                  : desallouer les structures|
|  de travail allouees par AlloueNu3P dans l'ordre |
|  inverse d'allocation                            |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         DetruVect                |
| Fonctions appelantes :  CCovNu3                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt DesAlloueNu3P(
                    TAct1t, BAct1t, TAct2t, BAct2t, TEff1t, BEff1t, TEff2t, BEff2t)


/* arguments de sortie */
TVect *TAct1t, *BAct1t, *TAct2t, *BAct2t, *TEff1t, *BEff1t, *TEff2t, *BEff2t;

/*--------------- Fin identification fonction ----*/



{
DetruVect( BEff2t);
DetruVect( TEff2t);
DetruVect( BEff1t);
DetruVect( TEff1t);
DetruVect( BAct2t);
DetruVect( TAct2t);
DetruVect( BAct1t);
DetruVect( TAct1t);

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : DesAlloueNu3T            |
| Role                  : desallouer les structures|
|  de travail allouees par AlloueNu3T dans l'ordre |
|  inverse d'allocation                            |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         DetruMat,                |
| Fonctions appelantes :  CCovNu3                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt DesAlloueNu3T(
                     AsVar13, AsVarNu23, AsVarNuB2t, AsVarNuT3t,
                     BVarZBP2t,
                     CovZ31t, CovZ32t, G13, G123, Trav)


/* arguments de sortie */
TMat *AsVar13, *AsVarNu23, *AsVarNuB2t, *AsVarNuT3t,
    *BVarZBP2t, *CovZ31t, *CovZ32t, *G13, *G123, *Trav;           

/*--------------- Fin identification fonction ----*/


{
DetruMat(Trav);
DetruMat(G123);
DetruMat(G13);
DetruMat(CovZ32t);
DetruMat(CovZ31t);
DetruMat(BVarZBP2t);
DetruMat(AsVarNuT3t);
DetruMat(AsVarNuB2t);
DetruMat(AsVarNu23);
DetruMat(AsVar13);


return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : DesAlloueNu3W            |
| Role                  : desallouer les structures|
|  de travail allouees par AlloueNu3W dans l'ordre |
|  inverse d'allocation                            |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         DetruMat,                |
| Fonctions appelantes :  CCovNu3                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt DesAlloueNu3W(
                     ValWB2t, ValWB3t, ValWT1t, ValWT2t, ValWInv)

/* arguments de sortie */
TMat *ValWB2t, *ValWB3t, *ValWT1t, *ValWT2t, *ValWInv;


/*--------------- Fin identification fonction ----*/

{
DetruMat(ValWInv);
DetruMat( ValWT2t);
DetruMat(ValWT1t);
DetruMat(ValWB3t);
DetruMat(ValWB2t);
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CCov3                    |
| Role                  : calculer la matrice de   |
|  variance-covariance asymptotique                |
|  a l'etape 3 (estimation des Theta seuls)        |
| Parametres d'entree   :                          |
|  Donnees: les donnees                            |
|  Theta: la structure des parametres Theta        |
|  Beta: la structure des parametres Beta          |
|  CTheta, CBeta: les contraintes sur les          |
|   parametres du modele                           |
|  CThetaE, CBetaE: les contraintes sur les        |
|    parametres differents Theta  et  Beta         |
|  CtxNum: contexte numerique                      |
|  ResNum: les resultats statistiques              |
|  Tous les arguments ci-dessus, excepte Donnees,  |
|  sont des tableaux d'au moins 2 structures       |
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
|  ResStat: les resultats statistiques de l'etape 3|
| Parametre de sortie   :                          |
|  Code = OK ou code de CCovNu3                    |
| Parametres de sortie  :                          |
| Effets de bord: voir programme CCovNu3           |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         CCovNu3,                 |
|                         CreerMat,                |
|                         DetruMat,                |
|                         EcrMat,                  |
|                         VNuAVMod                 |
| Fonctions appelantes :  NLAutres                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CCov3(  Donnees, Theta, Beta, CTheta, CBeta, CThetaE, CBetaE, CtxNum, ResNum,
                 TypeMu,  ResStat, Code)


/* arguments d'entree */
TDonnees *Donnees;
TParam Theta[] , Beta[];
TContr CTheta[], CBeta[];
TContr CThetaE[], CBetaE[];
TCtxNum CtxNum[];
TResNum ResNum[];

/* arguments d'entree-sortie */
TShortInt *TypeMu;
TResStat *ResStat;

/* argument de sortie */
TShortInt *Code;


/*--------------- Fin identification fonction ----*/

{
/* locals */
TMat AsVarNu3; /* matrice de variance asymptotique apres prise en compte
              des contraintes numeriques */
TShortInt NbAct, Etap3;

/* pointeur sur les fonctions du module appelees et non encore definies */
TShortInt CCovNu3();

/* Ecriture de la trace */
ECRTRACE("CCov3");


*Code = OK;

Etap3 = 2; /* indice dans les tableaux correspondant a l'etape 3 */

/* allocation de AsVarNu3 */
NbAct = Theta[Etap3].Act.nbele + Beta[Etap3].Act.nbele;

APPEL(CreerMat(NbAct, NbAct, &AsVarNu3));

/* calcul de la matrice de variance asymptotique apres prise en compte
des contraintes numeriques */

APPEL(CCovNu3 ( Donnees, Theta, Beta, CTheta, CBeta, CThetaE, CBetaE, CtxNum, ResNum,
                 TypeMu,  ResStat, &AsVarNu3, Code));

if (*Code == OK)
  {
  
  /* transformation en dimension parametres du modele */
  APPEL(VNuAVMod( &AsVarNu3, &(Theta[Etap3]), &(Beta[Etap3]), &(CTheta[Etap3]), &(CBeta[Etap3]), 
                  &(CThetaE[Etap3]), &(CBetaE[Etap3]),
                  &(ResStat->AsVar)));
  /* c'est VNuAVMod qui alloue ResStat->AsVar */
  /* desallocation de AsVarNu3 */
  }

DetruMat(&AsVarNu3);



/* retour */
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : CCovNu3                  |
| Role                  : calculer la matrice de   |
|  variance-covariance asymptotique des parametres |
|  apres prise en compte des contraintes           |
|  d'inegalite numerique, a l'etape 3              |
| Parametres d'entree   :                          |
|  Donnees: les donnees                            |
|  Theta: la structure des parametres Theta        |
|  Beta: la structure des parametres Beta          |
|  CTheta, CBeta: les contraintes sur les          |
|   parametres du modele                           |
|  CThetaE, CBetaE: les contraintes sur les        |
|    parametres differents Theta  et  Beta         |
|  CtxNum: contexte numerique                      |
|  ResNum: les resultats statistiques              |
|  Tous les arguments ci-dessus, excepte Donnees,  |
|  sont des tableaux d'au moins 2 structures       |
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
|  ResStat: les resultats statistiques de l'etape 2|
| Parametres de sortie  :                          |
|  AsVarNu3: variance asymptotique en dimension    |
|    ``actifs''                                    |
|   allouee avant l'appel                          |
|  Code = OK ou retour de CCovNu, CVarZmco, CVarZmv|
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                       AlloueNu3D,                |
|                       AlloueNu3N,                |
|                       AlloueNu3P,                |
|                       AlloueNu3T,                |
|                       AlloueNu3W,                |
|                       CCovNu,                    |
|                       CDmcb,  CDmvt,             |
|                       CopyMat,                   |
|                       CValW,                     |
|                       CVarZmco, CVarZmv,         |
|                       DesAlloueNu3D,             |
|                       DesAlloueNu3N,             |
|                       DesAlloueNu3P,             |
|                       DesAlloueNu3T,             |
|                       DesAlloueNu3W,             |
|                       DivMatVal,                 |
|                       DModAAct,                  |
|                       EcrMat,                    |
|                       InvMat,                    |
|                       ModAAct,                   |
|                       MultMat,                   |
|                       MultMatT,                  |
|                       PCValB des etapes 1 et 2,  |
|                       PCValD des etapes 1 et 2,  |
|                       TrValW                     |
| Fonctions appelantes :  CCov3                    |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CCovNu3( Donnees, Theta, Beta, CTheta, CBeta, CThetaE, CBetaE, CtxNum, ResNum, 
                 TypeMu,  ResStat, AsVarNu3, Code)




/* arguments d'entree */
TDonnees *Donnees;
TParam Theta[] , Beta[];
TContr CTheta[], CBeta[];
TContr CThetaE[], CBetaE[];
TCtxNum CtxNum[];
TResNum ResNum[];


/* arguments d'entree-sortie */
TShortInt *TypeMu;
TResStat *ResStat;

/* arguments de sortie */
TMat *AsVarNu3; /* dimension NbAct, NbAct , a l'etape 3 */
TShortInt *Code;


/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt NbAct1, NbAct2, NbAct3,
          NbTAct1, NbBAct1, NbTEff1, NbBEff1,
          NbTAct2, NbBAct2, NbTEff2, NbBEff2,
          NbZ1, NbZ2, NbZ3,
          Etap1, Etap2, Etap3,
          i, j , i1, i2, i3;
TDouble Val, NbObsT;
TLogic  VouluBVarZBP;

/* structures pour stocker les calculs intermediaires */
/* les derivees par rapport aux parametres */
TMat DEffT1t, DEffB1t, DEffT2t, DEffB2t,
                    DValf1t, DValf2t,
                    DVarYT1t,  DVarYB1t, DVarYT2t, DVarYB2t;
/* les equN */
TMat ValB1t, ValD1t,
                     ValB2t, ValD2t,
                     ValDB3t,
                     ValDT2t;

/* les valeurs des parametres */
TVect TAct1t, BAct1t, TAct2t, BAct2t, TEff1t, BEff1t, TEff2t, BEff2t;


/* les matrices ValW */
TMat ValWB2t, ValWB3t, ValWT1t, ValWT2t;



/* le reste */
TMat AsVar13, AsVarNu23, AsVarNuB2t, AsVarNuT3t,
                     BVarZBP2t,
                     CovZ31t, CovZ32t, G13, G123;

/* les tableaux de travail */
TMat ValWInv, Trav;
/* ValWInv: NbAct3, NbAct3 */
/* Trav: NbAct3, Max(NbAct1,NbAct2,NBAct3) */

/* pointeurs sur des elements des structures passees en argument:
sert a ameliorer la performance */
TDouble   **asvar13, **asvarnu23, **asvarnu3, **asvarnut3t,
          **covz31t, **covz32t, **g123, **g13, **valb, **valb1t, **valb2t;


/* Ecriture de la trace */
ECRTRACE("CCovNu3");

*Code = OK;

Etap1 = 0; /* indice dans les tableaux correspondant a l'etape 1 */
Etap2 = 1; /* indice dans les tableaux correspondant a l'etape 2 */
Etap3 = 2; /* indice dans les tableaux correspondant a l'etape 3 */

/* initialisations des dimensions */
NbTEff1 = Theta[Etap1].Eff.nbele;
NbBEff1 = Beta[Etap1].Eff.nbele;
NbTAct1 = Theta[Etap1].Act.nbele;
NbBAct1 = Beta[Etap1].Act.nbele;
NbTEff2 = Theta[Etap2].Eff.nbele;
NbBEff2 = Beta[Etap2].Eff.nbele;
NbTAct2 = Theta[Etap2].Act.nbele;
NbBAct2 = Beta[Etap2].Act.nbele;

NbAct1 = NbTAct1 + NbBAct1;
NbAct2 = NbTAct2 + NbBAct2;
NbAct3 = Theta[Etap3].Act.nbele + Beta[Etap3].Act.nbele;
NbZ1 = CtxNum[Etap1].NbZ * Donnees->NbObs;
NbZ2 = CtxNum[Etap2].NbZ * Donnees->NbObs;
NbZ3 = CtxNum[Etap3].NbZ * Donnees->NbObs;


/* Allocation des structures de travail */
/* ----------------------------------- */

/* allocation des structures relatives aux derivees par rapport aux parametres */
APPEL(AlloueNu3D( (TShortInt)(Donnees->NbObs),
          NbTEff1, NbBEff1,  NbTEff2, NbBEff2, NbTAct1, NbBAct1, NbTAct2, NbBAct2,
          &DEffT1t, &DEffB1t, &DEffT2t, &DEffB2t,
          &DValf1t, &DValf2t,
          &DVarYT1t,  &DVarYB1t, &DVarYT2t, &DVarYB2t));


/* allocation des structures relatives aux equations normales */
APPEL(AlloueNu3N(NbZ1, NbZ2, NbZ3, NbAct1, NbAct2,
                     &ValB1t, &ValD1t,
                     &ValB2t, &ValD2t,
                     &ValDB3t,
                     &ValDT2t));

/* allocation des structures relatives aux parametres */
APPEL(AlloueNu3P(NbTEff1, NbBEff1, NbTEff2,  NbBEff2,NbTAct1, NbBAct1, NbTAct2, NbBAct2,
                    &TAct1t, &BAct1t, &TAct2t, &BAct2t, &TEff1t, &BEff1t, &TEff2t, &BEff2t));

/* allocation des structures relatives aux matrices ValW */
APPEL(AlloueNu3W(NbAct1, NbAct2, NbAct3,
                 &ValWB2t, &ValWB3t, &ValWT1t, &ValWT2t, &ValWInv));


/* allocation des autres structures de travail */
APPEL(AlloueNu3T(NbAct1, NbAct2, NbAct3, NbZ1, NbZ2, NbZ3,
                 &AsVar13, &AsVarNu23, &AsVarNuB2t, &AsVarNuT3t,
                 &BVarZBP2t,
                 &CovZ31t, &CovZ32t, &G13, &G123, &Trav));


/*  calcul de AsVarNuT3t */
/* --------------------- */
APPEL(CCovNu( ResNum[Etap3].Sigma,
              &(CtxNum[Etap3]), Donnees,
              &(ResStat->EquN.ValB), &(ResStat->ValW), &(ResStat->Ajustes),
              TypeMu,  &(ResStat->Residus), &(ResStat->Mu3), &(ResStat->Mu4), 
              &(ResStat->BVarZBP),
              &AsVarNuT3t, Code));

if (*Code != OK)
  {
  /* Desallocation des structures de travail dans l'ordre des alloc */
  APPEL(DesAlloueNu3T(
                     &AsVar13, &AsVarNu23, &AsVarNuB2t, &AsVarNuT3t,
                     &BVarZBP2t,
                     &CovZ31t, &CovZ32t, &G13, &G123, &Trav));
  APPEL(DesAlloueNu3W(
                 &ValWB2t, &ValWB3t, &ValWT1t, &ValWT2t, &ValWInv));
  APPEL(DesAlloueNu3P(
                    &TAct1t, &BAct1t, &TAct2t, &BAct2t, &TEff1t, &BEff1t, &TEff2t, &BEff2t));
  APPEL(DesAlloueNu3N(
                     &ValB1t, &ValD1t,
                     &ValB2t, &ValD2t,
                     &ValDB3t,
                     &ValDT2t));
  APPEL(DesAlloueNu3D(
          &DEffT1t, &DEffB1t, &DEffT2t, &DEffB2t,
          &DValf1t, &DValf2t,
          &DVarYT1t,  &DVarYB1t, &DVarYT2t, &DVarYB2t));

  return(OK);
  }


if(GNLControle.SortImp != NULL)
  {
  /* Impressions */
  /*  fprintf(GNLControle.SortImp,
"\n\n * IMPRESSIONS INTERMEDIAIRES DANS LE CALCUL DE LA VARIANCE ASYMPTOTIQUE A L'ETAPE 3\n\n"); */
  NLWARNING((IMPCOV,2,PROMPT,"3",IMP));

  fprintf(GNLControle.SortImp,
 "SigmaTheta3\n");
  APPEL(EcrMat(GNLControle.SortImp, &AsVarNuT3t));
  }


/* si PCValEta[Etap3]== CEtamco, c'est fini */
if (CtxNum[Etap3].PCEquN.PCValEta == CEtamco)
  {
  APPEL(CopyMat(&AsVarNuT3t, AsVarNu3));
  /* Desallocation des structures de travail dans l'ordre des alloc */
  APPEL(DesAlloueNu3T(
                     &AsVar13, &AsVarNu23, &AsVarNuB2t, &AsVarNuT3t,
                     &BVarZBP2t,
                     &CovZ31t, &CovZ32t, &G13, &G123, &Trav));
  APPEL(DesAlloueNu3W(
                 &ValWB2t, &ValWB3t, &ValWT1t, &ValWT2t, &ValWInv));
  APPEL(DesAlloueNu3P(
                    &TAct1t, &BAct1t, &TAct2t, &BAct2t, &TEff1t, &BEff1t, &TEff2t, &BEff2t));
  APPEL(DesAlloueNu3N(
                     &ValB1t, &ValD1t,
                     &ValB2t, &ValD2t,
                     &ValDB3t,
                     &ValDT2t));
  APPEL(DesAlloueNu3D(
          &DEffT1t, &DEffB1t, &DEffT2t, &DEffB2t,
          &DValf1t, &DValf2t,
          &DVarYT1t,  &DVarYB1t, &DVarYT2t, &DVarYB2t));

  return(OK);
  }



if ( (CtxNum[Etap3].PCEquN.PCValEta != CEtamv) || 
     (CtxNum[Etap2].PCEquN.PCValEta != CEtamv))
  {
  /* Emettre un message */
  /* printf("On ne sait pas calculer la variance a l'etape 3 dans ces cas-la\n");  */
  NLWARNING((WARCCOV3,1,"CCov3",WNUMER));
  /* Desallocation des structures de travail dans l'ordre des alloc */
  APPEL(DesAlloueNu3T(
                     &AsVar13, &AsVarNu23, &AsVarNuB2t, &AsVarNuT3t,
                     &BVarZBP2t,
                     &CovZ31t, &CovZ32t, &G13, &G123, &Trav));
  APPEL(DesAlloueNu3W(
                 &ValWB2t, &ValWB3t, &ValWT1t, &ValWT2t, &ValWInv));
  APPEL(DesAlloueNu3P(
                    &TAct1t, &BAct1t, &TAct2t, &BAct2t, &TEff1t, &BEff1t, &TEff2t, &BEff2t));
  APPEL(DesAlloueNu3N(
                     &ValB1t, &ValD1t,
                     &ValB2t, &ValD2t,
                     &ValDB3t,
                     &ValDT2t));
  APPEL(DesAlloueNu3D(
          &DEffT1t, &DEffB1t, &DEffT2t, &DEffB2t,
          &DValf1t, &DValf2t,
          &DVarYT1t,  &DVarYB1t, &DVarYT2t, &DVarYB2t));

  return(OK);
  }



/* Calcul de ValWB2t et AsVarNuB2t */
/* ----------------------------- */

/* Transformation des parametres estimes a l'etape 3 en parametres actifs
selon les contraintes de l'etape 2 */

APPEL(ModAAct(&(Beta[Etap3].Estim), &(CBeta[Etap2]), &(CBetaE[Etap2]), 
              &BEff2t, &BAct2t));


/* Calcul des derivees des valeurs ajustees a l'etape 3 par rapport
a  BAct2t */
APPEL(DModAAct(CBeta[Etap2].Ctr, Donnees->NbObs, &(Donnees->NbObsC),
                &BAct2t, &(ResStat->FctSensib.DVarYBeta),
                &(Beta[Etap2].Pass), &(CBetaE[Etap2]),
                &DEffB2t, &DVarYB2t));


/* calcul des EquN */
APPEL(CtxNum[Etap2].PCEquN.PCValB(&(Donnees->NbRepet), &(ResStat->Ajustes.Valf),
                            &(ResStat->Ajustes.VarY),
                            &(Donnees->StatDon.S2),
                            &DValf2t, &DVarYT2t, &DVarYB2t,
                            &ValB2t));
APPEL(CtxNum[Etap2].PCEquN.PCValD(&(ResStat->Ajustes.Valf), &DValf2t, &DVarYT2t, &DVarYB2t,
                            &ValD2t));


/* calcul de ValWB2t */
APPEL(CValW(Donnees->NbObsT, &ValB2t, &ValD2t, &ValWB2t));

/* Transformation de ValWB2t */
if (CBeta[Etap2].Ctr == VRAI)
  {
    /* ValWB2t est en dimansion NbAct2,NbAct2 => type=0 */
  APPEL(TrValW(0, NbTEff2, NbBEff2,
               &TAct2t, &BAct2t, &(CThetaE[Etap2]), &(CBetaE[Etap2]), &ValWB2t));
  }


/* Transformation de ValWB2t */
if(GNLControle.SortImp != NULL)
  {
  fprintf(GNLControle.SortImp,
"W2Theta3Beta2\n");
  APPEL(EcrMat(GNLControle.SortImp, &ValWB2t));
  }

/* calcul de AsVarNuB2t */
/* sauvegarde du contexte */
VouluBVarZBP  = GNLControle.Voulu.BVarZBP;
/* mise a FAUX de ces valeurs pour que le calcul soit effectif 
sinon CCovNu considere que le calcul a deja ete fait */
GNLControle.Voulu.BVarZBP=FAUX;

APPEL(CCovNu( ResNum[Etap2].Sigma,
              &(CtxNum[Etap2]), Donnees,
              &ValB2t,&ValWB2t, &(ResStat->Ajustes),
              TypeMu,  &(ResStat->Residus), &(ResStat->Mu3), &(ResStat->Mu4), 
               &BVarZBP2t,
              &AsVarNuB2t, Code));
/* restauration du contexte */
GNLControle.Voulu.BVarZBP = VouluBVarZBP;

if (*Code != OK)
  {
  /* Desallocation des structures de travail dans l'ordre des alloc */
  APPEL(DesAlloueNu3T(
                     &AsVar13, &AsVarNu23, &AsVarNuB2t, &AsVarNuT3t,
                     &BVarZBP2t,
                     &CovZ31t, &CovZ32t, &G13, &G123, &Trav));
  APPEL(DesAlloueNu3W(
                 &ValWB2t, &ValWB3t, &ValWT1t, &ValWT2t, &ValWInv));
  APPEL(DesAlloueNu3P(
                    &TAct1t, &BAct1t, &TAct2t, &BAct2t, &TEff1t, &BEff1t, &TEff2t, &BEff2t));
  APPEL(DesAlloueNu3N(
                     &ValB1t, &ValD1t,
                     &ValB2t, &ValD2t,
                     &ValDB3t,
                     &ValDT2t));
  APPEL(DesAlloueNu3D(
          &DEffT1t, &DEffB1t, &DEffT2t, &DEffB2t,
          &DValf1t, &DValf2t,
          &DVarYT1t,  &DVarYB1t, &DVarYT2t, &DVarYB2t));

  return(OK);
  }

if(GNLControle.SortImp != NULL)
  {
  fprintf(GNLControle.SortImp,
 "SigmaBeta2\n");
  APPEL(EcrMat(GNLControle.SortImp, &AsVarNuB2t));
  }

/* calcul de ValWB3t */
/* ---------------- */
/* calcul de ValDB3t */
APPEL(CDmcb(&(ResStat->Ajustes.Valf), &DValf2t, &DVarYT2t, &DVarYB2t,
                          &ValDB3t));


/* calcul de ValWB3t */
APPEL(CValW(Donnees->NbObsT, &(ResStat->EquN.ValB), &ValDB3t, &ValWB3t));

/* transformation de ValWB3t */
if (CBeta[Etap2].Ctr == VRAI)
  {
    /* ValWB3t est en dimension NbAct3,NbAct2 => type=2 */
  APPEL(TrValW(2, NbTEff2, NbBEff2,
               &TAct2t, &BAct2t, &(CThetaE[Etap2]), &(CBetaE[Etap2]), &ValWB3t));
  }
if(GNLControle.SortImp != NULL)
  {
  fprintf(GNLControle.SortImp,
"W3Beta \n");
  APPEL(EcrMat(GNLControle.SortImp, &ValWB3t));
  }

/* Calcul de ValWT2t */
/* ----------------- */

/* Transformation des parametres estimes a l'etape 3 en parametres actifs
selon les contraintes de l'etape 1 */
APPEL(ModAAct(&(Theta[Etap3].Estim), &(CTheta[Etap1]), &(CThetaE[Etap1]), 
              &TEff1t, &TAct1t));


/* Calcul des derivees des valeurs ajustees a l'etape 3 par rapport
a TAct1t  */
APPEL(DModAAct(CTheta[Etap1].Ctr, Donnees->NbObs, &(Donnees->NbObsC),
                &TAct1t, &(ResStat->FctSensib.DValf),
                &(Theta[Etap1].Pass), &(CThetaE[Etap1]),
                &DEffT1t, &DValf1t));
APPEL(DModAAct(CTheta[Etap1].Ctr, Donnees->NbObs, &(Donnees->NbObsC),
                &TAct1t, &(ResStat->FctSensib.DVarYTheta),
                &(Theta[Etap1].Pass), &(CThetaE[Etap1]),
                &DEffT1t, &DVarYT1t));


/* calcul de ValDT2t */
APPEL(CDmvt(&(ResStat->Ajustes.Valf), &DValf1t, &DVarYT1t, &DVarYB1t,
                            &ValDT2t));

/* calcul de ValWT2t */
APPEL(CValW(Donnees->NbObsT, &ValB2t, &ValDT2t, &ValWT2t));

/* Transformation de ValWT2t */
if (CTheta[Etap1].Ctr == VRAI)
  {
    /* ValWT2t est en dimension NbAct2,NbAct1 => type=1 */
  APPEL(TrValW(1, NbTEff1, NbBEff1,
               &TAct1t, &BAct1t, &(CThetaE[Etap1]), &(CBetaE[Etap1]), &ValWT2t));
  }
if(GNLControle.SortImp != NULL)
  {
  fprintf(GNLControle.SortImp, "W2Theta\n");
  APPEL(EcrMat(GNLControle.SortImp, &ValWT2t));
  }

/* calcul de ValWT1t */
/* ----------------- */

/* calcul des EquN */
APPEL(CtxNum[Etap1].PCEquN.PCValB(&(Donnees->NbRepet), &(ResStat->Ajustes.Valf),
                            &(ResStat->Ajustes.VarY),
                            &(Donnees->StatDon.S2),
                            &DValf1t, &DVarYT1t, &DVarYB1t,
                            &ValB1t));
APPEL(CtxNum[Etap1].PCEquN.PCValD(&(ResStat->Ajustes.Valf), &DValf1t, &DVarYT1t, &DVarYB1t,
                            &ValD1t));

/* calcul de ValWT1t */
APPEL(CValW(Donnees->NbObsT, &ValB1t, &ValD1t, &ValWT1t));

/* Transformation de ValWT1t */
if (CTheta[Etap1].Ctr == VRAI)
  {
    /* ValWT1t est en dimension NbAct1,NbAct1 => type=0 */
  APPEL(TrValW(0, NbTEff1, NbBEff1,
               &TAct1t, &BAct1t, &(CThetaE[Etap1]), &(CBetaE[Etap1]), &ValWT1t));
  }
if(GNLControle.SortImp != NULL)
  {
  /* impression intermediaire */
  fprintf(GNLControle.SortImp, "W1Theta3Beta2\n");
  APPEL(EcrMat(GNLControle.SortImp, &ValWT1t));
  }


/* calcul de G13 */
/* ------------- */

/* calcul de CovZ31t */
if ((CtxNum[Etap1].PCValZ == CZmco) && (CtxNum[Etap3].PCValZ == CZmco))
  {
  APPEL(CVarZmco(Donnees->NbObsT, &(Donnees->NbRepet), &(Donnees->PoidsT),
                &(Donnees->ValY), &(ResStat->Ajustes),
                TypeMu, &(ResStat->Residus), &(ResStat->Mu3), &(ResStat->Mu4),
                &CovZ31t, Code));
  }
else
  {
  APPEL(CVarZmv(Donnees->NbObsT, &(Donnees->NbRepet), &(Donnees->PoidsT),
                &(Donnees->ValY), &(ResStat->Ajustes),
                TypeMu, &(ResStat->Residus), &(ResStat->Mu3), &(ResStat->Mu4),
                &CovZ31t, Code));

if (*Code != OK)
  {
  /* Desallocation des structures de travail dans l'ordre des alloc */
  APPEL(DesAlloueNu3T(
                     &AsVar13, &AsVarNu23, &AsVarNuB2t, &AsVarNuT3t,
                     &BVarZBP2t,
                     &CovZ31t, &CovZ32t, &G13, &G123, &Trav));
  APPEL(DesAlloueNu3W(
                 &ValWB2t, &ValWB3t, &ValWT1t, &ValWT2t, &ValWInv));
  APPEL(DesAlloueNu3P(
                    &TAct1t, &BAct1t, &TAct2t, &BAct2t, &TEff1t, &BEff1t, &TEff2t, &BEff2t));
  APPEL(DesAlloueNu3N(
                     &ValB1t, &ValD1t,
                     &ValB2t, &ValD2t,
                     &ValDB3t,
                     &ValDT2t));
  APPEL(DesAlloueNu3D(
          &DEffT1t, &DEffB1t, &DEffT2t, &DEffB2t,
          &DValf1t, &DValf2t,
          &DVarYT1t,  &DVarYB1t, &DVarYT2t, &DVarYB2t));

  return(OK);
  }
  /* remarque: 
  si PCValZ[Etap1] == CZmco et PCValZ[Etap3] == CZmv,
  dans le calcul de Asvar12 qui suit, on ne considera que les NbObs premieres lignes
  de CovZ21t, les NbObs suivantes seront ignorees 
  
  si PCValZ[Etap1] == CZmv et PCValZ[Etap3] == CZmco,
  dans le calcul de Asvar12 qui suit, on ne considera que les NbObs premieres colonnes
  de CovZ21t, les NbObs suivantes seront ignorees 
  
  */
  }

/* calcul de AsVar13 */
/* pour ameliorer la performance, on transfere les elements de structures */
NbObsT = (TDouble)(Donnees->NbObsT);
valb = ResStat->EquN.ValB.donnees;
covz31t = CovZ31t.donnees;
asvar13 = AsVar13.donnees;
valb1t = ValB1t.donnees;

for (i=0; i < NbAct3 ; i++)
  {
  for (j=0; j < NbAct1; j++)
    {
    asvar13[i][j] = 0.0;
    for (i1=0; i1 < NbZ1; i1++)
      {
      Val = 0.0;
      for (i3=0; i3 < NbZ3; i3++)
        {
        Val = Val + (valb[i3][i] * covz31t[i1][i3]);
        }
      asvar13[i][j] = asvar13[i][j] + (Val * valb1t[i1][j]);
      }
    asvar13[i][j] = asvar13[i][j] / NbObsT;
    }
  }
if(GNLControle.SortImp != NULL)
  {
  fprintf(GNLControle.SortImp, "Sigma13\n");
  APPEL(EcrMat(GNLControle.SortImp, &AsVar13));
  }


/* calcul de G13 */
/* inversion de ValWT1t */
APPEL(InvMat(&ValWT1t));
APPEL(MultMat(&AsVar13, &ValWT1t, &Trav));
APPEL(MultMatT(&Trav, &ValWT2t, &G13));
if(GNLControle.SortImp != NULL)
  {
 fprintf(GNLControle.SortImp, "G13\n");
  APPEL(EcrMat(GNLControle.SortImp, &G13));
  }

/* calcul de G13 - AsVarNu23 */
/* -------------------------- */
/* calcul de CovZ32t */

APPEL(CVarZmv(Donnees->NbObsT, &(Donnees->NbRepet), &(Donnees->PoidsT),
                &(Donnees->ValY), &(ResStat->Ajustes),
                TypeMu, &(ResStat->Residus), &(ResStat->Mu3), &(ResStat->Mu4),
                &CovZ32t, Code));

if (*Code != OK)
  {
  /* Desallocation des structures de travail dans l'ordre des alloc */
  APPEL(DesAlloueNu3T(
                     &AsVar13, &AsVarNu23, &AsVarNuB2t, &AsVarNuT3t,
                     &BVarZBP2t,
                     &CovZ31t, &CovZ32t, &G13, &G123, &Trav));
  APPEL(DesAlloueNu3W(
                 &ValWB2t, &ValWB3t, &ValWT1t, &ValWT2t, &ValWInv));
  APPEL(DesAlloueNu3P(
                    &TAct1t, &BAct1t, &TAct2t, &BAct2t, &TEff1t, &BEff1t, &TEff2t, &BEff2t));
  APPEL(DesAlloueNu3N(
                     &ValB1t, &ValD1t,
                     &ValB2t, &ValD2t,
                     &ValDB3t,
                     &ValDT2t));
  APPEL(DesAlloueNu3D(
          &DEffT1t, &DEffB1t, &DEffT2t, &DEffB2t,
          &DValf1t, &DValf2t,
          &DVarYT1t,  &DVarYB1t, &DVarYT2t, &DVarYB2t));

  return(OK);
  }

/* remarque: si PCValZ[Etap2] == CZmv et PCValZ[Etap3] == CZmco,
dans le calcul de Asvar12 qui suit, on ne considera que les NbObs premieres colonnes
de CovZ32t, les NbObs suivantes seront ignorees
Seul autre cas possible: PCValZ[Etap2] == CZmv et PCValZ[Etap3] == CZmv */

/* calcul de G13 - AsVarNu23 */
/* pour ameliorer la performance, on transfere les elements de structures */
covz32t = CovZ32t.donnees;
asvarnu23 = AsVarNu23.donnees;
valb2t = ValB2t.donnees;
g13 = G13.donnees;

for (i=0; i < NbAct3 ; i++)
  {
  for (j=0; j < NbAct2; j++)
    {
    asvarnu23[i][j] = 0.0;
    for (i2=0; i2 < NbZ2; i2++)
      {
      Val = 0.0;
      for (i3=0; i3 < NbZ3; i3++)
        {
        Val = Val + (valb[i3][i] * covz32t[i2][i3]);
        }
      asvarnu23[i][j] = asvarnu23[i][j] + (Val * valb2t[i2][j]);
      }
    asvarnu23[i][j] =  g13[i][j] - (asvarnu23[i][j] / NbObsT);
    }
  }
if(GNLControle.SortImp != NULL)
  {
  fprintf(GNLControle.SortImp, 
   "G13 - Sigma23\n");
  APPEL(EcrMat(GNLControle.SortImp, AsVarNu23));
  }

/* calcul de G123 */
/* inversion de ValWB2t */
APPEL(InvMat(&ValWB2t));
APPEL(MultMat(&AsVarNu23, &ValWB2t, &Trav));
APPEL(MultMatT(&Trav, &ValWB3t, &G123));

/* calcul de AsVarNu3 */
/* ------------------ */
/* calcul du 1ier terme de la somme :
ValW[Etap3]^-1 * ValWB3t * AsVarNuB2t * ValWB3t^t ValW[Etap3]^-1 */
/* inversion de ResStat->ValW */
APPEL(CopyMat(&(ResStat->ValW), &ValWInv));
APPEL(InvMat(&ValWInv));

APPEL(MultMat(&ValWInv, &ValWB3t, &Trav));
APPEL(MultMat(&Trav, &AsVarNuB2t, &AsVar13));
/* AsVar13 ne servant plus, on l'utilise ici pour stocker les resultats intermediaires */
APPEL(MultMatT(&AsVar13, &ValWB3t, &Trav));
APPEL(MultMat(&Trav, &ValWInv, AsVarNu3));

/* calcul du 3ieme terme :
ValW[Etap3]^-1 * G123 * ValW[Etap3]^-1 */
APPEL(MultMat(&ValWInv, &G123,  &Trav));
APPEL(MultMat(&Trav, &ValWInv, &G123));
/* G123 ne servant plus, on l'utilise ici pour stocker les resultats intermediaires */
Val = (TDouble)2.0 / NbObsT;

/* addition des 3 termes */
/* pour ameliorer la performance, on transfere les elements de structures */
asvarnu3 = AsVarNu3->donnees;
asvarnut3t = AsVarNuT3t.donnees;
g123 = G123.donnees;


for (i=0 ; i < NbAct3; i++)
  {
  for (j=0 ; j <= i; j++)
    {
    asvarnu3[i][j] = asvarnu3[i][j] + asvarnut3t[i][j]  + (g123[i][j] / Val) ;
    asvarnu3[j][i]= asvarnu3[i][j];
    }
  }



/* Desallocation des structures de travail dans l'ordre des alloc */
/* -------------------------------------------------------------- */

/* desallocation des autres structures de travail */
APPEL(DesAlloueNu3T(
                     &AsVar13, &AsVarNu23, &AsVarNuB2t, &AsVarNuT3t,
                     &BVarZBP2t,
                     &CovZ31t, &CovZ32t, &G13, &G123, &Trav));



/* desallocation des structures relatives aux matrices ValW */
APPEL(DesAlloueNu3W(
                 &ValWB2t, &ValWB3t, &ValWT1t, &ValWT2t, &ValWInv));



/* desallocation des structures relatives aux parametres */
APPEL(DesAlloueNu3P(
                    &TAct1t, &BAct1t, &TAct2t, &BAct2t, &TEff1t, &BEff1t, &TEff2t, &BEff2t));


/* desallocation des structures relatives aux equations normales */
APPEL(DesAlloueNu3N(
                     &ValB1t, &ValD1t,
                     &ValB2t, &ValD2t,
                     &ValDB3t,
                     &ValDT2t));

/* desallocation des structures relatives aux derivees */
APPEL(DesAlloueNu3D(
          &DEffT1t, &DEffB1t, &DEffT2t, &DEffB2t,
          &DValf1t, &DValf2t,
          &DVarYT1t,  &DVarYB1t, &DVarYT2t, &DVarYB2t));

return(OK);
}
