/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : MajContP                  |
| Date                 :  7Nov91 / 16:44:46        |
| Derniere mise a jour :                           |
| Concepteur           : A. Bouvier                |
| Role                 : Determiner les contraintes|
|  sur les parametres en dimension differents,     |
|  determiner les nombres de parametres differents,|
|  actifs et a estimer, initialiser les valeurs des|
|  parametres actifs et differents                 |
| Reference conception :                           |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|.
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <stdio.h>
#include <errno.h>
#include <float.h>

#include "nlchoix.h"
#include "nlcodes.h"
#include "nltypes.h"
#include "nlmacros.h"
#include "errcodes.h"


/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/
/* fonctions des autres modules */
TShortInt CopyVect(), CreerMatShort(), CreerVect(),
 ModAAct(), IMaxS();



/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : AffectCE                 |
| Role                  : Allouer eventuellement   |
|                         et affecter les valeurs  |
|                         des contraintes sur les  |
|                         parametres differents    |
| (fonction appelee successivement sur les Theta et|
|  Beta)                                           |
| Parametres d'entree   :                          |
|  NbEff: nombre de parametres differents de la    |
|   sorte consideree (Theta ou Beta)               |
|  ContrM: contraintes sur les parametres en       |
|    dimension multiples                           |
| Parametres de sortie  :                          |
|  ContrE: contraintes sur les parametres en       |
|    dimension differents                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CompCE                    |
| Fonctions appelantes : MajContP                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt AffectCE(NbEff,ContrM, ContrE)
  /* arguments d'entree */
  TShortInt NbEff; /* nombre de parametres differents */
  TContr *ContrM; /* contraintes en dimension multiples */

  /* arguments de sortie */
  TContr *ContrE; /* contraintes en dimension differents */

/*--------------- Fin identification fonction ----*/
  {
/* locals */
/* pointeur sur les fonctions du module appelees */
   TShortInt CompCE();

  /* mettre son nom dans la trace */
  ECRTRACE("AffectCE");

    ContrE->Ctr = VRAI;

  if (ContrM->Ctr == FAUX)
    /* Cas ou il n'y a aucune contrainte */
    {
    /* on a les memes contraintes sur les parametres
       en dimension differents et en dimension multiples 
       c'est-a-dire:
       EgalN, BInf, BSup ont 0 elements et EgalP est une suite
       croissante 
       et nombre de parametres multiples= nbre de param differents */
    ContrE->EgalN =  ContrM->EgalN;
    ContrE->BInf =  ContrM->BInf;
    ContrE->BSup =  ContrM->BSup;
    } /* fin du cas sans contrainte */
  else
    {
    /* cas ou il y a une contrainte */
    /* Allocation  et affectation des vecteurs composants les
      contraintes sur parametres differents */

    /* EgalN */
    APPEL(CompCE(NbEff, &(ContrM->EgalP), &(ContrM->EgalN), &(ContrE->EgalN)));

    /* BInf */
    APPEL(CompCE(NbEff, &(ContrM->EgalP), &(ContrM->BInf), &(ContrE->BInf)));

    /* BSup */
    APPEL(CompCE(NbEff, &(ContrM->EgalP), &(ContrM->BSup), &(ContrE->BSup)));

    } /* fin du cas ou il y a une contrainte */

  /* retour */
  return(OK);
  }


/*--------------- Identification fonction ----------
| Nom de la fonction    : AffectPass               |
| Role                  : Allouer et affecter la   |
|                         matrice de passage des   |
|                         parametres de la         |
|                         dimension multiples      |
|                         a la dimension differents|
| (fonction appelee successivement sur les Theta et|
|  Beta)                                           |
| Parametres d'entree   : NbEff= nombre de         |
|                           parametres differents  |
|                         EgalP= pointeur sur le   |
|                           vecteur des contraintes|
|                           d'egalite entre        |
|                           parametres             |
| Parametres de sortie  : Pass= pointeur sur la    |
|                         matrice de passage des   |
|                         parametres de la         |
|                         dimension "multiple" a la|
|                         dimension "differents"   |
| Retour fonction       : OK ou code d'erreur      |
|    renvoye par l'intermediaire de APPEL          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CreerMatShort             |
| Fonctions appelantes : MajContP                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt AffectPass(NbEff,EgalP,Pass)
  /* arguments d'entree */
  TShortInt NbEff; /* nombre de parametres differents */
  TVectShort *EgalP; /* vecteur des contraintes d'egalite entre parametres */

  /* arguments de sortie */
  TMatShort *Pass; /* matrice de passage des parametres en dimension
                     "multiples" a la dimension "differents" */

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TShortInt  i, j, nblig;
  /* pointeurs sur des elements de structure pour ameliorer la performance */
  TShortInt **pass;
  TShortInt *egalp;

  /* mettre son nom dans la trace */
  ECRTRACE("AffectPass");


  /* Allocation de Pass */
  /* On cree la matrice, meme s'il n'y a pas de parametres de la sorte
     consideree (Theta ou Beta) dans le modele ou differents: le nombre
     de lignes ou colonnes sera alors nul */
  nblig = EgalP->nbele;
  APPEL(CreerMatShort(nblig, NbEff, Pass));
  pass = Pass->donnees;
 
  /* Initialisation a 0 de Pass */
  if ((nblig > 0) && (NbEff > 0))
    {
    for(i = 0; i < nblig; i++)
      for(j = 0; j < NbEff; j++)
        {
        pass[i][j] =  (TShortInt)0;
        }

    /* Dans la ligne "i" correspondant a un parametre multiple,
       on met 1 dans la colonne qui correspond a l'indice du
       parametre different auquel il est egal */
    egalp = EgalP->donnees;
    for(i = 0; i < nblig; i++)
      {
      pass[i][egalp[i] - 1] = (TShortInt)1;
      }
    }

/* retour */
  return(OK);
  }



/*--------------- Identification fonction ----------
| Nom de la fonction    : CNbAct                   |
| Role                  : Calculer le nombre de    |
| parametres actifs, a partir du nombre de         |
| parametres differents et des contraintes sur les |
| parametres differents                            |
| (fonction appelee successivement sur les Theta et|
|  Beta, s'il y a des contraintes d'egalite        |
|  numerique sur ceux-ci)                          |
| Parametres d'entree   :                          |
|  EgalN= pointeur sur le vecteur des egalites     |
|   numeriques sur les parametres differents       |
|   Son nombre d'elements doit etre positif        |
| Parametres de sortie  :                          |
|  NbAct: nombre de parametres actifs              |
|     (eventuellement nul)                         |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Fonctions appelantes : MajContP                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt CNbAct(EgalN, NbAct)
  /* arguments d'entree */
  TVect *EgalN; 

  /* arguments de sortie */
  TShortInt *NbAct;

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TShortInt IndE; /* indice du parametre courant en dim differents */
  TShortInt NbP;

  /* mettre son nom dans la trace */
  ECRTRACE("CNbAct");

  *NbAct =  EgalN->nbele; /* c.a.d nombre de parametres differents */
  NbP = EgalN->nbele;
  for (IndE = 0; IndE < NbP; IndE++)
    {
    if (!myisnan(EgalN->donnees[IndE]) )
      {
      *NbAct = *NbAct - 1;
      }
    }

  /* retour */
  return(OK);
  }


/*--------------- Identification fonction ----------
| Nom de la fonction    : CompCE                   |
| Role                  : Allouer                  |
|                         et affecter les valeurs  |
|                         a un composant d'une     |
|                         contrainte sur les       |
|                         parametres differents    |
| (fonction appelee successivement sur les Theta et|
|  Beta et, pour chacun d'eux, pour chacun des     |
|  composants des contraintes en dimension         |
|  differents)                                     |
| Parametres d'entree   :                          |
|  NbEff=nombre de parametres differents           |
|  EgalP= pointeur sur le vecteur des egalites     |
|   entre parametres                               |
|  CompM= pointeur sur le composant (# EgalP)      |
|    d'une contrainte en dimension multiples       |
| Parametres de sortie  :                          |
|  CompE= pointeur sur le composant (# EgalP)      |
|    correspondant a CompM en dimension differents |
| Retour fonction       : OK ou code d'erreur      |
|    renvoye par l'intermediaire de APPEL          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CreerVect                 |
| Fonctions appelantes : AffectCE                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt CompCE(NbEff, EgalP, CompM, CompE)
  /* arguments d'entree */
  TShortInt NbEff;
  TVectShort *EgalP; 
  TVect *CompM;

  /* arguments de sortie */
  TVect *CompE;

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TShortInt IndE; /* indice du parametre courant en dim differents */
  TShortInt IndM; /* indice du parametre courant en dim multiple */
  TShortInt n;
/* pointeurs sur des elements de structure pour ameliorer la performance */
  TShortInt *egalp;
  TDouble *compm, *compe;

  /* mettre son nom dans la trace */
  ECRTRACE("CompCE");

  if(CompM->nbele <= 0)
    {
    /* pas de contrainte de ce type : le composant est alloue
       avec 0 elements */
    APPEL(CreerVect((TShortInt)0, CompE));
    } /* fin du cas pas de contrainte de ce type */

   else
    {
    /* Il existe une contrainte:
     On recopie la valeur de la contrainte de chaque
    parametre en dimension multiples dans l'element
    qui correspond dans les contraintes en dim differents */

    APPEL(CreerVect(NbEff, CompE));
    n = EgalP->nbele;
    egalp = EgalP->donnees;
    compe = CompE->donnees;
    compm = CompM->donnees;
    IndE = 0;
    for (IndM = 0; IndM < n; IndM++)
      {
      if(egalp[IndM] > IndE)
        {
        compe[IndE] = compm[IndM];
        IndE = IndE + 1;
        }
      }
    } /* cas du cas avec contrainte */


  /* retour */
  return(OK);
  }

  

/*--------------- Identification fonction ----------
| Nom de la fonction    : MajContP                 |
| Role                  : - Determiner les         |
|  contraintes sur les parametres en dimension     |
|  differents                                      |
|                         - Determiner les nombres |
|  de parametres differents,actifs et a estimer    |
|                         - Initialiser les valeurs|
|  des parametres actifs et differents             |
|  (fonction appelee successivement sur les Theta  |
|   et Beta)                                       |
| Parametres d'entree   :                          |
|  ContrM= pointeur  sur les contraintes en        |
|    dimension multiples de l'etape                |
| Parametres d'e./s.    :                          |
|  Param= pointeur sur les parametres              |
|   Les composants "Init" sont une entree          |
|   (valeurs initiales pour l'etape)               |
|   les composants "Eff", "Act", "Estim.nbele",    |
|   "Pass" sont des sorties                        |
| Parametres de sortie  :                          |
|  ContrE= pointeur  sur les contraintes           |
|   sur les parametres differents                  |
| Retour fonction       : OK ou code d'erreur      |
|    renvoye par l'intermediaire de APPEL          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                        AffectCE,                 |
|                        AffectPass,               |
|                        CNbAct,                   |
|                        CopyVect,                 |
|                        CreerVect,                |
|                        IMaxS,                    |
|                        ModAAct                   |
| Fonctions appelantes : NLEtape                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt MajContP( ContrM, Param, ContrE)

  /* arguments d'entree */
  TContr *ContrM; /* contraintes de l'etape */

  /* arguments d'entree-sortie */
  TParam *Param;
  /* Les composants "Init" sont une entree (valeurs initiales pour l'etape)
     les composants "Eff", "Act", "Estim.nbele", "Pass" sont des sorties */

  /* arguments de sortie */
  TContr *ContrE; /* contraintes sur les parametres differents */


/*--------------- Fin identification fonction ----*/

  {
  /* locals */
  TShortInt NbAct;


  /* mettre son nom dans la trace */
  ECRTRACE("MajContP");


  /* Allocation des parametres estimes:
     leur nombre est egal a ceux de Init, c'est-a-dire:
     nbre de param dans le modele * nbre de courbes */
  APPEL(CreerVect(Param->Init.nbele, &(Param->Estim)));
  /* initialisation: Estim=Init */
  APPEL(CopyVect(&(Param->Init), &(Param->Estim)));

  /* Allocation des parametres differents:
     leur nombre est egal a l'indice le plus eleve des parametres
     differents */

  if (ContrM->EgalP.nbele >0)
    {
    APPEL(CreerVect(IMaxS(&(ContrM->EgalP)), &(Param->Eff)));
    }
  else
    {
    APPEL(CreerVect( (TShortInt)0,  &(Param->Eff)));
    }


  /* Calcul de la matrice de passage de la dimension 
   modele * nbre de courbes a la  dimension differents */

  APPEL(AffectPass(Param->Eff.nbele  ,&(ContrM->EgalP), &(Param->Pass)));


  /* Calcul des contraintes sur les parametres differents */
  APPEL( AffectCE(Param->Eff.nbele, ContrM, ContrE ));

  /* Calcul du nombre de parametres actifs */
  if (ContrE->EgalN.nbele > 0)
    {
    /* il y a des contraintes d'egalite numerique */
    APPEL( CNbAct(&(ContrE->EgalN)  , &NbAct));
    }
  else
    {
    /* pas de contrainte d'egalite numerique:
     nombre de parametres actifs= nombre de parametres differents */
    NbAct = Param->Eff.nbele;
    }

  /* Allocation des composants Act*/
  APPEL(CreerVect(NbAct, &(Param->Act)));

  /* Transformation des valeurs initiales en dimension differents
     et actifs */
  if (NbAct >0)
    {
    APPEL(ModAAct( &(Param->Init),ContrM, ContrE,&(Param->Eff), &(Param->Act) ));
    }

  /* retour */
  return(OK);
  }

