

/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : NLAutres                  |
| Date                 : 1992                      |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : A. Bouvier                |
| Role                 : calculer les sorties      |
|   non calculees lors du  processus iteratif      |
| Reference conception :                           |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <stdio.h>
#include <float.h>

#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlglobal.h"
#include "nlmacros.h"
#include "errcodes.h"
#include "Rmath.h"
#include "prognmath.h"


/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/
TDouble pow();
TDouble sin();
TDouble sqrt();

/* fonctions des autres modules */
TShortInt CCov2(), CCov3(), CopyMat(), CreerMat(),  CreerVect(),
          CLv(), CScrNP(), CScrPv(), CValR(),
          DivMatVal(), GerMessage(), InvMat(),
          MultMatVal(),  VNuAVMod();

void  DetruVect(), DetruMat();
TShortInt TrValW(TShortInt type,
                 TShortInt NbTEff, TShortInt NbBEff, TVect *TAct, TVect *BAct,
		 TContr *CThetaE, TContr *CBetaE,
                 TMat *ValW);

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/




/*--------------- Identification fonction ----------
| Nom de la fonction    : CBVarZBP                 |
| Role                  : Creer et affecter des    |
|  valeurs a la matrice BVarZBP                    |
|  BVarZBP= (ValB^T * VarZ * ValB) / NbObsT        |
| Parametres d'entree   :                          |
|  ValB: valeur de l'equation B                    |
|  Ajustes: valeurs ajustees de Y et de sa variance|
|  Donnees: les donnees                            |
|   dont on utilise en entree:                     |
|   NbObs, NbObsT, NbRepet, PoidsT, ValY           |
|  CtxNum: contexte numerique                      |
|   dont on utilise NbZ, et PCVarZ                 |
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
|  Residus: les residus                            |
|  Mu3, Mu4: les moments d'ordre 3 et 4            |
| Parametres de sortie  :                          |
|  BVarZBP (non allouee avant l'appel)             |
|  Code : OK ou code de PCVarZ                     |
| Effets de bord    :                              |
| si PCVarz=CVarZmv:                               |
|   - Si  GNLControle.Voulu.Residus =VRAI          |
|      Residus en entree contient les residus      |
|      sinon, Residus est ignore en entree, et, en |
|      sortie, il  contient les residus            |
|   - Si  GNLControle.Voulu.Mu =VRAI               |
|      Mu3, Mu4 en entree contiennent les moments  |
|      sinon, ils sont ignores en entree, et, en   |
|      sortie, il  contient les moments            |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CreerMat, PCVarZ         |
| Fonctions appelantes :  CCovNu, NLAutres         |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/



TShortInt CBVarZBP( ValB, Ajustes, Donnees, CtxNum,
              TypeMu, Residus, Mu3, Mu4,
              BVarZBP,
              Code)

/* arguments d'entree */
TMat *ValB;
TAjustes *Ajustes;
TDonnees *Donnees;
TCtxNum *CtxNum;

/* arguments d'entree-sortie */
TShortInt *TypeMu;
TVect *Residus, *Mu3, *Mu4;

/* arguments de sortie */
TMat *BVarZBP;
TShortInt *Code;

/*--------------- Fin identification fonction ----*/
  {
  /* locals */
  TShortInt  NbObs, NbAct, Dim, i, j, k, l;
  TMat VarZ; /* dimension NbObs*NbZ, NbObs*NbZ */
  TDouble Val;
  TLongInt NbObsT;

  /* pointeurs sur des elements des structures passees en argument */
  TDouble **bvarzbp; /* dimension NbAct, NbAct */
  TDouble **valb; /* dimension NbObs*NbZ, NbAct */


  /* mettre son nom dans la trace */
  ECRTRACE("CBVarZBP");

  *Code = OK;
  NbAct = ValB->nbcol;
  NbObs = Donnees->NbObs;

  /* allouer la sortie */
  APPEL(CreerMat(NbAct, NbAct, BVarZBP));

  Dim = NbObs * CtxNum->NbZ;


  /* calculer VarZ */
  /* allouer la sortie */
  APPEL(CreerMat(Dim, Dim, &VarZ));
 
  APPEL(CtxNum->PCVarZ( Donnees->NbObsT, &(Donnees->NbRepet), &(Donnees->PoidsT), 
                        &(Donnees->ValY), Ajustes, 
                        TypeMu, Residus, Mu3, Mu4,
                        &VarZ, Code));

  if (*Code != OK)
    {
    return(OK);
    }


  bvarzbp = BVarZBP->donnees;
  valb = ValB->donnees;

  NbObsT = Donnees->NbObsT;

  /* on calcule la moitie inferieure gauche: */
  for (i=0; i< NbAct; i++)
    {
    for(k=0; k<= i; k++)
      {
      bvarzbp[i][k]= (TDouble)ZERO;
      for(j=0; j<Dim; j++)
        {
        Val=(TDouble)ZERO;
        for (l=0; l<Dim; l++)
          {
          Val = Val + (valb[l][i] * VarZ.donnees[l][j]);
          } /* fin boucle sur l */
        bvarzbp[i][k]=  bvarzbp[i][k] + (Val * valb[j][k]);
        } /* fin boucle sur j */
      bvarzbp[i][k]= bvarzbp[i][k] / (TDouble)NbObsT;

      /* on met la valeur dans l'element symetrique qui lui correspond */ 
      bvarzbp[k][i] = bvarzbp[i][k]; 
      } /* fin boucle sur k */
    } /* fin boucle sur i*/

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CCorr                    |
| Role                  : calculer la matrice de   |
|  correlation asymptotique                        |
| Parametres d'entree   :                          |
|  Etape: indice de l'etape courante               |
|   compte a partir de 1                           |
|  Donnees: les donnees                            |
|  Theta: la structure des parametres Theta        |
|  Beta: la structure des parametres Beta          |
|  CTheta, CBeta: les contraintes sur les          |
|   parametres du modele                           |
|  CThetaE: les contraintes sur les parametres     |
|   differents Theta                               |
|  CBetaE: les contraintes sur les parametres      |
|   differents Beta                                |
|  CtxNum: contexte numerique                      |
|  ResNum: resultats statistiques                  |
|   A l'exception de Donnees, tous les arguments   |
|   sont des tableaux de NbEtapes structures       |
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
|  ResStat: les resultats statistiques de l'etape  |
|     courante,
|     dont, en sortie, on utilise                  |
|     Corr: la matrice des  correlations en        |
|     dimension ``modele*NbCourbe''                |
|     non allouee avant l'appel                    |
| Effets de bord    :                              |
|   - Si  GNLControle.Voulu.Residus =VRAI          |
|   et que PCVarZ=CVarZmv,                         |
|      Residus en entree contient les residus      |
|      sinon, Residus est ignore en entree, et, en |
|      sortie, il  contient les residus            |
|   - Si  GNLControle.Voulu.Mu =VRAI               |
|   et que PCVarZ=CVarZmv,                         |
|      Mu3, Mu4 en entree contiennent les moments  |
|      sinon, ils sont ignores en entree, et, en   |
|      sortie, il  contient les moments            |
|   - Si  GNLControle.Voulu.BVarZBP =VRAI          |
|      BVarZBP est suppose contenir en entree le   |
|      produit des matrices correspondant,         |
|      sinon,  et si Effic=FAUX, ce produit est    |
|      calcule  en sortie                          |
|   - Si GNLControle.Voulu.AsVar = VRAI,           |
|  AsVar est suppose contenir la matrice des       |
|  variances sinon, celle-ci est calculee en sortie|
|  Ces effets de bord concernent l'etape courante  |
| Parametres de sortie  :                          |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CCov1, CCov2, CCov3      |
|                         CreerMat                 |
|  plus les fonctions systeme:                     |
|    sqrt
| Fonctions appelantes :  NLAutres                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CCorr(Etape, Donnees,Theta, Beta,CTheta, CBeta, CThetaE, CBetaE, CtxNum, ResNum,
                 TypeMu,  ResStat)


/* arguments d'entree */
TShortInt Etape;
TDonnees *Donnees;
TParam Theta[] , Beta[];
TContr CTheta[], CBeta[];
TContr CThetaE[], CBetaE[];
TCtxNum CtxNum[];
TResNum ResNum[];

/* arguments d'entree-sortie */
TShortInt *TypeMu;
TResStat *ResStat;



/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt NbMod, i,j, IndEtap;
TShortInt MonCode;
TDouble Valeur;

/* pointeur sur les fonctions du module appelees */
TShortInt CCov1();

/* pointeurs pour ameliorer la performance */
TDouble **Corr, **AsVar;

/* Ecriture de la trace */
ECRTRACE("CCorr");

MonCode  = OK;
IndEtap = Etape - 1; /* indice dans les tableaux */

/* calcul de AsVar si necessaire */
if (GNLControle.Voulu.AsVar == FAUX)
  {
  if (Etape ==1)
    {
    APPEL(CCov1( ResNum[IndEtap].Sigma,Donnees,&(Theta[IndEtap]), &(Beta[IndEtap]),
                 &(CTheta[IndEtap]), &(CBeta[IndEtap]),
                 &(CThetaE[IndEtap]), &(CBetaE[IndEtap]),
                 &(CtxNum[IndEtap]),
                 TypeMu,  ResStat, &MonCode));
    }
  if (Etape  == 2)
    {
    APPEL(CCov2(Donnees,Theta, Beta,CTheta, CBeta, CThetaE, CBetaE, CtxNum, ResNum,
                 TypeMu,  ResStat, &MonCode));
    }
  if (Etape == 3)
    {
    APPEL(CCov3(Donnees,Theta, Beta,
                CTheta, CBeta,
                CThetaE, CBetaE, CtxNum, ResNum,
                 TypeMu,  ResStat, &MonCode));
    } /* fin du if (Etape == 3)  */
  } /* fin du Voulu.AsVar= FAUX */

/* si AsVar a une dimension nulle ou que MonCode est non OK,
   c'est qu'on ne sait pas la calculer */
  if ((ResStat->AsVar.nblig == 0) || (MonCode != OK))
    {
    /*
    printf("On ne sait pas calculer la variance asymptotique, donc la matrice de correlation\n");
    */

    NLWARNING((WARCORR,1,"CCorr",WNUMER));
    APPEL(CreerMat( (TShortInt)0,  (TShortInt)0, &(ResStat->Corr)));
    }
  else
    {
    /* allocation de la sortie */
    NbMod = ResStat->AsVar.nblig;
    APPEL(CreerMat( NbMod,  NbMod, &(ResStat->Corr)));
    Corr = ResStat->Corr.donnees;
    AsVar = ResStat->AsVar.donnees;

    for (i=0; i<NbMod; i++)
      {
      for (j=0; j<=i; j++)
        {
        if ((myisnan(AsVar[i][j])) || (myisnan(AsVar[i][i])) || (myisnan(AsVar[j][j])))
          {
          Corr[i][j] = mysignaling_nan(n);
          }
        else
          {
          Valeur = AsVar[i][i] * AsVar[j][j];
          if ( Valeur < (TDouble)0)
            Corr[i][j] = mysignaling_nan(n);
          else
            Corr[i][j] = AsVar[i][j] / sqrt(Valeur);
          }
        Corr[j][i] = Corr[i][j];
        } /* fin boucle sur j */
      } /* fin boucle sur i */
    } /* fin du else AsVar.nblig=0 */

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CCovNu                   |
| Role                  : calculer la matrice de   |
|  variance-covariance asymptotique des parametres |
|  avant prise en compte des contraintes           |
| Parametres d'entree   :                          |
|  Sigma: la valeur de sigma**2                    |
|  CtxNum: contexte numerique                      |
|  Donnees: les donnees                            |
|  ValB: equation normale B                        |
|  ValW: la matrice W                              |
|  Ajustes: valeurs ajustees de Y et de sa variance|
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
|  Residus: les residus,                           |
|  Mu3, Mu4: les moments d'ordre 3 et 4            |
|  BVarZBP: la matrice  BVarZBP                    |
| Parametres de sortie  :                          |
|  AsVarNu: la matrice resultat                    |
|   allouee avant l'appel                          |
|  Code = OK ou code de CBVarZBP                   |
| Effets de bord    :                              |
|   - Si  GNLControle.Voulu.Residus =VRAI          |
|   et que PCVarZ=CVarZmv,                         |
|      Residus en entree contient les residus      |
|      sinon, Residus est ignore en entree, et, en |
|      sortie, il  contient les residus            |
|   - Si  GNLControle.Voulu.Mu =VRAI               |
|   et que PCVarZ=CVarZmv,                         |
|      Mu3, Mu4 en entree contiennent les moments  |
|      sinon, ils sont ignores en entree, et, en   |
|      sortie, il  contient les moments            |
|   - Si  GNLControle.Voulu.BVarZBP =VRAI          |
|      BVarZBP est suppose contenir en entree le   |
|      produit des matrices correspondant,         |
|      sinon,  et si Effic=FAUX, ce produit est    |
|      calcule  en sortie                          |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                       CBVarZBP,                  |
|                       CopyMat,                   |
|                       CreerMat,                  |
|                       DetruMat,                  |
|                       DivMatVal,                 |
|                       InvMat,                    |
|                       MultMatVal,                |
| Fonctions appelantes :  CCov1 , CCovNu2          |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CCovNu( Sigma,
                  CtxNum, Donnees,
                 ValB, ValW, Ajustes,
                 TypeMu,  Residus, Mu3, Mu4, BVarZBP,
                 AsVarNu, Code)


/* arguments d'entree */
TDouble Sigma;
TCtxNum *CtxNum;
TDonnees *Donnees;
TMat *ValB, *ValW;
TAjustes *Ajustes;

/* arguments d'entree-sortie */
TShortInt *TypeMu;
TVect *Residus, *Mu3, *Mu4;
TMat  *BVarZBP;

/* arguments de sortie */
TMat *AsVarNu; /* dimension NbAct, NbAct */
TShortInt *Code;


/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt NbAct, i, j ,k ,l;
TMat ValWInv;
TDouble Val, NbObsT;


/* pointeurs sur des elements des structures passees en argument */
TDouble **bvarzbp; /* dimension NbAct, NbAct */
TDouble **asvarnu;

/* Ecriture de la trace */
ECRTRACE("CCovNu");

*Code = OK;

NbAct = ValW->nblig;
NbObsT = (TDouble)(Donnees->NbObsT);

/* inversion de ValW */
APPEL(CreerMat(NbAct, NbAct, &ValWInv));
APPEL(CopyMat(ValW, &ValWInv));
APPEL(InvMat(&ValWInv));

if (CtxNum->Effic==VRAI)
  {
  APPEL(DivMatVal(&ValWInv, NbObsT, AsVarNu));

  if (CtxNum->Estim == MCOT)
    APPEL(MultMatVal(AsVarNu, Sigma, AsVarNu));
  }
else
  {
  /* calcul de BVarZBP si non deja fait */
  if (GNLControle.Voulu.BVarZBP==FAUX)
    {
    APPEL(CBVarZBP(ValB, Ajustes, Donnees,  CtxNum,
              TypeMu, Residus, Mu3, Mu4, BVarZBP, Code));
    if (*Code != OK)
      {
      return(OK);
      }
    }
  /* calcul de AsVarNu */
  bvarzbp = BVarZBP->donnees;
  asvarnu = AsVarNu->donnees;

  for (i=0; i<NbAct; i++)
    {
    for (l=0; l<= i; l++)
      {
      asvarnu[i][l]=(TDouble)ZERO;
      for(j=0; j<NbAct; j++)
        {
        Val=0;
        for(k=0; k<NbAct; k++)
          {
          Val=Val+ (ValWInv.donnees[i][k] * bvarzbp[k][j]);
          } /* fin boucle sur k*/
        asvarnu[i][l]= asvarnu[i][l] + (Val * ValWInv.donnees[j][l]);
        } /* fin boucle sur j */
      asvarnu[i][l]= asvarnu[i][l] / NbObsT;
      asvarnu[l][i]= asvarnu[i][l];
      } /* fin boucle sur l */
    } /* fin boucle sur i */
    
  } /* fin du Effic ==FAUX */

DetruMat( &ValWInv);


return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : CCov1                    |
| Role                  : calculer la matrice de   |
|  variance-covariance asymptotique                |
|  quand Etape =1 (estimation des Theta ou des     |
|   Theta et des Beta simultanement)               |
| Parametres d'entree   :                          |
|  Sigma: valeur de sigma2                         |
|  Donnees: les donnees                            |
|  Theta: la structure des parametres Theta        |
|  Beta: la structure des parametres Beta          |
|  CTheta, CBeta: les contraintes sur les          |
|   parametres du modele                           |
|  CThetaE: les contraintes sur les parametres     |
|   differents Theta                               |
|  CBetaE: les contraintes sur les parametres      |
|   differents Beta                                |
|  CtxNum: contexte numerique                      |
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
|  ResStat: les resultats statistiques             |
| Parametres de sortie :                           |
|  Code = OK ou code de CCovNu                     |
| Effets de bord    :                              |
|   - Si  GNLControle.Voulu.Residus =VRAI          |
|   et que PCVarZ=CVarZmv,                         |
|      Residus en entree contient les residus      |
|      sinon, Residus est ignore en entree, et, en |
|      sortie, il  contient les residus            |
|   - Si  GNLControle.Voulu.Mu =VRAI               |
|   et que PCVarZ=CVarZmv,                         |
|      Mu3, Mu4 en entree contiennent les moments  |
|      sinon, ils sont ignores en entree, et, en   |
|      sortie, il  contient les moments            |
|   - Si  GNLControle.Voulu.BVarZBP =VRAI          |
|      BVarZBP est suppose contenir en entree le   |
|      produit des matrices correspondant,         |
|      sinon,  et si Effic=FAUX, ce produit est    |
|      calcule  en sortie                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         CCovNu,                  |
|                         CreerMat,                |
|                         DetruMat,                |
|                         VNuAVMod                 |
| Fonctions appelantes :  NLAutres                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CCov1( Sigma, Donnees,Theta, Beta,CTheta, CBeta, CThetaE, CBetaE, CtxNum,
                 TypeMu,  ResStat,
                 Code)


/* arguments d'entree */
TDouble Sigma;
TDonnees *Donnees;
TParam *Theta , *Beta;
TContr *CTheta, *CBeta;
TContr *CThetaE, *CBetaE;
TCtxNum *CtxNum;

/* arguments d'entree-sortie */
TShortInt *TypeMu;
TResStat *ResStat;
 /* argument de sortie */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TMat AsVarNu; /* matrice de variance asymptotique en dimension actifs */

TShortInt NbAct, NbTEff, NbBEff;

/* pointeur sur les fonctions du module appelees */
TShortInt CCovNu();

/* Ecriture de la trace */
ECRTRACE("CCov1");

*Code = OK;
NbTEff = Theta->Eff.nbele;
NbBEff = Beta->Eff.nbele;

/* allocation de AsVarNu */
NbAct = Theta->Act.nbele + Beta->Act.nbele;
APPEL(CreerMat(NbAct, NbAct, &AsVarNu));

/* calcul de la matrice de variance asymptotique avant prise en compte
des contraintes numeriques */

APPEL(CCovNu( Sigma,
              CtxNum, Donnees,
             &(ResStat->EquN.ValB), &(ResStat->ValW), &(ResStat->Ajustes),
             TypeMu,  &(ResStat->Residus), &(ResStat->Mu3), &(ResStat->Mu4), 
             &(ResStat->BVarZBP),
             &AsVarNu, Code));

if (*Code == OK)
  {
  /* le calcul a pu etre effectue */
  /* Tenir compte des contraintes numeriques */
  if ((CTheta->Ctr == VRAI) || (CBeta->Ctr == VRAI))
    {
    APPEL(TrValW(0, NbTEff, NbBEff,
               &(Theta->Act), &(Beta->Act), CThetaE, CBetaE, &AsVarNu));

    }



  /* transformation en dimension parametres du modele */
  APPEL(VNuAVMod( &AsVarNu, Theta, Beta, CTheta, CBeta,  CThetaE, CBetaE,
                 &(ResStat->AsVar)));
  }

/* desallocation de AsVarNu */

DetruMat(&AsVarNu);



/* retour */
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : CMu                      |
| Role                  :                          |
|  calculer les moments d'ordre 3 et 4             |
| Parametres d'entree   :                          |
|  NbObsT: nombre d'observations en comptant les   |
|   repetitions                                    |
|  NbRepet: vecteur des repetitions                |
|  PoidsT: poids sur chacune des repetitions       |
|   y comprises celles de poids nul                |
|  ValY: valeurs observees de la reponse f         |
|   sur chacune des repetitions y comprises celles |
|   de poids nul                                   |
|  Valf: valeurs ajustees de f                     |
|  VarY: variance de Y ajustee                     |
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
|  Residus : les residus non reduits               |
|  ceux-ci sont calcules  uniquement si necessaires|
|  c'est-a-dire si TypeMu=MURES ou MURESREPET      |
|  et si pas deja calcules, c'est-a-dire, si       |
|  GNLControle.Voulu.Residus=FAUX                  |
| Parametres de sortie  :                          |
|  Mu3 : moments d'ordre 3                         |
|  Mu4 : moments d'ordre 4                         |
|   vecteurs non alloues avant l'appel             |
| Effets de bord    :                              |
|   - Si  GNLControle.Voulu.Residus =VRAI          |
|      Residus en entree est suppose contenir les  |
|      residus                                     |
|      sinon, Residus est ignore en entree, et, en |
|      sortie, il  contient les residus            |
|      si TypeMu=MURES ou MURESREPET               |
| Retour fonction       : OK ou ERRCMU             |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CreerVect, CResidus      |
|  plus la fonction systeme pow                    |
| Fonctions appelantes :  CVarZmv, NLAutres        |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CMu( NbObsT, NbRepet, PoidsT, ValY, Valf, VarY,
              TypeMu, Residus,
              Mu3, Mu4)

/* arguments d'entree */
TLongInt NbObsT;
TVectLong *NbRepet;
TVect *PoidsT, *ValY, *Valf, *VarY;

/* arguments d'entree-sortie */
TShortInt *TypeMu;
TVect *Residus;

/* arguments de sortie*/
TVect *Mu3, *Mu4;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt NbObs, i,j ,k,ksv;
TDouble Moy, Val, Val3, Val4;
TChar Mess1[5];

/* pointeur sur les fonctions du module appelees */
TShortInt CResidus();

/* pointeurs sur des elements de structure pour ameliorer la performance */
TLongInt *nbrepet;
TDouble  *vary, *residus, *mu3, *mu4;

/* Ecriture de la trace */
ECRTRACE("CMu");

if ((*TypeMu == MUGAUSS) || (*TypeMu == MURES) || (*TypeMu == MURESREPET))
  {
  /* allouer les sorties */
  NbObs = NbRepet->nbele;
  APPEL(CreerVect(NbObs, Mu3));
  APPEL(CreerVect(NbObs, Mu4));

  /* allocation de pointeurs pour la performance */
  mu3 = Mu3->donnees;
  mu4 = Mu4->donnees;
  vary = VarY->donnees;
  nbrepet = NbRepet->donnees;

  switch(*TypeMu)
    {
    case MUGAUSS :
      for(i = 0; i<NbObs; i++)
        {
        mu3[i] = (TDouble)ZERO;
        mu4[i] = (TDouble)3 * (vary[i] * vary[i]);
        }
      break;
    case MURES :
      /* estimation par les residus, la variance etant supposee constante */
      /* calcul des residus, si non deja fait */
      if (GNLControle.Voulu.Residus==FAUX)
        {
        APPEL(CResidus(NbObsT, NbRepet, PoidsT, ValY,  Valf, Residus));
        }
      /* calcul de la moyenne des residus */
      residus = Residus->donnees;
      Moy=(TDouble)ZERO;
      for(i = 0; i<NbObsT; i++)
        {
        Moy = Moy + residus[i];
        }
      Moy = Moy / (TDouble)NbObsT;
     /* calcul de somme des (residus - Moy) a la puissance 3 et 4 */
      Val3 = (TDouble)ZERO;
      Val4 = (TDouble)ZERO;
      for(i = 0; i<NbObsT; i++)
        {
        Val = residus[i] - Moy;
        Val3 =  Val3 + pow( Val, (TDouble)3);
        Val4 = Val4 + pow( Val, (TDouble)4);
        }
      
     /* calcul des Mu: ils ont tous la meme valeur */
      Val3 = Val3 / (TDouble)NbObsT;
      Val4 = Val4 / (TDouble)NbObsT;

      for(i = 0; i<NbObs; i++)
        {
        mu3[i] = Val3;
        mu4[i] = Val4;
        } /* fin boucle sur i */

      break;
    case MURESREPET :
    /* estimation par les residus en tenant compte des repet */
      /* calcul des residus, si non deja fait */
      if (GNLControle.Voulu.Residus==FAUX)
        {
        APPEL(CResidus(NbObsT, NbRepet, PoidsT, ValY, Valf, Residus));
        }
      /* calcul de la moyenne des residus sur les repetitions */
      residus = Residus->donnees;
      k=0;
      for(i = 0; i<NbObs; i++)
        {
        ksv=k; /* ksv: indice du 1ier tuple d'un ensemble de repet */
        mu3[i] =(TDouble)ZERO;
        mu4[i] =(TDouble)ZERO;
        Moy=(TDouble)ZERO;
        for(j=0; j<nbrepet[i]; j++)
          {
          Moy = Moy + residus[k];
          k =  k + 1;
          }
        Moy = Moy / (TDouble)(nbrepet[i]);
        /* calcul des Mu */
        k = ksv;
        for(j=0; j<nbrepet[i]; j++)
          {
          Val =  residus[k] - Moy;
          mu3[i] = mu3[i] + pow(Val, (TDouble)3);
          mu4[i] = mu4[i] + pow(Val, (TDouble)4);
          k =  k + 1;
          }
        mu3[i] = mu3[i] / (TDouble)(nbrepet[i]);
        mu4[i] = mu4[i] / (TDouble)(nbrepet[i]);
        } /* fin boucle sur i */
      break;
    } /* fin du switch */
  } /* fin du (*TypeMu == MUGAUSS) || (*TypeMu == MURES) ||  (*TypeMu == MURESREPET)) 
      Si on est pas passe par la, c'est que TypeMu=CONNU ou que l'on a pas demande la
      verif et qu'il est incorrect */
else
  {
  if (*TypeMu != CONNU)
    {
    sprintf(Mess1,"%hd", *TypeMu);
    /* actions effectuees par la macro NLERREUR :
   fprintf(stderr, "TypeMu, %s, n'est pas un code reconnu\n",
    Mess1);
    return(ERRCMU);
    */
    NLERREUR((ERRCMU,2,"CMu",Mess1, ERR));
    }
  }

    
/* retour */
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : CNorme                   |
| Role                  : calcul de la norme des   |
|    equations normales                            |
| Parametres d'entree   :                          |
|  NbObsT: nombre total de repetitions             |
|  ValR: =ValB*(ValZ-ValEta), vecteur NbAct        |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  Norme: somme des elements au carre de ValR      |
|   divisee par NbObsT                             |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  NLAutres                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CNorme( NbObsT, ValR, 
                  Norme)

/* arguments d'entree */
TLongInt NbObsT;
TVect *ValR;

/* arguments de sortie*/
TDouble *Norme;

/*--------------- Fin identification fonction ----*/


{
/* locals */
TShortInt i,n;

/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *valr;

/* Ecriture de la trace */
ECRTRACE("CNorme");

*Norme=(TDouble)ZERO;
n = ValR->nbele;
valr = ValR->donnees;

for (i = 0; i < n; i++)
  {
  *Norme = *Norme + (valr[i] * valr[i]);
  }
*Norme = *Norme / (TDouble)NbObsT;


/* retour */
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CResidus                 |
| Role                  : Calcul des residus       |
| c'est-a-dire de la difference entre les valeurs  |
| observees de la variance de Y et celles ajustees |
| Les observations de poids nuls ne figurent pas   |
| dans les sorties                                 |
| Parametres d'entree   :                          |
|  NbObsT: nombre d'observations en comptant les   |
|   repetitions                                    |
|  NbRepet: vecteur des repetitions                |
|  PoidsT: poids sur chacune des repetitions       |
|  y compris celles de poids nul                   | 
|  dimension nulle s'il n'y en a pas               |
|  ValY: valeurs observees de la reponse           |
|  sur chacune des repetitions y compris celles    |
|  de poids nul                                    |
|  Valf: valeurs ajustees de f                     |
| Parametres de sortie  :                          |
|  Residus: les residus                            |
|   non alloues avant l'appel                      |
| Parametres d'e./s.    :                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CreerVect                 |
| Fonctions appelantes : CMu                       |
|                        NLAutres                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt CResidus(NbObsT, NbRepet, PoidsT, ValY, Valf, 
                   Residus)

/* arguments d'entree */
TLongInt NbObsT;
TVectLong *NbRepet ;
TVect *PoidsT, *ValY, *Valf;

/* arguments de sortie */
TVect *Residus ;

/*--------------- Fin identification fonction ----*/

{
TShortInt i,j, IndObsTT, IndRes, NbObs;
TLogic YaPoids;
TDouble Poi;

/* pointeurs sur des elements de structure pour ameliorer la performance */
TLongInt *nbrepet;
TDouble *poidst, *valy, *valf, *residus;

/* Ecriture de la trace */
ECRTRACE("CResidus");

  /* allouer les sorties */
  APPEL(CreerVect((TShortInt)NbObsT, Residus));

  /* initialisations */
  IndObsTT = 0 ;
  IndRes = 0;
  NbObs = Valf->nbele;
  if (PoidsT->nbele == 0)
    {
    YaPoids=FAUX;
    }
  else
    {
    YaPoids=VRAI;
    }
  Poi=(TDouble)1;

  /* boucle sur les observations */
  /* affectation des pointeurs pour la performance */
  nbrepet = NbRepet->donnees;
  residus = Residus->donnees;
  valy = ValY->donnees;
  valf = Valf->donnees;
  poidst = PoidsT->donnees;

  for (i = 0 ; i < NbObs ; i++)
    {
    j = 0 ;
    while(j < nbrepet[i])
      {
      if (YaPoids==VRAI)
        {
        Poi=poidst[IndObsTT];
        }
 
      if (Poi != (TDouble)ZERO)
        {
        /* observation a  prendre en compte */
        residus[IndRes] = valy[IndObsTT] - valf[i] ;
        j = j + 1;
        IndRes = IndRes + 1;
        }
      IndObsTT = IndObsTT + 1;
      } /* fin de while sur j */
    } /* fin de boucle sur i */

  /* retour */
  return(OK);
}

        

/*--------------- Identification fonction ----------
| Nom de la fonction    : CResidusR                |
| Role                  : calculer les residus     |
|  reduits, c'est-a-dire, les residus divises par  |
|  la racine carree des valeurs ajustees de la     |
|  variance                                        |
| Parametres d'entree   :                          |
|  NbRepet: nombre de repetitions de chaque        |
|    observation
|  VarY: valeurs ajustees de la variance           |
|  Residus : les residus                           |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ResidusR: les residus reduits                   |
|   non alloue avant l'appel                       |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CreerVect, plus la       |
|  fonction systeme sqrt                           |
| Fonctions appelantes :  NLAutres                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CResidusR(NbRepet, VarY, Residus,
                    ResidusR)

/* arguments d'entree */
TVectLong *NbRepet;
TVect *VarY;
TVect *Residus;

/* arguments de sortie*/
TVect *ResidusR;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt NbObsT, NbObs, i,j,k;

/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *vary, *residus, *residusr;
TLongInt *nbrepet;

/* Ecriture de la trace */
ECRTRACE("CResidusR");

NbObsT = Residus->nbele;
NbObs = NbRepet->nbele;

/* allouer la sortie */
APPEL(CreerVect(NbObsT, ResidusR));

/* affectation des pointeurs pour ameliorer la performance */
residus = Residus->donnees;
residusr = ResidusR->donnees;
vary = VarY->donnees;
nbrepet = NbRepet->donnees;

k = 0;
for (i=0; i< NbObs; i++)
  {
  for (j=0; j<nbrepet[i]; j++)
    {
    residusr[k] = residus[k] / sqrt(vary[i]);
    k = k +1;
    }
  }

/* retour */
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : NLAutres                 |
| Role                 : calculer les sorties      |
|   non calculees lors du processus iteratif       |
| Parametres d'entree   :                          |
|  Donnees: les donnees                            |
|  CtxPuss: le contexte du processus               |
|  CtxNum: tableau des NbEtapes contextes          |
|          numeriques                              |
|  Theta, Beta: tableaux des NbEtapes structures   |
|   des parametres  Theta et Beta                  |
|  CTheta, CBeta: tableaux des NbEtapes structures |
|   des contraintes sur les parametres en dimension|
|    ``modele * NbCourbe''                         |
|  CThetaE, CBetaE:  tableaux des NbEtapes         |
|     structures des contraintes sur les           |
|    parametres differents                         |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
|  ResNum, ResStat: tableaux des NbEtapes          |
|     resultats numeriques et  statistiques        |
| Effets de bord    :                              |
|   - si  GNLControle.Voulu.Mu =VRAI,              |
|   et si TypeMu=MURES ou MURESREPET, Residus est  |
|   calcule                                        |
|   - si  GNLControle.Voulu.BVarZBP = VRAI,        |
|   et si PCVarZ=CVarZmv, Mu3 et Mu4 sont calcules |
|   (avec controle de TypeMu si AVerif.VCtxPuss)   |
|   ce qui implique les consequences ci-dessus     |
|   - si  GNLControle.Voulu.AsVar=VRAI,            |
|   si Effic=FAUX, BVarZBP est calcule ce qui      |
|   implique les  consequences ci-dessus           |
|   - si GNLControle.Voulu.Corr= VRAI,             |
|   AsVar est calcule en sortie, ce qui implique   |
|    les  consequences ci-dessus                   |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                        CBVarZBP,                 |
|                        CCorr                     |
|                        CCov1,CCov2, CCov3,       |
|                        CMu,                      |
|                        CNorme,                   |
|                        CreerVect,                |
|                        CResidus,                 |
|                        CResidusR,                |
|                        CScrNP,                   |
|                        CScrPV,                   |
|                        CValR,                    |
|                        DetruVect,                |
| Fonctions appelantes : NL                        |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt NLAutres( Donnees, CtxPuss, CtxNum, Theta, Beta, CTheta, CBeta,  CThetaE,  CBetaE,
                   ResNum, ResStat)

/* arguments d'entree */
TDonnees *Donnees;
TCtxPuss *CtxPuss; /* TypeMu est une entree-sortie */
TCtxNum CtxNum[];
TParam Theta[] , Beta[];
TContr CTheta[], CBeta[];
TContr CThetaE[], CBetaE[];

/* arguments d'entrees-sorties */
TResNum ResNum[]; /* Sigma est une entree */
TResStat ResStat[];

/*--------------- Fin identification fonction ----*/

{
/* locals */

TVect ValR;
TShortInt i,j,l, MonCode, IndEtap, Etape, NbAct;
TDouble n, y, proba,mu, freq, v,v1,v2, sqv;
 long nn;

/* mettre son nom dans la trace */
ECRTRACE("NLAutres");

MonCode = OK;

for (IndEtap =0; IndEtap < CtxPuss->NbEtapes; IndEtap++)
  {
  Etape = IndEtap + 1;  


  if (GNLControle.Voulu.ResNum == VRAI) 
    {
    /* calcul de Scr ponderee par VarY et divisee par n */
    APPEL(CScrPv(Donnees->NbObsT, ResNum[IndEtap].Sigma, 
                &(Donnees->NbRepet), &(Donnees->StatDon.Y1),
                &(Donnees->StatDon.Y2),
                &(ResStat[IndEtap].Ajustes.Valf),
                &(ResStat[IndEtap].Ajustes.VarY),
                 &(ResNum[IndEtap].Scr)));
    /* Sigma a ete calcule par NLVCtxPuss ou PussIter */
    ResNum[IndEtap].Scr =  ResNum[IndEtap].Scr *(TDouble)(Donnees->NbObsT) ;


    /* calcul de Scr non ponderee et non divisee */
    APPEL(CScrNP(Donnees->NbObsT,
                 &(Donnees->NbRepet), &(Donnees->StatDon.Y1), 
                 &(Donnees->StatDon.Y2), 
                 &(ResStat[IndEtap].Ajustes.Valf), 
                 &(ResNum[IndEtap].ScrNP)));

    ResNum[IndEtap].ScrNP =  ResNum[IndEtap].ScrNP * (TDouble)(Donnees->NbObsT);
    
    /* calcul des loglik selon la famille: */
    if (CtxPuss->Famille != GAUSS)
      ResNum[IndEtap].Log = mysignaling_nan(nn);


    switch (CtxPuss->Famille) {
    case GAUSS: 
    /* calcul de -2log(Vraisemblance)/n */
      APPEL( CLv( Donnees->NbObsT,
                &(Donnees->NbRepet), &(Donnees->StatDon.Y1),
                &(Donnees->StatDon.Y2),
                &(ResStat[IndEtap].Ajustes.Valf),  &(ResStat[IndEtap].Ajustes.VarY),
                &(ResNum[IndEtap].Log)));

      ResNum[IndEtap].LogFamille =
             - Donnees->NbObsT * ResNum[IndEtap].Log /2.0;
      /*      ICI, pas de Deviance ni de ResidusDev*/
      break;
    case  POISSON:
      /*
    mu_rep(out.nls2$response,out.nls2$replications)
    v_- sum(lgamma(y+1))
    v1_y*log(mu)-mu
    ResNum[IndEtap].LogFamille =    sum(v1) + v
        v2_y*log(y)-y
    v2[y==0]_0
    ResNum[IndEtap].Deviance = -2*(sum(v1)-sum(v2));
  APPEL(CreerVect(Donnees->NbObsT, &(ResNum[IndEtap].ResidusDev)));

    ResNum[IndEtap].ResidusDev = sqrt(-2*(v1-v2))*sign(y-mu); 
Ici pas de Loglik mais uniquement celui de la famille
  */
      ResNum[IndEtap].LogFamille = 0.0;
      ResNum[IndEtap].Deviance = 0.0;
      APPEL(CreerVect(Donnees->NbObsT, &(ResNum[IndEtap].ResidusDev)));
      l = 0;
      for (i=0; i<Donnees->NbObs; i++) {
	for (j=0; j< Donnees->NbRepet.donnees[i]; j++) {
	  y=Donnees->ValY.donnees[l];
	  mu = ResStat[IndEtap].Ajustes.Valf.donnees[i];
          v= - lgammafn(y+1);
	  v1 = (y*log(mu)) - mu;
          ResNum[IndEtap].LogFamille += (v1+v);
	  ResNum[IndEtap].Deviance +=  ( -2.0 * v1);
	  if (y != (double)0.0) {
	    v2= (y*log(y)) - y;
	    ResNum[IndEtap].Deviance += ( 2 * v2);
	  }
	  sqv = -2*(v1-v2);
	  if (sqv < 0.0)
	    ResNum[IndEtap].ResidusDev.donnees[l] = 
	      (double)mysignaling_nan(nn);
	  else {
	    ResNum[IndEtap].ResidusDev.donnees[l] = sqrt(sqv);
	  if ( (y-mu) <0) {
	    ResNum[IndEtap].ResidusDev.donnees[l] =
	      -ResNum[IndEtap].ResidusDev.donnees[l];
	  }
	  }
	  l=l+1;
	} /* fin j */
      } /* fin i */
      break;

    
    case  BINOM :
    case  BERNOULLI :
      /*
    if (n=="NULL") n_rep(1,length(y))
    proba_rep(out.nls2$response,out.nls2$replications)/n
    freq_y/n
    v_lgamma(n+1)-lgamma(y+1)- lgamma(n-y+1)
    v1_y*log(proba/(1-proba))+ n*log(1-proba)
    ResNum[IndEtap].LogFamille = sum(v1) + sum(v)
    indic_(freq>0)&(freq<1)
    v2_y*log(freq/(1-freq))+ n*log(1-freq)
    ResNum[IndEtap].Deviance = -2*(sum(v1)-sum(v2[indic]))
  APPEL(CreerVect(Donnees->NbObsT, &(ResNum[IndEtap].ResidusDev)));
    ResNum[IndEtap].ResidusDev = sqrt(-2*(v1-v2))*sign(freq-proba)
Ici pas de Log
		     */
      ResNum[IndEtap].LogFamille = 0.0;
      ResNum[IndEtap].Deviance = 0.0;
      APPEL(CreerVect(Donnees->NbObsT, &(ResNum[IndEtap].ResidusDev)));
      APPEL(CreerVect(Donnees->NbObsT, &(ResStat[IndEtap].ResidusR)));

      l = 0;
      for (i=0; i<Donnees->NbObs; i++) {
	for (j=0; j< Donnees->NbRepet.donnees[i]; j++) {
	  y=Donnees->ValY.donnees[l];
	  if (CtxPuss->Famille == BERNOULLI) 
	    n=1;
	  else
            n=Donnees->XObs.donnees[i][Donnees->IndiceN -1 ];
	  proba = ResStat[IndEtap].Ajustes.Valf.donnees[i] / (TDouble)n;
	  freq= (TDouble)(y/n);
          v= lgammafn(n+1) - lgammafn(y+1) -lgammafn( n-y+1);
	  v1 = y*log(proba/(1-proba))+ n*log(1-proba);

          ResNum[IndEtap].LogFamille += (v1+v);
	  ResNum[IndEtap].Deviance +=  ( -2.0 * v1);
	  if ((freq>0) && (freq <1)) {
	    v2= y*log(freq/(1-freq))+ n*log(1-freq);
	    ResNum[IndEtap].Deviance += ( 2 * v2);
	  } /* fin if ((freq>0) && (freq <1)) */

	  sqv = -2*(v1-v2);
	  if (sqv < 0.0)
	    ResNum[IndEtap].ResidusDev.donnees[l] = 
	      (double)mysignaling_nan(nn);
	  else {
	    ResNum[IndEtap].ResidusDev.donnees[l] = sqrt(sqv);
	  if ( (freq-proba) <0) {
	    ResNum[IndEtap].ResidusDev.donnees[l] =
	      -ResNum[IndEtap].ResidusDev.donnees[l];
	  }
	  }

	  /* Calcul des rsidus rduits */
	  sqv = n*proba*(1.0-proba);
	  if (sqv < 0.0)
	    ResStat[IndEtap].ResidusR.donnees[l] =
	      (double)mysignaling_nan(nn);
	  else {
	  ResStat[IndEtap].ResidusR.donnees[l] =
	    (y- (n*proba)) / sqrt(sqv);
	  }
	  l=l+1;
	} /* fin j */
      } /* fin i */

      break;
    
    case MULTINOM :
      /*
   p_rep(out.nls2$response,out.nls2$replications)/n
    v_sum(lgamma(n+1))
    v1_y*log(p)
    out.nls2$loglik_sum(v1) + v 
	    v2_y*log(y/n)
    v2[y==0]_0
    ResNum[IndEtap].Deviance = -2*(sum(v1)-sum(v2))
Ici pas de Loglik ni de ResidusDev
    */
      ResNum[IndEtap].LogFamille = 0.0;
      ResNum[IndEtap].Deviance = 0.0;
      APPEL(CreerVect(Donnees->NbObsT, &(ResStat[IndEtap].ResidusR)));
      l = 0;
      for (i=0; i<Donnees->NbObs; i++) {
	for (j=0; j< Donnees->NbRepet.donnees[i]; j++) {
	  y=Donnees->ValY.donnees[l];
          n=Donnees->XObs.donnees[i][Donnees->IndiceN -1 ];
	  proba = ResStat[IndEtap].Ajustes.Valf.donnees[i] / (TDouble)n;
          v= lgammafn(n+1);
	  v1 = y*log(proba);
          ResNum[IndEtap].LogFamille += (v1+v);
	  ResNum[IndEtap].Deviance +=  ( -2.0 * v1);
	  if (y != (double)0.0) {
	    v2= y*log(y/n);
	    ResNum[IndEtap].Deviance += ( 2 * v2);
	  } 
	  /* Calcul des rsidus rduits: idem que BINOM et BERNOULLI */
	  sqv = ResStat[IndEtap].Ajustes.Valf.donnees[i] *
	    (1.0 - ResStat[IndEtap].Ajustes.Valf.donnees[i] /n);
	  if (sqv < 0.0)
	    ResStat[IndEtap].ResidusR.donnees[l] =
	      (double)mysignaling_nan(nn);
	  else {
	  ResStat[IndEtap].ResidusR.donnees[l] =
	    (y-ResStat[IndEtap].Ajustes.Valf.donnees[i] ) / sqrt(sqv);
	  }
	  l=l+1;
	} /* fin j */
      } /* fin i */
    break;

    default:
      fprintf(stderr, "Error in NLAutres: family, %s, not valid.\n",
	     CtxPuss->Famille); 
  } /* fin du switch */
      
  
    /* Allocation de ValR */
    NbAct = Theta[IndEtap].Act.nbele + Beta[IndEtap].Act.nbele;
    APPEL(CreerVect(NbAct, &ValR));
  
    /* calcul de ValR = B(Z-eta) */
    APPEL( CValR(&(ResStat[IndEtap].ValZ),&(ResStat[IndEtap].EquN.ValEta),
                     &(ResStat[IndEtap].EquN.ValB), &ValR));
    APPEL(CNorme(Donnees->NbObsT, &ValR, &(ResNum[IndEtap].Norme)));
  
    /* desallocation de ValR */
    DetruVect (&ValR);

    } /* fin (GNLControle.Voulu.ResNum == VRAI) */
  
  
  /* Respecter l'ordre des appels suivants */
  
  if (GNLControle.Voulu.Residus == VRAI)
    {
    APPEL(CResidus(Donnees->NbObsT,&(Donnees->NbRepet), &(Donnees->PoidsT), 
                   &(Donnees->ValY), &(ResStat[IndEtap].Ajustes.Valf),
                   &(ResStat[IndEtap].Residus)));
    if ( (CtxPuss->Famille != BINOM) && 
	 (CtxPuss->Famille != BERNOULLI) &&
	 (CtxPuss->Famille != MULTINOM))
    APPEL(CResidusR( &(Donnees->NbRepet),&(ResStat[IndEtap].Ajustes.VarY),
                     &(ResStat[IndEtap].Residus), &(ResStat[IndEtap].ResidusR)));
  
    }
  
  if (GNLControle.Voulu.Mu== VRAI)
    {
    APPEL(CMu( Donnees->NbObsT, &(Donnees->NbRepet), &(Donnees->PoidsT), 
               &(Donnees->ValY), &(ResStat[IndEtap].Ajustes.Valf), &(ResStat[IndEtap].Ajustes.VarY),
               &(CtxPuss->TypeMu), &(ResStat[IndEtap].Residus), &(ResStat[IndEtap].Mu3), 
               &(ResStat[IndEtap].Mu4)));
    }
  
  if (GNLControle.Voulu.BVarZBP== VRAI)
    {
  
    APPEL(CBVarZBP(&(ResStat[IndEtap].EquN.ValB), &(ResStat[IndEtap].Ajustes),
                   Donnees, &(CtxNum[IndEtap]), &(CtxPuss->TypeMu),
                   &(ResStat[IndEtap].Residus), &(ResStat[IndEtap].Mu3), &(ResStat[IndEtap].Mu4),
                   &(ResStat[IndEtap].BVarZBP), &MonCode));
    }

  
  if ((GNLControle.Voulu.AsVar== VRAI) && (MonCode == OK))
    {
    if (Etape ==1)
      {
      APPEL(CCov1(  ResNum[IndEtap].Sigma,
                 Donnees,&(Theta[IndEtap]), &(Beta[IndEtap]), &(CTheta[IndEtap]), &(CBeta[IndEtap]),
                 &(CThetaE[IndEtap]), &(CBetaE[IndEtap]), &(CtxNum[IndEtap]),
                   &(CtxPuss->TypeMu),  &(ResStat[IndEtap]),  &MonCode));
      }
    if (Etape  == 2)
      {
      APPEL(CCov2(Donnees,Theta, Beta,CTheta, CBeta, CThetaE, CBetaE, CtxNum, ResNum,
                 &(CtxPuss->TypeMu),  &(ResStat[IndEtap]),  &MonCode));
      }
    if (Etape == 3)
      {
      APPEL(CCov3(Donnees,Theta, Beta,
                CTheta, CBeta,
                CThetaE, CBetaE, CtxNum, ResNum,
                &(CtxPuss->TypeMu),  &(ResStat[IndEtap]),  &MonCode));

      } /* fin du if (Etape == 3) */
    } /* fin du if (GNLControle.Voulu.AsVar== VRAI) */
  
  if ((GNLControle.Voulu.Corr== VRAI) && (MonCode == OK))
    {
    /* calcul de la matrice de correlation */
    APPEL(CCorr(Etape, Donnees, Theta, Beta,
                CTheta, CBeta,
                CThetaE, CBetaE, CtxNum, ResNum,
                &(CtxPuss->TypeMu),  &(ResStat[IndEtap])));
    }
  } /* fin boucle sur IndEtap */



/* retour*/
return(OK);
}



    

