/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : NLEquN                    |
| Date                 :  7Nov91 / 16:31:20        |
| Derniere mise a jour :                           |
| Concepteur           : A. Bouvier                |
| Role                 : Programmes de calcul des  |
|  equations normales: B, D, Eta                   |
| Reference conception :                           |
| Lecteur              : G. Husser                 |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlmacros.h"
#include "errcodes.h"
#include "nlglobal.h"

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/
/* fonctions des autres modules */
TShortInt CopyMat(), CopyVect();

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : CBmcmb                   |
| Role                  : Calculer ValB, quand     |
|  Estim=MCMB ou QVB                               |
|  les parametres actifs sont ceux de              |
|   la variance (les Beta)                         |             
|  et il y a 2 statistiques exhaustives            |
| Parametres d'entree   :                          |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Valf: valeurs de la fonction f                  |
|  VarY: valeurs de la variance de Y               |
|  S2: somme des carres, pour chaque observation   |
|   (inutilise ici)                                |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValB: matrice ValB                              |
|        dimension: 2*NbObs, (NbBeta)              | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CBmcmb(NbRepet, Valf, VarY, S2, DValf, DVarYTheta, DVarYBeta,
                 ValB)

/* arguments d'entree */
TVectLong *NbRepet; /* dimension NbObs */
TVect *Valf, *VarY, *S2; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta ou 0,0*/
TMat *DVarYBeta;  /* dimension NbObs, NbBeta ou 0,0*/


/* arguments de sortie */
TMat *ValB; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, a, NbObs, NbTheta, NbBeta;
TDouble X,U,T;


/* pointeurs pour ameliorer la performance */
TLongInt *nbrepet;
TDouble *vary, *valf ;
TDouble **valb, **dvarybeta;

/* Ecriture de la trace */
ECRTRACE("CBmcmb");

/* determination des dimensions */
NbObs = NbRepet->nbele;
NbTheta = DValf->nbcol;
NbBeta = DVarYBeta->nbcol;

/* affectation des pointeurs */
nbrepet = NbRepet->donnees;
vary = VarY->donnees;
valf = Valf->donnees;
valb = ValB->donnees;
dvarybeta = DVarYBeta->donnees;

for(i = 0; i < NbObs; i++)
  {
  X = (TDouble)nbrepet[i] / vary[i];
  U = X / vary[i];
  T =  U * valf[i];
  for(a = 0; a < NbBeta; a++)
    {
    valb[i][NbTheta+a] = - (T *  dvarybeta[i][a]);
    valb[i+NbObs][NbTheta+a] =  (U * dvarybeta[i][a]) / (TDouble)2;
    }
  } /* fin boucle sur i */
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CBmcmtb                  |
| Role                  : Calculer ValB, quand     |
|  Estim=MCMTB ou QVTB                             |
|  les parametres actifs sont ceux de f,(les Theta)|
|  et de la variance (les Beta),                   |
|  et il y a 2 statistiques exhaustives (NbZ=2)    |
| Parametres d'entree   :                          |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Valf: valeurs de la fonction f                  |
|  VarY: valeurs de la variance de Y               |
|  S2: somme des carres, pour chaque observation   |
|   (inutilise ici)                                |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|   (inutilise ici)                                |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValB: matrice ValB                              |
|        dimension: 2*NbObs, (NbTheta+NbBeta)      | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CBmcmtb(NbRepet, Valf, VarY, S2, DValf, DVarYTheta, DVarYBeta,
                 ValB)

/* arguments d'entree */
TVectLong *NbRepet; /* dimension NbObs */
TVect *Valf, *VarY, *S2; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta ou 0,0*/
TMat *DVarYBeta;  /* dimension NbObs, NbBeta ou 0,0*/


/* arguments de sortie */
TMat *ValB; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, a, NbObs, NbTheta, NbBeta;
TDouble X,U,M,T;


/* pointeurs pour ameliorer la performance */
TLongInt *nbrepet;
TDouble *vary, *valf ;
TDouble **valb, **Dvalf, **dvarybeta;

/* Ecriture de la trace */
ECRTRACE("CBmcmtb");

/* determination des dimensions */
NbObs = NbRepet->nbele;
NbTheta = DValf->nbcol;
NbBeta = DVarYBeta->nbcol;

/* affectation des pointeurs */
nbrepet = NbRepet->donnees;
vary = VarY->donnees;
valf = Valf->donnees;
valb = ValB->donnees;
Dvalf =  DValf->donnees;
dvarybeta = DVarYBeta->donnees;

for(i = 0; i < NbObs; i++)
  {
  X = (TDouble)nbrepet[i] / vary[i];
  U = X / vary[i];
  M =  (U / (TDouble)2);
  T =  U * valf[i];
  for(a = 0; a < NbTheta; a++)
    {
    valb[i][a] =   (X * Dvalf[i][a]);
    valb[i+NbObs][a] =  (TDouble)ZERO;
    }
  for(a = 0; a < NbBeta; a++)
    {
    valb[i][NbTheta+a] = - ( T * dvarybeta[i][a]);
    valb[i+NbObs][NbTheta+a] =  M * dvarybeta[i][a];
    }
  } /* fin boucle sur i */
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CBmcob                   |
| Role                  : Calculer ValB, quand     |
|  Estim=MCOB                                      |
|  les parametres actifs sont ceux de              |
|   la variance (les Beta)                         |             
|  et il y a 2 statistiques exhaustives            |
| Parametres d'entree   :                          |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Valf: valeurs de la fonction f                  |
|  VarY: valeurs de la variance de Y               |
|  S2: somme des carres, pour chaque observation   |
|   (inutilise ici)                                |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|   (inutilise ici)                                |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValB: matrice ValB                              |
|        dimension: 2*NbObs, (NbBeta)              | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CBmcob(NbRepet, Valf, VarY, S2, DValf, DVarYTheta, DVarYBeta,
                 ValB)

/* arguments d'entree */
TVectLong *NbRepet; /* dimension NbObs */
TVect *Valf, *VarY, *S2; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta ou 0,0*/
TMat *DVarYBeta;  /* dimension NbObs, NbBeta ou 0,0*/


/* arguments de sortie */
TMat *ValB; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, a, NbObs, NbTheta, NbBeta;


/* pointeurs pour ameliorer la performance */
TLongInt *nbrepet;
TDouble  *valf ;
TDouble **valb, **dvarybeta;

/* Ecriture de la trace */
ECRTRACE("CBmcob");

/* determination des dimensions */
NbObs = NbRepet->nbele;
NbTheta = DValf->nbcol;
NbBeta = DVarYBeta->nbcol;

/* affectation des pointeurs */
nbrepet = NbRepet->donnees;
valf = Valf->donnees;
valb = ValB->donnees;
dvarybeta = DVarYBeta->donnees;

for(i = 0; i < NbObs; i++)
  {
  for(a = 0; a < NbBeta; a++)
    {
    valb[i][NbTheta+a] = - (valf[i] * nbrepet[i] * dvarybeta[i][a]);
    valb[i+NbObs][NbTheta+a] =  (nbrepet[i] * dvarybeta[i][a]) / (TDouble)2;
    }
  } /* fin boucle sur i */
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CBmcot                   |
| Role                  : Calculer ValB, quand     |
|  Estim=MCOT                                      |
|  les parametres actifs sont ceux de f,(les Theta)|
|  et il y a 1 statistique exhaustive              |
| Parametres d'entree   :                          |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Valf: valeurs de la fonction f                  |
|   (inutilise ici)                                |
|  VarY: valeurs de la variance de Y               |
|   (inutilise ici)                                |
|  S2: somme des carres, pour chaque observation   |
|   (inutilise ici)                                |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|   (inutilise ici)                                |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
|   (inutilise ici)                                |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValB: matrice ValB                              |
|        dimension: NbObs, (NbTheta)               | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CBmcot(NbRepet, Valf, VarY, S2, DValf, DVarYTheta, DVarYBeta,
                 ValB)

/* arguments d'entree */
TVectLong *NbRepet; /* dimension NbObs */
TVect *Valf, *VarY, *S2; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta ou 0,0*/
TMat *DVarYBeta;  /* dimension NbObs, NbBeta ou 0,0*/


/* arguments de sortie */
TMat *ValB; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, a, NbObs, NbTheta;


/* pointeurs pour ameliorer la performance */
TLongInt *nbrepet;
TDouble **valb, **Dvalf;

/* Ecriture de la trace */
ECRTRACE("CBmcot");

/* determination des dimensions */
NbObs = NbRepet->nbele;
NbTheta = DValf->nbcol;

/* affectation des pointeurs */
nbrepet = NbRepet->donnees;
valb = ValB->donnees;
Dvalf =  DValf->donnees;

for(i = 0; i < NbObs; i++)
  {
  for(a = 0; a < NbTheta; a++)
    {
    valb[i][a] =  nbrepet[i] * Dvalf[i][a];
    }
  } /* fin boucle sur i */
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : CBmcpvit                 |
| Role                  : Calculer ValB, quand     |
|  Estim=MCPVIT                                    |
|  les parametres actifs sont ceux de f,(les Theta)|
|  et il y a 1 statistique exhaustive              |
| Parametres d'entree   :                          |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Valf: valeurs de la fonction f                  |
|   (inutilise ici)                                |
|  VarY: valeurs de la variance de Y               |
|   (inutilise ici)                                |
|  S2: somme des carres, pour chaque observation   |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|   (inutilise ici)                                |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
|   (inutilise ici)                                |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValB: matrice ValB                              |
|        dimension: NbObs, (NbTheta)               | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CBmcpvit(NbRepet, Valf, VarY, S2, DValf, DVarYTheta, DVarYBeta,
                 ValB)

/* arguments d'entree */
TVectLong *NbRepet; /* dimension NbObs */
TVect *Valf, *VarY, *S2; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta ou 0,0*/
TMat *DVarYBeta;  /* dimension NbObs, NbBeta ou 0,0*/


/* arguments de sortie */
TMat *ValB; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, a, NbObs, NbTheta;


/* pointeurs pour ameliorer la performance */
TLongInt *nbrepet;
TDouble *s2;
TDouble **valb, **Dvalf;

/* Ecriture de la trace */
ECRTRACE("CBmcpvit");

/* determination des dimensions */
NbObs = NbRepet->nbele;
NbTheta = DValf->nbcol;

/* affectation des pointeurs */
nbrepet = NbRepet->donnees;
valb = ValB->donnees;
Dvalf =  DValf->donnees;
s2 = S2->donnees;

for(i = 0; i < NbObs; i++)
  {
  for(a = 0; a < NbTheta; a++)
    {
    valb[i][a] =   (nbrepet[i] / s2[i]) * Dvalf[i][a];
    }
  } /* fin boucle sur i */
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CBmct                    |
| Role                  : Calculer ValB, quand     |
|  Estim=MCPT ou MCMT ou QVT                       |
|  les parametres actifs sont ceux de f,(les Theta)|
|  et il y a 1 statistique exhaustive              |
| Parametres d'entree   :                          |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Valf: valeurs de la fonction f                  |
|   (inutilise ici)                                |
|  VarY: valeurs de la variance de Y               |
|  S2: somme des carres, pour chaque observation   |
|   (inutilise ici)                                |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|   (inutilise ici)                                |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
|   (inutilise ici)                                |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValB: matrice ValB                              |
|        dimension: NbObs, (NbTheta)               | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CBmct(NbRepet, Valf, VarY, S2, DValf, DVarYTheta, DVarYBeta,
                 ValB)

/* arguments d'entree */
TVectLong *NbRepet; /* dimension NbObs */
TVect *Valf, *VarY, *S2; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta ou 0,0*/
TMat *DVarYBeta;  /* dimension NbObs, NbBeta ou 0,0*/


/* arguments de sortie */
TMat *ValB; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, a, NbObs, NbTheta;
TDouble X;


/* pointeurs pour ameliorer la performance */
TLongInt *nbrepet;
TDouble *vary ;
TDouble **valb, **Dvalf;

/* Ecriture de la trace */
ECRTRACE("CBmct");

/* determination des dimensions */
NbObs = NbRepet->nbele;
NbTheta = DValf->nbcol;

/* affectation des pointeurs */
nbrepet = NbRepet->donnees;
vary = VarY->donnees;
valb = ValB->donnees;
Dvalf =  DValf->donnees;

for(i = 0; i < NbObs; i++)
  {
  X = (TDouble)nbrepet[i] / vary[i];
  for(a = 0; a < NbTheta; a++)
    {
    valb[i][a] =  X * Dvalf[i][a];
    }
  } /* fin boucle sur i */
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CBmvt                    |
| Role                  : Calculer ValB, quand     |
|  Estim=MVT:                                      |
|  les parametres actifs sont ceux de f,(les Theta)|
|  et il y a 2 statistiques exhaustives (NbZ=2)    |
| Parametres d'entree   :                          |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Valf: valeurs de la fonction f                  |
|  VarY: valeurs de la variance de Y               |
|  S2: somme des carres, pour chaque observation   |
|   (inutilise ici)                                |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
|   (inutilise ici)                                |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValB: matrice ValB                              |
|        dimension: 2*NbObs, (NbTheta)             | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CBmvt(NbRepet, Valf, VarY, S2, DValf, DVarYTheta, DVarYBeta,
                 ValB)

/* arguments d'entree */
TVectLong *NbRepet; /* dimension NbObs */
TVect *Valf, *VarY, *S2; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta ou 0,0*/
TMat *DVarYBeta;  /* dimension  0,0*/


/* arguments de sortie */
TMat *ValB; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, a, NbObs, NbTheta;
TDouble X,U,T,M;


/* pointeurs pour ameliorer la performance */
TLongInt *nbrepet;
TDouble *vary, *valf ;
TDouble **valb, **Dvalf, **dvarytheta ;

/* Ecriture de la trace */
ECRTRACE("CBmvt");

/* determination des dimensions */
NbObs = NbRepet->nbele;
NbTheta = DValf->nbcol;


/* affectation des pointeurs */
nbrepet = NbRepet->donnees;
vary = VarY->donnees;
valf = Valf->donnees;
valb = ValB->donnees;
Dvalf =  DValf->donnees;
dvarytheta = DVarYTheta->donnees;

for(i = 0; i < NbObs; i++)
  {
  X = (TDouble)nbrepet[i] / vary[i];
  U = X / vary[i];
  T =  U * valf[i];
  M =   (U / (TDouble)2);
  for(a = 0; a < NbTheta; a++)
    {
    valb[i][a] =  (X * Dvalf[i][a]) - (T * dvarytheta[i][a]);
    valb[i+NbObs][a] =  M * dvarytheta[i][a];
    }
  } /* fin boucle sur i */
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : CBmvtb                   |
| Role                  : Calculer ValB, quand     |
|  Estim=MVTB:                                     |
|  les parametres actifs sont ceux de f,(les Theta)|
|  et de la variance (les Beta)                    |             
|  et il y a 2 statistiques exhaustives (NbZ=2)    |
| Parametres d'entree   :                          |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Valf: valeurs de la fonction f                  |
|  VarY: valeurs de la variance de Y               |
|  S2: somme des carres, pour chaque observation   |
|   (inutilise ici)                                |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValB: matrice ValB                              |
|        dimension: 2*NbObs, (NbTheta+NbBeta)      | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CBmvtb(NbRepet, Valf, VarY, S2, DValf, DVarYTheta, DVarYBeta,
                 ValB)

/* arguments d'entree */
TVectLong *NbRepet; /* dimension NbObs */
TVect *Valf, *VarY, *S2; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta */
TMat *DVarYBeta;  /* dimension NbObs, NbBeta */


/* arguments de sortie */
TMat *ValB; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, a, NbObs, NbTheta, NbBeta;
TDouble X,U,T,M;

/* pointeurs pour ameliorer la performance */
TLongInt *nbrepet;
TDouble *vary, *valf ;
TDouble **valb, **Dvalf, **dvarytheta, **dvarybeta;


/* Ecriture de la trace */
ECRTRACE("CBmvtb");

/* determination des dimensions */
NbObs = NbRepet->nbele;
NbTheta = DValf->nbcol;
NbBeta = DVarYBeta->nbcol;

/* affectation des pointeurs */
nbrepet = NbRepet->donnees;
vary = VarY->donnees;
valf = Valf->donnees;
valb = ValB->donnees;
Dvalf =  DValf->donnees;
dvarytheta = DVarYTheta->donnees;
dvarybeta = DVarYBeta->donnees;

for(i = 0; i<NbObs; i++)
  {
  X = (TDouble)nbrepet[i] / vary[i];
  U = X / vary[i];
  T =  U * valf[i];
  M =   (U / (TDouble)2);
  for(a  =  0; a < NbTheta; a++)
    {
    valb[i][a] =   (X * Dvalf[i][a]) - (T * dvarytheta[i][a]);
    valb[i+NbObs][a] =  M * dvarytheta[i][a];
    }
  for(a = 0; a < NbBeta; a++)
    {
    valb[i][NbTheta+a] = - (T * dvarybeta[i][a]);
    valb[i+NbObs][NbTheta+a] =  M * dvarybeta[i][a];
    }
  } /* fin boucle sur i */
return(OK);
}




/*--------------- Identification fonction ----------
| Nom de la fonction    : CDmcb                    |
| Role                  : Calculer ValD, quand     |
|  Estim=MCOB, MCMB ou QVB                         |
|  les parametres actifs sont ceux de              |
|  la variance (les Beta)                          |             
|  et il y a 2 statistiques exhaustives (NbZ=2)    |
| Parametres d'entree   :                          |
|  Valf: valeurs de la fonction f                  |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValD: matrice ValD                              |
|        dimension:   2*NbObs,(NbTheta+NbBeta)     | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2, CCovNu3         |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CDmcb( Valf, DValf, DVarYTheta, DVarYBeta,
                 ValD)

/* arguments d'entree */
TVect *Valf; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta */
TMat *DVarYBeta;  /* dimension NbObs, NbBeta */


/* arguments de sortie */
TMat *ValD; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, j, NbObs, NbBeta;

/* pointeurs pour ameliorer la performance */
TDouble **vald, **dvarybeta;

/* Ecriture de la trace */
ECRTRACE("CDmcb");



/* determination des dimensions */
NbObs = DValf->nblig;
NbBeta = DVarYBeta->nbcol;

/* affectation des pointeurs */
vald = ValD->donnees;
dvarybeta = DVarYBeta->donnees;

for (i = 0; i<NbObs; i++)
  {
  for (j = 0 ; j < NbBeta; j++)
    {
    vald[i][j] = (TDouble)ZERO;
    vald[i+NbObs][j] = dvarybeta[i][j];
    }
  }

return(OK);
}




/*--------------- Identification fonction ----------
| Nom de la fonction    : CDmct                    |
| Role                  : Calculer ValD, quand     |
|  Estim=MCPT, MCOT, MCMT, QVT, MCPVIT             |
|  les parametres actifs sont ceux de f,(les Theta)|
|  et il y a 1 statistique exhaustive (NbZ=1)      |
| Parametres d'entree   :                          |
|  Valf: valeurs de la fonction f                  |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValD: matrice ValD                              |
|        dimension:     NbObs,NbTheta              | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CopyMat                  |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2, CCovNu3         |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CDmct( Valf, DValf, DVarYTheta, DVarYBeta,
                 ValD)

/* arguments d'entree */
TVect *Valf; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta */
TMat *DVarYBeta;  /* dimension NbObs, NbBeta */


/* arguments de sortie */
TMat *ValD; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{

/* Ecriture de la trace */
ECRTRACE("CDmct");

APPEL(CopyMat(DValf, ValD));

return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : CDmv                     |
| Role                  : Calculer ValD, quand     |
|  Estim=MVTB, MCMTB ou QVTB                       |
|  les parametres actifs sont ceux de f,(les Theta)|
|  et de la variance (les Beta)                    |             
|  et il y a 2 statistiques exhaustives (NbZ=2)    |
| Parametres d'entree   :                          |
|  Valf: valeurs de la fonction f                  |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValD: matrice ValD                              |
|        dimension:   2*NbObs,(NbTheta+NbBeta)     | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2, CCovNu3         |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CDmv( Valf, DValf, DVarYTheta, DVarYBeta,
                 ValD)

/* arguments d'entree */
TVect *Valf; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta */
TMat *DVarYBeta;  /* dimension NbObs, NbBeta */


/* arguments de sortie */
TMat *ValD; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt NbObs, NbTheta, NbBeta, i, j;


/* pointeurs pour ameliorer la performance */
TDouble  *valf;
TDouble **vald, **Dvalf, **dvarytheta, **dvarybeta;

/* Ecriture de la trace */
ECRTRACE("CDmv");

/* determination des dimensions */
NbObs = DValf->nblig;
NbTheta = DValf->nbcol;
NbBeta = DVarYBeta->nbcol;

/* affectation des pointeurs */
valf = Valf->donnees;
vald = ValD->donnees;
Dvalf =  DValf->donnees;
dvarytheta = DVarYTheta->donnees;
dvarybeta = DVarYBeta->donnees;


for (i = 0; i<NbObs; i++)
  {
  for (j = 0 ; j < NbTheta; j++)
    {
    vald[i][j] = Dvalf[i][j];
    vald[i+NbObs][j] = dvarytheta[i][j] + 
               ((TDouble)2 * valf[i] * Dvalf[i][j]);
    }
  for (j = 0 ; j < NbBeta; j++)
    {
    vald[i][j+NbTheta] = (TDouble)ZERO;
    vald[i+NbObs][j+NbTheta] = dvarybeta[i][j];
    }
  }


return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : CDmvt                    |
| Role                  : Calculer ValD, quand     |
|  Estim=MVT                                       |
|  les parametres actifs sont ceux de f,(les Theta)|
|  et il y a 2 statistiques exhaustives (NbZ=2)    |
| Parametres d'entree   :                          |
|  Valf: valeurs de la fonction f                  |
|  DValf: derivees de la fonction f                |
|  DVarYTheta: derivees de la variance de Y par    |
|   rapport aux Theta actifs, ou matrice de        |
|   dimension 0                                    |
|  DVarYBeta: derivees de la variance de Y par     |
|   rapport aux Beta actifs, ou matrice de         |
|   dimension 0                                    |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValD: matrice ValD                              |
|        dimension:   2*NbObs,NbTheta              | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, Retablir,      |
|                         CCovNu2, CCovNu3         |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CDmvt( Valf, DValf, DVarYTheta, DVarYBeta,
                 ValD)

/* arguments d'entree */
TVect *Valf; /* dimension NbObs */
TMat *DValf; /* dimension NbObs, NbTheta */
TMat *DVarYTheta; /* dimension NbObs, NbTheta */
TMat *DVarYBeta;  /* dimension NbObs, NbBeta */


/* arguments de sortie */
TMat *ValD; /* dim: NbZ*NbObs, NbTheta+NbBeta */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt NbObs, NbTheta, i, j;

/* pointeurs pour ameliorer la performance */
TDouble  *valf;
TDouble **vald, **Dvalf, **dvarytheta;

/* Ecriture de la trace */
ECRTRACE("CDmvt");

/* determination des dimensions */
NbObs = DValf->nblig;
NbTheta = DValf->nbcol;

/* affectation des pointeurs */
valf = Valf->donnees;
vald = ValD->donnees;
Dvalf =  DValf->donnees;
dvarytheta = DVarYTheta->donnees;

for (i = 0; i<NbObs; i++)
  {
  for (j = 0 ; j < NbTheta; j++)
    {
    vald[i][j] = Dvalf[i][j];
    vald[i+NbObs][j] = dvarytheta[i][j] + 
               ((TDouble)2 * valf[i] * Dvalf[i][j]);
    }
  }


return(OK);
}




/*--------------- Identification fonction ----------
| Nom de la fonction    : CEtamco                  |
| Role                  : Calculer ValEta, quand   |
|  il y a 1 statistique exhaustive (NbZ=1)         |
| Parametres d'entree   :                          |
|  Valf: valeurs de la fonction f                  |
|  VarY: valeurs de la variance de Y               |
|   (inutilise ici)                                |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValEta: vecteur ValEta                          |
|        dimension:     NbObs                      | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CopyVect                 |
| Fonctions appelantes :  PussIter, Retablir       |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CEtamco( Valf, VarY,
                 ValEta)

/* arguments d'entree */
TVect *Valf; /* dimension NbObs */
TVect *VarY; /* dimension NbObs */

/* arguments de sortie */
TVect *ValEta; /* dim: NbZ*NbObs */

/*--------------- Fin identification fonction ----*/

{

/* Ecriture de la trace */
ECRTRACE("CEtamco");

APPEL(CopyVect(Valf, ValEta));


return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : CEtamv                   |
| Role                  : Calculer ValEta, quand   |
|  il y a 2 statistiques exhaustives (NbZ=2)       |
| Parametres d'entree   :                          |
|  Valf: valeurs de la fonction f                  |
|  VarY: valeurs de la variance de Y               |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValEta: vecteur ValEta                          |
|        dimension:   2*NbObs                      | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, Retablir       |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CEtamv( Valf, VarY,
                 ValEta)

/* arguments d'entree */
TVect *Valf; /* dimension NbObs */
TVect *VarY; /* dimension NbObs */

/* arguments de sortie */
TVect *ValEta; /* dim: NbZ*NbObs */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt NbObs, i;

/* pointeurs pour ameliorer la performance */
TDouble *vary, *valf, *valeta ;


/* Ecriture de la trace */
ECRTRACE("CEtamv");

/* determination des dimensions */
NbObs = Valf->nbele;


/* affectation des pointeurs */
vary = VarY->donnees;
valf = Valf->donnees;
valeta = ValEta->donnees;

for (i = 0; i<NbObs; i++)
  {
  valeta[i] = valf[i];
  valeta[i+NbObs] = (valf[i] * valf[i]) + vary[i];
  }

return(OK);
}

