/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : NLUtil                    |
| Date                 :  7Nov91 / 16:30:18        |
| Derniere mise a jour :                           |
| Concepteur           : A. Bouvier                |
| Role                 : Utilitaires generaux      |
| Reference conception :                           |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|.
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <stdio.h>

#include "nlchoix.h"
#include "nltypes.h"
#include "nlcodes.h"
#include "errcodes.h"
#include "nlmacros.h"

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : CreerMat                 |
| Role                  :                          |
|    Allouer le composant donnees et affecter      |
|    la valeur de nblig et nbcol,                  |
|    dans une structure de type TMat               |
| Parametres d'entree   :                          |
|    nblig: nombre de lignes                       |
|    nbcol: nombre de colonnes                     |
| Parametres d'e./s.    :                          |
|    pmat : adresse de la structure TMat           |   
|    en sortie les composants nblig et nbcol       |
|    de la structure pointee par pmat sont affectes|
|    et le composant donnees pointe sur un vecteur |
|    de nblig pointeurs, chacun d'entre eux        |
|    pointant sur un vecteur de nbcol elements     |
|    de type TDouble                               |
|    Pointeur sur un element fictif si nblig ou    |
|     nbcol=0                                      |
| Retour:                OK                        |
| S'il y a une erreur d'allocation, les macros     |
| d'allocation appelees provoquent un exit         |
| apres impression d'un message                    |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    :                           |
|       MajContP,                                  |
|       NLAutres,                                  |
|       NLDebut,                                   |
|       NLEtape,                                   |
|       NLVDon,                                    |
|       PussIter                                   |
| Macros utilisees: CREER_T2, CREER_T1             |
--------------------------------------------------*/


/*--------------- Definition fonction -----------*/
TShortInt CreerMat( TShortInt nblig, TShortInt nbcol, TMat *pmat)

/*--------------- Fin identification fonction ----*/

{
TShortInt i, nl, nc;

pmat->nblig=nblig;
pmat->nbcol=nbcol;


if ((nblig <= (TShortInt)0) || (nbcol <= (TShortInt)0))
  {
  /* creation d'un element bidon pour que le pointeur sur donnees existe */
  nl = (TShortInt)1;
  nc= (TShortInt)1;
  }
else
  {
  nl = nblig;
  nc = nbcol;
  }

CREER_T2(pmat->donnees, nl, TDouble);
for(i=0; i<nl; i++)
  {
  CREER_T1(pmat->donnees[i], nc, TDouble);
  }


return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CreerMatC                |
| Role                  :                          |
|    Allouer le composant donnees et affecter      |
|    la valeur de nblig et nbcol,                  |
|    dans une structure de type TMat               |
|    L'allocation des valeurs est dans des         |
|    emplacements memoire consecutifs              |
| Parametres d'entree   :                          |
|    nblig: nombre de lignes                       |
|    nbcol: nombre de colonnes                     |
| Parametres d'e./s.    :                          |
|    pmat : adresse de la structure TMat           |   
|    en sortie les composants nblig et nbcol       |
|    de la structure pointee par pmat sont affectes|
|    et le composant donnees pointe sur un vecteur |
|    de nblig pointeurs, chacun d'entre eux        |
|    pointant sur un vecteur de nbcol elements     |
|    de type TDouble                               |
|    Pointeur sur un element fictif si nblig ou    |
|     nbcol=0                                      |
| Retour:                OK                        |
| S'il y a une erreur d'allocation, les macros     |
| d'allocation appelees provoquent un exit         |
| apres impression d'un message                    |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    : NLEtape, NLVDon, PussIter |
| Macros utilisees: CREER_T2, CREER_T1             |
--------------------------------------------------*/


/*--------------- Definition fonction -----------*/
TShortInt CreerMatC( TShortInt nblig, TShortInt nbcol, TMat *pmat)
/*--------------- Fin identification fonction ----*/

{
TShortInt i, nl, nc;
double *pval; /* pointeur sur la zone qui contient les valeurs */

pmat->nblig=nblig;
pmat->nbcol=nbcol;


if ((nblig <= (TShortInt)0) || (nbcol <= (TShortInt)0))
  {
  /* creation d'un element bidon pour que le pointeur sur donnees existe */
  nl = (TShortInt)1;
  nc= (TShortInt)1;
  }
else
  {
  nl = nblig;
  nc = nbcol;
  }
CREER_T2(pmat->donnees, nl, TDouble);
/* allocation d'une zone memoire continue de nl*nc elements */
CREER_T1(pval, (nc*nl), TDouble);

for(i=0; i<nl; i++)
  {
  pmat->donnees[i]= pval+(i*nc);
  }

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CreerMatShort            |
| Role                  :                          |
|    Allouer le composant donnees et affecter      |
|    la valeur de nblig et nbcol,                  |
|    dans une structure de type                    |
|    TMatShort                                     |
| Parametres d'entree   :                          |
|    nblig: nombre de lignes                       |
|    nbcol: nombre de colonnes                     |
| Parametres d'e./s.    :                          |
|    pmat : adresse de la structure TMatShort      |   
|    en sortie les composants nblig et nbcol       |
|    de la structure pointee par pmat sont affectes|
|    et le composant donnees pointe sur un vecteur |
|    de nblig pointeurs, chacun d'entre eux        |
|    pointant sur un vecteur de nbcol elements     |
|    de type TShortInt                             |
|    Pointeur sur un element fictif si nblig ou    |
|     nbcol=0                                      |
| Retour:                OK                        |
| S'il y a une erreur d'allocation, les macros     |
| d'allocation appelees provoquent un exit         |
| apres impression d'un message                    |
| Reference conception  :                          |
--------------------------------------------------*/



/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    :                           |
|       MajContP,                                  |
| Macros utilisees: CREER_T2, CREER_T1             |
--------------------------------------------------*/

/*--------------- Definition fonction -----------*/
TShortInt CreerMatShort(TShortInt nblig, TShortInt nbcol, TMatShort *pmat)
/*--------------- Fin identification fonction ----*/

{
TShortInt i, nl, nc;

pmat->nblig=nblig;
pmat->nbcol=nbcol;

if ((nblig <= (TShortInt)0) || (nbcol <= (TShortInt)0))
  {
  /* creation d'un element bidon pour que le pointeur sur donnees existe */
  nl = (TShortInt)1;
  nc= (TShortInt)1;
  }
else
  {
  nl = nblig;
  nc = nbcol;
  }

CREER_T2(pmat->donnees, nl, TShortInt);

for(i=0; i<nl; i++)
  {
  CREER_T1(pmat->donnees[i], nc, TShortInt);
  }

return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : CreerVect                |
| Role                  :                          |
|    Allouer le composant donnees et affecter      |
|    la valeur de nbele, dans une structure de     |
|    type TVect                                    |
| Parametres d'entree   :                          |
|    nbele: nombre d'elements (eventuellement 0)   |
| Parametres d'e./s.    :                          |
|    pvect:                                        |
|      En entree: pointe sur une structure TVect.  |   
|      En sortie: le composant nbele de la         |
|      structure pointee par pvect contient nbele, |
|      et le composant donnees est un pointeur     |
|      sur un vecteur de nbele elements de type    | 
|      TDouble                                     |
|      ou pointeur sur un element fictif si nbele=0|
| Retour:                OK                        |
| S'il y a une erreur d'allocation, les macros     |
| d'allocation appelees provoquent un exit         |
| apres impression d'un message                    |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    :                           |
|       CModele,                                   |
|       MajContP,                                  |
|       NLAutres,                                  |
|       NLDebut,                                   |
|       NLEtape,                                   |
|       NLExhau,                                   |
|       NLVDon,                                    |
|       NLVMod,                                    |
|       PussIter                                   |
|       NLRacine                                   |
| Macros utilisees: CREER_T1                       |
--------------------------------------------------*/

/*--------------- Definition fonction -----------*/
TShortInt CreerVect( TShortInt nbele, TVect *pvect)
/*--------------- Fin identification fonction ----*/
{
TShortInt ne;
pvect->nbele=nbele;

if (nbele<=0)
  {
  ne =1;
  }
else
  {
  ne = nbele;
  }

CREER_T1(pvect->donnees, ne, TDouble);

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CreerVectLong            |
| Role                  :                          |
|    Allouer le composant donnees et affecter      |
|    la valeur de nbele, dans une structure        |
|    de type TVectLong                             |
| Parametres d'entree   :                          |
|    nbele: nombre d'elements (eventuellement 0)   |
| Parametres d'e./s.    :                          |
|    pvect:                                        |
|      En entree: pointe sur une structure TVect.  |   
|      En sortie: le composant nbele de la         |
|      structure pointee par pvect contient nbele, |
|      et le composant donnees est un pointeur     |
|      sur un vecteur de nbele elements de type    | 
|      TLongInt                                    |
|      ou pointeur sur un element fictif si nbele=0|
| Retour:                OK                        |
| S'il y a une erreur d'allocation, les macros     |
| d'allocation appelees provoquent un exit         |
| apres impression d'un message                    |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    :                           |
|       NLVDon,                                    |
| Macros utilisees: CREER_T1                       |
--------------------------------------------------*/

/*--------------- Definition fonction -----------*/
TShortInt CreerVectLong (TShortInt nbele, TVectLong  *pvect)
/*--------------- Fin identification fonction ----*/
{
TShortInt ne;

pvect->nbele=nbele;

if (nbele<=0)
  {
  ne =1;
  }
else
  {
  ne = nbele;
  }

CREER_T1(pvect->donnees, ne, TLongInt);
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CreerVectShort           |
| Role                  :                          |
|    Allouer le composant donnees et affecter      |
|    la valeur de nbele, dans une structure        |
|    de type TVectShort                            |
| Parametres d'entree   :                          |
|    nbele: nombre d'elements (eventuellement 0)   |
| Parametres d'e./s.    :                          |
|    pvect:                                        |
|      En entree: pointe sur une structure TVect.  |   
|      En sortie: le composant nbele de la         |
|      structure pointee par pvect contient nbele, |
|      et le composant donnees est un pointeur     |
|      sur un vecteur de nbele elements de type    | 
|      TShortInt                                   |
|      ou pointeur sur un element fictif si nbele=0|
| Retour:                OK                        |
| S'il y a une erreur d'allocation, les macros     |
| d'allocation appelees provoquent un exit         |
| apres impression d'un message                    |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    :                           |
|       NLAutres,                                  |
|       NLDebut,                                   |
|       NLVDon,                                    |
|       NLVMod,                                    |
| Macros utilisees: CREER_T1                       |
--------------------------------------------------*/

/*--------------- Definition fonction -----------*/
TShortInt CreerVectShort(TShortInt nbele, TVectShort *pvect)
/*--------------- Fin identification fonction ----*/
{
TShortInt ne;

pvect->nbele=nbele;

if (nbele<=0)
  {
  ne =1;
  }
else
  {
  ne = nbele;
  }

CREER_T1(pvect->donnees, ne, TShortInt);
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : CreerVectStr             |
| Role                  :                          |
|    Allouer le composant donnees et affecter      |
|    la valeur de nbele, dans une structure        |
|    de type TVectStr                              |
| Parametres d'entree   :                          |
|    nbele: nombre d'elements (eventuellement 0)   |
| Parametres d'e./s.    :                          |
|    pvect:                                        |
|      En entree: pointe sur une structure TVectStr|   
|      En sortie: le composant nbele de la         |
|      structure pointee par pvect contient nbele, |
|      et le composant donnees est un pointeur     |
|      sur un vecteur de nbele elements de type    | 
|      TChaine                                     |
|      ou pointeur sur un element fictif si nbele=0|
| Retour:                OK                        |
| S'il y a une erreur d'allocation, les macros     |
| d'allocation appelees provoquent un exit         |
| apres impression d'un message                    |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Definition fonction -----------*/
TShortInt CreerVectStr( TShortInt nbele, TVectStr *pvect)
/*--------------- Fin identification fonction ----*/
{
TShortInt i, ne;

pvect->nbele=nbele;

if (nbele<=0)
  {
  ne =1;
  }
else
  {
  ne = nbele;
  }


  CREER_T2(pvect->donnees, ne, TChar);

  for(i=0; i<ne; i++)
    {
    CREER_T1(pvect->donnees[i], MAXCH, char);
    }

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : EcrMat                   |
| Role                  :  Ecriture des valeurs    |
|   d'une matrice de type TMat,                    |
|   en format %g, 5 valeurs par ligne              |
| Parametres d'entree   :                          |
|  unite: unite d'impression                       |
|  MatIn: matrice  a ecrire                        |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    :                           |
|       NLIp                                       |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt EcrMat(FILE *unite,TMat *MatIn)
/* ........ parametres d'entree:.................................. */
     /*
FILE *unite;
TMat *MatIn;
     */

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i,j;
TDouble **matin;
TShortInt matinl, matinc;

/* ECRITURE DES VALEURS D'UNE MATRICE */
matin= MatIn->donnees;
matinl = MatIn->nblig;
matinc = MatIn->nbcol;
if ((matinl <= 0) || (matinc <=0 ))
  {
  fprintf(unite, "aucun element\n");
  return(OK);
  }


for (i = 0; i < matinl; i++)
  {
  for(j = 0; j < matinc; j++)
    {
    fprintf(unite," %20.10g", matin[i][j]);
    if ((j+1) % 5 == 0) fprintf(unite,"\n");
    }
  fprintf(unite,"\n");
  }
fprintf(unite,"\n");
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : EcrMatShort              |
| Role                  :  Ecriture des valeurs    |
|   d'une matrice de type TMatShort,               |
|   en format %d, 5 valeurs par ligne              |
| Parametres d'entree   :                          |
|  unite: unite d'impression                       |
|  MatIn: matrice  a ecrire                        |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    :                           |
|       NLIp                                       |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt EcrMatShort(FILE *unite,TMatShort *MatIn)
/* ........ parametres d'entree:.................................. */

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i,j;
TShortInt **matin;
TShortInt matinl, matinc;

/* ECRITURE DES VALEURS D'UNE MATRICE Short */
matin= MatIn->donnees;
matinl = MatIn->nblig;
matinc = MatIn->nbcol;

if ((matinl <= 0) || (matinc <=0 ))
  {
  fprintf(unite, "aucun element\n");
  return(OK);
  }



for (i = 0; i < matinl; i++)
  {
  for(j = 0; j < matinc; j++)
    {
    fprintf(unite," %d", matin[i][j]);
    if ((j+1) % 5 == 0) fprintf(unite,"\n");
    }
  fprintf(unite,"\n");
  }
fprintf(unite,"\n");
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : EcrVect                  |
| Role                  :  Ecriture des valeurs    |
|   d'un vecteur de type TVect,                    |
|   en format %g, 5 valeurs par ligne              |
| Parametres d'entree   :                          |
|  unite: unite d'impression                       |
|  VectIn: vecteur a ecrire                        |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    :                           |
|       CModele,                                   |
|       NLIp                                       |
|       PussIter                                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt EcrVect(FILE *unite,TVect *VectIn)
/* ........ parametres d'entree:.................................. */

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i;
TDouble *vectin;
TShortInt vectinn;

/* ECRITURE DES VALEURS D'UN VECTEUR */
vectin = VectIn->donnees;
vectinn = VectIn->nbele;
if (vectinn <= 0)
  {
  fprintf(unite, "aucun element\n");
  return(OK);
  }


for (i = 0; i < vectinn; i++)
  {
  fprintf(unite, " %20.10g", vectin[i]);
  if ((i+1) % 5 == 0) fprintf(unite,"\n");
  }
fprintf(unite,"\n");
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : EcrVectLong              |
| Role                  :  Ecriture des valeurs    |
|   d'un vecteur de type TVectLong,                |
|   en format %ld, 5 valeurs par ligne             |
| Parametres d'entree   :                          |
|  unite: unite d'impression                       |
|  VectIn: vecteur a ecrire                        |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    :                           |
|       NLIp                                       |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt EcrVectLong(FILE *unite,TVectLong *VectIn)
/* ........ parametres d'entree:.................................. */

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i;
TLongInt *vectin;
TShortInt vectinn;

/* ECRITURE DES VALEURS D'UN VECTEUR TVectLong*/
vectin = VectIn->donnees;
vectinn = VectIn->nbele;
if (vectinn <= 0)
  {
  fprintf(unite, "aucun element\n");
  return(OK);
  }


for (i = 0; i < vectinn; i++)
  {
  fprintf(unite, " %20ld", vectin[i]);
  if ((i+1) % 5 == 0) fprintf(unite,"\n");
  }
fprintf(unite,"\n");
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : EcrVectShort             |
| Role                  :  Ecriture des valeurs    |
|   d'un vecteur de type TVectShort,               |
|   en format %hd, 5 valeurs par ligne             |
| Parametres d'entree   :                          |
|  unite: unite d'impression                       |
|  VectIn: vecteur a ecrire                        |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    :                           |
|       NLIp                                       |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt EcrVectShort(FILE *unite,TVectShort *VectIn)
/* ........ parametres d'entree:.................................. */

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i;
TShortInt *vectin;
TShortInt vectinn;

/* ECRITURE DES VALEURS D'UN VECTEUR TVectShort*/
vectin = VectIn->donnees;
vectinn = VectIn->nbele;
if (vectinn <= 0)
  {
  fprintf(unite, "aucun element\n");
  return(OK);
  }


for (i = 0; i < vectinn; i++)
  {
  fprintf(unite, " %20hd", vectin[i]);
  if ((i+1) % 5 == 0) fprintf(unite,"\n");
  }
fprintf(unite,"\n");
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : EcrVectStr               |
| Role                  :  Ecriture des valeurs    |
|   d'un vecteur de type TVectStr,                 |
|   5 valeurs par ligne                            |
| Parametres d'entree   :                          |
|  unite: unite d'impression                       |
|  VectIn: vecteur a ecrire                        |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants    :                           |
|       NLIp                                       |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt EcrVectStr(FILE *unite,TVectStr *VectIn)
/* ........ parametres d'entree:.................................. */

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i;
TChar **vectin;
TShortInt vectinn;

/* ECRITURE DES VALEURS D'UN VECTEUR STR*/
vectin = VectIn->donnees;
vectinn = VectIn->nbele;
if (vectinn <= 0)
  {
  fprintf(unite, "aucun element\n");
  return(OK);
  }


for (i = 0; i < vectinn; i++)
  {
  fprintf(unite, " %s", vectin[i]);
  if ((i+1) % 5 == 0) fprintf(unite,"\n");
  }
fprintf(unite,"\n");
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : ReCreerMat               |
| Role                  :                          |
|    Reallouer le composant donnees precedemment   |
|    alloue avec des dimensions differentes dans   |
|    une matrice de type TMat                      |
| Parametres d'entree   :                          |
|    nblig: nombre de lignes                       |
|    nbcol: nombre de colonnes                     |
| Parametres d'e./s.    :                          |
|    pmat : en entree, pointe sur la matrice       |
|    precedemment allouee,                         |
|    en sortie, pointe sur la matrice allouee      |
|     avec nblig lignes et nbcol colonnes          |
|    Pointeur sur element fictif si nblig ou       |
|    nbcol=0                                       |
| Retour:                OK                        |
| S'il y a une erreur d'allocation, les macros     |
| d'allocation appelees provoquent un exit         |
| apres impression d'un message                    |
| Reference conception  :                          |
--------------------------------------------------*/


/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants: aucun pour l'instant          |
| Macros utilisees: CREER_T1, DETRU_T1,            |
|                   RECREER_T2, RECREER_T1         |
--------------------------------------------------*/

/*--------------- Definition fonction -----------*/
TShortInt ReCreerMat( TShortInt nblig, TShortInt nbcol, TMat *pmat)
/*--------------- Fin identification fonction ----*/

{
TShortInt i, nbligp, nbcolp, iborne;

nbligp = pmat->nblig;
nbcolp = pmat->nbcol;

pmat->nblig=nblig;
pmat->nbcol=nbcol;

if ((nblig>0) && (nbcol >0))
  {
  RECREER_T2(pmat->donnees, nblig, TDouble);

  /* recreer les lignes precedemment creees */
  iborne = MIN(nbligp, nblig);
  for(i=0; i<iborne; i++)
    {
    RECREER_T1(pmat->donnees[i], nbcol, TDouble);
    }

  /* creer les lignes supplementaires, si besoin */
  if (nblig > nbligp)
    {
    for(i=nbligp; i<nblig; i++)
      {
      CREER_T1(pmat->donnees[i], nbcol, TDouble);
      }
    }
  else
    {
    /* cas ou la matrice nouvelle comprend moins que precedemment:
    on detruit les lignes en surplus */
    for(i=nbligp; i<nblig; i++)
      {
      DETRU_T1(pmat->donnees[i], nbcolp, TDouble);
      }
    } /* fin du else sur nblig > nbligp */
  } /* fin du if (nblig>0) && (nbcol >0) */

if ((pmat->nblig <=0 ) || (pmat->nbcol <=0 ))
  {
  /* creer elements fictifs */
  CREER_T2(pmat->donnees, 1, TDouble);
  CREER_T1(pmat->donnees[0], 1, TDouble);
  }

  

return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : ReCreerVect              |
| Role                  :                          |
|    Reallouer un vecteur precedemment alloue avec |
|    une dimension differente de celle qu'il avait |
|    precedemment                                  |
|    Type du vecteur: TVect                        |
| Parametres d'entree   :                          |
|    nbele: nombre d'elements (eventuellement 0)   |
| Parametres d'e./s.    :                          |
|    pvect:                                        |
|      En entree: pointeur sur un vecteur          |
|       precedemment alloue,                       |
|      en sortie: pointeur sur un vecteur de       |
|      nbele elements,                             |
|      ou pointeur NULL si nbele=0                 |
| Retour:                OK                        |
| S'il y a une erreur d'allocation, les macros     |
| d'allocation appelees provoquent un exit         |
| apres impression d'un message                    |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Modules appelants: aucun pour l'instant          |
| Macros utilisees: RECREER_T1                     |
--------------------------------------------------*/

/*--------------- Definition fonction -----------*/
TShortInt ReCreerVect(TShortInt nbele, TVect *pvect)

/*--------------- Fin identification fonction ----*/
{

pvect->nbele=nbele;
if (nbele>0)
  {
  RECREER_T1(pvect->donnees, nbele, TDouble); 
  }
else
  {
  RECREER_T1(pvect->donnees, 1, TDouble);
  }
return(OK);
}

