/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : NLVCtxNum                 |
| Date                 :  5Nov91 / 14:40:46        |
| Derniere mise a jour :                           |
| Concepteur           : A. Bouvier                |
| Role                 : Verifier et initialiser   |
|   les contextes numeriques                       |
| Reference conception :                           |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|.
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlmacros.h"
#include "errcodes.h"

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/
/* fonctions des autres modules */
TShortInt  GerMessage();
TLongInt IMinL();

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/




/*--------------- Identification fonction ----------
| Nom de la fonction    : AffectNum                |
| Role                  : Affecter des valeurs     |
| aux composants du contexte numerique quand       |
| l'estimateur n'est pas defini par l'utilisateur  |
| (c'est-a-dire quand Estim!= LEMIEN)              |
| Parametres d'entree   :                          |
|  Etape: indice de l'etape courante               |
|  Vari: indications sur la variance               |
|  TypeMu: facon de calculer les Mu                |
| Parametres d'e.s      :                          |
|  CtxNum : le contexte numerique                  |
|  Estim est une entree, et les composants suivants|
|  sont des sorties:                               |
|  Effic: efficacite de l'estimateur               |
|  NbZ: dimension des statistiques exhaustives     |
|  Symm: forme de la matrice a inverser            |
|  PC..: pointeurs sur les programmes de calcul    |
|   des equations normales                         |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes : NLVCtxNum                 |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt AffectNum(Etape, Vari, TypeMu, CtxNum)

  /* arguments d'entree */
  TShortInt Etape, Vari, TypeMu;

  /* arguments d'entree sortie */
  TCtxNum *CtxNum;

/*--------------- Fin identification fonction ----*/
  {


  /* pointeurs sur les programmes  */
extern  TShortInt 
                  CBmcmb(), CBmcmtb(), CBmcob(), CBmcot(),
                    CBmcpvit(), CBmct(), CBmvt(), CBmvtb(),
                  CDmcb(), CDmct(), CDmv(),  CDmvt(),
                  CEtamco(), CEtamv(),
                  CVarZmco(), CVarZmv(),
                  CZmco(), CZmv(); 

                   

/* Mettre son nom dans la trace */
  ECRTRACE("AffectNum");


  if (Etape %2  == 1) /* etape impaire: estimation simultanee ou theta seuls */
    {
    switch(CtxNum->Estim)
      {
      case MVTB:
        CtxNum->NbZ = 2;
        CtxNum->Effic = OUI;
        CtxNum->Symm = SYM;
        CtxNum->PCValZ = CZmv;
        CtxNum->PCEquN.PCValB = CBmvtb;
        CtxNum->PCEquN.PCValEta = CEtamv;
        CtxNum->PCEquN.PCValD = CDmv;
        CtxNum->PCVarZ = CVarZmv;
        break;
  
      case MCMTB:
      case QVTB:
        CtxNum->NbZ = 2;
        if( (Vari != VST) && (Vari != VSTB) && (Vari != VTB))
          {
          /* la variance ne depend pas des theta */
          CtxNum->Effic = OUI;
          }
        else
          {
          CtxNum->Effic = NON;
          }
        CtxNum->Symm = SYMBLOC;
        CtxNum->PCValZ = CZmv;
        CtxNum->PCEquN.PCValB = CBmcmtb;
        CtxNum->PCEquN.PCValEta = CEtamv;
        CtxNum->PCEquN.PCValD = CDmv;
        CtxNum->PCVarZ = CVarZmv;
        break;
  
      case MVT:
        CtxNum->NbZ = 2;
        CtxNum->Effic = OUI;
        CtxNum->Symm = SYM;
        CtxNum->PCValZ = CZmv;
        CtxNum->PCEquN.PCValB = CBmvt;
        CtxNum->PCEquN.PCValEta = CEtamv;
        CtxNum->PCEquN.PCValD = CDmvt;
        CtxNum->PCVarZ = CVarZmv;
        break;
  
  
      case MCPT:
        CtxNum->NbZ  =  1;
        CtxNum->Effic = OUI;
        CtxNum->Symm = SYM;
        CtxNum->PCValZ = CZmco;
        CtxNum->PCEquN.PCValB = CBmct;
        CtxNum->PCEquN.PCValEta = CEtamco;
        CtxNum->PCEquN.PCValD = CDmct;
        CtxNum->PCVarZ = CVarZmco;
        break;
  
  
      case MCOT:
        CtxNum->NbZ = 1;
        if( Vari == CTE)
          {
          /* la variance est constante */
           CtxNum->Effic = OUI;
          }
        else
          {
          CtxNum->Effic = NON;
          }
        CtxNum->Symm = SYM;
        CtxNum->PCValZ = CZmco;
        CtxNum->PCEquN.PCValB = CBmcot;
        CtxNum->PCEquN.PCValEta = CEtamco;
        CtxNum->PCEquN.PCValD = CDmct;
        CtxNum->PCVarZ = CVarZmco;
        break;
  
  
      case MCMT:
      case QVT:
        CtxNum->NbZ = 1;
        CtxNum->Effic = NON;
        CtxNum->Symm = SYM;
        CtxNum->PCValZ = CZmco;
        CtxNum->PCEquN.PCValB = CBmct;
        CtxNum->PCEquN.PCValEta = CEtamco;
        CtxNum->PCEquN.PCValD = CDmct;
        CtxNum->PCVarZ = CVarZmco;
        break;
  
  
      case MCPVIT:
        CtxNum->NbZ = 1;
        CtxNum->Effic = OUI;
        CtxNum->Symm = SYM;
        CtxNum->PCValZ = CZmco;
        CtxNum->PCEquN.PCValB = CBmcpvit;
        CtxNum->PCEquN.PCValEta = CEtamco;
        CtxNum->PCEquN.PCValD = CDmct;
        CtxNum->PCVarZ = CVarZmco;
        break;
  
      default: ;
      /* cas impossible */
      } /*fin du switch */
    } /* fin de Etape impaire */


  if (Etape %2  == 0) /* etape paire: estimation des beta seuls */
    {
    switch(CtxNum->Estim)
      {
      case MCOB:
        CtxNum->NbZ = 2;
        CtxNum->Effic = NON;
        CtxNum->Symm = SYM;
        CtxNum->PCValZ = CZmv;
        CtxNum->PCEquN.PCValB = CBmcob;
        CtxNum->PCEquN.PCValEta = CEtamv;
        CtxNum->PCEquN.PCValD = CDmcb;
        CtxNum->PCVarZ = CVarZmv;
        break;
  
      case MCMB:
      case QVB:
        CtxNum->NbZ = 2;
        CtxNum->Effic = NON;
        CtxNum->Symm = SYM;
        CtxNum->PCValZ = CZmv;
        CtxNum->PCEquN.PCValB = CBmcmb;
        CtxNum->PCEquN.PCValEta = CEtamv;
        CtxNum->PCEquN.PCValD = CDmcb;
        CtxNum->PCVarZ = CVarZmv;
        break;
  
      default: ;
      /* cas impossible */
      } /*fin du switch */
    } /* fin de Etape paire */

/* correction de Effic selon TypeMu */
  if ((CtxNum->NbZ == 2 ) && (TypeMu != MUGAUSS))
    {
    CtxNum->Effic = NON;
    }

  return(OK);
  }


/*--------------- Identification fonction ----------
| Nom de la fonction    : EstDefaut                |
| Role                  : Determiner l'estimateur  |
|  quand l'utilisateur l'a initialise a DEFAUT     |
|  selon les valeurs de Vari (de Modele) et de     |
|  l'indice de l'etape courante                    |
| Parametres d'entree   :                          |
|  Vari: indications sur la variance (cf Modele)   |
|  Etape: indice de l'etape courante               |
|  (commence a  1)                                 |
|  NbEtapes: nombre d'etapes                       |
| Parametres de sortie:                            |
|  Estim: valeur du code de l'estimateur           |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes : NLVCtxNum                 |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt EstDefaut(Vari,Etape, NbEtapes,
                    Estim)

  /* arguments d'entree */
  TShortInt Vari;
  TShortInt Etape, NbEtapes;

  /* arguments de sortie */
  TShortInt *Estim;

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TChar MessE[5], Mess1[5];


/* Mettre son nom dans la trace */
  ECRTRACE("EstDefaut");

  switch(Vari)
    {
    case VB:
    case VSB:
    case VSTB:
    case VTB:
    /* la variance inclut Beta */
      if(NbEtapes == 1)
        {
        *Estim = MVTB;
        }
      else
        {
        if (Etape  == 1)
          {
          *Estim = MCOT;
          }
        if (Etape %2  == 0) /* etape paire */
          {
          *Estim = MCMB;
          }
         if(( Etape %2  == 1) && (Etape > 1)) /* etape impaire >1 */
          {
          *Estim = MCMT;
          }
        } /* fin du else */
      break; /* sortie du case */

    case CTE:
      *Estim = MCOT;
      break;

    case SP:
      *Estim = MCPT;
      break;

    case VI: /* variance intra */
      *Estim = MCPVIT;
      break;

    default: /* tous les autres cas */
      *Estim = MVT;
    } /* fin du switch */

  /* On signale le choix qui a ete fait */
  sprintf(MessE,"%hd", Etape);
  sprintf(Mess1,"%hd", *Estim);

   /* MESSAGE ecrit par NLWARNING: */
/*
  printf("L'estimateur choisi a l'etape %s etait l'estimateur par defaut\n", MessE);
  printf("Compte-tenu du type de la variance (Modele.Vari), du nombre d'etapes\n");
  printf(" et de l'indice de l'etape courante, on fixe celui-ci a %s\n", Mess1);
*/
  /* FIN MESSAGE */
  NLWARNING((WARESD1,3,"EstDefaut",MessE,Mess1,WMETHO));

  return(OK);
  }


/*--------------- Identification fonction ----------
| Nom de la fonction    : EstMien                  |
| Role                  : faire les verifications  |
| de CtxNum quand l'estimateur est LEMIEN          |
| et initialiser les pointeurs sur les programmes  |
| de calcul des equations normales                 |
| et des statistiques exhaustives                  |
| (le nom de ces programmes etant impose:          |
|   calcb, calcd, calceta, calcz, calcvarz)        |
| Parametres d'entree   :                          |
|  Etape: indice de l'etape courante               |
|  (sert uniquement pour les messages)             |
| Parametres d'e.s      :                          |
| CtxNum: le contexte numerique, dont les          |
| composants suivants sont des entrees:            |
|  Effic: efficacite de l'estimateur               |
|  NbZ: dimension des statistiques exhaustives     |
|  Symm: forme de la matrice a inverser            |
| et les composants suivants sont des sorties:     |
|  PC..: pointeurs sur les programmes de calcul    |
|   des equations normales                         |
| Parametres de sortie  :                          |
|  Code: OK ou code de warning                     |
|    Le code d'erreur n'est pas dans le retour     |
|    fonction, afin de revenir au programme        |
|    appelant quand il y a une erreur              |
|    ce qui ne serait pas le cas sinon, ce         |
|    programme etant appele via la macro APPEL     |
|    pour pouvoir etre jete proprement si le nombre|
|    de warnings est depasse                       |
| Retour fonction       : OK ou Nombre de warnings |
|   depasse                                        |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes : NLVCtxNum                 |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt EstMien(Etape,CtxNum, Code)

  /* arguments d'entree */
  TShortInt Etape;

  /* arguments d'entree sortie */
  TCtxNum *CtxNum;

  /* arguments de sortie */
  TShortInt *Code;

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TChar MessE[5], Mess1[5], Mess2[5], Mess3[5], Mess4[5];
  /* pointeurs sur les programmes supposes fournis */
   TShortInt calcb(), calcd(), calceta(), calcz(), calcvarz() ;


/* Mettre son nom dans la trace */
  ECRTRACE("EstMien");

  *Code = OK;

  /* Verif de NbZ */
  if(CtxNum->NbZ <= 0)
    {
    sprintf(MessE,"%hd", Etape);
    sprintf(Mess1,"%hd", CtxNum->NbZ);

   /* MESSAGE ecrit par NLWARNING: */
/*
    printf("L'estimateur choisi a l'etape %s est LEMIEN", MessE); 
    printf(" mais la dimension des statistiques exhaustives %s (CtxNum[%s].NbZ)  est erronee\n",
           Mess1);
    printf( " elle doit etre positive\n");
*/
    /* FIN MESSAGE */
    NLWARNING((WARESM1,3,"EstMien",MessE,Mess1, WMETHO));

    *Code =  WARESM1;
    }

  /* verif de Symm */
  if ((CtxNum->Symm != SYM) && (CtxNum->Symm != SYMBLOC) && (CtxNum->Symm != NONSYM) )
    {
    sprintf(MessE,"%hd", Etape);
    sprintf(Mess1,"%hd", CtxNum->Symm);
    sprintf(Mess2,"%hd", SYM);
    sprintf(Mess3,"%hd", SYMBLOC);
    sprintf(Mess4,"%hd", NONSYM);

   /* MESSAGE ecrit par NLWARNING: */
/*
    printf("L'estimateur choisi a l'etape %s est LEMIEN", MessE); 
    printf(  " mais la forme de la matrice a inverser %s (CtxNum[%s].Symm) est erronee\n", 
           Mess1);  
    printf(" elle doit etre %s (SYM) ou %s (SYMB) ou %s (NONSYM) \n", 
           Mess2, Mess3, Mess4);
*/
    /* FIN MESSAGE */
    NLWARNING((WARESM2,6,"EstMien",MessE,Mess1, Mess2, Mess3,Mess4, WMETHO));

    *Code = WARESM2;
    }

  /* Verif de Effic */
  if ((CtxNum->Effic != OUI) && (CtxNum->Effic != NON))
    {
    sprintf(MessE,"%hd", Etape);
    sprintf(Mess1,"%hd", CtxNum->Effic);
    sprintf(Mess2,"%hd", OUI);
    sprintf(Mess3,"%hd", NON);

   /* MESSAGE ecrit par NLWARNING: */
/*
    printf("L'estimateur choisi a l'etape %s est LEMIEN", MessE); 
    printf( " mais l'indication de son efficacite %s (CtxNum[%s].Effic)  est erronee\n", 
           Mess1);
    printf( " elle doit etre %s (OUI) ou %s (NON)\n" , Mess2, Mess3);
*/
    /* FIN MESSAGE */
    NLWARNING((WARESM3,5,"EstMien",MessE,Mess1, Mess2, Mess3, WMETHO));

    *Code = WARESM3;
    }


  /* Initialiser les pointeurs sur les programmes de calcul
     des equations normales: ceux-ci sont supposes fournis */
  CtxNum->PCEquN.PCValB =  calcb;
  CtxNum->PCEquN.PCValD =  calcd;
  CtxNum->PCEquN.PCValEta =  calceta;
  CtxNum->PCValZ =  calcz;
  CtxNum->PCVarZ =  calcvarz;
  return(OK);
  }
   


/*--------------- Identification fonction ----------
| Nom de la fonction    : VerifMien                |
| Role                  : faire les verifications  |
| de TypeCritStat quand l'estimateur est LEMIEN    |
| Parametres d'entree   :                          |
|  Etape: indice de l'etape courante               |
|  TypeCritStat                                    |
|  Vari: indications sur la variance (cf Modele)   |
|  NbRepet: nombre de repetitions en chaque point  |
|    d'observation                                 |
| Parametres de sortie  :                          |
|  Code: OK ou code de warning                     |
|    Le code d'erreur n'est pas dans le retour     |
|    fonction, afin de revenir au programme        |
|    appelant quand il y a une erreur              |
|    ce qui ne serait pas le cas sinon, ce         |
|    programme etant appele via la macro APPEL     |
|    pour pouvoir etre jete proprement si le nombre|
|    de warnings est depasse                       |
| Retour fonction       : OK ou Nombre de warnings |
|   depasse                                        |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes : NLVCtxNum                 |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt VerifMien(Etape,TypeCritStat, Vari, NbRepet, Code)

  /* arguments d'entree */
  TShortInt Etape, TypeCritStat, Vari;
  TVectLong *NbRepet;

  /* arguments de sortie */
  TShortInt *Code;

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TChar MessE[5], Mess1[5], Mess2[5];
  TLongInt MinRep;

/* Mettre son nom dans la trace */
  ECRTRACE("VerifMien");
  *Code = OK;

  /* Verif de TypeCritStat */
  if( (TypeCritStat != CARRET) && (TypeCritStat != CLV) &&
      (TypeCritStat != CSCRNP) && (TypeCritStat != CSCRPV) &&
      (TypeCritStat != CSCRPS2) && (TypeCritStat != CSCR2NP) && 
      (TypeCritStat != CRITSIG)  && (TypeCritStat != LEMIEN))
    {
    sprintf(MessE,"%hd", Etape);
    sprintf(Mess1,"%hd", TypeCritStat);

   /* MESSAGE ecrit par NLWARNING: */
/*
    printf("L'estimateur choisi a l'etape %s est LEMIEN", MessE); 
    printf(" mais le  mode de calcul du critere statistique %s (TypeCritStat) est errone\n",
           Mess1);
*/
    /* FIN MESSAGE */
    NLWARNING((WARVFM1,3,"VerifMien",MessE,Mess1, WMETHO));
    *Code =  WARVFM1;
    }

  /* verif de la concordance avec l'etape */
  if (Etape %2  == 1) /* etape impaire: estimation simultanee ou theta seuls */
    {
    /* ne peut etre  CSCR2NP */
    if (TypeCritStat == CSCR2NP)
      {
      sprintf(MessE,"%hd", Etape);
      sprintf(Mess1,"%hd", TypeCritStat);

     /* MESSAGE ecrit par NLWARNING: */
/*
      printf("L'estimateur choisi a l'etape impaire %s est LEMIEN", MessE);
    printf(" mais le  mode de calcul du critere statistique %s (TypeCritStat)\
 n'est possible que pour une etape paire (estimation des beta seuls)\n",
           Mess1);
*/
      /* FIN MESSAGE */
      NLWARNING((WARVFM2,3,"VerifMien",MessE,Mess1, WMETHO));
      *Code =  WARVFM2;
      }
    } /* fin de etape impaire */


  if (Etape %2  == 0) /* etape paire: estimation beta seuls */
    {
    /* ne peut etre  que CARRET, CSCR2NP, LEMIEN  */
    if ((TypeCritStat != CSCR2NP) && (TypeCritStat != CARRET) && (TypeCritStat != LEMIEN))
      {
      sprintf(MessE,"%hd", Etape);
      sprintf(Mess1,"%hd", TypeCritStat);

     /* MESSAGE ecrit par NLWARNING: */
/*
      printf("L'estimateur choisi a l'etape paire %s (estimation des beta seuls) est LEMIEN", MessE);
    printf(" mais le  mode de calcul du critere statistique %s (TypeCritStat)\
 n'est possible que pour une etape impaire\n",
           Mess1);
*/
      /* FIN MESSAGE */
      NLWARNING((WARVFM3,3,"VerifMien",MessE,Mess1, WMETHO));
      *Code =  WARVFM3;
      }
    } /* fin de etape paire */

  /* verif du nombre de repet et de la variance si CSCRPS2 */
  if (TypeCritStat == CSCRPS2)
    {
    MinRep = IMinL(NbRepet);
    if (MinRep <= MAXREP)
      {
      sprintf(MessE,"%hd", Etape);
      sprintf(Mess2,"%hd", TypeCritStat);
      sprintf(Mess1,"%hd", MAXREP);

      /* MESSAGE ecrit par NLWARNING: */
/*
      printf("L'estimateur choisi a l'etape %s est LEMIEN\n", MessE);
      printf(" et le mode de calcul du critere statistique %s (TypeCritStat)\n", Mess2);
      printf(               " mais il y a au moins un point d'observation \n");
      printf(               " pour lequel le nombre de repetitions est inferieur a %s (MAXREP)\n",
               Mess1);
*/
      /* FIN MESSAGE */
      NLWARNING((WARVFM4,4,"VerifMien",MessE,Mess2, Mess1,WMETHO));
      /* On ne change pas Code car ca ne doit pas engendrer un arret du programme:
      simple avertissement */
      }
    if (Vari != VI)
      {
      /* la variance doit etre le code de variance-intra */
      sprintf(MessE,"%hd", Etape);
      sprintf(Mess2,"%hd", TypeCritStat);
     /* MESSAGE ecrit par NLWARNING */
/*
      printf("L'estimateur choisi a l'etape %s est LEMIEN \n", MessE);
      printf(" et le mode de calcul du critere statistique %s (TypeCritStat)\n", Mess2);
      printf(" alors que la variance n'est pas variance intra (Modele.Vari)\n");
*/
      /* FIN MESSAGE */
      NLWARNING((WARVFM5,3,"VerifMien",MessE,Mess2, WMETHO));
      /* ca engendre un arret du programme */
      *Code = WARVFM5;
      }
    } /* fin de CSCR2NP */

  return(OK);
  }
   



/*--------------- Identification fonction ----------
| Nom de la fonction    : EstSimul                 |
| Role                  : Verifier la coherence    |
|   entre Vari (de Modele) et l'estimateur, quand  |
|   celui-ci est un estimateur simultane           |
| Parametres d'entree   :                          |
|  Vari: indications sur la variance (cf Modele)   |
|  Etape: indice de l'etape courante               |
|   (sert uniquement pour les messages)            |
| Parametres d'entree-sortie:                      |
|  Estim: valeur du code de l'estimateur           |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes : NLVCtxNum                 |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt EstSimul(Vari, Etape,
                    Estim)

  /* arguments d'entree */
  TShortInt Vari;
  TShortInt Etape;

  /* arguments d'entree-sortie */
  TShortInt *Estim;

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TChar MessE[5];


/* Mettre son nom dans la trace */
  ECRTRACE("EstSimul");


  switch(Vari)
    {
    case CTE:
      sprintf(MessE,"%hd", Etape);
     /* MESSAGE ecrit par NLWARNING: */
/*
      printf("L'estimateur choisi a l'etape %s est un estimateur simultane\n", MessE);
      printf( " alors que la variance est constante (Modele.Vari=CTE)\n");
      printf( "On fixe l'estimateur a MCOT\n");
*/
      /* FIN MESSAGE */
      NLWARNING((WARESS1,2,"EstSimul",MessE,WMETHO));

      *Estim = MCOT;
      break;
    case SP:
      sprintf(MessE,"%hd", Etape);

     /* MESSAGE ecrit par NLWARNING: */
/*
      printf("L'estimateur choisi a l'etape %s est un estimateur simultane\n", MessE);
      printf(" alors que la variance est ponderee (Modele.Vari=SP)\n");
      printf( "On fixe l'estimateur a MCPT\n");
*/
      /* FIN MESSAGE */
      NLWARNING((WARESS2,2,"EstSimul",MessE,WMETHO));

      *Estim = MCPT;
      break;

    case VST: /* le seul autre cas qui reste ou la variance n'inclut pas B */
      sprintf(MessE,"%hd", Etape);

     /* MESSAGE ecrit par NLWARNING */
/*
      printf("L'estimateur choisi a l'etape %s est un estimateur simultane\n", MessE);
      printf(        " alors que la variance n'inclut pas de Beta (Modele.Vari=VST)\n");
      printf(        "On enleve du code de l'estimateur la lettre B\n");
*/
      /* FIN MESSAGE */
      NLWARNING((WARESS3,2,"EstSimul",MessE,WMETHO));

      switch(*Estim)
        {
        case MVTB:
          *Estim = MVT;
          break;
        case MCMTB:
          *Estim = MCMT;
          break;
	case QVTB:
	  *Estim = QVT;
/*
        case MCOTB:
          *Estim = MCOT;
*/
        default: ;
        }
    default: ;
      } /* fin du switch sur Vari */

    return(OK);
  }


/*--------------- Identification fonction ----------
| Nom de la fonction    : EstTheta                 |
| Role                  : Verifier la coherence    |
|   entre Vari (de Modele), le nombre minimum de   |
|   repetitions en chaque point d'observation et   |
|   l'estimateur, quand                            |
|   celui-ci est un estimateur des Theta seuls     |
| Parametres d'entree   :                          |
|  Vari: indications sur la variance (cf Modele)   |
|  Etape: indice de l'etape courante               |
|  NbRepet: nombre de repetitions en chaque point  |
|    d'observation                                 |
| Parametres d'entree-sortie:                      |
|  Estim: valeur du code de l'estimateur           |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : IMinL                     |
| Fonctions appelantes : NLVCtxNum                 |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt EstTheta(Vari, Etape, NbRepet,
                    Estim)

  /* arguments d'entree */
  TShortInt Vari;
  TShortInt Etape;
  TVectLong *NbRepet;

  /* arguments d'entree-sortie */
  TShortInt *Estim;

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TChar MessE[5];
  TChar Mess1[5];
  TLongInt MinRep;



/* Mettre son nom dans la trace */
  ECRTRACE("EstTheta");

  switch(*Estim)
    {
    case MCPT:
      if ( (Etape == 1) && 
                 ((Vari == VST) || (Vari ==  VSTB) || (Vari == VTB)))
        {
        /* la variance depend des Theta */
        sprintf(MessE,"%hd", Etape);

        /* MESSAGE ecrit par NLWARNING: */
/*
        printf("L'estimateur choisi a l'etape %s est MCPT\n", MessE);
        printf(   " alors que la variance depend des Theta  (Modele.Vari)\n");
        printf(              "On fixe l'estimateur a MCOT\n");
*/
        /* FIN MESSAGE */
         NLWARNING((WAREST1,2,"EstTheta",MessE,WMETHO));

        *Estim = MCOT;
        }
      break;
    case MCMT:
    case QVT:
      if ((Vari != VST) && (Vari != VSTB) && (Vari != VTB))
        {
        /* la variance ne depend pas des Theta */
        sprintf(MessE,"%hd", Etape);
       /* MESSAGE ecrit par NLWARNING */
/*
        printf("L'estimateur choisi a l'etape %s est MCMT ou QVT \n", MessE);
        printf(" alors que la variance ne depend pas des Theta (Modele.Vari)\n");
        printf("On fixe l'estimateur a MCPT si Vari#SP et a MCOT sinon\n");
*/
        /* FIN MESSAGE */
        NLWARNING((WAREST2,2,"EstTheta",MessE,WMETHO));
        if (Vari == SP)
          {
          *Estim = MCPT;
          }
        else
          {
          *Estim = MCOT;
          }
        }
      break;
    case MCPVIT:
      MinRep = IMinL(NbRepet);
      if (MinRep  < MINREP)
        {
        sprintf(MessE,"%hd", Etape);
        sprintf(Mess1,"%hd", MINREP);

        /* MESSAGE ecrit par NLWARNING: */
/*
        printf("L'estimateur choisi a l'etape %s est MCPVIT\n", MessE);
        printf(        " alors qu'il faut au moins %s (MINREP) repetitions \n", Mess1);
        printf(               " en chacun des points d'observations pour pouvoir l'utiliser\n");
        printf(             " On fixe l'estimateur a MCOT\n");
*/
        /* FIN MESSAGE */
        NLWARNING((WAREST3,3,"EstTheta",MessE, Mess1, WMETHO));

        *Estim = MCOT;
        }
      if ((MinRep >= MINREP) && (MinRep <= MAXREP))
        {
        sprintf(MessE,"%hd", Etape);
        sprintf(Mess1,"%hd", MAXREP);

        /* MESSAGE ecrit par NLWARNING: */
/*
        printf("L'estimateur choisi a l'etape %s est MCPVIT\n", MessE);
        printf(               " mais il y a au moins un point d'observation \n");
        printf(               " pour lequel le nombre de repetitions est inferieur a %s (MAXREP)\n",
                 Mess1);
*/
        /* FIN MESSAGE */
        NLWARNING((WAREST4,3,"EstTheta",MessE,Mess1,WMETHO));

        }
      if (Vari != VI)
        {
        /* la variance doit etre le code de variance-intra */
        sprintf(MessE,"%hd", Etape);
       /* MESSAGE ecrit par NLWARNING */
/*
        printf("L'estimateur choisi a l'etape %s est MCPVIT \n", MessE);
        printf(" alors que la variance n'est pas variance intra (Modele.Vari)\n");
        printf("On fixe l'estimateur a MCPT si Vari=SP et a MCOT sinon\n");
*/
        /* FIN MESSAGE */
        NLWARNING((WAREST5,2,"EstTheta",MessE,WMETHO));
        if (Vari == SP)
          {
          *Estim = MCPT;
          }
        else
          {
          *Estim = MCOT;
          }
        }
      break;

    default: ;
     /* autre valeur: on ne fait rien */
    } /* fin du switch */

  return(OK);
  }


/*--------------- Identification fonction ----------
| Nom de la fonction    : NLVCtxNum                |
| Role                  : Verifier et initialiser  |
| le contexte numerique                            |
| Parametres d'entree   :                          |
|  Algo: algorithme                                |
|  TypeSigma: facon de calculer sigma              |
|  TypeMu: facon de calculer les Mu                |
|  Vari: indications sur la variance (cf Modele)   |
|  NbRepet: nombre de repetitions de chaque        |
|   observation                                    |
| Parametres d'entree-sortie  :                    |
|  NbEtapes: nombre d'etapes                       |
|  CtxNum: tableau des contextes numeriques (autant|
|   d'elements que d'etapes demandees)             |
| En entree, ces structures sont les valeurs       |
| demandees, en sortie, elles sont corrigees ou    |
| initialisees                                     |    
| Parametre de sortie   : OK ou code de warning    |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|    AffectNum,                                    |
|    EstDefaut,                                    |
|    EstMien,                                      |
|    EstSimul,                                     |
|    EstTheta                                      |
| Fonctions appelantes : NLVInit                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt NLVCtxNum(Algo, TypeSigma, TypeMu, Vari, NbRepet,
                    NbEtapes, CtxNum,
                    Code)

  /* arguments d'entree */
  TShortInt Algo;
  TShortInt TypeSigma;
  TShortInt TypeMu;
  TShortInt Vari;
  TVectLong *NbRepet;

  /* arguments d'entree- sortie */
  TShortInt *NbEtapes;
  TCtxNum CtxNum[];

  /* argu de sortie */
  TShortInt *Code;
/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TShortInt  Etape , IndEtap;
  TChar MessE[5];
  TLogic Trouve;


/* pointeurs sur les programmes appeles */
  TShortInt CDirGN(), CDirGNBloc(), CDirGM(), CDirGMBloc();

/* Mettre son nom dans la trace */
  ECRTRACE("NLVCtxNum");

  *Code = OK;

  Etape = 1;
  IndEtap = 0; /* indice dans les tableaux correspondant a la valeur de Etape
                evite de se melanger entre ce qui commence a 0 et a 1
                et rend le code plus lisible  */
  
  while(Etape <= *NbEtapes) /* attention: NbEtapes est modifie
                           dans le corps du while */
    {

    Trouve = FAUX; /* permet de savoir si l'estimateur a ete reconnu */
    /* VERIFICATION DE Estim */
    /* Je n'emploie pas une instruction switch car les tests 
       doivent etre executes successivement et dans cet ordre;
        ils ne sont pas exclusifs */
      
    if (CtxNum[IndEtap].Estim ==  DEFAUT)
      {
      /* l'utilisateur laisse le programme choisir */
      Trouve = VRAI;
      APPEL(EstDefaut(Vari,Etape, *NbEtapes,&(CtxNum[IndEtap].Estim)));
      }

    if (( CtxNum[IndEtap].Estim ==  MVTB) || 
	( CtxNum[IndEtap].Estim ==  MCMTB) ||
	( CtxNum[IndEtap].Estim ==  QVTB))
      {
      /* estimation simultanee des Theta et Beta */
      Trouve = VRAI;
      if(Etape > 1)
        {
        sprintf(MessE,"%hd", Etape);

        /* MESSAGE ecrit par NLWARNING: */
/*
        printf("L'estimateur choisi a l'etape %s est un estimateur simultane:\n", MessE);
        printf(          " on ignore cette etape et les eventuelles suivantes\n"); 
        printf("Les estimateurs impliquant l'estimation simultanee des Theta \n");
        printf(         " et des Beta ne sont possibles que s'il n'y a qu'une etape\n");
*/
        /* FIN MESSAGE */
        NLWARNING((WARES1,2,"NLVCtxNum",MessE,WMETHO));

         *NbEtapes = 1;
         } /* fin Etape >1 */
       else
         {
         APPEL(EstSimul(Vari,Etape, &(CtxNum[IndEtap].Estim))); 
         }
      } /* fin de estimation simultanee */

    if (( CtxNum[IndEtap].Estim ==  MVT) || 
	( CtxNum[IndEtap].Estim ==  MCPT) || 
	( CtxNum[IndEtap].Estim ==  MCOT) ||
	( CtxNum[IndEtap].Estim ==  MCMT) ||
	( CtxNum[IndEtap].Estim ==  QVT) ||
	( CtxNum[IndEtap].Estim ==  MCPVIT))
      {
      /* estimation des Theta seuls */
      Trouve = VRAI;
      if(Etape % 2  == 0)
        {
        /* Etape paire: estimation des beta seule possible */
        sprintf(MessE,"%hd", Etape);

        /* MESSAGE ecrit par NLWARNING: */
/*
        printf("L'estimateur choisi a l'etape paire %s est un estimateur des Theta seuls:\n", 
      MessE);
        printf( " on ignore cette etape et les eventuelles suivantes\n"); 
        printf("Les estimateurs impliquant l'estimation des Theta seuls\n");
        printf( " ne sont possibles qu'a une etape impaire\n");
*/
        /* FIN MESSAGE */
        NLWARNING((WARES2,2,"NLVCtxNum",MessE,WMETHO));

        *NbEtapes = Etape - 1;
        } /* fin Etape paire */
      else
        {
        if ((*NbEtapes == 1) && 
         ( (Vari == VB) || (Vari == VSB) || (Vari == VSTB) || (Vari == VTB)))
          {
          /* MESSAGE ecrit par NLWARNING: */
/*
          printf("L'estimateur choisi est un estimateur des Theta seuls:\n");
          printf(         " alors que la variance depend de Beta (Modele.Vari)\n");
*/
          /* FIN MESSAGE */
          NLWARNING((WARES3,1,"NLVCtxNum",WMETHO));
          *NbEtapes = Etape - 1;
          } /* fin du if ((*NbEtapes == 1) && (Vari  ) */
        else
          {
          APPEL(EstTheta(Vari, Etape,  NbRepet, &(CtxNum[IndEtap].Estim)));
          /* Code renvoye toujours OK */
          }
        } /* fin du else Etape paire */

      } /* fin des Theta seuls */


    if (( CtxNum[IndEtap].Estim ==  MCOB) || 
	( CtxNum[IndEtap].Estim ==  MCMB) ||
	( CtxNum[IndEtap].Estim == QVB))
      {
      /* estimation des Beta seuls */
      Trouve = VRAI;
      if(Etape % 2  == 1)
        {
        /* Etape impaire: estimation des theta seule possible */
        sprintf(MessE,"%hd", Etape);

        /* MESSAGE ecrit par NLWARNING: */
/*
        printf("L'estimateur choisi a l'etape impaire %s est un estimateur des Beta seuls:\n", MessE);
        printf(          " on ignore cette etape et les eventuelles suivantes\n"); 
        printf("Les estimateurs impliquant l'estimation des Beta seuls\n");
        printf(         " ne sont possibles qu'a une etape paire\n");
*/
        /* FIN MESSAGE */
        NLWARNING((WARES4,2,"NLVCtxNum",MessE,WMETHO));

        *NbEtapes = Etape - 1;
        } /* fin Etape impaire */
      else
        {
        if ( (Vari  == CTE) || (Vari == SP) || (Vari == VST) || (Vari == VI)) 
          /* Vari n'inclut pas Beta */
          {
          sprintf(MessE,"%hd", Etape);

          /* MESSAGE ecrit par NLWARNING: */
/*
          printf("L'estimateur choisi a l'etape %s est un estimateur des Beta seuls:\n", MessE);
          printf(         " alors que la variance ne depend pas de Beta (Modele.Vari)\n");
          printf(          " on ignore cette etape et les eventuelles suivantes\n"); 
          printf("Les estimateurs impliquant l'estimation des Beta seuls\n");
          printf(         " ne sont possibles que si Vari (de Modele) inclut des Beta\n");
*/
          /* FIN MESSAGE */
          NLWARNING((WARES6,2,"NLVCtxNum",MessE,WMETHO));

          *NbEtapes = Etape - 1;
          } /* fin du Vari == */
        } /* fin du else etape impaire */
     }  /* fin de Estim des Betas seuls */


    if ( CtxNum[IndEtap].Estim ==  LEMIEN)
      {
      /* estimateur defini par l'utilisateur */
      Trouve = VRAI;
      APPEL( EstMien(Etape,&(CtxNum[IndEtap]), Code));
      if(*Code == OK)
        {
        /* TypeCritStat est fourni par l'utilisateur: on le verifie */
        APPEL(VerifMien(Etape,CtxNum[IndEtap].TypeCritStat, Vari, NbRepet, Code));
        }  
      if(*Code != OK)
        {
        sprintf(MessE,"%hd", Etape);

        /* MESSAGE ecrit par NLWARNING: */
/*
        printf("L'estimateur choisi a l'etape %s est LEMIEN, \n", MessE);
        printf(         " mais les autres valeurs du contexte numerique etant incorrectes\n");
        printf(          " on ignore cette etape et les eventuelles suivantes\n"); 
*/
        /* FIN MESSAGE */
        NLWARNING((WARES7,2,"NLVCtxNum",MessE,WMETHO));

        *NbEtapes = Etape - 1;
        *Code = OK; /* on remet a OK, car ce n'est pas fatal: on ignore l'etape */
        }
      } /* fin de Estim == LEMIEN */

    if(Trouve == FAUX)
      {
      /* On est passe par aucun des tests ci-dessus: estimateur errone */
      sprintf(MessE,"%hd", Etape);

      /* MESSAGE ecrit par NLWARNING: */
/*
      printf("L'estimateur choisi a l'etape %s n'est pas un estimateur reconnu:\n", MessE);
      printf(          " on ignore cette etape et les eventuelles suivantes\n"); 
*/
      /* FIN MESSAGE */
      NLWARNING((WARES8,2,"NLVCtxNum",MessE,WMETHO));

      *NbEtapes = Etape - 1;
      } /* fin du Trouve=faux */

  if(*NbEtapes == 0)
    {
    /* MESSAGE ecrit par NLWARNING: */
/*
     printf("La correction effectuee sur NbEtapes provoque un nombre d'etapes nul\n");
*/
     /* FIN MESSAGE */
     NLWARNING((WARES5,1,"NLVCtxNum",WMETHO));

    *Code = WARES5;
    return(OK); /* on sort: pas la peine de faire la suite */
    } /* fin NbEtapes == 0 */

  if(Etape <= *NbEtapes)
    {
  /* aucune erreur decelee:
     affecter des valeurs aux autres composants du contexte */

    if(CtxNum[IndEtap].Estim != LEMIEN)
      {
      APPEL( AffectNum(Etape, Vari, TypeMu,
                      &(CtxNum[IndEtap])));
      }

    /* determiner TypeCritStat  selon la valeur de Estim */
    switch(CtxNum[IndEtap].Estim)
      {
      case MVTB:
      case MVT:
        CtxNum[IndEtap].TypeCritStat = CLV;
        break;
      case MCMTB:
      case QVTB:
      case MCMT:
      case QVT:
      case MCMB:
      case QVB:
        CtxNum[IndEtap].TypeCritStat = CARRET;
        break;
      case MCOT:
        CtxNum[IndEtap].TypeCritStat = CSCRNP;
        break;
      case MCPT:
        CtxNum[IndEtap].TypeCritStat = CSCRPV;
        break;
      case MCPVIT:
        CtxNum[IndEtap].TypeCritStat = CSCRPS2;
        break;
      case MCOB:
        CtxNum[IndEtap].TypeCritStat = CSCR2NP;
        break;
      case LEMIEN:
        /* TypeCritStat est fourni par l'utilisateur */
        break;
      } /* fin du switch */

    /* cas particulier quand TypeSigma=VARRESID:
      le critere de minimisation est egal a Sigma si Estim=MCOT et Vari=CTE
      et aussi, si  Estim=MCPT */
    if (TypeSigma == VARRESID)
      {
      if (((CtxNum[IndEtap].Estim == MCOT) && (Vari == CTE)) ||
         (CtxNum[IndEtap].Estim == MCPT))
        {
        CtxNum[IndEtap].TypeCritStat = CRITSIG;
        }
      }

    /* determiner PCDirec selon la valeur de Algo et Symm */
    if ((Algo == GN) && (CtxNum[IndEtap].Symm == SYMBLOC))
      {
      CtxNum[IndEtap].PCDirec = CDirGNBloc;
      }
    if ((Algo == GN) && (CtxNum[IndEtap].Symm != SYMBLOC))
      {
      CtxNum[IndEtap].PCDirec = CDirGN;
      }
    if ((Algo == GM) && (CtxNum[IndEtap].Symm == SYMBLOC))
      {
      CtxNum[IndEtap].PCDirec = CDirGMBloc;
      }
    if ((Algo == GM) && (CtxNum[IndEtap].Symm != SYMBLOC))
      {
      CtxNum[IndEtap].PCDirec = CDirGM;
      }

    } /* fin de aucune erreur decelee */

    /* incrementation des compteurs de boucle */
    Etape = Etape + 1;
    IndEtap = IndEtap + 1;

    } /* fin du while */


  /* retour */
  return(OK);
  }
