/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : ProgIter                  |
| Date                 : 1991                      |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : A. Bouvier                |
| Role                 : les programmes de calcul  |
|  de PussIter et les utilitaires necessaires a    |
|  ceux-ci                                         |
| Reference conception :                           |
| Lecteur              : G. Husser                 |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/


/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <errno.h>

#include "nlchoix.h"
#include "nlcodes.h"
#include "nltypes.h"
#include "nlglobal.h"
#include "nlmacros.h"
#include "errcodes.h"

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/
/* fonctions des autres modules */
TShortInt CopyMat(), GerMessage(), InvMat(),
MultVectMat(), MVect(), MultMat2(),
MultMat1(), MultMatVal();

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/



/*--------------- Identification fonction ----------
| Nom de la fonction    : ColleMat                 |
| Role                  : inserer une sous-matrice |
|  dans une matrice                                |
| Parametres d'entree   :                          |
|  LigDeb, ColDeb: indices de ligne et de colonne  |
|    du debut de la sous-matrice dans la matrice   |
|  NbLig, NbCol: nombre de lignes et de colonnes   |
|    de la sous-matrice                            | 
|  MatIn: la sous-matrice                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  MatOut: la matrice (supposee creee avant        | 
|                           l'appel)               |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  InvSb                    |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/


TShortInt ColleMat(LigDeb, ColDeb,  NbLig, NbCol, MatIn, MatOut)

/* arguments d'entree */
TShortInt LigDeb, ColDeb, NbLig, NbCol;
TMat *MatIn;

/* arguments de sortie*/
TMat *MatOut;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i,j,i1,j1;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble **matin, **matout;

/* Ecriture de la trace */
ECRTRACE("ColleMat");

matin = MatIn->donnees;
matout = MatOut->donnees;

i = LigDeb;
for(i1 = 0; i1 < NbLig; i1++)
  {
  j = ColDeb;
  for(j1 = 0; j1 < NbCol; j1++)
    {
    matout[i][j] =  matin[i1][j1];
    j = j + 1;
    }
  i = i + 1;
  }
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CPasOpt                  |
| Role                  : calculer le pas optimal  |
| Parametres d'entree   :                          |
| Algo= type de l'algorithme (GM ou GN)            |
| OmegaPas: valeur de correction de Omega          |
| LambdaC2: valeur de correction de Lambda         |
| Crit: tableau des 3 valeurs du critere d'arret   |
| Parametres d'e/s      :                          |
|  NbDeb : nombre de fois consecutives ou on repart|
|     du debut d'un intervalle                     |
|  Lambda: modifie si Algo=GM                      |
| Parametres de sortie  :                          |
|  Omega: valeur du pas optimal                    |
|  Fini: quand les criteres sont consideres comme  |
|        egaux                                     |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CPasOpt(Algo, OmegaPas,LambdaC1, LambdaC2, Crit, 
                  NbDeb, Lambda, 
                  Omega, CritEgaux)

/* arguments d'entree */
TShortInt Algo;
TDouble OmegaPas, LambdaC1, LambdaC2;
TDouble *Crit;

/* argument d'e/s */
TLongInt *NbDeb;
TDouble *Lambda;

/* arguments de sortie*/
TDouble *Omega;
TLogic *CritEgaux;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TDouble InfCrit;
TShortInt IInf;

/* Ecriture de la trace */
ECRTRACE("CPasOpt");


if (((float)Crit[0]== (float)Crit[1]) || ((float)Crit[0]== (float)Crit[2]) || ((float)Crit[2]== (float)Crit[1]))
  *CritEgaux=VRAI;
else
  *CritEgaux=FAUX;

/* Si les criteres sont egaux en 2 au moins des points
il faut faire comme si le mini etait au debut de l'intervalle ,
i.e il faut initialiser a InfCrit =0 */

/* recherche de l'indice dans Crit correspondant au minimum des valeurs*/
InfCrit = Crit[0];
IInf = 0;
if (Crit[1] < InfCrit)
  {
  IInf = 1;
  InfCrit = Crit[1];
  }
if (Crit[2] < InfCrit)
  {
  IInf = 2;
  }



/* on met a jour le nombre de fois ou on repart consecutivement du
debut d'un intervalle */
if (IInf == 0)
  {
  *NbDeb = *NbDeb + 1;
  }
else
  {
  *NbDeb = 0;
  *Lambda = LambdaC1 * *Lambda;
  }


switch(IInf)
  {
  case 0:
    *Omega = OmegaPas;
    if (Algo == GM)
      {
      *Lambda = LambdaC2 * *Lambda;
      }
    break;

  case 1:
    *Omega = (TDouble)1;
    break;

  case 2:
    *Omega = (-(TDouble)4 * Crit[2]) + Crit[1]
             + ( (TDouble)3 * Crit[0]);
    *Omega = *Omega / ( (TDouble)4 * (Crit[1] + Crit[0] - 
                     ( (TDouble)2 * Crit[2])));
  } /* fin du switch */

/* retour */
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : CValR                    |
| Role                  : calcul de ValR:          |
| ValR=ValB * (ValZ- ValEta)                       |
| Parametres d'entree   :                          |
|  ValZ: vecteur NbZ*NbObs                         |
|  ValEta: vecteur NbZ*NbObs                       |
|  ValB: matrice NbZ*NbObs, NbAct                  |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValR (alloue  avant l'appel): vecteur NbAct     |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CValR( ValZ, ValEta, ValB, ValR)

/* arguments d'entree */
TVect *ValZ, *ValEta;
TMat *ValB;

/* arguments de sortie*/
TVect *ValR;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, j, valbnl, valbnc;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *valr, *valz, *valeta;
TDouble **valb;

/* Ecriture de la trace */
ECRTRACE("CValR");

valbnl = ValB->nblig;
valbnc = ValB->nbcol;
valr = ValR->donnees;
valz = ValZ->donnees;
valeta = ValEta->donnees;
valb = ValB->donnees;

for (j = 0; j < valbnc; j++)
  {
  valr[j] = (TDouble)ZERO;
  for ( i = 0; i < valbnl; i++)
    {
    valr[j] = valr[j] + (valb[i][j] * (valz[i] - valeta[i]));
    } /* fin boucle sur i */
  }  /* fin boucle sur j */

/* retour */
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CValW                    |
| Role                  : calcul de ValW:          |
| ValW=  1/NbObsT * transposee(ValB) * ValD        |
| Parametres d'entree   :                          |
|  NbObsT: nombre total de repetitions             |
|  ValB: matrice NbZb*NbObs, NbActb                |
|  ValD: matrice NbZd*NbObs, NbActd                |
|  Les dimensions des matrices en entree peuvent ne|
|  pas correspondre: NbZb # NbZd et NbActb # NbActd|
|  on considere uniquement les elements sur les    |
|  dimensions qui correspondent                    |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValW (allouee avant l'appel)                    |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Fonctions appelantes :  PussIter, CCovNu2        |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CValW( NbObsT, ValB, ValD, ValW)

/* arguments d'entree */
TLongInt NbObsT;
TMat *ValB, *ValD;

/* arguments de sortie*/
TMat *ValW;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, j, k, iborne, jborne, kborne;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble **valw, **valb, **vald;

/* Ecriture de la trace */
ECRTRACE("CValW");


iborne = MIN(ValW->nblig, ValB->nbcol);
jborne = MIN(ValB->nblig, ValD->nblig);
kborne = MIN(ValW->nbcol, ValD->nbcol);

valw = ValW->donnees;
valb = ValB->donnees;
vald = ValD->donnees;


for (k = 0; k < kborne; k++)
  {
  for (i = 0; i < iborne; i++)
    {
    valw[i][k] = (TDouble)ZERO;
    for ( j = 0; j < jborne; j++)
      {
      valw[i][k] = valw[i][k] + (valb[j][i] * vald[j][k]);
      } /* fin boucle sur j */
    valw[i][k] = (valw[i][k] / (TDouble)NbObsT);
    }  /* fin boucle sur i */
  }  /* fin boucle sur k */

/* retour */
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : ExtrMat                  |
| Role                  : Extraire une sous-matrice|
|  dans une matrice                                |
| Parametres d'entree   :                          |
|  LigDeb, ColDeb: indices de ligne et de colonne  |
|    du debut de la sous-matrice                   |
|  NbLig, NbCol: nombre de lignes et de colonnes   |
|    de la sous-matrice                            |
|  MatIn: matrice d'origine                        |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  MatOut: la sous-matrice (supposee creee avant   |
|                           l'appel)               |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  CDirGM, InvSb            |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt ExtrMat(LigDeb, ColDeb,  NbLig, NbCol, MatIn, MatOut)

/* arguments d'entree */
TShortInt LigDeb, ColDeb, NbLig, NbCol;
TMat *MatIn;

/* arguments de sortie*/
TMat *MatOut;

/*--------------- Fin identification fonction ----*/

{

/* locals */
TShortInt i,j,i1,j1;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble **matin, **matout;

/* Ecriture de la trace */
ECRTRACE("ExtrMat");

i = LigDeb;
matin = MatIn->donnees;
matout = MatOut->donnees;

for(i1 = 0; i1 < NbLig; i1++)
  {
  j = ColDeb;
  for(j1 = 0; j1 < NbCol; j1++)
    {
    matout[i1][j1] =  matin[i][j];
    j = j + 1;
    }
  i = i + 1;
  }
return(OK);
}







/*--------------- Identification fonction ----------
| Nom de la fonction    : InvSb                    |
| Role                  : Inverser la matrice ValW |
|  quand elle est symetrique par bloc              |
| Parametres d'entree   :                          |
|  ValW: matrice ValW a inverser, de dimension     |
|       (NbTheta+NbBeta)*(NbTheta+NbBeta)          |
|       (nombres de parametres actifs)             |
| Parametres d'e/s      :                          |
|  Trav:   vecteur de travail de dimension         |
|     au moins egal a max(NbTheta,NbBeta)          |
| Parametres de sortie  :                          |
|  BlocP:  Inverse du bloc superieur gauche de ValW|
|          matrice de dimension  NbTheta*NbTheta   |
|  BlocQ:  Inverse du bloc inferieur gauche de ValW|
|          matrice de dimension NbBeta*NbTheta     |
|  Bloc0:  Inverse du bloc superieur droit  de ValW|
|          matrice de dimension  NbTheta*NbBeta    |
|  BlocR:  Inverse du bloc inferieur droit  de ValW|
|          matrice de dimension NbBeta*NbBeta      |
|  ValWInv: matrice inversee                       |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         ColleMat,                |
|                         ExtrMat,                 |
|                         InvMat                   |
| Fonctions appelantes :  CArret                   |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt InvSb (ValW, 
                  Trav, BlocP, BlocQ, Bloc0, BlocR, ValWInv)

/* arguments d'entree */
TMat *ValW;
/* arguments d'entree-sortie */
TVect *Trav;

/* arguments de sortie */
TMat *BlocP, *BlocQ, *Bloc0, *BlocR;
TMat *ValWInv;
/*--------------- Fin identification fonction ----*/


{
/* locals */
TShortInt NbTheta, NbBeta, Code;

/* Ecriture de la trace */
ECRTRACE("InvSb");


/* Determination des dimensions */
NbTheta =  BlocP->nblig;
NbBeta =  BlocQ->nblig;

/* Extraction des blocs qui sont symetriques */
APPEL(ExtrMat((TShortInt)0,(TShortInt)0,NbTheta,NbTheta, ValW, BlocP));
APPEL(ExtrMat(NbTheta,(TShortInt)0,NbBeta,NbTheta, ValW, BlocQ));
APPEL(ExtrMat((TShortInt)0,NbTheta,NbTheta,NbBeta,ValW, Bloc0));
APPEL(ExtrMat(NbTheta,NbTheta,NbBeta, NbBeta, ValW ,BlocR));


/* Inversion de BlocP et BlocR: resultat dans eux-memes */
Code = InvMat(BlocP);
if (Code != OK)
  {
  /* ACTIONS EFFECTUEES PAR LA MACRO NLERREUR */
/*
  fprintf(stderr,"InvSb:  matrice non inversible \n");
  return(ERRCALC);
*/
  NLERREUR((ERRCALC,1,"InvSb",ERR));
  }

Code = InvMat(BlocR);
if (Code != OK)
  {
  /* ACTIONS EFFECTUEES PAR LA MACRO NLERREUR */
/*
  fprintf(stderr,"InvSb: la matrice  est non inversible \n");
  return(ERRCALC);
*/
  NLERREUR((ERRCALC,1,"InvSb",ERR));
  }

/* Calcul de -BlocR-1* BlocQ * BlocP-1 dans BlocQ: */

/* 1)-calcul de BlocR-1* BlocQ dans BlocQ */
APPEL(MultMat2(BlocR, BlocQ, Trav));

/*2)- calcul de BlocQ * BlocP-1 dans BlocQ */
APPEL(MultMat1(BlocQ, BlocP, Trav));

/* 3)- multiplication par -1 */
APPEL(MultMatVal(BlocQ,(TDouble)(-1), BlocQ));
 

/* Constitution de ValWInv */
APPEL(ColleMat((TShortInt)0,(TShortInt)0,NbTheta,NbTheta,  BlocP, ValWInv));
APPEL(ColleMat(NbTheta,(TShortInt)0,NbBeta,NbTheta, BlocQ, ValWInv));
APPEL(ColleMat((TShortInt)0,NbTheta,NbTheta,NbBeta, Bloc0,ValWInv));
APPEL(ColleMat(NbTheta,NbTheta,NbBeta, NbBeta, BlocR, ValWInv));

/* retour */
return(OK);

}

