/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : deriv.c                   |
| Date                 : 13 dec. 1991              |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : O. Nicole                 |
| Role                 : deriver un ensemble d'expressions
| Reference conception : DCP module 17             |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/

#include "nltypes.h"
#include "nlcodes.h"
#include "dftypes.h"
#include "dfcodes.h"

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/

extern void *malloc(size_t size);
extern void *realloc(void *ptr, size_t size);

extern TArbre * constUn(TArbre * SsArbG, TShortInt fonction, TDouble CstVal);
extern TArbre * constBin(TArbre * SsArbG, TArbre * SsArbD, TShortInt fonction);
extern TShortInt erreAnal(TChaine message, TChaine param);
extern TLogic decorArb(TArbDe * arbre, TVectShort liste, TShortInt * typeIdent);
/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/

/*--------------- Identification fonction ----------
| Nom de la fonction    : simplArb                 |
| Role                  : simplifie une expression |
| Parametres d'entree   : arbre                    |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : erreAnal                  |
| Fonctions appelantes : deriv                     |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TArbre * simplArb(arbre)
     TArbre * arbre;

/*--------------- Fin identification fonction ----*/

{
  TArbre * gauche, * droit;

  if (arbre==NULL)
    {
      return(NULL);
    }
  gauche=(TArbre *)malloc(sizeof(TArbre));
  if (gauche==NULL)
    {
      erreAnal(EMEM, NULL);
      return(NULL);
    }
  droit=(TArbre *)malloc(sizeof(TArbre));
  if (droit==NULL)
    {
      erreAnal(EMEM, NULL);
      return(NULL);
    }

  /* simplification des sous-arbres */
  gauche=simplArb(arbre->filsG);
  droit=simplArb(arbre->filsD);

  /* simplification du noeud courant */
  switch(arbre->fonction)
    {
    case PLUS:
      if (gauche->fonction==CONSTANTE &&
	  gauche->valeur==(double)0)
	{
	  /* 0+F -> F */
	  return(droit);
	}
      else if (droit->fonction==CONSTANTE &&
	       droit->valeur==(double)0)
	{
	  /* F+0 -> F */
	  return(gauche);
	}
      break;
    case MOINS:
      if (gauche->fonction==CONSTANTE &&
	  gauche->valeur==(double)0)
	{
	  /* 0-F -> -1 * F */
	  return(constBin(constUn(NULL, CONSTANTE, (double)-1),
			 droit,
			 MULT));
	}
      else if (droit->fonction==CONSTANTE &&
	       droit->valeur==(double)0)
	{
	  /* F-0 -> F */
	  return(gauche);
	}
      break;
    case MULT:
      if ((gauche->fonction==CONSTANTE &&
	  gauche->valeur==(double)0) ||
	  (droit->fonction==CONSTANTE &&
	   droit->valeur==(double)0))
	{
	  /* 0*F = F * 0 -> 0 */
	  return(constUn(NULL, CONSTANTE, (double)0));
	}
      else if (gauche->fonction==CONSTANTE &&
	       gauche->valeur==(double)1)
	{
	  /* 1*F -> F */
	  return(droit);
	}
      else if (droit->fonction==CONSTANTE &&
	       droit->valeur==(double)1)
	{
	  /* F*1 -> F */
	  return(gauche);
	}
      break;
    case DIV:
      if (gauche->fonction==CONSTANTE &&
	  gauche->valeur==(double)0)
	{
	  /* 0/F -> 0 */
	  return(constUn(NULL, CONSTANTE, (double)0));
	}
      else if (droit->fonction==CONSTANTE &&
	       droit->valeur==(double)1)
	{
	  /* F/1 -> F */
	  return(gauche);
	}
      break;
    case LOG:
    case LN:
      if (gauche->fonction==CONSTANTE &&
	  gauche->valeur==(double)1)
	{
	  /* log(1) -> 0 */
	  /* ln(1) -> 0 */
	  return(constUn(NULL, CONSTANTE, (double)0));
	}
      break;
    case SIN:
    case TG:
      if (gauche->fonction==CONSTANTE &&
	  gauche->valeur==(double)0)
	{
	  /* sin(0) -> 0 */
	  /* tg(0) -> 0 */
	  return(constUn(NULL, CONSTANTE, (double)0));
	}
      break;
    case COS:
    case EXP:
      if (gauche->fonction==CONSTANTE &&
	  gauche->valeur==(double)0)
	{
	  /* cos(0) -> 1 */
	  /* exp(0) -> 1 */
	  return(constUn(NULL, CONSTANTE, (double)1));
	}
      break;
    case PUISSANCE:
      if (droit->fonction==CONSTANTE)
	{
	  if (droit->valeur==(double)0)
	    {
	      /* F^0 -> 1 */
	      return(constUn(NULL, CONSTANTE, (double)1));
	    }
	  else if (droit->valeur==(double)1)
	    {
	      /* F^1 -> F */
	      return(gauche);
	    }
	}
      break;
    default:
      break;
    }

  /* dans les autres cas, ou quand les conditions de simplification ne sont
     pas satisfaites, on retourne l'arbre initial */
  arbre->filsG=gauche;
  arbre->filsD=droit;
  return(arbre);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : dedecArb                 |
| Role                  : copie un arbre decore dans un arbre simple
| Parametres d'entree   : arbre                    |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : erreAnal                  |
| Fonctions appelantes : derivArb                  |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TArbre * dedecArb(arbre)
     TArbDe * arbre;

/*--------------- Fin identification fonction ----*/

{
  TArbre * Noeud;
  
  if (arbre==NULL)
    {
      return(NULL);
    }
  Noeud=(TArbre *)malloc(sizeof(TArbre));
  if (Noeud==NULL)
    {
      erreAnal(EMEM, NULL);
      return(NULL);
    }
  Noeud->fonction=arbre->fonction;
  Noeud->valeur=arbre->valeur;
  Noeud->filsG=dedecArb(arbre->filsG);
  Noeud->filsD=dedecArb(arbre->filsD);
  return(Noeud);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : derivArb                 |
| Role                  : calcule la derivee d'un arbre
| Parametres d'entree   : arbre variable type indic|
| Parametres de sortie  : error                    |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 17            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : erreAnal constUn constBin |
| Fonctions appelantes : deriv                     |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TArbre * derivArb(arbre, variable, error, type, indic)
     TArbDe * arbre;
     TShortInt * variable;
     TShortInt * error;
     TVectShort type;
     TLogic indic;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i;
  TLogic test;
  TArbre *gauche, *droit, *der;

  if (arbre==NULL || *error==OUI)
    {
      return(NULL);
    }

  /* teste si la variable de derivation est dans la liste du noeud courant */
  i=0;
  test=FAUX;
  while (i<arbre->Liste.nbele && test==FAUX)
    {
      if (arbre->Liste.donnees[i]==*variable)
	{
	  test=VRAI;
	}
      i++;
    }

  if (test==FAUX)
    {
      /* ca ne sert a rien de deriver cette branche */
      gauche=constUn(NULL, CONSTANTE, (double)0);
      if (gauche==NULL)
	{
	  *error=OUI;
	  return(NULL);
	}
      return(gauche);
    }

  /* on calcule les derivees des sous arbres gauches et droits */
  if (arbre->fonction>NOT && arbre->fonction<VALINT)
    {
      /* fonction unaire */
      gauche=derivArb(arbre->filsG, variable, error, type, indic);
      if (gauche==NULL || *error==OUI)
	{
	  return(NULL);
	}
      /* fonction binaire */
      if (arbre->fonction>=PLUS)
	{
	  droit=derivArb(arbre->filsD, variable, error, type, indic);
	  if (droit==NULL || *error==OUI)
	    {
	      return(NULL);
	    }
	}
    }

  /* on construit la derivee */
  switch(arbre->fonction)
    {
    case PLUS:
      der=constBin(gauche, droit, PLUS);
      break;
    case MOINS:
      der=constBin(gauche, droit, MOINS);
      break;
    case MULT:
      der=constBin(constBin(dedecArb(arbre->filsG), droit, MULT),
		   constBin(gauche, dedecArb(arbre->filsD), MULT), PLUS);
      break;
    case DIV:
      der=constBin(constBin(constBin(gauche, dedecArb(arbre->filsD), MULT),
			    constBin(dedecArb(arbre->filsG), droit, MULT),
			    MOINS),
		   constBin(dedecArb(arbre->filsD),
			    constUn(NULL, CONSTANTE, (double)2),
			    PUISSANCE),
		   DIV);
      break;
    case NOT:
    case AND:
    case OR:
    case EQ:
    case NEQ:
    case LE:
    case LEQ:
    case GE:
    case GEQ:
    case THENELSE:
      /* on ne devrait jamais passer par cette branche */
      der=NULL;
      break;
    case IF:
      gauche=derivArb(arbre->filsD->filsG, variable, error, type, indic);
      if (gauche==NULL || *error==OUI)
	{
	  return(NULL);
	}
      droit=derivArb(arbre->filsD->filsD, variable, error, type, indic);
      if (droit==NULL || *error==OUI)
	{
	  return(NULL);
	}
      der=constBin(dedecArb(arbre->filsG),
		   constBin(gauche,
			    droit,
			    THENELSE),
		   IF);
      break;
    case VALINT:
      if (indic==NON)
	{
	  /* la derivee de F[T] a un sens */
	  der=constBin(constUn(dedecArb(arbre->filsG),
			       arbre->filsG->fonction-type.nbele, 0),
		       constUn(dedecArb(arbre->filsD),
			       arbre->filsD->fonction, 0), VALINT);
	}
      else
	{
	  /* la derivee de F[T] n'a pas de sens, on utilise 0 */
	  der=constUn(NULL, CONSTANTE, (double)0);
	}
      break;
    case PUISSANCE:
      der=constBin(constBin(dedecArb(arbre->filsD),
			    constBin(constBin(dedecArb(arbre->filsG),
					      constBin(dedecArb(arbre->filsD),
						      constUn(NULL, CONSTANTE, (double)1),
						      MOINS),
					      PUISSANCE),
				     gauche,
				     MULT),
			    MULT),
		   constBin(dedecArb(arbre),
			    constBin(constUn(dedecArb(arbre->filsG), LN, 0),
				     droit,
				     MULT),
			    MULT),
		   PLUS);
      break;
    case COS:
      der=constBin(constUn(NULL, CONSTANTE, (double)-1),
		   constBin(gauche,
			    constUn(dedecArb(arbre->filsG), SIN, 0),
			    MULT),
		   MULT);
      break;
    case SIN:
      der=constBin(gauche,
		   constUn(dedecArb(arbre->filsG), COS, 0),
		   MULT);
      break;
    case EXP:
      der=constBin(gauche,
		   constUn(dedecArb(arbre->filsG), EXP, 0),
		   MULT);
      break;
    case TG:
      der=constBin(gauche,
		   constBin(constUn(dedecArb(arbre->filsG), COS, 0),
			    constUn(NULL, CONSTANTE, (double)2),
			    PUISSANCE),
		   DIV);
      break;
    case LN:
      der=constBin(gauche,
		   dedecArb(arbre->filsG),
		   DIV);
      break;
    case LOG:
      der=constBin(gauche,
		   constBin(dedecArb(arbre->filsG),
			    constUn(constUn(NULL, CONSTANTE, (double)10),
				    LN, 0),
			    MULT),
		   DIV);
      break;
    case CONSTANTE:
      der=dedecArb(arbre);
      break;
    default:
      if (type.donnees[-arbre->fonction]==IDEF)
	{
	  if (indic==NON)
	    {
	      /* la derivee de f a un sens */
	      der=constUn(NULL, arbre->fonction - type.nbele, 0);
	    }
	  else
	    {
	      /* la derivee de f n'a pas de sens, on utilise 0 */
	      der=constUn(NULL, CONSTANTE, (double)0);
	    }
	}
      else if(type.donnees[-arbre->fonction]==IDEAUX)
	{
	  der=constUn(NULL, arbre->fonction - type.nbele, 0);
	}
      else if (type.donnees[-arbre->fonction]==IDELESF)
	{
	  der=constUn(NULL, arbre->fonction - type.nbele, 0);
	}
      else if (-arbre->fonction==*variable)
	{
	  der=constUn(NULL, CONSTANTE, (double)1);
	}
      else
	{
	  der=constUn(NULL, CONSTANTE, (double)0);
	}
      break;
    }
  if (der==NULL)
    {
      *error=OUI;
      return(NULL);
    }
  return(der);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : deriv                    |
| Role                  : deriver un ensemble d'expressions
| Parametres d'entree   : ModStandard borne        |
| Parametres de sortie  : Deriv                    |
| Parametres d'e./s.    : DerivAux                 |
| Reference conception  : DCP module 17            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : erreAnal decorArb derivArb|
| Fonctions appelantes : prepDer                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt deriv(ModStandard, Deriv, DerivAux, borne)
     TModStandard ModStandard;
     TDeriv * Deriv;
     TDerivAux * DerivAux;
     TShortInt borne;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i, j, nbaux, nbderaux, nbparam, first, error;
  TVectShort listp;
  TLogic indic;

  nbaux=ModStandard.Auxiliaire.nbele;
  nbparam=ModStandard.IdentDeriv.nbele;
  nbderaux=nbaux*nbparam;
  indic=NON;
  if (DerivAux->nbele<nbderaux)
    {
      first=DerivAux->nbele/nbaux;
      /* les expressions auxiliaires sont deja derivees par rapport aux
       derivateurs 0 ,..., first-1 */
      listp.donnees=(TShortInt *)malloc((nbparam-first)*sizeof(TShortInt));
      if (listp.donnees==NULL)
	{
	  erreAnal(EMEM, NULL);
	  return(FAUX);
	}
      listp.nbele=nbparam-first;

      /* il suffit de decorer/deriver les arbres par rapport aux variables
	 non encore derivees */
      for (i=first; i<nbparam; i++)
	{
	  listp.donnees[i-first]=ModStandard.IdentDeriv.donnees[i];
	}

      /* on decore les aux */
      for (i=0; i<nbaux; i++)
	{
	  if (decorArb(ModStandard.Auxiliaire.UnAux[i].arbre, listp,
		       ModStandard.TypeIdent.donnees)==FAUX)
	    {
	      return(FAUX);
	    }
	  /* on cree DerivAux */
	  if (DerivAux->nbele==0)
	    {
	      DerivAux->arbre=(TArbre **)malloc(nbderaux*sizeof(TArbre *));
	      if (DerivAux->arbre==NULL)
		{
		  erreAnal(EMEM, NULL);
		  return(FAUX);
		}
	    }
	  else
	    {
	      DerivAux->arbre=(TArbre **)realloc(DerivAux->arbre,
						nbderaux*sizeof(TArbre *));
	      if (DerivAux->arbre==NULL)
		{
		  erreAnal(EMEM, NULL);
		  return(FAUX);
		}
	    }
	  DerivAux->nbele=nbderaux;

	  /* on derive l'aux[i] par rapport au param[j] */
	  error=NON;
	  for (j=0; j<listp.nbele; j++)
	    {
	      DerivAux->arbre[(first+j)*nbaux+i]=
		simplArb(derivArb(ModStandard.Auxiliaire.UnAux[i].arbre,
				  &listp.donnees[j], &error,
				  ModStandard.TypeIdent, indic));
	      if (error==OUI)
		{
		  return(FAUX);
		}
	    }
	}
    }

  /* on derive les expressions principales */
  Deriv->nbele=nbparam*ModStandard.Expressions.nbele;
  Deriv->arbre=(TArbre **)malloc(Deriv->nbele*sizeof(TArbre *));
  if (Deriv->arbre==NULL)
    {
      erreAnal(EMEM, NULL);
      return(FAUX);
    }

  for (i=0; i<ModStandard.Expressions.nbele; i++)
    {
      /* on decore les expressions */
      if (decorArb(ModStandard.Expressions.arbre[i], ModStandard.IdentDeriv,
		   ModStandard.TypeIdent.donnees)==FAUX)
	{
	  return(FAUX);
	}

      /* on derive */
      error=NON;
      for (j=0; j<nbparam; j++)
	{
	  if (borne>=0 && j>=borne)
	    {
	      indic=OUI;
	    }
	  Deriv->arbre[j*ModStandard.Expressions.nbele+i]=
	    simplArb(derivArb(ModStandard.Expressions.arbre[i],
			      &ModStandard.IdentDeriv.donnees[j], &error,
			      ModStandard.TypeIdent, indic));
	  if (error==OUI)
	    {
	      return(FAUX);
	    }
	}
    }
  return(VRAI);
}
