
/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : evalArbInv.c                 |
| Date                 : 16 decembre 1992          |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : O. Nicole                 |
| Role                 : evaluation formelle du modele par parcours des
                         arbres                    |
| Reference conception : DCP modules 24-29         |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/
/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <float.h>
#include <math.h>
#include <stdio.h>

#include "nltypes.h"
#include "nlcodes.h"
#include "dftypes.h"
#include "dfcodes.h"
#include "nlglobal.h"
#include "errcodes.h"
#include "nlmacros.h"
#include "nlchoix.h"
/* les arguments de NL: */
#include "nldcl.h"

/*----------------FONCTIONS EXTERNES ------------*/
TShortInt GerMessage( );

/*--------------- VARIABLES EXTERNES -------------*/

extern  TModSpecifInv GModSpecifInv;
extern TDeriveesInv DeriveesInv;

/*--------------- Identification fonction ----------
| Nom de la fonction    : imprErr                  |
| Role                  : sert d'appel a NLWARNING et retourne la valeur NaN
| Parametres d'entree   : fonction x y             |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : evalArbInv                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

static double imprErr(fonction, x, y)
     TShortInt fonction;
     TDouble x, y;

/*--------------- Fin identification fonction ----*/

{
  double r;
  char Mess1[30];
  char Mess2[30];
  
    switch(fonction)
      {
      case LOG:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "log10", Mess1, " ", WNUMER));
	break;
      case LN:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "log", Mess1, " ", WNUMER));
	break;
      case DIV:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "div", Mess1, " ", WNUMER));
	break;
      case EXP:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "exp", Mess1, " ", WNUMER));
	break;
      case PUISSANCE:
	sprintf(Mess1, "%g", x);
	sprintf(Mess2, "%g", y);
	NLWARNING((WARNUM, 3, "pow", Mess1, Mess2,  WNUMER));
	break;
      }

  r=mysignaling_nan(n);
  return(r);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : evalArbInv                  |
| Role                  : calcule formellement la valeur d'un arbre
| Parametres d'entree   : arbre                    |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 29            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : calcf_ calcv_ calcphi_ _
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TDouble evalArbInv(arbre)
     TArbre * arbre;

/*--------------- Fin identification fonction ----*/

{
  TDouble x, y;

  switch(arbre->fonction)
    {
    case CONSTANTE:
      return(arbre->valeur);
      break;
    case PLUS:
      return(evalArbInv(arbre->filsG) + evalArbInv(arbre->filsD));
      break;
    case MOINS:
      return(evalArbInv(arbre->filsG) - evalArbInv(arbre->filsD));
      break;
    case MULT:
      return(evalArbInv(arbre->filsG) * evalArbInv(arbre->filsD));
      break;
    case DIV:
      x=evalArbInv(arbre->filsD);
      return( (x==(TDouble)0) ?
	     imprErr(DIV, x, 0) :
	     evalArbInv(arbre->filsG) / x);
      break;
    case LOG:
      x=evalArbInv(arbre->filsG);
      return( (x<=(TDouble)0) ?
	     imprErr(LOG, x, 0) :
	     log(x)/log((double)10));
      break;
    case TG:
      return(tan(evalArbInv(arbre->filsG)));
      break;
    case LN:
      x=evalArbInv(arbre->filsG);
      return( (x<=(TDouble)0) ?
	     imprErr(LN, x, 0) :
	     log(x));
      break;
    case EXP:
      x=evalArbInv(arbre->filsG);
      return((x>(double)709) ?
	     imprErr(EXP, x, 0) :
	     ((x<(double)-745) ?
	       (double)0 :
		exp(x)));
      break;
    case SIN:
      return(sin(evalArbInv(arbre->filsG)));
      break;
    case COS:
      return(cos(evalArbInv(arbre->filsG)));
      break;
    case IF:
      if (arbre->filsD==NULL)
	{
	  return((double)0);
	}
      if (evalArbInv(arbre->filsG)!=0)
	{
	  return(evalArbInv(arbre->filsD->filsG));
	}
      else
	{
	  return(evalArbInv(arbre->filsD->filsD));
	}
      break;
    case OR:
      return(evalArbInv(arbre->filsG)+evalArbInv(arbre->filsD));
      break;
    case AND:
      return(evalArbInv(arbre->filsG)*evalArbInv(arbre->filsD));
      break;
    case EQ:
      return( (evalArbInv(arbre->filsG)==evalArbInv(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case NEQ:
      return( (evalArbInv(arbre->filsG)!=evalArbInv(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case LE:
      return( (evalArbInv(arbre->filsG)<evalArbInv(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case LEQ:
      return( (evalArbInv(arbre->filsG)<=evalArbInv(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case GE:
      return( (evalArbInv(arbre->filsG)>evalArbInv(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case GEQ:
      return( (evalArbInv(arbre->filsG)>=evalArbInv(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case NOT:
      return( (evalArbInv(arbre->filsG)==(TDouble)0) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case PUISSANCE:
      x=evalArbInv(arbre->filsG);
      y=evalArbInv(arbre->filsD);
      return(((x==0 && y<=0) || (x<0 && (int)y!=y)) ?
	     imprErr(PUISSANCE, x, y) :
	     pow(x, y));
      break;
    }
  /* AB */
return ((TDouble)0);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : calcinvf_                   |
| Role                  : calcule la reponse pour un modele simple
| Parametres d'entree   : nbt nbg nbl nbc t g x    |
| Parametres de sortie  : f df ie le               |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 24            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : evalArbInv                   |
| Fonctions appelantes :                           |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt calcinvf_(nbp, nbga, nbl, nbgv, p, ga, ord, gv, abs, dabsdp,
		    dabsdo, varord, le, ie)
      TShortInt nbp, nbga, nbl, nbgv;
      TDouble *p, *ga, *gv, *ord, *abs, **dabsdp, *dabsdo, *varord;
      TShortInt *le, *ie;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i, xt, j;

  ECRTRACE("calcinvf");
  /* recopie des parametres d'entree */
  for (i=0; i<nbp; i++)
    {
      GModSpecifInv.ParamEntre[i+1]->valeur=p[i];
    }
  xt=1+nbp;
  for (i=0; i<nbga; i++)
    {
      GModSpecifInv.ParamEntre[i+xt]->valeur=ga[i];
    }
  xt=xt+nbga;
  for (i=0; i<nbgv; i++)
    {
      GModSpecifInv.ParamEntre[i+xt]->valeur=gv[i];
    }
  /* boucle sur les nbl observations */
  for (i=0; i<nbl; i++)
    {
      /* recopie des parametres d'entree */
      GModSpecifInv.ParamEntre[0]->valeur=ord[i];
      
      /* evaluation de l'arbre de F */
      abs[i]=evalArbInv(GModSpecifInv.IdeAbs.arbre);
      if (finite(abs[i])!=1)
	{
	  *ie=i+1;
	  *le=F;
	  return(ERRMATH);
	}
      
      /* evaluation des arbres des dF */
      for (j=0; j<nbp; j++)
	{
	  dabsdp[i][j]=evalArbInv(DeriveesInv.DerivAbs.arbre[j]);
	  if (finite(dabsdp[i][j])!=1)
	    {
	      *ie=i+1;
	      *le=DFDT;
	      return(ERRMATH);
	    }
	}
      dabsdo[i]=evalArbInv(DeriveesInv.DerivAbs.arbre[nbp]);
      if (finite(dabsdo[i])!=1)
	{
	  *ie=i+1;
	  *le=DFDT;
	  return(ERRMATH);
	}
      if (GModSpecifInv.Vari==OUI)
	{
	  /* Calcul de VarOrd si existe */
	  varord[i]=evalArbInv(GModSpecifInv.IdeVarOrd.arbre);
	  if (finite(varord[i])!=1)
	    {
	      *ie=i+1;
	      *le=V;
	      return(ERRMATH);
	    }
	}
    }
  return(OK);
}
