/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <stdio.h>
#include <float.h>

#include "nltypes.h"
#include "nlcodes.h"
#include "dftypes.h"
#include "dfcodes.h"
#include "nlglobal.h"
#include "errcodes.h"
#include "nlchoix.h"
#include "nlmacros.h"

/* les arguments de NL: */
#include "nldcl.h"

/*--------------- VARIABLES EXTERNES -------------*/


extern  TModSpecifPsi GModSpecifPsi;
extern TDeriveesPsi DeriveesPsi;

/*----------------FONCTIONS EXTERNES ------------*/
TShortInt GerMessage( );


/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : evalArbFunc.c                 |
| Date                 : 16 decembre 1992          |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : O. Nicole                 |
| Role                 : evaluation formelle du modele par parcours des
                         arbres                    |
| Reference conception : DCP modules 24-29         |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/

/*--------------- Identification fonction ----------
| Nom de la fonction    : imprErr                  |
| Role                  : sert d'appel a NLWARNING et retourne la valeur NaN
| Parametres d'entree   : fonction x y             |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : evalArb                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

static double imprErr(fonction, x, y)
     TShortInt fonction;
     TDouble x, y;

/*--------------- Fin identification fonction ----*/

{
  double r;
  char Mess1[30];
  char Mess2[30];
  
    switch(fonction)
      {
      case LOG:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "log10", Mess1, " ", WNUMER));
	break;
      case LN:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "log", Mess1, " ", WNUMER));
	break;
      case DIV:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "div", Mess1, " ", WNUMER));
	break;
      case EXP:
	sprintf(Mess1, "%g", x);
	NLWARNING((WARNUM, 3, "exp", Mess1, " ", WNUMER));
	break;
      case PUISSANCE:
	sprintf(Mess1, "%g", x);
	sprintf(Mess2, "%g", y);
	NLWARNING((WARNUM, 3, "pow", Mess1, Mess2,  WNUMER));
	break;
      }

  r=mysignaling_nan(n);
  return(r);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : evalArbFunc                  |
| Role                  : calcule formellement la valeur d'un arbre
| Parametres d'entree   : arbre                    |
| Parametres de sortie  :                          |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 29            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : calcpsi_ 
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TDouble evalArbFunc(arbre)
     TArbre * arbre;

/*--------------- Fin identification fonction ----*/

{
  TDouble x, y;

  switch(arbre->fonction)
    {
    case CONSTANTE:
      return(arbre->valeur);
      break;
    case PLUS:
      return(evalArbFunc(arbre->filsG) + evalArbFunc(arbre->filsD));
      break;
    case MOINS:
      return(evalArbFunc(arbre->filsG) - evalArbFunc(arbre->filsD));
      break;
    case MULT:
      return(evalArbFunc(arbre->filsG) * evalArbFunc(arbre->filsD));
      break;
    case DIV:
      x=evalArbFunc(arbre->filsD);
      return( (x==(TDouble)0) ?
	     imprErr(DIV, x, 0) :
	     evalArbFunc(arbre->filsG) / x);
      break;
    case LOG:
      x=evalArbFunc(arbre->filsG);
      return( (x<=(TDouble)0) ?
	     imprErr(LOG, x, 0) :
	     log(x)/log((double)10));
      break;
    case TG:
      return(tan(evalArbFunc(arbre->filsG)));
      break;
    case LN:
      x=evalArbFunc(arbre->filsG);
      return( (x<=(TDouble)0) ?
	     imprErr(LN, x, 0) :
	     log(x));
      break;
    case EXP:
      x=evalArbFunc(arbre->filsG);
      return((x>(double)709) ?
	     imprErr(EXP, x, 0) :
	     ((x<(double)-745) ?
	       (double)0 :
		exp(x)));
      break;
    case SIN:
      return(sin(evalArbFunc(arbre->filsG)));
      break;
    case COS:
      return(cos(evalArbFunc(arbre->filsG)));
      break;
    case IF:
      if (arbre->filsD==NULL)
	{
	  return((double)0);
	}
      if (evalArbFunc(arbre->filsG)!=0)
	{
	  return(evalArbFunc(arbre->filsD->filsG));
	}
      else
	{
	  return(evalArbFunc(arbre->filsD->filsD));
	}
      break;
    case OR:
      return(evalArbFunc(arbre->filsG)+evalArbFunc(arbre->filsD));
      break;
    case AND:
      return(evalArbFunc(arbre->filsG)*evalArbFunc(arbre->filsD));
      break;
    case EQ:
      return( (evalArbFunc(arbre->filsG)==evalArbFunc(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case NEQ:
      return( (evalArbFunc(arbre->filsG)!=evalArbFunc(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case LE:
      return( (evalArbFunc(arbre->filsG)<evalArbFunc(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case LEQ:
      return( (evalArbFunc(arbre->filsG)<=evalArbFunc(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case GE:
      return( (evalArbFunc(arbre->filsG)>evalArbFunc(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case GEQ:
      return( (evalArbFunc(arbre->filsG)>=evalArbFunc(arbre->filsD)) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case NOT:
      return( (evalArbFunc(arbre->filsG)==(TDouble)0) ?
	     (TDouble)1 :
	     (TDouble)0 );
      break;
    case PUISSANCE:
      x=evalArbFunc(arbre->filsG);
      y=evalArbFunc(arbre->filsD);
      return(((x==0 && y<=0) || (x<0 && (int)y!=y)) ?
	     imprErr(PUISSANCE, x, y) :
	     pow(x, y));
      break;
    }
  /* AB: 29/06/2001 En principe, on ne passe pas par la */
return((TDouble)0);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : calcpsi_                   |
| Role                  : calcule la reponse pour un modele simple
| Parametres d'entree   : nbt nbg nbl nbc t g x    |
| Parametres de sortie  : f df ie le               |
| Parametres d'e./s.    :                          |
| Reference conception  : DCP module 24            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : evalArbFunc                   |
| Fonctions appelantes :                           |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt calcpsi_(nbp, nbg, nbl, nbc, p, g, z, psi, dpsi, le, ie)
      TShortInt nbp, nbg, nbl, nbc;
      TDouble *p, *g, **z, *psi, **dpsi;
      TShortInt *le, *ie;

/*--------------- Fin identification fonction ----*/

{
  TShortInt i, xt, j, ii;

  ECRTRACE("calcpsi");

  /* recopie des parametres d'entree */
  for (i=0; i<nbp; i++)
    {
      GModSpecifPsi.ParamEntre[i+nbc]->valeur=p[i];
    }
  xt=nbc+nbp;

  for (i=0; i<nbg; i++)
    {
      GModSpecifPsi.ParamEntre[i+xt]->valeur=g[i];
    }

  /* boucle sur les nbl observations */
  if (GModSpecifPsi.IdeZ.nbele!=0)
    {
      /* Cas ou l'on a plusieur Z et un seul PSI */
      for (i=0; i<nbl; i++)
	{
	  /* recopie des parametres d'entree */
	  for (j=0; j<nbc; j++)
	    {
	      GModSpecifPsi.ParamEntre[j]->valeur=z[i][j];
	    }
	  /* Evaluation de l'arbre de PSI */
	  psi[i]=evalArbFunc(GModSpecifPsi.IdePsi.UnPsi[0].arbre);
	  if (finite(psi[i])!=1)
	    {
	      *ie=i+1;
	      *le=F;
	      return(ERRMATH);
	    }
	  
	  /* evaluation des arbres des dPsi */
	  for (j=0; j<nbp; j++)
	    {
	      dpsi[i][j]=evalArbFunc(DeriveesPsi.DerivPsi.arbre[j]);
	      if (finite(dpsi[i][j])!=1)
		{
		  *ie=i+1;
		  *le=DFDT;
		  return(ERRMATH);
		}
	    }
	}
    }
  else
    {
      /* Cas ou l'on a plusieurs PSI et pas de Z */
      for (i=0; i<GModSpecifPsi.IdePsi.nbele; i++)
	{
	  /* evaluation des arbres de PSI */
	  psi[i]=evalArbFunc(GModSpecifPsi.IdePsi.UnPsi[i].arbre);
	  if (finite(psi[i])!=1)
	    {
	      *ie=i+1;
	      *le=F;
	      return(ERRMATH);
	    }
	  
	  /* evaluation des arbres des dPsi */
	  for (j=0; j<nbp; j++)
	    {
	      ii=j*GModSpecifPsi.IdePsi.nbele+i;
	      dpsi[i][j]=evalArbFunc(DeriveesPsi.DerivPsi.arbre[ii]);
	      if (finite(dpsi[i][j])!=1)
		{
		  *ie=i+1;
		  *le=DFDT;
		  return(ERRMATH);
		}
	    }
	}
    }
  return(OK);
}
