/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : CCov2                     |
| Date                 : 1992                      |
| Derniere mise a jour :                           |
| Concepteur           : A. Bouvier                |
| Role                 : les programmes de calcul  |
|   la variance asymptotique a l'etape 2           |
| Reference conception :                           |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <stdio.h>
#include <errno.h>

#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlglobal.h"
#include "nlmacros.h"
#include "errcodes.h"


/*--------------- FONCTIONS EXTERNES -------------*/
/* fonctions des autres modules */
TShortInt CCovNu(), CEtamv(), CopyMat(), CreerMat(), CreerVect(), CDmvt(),CVarZmv(), CValW(),
          DModAAct(),
          EcrMat(),
          InvMat(),
          ModAAct(), MultMat(), MultMatT(),
          TrValW(),
          VNuAVMod();

void DetruMat(TMat *pmat);
void DetruVect(TVect *pvect);
TShortInt GerMessage( );


/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueNu2N               |
| Role                  : allouer les structures de|
|  travail relatives aux equations normales        |
|  necessaires au programme CCovNu2                |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         CreerMat,                |
| Fonctions appelantes :  CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt AlloueNu2N( NbZ1, NbZ2, NbAct1, NbAct2,
                     ValB1t, ValD1t, ValD2t, ValW1t, ValW2t , ValWInv)

/* arguments d'entree */
TShortInt NbZ1, NbZ2, NbAct1, NbAct2;

/* arguments de sortie */
TMat *ValB1t, *ValD1t, *ValD2t, *ValW1t, *ValW2t , *ValWInv;

/*--------------- Fin identification fonction ----*/

{

APPEL(CreerMat(NbZ1, NbAct1,  ValB1t));
APPEL(CreerMat(NbZ1, NbAct1,  ValD1t));
APPEL(CreerMat(NbZ2, NbAct1,  ValD2t));
APPEL(CreerMat(NbAct1, NbAct1,  ValW1t));
APPEL(CreerMat(NbAct2, NbAct1,  ValW2t));
APPEL(CreerMat(NbAct2, NbAct2, ValWInv));

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueNu2P               |
| Role                  : allouer les structures de|
|  travail relatives aux parametres                |
|  necessaires au programme CCovNu2                |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         CreerMat,                |
|                         CreerVect                |
| Fonctions appelantes :  CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt AlloueNu2P(NbObs, NbTEff1, NbBEff1, NbTAct1, NbBAct1,
                    DEffT1t, DEffB1t, DValf1t, DVarYT1t, DVarYB1t,
                    TAct1t, BAct1t, TEff1t, BEff1t)

/* arguments d'entree */
TShortInt NbObs;
TShortInt NbTEff1, NbBEff1, NbTAct1, NbBAct1;

/* arguments de sortie */
TMat *DEffT1t, *DEffB1t, *DValf1t, *DVarYT1t, *DVarYB1t;
TVect *TAct1t, *BAct1t, *TEff1t, *BEff1t;

/*--------------- Fin identification fonction ----*/

{


APPEL(CreerMat(NbObs, NbTEff1, DEffT1t));
APPEL(CreerMat(NbObs, NbBEff1, DEffB1t));
APPEL(CreerMat(NbObs, NbTAct1, DValf1t));
APPEL(CreerMat(NbObs, NbTAct1, DVarYT1t));
APPEL(CreerMat(NbObs, NbBAct1, DVarYB1t));

APPEL(CreerVect(NbTAct1, TAct1t));
APPEL(CreerVect(NbBAct1, BAct1t));
APPEL(CreerVect(NbTEff1, TEff1t));
APPEL(CreerVect(NbBEff1, BEff1t));

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueNu2T               |
| Role                  : allouer les structures de|
|  travail non allouees par AlloueNu2N et          |
|  AlloueNu2P necessaires au programme CCovNu2     |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         CreerMat,                |
| Fonctions appelantes :  CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt AlloueNu2T(NbAct1, NbAct2, NbZ1, NbZ2,
                     AsVar12, AsVarNu1t, AsVarNu2t, BVarZBP1t,
                     CovZ21t, G12, Trav)


/* arguments d'entree */
TShortInt NbAct1, NbAct2, NbZ1, NbZ2;

/* arguments de sortie */
TMat *AsVar12, *AsVarNu1t, *AsVarNu2t, *BVarZBP1t,
                     *CovZ21t, *G12, *Trav;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt MaxNbAct, MaxNbZ;

MaxNbAct = MAX(NbAct1, NbAct2);
MaxNbZ = MAX(NbZ2, NbZ1);

APPEL(CreerMat(NbAct2, NbAct1, AsVar12));
APPEL(CreerMat(NbAct1, NbAct1, AsVarNu1t));
APPEL(CreerMat(NbAct2, NbAct2, AsVarNu2t));
APPEL(CreerMat(NbAct1, NbAct1, BVarZBP1t));
APPEL(CreerMat(MaxNbZ, MaxNbZ, CovZ21t));
APPEL(CreerMat(NbAct2, NbAct2, G12));
APPEL(CreerMat(MaxNbAct, MaxNbAct, Trav));

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : DesAlloueNu2N            |
| Role                  : desallouer les structures|
|  de travail allouees par AlloueNu2N dans l'ordre |
|  inverse d'allocation                            |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         DetruMat,                |
| Fonctions appelantes :  CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt DesAlloueNu2N(
                     ValB1t, ValD1t, ValD2t, ValW1t, ValW2t , ValWInv)

/* arguments d'entree */
TMat *ValB1t, *ValD1t, *ValD2t, *ValW1t, *ValW2t , *ValWInv;

/*--------------- Fin identification fonction ----*/

{
DetruMat( ValWInv);
DetruMat(  ValW2t);
DetruMat(  ValW1t);
DetruMat(  ValD2t);
DetruMat(  ValD1t);
DetruMat(  ValB1t);

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : DesAlloueNu2P            |
| Role                  : desallouer les structures|
|  de travail allouees par AlloueNu2P dans l'ordre |
|  inverse d'allocation                            |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         DetruMat,                |
|                         DetruVect                |
| Fonctions appelantes :  CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt DesAlloueNu2P(
                    DEffT1t, DEffB1t, DValf1t, DVarYT1t, DVarYB1t,
                    TAct1t, BAct1t, TEff1t, BEff1t)

/* arguments d'entree */
TMat *DEffT1t, *DEffB1t, *DValf1t, *DVarYT1t, *DVarYB1t;
TVect *TAct1t, *BAct1t, *TEff1t, *BEff1t;

/*--------------- Fin identification fonction ----*/

{
DetruVect( BEff1t);
DetruVect( TEff1t);
DetruVect( BAct1t);
DetruVect( TAct1t);
DetruMat( DVarYB1t);
DetruMat( DVarYT1t);
DetruMat( DValf1t);
DetruMat( DEffB1t);
DetruMat( DEffT1t);


return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : DesAlloueNu2T            |
| Role                  : desallouer les structures|
|  de travail allouees par AlloueNu2T dans l'ordre |
|  inverse d'allocation                            |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         DetruMat,                |
| Fonctions appelantes :  CCovNu2                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt DesAlloueNu2T(
                     AsVar12, AsVarNu1t, AsVarNu2t, BVarZBP1t,
                     CovZ21t, G12, Trav)


/* arguments d'entree */
TMat *AsVar12, *AsVarNu1t, *AsVarNu2t, *BVarZBP1t,
                     *CovZ21t, *G12, *Trav;

/*--------------- Fin identification fonction ----*/

{
DetruMat( Trav);
DetruMat( G12);
DetruMat( CovZ21t);
DetruMat( BVarZBP1t);
DetruMat( AsVarNu2t);
DetruMat( AsVarNu1t);
DetruMat( AsVar12);

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CCov2                    |
| Role                  : calculer la matrice de   |
|  variance-covariance asymptotique                |
|  a l'etape 2 (estimation des Beta seuls)         |
| Parametres d'entree   :                          |
|  Donnees: les donnees                            |
|  Theta: la structure des parametres Theta        |
|  Beta: la structure des parametres Beta          |
|  CTheta, CBeta: les contraintes sur les          |
|   parametres du modele                           |
|  CThetaE, CBetaE: les contraintes sur les        |
|    parametres  differents Theta et  Beta         |
|  CtxNum: contexte numerique                      |
|  ResNum: les resultats statistiques              |
|  Tous les arguments ci-dessus, excepte Donnees,  |
|  sont des tableaux d'au moins 2 structures       |
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
|  ResStat: les resultats statistiques de l'etape 2|
| Parametres de sortie  :                          |
|  Code = OK ou code de CCovNu2                    |
| Parametres de sortie  :                          |
| Effets de bord: voir programme CCovNu2           |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         CCovNu2,                 |
|                         CreerMat,                |
|                         DetruMat,                |
|                         VNuAVMod                 |
| Fonctions appelantes :  NLAutres                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CCov2(  Donnees, Theta, Beta, CTheta, CBeta, CThetaE, CBetaE, CtxNum, ResNum,
                 TypeMu,  ResStat,
                 Code)


/* arguments d'entree */
TDonnees *Donnees;
TParam Theta[] , Beta[];
TContr CTheta[], CBeta[];
TContr CThetaE[], CBetaE[];
TCtxNum CtxNum[];
TResNum ResNum[];

/* arguments d'entree-sortie */
TShortInt *TypeMu;
TResStat *ResStat;

/* argument de sortie */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TMat AsVarNu2; /* matrice de variance asymptotique apres prise en compte
              des contraintes numeriques */
TShortInt NbAct, Etap2;

/* pointeur sur les fonctions du module appelees et non encore definies */
TShortInt CCovNu2();

/* Ecriture de la trace */
ECRTRACE("CCov2");

*Code = OK;
Etap2 = 1; /* indice dans les tableaux correspondant a l'etape 2 */

/* allocation de AsVarNu2 */
NbAct = Theta[Etap2].Act.nbele + Beta[Etap2].Act.nbele;
APPEL(CreerMat(NbAct, NbAct, &AsVarNu2));

/* calcul de la matrice de variance asymptotique apres prise en compte
des contraintes numeriques */

APPEL(CCovNu2 ( Donnees, Theta, Beta, CTheta, CThetaE, CBetaE, CtxNum, ResNum,
                 TypeMu,  ResStat, &AsVarNu2, Code));

if(*Code == OK)
  {
  
  /* transformation en dimension parametres du modele */
  APPEL(VNuAVMod( &AsVarNu2, &(Theta[Etap2]), &(Beta[Etap2]), &(CTheta[Etap2]), &(CBeta[Etap2]), 
                &(CThetaE[Etap2]), &(CBetaE[Etap2]),
                &(ResStat->AsVar)));
  /* c'est VNuAVMod qui alloue ResStat->AsVar */
  /* desallocation de AsVarNu2 */
  } /* fin Code == OK */

DetruMat(&AsVarNu2);



/* retour */
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : CCovNu2                  |
| Role                  : calculer la matrice de   |
|  variance-covariance asymptotique des parametres |
|  apres prise en compte des contraintes           |
|  d'inegalite numerique, a l'etape 2              |
| Parametres d'entree   :                          |
|  Donnees: les donnees                            |
|  Theta: la structure des parametres Theta        |
|  Beta: la structure des parametres Beta          |
|  CTheta: les contraintes sur les parametres Theta|
|              du modele                           |
|  CThetaE, CBetaE: les contraintes sur les        |
|    parametres  differents Theta et  Beta         |
|  CtxNum: contexte numerique                      |
|  ResNum: les resultats statistiques              |
|  Tous les arguments ci-dessus, excepte Donnees,  |
|  sont des tableaux d'au moins 2 structures       |
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
|  ResStat: les resultats statistiques de l'etape 2|
| Parametres de sortie  :                          |
|  AsVarNu2: variance asymptotique en dimension    |
|    ``actifs''                                    |
|   allouee avant l'appel                          |
|  Code = OK ou retour de CBVarZBP                 |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                       AlloueNu2N,                |
|                       AlloueNu2P,                |
|                       AlloueNu2T,                |
|                       CCovNu,                    |
|                       CDmvt,                     |
|                       CopyMat,                   |
|                       CValW,                     |
|                       CVarZmv,                   |
|                       DesAlloueNu2N,             |
|                       DesAlloueNu2P,             |
|                       DesAlloueNu2T,             |
|                       DModAAct,                  |
|                       EcrMat,                    | 
|                       InvMat,                    |
|                       ModAAct,                   |
|                       MultMat,                   |
|                       MultMatT,                  |
|                       PCValB de l'etape 1,       |
|                       PCValD de l'etape 1,       |
|                       TrValW                     |
| Fonctions appelantes :  CCov2                    |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CCovNu2( Donnees, Theta, Beta, CTheta, CThetaE, CBetaE, CtxNum, ResNum, 
                 TypeMu,  ResStat, AsVarNu2, Code)




/* arguments d'entree */
TDonnees *Donnees;
TParam Theta[] , Beta[];
TContr CTheta[];
TContr CThetaE[], CBetaE[];
TCtxNum CtxNum[];
TResNum ResNum[];


/* arguments d'entree-sortie */
TShortInt *TypeMu;
TResStat *ResStat;

/* arguments de sortie */
TMat *AsVarNu2; /* dimension NbAct, NbAct , a l'etape 2*/
TShortInt *Code;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt NbAct1, NbAct2, NbTAct1, NbBAct1, NbTEff1, NbBEff1,
          NbZ1, NbZ2, i, j , i1, i2, Etap1, Etap2;
TDouble Val, NbObsT;
TLogic  VouluBVarZBP;

/* structures pour stocker les calculs intermediaires */
TMat DEffT1t, DEffB1t, DValf1t, DVarYT1t, DVarYB1t,
     ValB1t, ValD1t, ValD2t, ValW1t, ValW2t , ValWInv,
     AsVar12, AsVarNu1t, AsVarNu2t, BVarZBP1t, CovZ21t, G12, Trav;
    
TVect TAct1t, BAct1t, TEff1t, BEff1t; 


/* pointeurs sur des elements des structures passees en argument:
sert a ameliorer la performance */
TDouble **valb, **covz21t, **asvar12, **valb1t, **asvarnu2,
**asvarnu2t, **g12;


/* Ecriture de la trace */
ECRTRACE("CCovNu2");

*Code = OK;

Etap1 = 0; /* indice dans les tableaux correspondant a l'etape 1 */
Etap2 = 1; /* indice dans les tableaux correspondant a l'etape 2 */

if (CtxNum[Etap2].PCEquN.PCValEta != CEtamv)
  {
  /* Emettre un message */
  /*  printf("On ne sait pas calculer la variance a l'etape 2 quand PCValEta # CEtamv\n"); */
  NLWARNING((WARCCOV2,1,"CCov2",WNUMER));
  *Code = WARCCOV2;
  return(OK);
  }

/* initialisations des dimensions */
NbTEff1 = Theta[Etap1].Eff.nbele;
NbBEff1 = Beta[Etap1].Eff.nbele;
NbTAct1 = Theta[Etap1].Act.nbele;
NbBAct1 = Beta[Etap1].Act.nbele;

NbAct1 = NbTAct1 + NbBAct1;
NbAct2 = Theta[Etap2].Act.nbele + Beta[Etap2].Act.nbele;
NbZ1 = CtxNum[Etap1].NbZ * Donnees->NbObs;
NbZ2 = CtxNum[Etap2].NbZ * Donnees->NbObs;

/* Allocation des structures de travail */
/* ----------------------------------- */

/* allocation des structures relatives aux parametres */
APPEL(AlloueNu2P((TShortInt)(Donnees->NbObs), NbTEff1, NbBEff1, NbTAct1, NbBAct1,
                 &DEffT1t, &DEffB1t, &DValf1t, &DVarYT1t, &DVarYB1t,
                 &TAct1t, &BAct1t, &TEff1t, &BEff1t));

/* allocation des structures relatives aux equations normales */
APPEL(AlloueNu2N(NbZ1, NbZ2, NbAct1, NbAct2,
                &ValB1t, &ValD1t, &ValD2t, &ValW1t, &ValW2t , &ValWInv));

/* allocation des autres structures de travail */
APPEL(AlloueNu2T(NbAct1, NbAct2, NbZ1, NbZ2,
                 &AsVar12, &AsVarNu1t, &AsVarNu2t, &BVarZBP1t,
                 &CovZ21t, &G12, &Trav));


/* calcul de AsVarNu2t */
/* -------------------- */
APPEL(CCovNu( ResNum[Etap2].Sigma,
              &(CtxNum[Etap2]), Donnees,
              &(ResStat->EquN.ValB), &(ResStat->ValW), &(ResStat->Ajustes),
              TypeMu,  &(ResStat->Residus), &(ResStat->Mu3), &(ResStat->Mu4), 
               &(ResStat->BVarZBP),
              &AsVarNu2t, Code));

if (*Code != OK)
  {
  /* desallocation des structures de travail */
  APPEL(DesAlloueNu2T(&AsVar12, &AsVarNu1t, &AsVarNu2t, &BVarZBP1t, 
                 &CovZ21t, &G12, &Trav));
  APPEL(DesAlloueNu2N(&ValB1t, &ValD1t, &ValD2t, &ValW1t, &ValW2t , &ValWInv));
  APPEL(DesAlloueNu2P(&DEffT1t, &DEffB1t, &DValf1t, &DVarYT1t, &DVarYB1t,
                 &TAct1t, &BAct1t, &TEff1t, &BEff1t));
  return(OK);
  }

/* Impressions */
if(GNLControle.SortImp != NULL)
  {
  /*  fprintf(GNLControle.SortImp,
"\n\n IMPRESSIONS INTERMEDIAIRES DANS LE CALCUL DE LA VARIANCE ASYMPTOTIQUE A L'ETAPE 2\n\n");
  */
  NLWARNING((IMPCOV,2,PROMPT,"2",IMP));
  fprintf(GNLControle.SortImp, "SigmaBeta2\n");
  APPEL(EcrMat(GNLControle.SortImp, &AsVarNu2t));
  }


/* Calcul de ValW1t et AsVarNu1t */
/* ----------------------------- */

/* Transformation des parametres estimes a l'etape 2 en parametres actifs
selon les contraintes de l'etape 1:
A l'etape 1, il n'y a que des Theta actifs */
APPEL(ModAAct(&(Theta[Etap2].Estim), &(CTheta[Etap1]), &(CThetaE[Etap1]), 
              &TEff1t, &TAct1t));

/* Calcul des derivees des valeurs ajustees a l'etape 2 par rapport
a TAct1 */
APPEL(DModAAct(CTheta[Etap1].Ctr, Donnees->NbObs, &(Donnees->NbObsC),
                &TAct1t, &(ResStat->FctSensib.DValf),
                &(Theta[Etap1].Pass), &(CThetaE[Etap1]),
                &DEffT1t, &DValf1t));
APPEL(DModAAct(CTheta[Etap1].Ctr, Donnees->NbObs, &(Donnees->NbObsC),
                &TAct1t, &(ResStat->FctSensib.DVarYTheta),
                &(Theta[Etap1].Pass), &(CThetaE[Etap1]),
                &DEffT1t, &DVarYT1t));

/* calcul des EquN */
APPEL(CtxNum[Etap1].PCEquN.PCValB(&(Donnees->NbRepet), &(ResStat->Ajustes.Valf),
                            &(ResStat->Ajustes.VarY),
                            &(Donnees->StatDon.S2),
                            &DValf1t, &DVarYT1t, &DVarYB1t,
                            &ValB1t));
APPEL(CtxNum[Etap1].PCEquN.PCValD(&(ResStat->Ajustes.Valf), &DValf1t, &DVarYT1t, &DVarYB1t,
                            &ValD1t));

/* calcul de ValW1t */
APPEL(CValW(Donnees->NbObsT, &ValB1t, &ValD1t, &ValW1t));

/* Transformation de ValW1t */
if (CTheta[Etap1].Ctr == VRAI)
  {
    /* ValW1t est en dimension NbAct1,NbAct1 => type=0 */
  APPEL(TrValW(0, NbTEff1, NbBEff1,
               &TAct1t, &BAct1t, &(CThetaE[Etap1]), &(CBetaE[Etap1]), &ValW1t));
  }

if(GNLControle.SortImp != NULL)
  {
  fprintf(GNLControle.SortImp, "W1Theta1Beta2\n");
  APPEL(EcrMat(GNLControle.SortImp, &ValW1t));
  }

/* calcul de AsVarNu1t */
/* sauvegarde du contexte */
VouluBVarZBP  = GNLControle.Voulu.BVarZBP;
/* mise a FAUX de ces valeurs pour que le calcul soit effectif 
sinon CCovNu considere que le calcul a deja ete fait */
GNLControle.Voulu.BVarZBP=FAUX;

APPEL(CCovNu( ResNum[Etap1].Sigma,
              &(CtxNum[Etap1]), Donnees,
              &ValB1t, &ValW1t, &(ResStat->Ajustes),
              TypeMu,  &(ResStat->Residus), &(ResStat->Mu3), &(ResStat->Mu4), 
               &BVarZBP1t,
              &AsVarNu1t, Code));

/* restauration du contexte */
GNLControle.Voulu.BVarZBP = VouluBVarZBP;
if(GNLControle.SortImp != NULL)
  {
  fprintf(GNLControle.SortImp, "SigmaTheta1\n");
  APPEL(EcrMat(GNLControle.SortImp, &AsVarNu1t));
  }

if (*Code != OK)
  {
  /* desallocation des structures de travail */
  APPEL(DesAlloueNu2T(&AsVar12, &AsVarNu1t, &AsVarNu2t, &BVarZBP1t, 
                 &CovZ21t, &G12, &Trav));
  APPEL(DesAlloueNu2N(&ValB1t, &ValD1t, &ValD2t, &ValW1t, &ValW2t , &ValWInv));
  APPEL(DesAlloueNu2P(&DEffT1t, &DEffB1t, &DValf1t, &DVarYT1t, &DVarYB1t,
                 &TAct1t, &BAct1t, &TEff1t, &BEff1t));
  return(OK);
  }


/* calcul de ValW2t */
/* ---------------- */
/* calcul de ValD2t */
APPEL(CDmvt(&(ResStat->Ajustes.Valf), &DValf1t, &DVarYT1t, &DVarYB1t,
                          &ValD2t));
/* calcul de ValW2t */
APPEL(CValW(Donnees->NbObsT, &(ResStat->EquN.ValB), &ValD2t, &ValW2t));
/* transformation de ValW2t */
if (CTheta[Etap1].Ctr == VRAI)
  {
    /* ValW2t est en dimension NbAct2,NbAct1 => type=1 */
  APPEL(TrValW(1, NbTEff1, NbBEff1,
               &TAct1t, &BAct1t, &(CThetaE[Etap1]), &(CBetaE[Etap1]), &ValW2t));
  }
if(GNLControle.SortImp != NULL)
  {
  fprintf(GNLControle.SortImp, "W2Theta\n");
  APPEL(EcrMat(GNLControle.SortImp, &ValW2t));
  }

/* calcul de - G12 */
/* ------------- */

/* calcul de CovZ21t */
APPEL(CVarZmv(Donnees->NbObsT, &(Donnees->NbRepet), &(Donnees->PoidsT),
                &(Donnees->ValY), &(ResStat->Ajustes),
                TypeMu, &(ResStat->Residus), &(ResStat->Mu3), &(ResStat->Mu4),
                &CovZ21t, Code));

if (*Code != OK)
  {
  /* desallocation des structures de travail */
  APPEL(DesAlloueNu2T(&AsVar12, &AsVarNu1t, &AsVarNu2t, &BVarZBP1t, 
                 &CovZ21t, &G12, &Trav));
  APPEL(DesAlloueNu2N(&ValB1t, &ValD1t, &ValD2t, &ValW1t, &ValW2t , &ValWInv));
  APPEL(DesAlloueNu2P(&DEffT1t, &DEffB1t, &DValf1t, &DVarYT1t, &DVarYB1t,
                 &TAct1t, &BAct1t, &TEff1t, &BEff1t));
  return(OK);
  }

/* remarque: si PCValZ[Etap1] == CZmco et PCValZ[Etap2] == CZmv,
dans le calcul de Asvar12 qui suit, on ne considera que les NbObs premieres colonnes
de CovZ21t, les NbObs suivantes seront ignorees */


/* calcul de AsVar12 */
NbObsT = (TDouble)(Donnees->NbObsT);
/* pour ameliorer la performance, on transfere les elements de structures */
valb = ResStat->EquN.ValB.donnees;
covz21t = CovZ21t.donnees;
asvar12 = AsVar12.donnees;
valb1t = ValB1t.donnees;
  

for (i=0; i < NbAct2 ; i++)
  {
  for (j=0; j < NbAct1; j++)
    {
    asvar12[i][j] = 0.0;
    for (i1=0; i1 < NbZ1; i1++)
      {
      Val = 0.0;
      for (i2=0; i2 < NbZ2; i2++)
        {
        Val = Val + (valb[i2][i] * covz21t[i2][i1]);
        }
      asvar12[i][j] = asvar12[i][j] + (Val * valb1t[i1][j]);
      }
    asvar12[i][j] = asvar12[i][j] / NbObsT;
    }
  }
 if(GNLControle.SortImp != NULL)
  {
  fprintf(GNLControle.SortImp, "Sigma12\n");
  APPEL(EcrMat(GNLControle.SortImp, &AsVar12));
  }

/* inversion de ValW1t */
APPEL(InvMat(&ValW1t));
APPEL(MultMat(&AsVar12, &ValW1t, &Trav));
APPEL(MultMatT(&Trav, &ValW2t, &G12));

/* calcul de AsVarNu2 */
/* ------------------ */
/* calcul du 1ier terme de la somme :
ValW[Etap2]^-1 * ValW2t * AsVarNu1t * ValW2t^t ValW[Etap2]^-1 */
/* inversion de ResStat->ValW */
APPEL(CopyMat(&(ResStat->ValW), &ValWInv));
APPEL(InvMat(&ValWInv));

APPEL(MultMat(&ValWInv, &ValW2t, &Trav));
APPEL(MultMat(&Trav, &AsVarNu1t, &AsVar12));
/* AsVar12 ne servant plus, on l'utilise ici pour stocker les resultats intermediaires */
APPEL(MultMatT(&AsVar12, &ValW2t, &Trav));
APPEL(MultMat(&Trav, &ValWInv, AsVarNu2));

/* calcul du 3ieme terme :
ValW[Etap2]^-1 * G12 * ValW[Etap2]^-1 */
APPEL(MultMat(&ValWInv, &G12,  &Trav));
APPEL(MultMat(&Trav, &ValWInv, &G12));
/* G12 ne servant plus, on l'utilise ici pour stocker les resultats intermediaires */
Val = (TDouble)2.0 / NbObsT;

/* addition des 3 termes */
/* pour ameliorer la performance, on transfere les elements de structures */
asvarnu2 = AsVarNu2->donnees;
asvarnu2t = AsVarNu2t.donnees;
g12 = G12.donnees;


for (i=0 ; i < NbAct2; i++)
  {
  for (j=0 ; j <= i; j++)
    {
    asvarnu2[i][j] = asvarnu2[i][j] + asvarnu2t[i][j] -(g12[i][j] / Val) ;
    asvarnu2[j][i] =asvarnu2[i][j];
    }
  }


/* Desallocation des structures de travail dans l'ordre des alloc */
/* -------------------------------------------------------------- */

/* desallocation des autres structures de travail */
APPEL(DesAlloueNu2T(&AsVar12, &AsVarNu1t, &AsVarNu2t, &BVarZBP1t, 
                 &CovZ21t, &G12, &Trav));

/* desallocation des structures relatives aux equations normales */
APPEL(DesAlloueNu2N(&ValB1t, &ValD1t, &ValD2t, &ValW1t, &ValW2t , &ValWInv));

/* desallocation des structures relatives aux parametres */

APPEL(DesAlloueNu2P(&DEffT1t, &DEffB1t, &DValf1t, &DVarYT1t, &DVarYB1t,
                 &TAct1t, &BAct1t, &TEff1t, &BEff1t));



return(OK);
}
