/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : CDirec                    |
| Date                 : dec 91                    |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : A. Bouvier                |
| Role                 : les programmes de calcul  |
|   de la direction de descente                    |
| Reference conception :                           |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <stdio.h>

#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlglobal.h"
#include "nlmacros.h"
#include "errcodes.h"

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/
TDouble sqrt();

/* fonctions des autres modules */
TShortInt MultMatVal(), SysLin(), ExtrMat(), GerMessage();

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/





/*--------------- Identification fonction ----------
| Nom de la fonction    : CDirecNorme              |
| Role                  : calculer la matrice      |
|  diagonale par laquelle on va normer les valeurs |
|  de ValA=ValW*NbObsT                             |
|  La norme est 1/sqrt(elements diagonaux de ValA) |
| Parametres d'entree   :                          |
|  ValA : matrice (NbAct,NbAct)                    |
| (on appelle NbAct le nombre total de parametres  |
|  actifs)                                         |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  MNorme: vecteur qui contient les elements de la |
|   matrice diagonale de norme (dimension NbAct)   |
| Retour fonction       : OK ou ERRNORME           |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune, mais la fonction |
| systeme "sqrt" dont les arguments sont des double|
| Fonctions appelantes :  CDirGM                   |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CDirecNorme( ValA,
                       MNorme)

/* arguments d'entree */
TMat *ValA;

/* arguments de sortie*/
TVect *MNorme;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, NbAct;
TChar Mess1[5], Mess2[15];
TDouble **vala;
TDouble *mnorme;

/* Ecriture de la trace */
ECRTRACE("CDirecNorme");


NbAct = ValA->nblig;
vala = ValA->donnees;
mnorme = MNorme->donnees;

for(i = 0; i < NbAct; i++)
  {
  if (vala[i][i] <= (TDouble)ZERO)
    {
    sprintf(Mess1,"%hd", (i+1));
    sprintf(Mess2, "%g", vala[i][i]);
    /* Actions effectuees par la macro NLERREUR suivante:
    fprintf(stderr, "Le %s -ieme element de la matrice ValW est egal a %s (negatif ou nul)\n",
     Mess1, Mess2);
    return(ERRDIREC);
 */
    NLERREUR((ERRDIREC,3,"CDirecNorme", Mess1, Mess2, ERR));
    }
  mnorme[i] = (TDouble)1.0 / sqrt(vala[i][i]);
  }
return(OK);

}



/*--------------- Identification fonction ----------
| Nom de la fonction    : CDirGM                   |
| Role                  : calculer la direction    |
|  de descente quand Algo=GM                       |
|  et la matrice n'est pas de type SYMBLOC         |
| Parametres d'entree   :                          |
|  NbObsT: le nombre total de repetitions          |
|  Lambda: valeur courante de Lambda               |
|  ValW: matrice (NbAct,NbAct)                     |
| (on appelle NbAct le nombre total de parametres  |
|  actifs)                                         |
| Parametres de sortie  :                          |
|  Direc : vecteur contenant la direction de       |
|     descente (dimension NbTheta+NbBeta)          |
| Parametres d'e/s      :                          |
|  ValR: le vecteur ValB(ValZ-ValEta) en entree,   |
|    modifie en sortie                             |
|              (dimension NbAct)                   |
| Parametres de travail :                          |
|  MNorme: vecteur de travail qui sert a stocker   |
|   les valeurs par lesquelles on va normer        |
|              (dimension NbAct)                   |
|  Tgamma: vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  Tv    : vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  Tc    : vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  ValA: matrice de travail                        |
|              (dimension NbAct, NbAct)            |
|  ValRsv: matrice de travail pour SysLin          |
|              (dimension NbAct, 1)                |
|  Direcres: matrice de travail resultat de SysLin |
|              (dimension NbAct, 1)                |
| Parametres inutilises, introduits pour           |
|  homogeneiser l'appel aux programmes de calcul   |
|  de la direction de descente:                    |
|  BlocP: Matrice NbTheta, NbTheta                 |
|  partie superieure gauche de  ValW               |
|  BlocQ: Matrice NbBeta, NbTheta                  |
|  partie inferieure gauche de ValW                |
|  BlocR: Matrice NbBeta, NbBeta                   |
|  partie inferieure droite de ValW                |
| Retour fonction       : OK ou ERRSYS             |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CDirecNorme,             |
|                         MultMatVal,              |
|                         SysLin                   |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CDirGM( NbObsT, Lambda, ValW,
                  Direc,
                  ValR, 
                  MNorme, Tgamma, Tv, Tc, ValA, ValRsv, Direcres,
                  BlocP, BlocQ , BlocR)

/* arguments d'entree */
TLongInt NbObsT;
TDouble Lambda;
TMat *ValW;

/* arguments de sortie*/
TVect *Direc;

/* arguments d'entree- sortie*/
TVect *ValR;

/* arguments de travail */
TVect *MNorme, *Tgamma, *Tv;
TVectShort *Tc;
TMat *ValA,  *ValRsv, *Direcres;


/* arguments inutilises */
TMat  *BlocP, *BlocQ, *BlocR;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i,j, NbAct, Code, Rang;
TDouble nbobst;
TDouble  **vala, **valrsv, **direcres;
TDouble *valr, *mnorme, *direc;

/* Ecriture de la trace */
ECRTRACE("CDirGM");


/* initialisations */
nbobst = (TDouble)NbObsT;
NbAct = ValW->nblig;
vala = ValA->donnees;
valr = ValR->donnees;
valrsv = ValRsv->donnees;
mnorme = MNorme->donnees;
direc = Direc->donnees;
direcres = Direcres->donnees;

/* calcul de ValA= ValW * NbObsT */
APPEL(MultMatVal(ValW, nbobst, ValA));


/* determiner la matrice diagonale de NbAct elements par laquelle on va
normaliser; on la stocke dans le vecteur MNorme */
APPEL(CDirecNorme(ValA, MNorme));

/* Normaliser */
for(i = 0; i < NbAct; i++)
  {
  for(j = 0; j < NbAct; j++)
    {
    vala[i][j] =  vala[i][j] * mnorme[i] * mnorme[j];
    }
  valr[i] = valr[i] * mnorme[i];
  /* modifier la diagonale de ValA */
  vala[i][i] =  vala[i][i] + Lambda;
  } /* fin boucle sur i */

/* resoudre: ValA*Direc=ValR  */
/* Transformation de ValR en matrice pour l'appel a SysLin */
for (i=0; i< NbAct; i++)
  {
  valrsv[i][0] = valr[i];
  }


Code = SysLin(NbAct, NbAct, (TShortInt)1, (TDouble)EPSILON, 
       &(ValA->donnees[0]), &(ValRsv->donnees[0]),
       &(Direcres->donnees[0]), &Rang,
       &(Tv->donnees[0]), &(Tgamma->donnees[0]), &(Tc->donnees[0]));


if (Code != OK)
  {
  /* Actions effectuees par la macro NLERREUR suivante:
  fprintf(stderr, "La resolution du systeme lineaire ne peut etre effectuee.\n");
  return(ERRSYS);
*/
  NLERREUR((ERRSYS,1,"CDirGM", ERR));
  }

/* Transformation de la sortie en un vecteur et
   calculer Direc a la bonne echelle */
for(i = 0; i < NbAct; i++)
  {
  direc[i] = direcres[i][0] * mnorme[i];
  } /* fin boucle sur i */



/* retour */
return(OK);
}




/*--------------- Identification fonction ----------
| Nom de la fonction    : CDirGMBloc               |
| Role                  : calculer la direction    |
|  de descente quand Algo=GM                       |
|  et la matrice est de type SYMBLOC               |
| Parametres d'entree   :                          |
|  NbObsT: le nombre total de repetitions          |
|  Lambda: valeur courante de Lambda               |
|  ValW: matrice (NbAct,NbAct)                     |
| (on appelle NbAct le nombre total de parametres  |
|  actifs)                                         |
| Parametres de sortie  :                          |
|  Direc : vecteur contenant la direction de       |
|     descente (dimension NbTheta+NbBeta)          |
| Parametres d'e/s      :                          |
|  ValR: le vecteur ValB(ValZ-ValEta)              |
|              (dimension NbAct)                   |
| Parametres de travail :                          |
|  MNorme: vecteur de travail qui sert a stocker   |
|   les valeurs par lesquelles on va normer        |
|              (dimension NbAct)                   |
|  Tgamma: vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  Tv    : vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  Tc    : vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  ValA: matrice de travail                        |
|              (dimension NbAct, NbAct)            |
|  ValRsv: matrice de travail pour SysLin          |
|              (dimension NbAct, 1)                |
|  Direcres: matrice de travail resultat de SysLin |
|              (dimension NbAct, 1)                |
|  BlocP: Matrice NbTheta, NbTheta                 |
|  partie superieure gauche de  ValW               |
|  BlocQ: Matrice NbBeta, NbTheta                  |
|  partie inferieure gauche de ValW                |
|  BlocR: Matrice NbBeta, NbBeta                   |
|  partie inferieure droite de ValW                |
| Retour fonction       : OK ou ERRSYS             |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CDirecNorme,             |
|                         ExtrMat,                 |
|                         MultMatVal,              |
|                         SysLin                   |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CDirGMBloc( NbObsT, Lambda, ValW,
                  Direc,
                  ValR,
                  MNorme, Tgamma, Tv, Tc, ValA, ValRsv, Direcres,
                  BlocP, BlocQ , BlocR)


/* arguments d'entree */
TLongInt NbObsT;
TDouble Lambda;
TMat *ValW;

/* arguments de sortie*/
TVect *Direc;

/* arguments d'entree- sortie*/
TVect *ValR;

/* arguments de travail */
TVect *MNorme, *Tgamma, *Tv;
TVectShort *Tc;
TMat *ValA,  *ValRsv, *Direcres;
TMat  *BlocP, *BlocQ, *BlocR;


/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i,j, NbAct, NbTheta, NbBeta, Code , Rang;
TDouble Trav;
TDouble  **vala, **valrsv, **direcres, **blocq;
TDouble *valr, *mnorme, *direc;

/* Ecriture de la trace */
ECRTRACE("CDirGMBloc");


/* initialisations */
NbTheta = BlocP->nblig;
NbBeta = BlocQ->nblig;
NbAct = NbTheta + NbBeta;
vala = ValA->donnees;
valr = ValR->donnees;
valrsv = ValRsv->donnees;
mnorme = MNorme->donnees;
blocq = BlocQ->donnees;
direc = Direc->donnees;
direcres = Direcres->donnees;

/* calcul de ValA= ValW * NbObsT */
APPEL(MultMatVal(ValW, (TDouble)NbObsT, ValA));

/* determiner la matrice diagonale de NbAct elements par laquelle on va
normaliser; on la stocke dans le vecteur MNorme */
APPEL(CDirecNorme(ValA, MNorme));

/* Normaliser */
for(i = 0; i < NbAct; i++)
  {
  for(j = 0; j < NbAct; j++)
    {
    vala[i][j] =  vala[i][j] * mnorme[i] * mnorme[j];
    }
  valr[i] = valr[i] * mnorme[i];
  /* modifier la diagonale de ValA */
  vala[i][i] =  vala[i][i] + Lambda;
  } /* fin boucle sur i */

/* decomposer ValA en blocs qui sont symetriques */
APPEL(ExtrMat((TShortInt)0,(TShortInt)0,NbTheta,NbTheta, ValA, BlocP));
APPEL(ExtrMat(NbTheta,(TShortInt)0,NbBeta,NbTheta, ValA, BlocQ));
APPEL(ExtrMat(NbTheta,NbTheta,NbBeta, NbBeta, ValA ,BlocR));

/* resoudre BlocP*Direc1=ValR1
en appelant Direc1  les NbTheta 1iers elements de Direc,
            ValR1   les NbTheta 1iers elements de ValR */
/* Transformation de ValR en matrice pour l'appel a SysLin */
for (i=0; i< NbAct; i++)
  {
  valrsv[i][0] = valr[i];
  }

Code = SysLin(NbTheta, NbTheta, (TShortInt)1, (TDouble)EPSILON,
             &(BlocP->donnees[0]), &(ValRsv->donnees[0]),
             &(Direcres->donnees[0]), &Rang,
             &(Tv->donnees[0]), &(Tgamma->donnees[0]), &(Tc->donnees[0]));

if (Code != OK)
  {
  /* Actions effectuees par la macro NLERREUR suivante:
  fprintf(stderr, "La resolution du systeme lineaire  ne peut etre effectuee.\n");
  return(ERRSYS);
*/
  NLERREUR((ERRSYS,1,"CDirGMBloc", ERR));
  }

/* resoudre BlocR*Direc2=ValR2 - BlocQ*Direc1 
en appelant Direc1  les NbTheta 1iers elements de Direc,
          Direc2  les NbBeta  suivants,
          ValR2   les NbBeta derniers elements de ValR */
/* calcul de ValR2= ValR2 - BlocQ*Direc1 */
for(i = 0; i < NbBeta; i++)
  {
  Trav = (TDouble)ZERO;
  for(j = 0; j < NbTheta; j++)
    {
    Trav = Trav + (blocq[i][j] * direcres[j][0]);
    }
  valr[i+NbTheta] = valr[i+NbTheta] -  Trav;
  } /* fin boucle sur i */

/* resolution de BlocR*Direc2=ValR2 */
/* Transformation de ValR en matrice pour l'appel a SysLin */

Code = SysLin(NbBeta,NbBeta,(TShortInt)1, (TDouble)EPSILON,
       &(BlocR->donnees[0]), &(ValRsv->donnees[NbTheta]), 
       &(Direcres->donnees[NbTheta]), &Rang,
       &(Tv->donnees[0]), &(Tgamma->donnees[0]), &(Tc->donnees[0]));

if (Code != OK)
  {
  /* Actions effectuees par la macro NLERREUR suivante:
  fprintf(stderr, "La resolution du systeme lineaire ne peut etre effectuee\n");
  return(ERRSYS);
*/
  NLERREUR((ERRSYS,1,"CDirGMBloc", ERR));
  }


/* Transformation de la sortie en un vecteur et
   calculer Direc a la bonne echelle */
for(i = 0; i < NbAct; i++)
  {
  direc[i] = direcres[i][0] * mnorme[i];
  } /* fin boucle sur i */

/* retour */
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : CDirGN                   |
| Role                  : calculer la direction    |
|  de descente quand Algo=GN et la matrice n'est   |
|  pas de type SYMBLOC                             |
| Parametres d'entree   :                          |
|  NbObsT: le nombre total de repetitions          |
|  ValR: le vecteur ValB(ValZ-ValEta)              |
|              (dimension NbAct)                   |
| (on appelle NbAct le nombre total de parametres  |
|  actifs)                                         |
|  ValWInv:  inverse de ValW                       |
|        matrice (NbAct,NbAct)                     |
| Parametres de sortie  :                          |
|  Direc : vecteur contenant la direction de       |
|     descente (dimension NbAct)                   |
| Parametres inutilises, introduits pour           |
|  homogeneiser l'appel aux programmes de calcul   |
|  de la direction de descente:                    |
|  Lambda: valeur courante de Lambda               |
|  MNorme: vecteur de travail                      |
|              (dimension NbAct)                   |
|  Tgamma: vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  Tv    : vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  Tc    : vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  ValW: matrice (NbAct,NbAct)                     |
|  ValRsv: matrice de travail pour SysLin          |
|              (dimension NbAct, 1)                |
|  Direcres: matrice de travail resultat de SysLin |
|              (dimension NbAct, 1)                |
|  BlocP: Matrice NbTheta, NbTheta                 |
|  partie superieure gauche de  ValW               |
|  BlocQ: Matrice NbBeta, NbTheta                  |
|  partie inferieure gauche de ValW                |
|  BlocR: Matrice NbBeta, NbBeta                   |
|  partie inferieure droite de ValW                |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CDirGN( NbObsT, Lambda, ValW,
                  Direc,
                  ValR,
                  MNorme, Tgamma, Tv, Tc, ValWInv, ValRsv, Direcres,
                  BlocP, BlocQ , BlocR)


/* arguments d'entree */
TLongInt NbObsT;
TVect *ValR;
TMat *ValWInv;


/* arguments de sortie*/
TVect *Direc;

/* arguments inutilises */
TDouble Lambda;
TMat *ValW;
TVect *MNorme, *Tgamma, *Tv;
TVectShort *Tc;
TMat *ValRsv, *Direcres, *BlocP, *BlocQ, *BlocR;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i,j , NbAct;
TDouble Trav;
TDouble **valwinv;
TDouble *valr, *direc;

/* Ecriture de la trace */
ECRTRACE("CDirGN");


/* initialisations */
NbAct = ValR->nbele;
valr = ValR->donnees;
valwinv = ValWInv->donnees;
direc = Direc->donnees;

/* resoudre: NbObsT*ValW*Direc = ValR
   soit: Direc= (ValWInv*ValR)/NbObsT */

for(i = 0; i < NbAct; i++)
  {
  Trav = (TDouble)ZERO;

  for(j = 0; j < NbAct; j++)
    {
    Trav = Trav + (valwinv[i][j] * valr[j]);
    }
  direc[i] = Trav / (TDouble)NbObsT;
  } /* fin boucle sur i */

/* retour */
return(OK);
}




/*--------------- Identification fonction ----------
| Nom de la fonction    : CDirGNBloc               |
| Role                  : calculer la direction    |
|  de descente quand Algo=GN et la matrice est     |
|  symetrique par blocs                            |
| Parametres d'entree   :                          |
|  NbObsT: le nombre total de repetitions          |
|  ValW: matrice (NbAct,NbAct)                     |
| (on appelle NbAct le nombre total de parametres  |
|  actifs)                                         |
|  BlocPInv: Matrice NbTheta, NbTheta              |
|  partie superieure gauche de  ValW inverse       |
|  BlocRInv: Matrice NbBeta, NbBeta                |
|  partie inferieure droite de ValW inverse        |
| Parametres de sortie  :                          |
|  Direc : vecteur contenant la direction de       |
|     descente (dimension NbTheta+NbBeta)          |
| Parametres d'e/s      :                          |
|  ValR: le vecteur ValB(ValZ-ValEta) en entree,   |
|   modifie en sortie                              |
|              (dimension NbAct)                   |
| Parametres inutilises, introduits pour           |
|  homogeneiser l'appel aux programmes de calcul   |
|  de la direction de descente:                    |
|  Lambda: valeur courante de Lambda               |
|  MNorme: vecteur de travail                      |
|              (dimension NbAct)                   |
|  Tgamma: vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  Tv    : vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  Tc    : vecteur de travail pour SysLin          |
|              (dimension NbAct)                   |
|  ValA: matrice de travail                        |
|              (dimension NbAct, NbAct)            |
|  ValRsv: matrice de travail pour SysLin          |
|              (dimension NbAct, 1)                |
|  Direcres: matrice de travail resultat de SysLin |
|              (dimension NbAct, 1)                |
|  BlocQ: Matrice NbBeta, NbTheta                  |
|  partie inferieure gauche de ValW                |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt CDirGNBloc( NbObsT, Lambda, ValW,
                      Direc,  ValR,
                      MNorme, Tgamma, Tv, Tc, ValA, ValRsv, Direcres,
                      BlocPInv, BlocQ , BlocRInv)

/* arguments d'entree */
TLongInt NbObsT;
TMat *ValW;
TMat *BlocPInv, *BlocRInv;

/* arguments d'entree- sortie*/
TVect *ValR;


/* arguments de sortie*/
TVect *Direc;

/* arguments ignores */
TDouble Lambda;
TVect *MNorme, *Tgamma, *Tv;
TVectShort *Tc;
TMat *ValA,  *ValRsv, *Direcres, *BlocQ;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i,j, NbTheta, NbBeta;
TDouble Trav;
TDouble **blocpinv, **blocrinv, **valw;
TDouble *valr, *direc;

/* Ecriture de la trace */
ECRTRACE("CDirGNBloc");


/* initialisations */
NbTheta = BlocPInv->nblig;
NbBeta = BlocRInv->nblig;
blocpinv = BlocPInv->donnees;
blocrinv = BlocRInv->donnees;
valw = ValW->donnees;
valr = ValR->donnees;
direc =  Direc->donnees;


/* resoudre: NbObsT*BlocP*Direc = ValR
en ne considerant que  les NbTheta 1iers elements de Direc et ValR */

for(i = 0; i < NbTheta; i++)
  {
  Trav = (TDouble)ZERO;

  for(j = 0; j < NbTheta; j++)
    {
    Trav = Trav + (blocpinv[i][j] * valr[j]);
    }

  direc[i] = Trav / (TDouble)NbObsT;
  } /* fin boucle sur i */

  
/* resoudre: NbObsT*(BlocQ*Direc1 + BlocR*Direc2) = ValR2
en appelant Direc1  les NbTheta 1iers elements de Direc,
            Direc2  les NbBeta  suivants,
            BlocQ: bloc (NbBeta, NbTheta) inferieur gauche de ValW
            ValR2   les NbBeta derniers elements de ValR */
 /* calcul de VALR2= VALR2- NbObsT*BlocQ*Direc1 */
for(i = 0; i < NbBeta; i++)
  {
  Trav = (TDouble)ZERO;

  for(j = 0; j < NbTheta; j++)
    {
    Trav = Trav + (valw[i + NbTheta][j] * direc[j]);
    }
  valr[i+NbTheta] = valr[i+NbTheta] - ((TDouble)NbObsT * Trav);
  } /* fin boucle sur i */

/* calcul de NbObsT*BlocR*Direc2= ValR2 
c.a.d:       Direc2= (BlocRInv*ValR2)/NbObsT */
for(i = 0; i < NbBeta; i++)
  {
  Trav = (TDouble)ZERO;

  for(j = 0; j < NbBeta; j++)
    {
    Trav = Trav + (blocrinv[i][j] * valr[j+NbTheta]);
    }
  direc[i+NbTheta] = Trav / (TDouble)NbObsT;
  } /* fin boucle sur i */

/* retour */
return(OK);
}

