/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : CModele                   |
| Date                 : 1992                      |
| Derniere mise a jour :                           |
| Concepteur           : A. Bouvier                |
| Role                 : les programmes de calcul  |
|  du modele et ceux appeles lorsqu'il y a une     |
|  erreur                                          |
| Reference conception :                           |
| Lecteur              : G. Husser                 |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/


#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlmacros.h"
#include "errcodes.h"
#include "nlglobal.h"

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/

/* fonctions des autres modules */
TShortInt ActAMod(), DModAAct(), CreerVect(), EcrVect(),
GerMessage(), MultVectVal(), DivVectVect(),
DivMatVect(), MultMatVal();
TShortInt ModSedo(TShortInt nbt,TShortInt nbg,TShortInt nbl,TShortInt nbc,
		  TDouble *theta, TDouble *gamf,TDouble **xobs,
		  TDouble ***FSedo,TDouble ***DFSedo,
		  TDouble *f, TDouble **df,
		  TShortInt *le,TShortInt *ie,
		  TShortInt indc);

/* fonctions fournies par l'utilisateur */
 TShortInt  calcf_(), calcv_();

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : CModele                  |
| Role                  :  Calcul des valeurs du   |
|  modele                                          |
| Parametres d'entree   :                          |
|  Donnees: les donnees                            |
|  Modele: le modele,                              |
|  CtxPuss: le contexte du processus de l'etape    |
|           courante                               |
|  ThetaCour, BetaCour: vecteur des valeurs        |
|   courantes des parametres actifs                |
|  CTheta, CBeta: contraintes sur les parametres   |
|   du modele                                      |
|  CThetaE, CBetaE: contraintes sur les parametres |
|  differents                                      |
| Parametres d'e/s      :                          |
|  Theta, Beta: les parametres de l'etape courante:|
|   La valeur de Noms.nbele est utilisee comme une |
|   entree, pour determiner le nombre de parametres|
|   du modele.                                     |
|   Le composant Estim contient en sortie la valeur|
|   des parametres courants (ThetaCour et BetaCour)|
|   en dimension "modele*NbCourbe" et le composant |
|   Eff en dimension "differents"                  |
|  DEffT, DEffB: matrices de travail qui servent a |
|   stocker les derivees par rapport aux parametres|
|   en dimension "differents"                      |
| Parametres de sortie  :                          |
|  FctSA: fonctions de sensibilite en dimension    |
|   "actif"                                        |
|  ResStat: les resultats statistiques:            |
|   les composants FctSensib et Ajustes sont des   |
|   sorties                                        |
|  ResNum: les resultats numeriques:               |
|   les composants InfoModErr et Scr sont des      |
|   sorties                                        |
|  ModErr: OK ou code d'erreur ERRMODF ou ERRMODV  |
| Retour fonction       : OK ou code des fonctions |
|   appelees                                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                        ActAMod,                  |
|                        calcv,                    |
|                        CreerVect                 |
|                        CVariance,                |
|                        DModAAct,                 |
|                        calcf_ ou ModSedo         |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CModele(Donnees, Modele, CtxPuss, ThetaCour, BetaCour,
                                CTheta, CBeta, CThetaE, CBetaE, 
                  Theta, Beta,DEffT, DEffB, 
                   FctSA, ResStat ,ResNum, ModErr)
/* arguments d'entree */
TDonnees *Donnees;
TModele *Modele;
TCtxPuss *CtxPuss;
TVect *ThetaCour, *BetaCour;
TContr *CTheta, *CBeta;
TContr *CThetaE, *CBetaE;

/* arguments d'entree-sortie */
TParam *Theta, *Beta;
TMat *DEffT, *DEffB;

/* arguments de sortie */
TFctSensib *FctSA;
TResStat *ResStat;
TResNum *ResNum;
TShortInt *ModErr;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt IParamT, IParamB, IObs, ICourbe, NbTheta, NbBeta, NbObs, j, i, nbt, nbele;
TShortInt IndErr, nbcourbe, nbvarex;

/* pointeur sur les fonctions du module appelees */
TShortInt CVariance();

/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble **dvarytheta;
TInfoModErr *ptr;
TDouble *VarY;

/* Ecriture de la trace */
ECRTRACE("CModele");

/* initialisations */
NbTheta = Theta->Noms.nbele;
NbBeta = Beta->Noms.nbele;
NbObs = Donnees->NbObs;
nbcourbe = Donnees->NbCourbe;
nbvarex = Donnees->XObs.nbcol;
ptr = &(ResNum->InfoModErr);

*ModErr = OK;

/* Passer de la dimension "actif" des parametres a la dimension "NbParam*NbCourbe" */
if (Theta->Act.nbele > 0)
  {
  APPEL(ActAMod(ThetaCour, CTheta, CThetaE, &(Theta->Eff), &(Theta->Estim)));
  }
if (Beta->Act.nbele > 0)
  {
  APPEL(ActAMod(BetaCour, CBeta, CBetaE, &(Beta->Eff), &(Beta->Estim)));
  }

/* initialisation de la boucle sur les courbes */
IParamT = 0;
IParamB = 0;
IObs = 0;

for (ICourbe = 0 ; ICourbe < nbcourbe; ICourbe++)
  {
      ptr->CodeErr = ERRMODF;
  if (Modele->CasSedo == VRAI)
    {
    ptr->CodeErr = ModSedo(NbTheta, 
                   Modele->GamF.nbele, 
                   Donnees->NbObsC.donnees[ICourbe],
                   nbvarex,
                   &(Theta->Estim.donnees[IParamT]), 
                   &(Modele->GamF.donnees[0]),
                   &(Donnees->XObs.donnees[IObs]),
                   &(ResStat->Sedo.FSedo[IObs]),
                   &(ResStat->Sedo.DFSedo[IObs]),
                   &(ResStat->Ajustes.Valf.donnees[IObs]), 
                   &(ResStat->FctSensib.DValf.donnees[IObs]),
                   &(ResNum->InfoModErr.LieuErr), &IndErr, ICourbe);

    }
  else
    {
    ptr->CodeErr =calcf_(NbTheta, 
                   Modele->GamF.nbele, 
                   Donnees->NbObsC.donnees[ICourbe],
                   nbvarex,
                   &(Theta->Estim.donnees[IParamT]), 
                   &(Modele->GamF.donnees[0]),
                   &(Donnees->XObs.donnees[IObs]),
                   &(ResStat->Ajustes.Valf.donnees[IObs]), 
                   &(ResStat->FctSensib.DValf.donnees[IObs]),
                   &(ResNum->InfoModErr.LieuErr), &IndErr);
    }

  if (ptr->CodeErr != OK)
    {
    /* Une erreur est decelee dans le calcul de f ou df */
    /* Finir de remplir InfoModErr */
    APPEL(CreerVect(nbvarex, &(ResNum->InfoModErr.XErr)));
      /*Si l'indice  de l'observation est erron: on met 0 a la place */
    if (( IndErr < 1)  || 
        (IndErr > Donnees->NbObsC.donnees[ICourbe])) 
     IndErr=1;
    for(i = 0; i < nbvarex; i++)
      {
      ptr->XErr.donnees[i] = Donnees->XObs.donnees[IObs+IndErr-1][i];
      }
    ptr->NomXErr = Donnees->NomObs.donnees[IObs+IndErr-1];
    *ModErr=ERRMODF;
    return(OK); /* ne pas retourner ERRMODF a cause de la macro APPEL:
                   ici, il faut retourner normalement au programme appelant */
    }

  if(Modele->YaCalcV == VRAI)
    {
      ptr->CodeErr = ERRMODV;
    ptr->CodeErr =
           calcv_(NbTheta, NbBeta, Modele->GamV.nbele, 
                 Donnees->NbObsC.donnees[ICourbe],
                 nbvarex,
                 &(Theta->Estim.donnees[IParamT]), 
                 &(Beta->Estim.donnees[IParamB]),
                 &(Modele->GamV.donnees[0]),
                 &(ResStat->Ajustes.Valf.donnees[IObs]),
                 &(ResStat->FctSensib.DValf.donnees[IObs]),
                 &(Donnees->XObs.donnees[IObs]),
                 &(ResStat->Ajustes.VarY.donnees[IObs]),
                 &(ResStat->FctSensib.DVarYTheta.donnees[IObs]),
                 &(ResStat->FctSensib.DVarYBeta.donnees[IObs]),
                 &(ResNum->InfoModErr.LieuErr), &IndErr);

    if (ptr->CodeErr != OK)
      {
      /* Une erreur est decelee dans le calcul de v ou dv */
      /* Finir de remplir InfoModErr */
      APPEL(CreerVect(nbvarex, &(ResNum->InfoModErr.XErr)));

      /*Si l'indice  de l'observation est erron: on met 0 a la place */
    if (( IndErr < 1)  || 
        (IndErr > Donnees->NbObsC.donnees[ICourbe])) 
     IndErr=1;

      for(i = 0; i < nbvarex; i++)
        {
        ptr->XErr.donnees[i] = Donnees->XObs.donnees[IObs+IndErr-1][i];
        }
      ptr->NomXErr = Donnees->NomObs.donnees[IObs+IndErr-1];
      *ModErr = ERRMODV;
      return(OK);
      } /* fin CodeErr != OK */
    } /* fin YaCalcV */

  /* incrementation des indices de boucle */
  IParamT = IParamT + NbTheta;
  IParamB = IParamB + NbBeta;
  IObs = IObs + Donnees->NbObsC.donnees[ICourbe];
  } /* fin boucle sur les courbes */


/* calcul de la variance de Y si necessaire */
if (Modele->Vari != VI)
  {
  if (Modele->YaCalcV == FAUX)
    {
    VarY = ResStat->Ajustes.VarY.donnees;
    nbele = ResStat->Ajustes.VarY.nbele;
    /* initialiser VarY car CVariance considera sinon
    les anciennes valeurs 
     pas la peine de reinitialiser ses derivees car elles
     sont deja initialisees a 0 et CVariance les divisera mais
     ca ne changera pas leur valeur */
    for (i=0; i<nbele; i++)
      {
      VarY[i]=(TDouble)1.0;
      }
    } /* fin de YaCalc FAUX */
  
    APPEL(CVariance(Modele->YaCalcV, Modele->Vari, CtxPuss->TypeSigma, 
                    ResNum->NbIter,
                    Donnees->NbObsT,
                    &(Donnees->NbRepet), 
                    &(Donnees->Poids), 
                    &(ResStat->Ajustes.Valf), 
                    &(Donnees->StatDon),
                    &(ResNum->Sigma), 
                    &(ResStat->Ajustes.VarY), 
                    &(ResStat->FctSensib.DVarYTheta),
                    &(ResStat->FctSensib.DVarYBeta),
                    &(ResNum->Scr) ));

    } /* fin du  Modele->Vari != VI */

/* passer des derivees par rapport au modele aux derivees par rapport
   aux parametres actifs */
  if (Theta->Act.nbele > 0)
    {
    APPEL(DModAAct(CTheta->Ctr, Donnees->NbObs,&(Donnees->NbObsC), ThetaCour, 
                 &(ResStat->FctSensib.DValf), 
                 &(Theta->Pass), CThetaE,  DEffT, &(FctSA->DValf)));
    }

  if (Modele->YaCalcV == VRAI)
    {
    APPEL(DModAAct(CTheta->Ctr, Donnees->NbObs, &(Donnees->NbObsC),ThetaCour, 
                   &(ResStat->FctSensib.DVarYTheta), 
                   &(Theta->Pass), CThetaE,  DEffT, &(FctSA->DVarYTheta)));
    }
  else
    {
    /* mettre les derivees a 0 */
    dvarytheta = FctSA->DVarYTheta.donnees;
    nbt = FctSA->DVarYTheta.nbcol;
    for (i=0 ; i< NbObs; i++)
      {
      for (j=0; j < nbt; j++)
        {
        dvarytheta[i][j] = (TDouble)ZERO;
        /* inutile de mettre les derivees des beta a zero
        car si YaCalcV=FAUX, il ne peut y avoir de beta */
        }
      }
    } /* fin du else */


  if(ResStat->FctSensib.DVarYBeta.nblig > 0)
    {
    APPEL(DModAAct(CBeta->Ctr, Donnees->NbObs, &(Donnees->NbObsC),BetaCour, 
                   &(ResStat->FctSensib.DVarYBeta), 
                   &(Beta->Pass), CBetaE,  DEffB, &(FctSA->DVarYBeta)));

    }

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CModErr                  |
| Role                  :  Traiter les erreurs     |
|  dans le calcul du modele:                       |
|   - ecrire les valeurs courantes                 |
|   - effectuer les modifications necessaires quand|
|   l'erreur est recuperable.                      |
| L'erreur est recuperable si:                     |
|  on n'est pas a la 1iere iteration,              |
|  on est a l'extremite d'un intervalle,           |
|  le nombre maximal d'erreurs tolerees n'est pas  |
|  atteint)                                        |
|  Dans ce cas:                                    |
|  - on modifie la direction de descente,          |
|  - on modifie les valeurs courantes              |
|  des parametres                                  |
| Parametres d'entree   :                          | 
|   ModErr: precise si l'erreur a eu lieu dans le  |
|     calcul du modele de l'esperance ou de v      |
|     = ERRMODF ou ERRMODV                         |
|   IPas: position dans l'intervalle d'etude       |
|    (1: debut, 2:extremite, 3: milieu)            |
|   NbIter: indice de l'iteration courante         |
|    (commence a 0)                                |
|   MaxErr: le nombre de reprises en cas d'erreur  |
|        tolerees                                  |
|   DirecC: valeur par laquelle il faut multiplier |
|    la direction de descente pour la corriger     |
|   ThetaAct, BetaAct: valeurs des parametres      |
|    actifs au debut de l'intervalle d'etude       |
|   InfoModErr: informations sur l'erreur          |
| Parametres d'e/s      :                          |
|   Encore: VRAI si l'erreur est recuperable       |
|   NbErr: nombre d'erreurs rencontrees            |
|   ThetaCour, BetaCour: valeurs courantes des     |
|     parametres actifs                            |
|   Direc: direction de descente                   |
| Parametres de sortie  :                          |
| Retour fonction       : OK ou code d'erreur      |
|   des fonctions appelees                         |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  ErrIp, MultVectVal       |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/


TShortInt CModErr(ModErr, IPas, NbIter, MaxErr, DirecC, ThetaAct, BetaAct, InfoModErr,
                  Encore, NbErr,
                  ThetaCour, BetaCour,  Direc )

/* arguments d'entree */
TShortInt ModErr, IPas;
TLongInt  NbIter;
TLongInt MaxErr;
TDouble  DirecC;
TVect *ThetaAct, *BetaAct;
TInfoModErr *InfoModErr;

/* arguments d'entree-sortie */
TLogic *Encore; /* indique si un autre essai peut etre tente */
TLongInt  *NbErr;
TVect *ThetaCour, *BetaCour;
TVect *Direc;


/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt i;
TChar Mess1[2], Mess2[5];
FILE *Imp;
TShortInt thetaactn, betaactn;

/* pointeur sur les fonctions du module appelees */
TShortInt ErrIp();

/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *thetaact, *thetacour, *betaact, *betacour, *direc;

/* Ecriture de la trace */
ECRTRACE("CModErr");


/* Attention: il s'agit ici d'impressions intermediaires car il ne faut pas,
  a chaque impression, incrementer le compteur de warnings
  Mais ces impressions doivent se faire sur l'unite d'impression des warnings
  ce qui explique la modification de SortImp suivante:
*/

  Imp = GNLControle.SortImp;
  GNLControle.SortImp = GNLControle.SortWE;

  *NbErr = *NbErr + 1;

/* impression d'un message et des valeurs courantes */
  if(GNLControle.SortImp != NULL)
    {
    APPEL(ErrIp(ModErr, IPas, NbIter, *NbErr, ThetaCour, BetaCour, InfoModErr));
    }

/* determiner si c'est dans le calcul de f ou de v */
  if (ModErr == ERRMODF)
    {
    sprintf(Mess1,"f");
    }
  else
    {
    sprintf(Mess1,"v");
    }

  /* TRAITEMENT DES CAS OU L'ERREUR  EST IRRECUPERABLE*/
  if ((NbIter == 0) && (IPas ==1))
    {
  /* MESSAGE ECRIT PAR NLERREUR: */
/*
    fprintf(GNLControle.SortErr,"CModErr:");
    fprintf(GNLControle.SortErr,
    " Le calcul des valeurs du modele de %s\
est impossible pour les valeurs initiales choisies\n ", Mess1);
*/
    NLERREUR((ModErr,1,"CModErr", ERR));
    }

  if((IPas == 1) || (IPas == 3))
  /* debut de l'intervalle ou milieu */
    {
    sprintf(Mess2,"%hd", IPas);
    /* MESSAGE ECRIT PAR NLWARNING: */
    /*
    fprintf(GNLControle.SortImp,"CModErr:");
    fprintf(GNLControle.SortImp,
      "Le calcul des valeurs du modele de %s est impossible au  pas %s \n",
      Mess1, Mess2);
    */
    NLWARNING((WARCM2,3,"CModErr",Mess1,Mess2, IMP));
    } /* fin de  IPas=1 ou 3 */
  
  if( (IPas == 2) && (*NbErr > MaxErr))
    {
    /* NbIter #0, on est a l'extremite de l'intervalle et tous les
       essais de reprise ont ete tentes */
    sprintf(Mess2,"%ld", MaxErr);
    /* MESSAGE ECRIT PAR NLWARNING:
    fprintf(GNLControle.SortImp,
    "Le nombre maximal d'erreurs tolerees %s  est atteint\n",
            Mess2);
    */
    NLWARNING((WARCM3,2,"CModErr",Mess2,IMP));
    *Encore = FAUX;
    } /* fin de  IPas=2 et NbErr >MaxErr */
  

  if( (IPas == 2) && (*NbErr <= MaxErr))
    {  
    /* TRAITEMENT DES CAS OU C'EST UN WARNING */
    /*   cas ou NbIter#0, on est a l'extremite de l'intervalle et 
    tous les essais de reprise n'ont pas ete tentes */
    *Encore = VRAI;
    sprintf(Mess1,"%ld", (*NbErr+1));
    /* MESSAGE ECRIT PAR NLWARNING:
    fprintf(GNLControle.SortImp,"On va essayer un %s i-eme essai\n", Mess1);
    */
    NLWARNING((WARCM4,2,"CModErr",Mess1, IMP));
    /* modification de la direction */
    /*      *Direc =*Direc * DirecC;  */
    APPEL(MultVectVal(Direc, DirecC, Direc));
    /* actualisation de la valeur des parametres */
    thetaact = ThetaAct->donnees;
    thetaactn = ThetaAct->nbele;
    thetacour = ThetaCour->donnees;
    direc = Direc->donnees;

    for(i = 0; i < thetaactn; i++)
      {
      thetacour[i] = thetaact[i] + direc[i];
      }

    betaact = BetaAct->donnees;
    betaactn = BetaAct->nbele;
    betacour = BetaCour->donnees;

    for(i = 0; i < betaactn; i++)
      {
      betacour[i] = betaact[i] + direc[i+thetaactn];
      }
    } /* fin IPas=2 et NbErr<=MaxErr */      

  /* retablir l'unite d'impression */
  GNLControle.SortImp=Imp;

  /* retour */
  return(OK);
  }


/*--------------- Identification fonction ----------
| Nom de la fonction    : CScr                     |
| Role                  : calculer la variance     |
|  residuelle                                      |
| Parametres d'entree   :                          |
|  NbObsT: nombre total de repetitions             |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Y1, Y2: statistiques sur les donnees            |
|  Poi: ponderations                               |
|    (peut etre NULL ou avoir 0 elements)          |
|  Valf: valeurs ajustees de l'esperance           |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  Scr: variance residuelle                        |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  CVariance                |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CScr(NbObsT, NbRepet, Y1, Y2,  Poi, Valf, Scr)

/* arguments d'entree */
TLongInt NbObsT;
TVectLong *NbRepet;
TVect *Y1, *Y2, *Poi,  *Valf;

/* arguments de sortie*/
TDouble *Scr;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, nbrepetn;
TDouble ValPoids;
TLogic YaPoids;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TLongInt *nbrepet;
TDouble *y1, *y2, *valf;

/* Ecriture de la trace */
ECRTRACE("CScr");

/* initialisations */
nbrepet = NbRepet->donnees;
y1 = Y1->donnees;
y2 = Y2->donnees;
valf = Valf->donnees;
nbrepetn = NbRepet->nbele;

*Scr=(TDouble)ZERO;
if(Poi == NULL || Poi->nbele < 0)
  {
  YaPoids=FAUX;
  ValPoids = (TDouble)1;
  }
else
  {
  YaPoids=VRAI;
  }



for(i = 0; i < nbrepetn; i++)
  {
  if(YaPoids == VRAI)
    {
    ValPoids = Poi->donnees[i];
    }
  *Scr = *Scr + (((TDouble)nbrepet[i] / ValPoids) *
                 (y2[i] + (valf[i] * valf[i])
                    - ((TDouble)2 * valf[i] * y1[i])));
  }
*Scr = *Scr / (TDouble)NbObsT;

/* retour */
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : CVariance                |
| Role                  : Ponderation et           |
|  multiplication par Sigma des sorties de calcv   |
|  ou d'un vecteur de 1                            |
|  Fonction appelee que si Vari # VI               |
| Parametres d'entree   :                          |
|  Vari: type de la variance                       |
|  TypeSigma: indique comment calculer Sigma       |
|  NbIter: indice courant de l'iteration           |
|   sert a distinguer le cas de la 1iere iteration |
|   dans le message et le code qui est renvoye     |
|   en cas de fonction v negative                  |
|  NbObsT: nombre total de repetitions             |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Poids: ponderations                             |
|    (peut avoir 0 element)                        |
|  Valf: valeurs ajustees de l'esperance           |
|  StatDon: statistiques sur les donnees           |
| Parametres d'e/s      :                          |
|  Sigma: une entree si deja calcule et necessaire |
|   une sortie si necessaire et non deja calcule   |
|  (necessaire si Vari inclut S ou egal CTE,       |
|   non calcule si TypeSigma=VARRESID)             |
|  ValV: valeurs de la fonction v                  |
|  DValVT: derivees de celles-ci par rapport aux   |
|   Theta                                          |
|  DValVB: derivees de celles-ci par rapport aux   |
|    Beta                                          |
| (s'il n'y a pas de fonction v, ValV est un       |
|   vecteur de NbObs "1", et les derivees sont des |
|   matrices de 0 element)                         |
| En entree, il s'agit des sorties de la fonction  |
| calcv, et en sortie, elles sont ponderees et     |
| multipliees par Sigma                            |
| Parametres de sortie  :                          |
|  Scr: variance residuelle. Celle-ci est          |
|   calculee, uniquement si on en a besoin, c.a.d: |
|   Vari inclut S ou egal CTE et TypeSigma=VARRESID|
| Retour fonction       : OK ou erreur             |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|   CScr                                           |
|   DivMatVect                                     |
|   DivVectVect                                    |
|   MultMatVal                                     |
|   MultVectVal                                    |
| Fonctions appelantes :  CModele                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CVariance(YaCalcV, Vari, TypeSigma, NbIter,
                    NbObsT, NbRepet, Poids, Valf, StatDon, 
                    Sigma, ValV, DValVT, DValVB, Scr)

/* arguments d'entree */
TLogic YaCalcV;
TShortInt Vari, TypeSigma;
TLongInt NbIter, NbObsT;
TVectLong *NbRepet;
TVect *Poids,  *Valf ;
TStatDon *StatDon;

/* arguments d'entree-sortie */
TDouble *Sigma;
TVect *ValV;
TMat *DValVT, *DValVB;

/* arguments de sortie*/
TDouble *Scr;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, valvn;
TChar Mess1[5];
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *valv;

/* Ecriture de la trace */
ECRTRACE("CVariance");


if (YaCalcV == VRAI)
  {
  /* verification de ValV: les valeurs doivent etre positives */
  valv = ValV->donnees;
  valvn = ValV->nbele;
  for(i=0; i<valvn; i++)
    {
    if (valv[i] <= (TDouble)ZERO)
      {
      sprintf(Mess1,"%hd", (i+1));
      /*
      printf("La %s-ieme valeur de la variance de Y est <=0\n", Mess1);
      return(ERRVAR);
      */
      if (NbIter >0)
        {
        NLERREUR((ERRVAR,2,"CVariance", Mess1, ERR));
        }
      else
        {
         NLERREUR((ERRVAR1,2,"CVariance", Mess1, ERR));
        }
      }
    }
  }
/* si YaCalcV=faux, les valv ont ete fixees a 1 */

    
/* Ponderation */
if (Poids->nbele > 0 )
  {
  APPEL(DivVectVect(ValV, Poids, ValV));
  APPEL(DivMatVect(DValVT, Poids, DValVT));

  if (DValVB->nblig > 0)
    {
    APPEL(DivMatVect(DValVB, Poids, DValVB));
    }
  }


if ( (Vari == CTE) || (Vari == SP) || (Vari == VST) || (Vari == VSB) || (Vari == VSTB))
  {
  /* cas ou on a besoin de Sigma */
  if (TypeSigma == VARRESID)
    {
    /* cas ou Sigma n'a pas ete calcule */
    APPEL(CScr(NbObsT, NbRepet, &(StatDon->Y1),  &(StatDon->Y2), ValV, Valf, Scr));
    *Sigma = *Scr;
    } /* fin du cas TypeSigma == VARRESID */

  /* Multiplication par Sigma */
  APPEL(MultVectVal(ValV, *Sigma, ValV));
  APPEL(MultMatVal(DValVT, *Sigma, DValVT));

  if (DValVB->nblig > 0)
    {
    APPEL(MultMatVal(DValVB, *Sigma, DValVB));
    }
  } /* fin du if sur Vari */

/* retour */
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : ErrIp                    |
| Role                  :  ecrire des informations |
|  quand il y a eu une erreur dans le calcul du    |
|  modele                                          |
|  Les valeurs courantes des parametres sont       |
|  imprimees en dimension "actifs"                 |
| Parametres d'entree   :                          |
|   ModErr: OK ou code d'erreur ERRMODF ou ERRMODV |
|   IPas: position dans l'intervalle d'etude       |
|    (1: debut, 2:extremite, 3: milieu)            |
|   NbIter: indice de l'iteration courante         |
|    (commence a 0)                                |
|   NbErr: le nombre d'erreurs deja decelees       |
|   ThetaCour, BetaCour: valeurs courantes des     |
|     parametres actifs                            |
|   InfoModErr: informations sur l'erreur          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK ou nombre maxi de     |
|   warnings atteint                               |
| Conditions d'appel    : GNLControle.SortIMP n'est|
|   pas NULL                                       |
| Remarque              :                          |
|   SortImp,et non pas SortWE, ne doit pas etre nul|
|   bien qu'il s'agisse de warnings, car,          |
|   pour ne pas incrementer                        |
|   le compteur de warnings a chaque impression,   |
|   les messages sont de type IMP (impression      |
|   intermediaire). Cependant, ils sont bien ecrits|
|   sur l'unite de warnings, car le programme      |
|   appelant a echange les unites d'erreur et      |
|   d'impression.                                  |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  EcrVect                  |
| Fonctions appelantes :  CModErr                  |
--------------------------------------------------*/



TShortInt ErrIp(ModErr, IPas, NbIter, NbErr, ThetaCour, BetaCour, InfoModErr)

/* arguments d'entree */
TShortInt ModErr, IPas;
TLongInt  NbIter, NbErr;
TVect *ThetaCour, *BetaCour;
TInfoModErr *InfoModErr;

/*--------------- Fin identification fonction ----*/
{
/* locals */
TChar Mess4[2];
TChar Mess0[15], Mess1[15], Mess2[15], Mess3[5];


/* Ecriture de la trace */
ECRTRACE("ErrIp");

/* IMPRESSION D'UN WARNING SIGNALANT L'ERREUR*/
/* determiner si c'est dans le calcul de f ou de v */
  if (ModErr == ERRMODF)
    {
    sprintf(Mess4,"f");
    }
  else
    {
    sprintf(Mess4,"v");
    }

  sprintf(Mess0,"%ld", NbErr);
  sprintf(Mess1,"%d", InfoModErr->CodeErr);
  sprintf(Mess2,"%d", IPas);
  sprintf(Mess3,"%ld", NbIter);

  /* MESSAGE ECRIT PAR NLWARNING:
  fprintf(GNLControle.SortImp,
"Une %s -ieme erreur est survenue lors du calcul des valeurs du modele de %s\n", 
Mess0, Mess4);
  fprintf(GNLControle.SortImp," de code %s au pas %s de l'iteration %s\n", 
Mess1, Mess2, Mess3);
*/
  NLWARNING((IMPERRIP1,6,"ErrIp",Mess0, Mess4, Mess1, Mess2, Mess3, IMP));

/* IMPRESSION DU LIEU DE L'ERREUR */
  switch(InfoModErr->LieuErr)
    {
    case F:
      sprintf(Mess1,"f");
      break;
    case V:
      sprintf(Mess1,"v");
      break;
    case DFDT:
      sprintf(Mess1,"df/dTheta");
      break;
    case DVDT:
      sprintf(Mess1,"dv/dTheta");
      break;
    case DVDB:
      sprintf(Mess1,"dv/dBeta");
      break;
    case A:
      if (ModErr == ERRMODF)
        {
        sprintf(Mess1,"auxf");
        InfoModErr->LieuErr = AUXF;
        }
      else
        {
        sprintf(Mess1,"auxv");
        InfoModErr->LieuErr=AUXV;
        }
      break;
    case DADT:
      sprintf(Mess1,"daux/dTheta");
      break;
    case DADB:
      sprintf(Mess1,"daux/dBeta");
      break;
    case SEDO:
      sprintf(Mess1,"odes");
      break;
    }

  /* MESSAGE ECRIT PAR NLWARNING:
  fprintf(GNLControle.SortImp,"L'erreur a eu lieu dans le calcul de %s \n", Mess1);
  fprintf(GNLControle.SortImp,"sur l'observation:\n");
*/
  NLWARNING((IMPERRIP2,2,"ErrIp",Mess1, IMP));


  fprintf(GNLControle.SortImp,"%s", InfoModErr->NomXErr);
  APPEL(EcrVect(GNLControle.SortImp,&(InfoModErr->XErr)));


/* IMPRESSION DES VALEURS COURANTES DES PARAMETRES */
  /* MESSAGE ECRIT PAR NLWARNING: 
  fprintf(GNLControle.SortImp,"Les valeurs courantes des parametres actifs Theta sont:\n");
*/
  NLWARNING((IMPERRIP3,2,"ErrIp","regression", IMP));

  APPEL(EcrVect(GNLControle.SortImp,ThetaCour));
  if(BetaCour->nbele>0)
    {
    /* MESSAGE ECRIT PAR NLWARNING: */
/*
    fprintf(GNLControle.SortImp,"Les valeurs courantes des parametres actifs Beta sont:\n");
*/
    NLWARNING((IMPERRIP3,2,"ErrIp","variance", IMP));
    APPEL(EcrVect(GNLControle.SortImp,BetaCour));
    }
return(OK);
}
