/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : Critere                   |
| Date                 : 1991                      |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : A. Bouvier                |
| Role                 : les programmes de calcul  |
|  des criteres                                    |
| Reference conception :                           |
| Lecteur              : G. Husser                 |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <errno.h>

#include "nlchoix.h"
#include "nlcodes.h"
#include "nltypes.h"
#include "nlglobal.h"
#include "nlmacros.h"
#include "errcodes.h"

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/
TDouble log();

/* fonctions des autres modules */
TShortInt CopyMat(), CScrPv(),
          GerMessage(),
          myinv(),
          MultVectMat(), MVect();
TDouble myln(double);

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : CArret                   |
| Role                  : Calculer le critere      |
|  d'arret                                         |
| Parametres d'entree   :                          |
|  Symm : indique si la matrice ValW est symetrique|
|  NbObsT: nombre total de repetitions             |
|  ValW: matrice ValW , de dimension               |
|       (NbTheta+NbBeta)*(NbTheta+NbBeta)          |
|       (nombres de parametres actifs)             |
|  ValR: = ValB*(ValZ-ValEta)                      |
|       vecteur de dimension (NbTheta+NbBeta)      |
| Parametres d'e/s      :                          |
|  Trav1, Trav2: vecteurs de travail de dimension  |
|          NbBeta+NbBeta                           |
|  BlocP:  matrice de travail de dimension         |
|          NbTheta*NbTheta                         |
|  BlocQ:  matrice de travail de dimension         |
|          NbBeta*NbTheta                          |
|  Bloc0:  matrice de travail de dimension         |
|          NbTheta*NbBeta                          |
|  BlocR:  matrice de travail de dimension         |
|          NbBeta*NbBeta                           |
| Quand Symm=SYMBLOC, les blocs contiennent en     |
|  sortie, les blocs de l'inverse de ValW          |
| Parametres de sortie  :                          |
|  CritArret: le critere d'arret                   |
|  ValWInv: matrice qui contient   l'inverse de    |
|     ValW (allouee avant l'appel)                 |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  InvMat, InvSb, myinv,   |
|                          MultVectMat, MVect      |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CArret(Symm, NbObsT, ValW, ValR, 
                 Trav1, Trav2, BlocP, BlocQ, Bloc0, BlocR,
                 CritArret, ValWInv)

/* arguments d'entree */
TShortInt Symm;
TLongInt NbObsT;
TMat *ValW;
TVect *ValR;

/* arguments d'entree-sortie */
TVect *Trav1, *Trav2; 
TMat  *BlocP, *BlocQ, *Bloc0, *BlocR; /* tableaux de travail */

/* arguments de sortie */
TDouble *CritArret;
TMat *ValWInv;

/*--------------- Fin identification fonction ----*/


{
/* locals */
TShortInt i, Code, valrn;

/* pointeurs sur les fonctions du module appelees */
TShortInt InvMat(), InvSb();

/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *valr, *trav1;

/* Ecriture de la trace */
ECRTRACE("CArret");


/* Inversion de ValW */
switch (Symm)
  {
  case SYMBLOC:
    APPEL(InvSb(ValW, Trav1, BlocP, BlocQ, Bloc0, BlocR,  ValWInv));
    break;
  case SYM:
    APPEL(CopyMat(ValW, ValWInv));
    Code = InvMat( ValWInv);
    if (Code != OK)
      {

      /* ACTIONS EFFECTUEES PAR LA MACRO NLERREUR */
/*
      fprintf(stderr,"CArret: la matrice ValW est non inversible \n");
      return(ERRCALC);
*/
      NLERREUR((ERRCALC,1,"CArret",ERR));
      }
    break;
  default:
    Code = myinv(ValW,ValWInv);
    if (Code != OK)
      {
      /* ACTIONS EFFECTUEES PAR LA MACRO NLERREUR */
/*
      fprintf(stderr,"CArret: la matrice ValW est non inversible \n");
      return(ERRCALC);
*/
      NLERREUR((ERRCALC,1,"CArret",ERR));
      }
  }

/* Division de ValR par NbObsT */
valrn = ValR->nbele;
valr = ValR->donnees;
trav1 = Trav1->donnees;

for(i = 0; i < valrn; i++)
  {
  trav1[i] =  valr[i] / (TDouble)NbObsT;
  }

/* calcul de: Transposee(Trav1)* ValWInv. 
   Le resultat est dans le vecteur Trav2, considere comme une matrice ligne */
APPEL(MultVectMat(Trav1, ValWInv, Trav2));

/* calcul de Trav2* Trav1: Trav1 est considere comme une matrice ligne et 
                           Trav2 comme une matrice colonne.
  Le resultat est le scalaire CritArret */
APPEL(MVect(Trav2, Trav1, CritArret));

/* retour*/
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : CLv                      |
| Role                  : calculer -2log(vraisemb) |
|  divise par le nombre total de repetitions       |
| Parametres d'entree   :                          |
|  NbObsT: nombre total de repetitions             |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Y1, Y2: statistiques sur les donnees            |
|  Valf: valeurs ajustees de l'esperance           |
|  VarY: variance de Y                             |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  Lv: 1/NbObsT * -2log(vraisemblance)             |    
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|    ATTENTION: appelle la fonction systeme        |
|  "log" dont l'argument doit etre un double       |
| et utilise la constante machine pi: M_PI         |
| Fonctions appelantes :  PussIter, NLAutres       |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt CLv( NbObsT,  NbRepet, Y1, Y2, Valf, VarY,
               Lv)

/* arguments d'entree */
TLongInt NbObsT;
TVectLong *NbRepet;
TVect *Y1, *Y2, *Valf, *VarY;

/* arguments de sortie*/
TDouble *Lv;

/*--------------- Fin identification fonction ----*/


{
/* locals */
TShortInt i, n;


/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *vary, *valf, *y1, *y2;
TLongInt *nbrepet;


/* Ecriture de la trace */
ECRTRACE("CLv");
n = NbRepet->nbele;
nbrepet = NbRepet->donnees;
vary = VarY->donnees;
valf = Valf->donnees;
y1 = Y1->donnees;
y2 = Y2->donnees;

*Lv=(TDouble)ZERO;

for (i = 0; i < n; i++)
  {
  *Lv = *Lv + (nbrepet[i] * (myln(vary[i])
    + (( (TDouble)1 / vary[i])
               * (y2[i] + (valf[i] * valf[i] )
               - ((TDouble)2 * valf[i] * y1[i])))));
  }

  *Lv = myln(((TDouble)2 * (TDouble)M_PI)) + (*Lv / (TDouble)NbObsT);

/* retour */
if (!finite(*Lv))
  {
    /* OTER ou REMETTRE       fprintf( GNLControle.SortWE,"LogVrais= %g\n", *Lv); */
      *Lv=(TDouble)ZERO;
  }

return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : CritSig                  |
| Role                  : renvoyer la valeur de    |
|  Sigma: fonction appelee quand le critere        |
|  statistique est egal a Sigma                    |
| Parametres d'entree   :                          |
|  Sigma                                           | 
| Parametres de sortie  :                          |
|  Crit=Sigma                                      |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt CritSig( Sigma,
                   Crit)  

/* arguments d'entree */
TDouble Sigma;

/* arguments de sortie*/
TDouble *Crit;

/*--------------- Fin identification fonction ----*/


{

/* Ecriture de la trace */
ECRTRACE("CritSig");

*Crit = Sigma;

/* retour */
return(OK);
}







/*--------------- Identification fonction ----------
| Nom de la fonction    : CScrNP                   |
| Role                  : calculer la variance     |
|  residuelle non ponderee                         |
| (Programme CScr sans les ponderations)           |
| Parametres d'entree   :                          |
|  NbObsT: nombre total de repetitions             |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Y1, Y2: statistiques sur les donnees            |
|  Valf: valeurs ajustees de l'esperance           |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  Scr: variance residuelle                        |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, NLAutres       |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CScrNP(NbObsT,  NbRepet, Y1, Y2, Valf,
                 Scr)

/* arguments d'entree */
TLongInt NbObsT;
TVectLong *NbRepet;
TVect *Y1, *Y2,  *Valf;

/* arguments de sortie*/
TDouble *Scr;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, n;

/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble  *valf, *y1, *y2;
TLongInt *nbrepet;

/* Ecriture de la trace */
ECRTRACE("CScrNP");

/* initialisations */
*Scr=(TDouble)ZERO;
n = NbRepet->nbele;
nbrepet = NbRepet->donnees;
valf = Valf->donnees;
y1 = Y1->donnees;
y2 = Y2->donnees;

for(i = 0; i < n; i++)
  {
  *Scr = *Scr + ((TDouble)nbrepet[i]  *
                 (y2[i] + (valf[i] * valf[i])
                    - ((TDouble)2 * valf[i] * y1[i])));
  }
*Scr = *Scr / (TDouble)NbObsT;
   

/* retour */
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : CScr2NP                  |
| Role                  : calculer la somme des    |
| carres dans le contexte d'estimation de la       |
| variance, divisee par le nombre total            |
|  de repetitions                                  |
| Version sans ponderation                         |
| Parametres d'entree   :                          |
|  NbObsT: nombre total de repetitions             |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Y1, Y2: statistiques sur les donnees            |
|  Valf: valeurs ajustees de l'esperance           |
|  VarY: variance de Y                             |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  Scr2: variance residuelle                       |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CScr2NP(NbObsT,  NbRepet, Y1, Y2, Valf, VarY, 
                  Scr2)

/* arguments d'entree */
TLongInt NbObsT;
TVectLong *NbRepet;
TVect *Y1, *Y2,  *Valf, *VarY;

/* arguments de sortie*/
TDouble *Scr2;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, n;
TDouble Valeur;

/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *vary, *valf, *y1, *y2;
TLongInt *nbrepet;


/* Ecriture de la trace */
ECRTRACE("CScr2NP");

/* initialisations */
n = NbRepet->nbele;
nbrepet = NbRepet->donnees;
vary = VarY->donnees;
valf = Valf->donnees;
y1 = Y1->donnees;
y2 = Y2->donnees;


*Scr2=(TDouble)ZERO;

for(i = 0; i < n; i++)
  {
  Valeur = (nbrepet[i]  * y2[i]) + 
       (valf[i] * valf[i])
       - ((TDouble)2 * nbrepet[i] * valf[i] * y1[i])
       - vary[i];
  *Scr2 = *Scr2 + (Valeur * Valeur);
/* Remarque: dans une version avec ponderation, il faudrait diviser 
  (Valeur * Valeur) par Poids[i] */
  }
*Scr2 = *Scr2 / (TDouble)NbObsT;
   

/* retour */
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : CScrPS2                  |
| Role                  : calculer la variance     |
|  residuelle ponderee par S2                      |
|  (programme CScr avec ponderations=S2)           |
| Parametres d'entree   :                          |
|  NbObsT: nombre total de repetitions             |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Y1, Y2: statistiques sur les donnees            |
|  Valf: valeurs ajustees de l'esperance           |
|  S2: somme des carres                            |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  Scr: variance residuelle                        |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CScrPS2(NbObsT,  NbRepet, Y1, Y2,  Valf, S2,
                  Scr)

/* arguments d'entree */
TLongInt NbObsT;
TVectLong *NbRepet;
TVect *Y1, *Y2,  *Valf, *S2;

/* arguments de sortie*/
TDouble *Scr;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, n;

/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *s2, *valf, *y1, *y2;
TLongInt *nbrepet;

/* Ecriture de la trace */
ECRTRACE("CScrPS2");

/* initialisations */
n = NbRepet->nbele;
nbrepet = NbRepet->donnees;
s2 = S2->donnees;
valf = Valf->donnees;
y1 = Y1->donnees;
y2 = Y2->donnees;

*Scr=(TDouble)ZERO;

for(i = 0; i < n; i++)
  {
  *Scr = *Scr + ((nbrepet[i] / s2[i]) *
                 (y2[i] + (valf[i] * valf[i])
                    - ((TDouble)2 * valf[i] * y1[i])));
  }
*Scr = *Scr / (TDouble)NbObsT;
   

/* retour */
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CScrPv                   |
| Role                  : calculer la variance     |
|  residuelle ponderee par VarY/Sigma              |
|  (programme CScr avec ponderations=VarY/Sigma)   |
| Parametres d'entree   :                          |
|  NbObsT: nombre total de repetitions             |
|  Sigma                                           |
|  NbRepet: nombre de repetitions de chaque        |
|          observation                             |
|  Y1, Y2: statistiques sur les donnees            |
|  Valf: valeurs ajustees de l'esperance           |
|  VarY: variance de Y                             |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  Scr: variance residuelle                        |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, NLAutres       |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CScrPv(NbObsT, Sigma, NbRepet, Y1, Y2,  Valf, VarY, 
                 Scr)

/* arguments d'entree */
TLongInt NbObsT;
TDouble Sigma;
TVectLong *NbRepet;
TVect *Y1, *Y2,  *Valf, *VarY;

/* arguments de sortie*/
TDouble *Scr;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i, n;

/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *vary, *valf, *y1, *y2;
TLongInt *nbrepet;

/* Ecriture de la trace */
ECRTRACE("CScrPv");

/* initialisations */
n = NbRepet->nbele;
nbrepet = NbRepet->donnees;
vary = VarY->donnees;
valf = Valf->donnees;
y1 = Y1->donnees;
y2 = Y2->donnees;


*Scr=(TDouble)ZERO;

for(i = 0; i < n; i++)
  {
  *Scr = *Scr + (((TDouble)nbrepet[i] / vary[i] ) *
                 (y2[i] + (valf[i] * valf[i])
                    - ((TDouble)2 * valf[i] * y1[i])));
  }
*Scr = (*Scr * Sigma) / (TDouble)NbObsT;
   

/* retour */
return(OK);
}

