/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : NLEtape                   |
| Date                 :  7Nov91 / 16:29:33        |
| Derniere mise a jour :                           |
| Concepteur           : A.Bouvier                 |
| Role                 : Gerer les calculs a une   |
|                        etape                     |
| Reference conception :                           |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/

/*--------------- INCLUDES ----------------------- */
#include <stdio.h>
#include <math.h>
#include <float.h>

#include "nlchoix.h"
#include "nltypes.h"
#include "nlcodes.h"
#include "nlmacros.h"
#include "errcodes.h"
#include "nlglobal.h"


/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/
short int calcv_(short int nbt, short int nbb, short int nbg, short int nbl, 
		 short int nbc, 
		 double *t, double *b, double *g,
		 double *f, double **df, double **x, double *v, 
		 double **dtv, double **dbv, short int *le, short int *ie);

/* fonctions des autres modules */
TShortInt  MajContP(),
          PussIter();

TShortInt CreerMat( TShortInt nblig, TShortInt nbcol, TMat *pmat);
TShortInt CreerMatC( TShortInt nblig, TShortInt nbcol, TMat *pmat);
TShortInt CreerVect( TShortInt nbele, TVect *pvect);
TShortInt CreerVectShort(TShortInt nbele, TVectShort *pvect);
TShortInt CopyVect(TVect *VectIn, TVect *VectOut);

void DetruVect(TVect *pvect);
void DetruMat(TMat *pmat);
void DetruVectShort(TVectShort *pvect);
TShortInt GerMessage( );

TShortInt DivVectVect(TVect *Vect1In, TVect *Vect2In, TVect *VectOut);
TShortInt MultVectVect2(TVect *VectIn, TVect *VectOut);


/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : AffectC                  |
| Role                  : Determiner les           |
|                         contraintes sur les      |
|                         parametres en dimension  |
|                         modele  pour l'etape     |
| Parametres d'entree   : Etape: indice de l'etape |
|                           (commence a 1)         |
|                         NbEtapes: nombre d'etapes|
|                         a effectuer              |
|                         TInit et BInit= pointeurs|
|                          sur les vecteurs des    |
|                          valeurs initiales des   |
|                          parametres pour l'etape |
|                         CTheta0 et CBeta0=       |
|                          pointeurs sur les       |
|                          contraintes initiales   |
| Parametres de sortie  : CTheta et CBeta=         |
|                          pointeurs sur les       |
|                          contraintes de l'etape  |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CreerVect, CreerVectShort |
| Fonctions appelantes : NLEtape                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt AffectC(Etape, NbEtapes, TInit, BInit, CTheta0, CBeta0, CTheta, CBeta)
  /* arguments d'entree */
  TShortInt Etape, NbEtapes;
  TVect *TInit, *BInit;
  TContr *CTheta0, *CBeta0;

  /* arguments de sortie */
  TContr *CTheta, *CBeta;

/*--------------- Fin identification fonction ----*/

  {

  /* mettre son nom dans la trace */
  ECRTRACE("AffectC");


  if ((Etape == 1) && (NbEtapes ==1))
    {
    /* On prend les contraintes fixees par l'utilisateur */
    *CTheta = *CTheta0;
    *CBeta = *CBeta0;
    }

  if((NbEtapes > 1) && (Etape %2  == 1))
    {
    /* Etape impaire: estimation des theta 
       Les contraintes sur les Theta sont celles fixees par l'utilisateur
       et on fixe les Beta a leur valeur initiale pour l'etape */
    *CTheta = *CTheta0;
    CBeta->EgalN = *BInit;
    CBeta->Ctr = VRAI;
    APPEL(CreerVect( (TShortInt)0, &(CBeta->BInf) ));
    APPEL(CreerVect( (TShortInt)0, &(CBeta->BSup) ));
    APPEL(CreerVectShort( (TShortInt)0, &(CBeta->EgalP) ));
    }

  if((Etape >= 2) && (Etape %2  == 0))
    {
    /* Etape paire: estimation des beta 
       Les contraintes sur les Beta sont celles fixees par l'utilisateur
       et on fixe les Theta a leur valeur initiale pour l'etape  */

    *CBeta = *CBeta0;
    CTheta->EgalN = *TInit;
    CTheta->Ctr = VRAI;
    APPEL(CreerVect( (TShortInt)0, &(CTheta->BInf) ));
    APPEL(CreerVect( (TShortInt)0, &(CTheta->BSup) ));
    APPEL(CreerVectShort( (TShortInt)0, &(CTheta->EgalP) ));
    }

  /* retour */
  return(OK);
  }

/*--------------- Identification fonction ----------
| Nom de la fonction    : Init3MCPT                |
| Role                  : Appelee dans le cas ou   |
|  on est a une etape>1 et impaire, et Estim=MCPT, |
|  cette fonction calcule v pour les precedentes   |
|  valeurs estimees des parametres, en pondere les |
|  poids, et supprime le calcul de v pour l'etape  |
| Parametres d'entree   :                          |
|  
| Parametres d'e/s      :                          |

| Parametres de sortie  :                          |
| Retour fonction       :
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : calcv_, CreerVect,        |
|                  CreerMat, DetruMat,             |
|                  MultVectVect2                   |
| Fonctions appelantes : NLEtape                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt Init3MCPT(NbTheta, NbBeta,ThetaPred,  BetaPred, ValfPred, DValfPred,
                       Modele, Donnees,
                       YaCalcVSv, YaPoidsSv, Valv, InfoModErr)

  /* arguments d'entree */
  TShortInt NbTheta, NbBeta;
        /* valeurs estimees aux etapes -1 et -2 */
  TVect *ThetaPred, *BetaPred;
  TVect *ValfPred;
  TMat *DValfPred;
  TModele *Modele;
  TDonnees *Donnees;


  /* arguments de sortie */
  /* pour retablir en fin d'etape */
  TLogic *YaCalcVSv, *YaPoidsSv;
  TVect *Valv ; 
  /* en cas d'erreur dans le calcul de v */
  TInfoModErr *InfoModErr;

/*--------------- Fin identification fonction ----*/

  {
  /* locals */
TShortInt NbObs, NbCourbe, ICourbe, IParamT, IParamB, NbVarEx;
TShortInt IObs, IndErr, i;
TMat  DValvT, DValvB;

/* mettre son nom dans la trace */
ECRTRACE("Init3MCPT");

/* allouer les sorties */
APPEL(CreerVect(Donnees->NbObs, Valv));


/* initialisations */
NbObs = Donnees->NbObs;
NbCourbe = Donnees->NbCourbe;
NbVarEx = Donnees->XObs.nbcol;


/* calcul de v */

if(Modele->YaCalcV == VRAI)
  {
  
  /* initialisation de la boucle sur les courbes */
  IParamT = 0;
  IParamB = 0;
  IObs = 0;

  /* pour contenir les derivees */
  APPEL(CreerMat(Donnees->NbObs, ThetaPred->nbele, &DValvT));
  APPEL(CreerMat(Donnees->NbObs, BetaPred->nbele, &DValvB));

  for (ICourbe = 0 ; ICourbe < NbCourbe; ICourbe++)
    {
    InfoModErr->CodeErr =
           calcv_(NbTheta, NbBeta, Modele->GamV.nbele, 
                 Donnees->NbObsC.donnees[ICourbe],
                 NbVarEx,
                 &(ThetaPred->donnees[IParamT]), 
                 &(BetaPred->donnees[IParamB]),
                 &(Modele->GamV.donnees[0]),
                 &(ValfPred->donnees[IObs]),
                 &(DValfPred->donnees[IObs]),
                 &(Donnees->XObs.donnees[IObs]),
                 &(Valv->donnees[IObs]),
                 &(DValvT.donnees[IObs]),
                 &(DValvB.donnees[IObs]),
                 &(InfoModErr->LieuErr), &IndErr);

    if (InfoModErr->CodeErr != OK)
      {
      /* Une erreur est decelee dans le calcul de v ou dv */
      /* Finir de remplir InfoModErr */
      APPEL(CreerVect(NbVarEx, &(InfoModErr->XErr)));
      for(i = 0; i < NbVarEx; i++)
        {
        InfoModErr->XErr.donnees[i] = Donnees->XObs.donnees[IObs+IndErr-1][i];
        }
      InfoModErr->NomXErr = Donnees->NomObs.donnees[IObs+IndErr-1];
      /* MESSAGE ECRIT PAR NLERREUR: */
      /* on ne peut calculer v */
      NLERREUR((ERRV3,1,"Init3MCPT", ERR));
      } /* fin CodeErr != OK */

    /* incrementation des indices de boucle */
    IParamT = IParamT + NbTheta;
    IParamB = IParamB + NbBeta;
    IObs = IObs + Donnees->NbObsC.donnees[ICourbe];
    } /* fin boucle sur les courbes */

  /* desallocation des derivees */
  DetruMat(&DValvT);
  DetruMat(&DValvB);

  } /* fin YaCalcV */
else
  {
  /* Valv= un vecteur de 1 */
  for (i=0; i< Valv->nbele; i++)
    {
    Valv->donnees[i]=(TDouble)1;
    }
  } /* fin du cas ou pas de YaCalcV */

/* calcul des poids */

if (Donnees->Poids.nbele >0)
  {
  *YaPoidsSv=VRAI;
  APPEL(MultVectVect2(Valv, &(Donnees->Poids)));
  }
else
  {
  *YaPoidsSv=FAUX;
  APPEL(CreerVect(Donnees->NbObs, &(Donnees->Poids)));
  APPEL(CopyVect(Valv, &(Donnees->Poids)));
  }


/* sauvegarde de YaCalcV */
*YaCalcVSv = Modele->YaCalcV;
Modele->YaCalcV=FAUX;
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : InitResNum               |
| Role                  : Initialiser la structure |
|  ResNum par des valeurs manquantes               |
| Parametres d'entree   :                          |
|  Sigma: valeur de Sigma initialisee dans CtxPuss |
| Parametres d'e/s      :                          |
|  ResNum: la structure qui contient les resultats |
|  numeriques pour une etape                       |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Fonctions appelantes : NLEtape                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt InitResNum(Sigma, ResNum)

  /* argu d'entree */
  TDouble Sigma;

  /* arguments d'entree-sortie */
  TResNum *ResNum;

/*--------------- Fin identification fonction ----*/

  {
  /* locals */
  TLongInt n=0;


  /* mettre son nom dans la trace */
  ECRTRACE("InitResNum");

  ResNum->CodePuss = NONFAIT;
  ResNum->CritArret= mysignaling_nan(n);
  ResNum->CritStat= mysignaling_nan(n);
  ResNum->Lambda= mysignaling_nan(n);
  ResNum->Log= mysignaling_nan(n);
  ResNum->LogFamille= mysignaling_nan(n);
  ResNum->Deviance= mysignaling_nan(n);
  APPEL(CreerVect(0, &(ResNum->ResidusDev)));
  ResNum->NbIter= 0;
  ResNum->Norme= mysignaling_nan(n);
  ResNum->Scr= mysignaling_nan(n);
  ResNum->ScrNP= mysignaling_nan(n);
  ResNum->Sigma = Sigma;
  ResNum->InfoModErr.CodeErr = OK;
  APPEL(CreerVect(0, &(ResNum->InfoModErr.XErr)));

  return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : InitResStat              |
| Role                  : Initialiser la structure |
|  ResStat par des valeurs manquantes              |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
|  ResStat: la structure qui contient les resultats|
|  statistiques pour une etape                     |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CreerMat, CreerMatC,      |
|                        CreerVect                 |
| Fonctions appelantes : NLEtape                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt InitResStat(ResStat)

  /* arguments d'entree - sortie */
  TResStat *ResStat;

/*--------------- Fin identification fonction ----*/

  {

  /* mettre son nom dans la trace */
  ECRTRACE("InitResStat");

  /* allocation de structures a 0 elements */
  APPEL(CreerMat((TShortInt)0, (TShortInt)0, &(ResStat->AsVar)));
  APPEL(CreerVect((TShortInt)0, &(ResStat->Ajustes.Valf)));
  APPEL(CreerVect((TShortInt)0, &(ResStat->Ajustes.VarY)));
  APPEL(CreerMat((TShortInt)0, (TShortInt)0, &(ResStat->BVarZBP)));
  APPEL(CreerMat((TShortInt)0, (TShortInt)0, &(ResStat->Corr)));
  APPEL(CreerMat((TShortInt)0, (TShortInt)0, &(ResStat->EquN.ValB)));
  APPEL(CreerVect((TShortInt)0, &(ResStat->EquN.ValEta)));
  APPEL(CreerMat((TShortInt)0, (TShortInt)0, &(ResStat->EquN.ValD)));
  APPEL(CreerMatC((TShortInt)0, (TShortInt)0, &(ResStat->FctSensib.DValf)));
  APPEL(CreerMatC((TShortInt)0, (TShortInt)0, &(ResStat->FctSensib.DVarYTheta)));
  APPEL(CreerMatC((TShortInt)0, (TShortInt)0, &(ResStat->FctSensib.DVarYBeta)));
  APPEL(CreerVect((TShortInt)0, &(ResStat->Mu3)));
  APPEL(CreerVect((TShortInt)0, &(ResStat->Mu4)));
  APPEL(CreerVect((TShortInt)0, &(ResStat->Residus)));
  APPEL(CreerVect((TShortInt)0, &(ResStat->ResidusR)));
  APPEL(CreerMat((TShortInt)0, (TShortInt)0, &(ResStat->ValW)));
  APPEL(CreerVect((TShortInt)0, &(ResStat->ValZ)));
  ResStat->Sedo.FSedo = NULL;
  ResStat->Sedo.DFSedo = NULL;

return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : Maj3MCPT                 |
| Role                  : Appelee dans le cas ou   |
|  on est a une etape>1 et impaire, et Estim=MCPT, |
|  cette fonction retablit les poids et YaCalcV    |
|  modifies par Init3MCPT                          |
| Parametres d'entree   :                          |
|  
| Parametres d'e/s      :                          |

| Parametres de sortie  :                          |
| Retour fonction       :
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : DivVectVect, DetruVect    |
| Fonctions appelantes : NLEtape                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt Maj3MCPT(YaCalcVSv, YaPoidsSv, YaCalcV, Poids, Valv)

  /* arguments d'entree */
  TLogic YaCalcVSv, YaPoidsSv;

  /* arguments de e/sortie */
  TLogic *YaCalcV;
  TVect *Poids, *Valv;

/*--------------- Fin identification fonction ----*/

{

/* mettre son nom dans la trace */
ECRTRACE("Maj3MCPT");

if (YaPoidsSv == VRAI)
  {
  if (*YaCalcV == VRAI)
    {
    /* retablir les poids : on les avait multiplies par v */
    APPEL(DivVectVect(Poids, Valv, Poids));
    }
  } /* fin du cas ou il y a des poids */
else
  {
  /* on desalloue les poids artificiellement crees */
  DetruVect(Poids);
  }

/* Desallouer Valv */
DetruVect(Valv);

/* Retablir YaCalcV */
*YaCalcV= YaCalcVSv;

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : NLEtape                  |
| Role                  : Gerer les calculs a une  |
|                         etape                    |
| Parametres d'entree   :                          |
|  Etape= indice de l'etape (commence a 1)         |
|  Donnees, Modele, CtxPuss, CtxNum = pointeurs sur|
|   respectivement les donnees, le modele et les   |
|   contextes du processus d'estimation et         |
|   numerique                                      |
|  ThetaPred1, ThetaPred2, BetaPred1, BetaPred2 =  |
|   pointeurs sur, respectivement, les valeurs     |
|   estimees des parametres Theta a l'etape        |
|   precedente et a celle d'avant, et des Beta     |
|  Valf1: valeurs ajustees de f a l'etape          |
|   precedente (ne sert que aux etapes >3 et       |
|   impaires et que si Estim est MCPT)             |
|  DValf1: derivees de f par rapport aux Theta     |
|   calculees a l'etape precedente (ne sert que    |
|   aux etapes >3 et impaires et que si Estim est  |
|   MCPT)                                          |
| Parametres de sortie  :                          |
|  (Toutes les structures de sortie concernent les |
|   valeurs pour l'etape)                          |
|  Theta, Beta= pointeurs sur les parametres       |
|  CTheta, CBeta= pointeurs sur les contraintes    |
|   sur les parametres                             |
|  CThetaE, CBetaE= pointeurs sur les contraintes  |
|   sur les parametres en dimension differents     |
|  ResNum= pointeur sur les resultats numeriques   |
|  ResStat= pointeur sur les resultats statistiques|
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                         AffectC                  |
|                         Init3MCPT                |
|                         InitResStat              |
|                         InitResNum               |
|                         MajContP                 |
|                         Maj3MCPT                 |
|                         PCValZ                   |
|                         PussIter                 |
| Fonctions appelantes :  NL                       |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt NLEtape(Etape, Donnees, Modele, CtxPuss, CtxNum,
                  ThetaPred1, ThetaPred2, BetaPred1, BetaPred2,
                  Valf1, DValf1,
                  Theta, Beta, CTheta, CBeta, CThetaE, CBetaE, ResNum, ResStat,
                  NbItSv, ItNum, ItStat, ItTheta, ItBeta, ItDirec, ItOmega)


  /* arguments d'entree */
  TShortInt Etape;
  TDonnees *Donnees;
  TModele *Modele;
  TCtxPuss *CtxPuss;
  TCtxNum *CtxNum;
        /* valeurs estimees aux etapes -1 et -2 */
  TVect *ThetaPred1, *ThetaPred2, *BetaPred1, *BetaPred2;
  TVect *Valf1;
  TMat *DValf1;

  /* arguments de sortie */
  TParam  *Theta, *Beta;
  TContr *CTheta, *CBeta;
  TContr *CThetaE, *CBetaE;
  TResNum *ResNum;
  TResStat *ResStat;


  /* les resultats des iterations */
  TShortInt *NbItSv;
  TResNum *ItNum;
  TResStat *ItStat;
  TVect *ItTheta, *ItBeta, *ItDirec;
  TDouble *ItOmega;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TChar Message[5];
TShortInt Code, ItSv, dim;


/* pour l'appel de Init3MCPT */
TLogic YaCalcVSv, YaPoidsSv;
TVect Valv;

  /* mettre son nom dans la trace */
  ECRTRACE("NLEtape");

 
  /* Message de debut d'etape */
  if (GNLControle.SortImp != NULL)
    {
    sprintf(Message, "%hd", Etape);
    NLWARNING((IMPDEBUT,2," ",Message,IMP));
    }


  /* initialiser le code d'execution */
  ResNum->CodePuss = NONFAIT;

  /* Initialiser la structure ResStat */
  APPEL(InitResStat(ResStat));

  /* Calcul des statistiques exhaustives */
  /* allocation */
  dim = (Donnees->NbObs) * (CtxNum->NbZ) ;
  APPEL(CreerVect(dim, &(ResStat->ValZ)));
  APPEL( CtxNum->PCValZ (&(Donnees->StatDon.Y1 ),&(Donnees->StatDon.Y2),
                  &(ResStat->ValZ)));

  /* Initialisation de Theta, Beta: leurs noms et leurs valeurs initiales */
  Theta->Noms = Modele->NomTheta;
  Beta->Noms = Modele->NomBeta;


  if(Etape  == 1)
    {
    Theta->Init = CtxPuss->Theta0;
    Beta->Init = CtxPuss->Beta0;
    }
  if(Etape == 2)
    {
    Theta->Init = *ThetaPred1;
    Beta->Init = CtxPuss->Beta0;
    }

  if ((Etape > 1) && (Etape % 2  == 1))
  /* Etape impaire: estimation des theta */
    {
    Theta->Init = *ThetaPred2;
    Beta->Init = *BetaPred1;
    if (CtxNum->Estim == MCPT)
      {
      /* cas particulier: aux etapes impaires >1, quand Estim=MCPT,
      il faut que v soit calculee avec toujours les memes valeurs
      des Theta et Beta: celles estimees a l'etape precedente.
      On calcule une fois pour toutes les valeurs de v,
      on les affecte aux poids (poids=poids*v) et on fait
      comme si il n'y avait plus de fonction v (YaCalcV=F) */
      APPEL(Init3MCPT(Theta->Noms.nbele, Beta->Noms.nbele,
                       ThetaPred2,  BetaPred1, Valf1, DValf1,
                       Modele,  Donnees,
                       &YaCalcVSv, &YaPoidsSv, &Valv, 
                       &(ResNum->InfoModErr)));
      } /* fin du cas particulier */
    }
  if((Etape > 2) && (Etape % 2  == 0))
  /* Etape paire: estimation des beta */
    {
    Theta->Init = *ThetaPred1;
    Beta->Init = *BetaPred2;
    }


 /* Initialisation des contraintes sur les parametres en dimension
    modele de l'etape: CTheta et CBeta */
  APPEL(AffectC(Etape, CtxPuss->NbEtapes, &(Theta->Init), &(Beta->Init),
          &(Modele->CTheta0),&(Modele->CBeta0),CTheta,CBeta));


  /* mise a jour des contraintes sur les parametres en dimension
     differents  et du nombre de parametres */
  APPEL( MajContP( CTheta, Theta,  CThetaE));
  APPEL( MajContP( CBeta, Beta,  CBetaE));
  /* Initialisation des sorties ResNum */
  APPEL(InitResNum(CtxPuss->Sigma, ResNum));


   /* Determination de ItSv:
            ItSv=VRAI si on veut garder les resultats par iteration de l'etape */
  ItSv=GNLControle.EtapeIt[Etape-1];

  /* Processus iteratif */
  Code = PussIter(ItSv,
                 Donnees, Modele, CtxPuss, CtxNum, CTheta, CBeta,
                 CThetaE, CBetaE,
                 Theta, Beta,
                 ResStat, ResNum,
                 NbItSv, ItNum, ItStat, ItTheta, ItBeta, ItDirec, ItOmega);


  if ((Etape > 1) && (Etape % 2  == 1))
    {
    if (CtxNum->Estim == MCPT)
      {
      /* Retablir les valeurs origine */
      APPEL(Maj3MCPT(YaCalcVSv, YaPoidsSv, 
                     &(Modele->YaCalcV), &(Donnees->Poids), &Valv));
      }
    }

  if ((Code == NONFAIT) || (Code == ERRTRACE) || (Code == ERRVINIT) ||
      (Code == ERRMODF) || (Code == ERRMODV) || (Code == ERRVAR1)   || 
      (Code == ERRTYPES) || (Code == ERRSYS) || 
      ((ResNum->NbIter == 0) && (Code !=OK)))
    {
    /* cas d'une erreur qui bloque la suite */
    CtxPuss->NbEtapes = Etape;
    ResNum->CodePuss = Code;
    return(Code);
    } 


  /* Message de fin d'etape */
  if (GNLControle.SortImp != NULL)
    {
    NLWARNING((IMPFIN,2," ",Message,IMP));
    }

  /* Retour */
  return(OK);
}

