/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : NLExhau                   |
| Date                 :  7Nov91 / 16:31:03        |
| Derniere mise a jour :                           |
| Concepteur           : A. Bouvier                |
| Role                 : Programmes de calcul des  |
|  statistiques exhaustives Z, et de leur variance |
| Reference conception :                           |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <stdio.h>

#include "nlchoix.h"
#include "nltypes.h"
#include "nlcodes.h"
#include "nlmacros.h"
#include "errcodes.h"
#include "nlglobal.h"

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/
TShortInt GerMessage( );
TShortInt CMu( TLongInt NbObsT, 
	       TVectLong *NbRepet, 
	       TVect *PoidsT, TVect *ValY, TVect *Valf, TVect *VarY,
              TShortInt *TypeMu, TVect *Residus,
              TVect *Mu3, TVect *Mu4);

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : CZmco                    |
| Role                  : Affecter des             |
|  valeurs aux statistiques  exhaustives,          |
|  quand leur nombre est egal a 1                  |
| Parametres d'entree   :                          |
|  Y1 et Y2=pointeurs sur les composants Y1 et Y2  |
|  de StatDon (Y2 est indique uniquement pour      |
|  analogie avec CZmv)                             |
| Parametres de sortie  :                          |
|  ValZ =pointeur sur les statistiques exhaustives |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : NLEtape (via le pointeur  |
|  de fonction PCValZ)                             |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt CZmco(Y1,Y2,ValZ)

  /* arguments d'entree */
  TVect *Y1,*Y2;
  /* ici, Y2 ne sert pas, mais on l'indique pour que la liste
     des arguments soit identique a CZmv */
  /* argument de sortie */
  TVect *ValZ;


/*--------------- Fin identification fonction ----*/

  {
  /* locals */
  TShortInt i; /* indice de boucle */
  /* pointeurs pour ameliorer la performance */
  TDouble *valz, *y1;

  /* mettre son nom dans la trace */
  ECRTRACE("CZmco");

  /* Affectation des pointeurs */
  valz = ValZ->donnees;
  y1 = Y1->donnees;

  /* Affectation des valeurs */
  /* les valeurs de la  statistique */
  for(i = 0; i < Y1->nbele; i++)
    {
    valz[i] = y1[i];
    }

  /* Retour */
  return(OK);
  }

/*--------------- Identification fonction ----------
| Nom de la fonction    : CZmv                     |
| Role                  : Affecter des             |
|  valeurs aux statistiques  exhaustives,          |
|  quand leur nombre est egal a 2                  |
| Parametres d'entree   :                          |
|  Y1 et Y2=pointeurs sur les composants Y1 et Y2  |
|  de StatDon                                      |
| Parametres de sortie  :                          |
|  ValZ =pointeur sur les statistiques exhaustives |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes : NLEtape (via le pointeur  |
|  de fonction PCValZ)                             |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/



TShortInt CZmv(Y1,Y2,ValZ)

  /* arguments d'entree */
  TVect *Y1,*Y2;
  /* argument de sortie */
  TVect *ValZ;


/*--------------- Fin identification fonction ----*/
  {
  /* locals */
  TShortInt i, NbObs;
  /* pointeurs pour ameliorer la performance */
  TDouble *valz, *y1, *y2;

  /* mettre son nom dans la trace */
  ECRTRACE("CZmv");

  /* Affectation des pointeurs */
  valz = ValZ->donnees;
  y1 = Y1->donnees;
  y2 = Y2->donnees;

  /* Affectation des valeurs */
  NbObs = Y1->nbele;
  for(i = 0; i < NbObs; i++)
    {
    valz[i] = y1[i];
    valz[i+NbObs] = y2[i];
    }

  /* Retour */
  return(OK);
  }



/*--------------- Identification fonction ----------
| Nom de la fonction    : CVarZmco                 |
| Role                  : Affecter des             |
|  valeurs a la variance de Z                      |
|  quand Estim=MCPT, MCOT, MCMT,QVT, MCPVIT        |
|  soit quand NbZ=1                                |
| Parametres d'entree   :                          |
|  NbObsT: nombre d'observations en comptant les   |
|   repetitions                                    |
|   argument ignore                                |
|  NbRepet: vecteur des repetitions                |
|  PoidsT: poids sur chacune des repetitions       |
|   argument ignore                                |
|  ValY: valeurs observees de la reponse f         |
|   argument ignore                                |
|  Ajustes: structure qui contient:                |
|     Valf: valeurs ajustees de f                  |
|      argument ignore                             |
|     VarY: variance de Y ajustee                  |
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
|   argument ignore                                |
|  Residus : les residus non reduits               |
|   argument ignore                                |
|  Mu3 : moments d'ordre 3                         |
|   argument ignore                                |
|  Mu4 : moments d'ordre 4                         |
|   argument ignore                                |
| Parametres de sortie  :                          |
|  VarZ: matrice de la variance de Z               |
|       allouee avant l'appel                      |
|  Code = OK                                       |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes :  CBVarZBP (via le pointeur|
|  de fonction PCValZ)                             |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/



TShortInt CVarZmco( NbObsT, NbRepet, PoidsT, ValY, Ajustes,
              TypeMu, Residus, Mu3, Mu4,
              VarZ, Code)

/* arguments d'entree */
TLongInt NbObsT;
TVectLong *NbRepet;
TVect *PoidsT, *ValY;
TAjustes *Ajustes;

/* arguments d'entree-sortie */
TShortInt *TypeMu;
TVect *Residus;
TVect *Mu3, *Mu4;

/* arguments de sortie*/
TMat *VarZ; /* dimension NbObs*NbZ, NbObs*NbZ  */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/
  {
  /* locals */
  TShortInt NbObs, i, j;

/* pointeurs sur des elements de structure pour ameliorer la performance */
  TLongInt *nbrepet;
  TDouble **varz;
  TDouble *vary;

  /* mettre son nom dans la trace */
  ECRTRACE("CVarZmco");

  *Code = OK;
  NbObs = NbRepet->nbele;

  /* affectation de pointeurs pour la performance */
  varz = VarZ->donnees;
  vary = Ajustes->VarY.donnees;
  nbrepet = NbRepet->donnees;

  for (i=0; i< NbObs; i++)
    {
    /* initialisation a zero */
    for(j=0; j<NbObs; j++)
      {
      varz[i][j] = (TDouble)ZERO;
      }

    varz[i][i] = vary[i] / nbrepet[i];
    }
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : CVarZmv                  |
| Role                  : Creer et affecter des    |
|  valeurs a la variance de Z                      |
|  quand Estim=MVTB, MCMTB, QVTB, MVT,MCOB, MCMB   |
|  soit quand NbZ=2                                |
| Parametres d'entree   :                          |
|  NbObsT: nombre d'observations en comptant les   |
|   repetitions                                    |
|  NbRepet: vecteur des repetitions                |
|  PoidsT: poids sur chacune des repetitions       |
|  ValY: valeurs observees de la reponse f         |
|  Ajustes: structure qui contient:                |
|     Valf: valeurs ajustees de f                  |
|     VarY: variance de Y ajustee                  |
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
|  Residus : les residus non reduits               |
|  ceux-ci sont calcules  uniquement si necessaires|
|  c'est-a-dire si TypeMu=MURES ou MURESREPET      |
|  C'est une entree car ils peuvent avoir deja ete |
|  calcules: dans ce cas,                          |
|      GNLControle.Voulu.Residus =VRAI et,         |
|      Residus en entree contient les residus      |
|  Si non deja calcules:                           |
|      GNLControle.Voulu.Residus=FAUX et,          |
|      Residus est ignore en entree, et, en        |
|      sortie contient les residus                 |
|  Mu3 : moments d'ordre 3                         |
|  Mu4 : moments d'ordre 4                         |
|  C'est une entree car ils peuvent avoir deja ete |
|  calcules: dans ce cas,                          |
|      GNLControle.Voulu.Mu=VRAI et,               |
|      Mu3 et Mu4 en entree contiennent les moments|
|  Si non deja calcules:                           |
|      GNLControle.Voulu.Mu=FAUX et,               |
|      Mu3 et Mu4 sont ignores en entree, et, en   |
|      sortie contiennent les moments              |
| Parametres de sortie  :                          |
|  VarZ: matrice de la variance de Z               |
|       allouee avant l'appel                      |
|  Code = OK ou WARVARZ1                           |
| Effets de bord    :                              |
|   - Si  GNLControle.Voulu.Residus =VRAI          |
|      Residus en entree contient les residus      |
|      sinon, Residus est ignore en entree, et, en |
|      sortie, il  contient les residus            |
|   - Si  GNLControle.Voulu.Mu =VRAI               |
|      Mu3, Mu4 en entree contiennent les moments  |
|      sinon, ils sont ignores en entree, et, en   |
|      sortie, il  contient les moments            |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CMu                       |
| Fonctions appelantes : CBVarZBP (via le pointeur |
|  de fonction PCValZ), CCovNu2                    |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/



TShortInt CVarZmv( NbObsT, NbRepet, PoidsT, ValY, Ajustes,
              TypeMu, Residus, Mu3, Mu4,
              VarZ, Code)

/* arguments d'entree */
TLongInt NbObsT;
TVectLong *NbRepet;
TVect *PoidsT, *ValY;
TAjustes *Ajustes;

/* arguments d'entree-sortie */
TShortInt *TypeMu;
TVect *Residus;
TVect *Mu3, *Mu4;

/* arguments de sortie*/
TMat *VarZ; /* dimension NbObs*NbZ, NbObs*NbZ  */
TShortInt *Code;  

/*--------------- Fin identification fonction ----*/
  {
  /* locals */
  TShortInt NbObs, Dim, i, j, ii;
  TChar Mess1[5], Mess2[50];


/* pointeurs sur des elements de structure pour ameliorer la performance */
  TLongInt *nbrepet;
  TDouble **varz;
  TDouble *vary, *valf, *mu3, *mu4;

  /* mettre son nom dans la trace */
  ECRTRACE("CVarZmv");

  *Code = OK;
  NbObs = NbRepet->nbele;
  Dim = NbObs * 2;

  /* calculer les Mu */
  if (GNLControle.Voulu.Mu==FAUX)
    {
    /* ils n'ont pas encore ete calcules */
    APPEL(CMu( NbObsT, NbRepet, PoidsT, ValY, &(Ajustes->Valf), &(Ajustes->VarY),
              TypeMu, Residus,
              Mu3, Mu4));
    }

  /* affectation des pointeurs pour ameliorer la performance */
  varz = VarZ->donnees;
  vary = Ajustes->VarY.donnees;
  valf = Ajustes->Valf.donnees;
  nbrepet = NbRepet->donnees;
  mu3 = Mu3->donnees;
  mu4 = Mu4->donnees;

  /* initialisation a zero */

  for (i=0; i<Dim; i++)
    {
    for(j=0; j<Dim; j++)
      {
      varz[i][j] = (TDouble)ZERO;
      }
    }


  for (i=0; i< NbObs; i++)
    {
    ii = NbObs+i;
    varz[i][i] = vary[i] / nbrepet[i];
    varz[i][ii] = 
       (mu3[i] + ((TDouble)2 * valf[i]  * vary[i]))
       / nbrepet[i];
    varz[ii][i] = varz[i][ii];
    varz[ii][ii] = 
       (mu4[i] + ((TDouble)4 * valf[i] * valf[i] * vary[i])
       - (vary[i] * vary[i]) 
       + ((TDouble)4 * valf[i] * mu3[i]))
       / nbrepet[i];
    if (varz[ii][ii] <= (TDouble)ZERO)
      {
      /* theoriquement ca n'est pas possible,
      mais ca peut le devenir, suite a des arrondis de calcul */
      sprintf(Mess1,"%d", (ii+1));
      sprintf(Mess2, "%10.20e", varz[ii][ii]);
/*      printf("CVarZmv: le %s-ieme terme de la diagonale de VarZ est <=0 (= %s)\n",
        Mess1, Mess2);
*/

      NLWARNING((WARVARZ1,3,"CVarZmv",Mess1, Mess2,WNUMER));
      /* dans ce cas, les calculs qui incluent la variance
      n'ont pas de sens */
      *Code = WARVARZ1;
      return(OK);
      } /* fin du cas <=0 */

    }


  return(OK);
  }


