/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : NLVCtxPuss                |
| Date                 : 1991                      |
| Derniere mise a jour :                           |
| Concepteur           : A.Bouvier                 |
| Role                 : Module d'initialisation et|
|  de verification du contexte du processus        |
| Reference conception :                           |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlmacros.h"
#include "errcodes.h"
#include "nlglobal.h"


/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/

/* fonctions des autres modules */
TLongInt IMinL();
TShortInt GerMessage();
TShortInt VerifParam();

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : CSigmaR                  |
| Role                  : Calculer Sigma quand     |
|  TypeSigma=VARREPET                              |
| Parametres d'entree   :                          |
|  NbRepet: le nombre de repetitions en chaque     |
|    point                                         |
|  S2: vecteur de la somme des carres              | 
| Parametres de sortie  :                          |
|  Sigma: valeur de l'ecart-type (au carre)        |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Fonctions appelantes : NLVCtxPuss                |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt CSigmaR(NbRepet, S2,
                     Sigma)

  /* arguments d'entree */
  TVectLong *NbRepet;
  TVect *S2;

  /* arguments de sortie */
  TDouble *Sigma;

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TShortInt i, nbele;
  TDouble A1, B1;

/* pointeurs pour ameliorer la performance */
  TLongInt *nbrepet;
  TDouble *s2;

/* Mettre son nom dans la trace */
  ECRTRACE("CSigmaR");

/* Affectation des pointeurs */
  nbrepet = NbRepet->donnees;
  s2 = S2->donnees;
  nbele = NbRepet->nbele;

  A1 = 0;
  B1 = 0;
  for (i = 0; i < nbele; i++)
    {
    A1 = A1 + ((TDouble)(nbrepet[i] - 1) * s2[i]);
    B1 = B1 + (TDouble)(nbrepet[i] - 1);
    }
  *Sigma = A1 / B1;

  return(OK);
  }
 

/*--------------- Identification fonction ----------
| Nom de la fonction    : NLVCtxPuss               |
| Role                  : Verifier et initialiser  |
| le contexte du processus d'estimation            |
| Parametres d'entree   :                          |
|  Vari: indications sur la variance               |
|  NbThetaEs: nombre de parametres Theta totaux    |
|  (=NbTheta*NbCourbe)                             |
|  NbBetaEs: nombre de parametres Beta totaux      |
|  (=NbBeta*NbCourbe)                              |
|  NbRepet: le nombre de repetitions en chaque     |
|    point                                         |
|  S2: vecteur de la somme des carres              | 
| Parametres d'entree-sortie  :                    |
|   CtxPuss: le contexte du processus d'estimation |
|    initialise et verifie                         |
| Parametres de sortie :                           |
|   Code : OK ou ERRVPUSS                          | 
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CSigmaR, IMinL, VerifMu,  |
|                        VerifParam                |
| Fonctions appelantes : NLVInit                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt NLVCtxPuss(Vari, NbThetaEs, NbBetaEs, NbRepet, S2,
                     CtxPuss, Code)

  /* arguments d'entree */
  TShortInt Vari , NbThetaEs, NbBetaEs;
  TVectLong *NbRepet;
  TVect *S2;

  /* arguments d'entree- sortie */
  TCtxPuss *CtxPuss;

  /* argument de sortie */
  TShortInt *Code;


/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TChar Mess1[15], Mess2[15],Mess3[15],Mess4[15],Mess5[15],Mess6[15];
  TLongInt MinRep;
  TShortInt MonCode;

  /* pointeur sur les fonctions appelees non encore apparues dans le module */
   TShortInt VerifMu();

/* Mettre son nom dans la trace */
  ECRTRACE("NLVCtxPuss");

  *Code =OK;

/* Verification de l'algorithme */
  if ((CtxPuss->Algo != GN) && (CtxPuss->Algo != GM))
    {
    sprintf(Mess1, "%hd", DEFALGO);
    sprintf(Mess2, "%hd", GM);
    sprintf(Mess3, "%hd", GN);
    NLWARNING((WARVPUSS1,4,"NLVCtxPuss",Mess1, Mess2, Mess3, WMETHO));
    /* Actions effectuees par la macro NLWARNING: */
   /* Ecriture de:
  Valeur allouee a CtxPuss.Algo: %s%.
  Les codes possibles sont: %s% (GM) et %s% (GN).
$ Le code fixe pour l'algorithme n'etant pas un code valide,
  on le modifie et on continue.
  */

    CtxPuss->Algo = DEFALGO;
    CtxPuss->Lambda0 = DEFLAMBDA0;
    CtxPuss->LambdaC1 = DEFLAMBDAC1;
    CtxPuss->LambdaC2 = DEFLAMBDAC2 / CtxPuss->LambdaC1;
    CtxPuss->OmegaPas = 0;
    }

/* Verification de la correction de la direction de descente */
  if ((CtxPuss->DirecC <= (TDouble)ZERO) || (CtxPuss->DirecC >= (TDouble)1))
    {
    sprintf(Mess1, "%f", DEFDIRECC);
    NLWARNING((WARVPUSS2,2,"NLVCtxPuss",Mess1, WMETHO));
    /* Actions effectuees par la macro NLWARNING: */
    /* Ecriture de:
    Valeur allouee a CtxPuss.DirecC: %s%.
$ La correction de la direction de descente
  doit etre positive et  inferieure a 1.
  On la modifie et on continue.
   */

    CtxPuss->DirecC = DEFDIRECC;
    }



  if(CtxPuss->Algo == GM)
    {
    /* Verification de Lambda0  si Algo=GM */
    if (CtxPuss->Lambda0 <= (TDouble)ZERO)
      {
      sprintf(Mess1, "%d", DEFLAMBDA0);
      NLWARNING((WARVPUSS3, 2 ,"NLVCtxPuss",Mess1, WMETHO));
      /* Actions effectuees par la macro NLWARNING: */
      /* Ecriture de:
      Valeur allouee a CtxPuss.Lambda0: %s%.
$ La valeur initiale du parametre de Gauss-Marquardt
  doit etre positive. On la modifie et on continue.
      */
      CtxPuss->Lambda0 = DEFLAMBDA0;
      }

    /* Verification de LambdaC1  si Algo=GM */
    if ((CtxPuss->LambdaC1 <= (TDouble)ZERO) || (CtxPuss->LambdaC1 >= 1.))
      {
      sprintf(Mess1, "%f", DEFLAMBDAC1);
      NLWARNING((WARVPUSS4, 2 ,"NLVCtxPuss",Mess1, WMETHO));
      /* Actions effectuees par la macro NLWARNING: */
      /* Ecriture de:
      Valeur allouee a CtxPuss.LambdaC1: %s%.
$ La valeur de correction du parametre de Gauss-Marquardt a chaque
  iteration doit etre positive et inferieure a 1.
  On la modifie et on continue.
      */

      CtxPuss->LambdaC1 = DEFLAMBDAC1;
      }

    /* Verification de LambdaC2  si Algo=GM */
    if (CtxPuss->LambdaC2 <= (TDouble)1 / CtxPuss->LambdaC1)
      {
      CtxPuss->LambdaC2 = (TDouble)DEFLAMBDAC2 / CtxPuss->LambdaC1;
      sprintf(Mess1,"%f", CtxPuss->LambdaC2);
      NLWARNING((WARVPUSS5, 2 ,"NLVCtxPuss",Mess1, WMETHO));
      /* Actions effectuees par la macro NLWARNING: */
      /* Ecriture de:
      Valeur allouee a CtxPuss.LambdaC2: %s%.
$ La valeur de correction du parametre de Gauss-Marquardt lors de la
  recherche du pas optimal doit etre superieure a 1/LambdaC1.
   On la modifie et on continue.
      */  
      }

    /* Verification de MaxLambda  si Algo=GM */
    if ((CtxPuss->MaxLambda <= (TDouble)ZERO) ||
        (CtxPuss->MaxLambda >   CtxPuss->Lambda0))
      {
      sprintf(Mess1, "%f", DEFMAXLAMBDA);
      NLWARNING((WARVPUSS9, 2 ,"NLVCtxPuss",Mess1, WMETHO));
      /* Actions effectuees par la macro NLWARNING: */
      /* Ecriture de:
      Valeur allouee a CtxPuss.MaxLambda: %s%.
$ La valeur maximale du parametre de Gauss-Marquardt
  doit etre positive et inferieure a Lambda0. On la modifie et on continue.
      */
      CtxPuss->MaxLambda = DEFMAXLAMBDA;
      }

    } /* fin du cas ou Algo =GM */

  else
    {
    /* quand Algo != GM, on initialise quand meme les Lambda,
    pour avoir ensuite un traitement homogene, c.a.d eviter des "if(Algo..." */
    CtxPuss->Lambda0 = DEFLAMBDA0;
    CtxPuss->LambdaC1 = DEFLAMBDAC1;
    CtxPuss->LambdaC2 = DEFLAMBDAC2 / CtxPuss->LambdaC1;
    }


/* Verification de MaxCritArret */
  if (CtxPuss->MaxCritArret < (TDouble)ZERO)
    {
    sprintf(Mess1, "%f", DEFCRITA);
    NLWARNING((WARVPUSS6,2,"NLVCtxPuss",Mess1,WMETHO));
     /* Actions effectuees par la macro NLWARNING:
     Ecriture de:
     Valeur allouee a CtxPuss.MaxCritArret: %s%.
$ La borne du critere d'arret doit etre positive ou nulle.
  On la modifie et on continue.
     */
    CtxPuss->MaxCritArret = DEFCRITA;
    }


/* Verification de MaxErr */
  if (CtxPuss->MaxErr < (TLongInt)0)
    {
    sprintf(Mess1, "%d", DEFMAXERR);
    NLWARNING((WARVPUSS7,2,"NLVCtxPuss",Mess1, WMETHO));
     /* Actions effectuees par la macro NLWARNING:
     Ecriture de:
     Valeur allouee a CtxPuss.MaxErr: %s%.
$ Le nombre de reprises en cas d'erreur dans le calcul du modele
  doit etre  positif ou nul.
  On le modifie et on continue.
    */
    CtxPuss->MaxErr = DEFMAXERR;
    }



/* Verification de MaxIter */
    if (CtxPuss->MaxIter < (TLongInt)0)
    {
    CtxPuss->MaxIter = DEFMAXITER * (NbThetaEs + NbBetaEs);
    sprintf(Mess1,"%ld", CtxPuss->MaxIter);
    NLWARNING((WARVPUSS8,2,"NLVCtxPuss",Mess1, WMETHO));
    /* Actions effectuees par la macro NLWARNING:
    Ecriture de:
    Valeur allouee a CtxPuss.MaxIter: %s%.
$ Le nombre maximum d'iterations est egal a la valeur par defaut,
  ou bien est negatif, ou bien est superieur au maximum autorise.
  On modifie sa valeur selon le nombre de parametres et on continue.
    */
    }

/* Verification de MaxDeb */
  if (CtxPuss->MaxDeb <= (TLongInt)1)
    {
    sprintf(Mess1, "%d", DEFMAXDEB);
    NLWARNING((WARVPUSS12,2,"NLVCtxPuss",Mess1, WMETHO));
     /* Actions effectuees par la macro NLWARNING:
    Ecriture de:
    Valeur allouee a CtxPuss.MaxDeb: %s%.
$ Le nombre de fois ou, consecutivement, on repart du debut
  d'un intervalle doit etre superieur a 1.
  On le modifie et on continue.
    */
    CtxPuss->MaxDeb = DEFMAXDEB;
    }

/* Verification de NbEtapes */
  if ((CtxPuss->NbEtapes <= 0) || (CtxPuss->NbEtapes > MAXETAP))
    {
    sprintf(Mess1,"%d", CtxPuss->NbEtapes);
    sprintf(Mess2, "%d", MAXETAP);
    /* Actions effectuees par la macro NLWARNING suivante: 
    fprintf(stderr, "La valeur de CtxPuss.NbEtapes (%s) est erronee\n",
      Mess1); 
    fprintf(stderr, "Elle doit etre positive et inferieure a %s\n",
      Mess2);*/
    NLWARNING((ERRVPUSS,3,"NLVCtxPuss",Mess1, Mess2, WMETHO)); 
     /* On appelle NLWarning avec un un code d'erreur, car ca provoquera 
     une erreur dans NLVInit */
    *Code = ERRVPUSS;
    }

/* Initialisation de OmegaPas */
  if (CtxPuss->Algo == GM)
    {
    CtxPuss->OmegaPas = 0;
    }
  if ((CtxPuss->Algo == GN) && (CtxPuss->OmegaPas <= (TDouble)ZERO))
    {
    sprintf(Mess1,"%f", DEFOMEGA);
    NLWARNING((WARVPUSS10,2,"NLVCtxPuss",Mess1, WMETHO));
    /* Actions effectuees par la macro NLWARNING:
    Ecriture de:
    Valeur allouee a CtxPuss.OmegasPas: %s%.
$ La valeur de correction du pas optimal doit etre positive.
  On la modifie et on continue.
    */
    CtxPuss->OmegaPas = DEFOMEGA;
    }

/* Verification de TypeSigma */
  if ((CtxPuss->TypeSigma != CONNU) && (CtxPuss->TypeSigma != VARREPET)
      && (CtxPuss->TypeSigma != VARRESID) && (CtxPuss->TypeSigma != IGNORE)
      && (CtxPuss->TypeSigma != VARINTRA) )
    {
    sprintf(Mess2, "%hd", CONNU);
    sprintf(Mess3, "%hd", VARREPET);
    sprintf(Mess4, "%hd", VARRESID);
    sprintf(Mess5, "%hd", IGNORE);
    sprintf(Mess6, "%hd", VARINTRA);
    if((Vari == VB) || (Vari == VTB))
      {
      CtxPuss->TypeSigma = IGNORE;
      }
    else
      {
      if (Vari == VI)
        {
        CtxPuss->TypeSigma = VARINTRA;
        }
      else
        {
        CtxPuss->TypeSigma = VARRESID;
        }
      }
    sprintf(Mess1,"%hd", CtxPuss->TypeSigma);
    NLWARNING((WARVPUSS11,7,"NLVCtxPuss", Mess1, Mess2, Mess3, Mess4, Mess5, 
                   Mess6, WMETHO));
    /* Actions effectuees par la macro NLWARNING:
    Ecriture de:
     Valeur allouee a CtxPuss.TypeSigma: %s%.
 Les codes sont:
  %s% (CONNU), %s% (VARREPET), %s% (VARRESID), %s% (IGNORE), %s% (VARINTRA).
$ La facon de calculer l'ecart-type n'est pas un code valide ou bien
  c'est la valeur par defaut.
  On la modifie et on continue.
    */

    }


  /* Verification de TypeSigma=VARREPET et du nombre de repetitions en chaque point  */
  if (CtxPuss->TypeSigma == VARREPET)
    {
    MinRep = IMinL(NbRepet);
    if (MinRep  < MINREP)
      {
      sprintf(Mess1,"%hd", MINREP);
      /* MESSAGE ecrit par NLWARNING: */
/*
      printf("Le type de calcul de Sigma ne peut etre VARREPET, car\n");
      printf("il faut au moins %s (MINREP) repetitions \n", Mess1);
      printf(" en chacun des points d'observations pour pouvoir l'utiliser\n");
      printf(" On fixe TypeSigma a VARRESID\n");
*/
      NLWARNING((WARVPUSS16,2,"NLVCtxPuss", Mess1, WMETHO));
      CtxPuss->TypeSigma = VARRESID;
      }
    if ((MinRep >= MINREP) && (MinRep <= MAXREP))
      {
      sprintf(Mess1,"%hd", MAXREP);

      /* MESSAGE ecrit par NLWARNING: */
/*
      printf("Le type de calcul de Sigma est VARREPET\n");
      printf(" mais il y a au moins un point d'observation \n");
      printf(" pour lequel le nombre de repetitions est inferieur a %s (MAXREP)\n",
                 Mess1);
*/
      NLWARNING((WARVPUSS17,2,"NLVCtxPuss", Mess1, WMETHO));
      }
    } /* fin TypeSigma=VARREPET */

  /* Initialisation de Sigma */
  if (CtxPuss->TypeSigma == VARREPET)
    {
    APPEL(CSigmaR(NbRepet, S2, &(CtxPuss->Sigma)));
    }
   if ((CtxPuss->TypeSigma == IGNORE) || (CtxPuss->TypeSigma == VARINTRA) ||
        ((CtxPuss->TypeSigma == VARRESID) && ((Vari == VB) || (Vari == VTB))))
    {
    CtxPuss->Sigma = (TDouble)1;
    }


  /* verification de TypeMu */
  APPEL(VerifMu(NbRepet, &(CtxPuss->Mu3), &(CtxPuss->Mu4), &(CtxPuss->TypeMu),
                &MonCode));
  if (MonCode !=OK)
    {
    *Code = MonCode;
    }

  /* verification de  Famille */
  if (  (CtxPuss->Famille  != GAUSS) &&
	(CtxPuss->Famille  != POISSON) &&
	(CtxPuss->Famille  != BINOM) &&
	(CtxPuss->Famille  != MULTINOM) &&
	(CtxPuss->Famille  != BERNOULLI)) {
    sprintf(Mess1, "%d", CtxPuss->Famille);
    NLWARNING((WARVPUSS18,2,"NLVCtxPuss",Mess1, WMETHO));
    *Code =WARVPUSS18;
  }

  /* verification de la dimension des parametres */
  APPEL(VerifParam( NbThetaEs, "f", &(CtxPuss->Theta0),  &MonCode));
  if (MonCode !=OK)
    {
    /* je continue les verif mais je sauvegarde le code d'erreur */
    *Code = MonCode;
    }
  APPEL(VerifParam( NbBetaEs, "v", &(CtxPuss->Beta0),  &MonCode));
  if (MonCode !=OK)
    {
    *Code = MonCode;
    }
  
  /* retour */
  return(OK);
  }


/*--------------- Identification fonction ----------
| Nom de la fonction    : VerifMu                  |
| Role                  : Verifier TypeMu          |
| Parametres d'entree   :                          |
|  NbRepet: vecteur des repetitions                |
|  Mu3, Mu4: les moments                           |
| Parametres d'e/s      :                          |
|  TypeMu: facon de calculer les Mu                |
| Parametres de sortie  :                          |
|  Code =OK ou ERRMU                               |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  IMinL                    |
| Fonctions appelantes :  NLVCtxPuss               |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt VerifMu(NbRepet, Mu3, Mu4,
                  TypeMu, Code)

/* arguments d'entree */
TVectLong *NbRepet;
TVect *Mu3, *Mu4;

/* arguments d'entree-sortie */
TShortInt *TypeMu;

/* argument de sortie */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt NbObs;
TLongInt IMinR;
TChar Mess1[5], Mess2[5], Mess3[5], Mess4[5], Mess5[5];

/* Ecriture de la trace */
ECRTRACE("VerifMu");

*Code = OK;

/* Verification de TypeMu */
  if ((*TypeMu != CONNU) && (*TypeMu != MUGAUSS)
       && (*TypeMu != MURES) && (*TypeMu != MURESREPET))
    {
    sprintf(Mess2, "%hd", CONNU);
    sprintf(Mess3, "%hd", MUGAUSS);
    sprintf(Mess4, "%hd", MURES);
    sprintf(Mess5, "%hd", MURESREPET);
    sprintf(Mess1,"%hd", DEFMU);
    NLWARNING((WARMU1,6,"VerifMu",Mess1, Mess2, Mess3, Mess4, Mess5, WMETHO));
    /* Actions effectuees par la macro NLWARNING:
    Ecriture de:
     Valeur allouee a CtxPuss.TypeMu: %s%.
  Les codes sont: %s% (CONNU), %s% (MUGAUSS), %s% (MURES), %s% (MURESREPET).
$ La facon de calculer les moments n'est pas un code valide.
  On la modifie et on continue.
    */

    *TypeMu = DEFMU;
    }

  if (*TypeMu == MURESREPET)
    {
    IMinR= IMinL(NbRepet);
    if (IMinR < MINREPMU)
      {
      sprintf(Mess3,"%ld", IMinR);
      sprintf(Mess1,"%hd", MINREPMU);
      sprintf(Mess2,"%hd", DEFMU);

      /* Actions effectuees par la macro NLWARNING suivante: */
/*
      fprintf(stderr, "Le nombre minimum de repetitions, %s,\n", Mess3);
      fprintf(stderr," est inferieur au minimum %s admis pour que TypeMu soit egal a MURESREPET\n",
       Mess1);
      fprintf(stderr, " On fixe TypeMu a la valeur par defaut: %s\n", Mess2);
*/
      NLWARNING((WARMU2,4,"VerifMu", Mess3, Mess1, Mess2, WMETHO));
      *TypeMu = DEFMU;
      }
    }

  if ( *TypeMu == CONNU)
    {
    NbObs = NbRepet->nbele;
    if ((Mu3->nbele != NbObs) || (Mu4->nbele != NbObs))
      {
      sprintf(Mess1,"%d", NbObs);
      /* Actions effectuees par la macro NLWARNING suivante: */
/*
      fprintf(stderr, "La dimension des vecteurs Mu3 et Mu4 doit etre  %s\n", Mess1);
*/
      NLWARNING((ERRMU,2,"VerifMu", Mess1, WMETHO)); 
     /* On appelle NLWarning avec un un code d'erreur, car ca provoquera 
     une erreur dans NLVInit */
      *Code = ERRMU;
      }
    } /* fin de TypeMu=CONNU */
  /* retour */
return(OK);
}


