/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : NLVMod                    |
| Date                 : 1991                      |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : A. Bouvier                |
| Role                 :Verifier et initialiser    |
|                      le Modele                   |
| Reference conception :                           |
| Lecteur              :  A. Bouvier               |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <stdio.h>
#include <float.h>

#include "nlchoix.h"
#include "nlcodes.h"
#include "nltypes.h"
#include "nlmacros.h"
#include "errcodes.h"

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/

/* fonction de l'analyseur formel */
TShortInt  ChargArb();

/* fonctions des autres modules */
TShortInt CreerVect(), GerMessage(), CreerVectShort(),
           DetruVectShort(), IMaxS();



/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : VDimComp                 |
| Role                  : Verifier la dimension    |
|  d'un vecteur composant  des contraintes sur les |
|  parametres Theta ou bien Beta du modele         |
|  Ce composant est BInf, ou BSup ou EgalN         |
|  mais pas EgalP car il n'a pas le meme type      |
| Parametres d'entree   :                          |
|  NbValParam : nombre de valeurs des parametres   |
| (= nombre de parametres du modele * NbCourbe)    |
|  NomContr: nom du composant                      |
|  NomFonc: "f" ou "v"                             |
|  Comp: vecteur du composant                      |
| Parametres d'e./s.    :                          |
| Parametres de sortie  :                          |
|  Code : OK ou code de warning                    |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Fonctions appelantes : VIcont                    |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt VDimComp(NbValParam,  NomContr, NomFonc, Comp, Code)
/* ............... Arguments d'entree ............................ */
TShortInt NbValParam;
TChar * NomContr, *NomFonc;
TVect *Comp;

/* argu de sortie */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TChar Mess1[5], Mess2[5];


/* Ecriture de la trace */
ECRTRACE("VDimComp");

*Code =OK;

if ((Comp->nbele != 0) && (Comp->nbele != NbValParam))
  {
  sprintf (Mess1,"%d", Comp->nbele);
  sprintf (Mess2, "%d", NbValParam);
  /* Actions effectuees par la macro :
  printf ("Le nombre de valeurs de la contrainte %s sur les parametres de %s, %s,  doit etre egal au \n",
         NomContr, NomFonc, Mess1);
  printf("nombre de parametres du modele de %s multiplie par le nombre de courbes\n",
        NomFonc);
  printf("soit: %s\n", Mess2);
  */
  NLWARNING ((WARVMOD7,6,"VDimComp",NomContr, NomFonc, Mess1, NomFonc, Mess2, WMETHO)) ;
  *Code = WARVMOD7;
  }

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : VEgalP                   |
| Role                  : Verifier les indices     |
|  differents des parametres:                      |
|  Les valeurs doivent etre positives et           |
|  inferieures ou egales a :                       |
|  nombre de parametres du modele * NbCourbes      |
|  Elles doivent etre consecutives                 |
| Parametres d'entree   :                          |
|  NbValParam : nombre de valeurs des parametres   |
| (= nombre de parametres du modele * NbCourbe)    |
|  NomFonc: type de la fonction ="f" ou bien "v"   |
|  EgalP: vecteur des indices differents           |
| Parametres d'e./s.    :                          |
| Parametres de sortie  :                          |
|  Code : OK ou code de warning                    |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CreerVectShort,           |
|                        DetruVectShort            |
|                        IMaxS                     |
| Fonctions appelantes : VICont                    |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt VEgalP(NbValParam,  NomFonc, EgalP, Code)
/* ............... Arguments d'entree ............................ */
TShortInt NbValParam;
TChar *NomFonc;
TVectShort *EgalP;

/* argu de sortie */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TChar Mess1[5], Mess2[5], Mess3[5];
TShortInt i, NbEff;
TVectShort IndVect;


/* Ecriture de la trace */
ECRTRACE("VEgalP");

*Code =OK;

/* Creation d'un vecteur pour mettre les indices dans l'ordre croissant */
APPEL(CreerVectShort(NbValParam, &IndVect));

/* Initialisation a une valeur impossible */
for (i=0; i < NbValParam; i++)
  {
  IndVect.donnees[i]= -1;
  }

/* On met dans IndVect[i] la valeur (i+1) 
   si (i+1) est un indice de parametre different rencontre */
for (i=0; i < NbValParam; i++)
  {
  if ((EgalP->donnees[i] <= 0) || (EgalP->donnees[i] > NbValParam))
    {
    /* verifier que les indices d'egalite entre parametres sont positifs et inferieurs 
    ou egaux au nombre total de parametres de la sorte consideree */
    sprintf(Mess1,"%hd", (i+1));
    sprintf(Mess2,"%hd", EgalP->donnees[i]);
    sprintf(Mess3,"%d", NbValParam);
    /* Actions effectuees par la macro NLWARNING :
    fprintf(stderr,
" La %s-ieme valeur de la contrainte d'egalite entre les parametres de %s (EgalP)\
  est egale a%s\n",    Mess1, NomFonc, Mess2);
     fprintf(stderr," Elle doit etre positive et inferieure au nombre total de parametres, soit: %s\n",
             Mess3);
    fprintf(stderr," On continue les verifications possibles, puis on s'arretera en erreur\n");
    */
    NLWARNING((WARVEGALP1, 5,"VEgalP", Mess1, NomFonc, Mess2,Mess3, WMETHO)) ;
    *Code = WARVEGALP1;
    }
  else
    {
    IndVect.donnees[(EgalP->donnees[i]-1)] = EgalP->donnees[i];
    }
  }

/* On verifie que toutes les valeurs consecutives ont ete bien rencontrees */
if (*Code == OK)
  {
  NbEff = IMaxS(EgalP);
  for (i=0; i < NbEff; i++)
    {
    if (IndVect.donnees[i] != (i+1))
      {
      sprintf(Mess1,"%hd", (i+1));
      /* Actions effectuees par la macro NLWARNING :
      fprintf(stderr,
" Dans les contraintes d'egalite entre les parametres de %s (EgalP),\
 il manque le parametre d'indice %s\n", NomFonc, Mess1);
      fprintf(stderr," On continue les verifications possibles, puis on s'arretera en erreur\n");
      */
      NLWARNING((WARVEGALP2, 3,"VEgalP", NomFonc, Mess1, WMETHO)) ;
      *Code = WARVEGALP2;
      }
    }
  }

DetruVectShort(&IndVect);
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : VerifCont                |
| Role                  : Verifier et initialiser  |
|  les contraintes sur les parametres du modele    |
| Parametres d'entree   :                          |
|  NbValTheta : nombre de valeurs des parametres   |
|   Theta                                          |
| (= nombre de parametres du modele * NbCourbe)    |
|  NbValBeta : nombre de valeurs des parametres    |
|   Beta (eventuellement 0)                        |
| (= nombre de parametres du modele * NbCourbe)    |
|  Theta0: vecteur des valeurs initiales des       |
|   parametres Theta                               |
|  Beta0: vecteur des valeurs initiales des        |
|   parametres Beta                                |
| Parametres d'e./s.    :                          |
|  CTheta0: contraintes sur les Theta              |
|  CBeta0: contraintes sur les Beta                |
| En entree, ce sont les contraintes fixees par    |
|  l'utilisateur, en sortie, elles sont            |
|  eventuellement modifiees                        |
| Parametres de sortie  :                          |
|  Code : OK ou code de warning                    | 
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : VICont                    |
| Fonctions appelantes : NLVMod                    |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt VerifCont(NbValTheta, NbValBeta,Theta0, Beta0,
                   CTheta0, CBeta0, Code)
/* ............... Arguments d'entree ............................ */
TShortInt NbValTheta, NbValBeta;
TVect *Theta0, *Beta0;

/* ............... Arguments d'entree-sortie ...................... */
TContr *CTheta0, *CBeta0;

/* argu de sortie */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt MonCode;

/* pointeur sur la fonction du module appelee */
TShortInt VICont();

/* Ecriture de la trace */
ECRTRACE("VerifCont");

*Code = OK;

/* determiner Ctr = FAUX si aucune contrainte */
if ((CTheta0->BInf.nbele == 0) && (CTheta0->BSup.nbele == 0) &&
    (CTheta0->EgalN.nbele == 0) && (CTheta0->EgalP.nbele == 0))
  {
  CTheta0->Ctr = FAUX;
  }
else
  {
  CTheta0->Ctr = VRAI;
  }


/* verifier les  composants de CTheta0 et initialiser EgalP */
APPEL(VICont(NbValTheta, "f", Theta0, CTheta0, &MonCode));
if (MonCode != OK)
  {
  *Code = MonCode;
  }


/* idem sur les Beta */
if (NbValBeta > 0)
  {
  if ((CBeta0->BInf.nbele == 0) && (CBeta0->BSup.nbele == 0) &&
      (CBeta0->EgalN.nbele == 0) && (CBeta0->EgalP.nbele == 0))
    {
    CBeta0->Ctr = FAUX;
    }
  else
    {
    CBeta0->Ctr = VRAI;
    }
  APPEL(VICont(NbValBeta, "v", Beta0, CBeta0, &MonCode));
  if (MonCode != OK)
    {
    *Code = MonCode;
    }
  }
else
  {
  CBeta0->Ctr = FAUX;
  }

return(OK);
}
  
/*--------------- Identification fonction ----------
| Nom de la fonction    : VerifParam               |
| Role                  : Verifier la dimension des|
|  vecteurs contenant les valeurs initiales des    |
|  parametres Theta ou bien Beta                   |
| Parametres d'entree   :                          |
|  NbValParam : nombre de valeurs des parametres   |
|   (= NbParam*NbCourbe)                           |
|  NomFonc: type de la fonction ="f" ou bien "v"   |
| Parametres d'e/s      :                          |
|  Param0: vecteur des valeurs initiales des       |
|   parametres                                     |
|  En entree, s'il y a des parametres de la sorte  |
|   consideree, ces vecteurs contiennent leurs     |
|   valeurs, sinon, ce sont des vecteurs a 0       |
|   element                                        |
| Parametres de sortie  :                          |
|   Code : OK ou code de warning                   |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CreerVect                 |
| Fonctions appelantes : NLVMod, NLVCtxPuss        |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt VerifParam( NbValParam,  NomFonc,
                     Param0,  Code)
/* ............... Arguments d'entree ............................ */
TShortInt NbValParam;
TChar *NomFonc;

/* argu d'entree- sortie */
TVect *Param0;

/* argu de sortie */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TChar Mess1[5], Mess2[5];


/* Ecriture de la trace */
ECRTRACE("VerifParam");

*Code =OK;

if (NbValParam>0)
  {
  /* verification de la dimension du vecteur des parametres */
  if (Param0->nbele != NbValParam)
    {
    sprintf (Mess1,"%d",Param0->nbele);
    sprintf (Mess2, "%d", NbValParam);
    /* Actions effectuees par la macro:
    printf ("Le nombre de valeurs initiales des parametres de %s , %s, doit etre egal au\n",
            NomFonc, Mess1);
    printf(" nombre de parametres du modele de %s multiplie par le nombre de courbes\n",
             NomFonc);
    printf("soit: %s\n",  Mess2);
    */
    NLWARNING ((WARVMOD5,5,"VerifParam",NomFonc, Mess1,NomFonc, Mess2, WMETHO)) ;
    *Code = WARVMOD5;
    }
  }

/* Retour */
return(OK);

}

  
/*--------------- Identification fonction ----------
| Nom de la fonction    : VerifPBis                |
| Role                  : Verifier la dimension des|
|  des vecteurs contenant les valeurs des          |
|  parametres de second niveau                     |
| Parametres d'entree   :                          |
|  NbGam  : nombre de parametres de second niveau  |
|  NomFonc: type de la fonction ="f" ou bien "v"   |
| Parametres d'e/s      :                          |
|  Gam : vecteur des valeurs des parametres        |
|   de second niveau                               |
|  En entree, s'il y a des parametres de la sorte  |
|   consideree, ces vecteurs contiennent leurs     |
|   valeurs, sinon, ce sont des vecteurs a 0       |
|   element                                        |
| Parametres de sortie  :                          |
|   Code : OK ou code de warning                   |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CreerVect                 |
| Fonctions appelantes : NLVMod                    |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt VerifPBis(  NbGam, NomFonc,
                      Gam, Code)
/* ............... Arguments d'entree ............................ */
TShortInt  NbGam;
TChar *NomFonc;

/* argu d'entree- sortie */
TVect  *Gam;

/* argu de sortie */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TChar Mess1[5], Mess2[5];


/* Ecriture de la trace */
ECRTRACE("VerifPBis");

*Code =OK;

/* verification de la dimension du vecteur des parametres-bis */

if (NbGam >0)
  {
  if (Gam->nbele != NbGam)
    {
    sprintf (Mess1,"%d",Gam->nbele);
    sprintf (Mess2, "%d", NbGam);
    /* Actions effectuees par la macro:
    printf ("Le nombre de valeurs des parametres de second niveau de %s \n",
            NomFonc);
    printf(" soit %s, doit etre egal au nombre de ceux-ci dans la definition du modele de %s\n", 
            Mess1, NomFonc);
    printf("soit: %s\n",  Mess2);
    */
    NLWARNING ((WARVMOD6,5,"VerifParam",NomFonc, Mess1,NomFonc, Mess2, WMETHO)) ;
    *Code = WARVMOD6;
    }
  }

/* Retour */
return(OK);

}

/*--------------- Identification fonction ----------
| Nom de la fonction    : VerifVari                |
| Role                  : Initialiser,             |
|  eventuellement, le mode de calcul de la variance|
|  Vari, et verifier sa concordance avec           |
|  l'existence d'un programme qui calcule v,       |
|  le nombre de parametres de v  et                |
|  la presence de poids                            |
| Parametres d'entree   :                          |
|  YaCalcV: Vrai s'il y a une fonction de calcul de|
|    v.                                            |
|  NbBeta : nombre de parametres du modele de la   |
|   variance                                       |
|  Poids: poids sur les observations               |
|   s'il n'y en a pas, c'est un vecteur a 0 element|
| Parametres d'e./s.    :                          |
|  Vari: indique le type de la variance            |
| Parametres de sortie  :                          |
|  Code : OK ou code de warning                    |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune                    |
| Fonctions appelantes : NLVMod                    |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt VerifVari(YaCalcV,  NbBeta, Poids,
                    Vari, Code)
/* ............... Arguments d'entree ............................ */
TLogic YaCalcV;
TShortInt NbBeta;
TVect *Poids;

/* ............... Arguments d'entree-sortie ..................... */
TShortInt *Vari;

/* argument de sortie */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/
{

/* Ecriture de la trace */
ECRTRACE("VerifVari");

*Code = OK;

/* affectation de la valeur par defaut de Vari,
    si l'utilisateur l'a laisse indefinie */
if (*Vari == DEFAUT)
  {
  if (NbBeta > 0)
    {
    *Vari = VSTB;
    }
  else
    {
    if (YaCalcV == VRAI)
      {
      *Vari = VST;
      }
    else
      {
      if (Poids->nbele > 0)
        {
        *Vari = SP;
        }
      else
        {
        *Vari = CTE;
        }
      } /* fin du else sur YaCalcV */
    } /* fin du  else (NbBeta > 0) */
  } /* fin du if (*Vari == DEFAUT) */

if ((*Vari != CTE) && (*Vari != SP) && (*Vari != VI) && (YaCalcV == FAUX))
  {
  /* Actions effectuees par la macro :
  printf ("Lorsque Modele.Vari n'est egal ni a CTE, ni a SP\n");
  printf( "il faut qu'il y ait une fonction de calcul de v\n");
  */
  NLWARNING ((WARVMOD3,1,"VerifVari",WMETHO)) ;
  *Code = WARVMOD3;
  }

if (((*Vari == VB) || (*Vari == VSB) || (*Vari ==VTB) || (*Vari == VSTB))
    && (NbBeta <= 0))
  {
  /* Actions effectuees par la macro :
  printf ("Lorsque Modele.Vari est egal a VB ou VSB ou VTB ou VSTB\n");
  printf( "il faut qu'il y ait des parametres Beta\n");
  */
  NLWARNING ((WARVMOD4,1,"VerifVari",WMETHO)) ;
 *Code = WARVMOD4;
  }

if ((*Vari == SP) && (Poids->nbele == 0))
  {
  /* Actions effectuees par la macro NLWARNING:
  printf ("Lorsque Modele.Vari est egal a SP\n");
  printf("Il faut qu'il y ait des poids pour qu'on puisse ponderer\n");
  printf("On affecte la valeur CTE a Vari\n");
  */
  NLWARNING ((WARVMOD1,1,"VerifVari", WMETHO));
  *Vari = CTE;
  }

 if (((*Vari == CTE) || (*Vari == SP) || (*Vari == VI) ) 
    && (YaCalcV == VRAI))
  {
 /* Actions effectuees par la macro NLWARNING:
  printf ("Lorsque Modele.Vari est egal a CTE ou SP ou VI\n");
  printf(" il ne faut pas qu'il y ait une fonction de calcul de v\n");
  */
  NLWARNING ((WARVMOD2,1,"VerifVari", WMETHO));
  *Code = WARVMOD2;
  }


/* Retour */
return(OK);

}



/*--------------- Identification fonction ----------
| Nom de la fonction    : VICont                   |
| Role                  : Verifier les composants  |
|  des contraintes sur les parametres Theta ou Beta|
|  du modele                                       |
|  initialiser, eventuellement, le composant EgalP |
| Parametres d'entree   :                          |
|  NbValParam : nombre de valeurs des parametres   |
| (= nombre de parametres du modele * NbCourbe)    |
|  NomFonc: "f" ou "v"                             |
|  Param0: valeurs initiales des parametres        |
| Parametres d'e./s.    :                          |
|  Contr0: contraintes                             |
| En entree, ce sont les contraintes fixees par    |
|  l'utilisateur, en sortie, le composant EgalP est|
|  eventuellement cree et rempli                   |
| Parametres de sortie  :                          |
|  Code : OK ou code de warning                    |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : VDimComp, VEgalP          |
| Fonctions appelantes : VerifCont                 |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt VICont(NbValParam, NomFonc, Param0, Contr0, Code)
/* ............... Arguments d'entree ............................ */
TShortInt NbValParam;
TChar *NomFonc;
TVect *Param0;

/* ............... Arguments d'entree-sortie......................... */
TContr *Contr0;

/* argu de sortie */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TChar Mess1[15], Mess2[15], Mess3[15], Mess4[15];
TShortInt i, ii, MonCode, EgalPCode;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TShortInt NbEgalP, NbEgalN, NbBInf, NbBSup;
TDouble *param0, *binf, *bsup, *egaln;
TShortInt *egalp;


/* Ecriture de la trace */
ECRTRACE("VICont");

*Code =OK;
param0 = Param0->donnees;

/* Verification de la dimension des vecteurs de Contr0 */

APPEL(VDimComp(NbValParam, "inferiority-bound", NomFonc , &(Contr0->BInf), &MonCode));
if (MonCode !=OK)
  {
  /* il y a eu une erreur mais je continue a verifier en sauvegardant
  le code d'erreur */
  *Code = MonCode;
  }

APPEL(VDimComp(NbValParam, "superiority-bound", NomFonc , &(Contr0->BSup), &MonCode));
if (MonCode !=OK)
  {
  /* il y a eu une erreur mais je continue a verifier en sauvegardant
  le code d'erreur */
  *Code = MonCode;
  }
APPEL(VDimComp(NbValParam, "numerical-equality", NomFonc , &(Contr0->EgalN), &MonCode));
if (MonCode !=OK)
  {
  /* il y a eu une erreur mais je continue a verifier en sauvegardant
  le code d'erreur */
  *Code = MonCode;
  }

/* Verification de la dimension de EgalP :
  on ne peut pas faire cette verif par VDimComp car le type de EgalP
  n'est pas TVect */

if ((Contr0->EgalP.nbele != 0) && (Contr0->EgalP.nbele != NbValParam))
  {
  sprintf (Mess1,"%d",Contr0->EgalP.nbele);
  sprintf (Mess2, "%d", NbValParam);
  /* Actions effectuees par la macro:
  printf ("Le nombre de valeurs de la contrainte %s sur les parametres de %s, %s,  doit etre egal au \n",
         "EgalP", NomFonc, Mess1);
  printf("nombre de parametres du modele de %s multiplie par le nombre de courbes\n",
        NomFonc);
  printf("soit: %s\n", Mess2);
  */
  NLWARNING ((WARVMOD7,6,"VICont","parameter-equality", NomFonc, Mess1, NomFonc, Mess2, WMETHO)) ;
  *Code = WARVMOD7;
  }

if (*Code != OK)
  {
  /* je retourne car, les dimensions etant fausses, je ne peux continuer les verif */
  return(OK);
  }


NbEgalP = Contr0->EgalP.nbele;

if (NbEgalP == 0)
  {
  /* pas de contraintes d'egalite entre parametres:
   on cree le vecteur EgalP et on le remplira par des indices
   de parametres croissant a partir de 1 (chaque parametre
   est egal a lui-meme) */
  APPEL(CreerVectShort(NbValParam, &(Contr0->EgalP)));
  }
else
  {
  /* on verifie que les indices de parametres differents sont valides */
  APPEL(VEgalP(NbValParam, NomFonc, &(Contr0->EgalP), &EgalPCode));
  if (EgalPCode != OK)
    {
    *Code = EgalPCode;
    }
  }


/* verifier les autres contraintes */
egalp = Contr0->EgalP.donnees;
NbEgalN = Contr0->EgalN.nbele;
egaln = Contr0->EgalN.donnees;
NbBInf = Contr0->BInf.nbele;
binf = Contr0->BInf.donnees;
NbBSup = Contr0->BSup.nbele;
bsup = Contr0->BSup.donnees;

for (i=0; i<NbValParam; i++)
  {
  /* verifier que les egalites numeriques sont egales aux valeurs initiales */
  if (NbEgalN != 0)
    {
    if (!myisnan(egaln[i]))
      {
      /* verifier qu'elles sont egales aux valeurs initiales */
      if (egaln[i] != param0[i])
        {
        sprintf (Mess1,"%d",(i+1));
        sprintf (Mess2,"%g",egaln[i]);
        sprintf (Mess3, "%g", param0[i]);
        /* Actions effectuees par la macro :
        printf ("La %s ieme valeur de la contrainte EgalN sur les parametres de %s, %s, \n",
          Mess1, NomFonc, Mess2);
        printf("est differente de la valeur initiale du parametre, soit: %s\n",
          Mess3);
        */
        NLWARNING ((WARVMOD11,5,"VICont",Mess1, NomFonc, Mess2, Mess3, WMETHO)) ;
        *Code = WARVMOD11;
        }
      } /* fin du if !myisnan(egaln[i] */
    } /* fin du cas ou il y a des EgalN */


  /* verifier que les bornes inf  sont <= aux valeurs initiales */
  if (NbBInf != 0)
    {
    if (!myisnan(binf[i]))
      {
      if (binf[i] > param0[i])
        {
        /* verifier que les bornes inf soient inf ou egales aux valeurs initiales */
        sprintf (Mess1,"%d",(i+1));
        sprintf (Mess2,"%g",binf[i]);
        sprintf (Mess3, "%g", param0[i]);
        /* Actions effectuees par la macro :
        printf ("La %s ieme valeur de la contrainte BInf sur les parametres de %s, %s, \n",
          Mess1, NomFonc, Mess2);
        printf("est superieure a la valeur initiale du parametre, soit: %s\n",
          Mess3);
        */
        NLWARNING ((WARVMOD9,5,"VICont",Mess1, NomFonc, Mess2, Mess3, WMETHO)) ;
        *Code = WARVMOD9;
        }
      } /* fin du if !myisnan(binf[i] */
    } /* fin du cas ou il y a des BInf */


  /* verifier que les bornes sup soient >= aux valeurs initiales */
  if (NbBSup != 0)
    {
    if (!myisnan(bsup[i]))
      {
      if (bsup[i] < param0[i])
        {
        /* verifier que les bornes sup soient sup ou egales aux valeurs initiales */
        sprintf (Mess1,"%d",(i+1));
        sprintf (Mess2,"%g",bsup[i]);
        sprintf (Mess3, "%g", param0[i]);
        /* Actions effectuees par la macro:
        printf ("La %s ieme valeur de la contrainte BSup sur les parametres de %s, %s, \n",
          Mess1, NomFonc, Mess2);
        printf("est inferieure a la valeur initiale du parametre, soit:%s\n",
          Mess3);
*/

        NLWARNING ((WARVMOD10,5,"VICont",Mess1, NomFonc, Mess2, Mess3, WMETHO)) ;
        *Code = WARVMOD10;
        }
      } /* fin du if !myisnan(bsup[i] */
    } /* fin du cas ou il y a des BSup */



  /* verifier que les bornes sup soient != aux bornes inf */
  if ((NbBSup != 0) && (NbBInf != 0))
    {
    if (!myisnan(bsup[i])  && !myisnan(binf[i]))
      {
      if (bsup[i] == binf[i])
        {
        sprintf (Mess1,"%d",(i+1));
        sprintf (Mess2,"%g",bsup[i]);
        sprintf (Mess3, "%g", binf[i]);
        /* Actions effectuees par la macro:
        printf ("La %s ieme valeur de la contrainte BSup sur les parametres de %s, %s, \n",
          Mess1, NomFonc, Mess2);
        printf("est egale a la valeur de la contrainte BInf du parametre, soit:%s\n",
          Mess3);
*/

        NLWARNING ((WARVMOD15,5,"VICont",Mess1, NomFonc, Mess2, Mess3, WMETHO)) ;
        *Code = WARVMOD15;
        } /* fin de (bsup[i] == binf[i]) */
      } /* fin du if !myisnan(bsup[i])  && !myisnan(binf[i]) */
    } /* fin du cas ou il y a des BSup et BInf */


  if ((NbEgalP != 0) && (EgalPCode == OK))
    {
     /* il y a des EgalP valides */
     /* verifier que les contraintes sur le prochain parametre egal 
       sont identiques */
    for (ii = (i+1); ii < NbValParam; ii++)
      {
      if (egalp[ii] == egalp[i])
        {
        /* ii identique a i */
        /* verifier  ValInit  */
        if (param0[i] != param0[ii])
          {
          sprintf (Mess1,"%d",(i+1));
          sprintf (Mess2,"%g",param0[i]);
          sprintf (Mess3,"%d",(ii+1));
          sprintf (Mess4, "%g", param0[ii]);
          /* Actions effectuees par la macro:
          printf ("La valeur initiale du %s ieme parametre de %s est egale a %s, \n",
                  Mess1, NomFonc, Mess2);
          printf("alors que celle fixee sur le %s parametre, egal au %s ieme, est  %s\n",
                 Mess3, Mess1, Mess4);
          */
          NLWARNING ((WARVMOD14,7,"VICont", Mess1, NomFonc, Mess2, Mess3, Mess1, Mess4, WMETHO)) ;
          *Code = WARVMOD14;
          }  /* fin de (param0[i] != param0[ii]) */

        /* verifier  EgalN  */
        if (NbEgalN >0)
          { 
          if (((egaln[i] != egaln[ii]) && !myisnan(egaln[i]) && !myisnan(egaln[ii])) || 
               (myisnan(egaln[i]) && !myisnan(egaln[ii])) ||
               (!myisnan(egaln[i]) && myisnan(egaln[ii])) )
            {
            sprintf (Mess1,"%d",(i+1));
            sprintf (Mess2,"%g",egaln[i]);
            sprintf (Mess3,"%d",(ii+1));
            sprintf (Mess4, "%g", egaln[ii]);
            /* Actions effectuees par la macro:
            printf ("La contrainte %s fixee sur le %s ieme parametre de %s est egale a %s, \n",
                   "EgalN", Mess1, NomFonc, Mess2);
            printf("alors que celle fixee sur le %s parametre, parametre egal au %s ieme, est  %s\n",
                   Mess3, Mess1, Mess4);
    
            */
            NLWARNING ((WARVMOD12,8,"VICont","numerical-equality", Mess1, NomFonc, Mess2, Mess3, Mess1, Mess4, WMETHO)) ;
            *Code = WARVMOD12;
            }
        } /* fin du cas ou il y a des EgalN */

        /* verifier  BInf */
        if (NbBInf >0)
          { 
          if (((binf[i] != binf[ii]) && !myisnan(binf[i]) && !myisnan(binf[ii])) || 
               (myisnan(binf[i]) && !myisnan(binf[ii])) ||
               (!myisnan(binf[i]) && myisnan(binf[ii])) )
            {
            sprintf (Mess1,"%d",(i+1));
            sprintf (Mess2,"%g",binf[i]);
            sprintf (Mess3,"%d",(ii+1));
            sprintf (Mess4, "%g", binf[ii]);
            /* Actions effectuees par la macro:
            printf ("La contrainte %s fixee sur le %s ieme parametre de %s est egale a %s, \n",
                   "BInf", Mess1, NomFonc, Mess2);
            printf("alors que celle fixee sur le %s parametre, parametre egal au %s ieme, est  %s\n",
                   Mess3, Mess1, Mess4);
            */
            NLWARNING ((WARVMOD12,8,"VICont","inferiority-bound", Mess1, NomFonc, Mess2, Mess3, Mess1, Mess4, WMETHO)) ;
            *Code = WARVMOD12;
            }
          } /* fin de NbBInf >0 */

        /* verifier  BSup */
        if (NbBSup >0)
          { 
          if (((bsup[i] != bsup[ii]) && !myisnan(bsup[i]) && !myisnan(bsup[ii])) || 
              (myisnan(bsup[i]) && !myisnan(bsup[ii])) ||
              (!myisnan(bsup[i]) && myisnan(bsup[ii])) )
            {
            sprintf (Mess1,"%d",(i+1));
            sprintf (Mess2,"%g",bsup[i]);
            sprintf (Mess3,"%d",(ii+1));
            sprintf (Mess4, "%g", bsup[ii]);
            /* Actions effectuees par la macro:
            printf ("La contrainte %s fixee sur le %s ieme parametre de %s est egale a %s, \n",
                   "BSup", Mess1, NomFonc, Mess2);
            printf("alors que celle fixee sur le %s parametre, parametre egal au %s ieme, est  %s\n",
                   Mess3, Mess1, Mess4);
            */
            NLWARNING ((WARVMOD12,8,"VICont","superiority-bound", Mess1, NomFonc, Mess2, Mess3, Mess1, Mess4, WMETHO)) ;
            *Code = WARVMOD12;
            }
          } /* fin de NbBSup >0 */

        break ; /* sortir de la boucle sur ii car s'il y a d'autres
                parametres egaux a i et ii, ceux-ci seront verifier par
                rapport a l'actuel ii dans les iterations suivantes de
                la boucle sur i */
        } /* fin de egalp[ii] == egalp[i] */
      }  /* fin boucle sur ii */
    } /* fin du cas il y a des EgalP */
  else
    {
    if ( NbEgalP == 0)
      {
      /* on  remplit EgalP s'il n'y en avait pas, par des indices
      de parametres croissant a partir de 1 (chaque parametre
      est egal a lui-meme) */
      egalp[i] = (i+1);
      }
    }
  } /* fin boucle sur i */

return(OK);
}

  

/*--------------- Identification fonction ----------
| Nom de la fonction    : NLVMod                   |
| Role                  : Initialiser et verifier  |
|  le modele en entree                             |
| Parametres d'entree   :                          |
|  NbCourbe: le nombre de courbes                  |
|  Poids: poids sur les observations               |
|   s'il n'y en a pas, c'est un vecteur a 0 element|
|  Theta0: valeurs initiales des Theta             |
|  Beta0: valeurs initiales des Beta               |
| Parametres d'e./s.    :                          |
|  Modele: la structure du modele                  |
| Parametres de sortie  :                          |
|  Code : OK ou code d'erreur                      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                        ChargerArb,               |
|                        VerifCont,                |
|                        VerifParam,               |
|                        VerifVari                 |
| Fonctions appelantes : NLVInit                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/

TShortInt NLVMod(NbCourbe, Poids, Theta0, Beta0,
                 Modele, Code)
/* ............... Arguments d'entree ............................ */
TShortInt NbCourbe;
TVect *Poids, *Theta0, *Beta0;

/* ............... Arguments d'entree-sortie ..................... */
TModele *Modele;

/* argu de sortie */
TShortInt *Code;

/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt NbTheta, NbBeta, NbValTheta, NbValBeta, NbGamF, NbGamV;
TShortInt MonCode;

/* Ecriture de la trace */
ECRTRACE("NLVMod");

/* initialisations */
MonCode =OK;
NbTheta = Modele->NomTheta.nbele;
NbBeta = Modele->NomBeta.nbele;
NbValTheta = NbTheta * NbCourbe;
NbValBeta = NbBeta * NbCourbe;
NbGamF = Modele->NomGamF.nbele;
NbGamV = Modele->NomGamV.nbele;


/* verification de Vari */
APPEL(VerifVari(Modele->YaCalcV,  NbBeta, Poids,   &(Modele->Vari),
  &MonCode));
if (MonCode !=OK)
  {
  /* je continue les verif mais je sauvegarde le code d'erreur */
  *Code = MonCode;
  }

/* verification de la dimension des parametres */
APPEL(VerifParam( NbValTheta, "f", Theta0,  &MonCode));
if (MonCode !=OK)
  {
  /* je continue les verif mais je sauvegarde le code d'erreur */
  *Code = MonCode;
  }

APPEL(VerifParam( NbValBeta, "v", Beta0,  &MonCode));

if (MonCode !=OK)
  {
  /*  je sauvegarde le code d'erreur */
  *Code = MonCode;
  }


/* verification de la dimension des parametres de second niveau */
APPEL(VerifPBis(  NbGamF, "f",  &(Modele->GamF),  &MonCode));
if (MonCode !=OK)
  {
  /* je continue les verif mais je sauvegarde le code d'erreur */
  *Code = MonCode;
  }

APPEL(VerifPBis(  NbGamV, "v",  &(Modele->GamV),  &MonCode));
if (MonCode !=OK)
  {
  /* je continue les verif mais je sauvegarde le code d'erreur */
  *Code = MonCode;
  }


if (*Code !=OK)
  {
  /* erreur de dimension: j'arrete la suite des verif sur le modele */
  return(OK);
  }

/* verification: si il y a des beta, il faut une fonction de calcul de v */
if ((NbBeta > 0) && (Modele->YaCalcV == FAUX))
  {
  NLWARNING ((WARVMOD13,1,"NLVMod", WMETHO)) ;
  *Code = WARVMOD13;
  }

/* verification des contraintes */
APPEL(VerifCont(NbValTheta, NbValBeta,Theta0, Beta0,
                &(Modele->CTheta0),  &(Modele->CBeta0), &MonCode));


if (MonCode !=OK)
  {
  *Code = MonCode;
  }

return(OK);
}

