/* -------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : ProgMath                  |
| Date                 : 1991                      |
| Derniere mise a jour :                           |
| Concepteur           : A. Bouvier                |
| Role                 : Programmes de calculs de  |
|  base sur vecteurs et matrices                   |
| PAR RAPPORT A LA VERSION NON INTERFACEE AVEC SPLUS
| On ote la fonction "matherr" qui provoque:       |
|  "multiply defined"                              |
| Reference conception :                           |
| Lecteur              :                           |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|.
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/

#include <math.h>

#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlmacros.h"
#include "errcodes.h"
#include "nlglobal.h"

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/
/* fonctions des autres modules */
TShortInt GerMessage();
void AffMess();

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/



/*--------------- Identification fonction ----------
| Nom de la fonction    : CopyMat                  |
| Role                  :  Recopie des valeurs     |
|   d'une matrice dans les elements d'une autre    |
| Parametres d'entree   :                          |
|  MatIn: matrice origine                          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  MatOut: matrice resultat                        |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  ProgIter, CDmct, CDmcb   |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt CopyMat (MatIn, MatOut)
/* ........ parametres d'entree:.................................. */
TMat *MatIn;
/* ........ parametres de sortie:.................................. */
TMat *MatOut;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
  TShortInt i, j, matinl, matinc;
/* pointeurs sur des elements de structure pour ameliorer la performance */
  TDouble **matin, **matout;


/* ........ algorithme : .................................................. */
  matinl = MatIn->nblig;
  matinc = MatIn->nbcol;
  matin = MatIn->donnees;
  matout = MatOut->donnees;


  for (i = 0 ; i <  matinl ; i++)
    {
    for (j = 0 ; j < matinc ; j++)
      {
      matout[i][j] = matin[i][j] ;
      }
    }
  return(OK);

}


/*--------------- Identification fonction ----------
| Nom de la fonction    : CopyVect                 |
| Role                  :  Recopie des valeurs     |
|   d'un vecteur dans les elements d'un autre      |
| Parametres d'entree   :                          |
|  VectIn: vecteur origine                         |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  VectOut: vecteur resultat                       |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  PussIter, CEtamco        |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt CopyVect(VectIn, VectOut)
/* ........ parametres d'entree:.................................. */
TVect *VectIn;
/* ........ parametres de sortie:.................................. */
TVect *VectOut;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i, vectinn;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *vectin, *vectout;


vectin = VectIn->donnees;
vectout = VectOut->donnees;
vectinn = VectIn->nbele;

/* COPIE D'UN VECTEUR DANS UN AUTRE */
for(i=0; i<vectinn; i++)
  {
  vectout[i]= vectin[i];
  }
return(OK);
}





/*--------------- Identification fonction ----------
| Nom de la fonction    : DivMatVal                |
| Role                  :  division des valeurs    |
| d'une matrice par une valeur TDouble             |
| Attention: on suppose que cette valeur n'est pas |
| nulle                                            |
| Parametres d'entree   :                          |
|  MatIn: matrice 1ier terme                       |
|  Val: valeur par laquelle on divise #0           |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  MatOut : matrice resultat                       |
| Retour fonction       : OK 
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  CCovNu                   |
-------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt DivMatVal(MatIn, Val, MatOut)

/* argument d'entree */
TMat *MatIn;
TDouble Val;

/* argument de sortie */
TMat *MatOut;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt  i, j, matinl, matinc;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble **matin, **matout;


  matinl = MatIn->nblig;
  matinc = MatIn->nbcol;
  matin = MatIn->donnees;
  matout = MatOut->donnees;

/* DIVISION D'UNE MATRICE PAR UNE VALEUR */

for (i = 0; i < matinl; i++)
  {
  for (j = 0 ; j < matinc; j++)
    {
    matout[i][j] = matin[i][j] / Val;
    }
  }
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : DivMatVect               |
| Role                  :  division de             |
|  chaque ligne d'une matrice par chaque element   |
|  d'un vecteur                                    |
| Attention: on suppose que le vecteur n'a         |
| pas de valeurs nulles (l'erreur peut etre, dans  |
| cas, recuperee par un mecanisme d'exception      |
| s'il existe)                                     |
| Parametres d'entree   :                          |
|  MatIn: matrice a diviser                        |
|  VectIn: vecteur par lequel on divise            |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  MatOut: matrice  resultat                       |
| Retour fonction       : OK
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  CVariance                |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt DivMatVect(MatIn, VectIn, MatOut)

/* argument d'entree */
TMat *MatIn;
TVect *VectIn;
/* argument de sortie */
TMat *MatOut;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
  TShortInt i,j,  matinc, vectinn;
/* pointeurs sur des elements de structure pour ameliorer la performance */
  TDouble **matin, **matout;
  TDouble *vectin;


  matinc = MatIn->nbcol;
  matin = MatIn->donnees;
  matout = MatOut->donnees;
  vectin = VectIn->donnees;
  vectinn = VectIn->nbele;


/* DIVISION DE CHAQUE LIGNE D'UNE MATRICE PAR CHAQUE ELEMENT D'UN VECTEUR*/

  for(i=0; i< vectinn; i++)
  {
  for(j=0;j<matinc; j++)
    {
    matout[i][j] =  matin[i][j] / vectin[i];
    }
  }
return(OK);
}




/*--------------- Identification fonction ----------
| Nom de la fonction    : DivVectVect              |
| Role                  :  division des valeurs    |
| d'un vecteur par les valeurs d'un autre          |
| Attention: on suppose que le 2ieme vecteur n'a   |
| pas de valeurs nulles (l'erreur peut etre, dans  |
| cas, recuperee par un mecanisme d'exception      |
| s'il existe)                                     |
| Parametres d'entree   :                          |
|  Vect1In: vecteur 1ier terme                     |
|  Vect2In: vecteur 2ieme terme                    | 
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  VectOut: vecteur resultat                       |
| Retour fonction       : OK
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  CVariance                |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt DivVectVect(Vect1In, Vect2In, VectOut)

/* argument d'entree */
TVect *Vect1In;
TVect *Vect2In;

/* argument de sortie */
TVect *VectOut;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i, vect1inn;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *vect1in, *vect2in, *vectout;

vect1in = Vect1In->donnees;
vect2in = Vect2In->donnees;
vectout = VectOut->donnees;
vect1inn = Vect1In->nbele;

/* DIVISION DE 2 VECTEURS */
for (i = 0; i < vect1inn; i++)
  {
  vectout[i] = vect1in[i] / vect2in[i];
  }
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : IMaxS                    |
| Role                  :  Renvoyer le maximum des |
|                          valeurs d'un vecteur    |
|                          TVectShort              |
| Parametres d'entree   :  pvect: pointeur sur un  |
|                          TVectShort, dont,       |
|                          eventuellement, le      |
|                          nombre d'elements est   |
|                          nul                     |
| Retour fonction       :  Le retour de la fonction|
|                          est le maximum des      |
|                          valeurs de pvect et     |
|                          UNDEF si celui-ci a 0   |
|                          elements                |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes :  MajContP, VEgalP         |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/


TShortInt IMaxS(pvect)
TVectShort *pvect;

/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt i, n; 
TShortInt maxi; /* valeur du maximum */
/* pointeurs sur des elements de structure pour ameliorer la performance */
TShortInt *vect;


if (pvect->nbele == 0)
  {
  return(UNDEF);
  }

n= pvect->nbele;
vect = pvect->donnees;
maxi = vect[0];
for (i = 1; i < n; i++)
  {
  maxi = MAX(maxi, vect[i]);
  }

return(maxi);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : IMinL                    |
| Role                  :  Renvoyer le minimum des |
|                          valeurs d'un vecteur    |
|                          TVectLong               |
| Parametres d'entree   :  pvect: pointeur sur un  |
|                          TVectLong , dont,       |
|                          eventuellement, le      |
|                          nombre d'elements est   |
|                          nul                     |
| Retour fonction       :  Le retour de la fonction|
|                          est le minimum des      |
|                          valeurs de pvect et     |
|                          UNDEF si celui-ci a 0   |
|                          elements                |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  EstTheta, NLVCtxPuss     |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/


TLongInt IMinL(pvect)
TVectLong *pvect;

/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt i; /* indice */
TLongInt mini; /* valeur du minimum */
TShortInt n;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TLongInt *vect;

if (pvect->nbele==0)
  {
  return(UNDEF);
  }

vect = pvect->donnees;
n = pvect->nbele;

mini=vect[0];
for (i=1; i<n; i++)
  {
  if (vect[i]< mini) mini=vect[i];
  }

return(mini);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : InvMat                   |
| Role                  :  Inversion de matrice    |
|                          symetrique              |
| Parametres d'entree   :                          |
| Parametres d'e/s      :                          |
|  Mat: matrice a inverser en entree, et matrice   |
|       inversee en sortie                         |
| Parametres de sortie  :                          |
| Retour fonction       : OK ou ERRCALC            |
| Reference conception  :                          |
|  voir bouquin: bibliotheque mathematique en C ,  |
|   programme minv, p34                            |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune, mais la fonction |
|  systeme fabs                                    |
| Fonctions appelantes :                           |
|                        CArret,                   |
|                        CCovNu, CCovNu2, CCovNu3, |
|                        InvSb                     |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt InvMat(Mat)
TMat *Mat;
/*--------------- Fin identification fonction ----*/

{
 /* INVERSION DE MATRICE SANS RECHERCHE DE PIVOT */

/* locals */
TShortInt matl;
TShortInt i, j , k;
TDouble pv, fabs();
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble **mat;


matl = Mat->nblig;
mat = Mat->donnees;

for (k = 0 ; k < matl; k++)
  {
  pv = mat[k][k];
  if (fabs(pv) < MINSG)
    {
      return(ERRCALC);
    }

  mat[k][k]=(TDouble)1;
  for (j = 0; j < matl; j++)
    {
    mat[k][j] /= pv;
    }
  for (i = 0; i < matl; i++)
    {
    if (i != k)
      {
       pv = mat[i][k];
       mat[i][k]= (TDouble)ZERO;
       for (j = 0; j < matl; j++)
         {
         mat[i][j] -= pv * mat[k][j];
         }
      }  
    }
  }
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : MultMat                  |
| Role                  :  produit matriciel       |
|  en ignorant les elements sur les dimensions qui |
|  provoqueraient un debordement de tableaux       |
| Parametres d'entree   :                          |
|   Mat1In: matrice 1ier facteur                   |
|   Mat2In: matrice 2ieme facteur                  |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|   MatOut: matrice resultat                       |
|     allouee avant l'appel                        |
| Retour fonction       : OK
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :                           |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt MultMat (Mat1In, Mat2In, MatOut)
/* ........ parametres d'entree:.................................. */
TMat *Mat1In , *Mat2In;
/* ........ parametres de sortie:.................................. */
TMat *MatOut;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
  TShortInt i, j ,k, iborne, jborne, kborne;
  TDouble Somme;
/* pointeurs sur des elements de structure pour ameliorer la performance */
  TDouble **mat1in, **mat2in, **matout;


  iborne = MIN(Mat1In->nblig, MatOut->nblig);
  jborne = MIN(Mat2In->nbcol, MatOut->nbcol);
  kborne = MIN( Mat1In->nbcol, Mat2In->nblig);

  mat1in = Mat1In->donnees;
  mat2in = Mat2In->donnees;
  matout = MatOut->donnees;

  for (i = 0 ; i < iborne ; i++)
    {
    for (j = 0 ; j < jborne ; j++)
      {
      Somme=(TDouble)ZERO;
      for (k = 0 ; k < kborne ; k++)
        {
        Somme = Somme + (mat1in[i][k] * mat2in[k][j]);
        }
      matout[i][j] = Somme;
      }
    }
  return(OK);

}


/*--------------- Identification fonction ----------
| Nom de la fonction    : MultMatT                 |
| Role                  :  produit d'une matrice   |
|  par la transposee d'une autre                   |
|  en ignorant les elements sur les dimensions qui |
|  provoqueraient un debordement de tableaux       |
| Parametres d'entree   :                          |
|   Mat1In: matrice 1ier facteur                   |
|   Mat2In: matrice 2ieme facteur                  |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|   MatOut: matrice resultat                       |
|     allouee avant l'appel                        |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :                           |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt MultMatT (Mat1In, Mat2In, MatOut)
/* ........ parametres d'entree:.................................. */
TMat *Mat1In , *Mat2In;
/* ........ parametres de sortie:.................................. */
TMat *MatOut;

/*--------------- Fin identification fonction ----*/

{


/* ........ variables internes : .......................................... */
  TShortInt i, j ,k, iborne, jborne, kborne;
  TDouble Somme;
/* pointeurs sur des elements de structure pour ameliorer la performance */
  TDouble **mat1in, **mat2in, **matout;


  iborne = MIN(Mat1In->nblig, MatOut->nblig);
  jborne = MIN(Mat2In->nblig, MatOut->nbcol);
  kborne = MIN( Mat1In->nbcol, Mat2In->nbcol);

  mat1in = Mat1In->donnees;
  mat2in = Mat2In->donnees;
  matout = MatOut->donnees;

  for (i = 0 ; i < iborne ; i++)
    {
    for (j = 0 ; j < jborne ; j++)
      {
      Somme=(TDouble)ZERO;
      for (k = 0 ; k < kborne ; k++)
        {
        Somme = Somme + (mat1in[i][k] * mat2in[j][k]);
        }
      matout[i][j] = Somme;
      }
    }
  return(OK);

}

/*--------------- Identification fonction ----------
| Nom de la fonction    : MultMat1                 |
| Role                  :  produit matriciel       |
|   de matrices carrees de meme dimension          |
|   avec resultat dans la 1iere matrice            |
| Parametres d'entree   :                          |
|  Mat2: matrice 2ieme facteur du produit          |
| Parametres d'e/s      :                          |
|  Mat1: matrice 1ier facteur du produit           |
|  Trav: tableau de travail dont la diemnsion doit |
|        etre egale au moins au nombre de colonnes |
|        de Mat1                                   |
| Parametres de sortie  :                          |
| Retour fonction       : OK
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  ProgIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt MultMat1(Mat1, Mat2, Trav)
/* argument d'entree-sortie */
TMat *Mat1;
TVect *Trav; /* tableau de travail de dimension au moins egal a
                Mat1.nbcol */
/* argument d'entree */
TMat *Mat2;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i,j,j0, mat1l, mat1c, mat2c;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble **mat1, **mat2;
TDouble *trav;



mat1l = Mat1->nblig;
mat1c = Mat1->nbcol;
mat2c = Mat2->nbcol;
mat1 = Mat1->donnees;
mat2 = Mat2->donnees;
trav = Trav->donnees;

/* MULTIPLICATION MATRICIELLE AVEC RESULTAT DANS LA 1IERE MATRICE */

for (i = 0; i < mat1l; i++)
  {
  for (j0 = 0; j0 < mat2c; j0++)
    {
    trav[j0] = (TDouble)ZERO;
    for (j = 0; j < mat1c; j++)
      {
      trav[j0] = trav[j0] + (mat1[i][j] * mat2[j][j0]);
      }
    }
  for (j0 = 0; j0 < mat2c; j0++)
    {
    mat1[i][j0] = trav[j0];
    }
  }
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : MultMat2                 |
| Role                  :  produit matriciel       |
|   de matrices carrees de meme dimension          |
|   avec resultat dans la 2ieme matrice            |
| Parametres d'entree   :                          |
|  Mat1: matrice 1ier facteur du produit           |
| Parametres d'e/s      :                          |
|  Mat2: matrice 2ieme facteur du produit          |
|  Trav: tableau de travail dont la diemnsion doit |
|        etre egale au moins au nombre de colonnes |
|        de Mat1                                   |
| Parametres de sortie  :                          |
| Retour fonction       : OK
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  ProgIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt MultMat2(Mat1, Mat2, Trav)
/* argument d'entree */
TMat *Mat1;
/* argument d'entree-sortie */
TMat *Mat2;
TVect *Trav; /* tableau de travail de dimension au moins egal a
                Mat1.nbcol */

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i,j,j0, mat1l, mat1c, mat2l, mat2c;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble **mat1, **mat2;
TDouble *trav;



mat1l = Mat1->nblig;
mat1c = Mat1->nbcol;
mat2c = Mat2->nbcol;
mat2l = Mat2->nblig;
mat1 = Mat1->donnees;
mat2 = Mat2->donnees;
trav = Trav->donnees;

/* MULTIPLICATION MATRICIELLE AVEC RESULTAT DANS LA 2IEME MATRICE */

for (j0 = 0; j0 < mat2c; j0++)
  {
  for (i = 0; i < mat1l; i++)
    {
    trav[i] = (TDouble)ZERO;
    for (j = 0; j < mat1c; j++)
      {
      trav[i] = trav[i] + (mat1[i][j] * mat2[j][j0]);
      }
    }
  for ( i = 0; i < mat2l; i++)
    {
    mat2[i][j0] = trav[i];
    }
  }
return(OK);
}
 


/*--------------- Identification fonction ----------
| Nom de la fonction    : MultMatVal               |
| Role                  :  multiplication d'une    |
|  matrice par une valeur                          |
| Parametres d'entree   :                          |
|  MatIn: matrice 1ier facteur du produit          |
|  Valeur: valeur 2ieme facteur du produit         |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  MatOut: matrice  resultat                       |
| Retour fonction       : OK
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  CModele, ProgIter        |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt MultMatVal(MatIn, Valeur, MatOut)

/* argument d'entree */
TMat *MatIn;
TDouble Valeur;
/* argument de sortie */
TMat *MatOut;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i,j, matinl, matinc;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble **matin, **matout;



  matinl = MatIn->nblig;
  matinc = MatIn->nbcol;
  matin = MatIn->donnees;
  matout = MatOut->donnees;


/* MULTIPLICATION D'UNE MATRICE PAR UNE VALEUR */

for (i = 0; i < matinl; i++)
  {
  for (j = 0; j < matinc; j++)
    {
    matout[i][j] = matin[i][j] * Valeur;
    }
  }
return(OK);
}





/*--------------- Identification fonction ----------
| Nom de la fonction    : MultVectMat              |
| Role                  :  multiplication d'un     |
|   vecteur par une matrice                        |
| Parametres d'entree   :                          |
|  VectIn: vecteur 1ier facteur du produit         |
|  MatIn: matrice 2ieme facteur du produit         |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  VectOut: vecteur resultat                       |
| Retour fonction       : OK
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  ProgIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt MultVectMat(VectIn, MatIn, VectOut)

/* argument d'entree */
TVect *VectIn;
TMat *MatIn;
/* argument de sortie */
TVect *VectOut;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
  TShortInt i,j,  matinc, vectinn;
/* pointeurs sur des elements de structure pour ameliorer la performance */
 TDouble **matin;
 TDouble *vectin, *vectout;


  matinc = MatIn->nbcol;
  matin = MatIn->donnees;
  vectin = VectIn->donnees;
  vectout = VectOut->donnees;
  vectinn = VectIn->nbele;


/* MULTIPLICATION D'UN VECTEUR PAR UNE MATRICE */
for (j = 0; j < matinc; j++)
  {
  vectout[j] = (TDouble)ZERO;
  for ( i = 0; i < vectinn; i++)
    {
    vectout[j] = vectout[j] + (vectin[i] * matin[i][j]);
    }
  }
return(OK);
}




/*--------------- Identification fonction ----------
| Nom de la fonction    : MultVectVal              |
| Role                  :  multiplication d'un     |
|  vecteur par une valeur                          |
| Parametres d'entree   :                          |
|  VectIn: vecteur 1ier facteur                    |
|  Val: valeur 2ieme terme                         |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  VectOut: vecteur resultat                       |
| Retour fonction       : OK
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  CModele                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt MultVectVal(VectIn, Valeur, VectOut)

/* argument d'entree */
TVect *VectIn;
TDouble Valeur;
/* argument de sortie */
TVect *VectOut;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i, vectinn;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *vectin, *vectout;


vectin = VectIn->donnees;
vectout = VectOut->donnees;
vectinn = VectIn->nbele;

/* MULTIPLICATION D'UN VECTEUR PAR UNE VALEUR */
for (i = 0; i < vectinn; i++)
  {
  vectout[i] = vectin[i] * Valeur;
  }
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : MultVectVect2            |
| Role                  :  multiplication d'un     |
|  vecteur par un vecteur, resultat dans le 2ieme  |
| Parametres d'entree   :                          |
|  VectIn: vecteur 1ier facteur                    |
| Parametres d'e/s      :                          |
|  VectOut: vecteur resultat                       |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  Init3MCMPT               |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt MultVectVect2(VectIn, VectOut)

/* argument d'entree */
TVect *VectIn;

/* argument de e/sortie */
TVect *VectOut;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i, vectinn;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *vectin, *vectout;


vectin = VectIn->donnees;
vectout = VectOut->donnees;
vectinn = VectIn->nbele;

/* MULTIPLICATION D'UN VECTEUR PAR UN VECTEUR */
for (i = 0; i < vectinn; i++)
  {
  vectout[i] = vectin[i] * vectout[i];
  }
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : MVect                    |
| Role                  :  multiplication de 2     |
|  vecteurs, le 1ier considere comme une matrice   |
|  ligne et le second, comme une matrice colonne   |
| Parametres d'entree   :                          |
|  Vect1In: vecteur 1ier facteur du produit        |
|  Vect2In: vecteur 2ieme facteur du produit       |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ValOut: valeur   resultat                       |
| Retour fonction       : OK
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  aucune                   |
| Fonctions appelantes :  ProgIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt MVect(Vect1In, Vect2In, ValOut)
/* argument d'entree */
TVect *Vect1In, *Vect2In;
/* argument de sortie */
TDouble *ValOut;

/*--------------- Fin identification fonction ----*/

{

/* ........ variables internes : .......................................... */
TShortInt i, vect1inn;
/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *vect1in, *vect2in;


vect1in = Vect1In->donnees;
vect2in = Vect2In->donnees;
vect1inn = Vect1In->nbele;

/* MULTIPLICATION DE 2 VECTEURS, CONSIDERES COMME DES MATRICES LIGNE ET COLONNE */

*ValOut=(TDouble)ZERO;

for (i = 0; i < vect1inn; i++)
  {
  *ValOut = *ValOut + (vect1in[i] * vect2in[i]);
  }
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : SysLin                   |
| Role                  :  Resolution d'un systeme |
|  lineaire A.X = B par la methode QR avec pivot   |
|  sur les colonnes, comprenant la detection       |
|  eventuelle de matrice singuliere                |
| Parametres d'entree   :                          |
|  m: nombre de lignes des matrices A et B         |
|  n: nombre de colonnes de la matrice A =         |
|     nombre de lignes de la matrice X             |
|  o: nombre de colonnes des matrices X et B       |
|  epsilon: plus petit pivot "acceptable"          |
| Parametres d'e/s      :                          |
|  TTa: Contenu successif des matrices :           |
|       A (la matrice premier membre du systeme a  |
|          resoudre)                               |
|       Q1.A.P1 = R1                               |
|       Q2.Q1.A.P1.P2 = R2                         |
|       Qr...Q2.Q1.A.P1.P2...Pr = Rr = Q.A.P = R   |
|  TTb:  Contenu successif des matrices :          |
|       B (la matrice second membre du systeme a   |
|          resoudre)                               |
|       Q1.B                                       |
|       Q2.Q1.B                                    |
|       Qr...Q2.Q1.B = Q.B                         |
| Parametres de sortie  :                          |
|  TTx: Contenu de la matrice X ("solution" du     |
|       systeme)                                   |
|  Prang: Rang de la matrice A                     |
| Parametres de travail :                          | 
|  Tv: Contenu successif des vecteurs de longueur  |
|      au plus m                                   |
|      V1, V2, ..., Vr associes aux matrices de    |
|                   Householder                    |
|      Q1, Q2, ... Qr (Vi est de longueur m-i+1)   |
|  Tgamma: Contenu successif des vecteurs de       |
|          longueur au plus n                      |
|          G1, G2, ..., Gr  permettant de          |
|          determiner les pivots                   |
|          (Gi contient les normes des n-i+1       |
|           dernieres colonnes de la matrice Ri)   |
|  Tc: Contenu successif des vecteurs de longueur n|
|      permettant de definir les permutations de   |
|      colonnes associees aux matrices             |
|      P1, P1.P2, ..., P1.P2...Pr                  |
| Retour fonction       : OK ou ERRSYS             |
| Reference   :                                    |
|    GOLUB - VAN LOAN                              |
|    Matrix computations                           | 
|    North Oxford Academic 1983                    |
|    Algorithme 6.4-1.                             |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  fonctions systeme:       |
|                       sqrt,fabs                  |
| Fonctions appelantes :  CDirec                   |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/
TShortInt SysLin (m, n, o, epsilon, 
                  TTa, TTb, 
                  TTx, Prang, 
                  Tv, Tgamma, Tc)

/* ........ parametres d'entree:.................................. */
 TShortInt m, n, o ;
 TDouble epsilon ;
/* ........ parametres d'e/s:.................................. */
TDouble **TTa, **TTb ;
/* ........ parametres de sortie:.................................. */
TDouble **TTx ;
TShortInt *Prang ;
/* ........ parametres de travail:.................................. */
TDouble *Tv, *Tgamma ;
 TShortInt *Tc ;

/*--------------- Fin identification fonction ----*/

{
 /* ...................................................................... */
 /* ........ Principe : Remplacement de la resolution du systeme A.X = B   */
 /* ........            par celle du systeme : Q.A.P.inv(P).X = Q.B        */
 /* ........            dans lequel, soit r le rang de la matrice A,       */
 /* ........            Q est une matrice orthogonale obtenue par produit  */
 /* ........            de r matrices de Householder, P est une matrice de */
 /* ........            permutation de colonnes obtenue par produit d'au   */
 /* ........            plus r transpositions, et le produit Q.A.P = R est */
 /* ........            une matrice triangulaire superieure dont seules les*/
 /* ........            r premieres lignes sont considerees comme non nulles*/
 /* ...................................................................... */

 /* ........ variables internes : ..........................................*/
    TDouble pivot, tr, max, alpha, beta, sum, x ;
    TShortInt itr, i, j, k, p , Code;

 

 /* ........ algorithme : ................................................. */
    Code =OK;
    *Prang = n ;
 /* ........ initialisation a l'identite de la permutation ................ */
 /* ........ stockee dans le vecteur Tc : ................................. */
    for (j = 0 ; j < n ; j++)
    {
       Tc[j] = j ;
    }  /* fait (j) */
 /* ........ calcul des carres des normes des colonnes de la matrice TTa :  */
    for (j = 0 ; j < n ; j++)  
    {
       Tgamma[j] = (TDouble)ZERO ;
       for (i = 0 ; i < m ; i++)
       {
	  Tgamma[j] += TTa[i][j] * TTa[i][j] ;
       }  /* fait (i) */
    }  /* fait (j) */
    for (k = 0 ; k < n ; k++)
    {  
 /* ........ determination du pivot (colonne de norme maximale) : ......... */
       pivot = (TDouble)ZERO ;
       for (j = k ; j < n ; j++)
       {
	  if (Tgamma[j] > pivot)
	  {
	     pivot = Tgamma[j] ;
	     p     = j ;
	  }  /* finsi */
       }  /* fait (j) */
 /* ........ arret lorsque le pivot est trop petit : ...................... */
       if (pivot < epsilon)
       {
	  *Prang = k ;
          Code = ERRSYS;
	  break ;
       }  /* finsi */
 /* ........ echange des elements k et p des vecteurs Tc et Tgamma ........ */
 /* ........ et des colonnes k et p de la matrice TTa : ................... */
       itr = Tc[k] ;
       Tc[k] = Tc[p] ;
       Tc[p] = itr ;
       tr  = Tgamma[k] ;
       Tgamma[k] = Tgamma[p] ;
       Tgamma[p] = tr ;
       for (i = 0 ; i < m ; i++)
       {
	  tr = TTa[i][k] ;
	  TTa[i][k] = TTa[i][p] ;
	  TTa[i][p] = tr ;
       }  /* fait (i) */
 /* ........ determination du vecteur Tv = (0, ..., 0, vk, ..., vm-1)  .... */
 /* ........ generateur de la matrice de Householder Qk+1 telle que    .... */
 /* ........ les composants allant de k+1 a m-1 du produit de Qk+1 par .... */
 /* ........ la colonne k de la matrice TTa soient nuls et calcul de   .... */
 /* ........ beta = 2 / t(Tv).Tv :                                     .... */
       max = 0. ;
       for (i = k ; i < m ; i++)
       {
	  if (fabs (TTa[i][k]) > max)
	  {
	     max = fabs (TTa[i][k]) ;
	  }  /* finsi */
       }  /* fait (i) */
       alpha = (TDouble)ZERO ;
       for (i = k ; i < m ; i++)
       {
	  Tv[i] = TTa[i][k] / max ;
	  alpha += Tv[i] * Tv[i] ;
       }  /* fait (i) */
       alpha = sqrt (alpha) ;
       beta  = 1. / (alpha * (alpha + fabs (Tv[k]))) ;
       if (Tv[k] >= (TDouble)ZERO)
       {
	  Tv[k] += alpha ;
       }  /* fin alors */
       else
       {
	  Tv[k] -= alpha ;
       }  /* fin sinon */
 /* ........ remplacement de la matrice TTa par le produit Qk+1.TTa :       */
 /* ........ (pour i = k, ... m-1 et j = 0, ..., k-1 on a : TTa[i][j] = 0.) */
       for (j = k ; j < n ; j++)
       {
	  sum = (TDouble)ZERO ;
	  for (i = k ; i < m ; i++)
	  {
	     sum += Tv[i] * TTa[i][j] ;
	  }  /* fait (i) */
	  sum *= beta ;
	  for (i = k ; i < m ; i++)
	  {
	     TTa[i][j] -= sum * Tv[i] ;
	  }  /* fait (i) */
       }  /* fait (j) */
 /* ........ remplacement de la matrice TTb par le produit Qk+1.TTb :        */
       for (j = 0 ; j < o ; j++)
       {
	  sum = (TDouble)ZERO  ;
	  for (i = k ; i < m ; i++)
	  {
	     sum += Tv[i] * TTb[i][j] ;
	  }  /* fait (i) */
	  sum *= beta ;
	  for (i = k ; i < m ; i++)
	  {
	     TTb[i][j] -= sum * Tv[i] ;
	  }  /* fait (i) */
       }  /* fait (j) */
 /* ........ mise a jour des carres des normes des colonnes ............... */
 /* ........ de la matrice TTa non encore traitees :        ............... */
       for (j = k+1 ; j < n ; j++)
       {
	  Tgamma[j] -= TTa[k][j] * TTa[k][j] ;
       }  /* fait (j) */
    }  /* fait (k) */
 /* ....... resolution du systeme R.inv(P).X = Q.B : ...................... */
 /* ....... (TTa contient R et TTb contient Q.B)     ...................... */
    for (j = 0 ; j < o ; j++)
    {
       for (i = n-1 ; i >= *Prang ; i--)
       {
	  TTx[Tc[i]][j] = (TDouble)ZERO ;
       }  /* fait (i) */
       for (i = *Prang-1 ; i >= 0 ; i--)
       {
	  x = TTb[i][j] ;
	  for (k = i+1 ; k < n ; k++)
	  {
	     x -= TTa[i][k] * TTx[Tc[k]][j] ;
	  }  /* fait (k) */
	  TTx[Tc[i]][j] = x / TTa[i][i] ;
       }  /* fait (i) */
    }  /* fait (j) */
 /* ....................................................................... */


return(Code);
}

