/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : PussIter                  |
| Date                 : 1991                      |
| Derniere mise a jour : %e%     / %u%             |
| Concepteur           : A. Bouvier                |
| Role                 : Effectuer le processus    |
|  iteratif de l'estimation                        |
| Reference conception :                           |
| Lecteur              : G. Husser                 |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|%c%--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <float.h>
/* Utilise par mysignaling_nan qui est dans nlmacros.h */
#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlmacros.h"
#include "errcodes.h"
#include "nlglobal.h"

/*--------------- MACROS -------------*/

/* On remplace la macro APPEL par celle-ci dans PussIter
de facon a ce que les structures locales soint desallouees */
#define APPELP(NLFonc)       \
  { \
  TShortInt ErrCode; \
  ErrCode=NLFonc ;   \
  if(ErrCode!=OK) \
    { \
    DesalloueTA (&Trav1, &Trav2, &Trav3, &TravS, \
     &MatTrav,  &MatTrav1,  &MatTrav2, &BlocP, &BlocQ, &Bloc0, &BlocR); \
    DesalloueT(  &ThetaCour, &BetaCour,  &ValR, &Direc, &Correc, \
                   &DEffT, &DEffB, &FctSA); \
    return(ErrCode);  \
    } \
  }



/*--------------- FONCTIONS EXTERNES -------------*/
/* extern char *calloc(); */
TDouble fabs();
/* fonctions des autres modules */
TShortInt  CArret(), CopyVect(),
           CLv(), 
           CModele(), CModErr(),
           CPasOpt(),
           CScrNP(), CScrPv(), CScrPS2(), CScr2NP(),
           CritSig(),
           CValR(), CValW(),
           EcrMat(), EcrVect(), 
           MultVectVal();

TShortInt CreerMat( TShortInt nblig, TShortInt nbcol, TMat *pmat);
TShortInt CreerMatC( TShortInt nblig, TShortInt nbcol, TMat *pmat);
TShortInt CreerVect( TShortInt nbele, TVect *pvect);
TShortInt CreerVectShort(TShortInt nbele, TVectShort *pvect);
TShortInt CopyMat (TMat *MatIn, TMat *MatOut);

void DetruMat(TMat *pmat);
void DetruVect(TVect *pvect);
void DetruVectShort(TVectShort *pvect);
TShortInt GerMessage( );
TShortInt calcc(TDouble Sigma2, TLongInt NbObsT, TVectLong *NbRepet,
		TVect *Y1, TVect *Y2, TVect *S2, TVect *Valf, TVect *VarY,
                TDouble *Crit);

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueS                  |
| Role                  :  Allouer les structures  |
|     de sortie de PussIter                        |
| Parametres d'entree   :                          |
|    NbZO: dimension totale des statistiques       |
|          exhaustives (=NbZ*NbObs)                |
|    NbObs: nombre d'observations                  |
|    Theta, Beta: structures des parametres        |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|    ValW: matrice W                               |
|    Ajustes: les ajustes                          |
|    EquN: les equations normales                  |
|    FctSensib: les fonctions de sensibilite       |
| Retour fonction       : OK ou le code d'erreur   |
|  des fonctions appelees                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CreerMat, CreerMatC,     |
|                         CreerVect                |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/


TShortInt AlloueS ( NbZO, NbObs,
                   Theta, Beta,
                   ValW, Ajustes, EquN, FctSensib)


/* arguments d'entree */
TShortInt    NbZO;
TLongInt NbObs;
TParam *Theta, *Beta;

/* arguments de sortie */
TMat *ValW;
TAjustes *Ajustes;
TEquN *EquN;
TFctSensib *FctSensib;


/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt NbAct, NbBeta, NbTheta;


/* Ecriture de la trace */
ECRTRACE("AlloueS");

/* ................................................................. */
/* determination des dimensions */
/* ................................................................. */
NbAct = Theta->Act.nbele + Beta->Act.nbele; /* nombre total de parametres actifs */
NbTheta =  Theta->Noms.nbele; /* nombre de parametres du modele */
NbBeta =  Beta->Noms.nbele;

/* Allocation des   Ajustes */
APPEL(CreerVect((TShortInt)NbObs, &(Ajustes->Valf)));
APPEL(CreerVect( (TShortInt)NbObs, &(Ajustes->VarY)));

/* Allocation des equations normales */
APPEL(CreerMat(NbZO, NbAct, &(EquN->ValB)));
APPEL(CreerMat(NbZO, NbAct, &(EquN->ValD)));
APPEL(CreerVect( NbZO, &(EquN->ValEta)));

/*  Allocation des FctSensib */
APPEL(CreerMatC( (TShortInt)NbObs, NbTheta, &(FctSensib->DValf)));
APPEL(CreerMatC( (TShortInt)NbObs, NbTheta, &(FctSensib->DVarYTheta)));
APPEL(CreerMatC((TShortInt)NbObs,  NbBeta, &(FctSensib->DVarYBeta)));
/* si le nombre de beta=0, mettre a 0 le nombre de lignes, car c'est
equivalent a structure vide */
if (NbBeta ==0) 
  {
  FctSensib->DVarYBeta.nblig = 0;
  }


/*  Allocation de   ValW */
APPEL(CreerMat(NbAct, NbAct, ValW));

      
/* retour */
return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueSedo               |
| Role                  :  Allouer les structures  |
|     de sortie du Sedo                            |
|     Fonction appelee que si CasSedo =VRAI        |
| Parametres d'entree   :                          |
|    NbObs:  nombre d'observations                 |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|    Sedo: les valeurs du Sedo                     |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  fonction systeme: calloc |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/


TShortInt AlloueSedo ( TLongInt NbObs,
                       TSedo *Sedo)


/* arguments d'entree 
TLongInt NbObs;

arguments de sortie
TSedo *Sedo;
*/


/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt  i, j, Dim4DF;

/* Ecriture de la trace */
  ECRTRACE("AlloueSedo");

  Dim4DF = GNLCtxInteg.LongSys - GNLCtxInteg.NbEq;

  Sedo->FSedo = (TDouble ***) calloc ((unsigned) NbObs, sizeof (TDouble **));
  Sedo->DFSedo = (TDouble ***) calloc ((unsigned) NbObs, sizeof (TDouble **));

  for (i = 0; i < NbObs; i++)
    {
    Sedo->FSedo[i] = (TDouble **)  
       calloc ((unsigned) GNLCtxInteg.NbJ, sizeof (TDouble *));
    Sedo->DFSedo[i] = (TDouble **) 
       calloc ((unsigned) GNLCtxInteg.NbJ, sizeof (TDouble *));
    for (j =0; j < GNLCtxInteg.NbJ; j++)
      {
      Sedo->FSedo[i][j] = (TDouble *)
        calloc ((unsigned) GNLCtxInteg.NbEq, sizeof (TDouble ));
      Sedo->DFSedo[i][j] = (TDouble *)
        calloc ((unsigned) Dim4DF, sizeof (TDouble ));
      } /* fin boucle sur j */
    } /* fin boucle sur i */


/* retour */
return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueT                  |
| Role                  :  Allouer les structures  |
|     de travail de PussIter                       |
| Parametres d'entree   :                          |
|  NbThetaA, NbBetaA: nombre de parametres Theta   |
|   actifs et de parametres Beta actifs            |
|  NbThetaE, NbBetaE: nombre de parametres Theta   |
|   differents et de parametres Beta differents    |
|  NbObs: nombre d'observations                    |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ThetaCour, BetaCour: vecteurs des valeurs       |
|   courantes des parametres                       |
|  ValR: Vecteur de travail                        |
|  Direc: Vecteur direction de descente            |
|  Correc  : Vecteur correction de la direction de |
|     descente                                     |
|  DEffT, DEffB: derivees par rapport aux Theta et |
|    Beta                                          |
|  FctSA: fonctions de sensibilite en dimension    |
|    "actifs"                                      |
| Retour fonction       : OK ou le code d'erreur   |
|  des fonctions appelees                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CreerMat, CreerVect      |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/



TShortInt AlloueT ( NbThetaA, NbBetaA, NbThetaE, NbBetaE, NbObs,
                  ThetaCour,BetaCour,ValR, Direc, Correc,
                  DEffT, DEffB, FctSA)

/* arguments d'entree */
TShortInt NbThetaA, NbBetaA, NbThetaE, NbBetaE;
TLongInt NbObs;

/* arguments de sortie */
TVect *ThetaCour, *BetaCour;
TVect *ValR, *Direc, *Correc;
TMat *DEffT, *DEffB;
TFctSensib *FctSA;

/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt NbAct;


/* Ecriture de la trace */
ECRTRACE("AlloueT");

NbAct = NbThetaA + NbBetaA;

/* Allocation des  ThetaCour, BetaCour */
APPEL(CreerVect(NbThetaA, ThetaCour));
APPEL(CreerVect(NbBetaA, BetaCour));
/* Remarque: BetaCour existe toujours, meme s'il n'y a pas de parametres Beta, 
mais le nombre de ses elements est alors nul */


/* Allocation de ValR */
APPEL(CreerVect(NbAct, ValR));

/* Allocation des Direc et Correc  */
APPEL(CreerVect(NbAct , Direc));
APPEL(CreerVect(NbAct , Correc));


/*  Allocation des FctSensib  en dimension actifs*/
APPEL(CreerMat((TShortInt)NbObs, NbThetaA, &(FctSA->DValf)));
APPEL(CreerMat( (TShortInt)NbObs, NbThetaA, &(FctSA->DVarYTheta)));
APPEL(CreerMat( (TShortInt)NbObs, NbBetaA, &(FctSA->DVarYBeta)));
/* si le nombre de beta=0, mettre a 0 le nombre de lignes, car c'est
equivalent a structure vide */

if (NbBetaA ==0) 
  {
  FctSA->DVarYBeta.nblig = 0;
  }

/*  Allocation de DEffT, DEffB */
APPEL(CreerMat( (TShortInt)NbObs,NbThetaE, DEffT));
APPEL(CreerMat( (TShortInt)NbObs,NbBetaE, DEffB));


/* retour */
return(OK);
}






/*--------------- Identification fonction ----------
| Nom de la fonction    : AlloueTA                 |
| Role                  :  Allouer les structures  |
|     de travail des fonctions appelees par        |
|      PussIter                                    |
| Parametres d'entree   :                          |
|  Algo: algorithme                                |
|  Symm: indique si ValW est symetrique            |
|  NbThetaA, NbBetaA: nombre de parametres Theta   |
|   actifs et de parametres Beta actifs            |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  Trav1,Trav2, Trav3: vecteurs de travail NbAct   |
|   de type TVect                                  |
|  TravS: vecteur de travail NbAct de type         |
|    VectShort                                     |
|  MatTrav: matrice NbAct, NbAct                   |
|   (NbAct: NbThetaA+NbBetaA),                     |
|  MatTrav1:  matrice NbAct, 1                     |
|  MatTrav2: matrice NbAct, 1                      |
|  BlocP: matrice NbThetaA, NbThetaA si SYMBLOC,   |
|         dimension 0,0 sinon                      |
|  BlocQ: matrice NbBetaA, NbThetaA si SYMBLOC,    |
|         dimension 0,0 sinon                      |
|  Bloc0: matrice NbThetaA, NbBetaA si SYMBLOC,    |
|         dimension 0,0 sinon                      |
|  BlocR: matrice NbBetaA, NbBetaA si SYMBLOC,     |
|         dimension 0,0 sinon                      |
| Retour fonction       : OK ou le code d'erreur   |
|  des fonctions appelees                          |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CreerMat, CreerVect      |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/



TShortInt AlloueTA( Algo, Symm, NbThetaA, NbBetaA,
                   Trav1, Trav2, Trav3, TravS,
                   MatTrav,MatTrav1, MatTrav2, BlocP, BlocQ, Bloc0, BlocR)


/* arguments d'entree */
TShortInt Algo, Symm;
TShortInt NbThetaA, NbBetaA;

/* arguments de sortie */
TVect *Trav1, *Trav2, *Trav3;
TVectShort *TravS;
TMat *MatTrav, *MatTrav1, *MatTrav2, *BlocQ, *BlocR, *Bloc0, *BlocP;

/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt NbAct;

/* Ecriture de la trace */
ECRTRACE("AlloueTA");

/* determination du nombre total de parametres actifs */
NbAct= NbThetaA + NbBetaA;

/*  MatTrav */
APPEL(CreerMat(NbAct, NbAct, MatTrav));

/* pour la decomposition en blocs de ValW */
if(Symm == SYMBLOC)
  {
  APPEL(CreerMat(NbThetaA, NbThetaA, BlocP));
  APPEL(CreerMat(NbBetaA, NbThetaA, BlocQ));
  APPEL(CreerMat(NbThetaA, NbBetaA, Bloc0));
  APPEL(CreerMat(NbBetaA, NbBetaA, BlocR));
  }
else
  {
  APPEL(CreerMat(0, 0, BlocP));
  APPEL(CreerMat(0, 0, BlocQ));
  APPEL(CreerMat(0, 0, Bloc0));
  APPEL(CreerMat(0, 0, BlocR));
  }
APPEL(CreerVect(NbAct, Trav1));
APPEL(CreerVect(NbAct, Trav2));

/* pour l'appel a SysLin */
if (Algo == GM)
  {
  APPEL(CreerVect(NbAct, Trav3));
  APPEL(CreerVectShort(NbAct, TravS));
  APPEL(CreerMat(NbAct, 1, MatTrav1));
  APPEL(CreerMat(NbAct, 1, MatTrav2));
  }
else
  {
  APPEL(CreerVect(0, Trav3));
  APPEL(CreerVectShort(0, TravS));
  APPEL(CreerMat(0, 0, MatTrav1));
  APPEL(CreerMat(0, 0, MatTrav2));
  }

/* retour */
return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : DesalloueT               |
| Role                  : Desallouer les structures|
|     de travail de PussIter                       |
|     Attention: des structures peuvent avoir un   |
|     nombre d'elements nul                        |
| Parametres d'entree   :                          |
|  ThetaCour, BetaCour: vecteurs des valeurs       |
|   courantes des parametres                       |
|  ValR: Vecteur de travail                        |
|  Direc: Vecteur direction de descente            |
|  Correc  : Vecteur correction de la direction de |
|     descente                                     |
|  DEffT, DEffB: derivees par rapport aux Theta et |
|    Beta                                          |
|  FctSA: fonctions de sensibilite en dimension    |
|    "actifs"                                      |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  DetruMat , DetruVect     |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt DesalloueT(ThetaCour, BetaCour, ValR, Direc, Correc,
                     DEffT, DEffB, FctSA)

TVect *ThetaCour, *BetaCour;
TVect *ValR, *Direc, *Correc;
TMat *DEffT, *DEffB;
TFctSensib *FctSA;

/*--------------- Fin identification fonction ----*/

{
/* Ecriture de la trace */
ECRTRACE("DesalloueT");

/* on procede dans l'ordre inverse d'allocation */
DetruMat(DEffB);
DetruMat(DEffT);

DetruMat(&(FctSA->DVarYBeta));
DetruMat(&(FctSA->DVarYTheta));
DetruMat(&(FctSA->DValf));
DetruVect(Correc);
DetruVect(Direc);
DetruVect(ValR);

DetruVect(BetaCour);
DetruVect(ThetaCour);

return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : DesalloueTA              |
| Role                  : Desallouer les structures|
|     de travail des fonctions appelees par        |
|     PussIter                                     |
|     Attention: des structures peuvent avoir un   |
|     nombre d'elements nul                        |
| Parametres d'entree   :                          |
|  Trav1,Trav2, Trav3:  vecteurs de travail de type|
|                       TVect                      |
|  TravS: vecteur de travail de type TVectShort    |
|  MatTrav,MatTrav1, MatTrav2,                     |
|  BlocP,BlocQ, Bloc0, BlocR:                      |
|      matrices de travail                         |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  DetruMat, DetruVect      |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt DesalloueTA (Trav1, Trav2, Trav3, TravS,
                      MatTrav, MatTrav1, MatTrav2, BlocP, BlocQ, Bloc0, BlocR)

TVect *Trav1, *Trav2, *Trav3;
TVectShort *TravS;
TMat *MatTrav, *MatTrav1, *MatTrav2, *BlocP, *BlocQ, *Bloc0, *BlocR;

/*--------------- Fin identification fonction ----*/

{
/* Ecriture de la trace */
ECRTRACE("DesalloueTA");

/* on procede dans l'ordre inverse d'allocation */

DetruMat(MatTrav2);
DetruMat(MatTrav1);
DetruVectShort(TravS);
DetruVect(Trav3);
DetruVect(Trav2);
DetruVect(Trav1);

DetruMat(BlocP);
DetruMat(BlocQ);
DetruMat(Bloc0);
DetruMat(BlocR);

DetruMat(MatTrav);

return(OK);
}





/*--------------- Identification fonction ----------
| Nom de la fonction    : PussIp                   |
| Role                  :  Gerer les impressions   |
|  intermediaires effectuees par PussIter          |
| Parametres d'entree   :                          |
|    Algo: l'algorithme                            |
|    NbIter : le nombre d'iterations               |
|    CritArret: le critere d'arret                 |
|    CritStat: le critere de minimisation          |
|    Lambda: la valeur de lambda                   |
|    Omega, Sigma, Direc                           |
|    ThetaCour, BetaCour: valeurs courantes des    |
|      parametres en dimension "modele * NbCourbe" |
|    Ajustes: les ajustes                          |
|    EquN: les equations normales                  |
|    FctSensib: les fonctions de sensibilites      |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Conditions d'appel    : GNLControle.SortImp!=NULL|
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  EcrMat, EcrVect          |
| Fonctions appelantes :  PussIter, Retablir       |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/


TShortInt PussIp(Algo, NbIter, CritArret,CritStat,Lambda, 
                 Omega, Sigma, Direc, ThetaCour, BetaCour,
                 Ajustes, EquN, FctSensib)


/* arguments d'entree */
TShortInt Algo;
TLongInt NbIter;
TDouble CritArret, CritStat, Lambda;
TDouble Omega, Sigma;
TVect *Direc;
TVect *ThetaCour, *BetaCour;
TAjustes *Ajustes;
TEquN *EquN;
TFctSensib *FctSensib;

/*--------------- Fin identification fonction ----*/
{
/* locals */
TChar Mess1[20];

/* Ecriture de la trace */
ECRTRACE("PussIp");

/* IMPRESSION DU No D'ITERATION */
if(GNLControle.VouluIp.NbIter)
  {
  sprintf(Mess1,"%ld",NbIter);
  /* MESSAGE ECRIT PAR NLWARNING */
/*
  fprintf(GNLControle.SortImp," %s: Iteration %s\n",PROMPT, Mess1);
*/
  NLWARNING((IMPIP0,2," ",Mess1, IMP));

  }

/* IMPRESSION DU CRITERE D'ARRET */
if(GNLControle.VouluIp.ResNum)
  {
  sprintf(Mess1,"%g",CritArret);
  /* MESSAGE ECRIT PAR NLWARNING */
/*
  fprintf(GNLControle.SortImp,"%s: Critere d'arret: %s\n",PROMPT, Mess1);
*/
  NLWARNING((IMPIP1,2,PROMPT,Mess1, IMP));

/* IMPRESSION DU CRITERE STATISTIQUE */
  sprintf(Mess1,"%g",CritStat);
  /* MESSAGE ECRIT PAR NLWARNING */
/*
  fprintf(GNLControle.SortImp,"%s: Critere de minimisation: %s\n",PROMPT, Mess1);
*/
  NLWARNING((IMPIP2,2,PROMPT,Mess1, IMP));


/* IMPRESSION DE SIGMA */

  sprintf(Mess1,"%g",Sigma);
  NLWARNING((IMPIP14,2,PROMPT,Mess1, IMP));

  if (Algo == GM)
    {
    /* IMPRESSION DE LAMBDA SI Algo =GM*/
    sprintf(Mess1,"%g",Lambda);
    /* MESSAGE ECRIT PAR NLWARNING */
  /*
    fprintf(GNLControle.SortImp,"%s: Valeur de lambda: %s\n",PROMPT, Mess1);
  */
    NLWARNING((IMPIP3,2,PROMPT,Mess1, IMP));
    }
  if (NbIter> 0)
    {
    /* IMPRESSION DE OMEGA ET DIREC */
    sprintf(Mess1,"%g",Omega);
    NLWARNING((IMPIP12,2,PROMPT,Mess1, IMP));
    NLWARNING((IMPIP13,1,PROMPT, IMP));
    APPEL(EcrVect(GNLControle.SortImp,Direc));
    } /* fin NbIter>0 */
  } /* fin VouluIp.ResNum */


/* IMPRESSION DES VALEURS ESTIMEES DES PARAMETRES */
if(GNLControle.VouluIp.Estim)
  {
  /* MESSAGE ECRIT PAR NLWARNING */
/*
  fprintf(GNLControle.SortImp,"Les valeurs courantes des parametres Theta sont:\n");
*/
   NLWARNING((IMPIP9,2,PROMPT,"regression", IMP));
  APPEL(EcrVect(GNLControle.SortImp,ThetaCour));
  if(BetaCour->nbele > 0)
    {
    /* MESSAGE ECRIT PAR NLWARNING */
/*
    fprintf(GNLControle.SortImp,"Les valeurs courantes des parametres Beta sont:\n");
*/
     NLWARNING((IMPIP9,2,PROMPT,"variance", IMP));
    APPEL(EcrVect(GNLControle.SortImp,BetaCour));
    }
  }

/* IMPRESSION DES VALEURS AJUSTES DE f ET VarY */
if(GNLControle.VouluIp.Ajustes)
  {
  /* MESSAGE ECRIT PAR NLWARNING */
/*
  fprintf(GNLControle.SortImp,"Les valeurs ajustees de l'esperance sont:\n");
*/
     NLWARNING((IMPIP4,1,PROMPT, IMP));
  APPEL(EcrVect(GNLControle.SortImp,&(Ajustes->Valf)));

  if(Ajustes->VarY.nbele > 0)
    {
    /* MESSAGE ECRIT PAR NLWARNING */
/*
    fprintf(GNLControle.SortImp,"Les valeurs ajustes de la variance de Y sont:\n");
*/
     NLWARNING((IMPIP5,1,PROMPT, IMP));
    APPEL(EcrVect(GNLControle.SortImp,&(Ajustes->VarY)));
    }
  }




/* IMPRESSION DES FONCTIONS DE SENSIBILITE */
if(GNLControle.VouluIp.FctSensib)
  {
  /* MESSAGE ECRIT PAR NLWARNING */
/*
  fprintf(GNLControle.SortImp,
 "Les valeurs des derivees de l'esperance par rapport aux Theta sont:\n");
*/
     NLWARNING((IMPIP7,1,PROMPT, IMP));
  APPEL(EcrMat(GNLControle.SortImp,&(FctSensib->DValf)));
  if (FctSensib->DVarYTheta.nblig > 0)
    {
    /* MESSAGE ECRIT PAR NLWARNING */
/*
      fprintf(GNLControle.SortImp,
 "Les valeurs des derivees de la variance de Y par rapport aux %s sont:\n","Theta");
*/
       NLWARNING((IMPIP8,2,PROMPT, "regression",IMP));
    APPEL(EcrMat(GNLControle.SortImp,&(FctSensib->DVarYTheta)));
      }
  if (FctSensib->DVarYBeta.nblig > 0)
      {
      /* MESSAGE ECRIT PAR NLWARNING */
/*
      fprintf(GNLControle.SortImp,
 "Les valeurs des derivees de la variance de Y par rapport aux %s sont:\n","Beta");
*/
      NLWARNING((IMPIP8,2,PROMPT,"variance",IMP));
      APPEL(EcrMat(GNLControle.SortImp,&(FctSensib->DVarYBeta)));
      }
  }
/* IMPRESSION DES EQUATIONS NORMALES */
if(GNLControle.VouluIp.EquN)
  {
  /* MESSAGE ECRIT PAR NLWARNING */
/*
  fprintf(GNLControle.SortImp,"Les valeurs de l'equation normale %s sont:\n","B");
*/
     NLWARNING((IMPIP6,2,PROMPT, "B",IMP));
  APPEL(EcrMat(GNLControle.SortImp,&(EquN->ValB)));
  /* MESSAGE ECRIT PAR NLWARNING */
/*
  fprintf(GNLControle.SortImp,"Les valeurs de l'equation normale %s sont:\n","Eta");
*/
     NLWARNING((IMPIP6,2,PROMPT, "Eta",IMP));
  APPEL(EcrVect(GNLControle.SortImp,&(EquN->ValEta)));
  /* MESSAGE ECRIT PAR NLWARNING */
/*
  fprintf(GNLControle.SortImp,"Les valeurs de l'equation normale %s sont:\n","D");
*/
     NLWARNING((IMPIP6,2,PROMPT, "D",IMP));
  APPEL(EcrMat(GNLControle.SortImp,&(EquN->ValD)));
  }

return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : PussIpSedo               |
| Role                  :  Gerer les impressions   |
|  intermediaires du sedo effectuees par PussIter  |
| Parametres d'entree   :                          |
|    NbObs: nombre d'observations                  |
|    Sedo: les valeurs de FSedo et DFSedo          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Conditions d'appel    : GNLControle.SortImp!=NULL|
|                         et cas sedo et impression|
|                         desiree                  |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes :  PussIter, Retablir       |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/


TShortInt PussIpSedo(NbObs, Sedo)

/* arguments d'entree */
TLongInt  NbObs ;
TSedo *Sedo;
/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt i, j, e;

/* Ecriture de la trace */
ECRTRACE("PussIpSedo");


/* IMPRESSION DES VALEURS AJUSTES DU SEDO */
  /* FSedo */
/*
  fprintf(GNLControle.SortImp,"Les valeurs ajustees de F sont:\n");
*/
     NLWARNING((IMPIP10,1,PROMPT, IMP));
  for (i =0; i <NbObs; i++)
    {

    for (j=0; j <GNLCtxInteg.NbJ; j++)
      {
      for (e=0; e < GNLCtxInteg.NbEq; e++)
        {
        fprintf(GNLControle.SortImp," %13.6g", Sedo->FSedo[i][j][e]);
        if ((e+1) % 5 == 0) fprintf(GNLControle.SortImp," \n");
        } /* fin boucle sur e */
      fprintf(GNLControle.SortImp," \n");
      } /* fin boucle sur j */
    } /* fin boucle sur i */



return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : IterSv                   |
| Role                  :  Stockage des resultats  |
|  intermediaires de PussIter                      |
| Parametres d'entree   :                          |
|    Algo: l'algorithme                            |
|    CritArret: le critere d'arret                 |
|    CritStat: le critere de minimisation          |
|    Lambda: la valeur de lambda                   |
|    NbIter : le nombre d'iterations               |
|    Sigma, Omega, Direc                           |
|    ThetaEstim, BetaEstim: valeurs courantes des  |
|      parametres en dimension  multiples
|    Ajustes: les ajustes                          |
|    EquN: les equations normales                  |
|    FctSensib: les fonctions de sensibilites      |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  ItNum, ItStat, ItTheta, ItBeta, ItDirec, ItOmega:|
|                      les structures de stockage  |
| Retour fonction       : OK                       |
| Conditions d'appel    : GNLControle.EtapeIt=     |
|  l'indice de l'etape courante                    |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CopyMat, CopyVect          |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/


TShortInt IterSv(NbObs, NbZO, NbTheta, NbBeta,NbThetaA, NbBetaA,
                 Algo, CritArret,CritStat,Lambda, 
                  NbIter,  Sigma, Omega, Direc, ThetaEstim, BetaEstim,
                 Ajustes, EquN, FctSensib,
                   ItNum, ItStat, ItTheta, ItBeta, ItDirec, ItOmega)


/* arguments d'entree */
TLongInt NbObs;
TShortInt NbZO, NbTheta, NbBeta, NbThetaA, NbBetaA;
TShortInt Algo;
TDouble CritArret, CritStat, Lambda;
TLongInt NbIter;
TDouble  Sigma, Omega;
TVect *Direc;
TVect *ThetaEstim, *BetaEstim;
TAjustes *Ajustes;
TEquN *EquN;
TFctSensib *FctSensib;

/* arguments de sortie */
TResNum *ItNum;
TResStat *ItStat;
TVect *ItTheta, *ItBeta;
TVect *ItDirec;
TDouble *ItOmega;

/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt i,NbAct;

/* Ecriture de la trace */
ECRTRACE("IterSv");

NbAct = NbThetaA + NbBetaA;

/*  No D'ITERATION: Dans tous les cas */
ItNum->NbIter = NbIter;

if  (GNLControle.VouluIt.ResNum == VRAI)
  {
/*  CRITERE D'ARRET */
  ItNum->CritArret=CritArret;
/*  CRITERE STATISTIQUE */
  ItNum->CritStat=CritStat;

/*  LAMBDA SI Algo =GM*/
  if (Algo == GM)
    {
    ItNum->Lambda=Lambda;
    }

/*  SIGMA */
   ItNum->Sigma=Sigma;

/*  DIREC */
    APPEL(CreerVect(NbAct , ItDirec));
    if (NbIter> 0)
      {
	APPEL(CopyVect(Direc, ItDirec));
      }
    else
      {
      /* A l'iter 0: Direc n'est pas calculee */
      for (i=0; i<NbAct; i++)
	{
	 ItDirec->donnees[i]= (double)mysignaling_nan(nn);
	}
      } /* fin else, i.e iter 0*/
  } /* fin VouluIt.resNum */

/* Pas optimal: il n'est pas dans ItNum, donc il faut le distinguer */
if (GNLControle.VouluIt.ResNum == VRAI)
  {
    if (NbIter> 0)
      *ItOmega = Omega;
    else
      *ItOmega = (double)mysignaling_nan(nn);
  } /* fin if (GNLControle.VouluIt.ResNum == VRAI) */


/*  VALEURS ESTIMEES DES PARAMETRES  (multiples)*/
if(GNLControle.VouluIt.Estim)
  {
  APPEL(CreerVect(ThetaEstim->nbele, ItTheta));
  APPEL(CreerVect(BetaEstim->nbele, ItBeta));
  /* Remarque: ItBeta existe toujours, meme s'il n'y a pas de parametres Beta,
  mais le nombre de ses elements est alors nul */
  APPEL(CopyVect(ThetaEstim, ItTheta));

  if(BetaEstim->nbele > 0)
    {
    APPEL(CopyVect(BetaEstim, ItBeta));
    }
  }

/*  VALEURS AJUSTES DE f ET VarY */
if(GNLControle.VouluIt.Ajustes)
  {
  APPEL(CreerVect((TShortInt)NbObs, &(ItStat->Ajustes.Valf)));
  APPEL(CopyVect(&(Ajustes->Valf), &(ItStat->Ajustes.Valf)));
  APPEL(CreerVect( (TShortInt)NbObs, &(ItStat->Ajustes.VarY)));
  if(Ajustes->VarY.nbele > 0)
    {
     APPEL(CopyVect(&(Ajustes->VarY), &(ItStat->Ajustes.VarY)));
    }
  }


/*  EQUATIONS NORMALES */
if(GNLControle.VouluIt.EquN)
  {
  APPEL(CreerMat(NbZO, NbAct, &(ItStat->EquN.ValB)));
  APPEL(CreerMat(NbZO, NbAct, &(ItStat->EquN.ValD)));
  APPEL(CreerVect( NbZO, &(ItStat->EquN.ValEta)));

  APPEL(CopyMat(&(EquN->ValB), &(ItStat->EquN.ValB)));
  APPEL(CopyVect(&(EquN->ValEta), &(ItStat->EquN.ValEta)));
  APPEL(CopyMat(&(EquN->ValD), &(ItStat->EquN.ValD)));
  }



/*  FONCTIONS DE SENSIBILITE */
if(GNLControle.VouluIt.FctSensib)
  {
  APPEL(CreerMatC( (TShortInt)NbObs, NbTheta, &(ItStat->FctSensib.DValf)));
  APPEL(CreerMatC( (TShortInt)NbObs, NbTheta, &(ItStat->FctSensib.DVarYTheta)));
  APPEL(CreerMatC((TShortInt)NbObs,  NbBeta, &(ItStat->FctSensib.DVarYBeta)));
  /* si le nombre de beta=0, mettre a 0 le nombre de lignes, car c'est
  equivalent a structure vide */
  if (NbBeta ==0)
    {
    ItStat->FctSensib.DVarYBeta.nblig = 0;
    }

  APPEL(CopyMat(&(FctSensib->DValf),&(ItStat->FctSensib.DValf)));
  if (FctSensib->DVarYTheta.nblig > 0)
    {
    APPEL(CopyMat(&(FctSensib->DVarYTheta), &(ItStat->FctSensib.DVarYTheta)));
      }
  if (FctSensib->DVarYBeta.nblig > 0)
      {
      APPEL(CopyMat(&(FctSensib->DVarYBeta), &(ItStat->FctSensib.DVarYBeta)));
      }
  }

return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : IterSvSedo               |
| Role                  :  Sauvegarder les calculs
|  intermediaires du sedo effectuees par PussIter  |
| Parametres d'entree   :                          |
|    NbObs: nombre d'observations                  |
|    Sedo: les valeurs de FSedo et DFSedo          |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Conditions d'appel    : 
|                          cas sedo et sauvegarde  |
|                         desiree                  |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/


TShortInt IterSvSedo(NbObs, Sedo, ItStat)

/* arguments d'entree */
TLongInt  NbObs ;
TSedo *Sedo; 

/* argu de sortie */
TResStat *ItStat;
/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt i, j, e;

/* Ecriture de la trace */
ECRTRACE("IterSvSedo");


  ItStat->Sedo.FSedo = (TDouble ***) calloc ((unsigned) NbObs, sizeof (TDouble **));

  /* FSedo  uniquement */
  for (i =0; i <NbObs; i++)
    {
    ItStat->Sedo.FSedo[i] = (TDouble **)calloc ((unsigned) GNLCtxInteg.NbJ, sizeof (TDouble *));
    for (j=0; j <GNLCtxInteg.NbJ; j++)
      {
      ItStat->Sedo.FSedo[i][j] = (TDouble *) calloc ((unsigned) GNLCtxInteg.NbEq, sizeof (TDouble ));

      for (e=0; e < GNLCtxInteg.NbEq; e++)
        {
        ItStat->Sedo.FSedo[i][j][e]= Sedo->FSedo[i][j][e];
        } /* fin boucle sur e */
      } /* fin boucle sur j */
    } /* fin boucle sur i */


return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : Retablir                 |
| Role                  :  Mettre a jour les       |
|  sorties de PussIter lorsqu'une erreur non       |
|  recuperable a ete detectee dans le calcul du    |
|   modele, ou bien lorsque MaxDeb est atteint,    |
|  c'est-a-dire retablir les valeurs               |
|   correspondant au debut du dernier intervalle ou|
|   le calcul du modele a ete  possible.           |
| Parametres d'entree   :                          |
|    IPas: pas courant                             |
|    Donnees: les donnees,                         |
|    Modele: le modele,                            |
|    Theta, Beta: les parametres de l'etape,       |
|    CTheta, CBeta: les contraintes sur les        | 
|            parametres du modele relatives a      |
|            l'etape courante                      |
|    CThetaE, CBetaE: les contraintes sur les      |
|            parametres differents                 |
|    CtxNum: contexte numerique de l'etape         |
|    CtxPuss: contexte du processus pour l'etape   |
|    ResNum: les resultats numeriques de           |
|      l'etape courante                            |
|    Omega, Direc: valeurs courantes de Omega et de|
|      la direction de descente                    |
| Parametres d'e/s      :                          | 
|    DEffT: derivees par rapport aux Theta en      |
|     dimension differents                         |
|     (tableau de travail de CModele)              |
|    DEffB: derivees par rapport aux Beta en       |
|     dimension differents                         |
|     (tableau de travail de CModele)              |
|    FctSA: fonction de sensibilite en dimension   |
|     actifs (tableau de travail de CModele)       |
|    ResStat: les resultats statistiques de        |
|      l'etape courante:                           |
|      le composant ValZ est une entree            |
|      les composants Ajustes, EquN, FctSensib et  |
|      ValW sont des sorties                       |
| Parametres de sortie  :                          |
| Retour fonction       :  OK                      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|         CModele,                                 |
|         CopyVect,                                |
|         CValW,                                   |
|         PCEquN: PCValB, PCValD, PCValEta,        |
|         PussIp, PussIpSedo                       |
| Fonctions appelantes :  PussIter                 |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/


TShortInt Retablir( ItSv, 
                    IPas, Donnees, Modele,Theta, Beta, 
                    CTheta, CBeta, CThetaE, CBetaE, CtxNum, CtxPuss,
                    ResNum, Omega, Direc, DEffT, DEffB, FctSA, ResStat,
                    NbItSv,ItNum, ItStat, ItTheta, ItBeta, ItDirec, ItOmega)

/* arguments d'entree */
TLogic ItSv;
TShortInt IPas;
TDonnees *Donnees;
TModele *Modele;
TParam *Theta, *Beta;
TContr *CTheta, *CBeta;
TContr *CThetaE, *CBetaE;
TCtxNum *CtxNum;
TCtxPuss *CtxPuss;
TResNum *ResNum;
TDouble Omega;
TVect *Direc;
TDouble *ItOmega;

/* arguments d'entree-sortie */
TMat *DEffT, *DEffB;
TFctSensib *FctSA;
TResStat *ResStat;

/* les resultats des iterations */
TShortInt *NbItSv;
TResNum *ItNum;
TResStat *ItStat;
TVect *ItTheta, *ItBeta, *ItDirec;

/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt ModErr;


/* Ecriture de la trace */
ECRTRACE("Retablir");

if ((IPas == 1) && (ResNum->NbIter == 0))
  {
  /* cas ou on a encore rien calcule: les valeurs estimees des parametres
  sont les valeurs initiales */
  APPEL(CopyVect(&(Theta->Init), &(Theta->Estim)));
  APPEL(CopyVect(&(Beta->Init), &(Beta->Estim)));
  }

else
  {
  /*
  on retablit les valeurs correspondant aux valeurs des parametres stockees
  dans le composant Act. */
  APPEL(CModele(Donnees, Modele, CtxPuss, &(Theta->Act), &(Beta->Act),
          CTheta, CBeta, CThetaE, CBetaE, Theta, Beta,DEffT, DEffB, FctSA, 
          ResStat ,ResNum, &ModErr));

  if ((IPas != 2) ||
          (CtxNum->TypeCritStat == CARRET) || (CtxNum->TypeCritStat == LEMIEN) )
    {
    /* retablir les EquN */
    APPEL(CtxNum->PCEquN.PCValB(&(Donnees->NbRepet),
                &(ResStat->Ajustes.Valf),  &(ResStat->Ajustes.VarY),
                &(Donnees->StatDon.S2), 
                &(FctSA->DValf), &(FctSA->DVarYTheta),  &(FctSA->DVarYBeta),
                &(ResStat->EquN.ValB)));
    APPEL(CtxNum->PCEquN.PCValD(&(ResStat->Ajustes.Valf),
        &(FctSA->DValf), &(FctSA->DVarYTheta),  &(FctSA->DVarYBeta),
        &(ResStat->EquN.ValD)));
    APPEL(CtxNum->PCEquN.PCValEta(&(ResStat->Ajustes.Valf),
                   &(ResStat->Ajustes.VarY),
                   &(ResStat->EquN.ValEta)));

    /* retablir  ValW */
    APPEL( CValW(Donnees->NbObsT, &(ResStat->EquN.ValB),
                &(ResStat->EquN.ValD), &(ResStat->ValW)));
    }
  }

/* Puisqu'on sort du processus iteratif, il faut faire comme si on
   etait a la derniere iteration, c'est-a-dire ecrire ou sauver les sorties demandees 
   si non deja fait */

if ((GNLControle.SortImp != NULL) || (ItSv==VRAI))
  {
  if ( (GNLControle.FreqIp < 0)  ||
       ((GNLControle.FreqIp > 0) && (ResNum->NbIter>0) &&
         (ResNum->NbIter % GNLControle.FreqIp != 0)))
    {
    if (GNLControle.SortImp != NULL)
      {
      APPEL(PussIp(CtxPuss->Algo,ResNum->NbIter, ResNum->CritArret,ResNum->CritStat,
                 ResNum->Lambda, Omega, ResNum->Sigma, Direc,
                 &(Theta->Estim), &(Beta->Estim),
                 &(ResStat->Ajustes), &(ResStat->EquN), &(ResStat->FctSensib)));
      if ((Modele->CasSedo == VRAI) && (GNLControle.VouluIp.Sedo == VRAI))
         {
         APPEL(PussIpSedo(Donnees->NbObs, &(ResStat->Sedo)));
         }
      } /* fin de (GNLControle.SortImp != NULL) */

    if (( ItSv==VRAI) && (*NbItSv <= MAXSVIT))
      {
      /* sauvegarde */
      APPEL(IterSv(Donnees->NbObs,ResStat->ValZ.nbele,
                    Theta->Noms.nbele, Beta->Noms.nbele,
                    Theta->Act.nbele, Beta->Act.nbele,
                    CtxPuss->Algo,ResNum->CritArret,ResNum->CritStat,
                    ResNum->Lambda,ResNum->NbIter,ResNum->Sigma, Omega, &Direc,
                    &(Theta->Estim), &(Beta->Estim),
                    &(ResStat->Ajustes), &(ResStat->EquN), &(ResStat->FctSensib),
                    &(ItNum[*NbItSv]), &(ItStat[*NbItSv]), 
                    &(ItTheta[*NbItSv]), &(ItBeta[*NbItSv]), &(ItDirec[*NbItSv]),
                    &(ItOmega[*NbItSv]) ));

      if ((Modele->CasSedo == VRAI) && (GNLControle.VouluIt.Sedo == VRAI))
        {
        APPEL(IterSvSedo(Donnees->NbObs, &(ResStat->Sedo), 
                      &(ItStat[*NbItSv])));
        }
      *NbItSv = *NbItSv + 1;
      } /* fin de (ItSv==VRAI) */
    } /* fin de if ( (GNLControle.FreqIp < 0)  || etc.. */
  } /* fin de if ( ((GNLControle.SortImp etc ... */


return(OK);
}



/*--------------- Identification fonction ----------
| Nom de la fonction    : PussIter                 |
| Role                  :  Gerer le processus      |
| iteratif des calculs de l'etape courante         |
| Parametres d'entree   :                          |
|    Donnees: les donnees,                         |
|    Modele: le modele,                            |
|    CtxPuss: le contexte du processus de calcul   |
|    CtxNum: le contexte numerique relatif a       |
|            l'etape courante                      |
|    CTheta, CBeta: les contraintes sur les        | 
|            parametres du modele relatives a      |
|            l'etape courante                      |
|    CThetaE, CBetaE: les contraintes sur les      |
|            parametres differents                 |
| Parametres d'e/s      :                          | 
|    Theta, Beta: les parametres de l'etape:       |
|      les composants Eff et Act sont des          |
|      entrees-sorties,                            |
|      le composant Estim est une sortie           |
|    ResStat: les resultats statistiques de        |
|      l'etape courante:                           |
|      le composant ValZ est une entree            |
|      les composants Ajustes, EquN, FctSensib et  |
|      ValW sont des sorties                       |
| Parametres de sortie  :                          |
|    ResNum: les resultats numeriques de           |
|      l'etape courante:                           |
|      Les composants:                             |
|      CritArret, Scr, NbIter, Lambda, InfoModErr, |
|      CritStat                                    |
|      sont des sorties                            |
|      Sigma est une entree quand sa valeur ne     |
|      depend pas des ajustes, une sortie sinon    |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|         AlloueS, AlloueSedo, AlloueT, AlloueTA,  |
|         CArret,                                  |
|         CModele, CModErr,                        |
|         CopyVect,                                |
|         CPasOpt,                                 |
|         CValR,                                   |
|         CValW,                                   |
|         DesalloueT, DesalloueTA,                 |
|         MultVectVal                              |
|         les programmes de calcul du critere      |
|         de minimisation: CLv, CArret, CScrNP,    |
|         CScrPV, CScrPS2, CScr2NP, CritSig        |
|         PCDirec (=CDirGM ou CDirGMBloc ou        |
|                  CDirGN ou CDirGNBloc)           |
|         PCValB, PCValD, PCValEta,                |
|         PussIp, PussIpSedo                       |
|         Retablir                                 |
|   plus la fonction systeme: fabs                 |
| Fonctions appelantes :  NLEtape                  |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/


TShortInt PussIter( ItSv,
                    Donnees, Modele, CtxPuss, CtxNum, CTheta, CBeta,
                    CThetaE, CBetaE,
                    Theta, Beta, ResStat, ResNum,
                    NbItSv, ItNum, ItStat, ItTheta, ItBeta, ItDirec, ItOmega)


/* arguments d'entree */
TLogic ItSv;
TDonnees *Donnees;
TModele *Modele;
TCtxPuss *CtxPuss;
TCtxNum *CtxNum;
TContr *CTheta, *CBeta;
TContr *CThetaE, *CBetaE;

/* arguments d'entree-sortie */
TParam *Theta, *Beta;
TResStat *ResStat;

/* arguments de sortie */
TResNum *ResNum;

/* les resultats des iterations */
TShortInt *NbItSv;
TResNum *ItNum;
TResStat *ItStat;
TVect *ItTheta, *ItBeta, *ItDirec;
TDouble *ItOmega;

/*--------------- Fin identification fonction ----*/
{
/* locals */
TShortInt  IPas, i, j, nbtheta, varyn, NbTAct, NbBAct;
TShortInt Code,  ModErr;
TLongInt NbErr, NbDeb;
TLogic Encore, CritEgaux;
TDouble Lambda,  Omega;
TDouble CritSCour[3]; /* pour mettre la valeur du critere
                         de minimisation calculee en chacun des 3 points
                         de l'intervalle etudie */
TVect ValR, Direc, Correc;
TFctSensib FctSA;  /* derivee en dimension actif:
                 servent pour le calcul du modele;
                 voir les arguments de  CModele */
TMat DEffT, DEffB; /* derivee en dimension differents:
                 servent pour le calcul du modele;
                 voir les arguments de  CModele */
TVect ThetaCour, BetaCour;
TMat MatTrav, MatTrav1, MatTrav2, BlocQ, BlocR, Bloc0, BlocP;
TVect Trav1, Trav2, Trav3;
TVectShort TravS;
TChar Mess1[5], Mess2[10];


/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble *vary, **dvarytheta, *s2;

/* Ecriture de la trace */
ECRTRACE("PussIter");

/* ................................................................. */
/*           Allocations des structures                              */
/* ................................................................. */


/*   Allocation des sorties                    */        
APPEL( AlloueS ( ResStat->ValZ.nbele, Donnees->NbObs,
                Theta, Beta,
                &(ResStat->ValW), &(ResStat->Ajustes), &(ResStat->EquN), 
                &(ResStat->FctSensib)));

if (Modele->CasSedo == VRAI)
  {
  APPEL(AlloueSedo (Donnees->NbObs,&(ResStat->Sedo)));
  }


/* initialisation des sorties */
/* si YaCalcV FAUX, VarY est un vecteur de 1 ou egal a S2
  et DVarYTheta une matrice de 0
  si YaCalcV FAUX, il n'y a pas de Beta, donc pas de matrice
  DVarYBeta: inutile de mettre DVarYBeta a Zero  */
if(Modele->YaCalcV == FAUX)
  {
  s2 = Donnees->StatDon.S2.donnees;
  vary = ResStat->Ajustes.VarY.donnees;
  dvarytheta = ResStat->FctSensib.DVarYTheta.donnees;
  varyn = ResStat->Ajustes.VarY.nbele;
  nbtheta = ResStat->FctSensib.DVarYTheta.nbcol;

  for(i = 0; i < varyn; i++)
    {
    if (Modele->Vari == VI)
      {
      /* variance intra: V=S2 */
      vary[i] = s2[i];
      }
    else
      {
      vary[i] = (TDouble)1.0;
      }
    for (j=0; j < nbtheta; j ++)
      {
      dvarytheta[i][j] = (TDouble)ZERO;
      }
    }
  }

/*   Allocation des structures de travail                    */        
APPEL(AlloueT ( Theta->Act.nbele, Beta->Act.nbele, 
               Theta->Eff.nbele, Beta->Eff.nbele, 
               Donnees->NbObs,
               &ThetaCour, &BetaCour,  &ValR, &Direc, &Correc,
               &DEffT, &DEffB, &FctSA));

/* Allocation des structures de travail des fonctions appelees */
APPEL(AlloueTA( CtxPuss->Algo, CtxNum->Symm, Theta->Act.nbele, Beta->Act.nbele, 
                &Trav1, &Trav2, &Trav3, &TravS,
                &MatTrav, &MatTrav1, &MatTrav2,
                &BlocP, &BlocQ, &Bloc0, &BlocR));


/* ................................................................. */
/* ......... initialisations du processus iteratif...................*/
/* ................................................................. */
Lambda =  CtxPuss->Lambda0;
ResNum->Lambda =  CtxPuss->Lambda0;
NbDeb = 0;
ResNum->NbIter =  0;
/* on incremente NbIter en fin d'etude d'un intervalle:
NbIter = 0 correspond a l'etude de l'intervalle commencant aux valeurs initiales
des parametres */
*NbItSv=0;

Code = WARPUSS1;  /* initialisation a convergence non atteinte */
NbTAct = Theta->Act.nbele;
NbBAct = Beta->Act.nbele;

/* initialisation de la valeur courante des parametres */
APPEL(CopyVect(&(Theta->Act), &ThetaCour));
APPEL(CopyVect(&(Beta->Act), &BetaCour));


/* ................................................................. */
/*  A chaque iteration, on va calculer le critere a minimiser en 3
    points d'un intervalle de taille Direc (direction de descente)  
    La  boucle do est effectuee autant de fois que d'intervalles etudies
*/
/* ................................................................. */
do
  {


/* ................................................................. */
/* boucle qui sera effectue, pour un intervalle donne, 3 fois:       */
/* au debut, (IPas=1), a l'extremite (IPas=2), a la fin(IPas=3)      */
/* ................................................................. */
  for(IPas = 1; IPas <= 3; IPas++)
    {
    /* ce qu'on fait en chacun des 3 points de l'intervalle */
    NbErr =   0;
    
    /* calcul du modele  */
    /* on boucle sur l'appel a CModele car s'il y a une erreur,
       il est peut-etre possible de reessayer a partir
       d'autres valeurs */
    do
      {

      APPELP(CModele(Donnees, Modele, CtxPuss, &ThetaCour, &BetaCour,
                    CTheta, CBeta, CThetaE, CBetaE, Theta,Beta,
                    &DEffT, &DEffB, &FctSA, 
                    ResStat ,ResNum, &ModErr));

      /* En sortie de CModele, s'il n'y a pas d'erreur, on a 
      dans Theta->Estim et Beta->Estim les valeurs courantes
      en dimension "modele * NbCourbe" . 
      Les Theta->Act, Beta->Act sont inchanges */
      Encore = FAUX;
      if(ModErr != OK)
        {
        /* Il y a une erreur: on emet des warnings et on essaie de corriger: */
        APPELP(CModErr(ModErr, IPas, ResNum->NbIter, CtxPuss->MaxErr, 
                   CtxPuss->DirecC, &(Theta->Act), &(Beta->Act), &(ResNum->InfoModErr),
                 &Encore, &NbErr, &ThetaCour, &BetaCour, &Direc ));


        if(Encore == FAUX)
          {
          /* on ne peut pas corriger: retablir en sortie les valeurs
             calculees au debut du dernier intervalle ou le calcul a 
             ete possible */
          APPELP(Retablir(ItSv,
                         IPas,Donnees, Modele, Theta, Beta, CTheta, CBeta,
                         CThetaE, CBetaE, CtxNum, CtxPuss, ResNum, 
                         Omega, &Direc ,
                         &DEffT, &DEffB, &FctSA, ResStat,
                         NbItSv, ItNum, ItStat,
                         ItTheta, ItBeta, ItDirec, ItOmega ));


          /* provoquer une erreur */
          /* MESSAGE ECRIT PAR NLWARNING: */
          /*
          fprintf(GNLControle.SortWE,
              " Le calcul du modele est impossible\n");
          */
          sprintf(Mess2,"%ld", CtxPuss->MaxErr);
          NLWARNING((WARPUSS1,2,"PussIter", Mess2,WNUMER));
          /* L'ecrire sur SortImp  */
          NLWARNING((IMPIT4,1," ", IMP));
          /* Desallouer les structures de travail */
          APPEL(DesalloueTA (&Trav1, &Trav2, &Trav3, &TravS,
                &MatTrav,  &MatTrav1,  &MatTrav2, &BlocP, &BlocQ, 
                &Bloc0, &BlocR));
          APPEL(DesalloueT(  &ThetaCour, &BetaCour,  &ValR,
                   &Direc, &Correc,
                   &DEffT, &DEffB, &FctSA));
          return(WARPUSS1);

          } /* fin de Encore=FAUX */
        } /* fin de ModErr !=0 */
      else
        {
        /* cas ou le modele a pu etre calcule */
        if (NbErr > 0)
          {
          /* cas ou il y a eu de precedentes erreurs de calcul */
          sprintf(Mess1,"%ld", (NbErr+1));
          /* MESSAGE ECRIT PAR NLWARNING: */
          /*  fprintf(GNLControle.SortWE,
              "%s% -ieme reussi\n", Mess1 ); */
          NLWARNING((IMPIT1,2,"PussIter",Mess1, WNUMER));
          }
        } /* fin du cas ou le modele a pu etre calcule */
      }
      while(Encore); 

    /* si on en a besoin pour la suite, calcul des EquN, ValW, et ValR .
    On en a besoin si ipas =1 (pour calculer le critere d'arret), et, sinon,
    pour calculer le critere stat si celui-ci est le critere d'arret ou LEMIEN */
    
    if ( (IPas == 1) || (CtxNum->TypeCritStat == CARRET) || (CtxNum->TypeCritStat == LEMIEN) )
      {
      /* calcul des EquN */
      APPELP(CtxNum->PCEquN.PCValB(&(Donnees->NbRepet),
                  &(ResStat->Ajustes.Valf),  &(ResStat->Ajustes.VarY),
                  &(Donnees->StatDon.S2), 
                  &(FctSA.DValf), &(FctSA.DVarYTheta),  &(FctSA.DVarYBeta),
                  &(ResStat->EquN.ValB)));

      APPELP(CtxNum->PCEquN.PCValD(&(ResStat->Ajustes.Valf),
          &(FctSA.DValf), &(FctSA.DVarYTheta),  &(FctSA.DVarYBeta),
          &(ResStat->EquN.ValD)));

      APPELP(CtxNum->PCEquN.PCValEta(&(ResStat->Ajustes.Valf),
                     &(ResStat->Ajustes.VarY),
                     &(ResStat->EquN.ValEta)));
  
       /* calcul de ValW */
      APPELP( CValW(Donnees->NbObsT, &(ResStat->EquN.ValB),
                  &(ResStat->EquN.ValD), &(ResStat->ValW)));
  
       /* calcul de ValR = B(Z-eta) */
      APPELP( CValR(&(ResStat->ValZ),&(ResStat->EquN.ValEta),
                   &(ResStat->EquN.ValB), &ValR));

      }

     /* calcul du critere stat */
    switch(CtxNum->TypeCritStat)
      {
      case CLV:
        APPELP(CLv(Donnees->NbObsT, &(Donnees->NbRepet), 
                  &(Donnees->StatDon.Y1), &(Donnees->StatDon.Y2),
                  &(ResStat->Ajustes.Valf),  &(ResStat->Ajustes.VarY),
                  &(CritSCour[IPas-1])));

        break;
      case CARRET:
        APPELP(CArret(CtxNum->Symm, Donnees->NbObsT, &(ResStat->ValW), &ValR, 
                  &Trav1, &Trav2,  &BlocP, &BlocQ, &Bloc0, &BlocR,
                  &(CritSCour[IPas-1]), &MatTrav));
        break;
      case CSCRNP:
        APPELP(CScrNP(Donnees->NbObsT, 
                     &(Donnees->NbRepet),
                     &(Donnees->StatDon.Y1), &(Donnees->StatDon.Y2),
                     &(ResStat->Ajustes.Valf),
                     &(CritSCour[IPas-1])));

        break;
      case CSCRPV:
        APPELP(CScrPv(Donnees->NbObsT, ResNum->Sigma,
                    &(Donnees->NbRepet),
                    &(Donnees->StatDon.Y1), &(Donnees->StatDon.Y2),
                    &(ResStat->Ajustes.Valf),  &(ResStat->Ajustes.VarY),
                    &(CritSCour[IPas-1])));

        break;
      case CSCRPS2:
        APPELP(CScrPS2(Donnees->NbObsT,
                      &(Donnees->NbRepet),
                      &(Donnees->StatDon.Y1), &(Donnees->StatDon.Y2),
                      &(ResStat->Ajustes.Valf),
                      &(Donnees->StatDon.S2) ,
                      &(CritSCour[IPas-1])));

        break;
      case CSCR2NP:
        APPELP(CScr2NP( Donnees->NbObsT,
                      &(Donnees->NbRepet),
                      &(Donnees->StatDon.Y1),
                      &(Donnees->StatDon.Y2),
                      &(ResStat->Ajustes.Valf),  &(ResStat->Ajustes.VarY),
                      &(CritSCour[IPas-1])));

        break;
      case CRITSIG:
        APPELP(CritSig(ResNum->Sigma,
                      &(CritSCour[IPas-1])));

        break;
      case LEMIEN:
        APPELP(calcc( ResNum->Sigma,
                      Donnees->NbObsT,
                      &(Donnees->NbRepet),
                      &(Donnees->StatDon.Y1), &(Donnees->StatDon.Y2), &(Donnees->StatDon.S2) ,
                      &(ResStat->Ajustes.Valf),&(ResStat->Ajustes.VarY),
                      &(CritSCour[IPas-1])));

        break;
      default:
        /* provoquer une erreur */
        sprintf(Mess1, "%hd", CtxNum->TypeCritStat);
        /* MESSAGE ECRIT PAR NLERREUR: */
        /*  fprintf(GNLControle.SortWE,
              "PussIter: TypeCritStat, %s, errone", Mess1); */
        ResNum->CodePuss = ERRTYPES;
        NLERREUR((ERRTYPES,2,"PussIter",Mess1, ERR));
        break;
      } /* fin du switch */


/* --------- Actions selon la valeur de ipas ----------------------- */
/* Remarque: ne pas mettre un switch sur IPas car, si la convergence est atteinte
ou le nombre maxi d'iterations ou MaxDeb, 
il faudrait sortir du switch et de la boucle sur IPas,
c'est-a-dire de 2 blocs enveloppants, ce qui n'est pas possible en C */
    if(IPas == 1)
      {
      /* ce qu'on fait au debut de l'intervalle */

      /* Le modele a pu etre calcule au debut de l'intervalle: 
      les valeurs a sortir sont celles calculees a cet endroit,
      qu'il y ait ou non une erreur ensuite.
      Mise a jour du critere de minimisation, de Lambda et
       des valeurs des parametres */
      ResNum->CritStat =  CritSCour[0];
      ResNum->Lambda = Lambda;


      APPELP(CopyVect(&ThetaCour, &(Theta->Act)));
      APPELP(CopyVect(&BetaCour, &(Beta->Act)));

      /* calcul du critere d'arret */
      APPELP(CArret(CtxNum->Symm, Donnees->NbObsT, &(ResStat->ValW), &ValR, 
                  &Trav1, &Trav2,  &BlocP, &BlocQ, &Bloc0, &BlocR,
                  &(ResNum->CritArret), &MatTrav));
      /* en sortie, MatTrav est l'inverse de ValW et
      si SYMBLOC, les blocs contiennent l'inverse de ValW par blocs */

      /* impression ou sauvegarde */
      if ((GNLControle.SortImp != NULL) || (ItSv==VRAI))
        {
        if( ((GNLControle.FreqIp > 0) && (ResNum->NbIter % GNLControle.FreqIp == 0)) ||
            ((GNLControle.FreqIp != 0) && (ResNum->NbIter == 0)))
          {
          if (GNLControle.SortImp != NULL)
            {
            /* impression */
            APPELP(PussIp(CtxPuss->Algo, ResNum->NbIter, 
                ResNum->CritArret,CritSCour[0],
                Lambda, Omega, ResNum->Sigma, &Direc,
                &(Theta->Estim), &(Beta->Estim),
                &(ResStat->Ajustes), &(ResStat->EquN), &(ResStat->FctSensib)));
            if ((Modele->CasSedo == VRAI) && (GNLControle.VouluIp.Sedo == VRAI))
              {
              APPELP(PussIpSedo(Donnees->NbObs, &(ResStat->Sedo)));
              }
            } /* fin de (GNLControle.SortImp != NULL) */

          if (( ItSv==VRAI) && (*NbItSv <= MAXSVIT))
            {
            /* sauvegarde */
            APPELP(IterSv(Donnees->NbObs,ResStat->ValZ.nbele,
                          Theta->Noms.nbele, Beta->Noms.nbele,
                          Theta->Act.nbele, Beta->Act.nbele,
                          CtxPuss->Algo,ResNum->CritArret,CritSCour[0],
                          Lambda,ResNum->NbIter,ResNum->Sigma, Omega, &Direc,
                          &(Theta->Estim), &(Beta->Estim),
                          &(ResStat->Ajustes), &(ResStat->EquN), &(ResStat->FctSensib),
                          &(ItNum[*NbItSv]), &(ItStat[*NbItSv]), 
                          &(ItTheta[*NbItSv]), &(ItBeta[*NbItSv]), &(ItDirec[*NbItSv]),
                          &(ItOmega[*NbItSv]) ));

            if ((Modele->CasSedo == VRAI) && (GNLControle.VouluIt.Sedo == VRAI))
              {
             APPELP(IterSvSedo(Donnees->NbObs, &(ResStat->Sedo),
                              &(ItStat[*NbItSv])));

              }
            *NbItSv = *NbItSv + 1;
            } /* fin de (ItSv==VRAI) */
          }
        }

      /* tester la convergence */
      if ((fabs(ResNum->CritArret) <= CtxPuss->MaxCritArret) &&
          ( (CtxPuss->Algo != GM) || (Lambda <= CtxPuss->MaxLambda)))
        {
        /* sortir des boucles : il y a convergence */
        Code = OK;
        break; /* sortir de la boucle sur IPas */
        }

      if (ResNum->NbIter >= CtxPuss->MaxIter)
        {
        if(CtxPuss->MaxIter > 0)
          {
          /* On teste si CtxPuss->MaxIter >0, car si celui-ci est nul,
            c'est que l'utilisateur ne veut pas d'estimation, 
            donc on n'ecrit pas de message dans ce cas */
          /* MESSAGE ECRIT PAR NLWARNING: */
          sprintf(Mess1,"%ld", CtxPuss->MaxIter);
/*
          fprintf(GNLControle.SortWE,"Le nombre maximal d'iterations %s est atteint\n", Mess1);
*/
          NLWARNING((WARPUSS2,2,"PussIter",Mess1, WNUMER));
          }
        Code = WARPUSS2;
        break; /* sortir de la boucle sur IPas */
        } /* fin de NbIter trop grand */

      /* calcul de la direction de descente */
     APPELP(CtxNum->PCDirec( Donnees->NbObsT, Lambda, &(ResStat->ValW),
                    &Direc,
                    &ValR,  
                    &Trav1, &Trav2, &Trav3, &TravS,
                    &MatTrav,&MatTrav1, &MatTrav2,
                    &BlocP, &BlocQ, &BlocR));

      /* actualisation des parametres courants */
      for(i = 0; i < NbTAct; i++)
        {
        ThetaCour.donnees[i] = Theta->Act.donnees[i] + Direc.donnees[i];
        }
      for(i = 0; i < Beta->Act.nbele; i++)
        {
        BetaCour.donnees[i] = Beta->Act.donnees[i] + Direc.donnees[i+Theta->Act.nbele];
        }
      } /* fin de IPAS=1 */

    if(IPas == 2)
      {
      /* ce qu'on fait au point extreme */
      /* actualisation des parametres courants */
      /*    Correc = Direc * 0.5  */
      APPELP( MultVectVal(&Direc, (TDouble)0.5, &Correc)); 
      for(i = 0; i < NbTAct; i++)
        {
        ThetaCour.donnees[i] = Theta->Act.donnees[i] + Correc.donnees[i];
        }
      for(i = 0; i < NbBAct; i++)
        {
        BetaCour.donnees[i] = Beta->Act.donnees[i] + Correc.donnees[i+Theta->Act.nbele];
        }
      } /* fin de IPas=2 */

    if(IPas == 3)
      {
      /* ce qu'on fait au point median */
      /* calcul du pas optimal */
      APPELP( CPasOpt(CtxPuss->Algo, CtxPuss->OmegaPas, 
                     CtxPuss->LambdaC1, CtxPuss->LambdaC2, CritSCour,
                     &NbDeb,  &Lambda,
                     &Omega, & CritEgaux));
      /* tester la convergence */
      /* 3/07/98
      Pour les moindres carrs modifis, le critere d'arret
      est egal a la matrice Q: si sa valeur est egale aux
      3 points de l'intervalle, c'est que l'intervalle entre
      ces points est petit: sortir des boucles :
       on est pres de la solution,  y a convergence */


      if (  (CritEgaux == VRAI) &&
            ( (CtxNum->Estim == MCMTB) ||  (CtxNum->Estim == QVTB) ||
	      (CtxNum->Estim == MCMB) || (CtxNum->Estim == QVB) ||
	      (CtxNum->Estim == MCMT)  ||(CtxNum->Estim == QVT)))
        {
	  if ((fabs(ResNum->CritArret) <= CtxPuss->MaxCritArret) &&
          ( (CtxPuss->Algo != GM) || (Lambda <= CtxPuss->MaxLambda)))
	    {
	      /* sortir des boucles : il y a convergence */
	      Code = OK;
	      break; /* sortir de la boucle sur IPas */
	    }
	} /* fin de (CritEgaux == VRAI) && etc */

      if (NbDeb >= CtxPuss->MaxDeb)
        {
        sprintf(Mess1,"%ld", CtxPuss->MaxDeb);
        if (CritEgaux == FAUX)
          {  
          /* MESSAGE ECRIT PAR NLWARNING: */
  /*
          fprintf(GNLControle.SortWE,
   "Le nombre maximal de fois, ou, consecutivement, le critere de minimisation\
   est minimal au debut d'un intervalle %s est atteint\n", Mess1);
  */
          NLWARNING((WARPUSS3,2,"PussIter",Mess1, WNUMER));
          Code = WARPUSS3;
          }
        else
          {  
          /* MESSAGE ECRIT PAR NLWARNING: */
  /*
          fprintf(GNLControle.SortWE,
   "Les criteres de minimisation sont egaux au debut, au milieu et a la fin\
d'un intervalle, meme apres que le nombre maximal d'essais (%s) ait ete tente\n",Mess1);
  */
          NLWARNING((WARPUSS4,2,"PussIter",Mess1, WNUMER));
          Code = WARPUSS4;
          }

        APPELP(Retablir(ItSv,
                       IPas,Donnees, Modele, Theta, Beta, CTheta, CBeta,
                       CThetaE, CBetaE, CtxNum, CtxPuss, ResNum, 
                       Omega, &Direc,
                       &DEffT, &DEffB, &FctSA, ResStat,
                       NbItSv,ItNum, ItStat,
                       ItTheta, ItBeta, ItDirec, ItOmega ));


        break; /* sortir de la boucle sur IPas */
        } /* fin de NbDeb trop grand */



      /* actualisation des parametres:
         valeurs au debut de l'intervalle suivant */
      /*      Correc = Omega * Direc; */
      APPELP( MultVectVal(&Direc, Omega, &Correc));
      for(i = 0; i < NbTAct; i++)
        {
        ThetaCour.donnees[i] = Theta->Act.donnees[i] + Correc.donnees[i];
        }
      for(i = 0; i < NbBAct; i++)
        {
        BetaCour.donnees[i] = Beta->Act.donnees[i] + Correc.donnees[i+Theta->Act.nbele];
        }
      ResNum->NbIter = ResNum->NbIter + 1;
    
      } /* fin IPas=3 */

    }/* fin du for sur IPas */

  } while(Code  == WARPUSS1);


/* impression ou sauvegarde finale si non deja faite  */
if ( ((GNLControle.SortImp != NULL) || (ItSv==VRAI)) && (Code != WARPUSS3))
/* si WARPUSS3, l'impression a deja ete faite par Retablir */
  {
  if ( (GNLControle.FreqIp < 0)  ||
       ((GNLControle.FreqIp > 0) && (ResNum->NbIter>0) &&
         (ResNum->NbIter % GNLControle.FreqIp != 0)))
    {
    if (GNLControle.SortImp != NULL)
      {
      APPELP(PussIp(CtxPuss->Algo, ResNum->NbIter, ResNum->CritArret,
                  CritSCour[0],
                  ResNum->Lambda, Omega, ResNum->Sigma, &Direc,
                  &(Theta->Estim), &(Beta->Estim),
                  &(ResStat->Ajustes), &(ResStat->EquN), &(ResStat->FctSensib)));
      if ((Modele->CasSedo == VRAI) && (GNLControle.VouluIp.Sedo == VRAI))
        {
        APPELP(PussIpSedo(Donnees->NbObs, &(ResStat->Sedo)));
        }
      } /* fin de (GNLControle.SortImp != NULL) */

    if (( ItSv==VRAI) && (*NbItSv <= MAXSVIT))
      {
      /* sauvegarde */
      APPELP(IterSv(Donnees->NbObs,ResStat->ValZ.nbele,
                    Theta->Noms.nbele, Beta->Noms.nbele,
                    Theta->Act.nbele, Beta->Act.nbele,
                    CtxPuss->Algo,ResNum->CritArret,CritSCour[0],
                    Lambda,ResNum->NbIter,ResNum->Sigma, Omega, &Direc,
                    &(Theta->Estim), &(Beta->Estim),
                    &(ResStat->Ajustes), &(ResStat->EquN), &(ResStat->FctSensib),
                    &(ItNum[*NbItSv]), &(ItStat[*NbItSv]), 
                    &(ItTheta[*NbItSv]), &(ItBeta[*NbItSv]), &(ItDirec[*NbItSv]),
                    &(ItOmega[*NbItSv]) ));

      if ((Modele->CasSedo == VRAI) && (GNLControle.VouluIt.Sedo == VRAI))
        {
        APPELP(IterSvSedo(Donnees->NbObs, &(ResStat->Sedo), 
                        &(ItStat[*NbItSv])));
        }
      *NbItSv = *NbItSv + 1;
      } /* fin de (ItSv==VRAI) */
    } /* fin de if ( (GNLControle.FreqIp < 0)  || etc.. */
  } /* fin de if ( ((GNLControle.SortImp etc ... */

/* Ecrire un message sur le fichier des impressions 
   sur la facon dont se termine le processus */
if (Code == OK)
  {
  if ((ResNum->NbIter == 0) && (CtxPuss->MaxIter > 0))
    {
      /* MESSAGE ECRIT PAR NLWARNING */
      /* fprintf(GNLControle.SortImp,
"Pas d'estimation : convergence atteinte sur les valeurs initiales"); */
       NLWARNING((IMPIT00,1," ", IMP));
       /* Dans ce cas, le code est NONFAIT */
       Code = NONFAIT ;
    } /* fin if ((ResNum->NbIter == 0) && (CtxPuss->MaxIter > 0)) */
   else
     /*   fprintf(GNLControle.SortImp,"Convergence atteinte");  */
  NLWARNING((IMPIT0,1," ", IMP));
  } /* fin de if (Code == OK) */

if (Code == WARPUSS2)
  {
  /*   fprintf(GNLControle.SortImp,"MaxIter atteinte");  */
  NLWARNING((IMPIT2,1," ",IMP));
  }
if (Code == WARPUSS3)
  {
  /*   fprintf(GNLControle.SortImp,"MaxDeb atteint");  */
  NLWARNING((IMPIT3,1," ",IMP));
  }

ResNum->CodePuss = Code;
  

/* Desallouer les structures de travail dans l'ordre inverse d'allocation. */
APPEL(DesalloueTA (&Trav1, &Trav2, &Trav3, &TravS,
                   &MatTrav,  &MatTrav1,  &MatTrav2, &BlocP, &BlocQ, &Bloc0, &BlocR));

APPEL(DesalloueT(  &ThetaCour, &BetaCour,  &ValR,
                   &Direc, &Correc,
                   &DEffT, &DEffB, &FctSA));

/* retour */
return(OK);
}
