/*--------------- COPYRIGHT ------------------------
| INRA - Laboratoire de Biometrie de Jouy en Josas |
--------------------------------------------------*/

/*--------------- IDENTIFICATION PRODUIT -----------
| Produit              : Transfo                   |
| Date                 :  7Nov91 / 16:31:54        |
| Derniere mise a jour :                           |
| Concepteur           : A. Bouvier                |
| Role                 : Programmes utilitaires de |
|   transformation des dimensions                  |
|   ``multiples'' a ``differents'' et ``actifs'' et|
|     vice-versa                                   |
| Reference conception :                           |
| Lecteur              : G. Husser                 |
--------------------------------------------------*/

/*--------------- HISTORIQUE -----------------------
|.
--------------------------------------------------*/

/*--------------- INCLUDES -----------------------*/
#include <math.h>
#include <stdio.h>
#include <errno.h>
#include <float.h>

#include "nlcodes.h"
#include "nlchoix.h"
#include "nltypes.h"
#include "nlglobal.h"
#include "nlmacros.h"
#include "errcodes.h"

/*--------------- VARIABLES EXTERNES -------------*/

/*--------------- FONCTIONS EXTERNES -------------*/

/* fonctions des autres modules */
TShortInt CopyVect(), CopyMat();
TShortInt GerMessage();
TShortInt CreerMat( TShortInt nblig, TShortInt nbcol, TMat *pmat);
TShortInt CreerVect( TShortInt nbele, TVect *pvect);
TShortInt CreerVectShort(TShortInt nbele, TVectShort *pvect);

void DetruVect(TVect *pvect);
void DetruVectShort(TVectShort *pvect);

/*--------------- CONSTANTES ---------------------*/

/*--------------- MACROS -------------------------*/

/*--------------- VARIABLES STATIQUES ------------*/

/*--------------- TYPES --------------------------*/

/*--------------- FIN IDENTIFICATION PRODUIT -----*/


/*--------------- Identification fonction ----------
| Nom de la fonction    : ActAMod                  |
| Role                  : Transformer les          |
|   parametres de la dimension "actifs" a la       |
|   dimension "multiples" et "differents"          |
| (fonction appelee successivement sur les Theta et|
|  Beta)                                           |
| Parametres d'entree   :                          |
|  ParamA: pointeur sur le vecteur   des parametres|
|   en dimension Actifs                            |
|   (le nombre d'elements doit etre positif)       |
|  ContrM: contraintes sur les parametres en       |
|   dimension "multiples"                          |
|  ContrE: contraintes sur les parametres en       |
|   dimension "differents"                         |
| Parametres d'entree-sortie  :                    |
|  ParamE: pointeur sur le vecteur   des parametres|
|   en dimension "differents"                      |
|  ParamM: pointeur sur le vecteur   des parametres|
|   en dimension "multiples"                       |
| En entree: ces structures ont ete allouees       |
| En sortie, leurs valeurs sont affectees          |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CopyVect, plus les        |
|  fonctions systeme "sin"                         |
| Fonctions appelantes : CModele                   |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt ActAMod(ParamA, ContrM, ContrE,
                  ParamE, ParamM)

  /* arguments d'entree */
  TVect *ParamA;
  TContr *ContrM;
  TContr *ContrE;

  /* arguments d'entree- sortie */
  TVect *ParamE, *ParamM;

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TShortInt IndM, IndE, IndA; 
/* respectivement , indices en dimension multiples, differents et actif */
  TDouble Valeur;
  TChar Mess4[5];
  TChar Message[5], Mess1[20], Mess2[20], Mess3[20]; 
  TShortInt egalnn, binfn, bsupn, NbEff, NbMod;

  /* pointeurs sur des elements de structure pour ameliorer la performance */
  TDouble *egaln, *binf, *bsup, *parame, *parama, *paramm;
  TShortInt *egalp;

 /* mettre son nom dans la trace */
  ECRTRACE("ActAMod");


  if(ContrM->Ctr ==NON )
    /* cas ou il n'y a pas de contrainte sur Param  */
    {
    APPEL(CopyVect(ParamA, ParamE));
    APPEL(CopyVect(ParamA, ParamM));
    }

  else
    {
    /* cas ou il y a des contraintes dessus */
    /*  calcul des  parametres differents */
    /* Tenir compte des contraintes d'egalite numerique et d'inegalite:
       transformation de dimension actif  a differents*/

   /* Affectation des pointeurs designant des elements de structures */
    egalp = ContrM->EgalP.donnees;
    egaln = ContrE->EgalN.donnees;
    binf = ContrE->BInf.donnees;
    bsup = ContrE->BSup.donnees;
    parame = ParamE->donnees;
    parama = ParamA->donnees;
    paramm = ParamM->donnees;
    egalnn = ContrE->EgalN.nbele;
    binfn = ContrE->BInf.nbele;
    bsupn = ContrE->BSup.nbele;
    NbEff = ParamE->nbele;
    NbMod = ParamM->nbele;

    IndA=0;
    for ( IndE =0; IndE <NbEff; IndE++)
      {
      if (((egalnn ==0) || myisnan(egaln[IndE])) &&
        ((binfn ==0) || myisnan(binf[IndE])) &&
        ((bsupn ==0) || myisnan(bsup[IndE])))
        {
        /* pas de contrainte numerique */
        parame[IndE]= parama[IndA];
        IndA=IndA+1;
        continue; /* on continue la boucle sur IndE */
        }

      if ((egalnn >0) && !myisnan(egaln[IndE]))    
        {
        /* contrainte numerique d'egalite sur IndE */
        parame[IndE]= egaln[IndE];
        continue; /* on continue la boucle sur IndE */
        }

      if (((binfn >0) && !myisnan(binf[IndE])) &&
          ((bsupn==0) || myisnan(bsup[IndE])))
        {
        /* contrainte : borne inferieure  et pas de borne superieure */
        parame[IndE]= 
            binf[IndE] +((parama[IndA])* (parama[IndA]));

        if ((errno==EDOM) || (errno== ERANGE))
          {
          /* overflow */
          sprintf(Mess3, "%hd", (IndA+1));
          sprintf(Message, "%hd", (IndE+1));
          sprintf(Mess1, "%g", parame[IndE]);
          sprintf(Mess2, "%g", binf[IndE]);

         /* Actions effectuees par la macro suivante NLERREUR: 
          fprintf(stderr,
  "ActAMod: Overflow dans la transformation du %s%-ieme parametre actif\
en le %s ieme parametre different\n", Mess3, Message);
          fprintf(stderr," La valeur du parametre differents est: %s\n", Mess1);
          fprintf(stderr," La valeur de sa borne inferieure est: %s\n", Mess2);
          return(ERROVFL1);
          fin actions de NLERREUR */
          NLERREUR((ERROVFL1,5,"ActAMod",Mess3,Message,Mess1, Mess2, ERR));
          }
        IndA=IndA+1;
        continue; /* on continue la boucle sur IndE */
        }

      if (((bsupn >0) && !myisnan(bsup[IndE] )) &&
          ((binfn==0) || myisnan(binf[IndE])))
        {
        /* contrainte : borne superieure  et pas inferieure */
        parame[IndE]= bsup[IndE] - ((parama[IndA])*(parama[IndA]));
        if ((errno==EDOM) || (errno== ERANGE))
          {
          /* overflow */
          sprintf(Mess3, "%hd", (IndA+1));
          sprintf(Message, "%hd", (IndE+1));
          sprintf(Mess1, "%g", parame[IndE]);
          sprintf(Mess2, "%g", bsup[IndE]);
         /* Actions effectuees par la macro suivante NLERREUR: 
          fprintf(stderr,
  "ActAMod: Overflow dans la transformation du %s%-ieme parametre actif\
en le %s ieme parametre different\n", Mess3, Message);
          fprintf(stderr," La valeur du parametre different est: %s\n", Mess1);
          fprintf(stderr," La valeur de sa borne superieure est: %s\n", Mess2);
          return(ERROVFL2);
          fin actions de NLERREUR */
          NLERREUR((ERROVFL2,5,"ActAMod",Mess3, Message,Mess1,Mess2,ERR));
          }
        IndA=IndA+1;
        continue; /* on continue la boucle sur IndE */
        }

      if (((bsupn >0) && !myisnan(bsup[IndE])) &&
          ((binfn  >0) && !myisnan(binf[IndE])))
        {
        /* borne inferieure et superieure */
        Valeur=sin(parama[IndA]);
        Valeur= Valeur*Valeur;
        parame[IndE]= binf[IndE] + ((bsup[IndE] - binf[IndE]) * Valeur);
        if ((errno==EDOM) || (errno== ERANGE))
          {
          sprintf(Mess4, "%hd", (IndA+1));
          sprintf(Message, "%hd", (IndE+1));
          sprintf(Mess1, "%g", parame[IndE]);
          sprintf(Mess2, "%g", binf[IndE]);
          sprintf(Mess3, "%g", bsup[IndE]);

         /* Actions effectuees par la macro suivante NLERREUR:
          fprintf(stderr,
"ActAMod: Overflow dans la transformation %s ieme parametre actif en le %s% ieme param. different\n", Mess4, Message);
          fprintf(stderr," La valeur du parametre different est: %s\n", Mess1);
          fprintf(stderr," La valeur de sa borne inferieure est: %s\n", Mess2);
          fprintf(stderr," La valeur de sa borne superieure est: %s\n", Mess3);
          return(ERROVFL3);
          fin actions de NLERREUR */
          NLERREUR((ERROVFL3,6,"ActAMod",Mess4, Message,Mess1,Mess2,Mess3, ERR));
          }
        IndA=IndA+1;
        }

     } /*  Fin boucle sur IndE */

  /* Transformation de differents a multiples */
    for ( IndM =0; IndM < NbMod; IndM++)
      {
      paramm[IndM]= parame[egalp[IndM]-1];
      } /*  Fin boucle sur IndM */

  } /* fin du else Ctr=NON */

  /* retour */
  return(OK);
  }


/*--------------- Identification fonction ----------
| Nom de la fonction    : CalcTr                   |
| Role                  : calculer le facteur par  |
|  lequel il faut multiplier une ligne de ValW     |
|  pour tenir compte des contraintes numeriques    |
|  d'inegalite                                     |
|  et mettre a jour l'indice courant du parametre  |
|  actif                                           |
| Parametres d'entree   :                          |
|   IndE: indice courant du parametre different    |
|   ParamA: valeur des parametres actifs           |
|   ContrE: contraintes sur les parametres         |
|   differents                                     |
| Parametres d'e/s      :                          |
|   IndA: en entree, indice courant du parametre   |
|     actif, en sortie, il est incremente si le    |
|     suivant est un autre parametre actif         |
| Parametres de sortie  :                          |
|   Val: facteur de correction de la ligne de ValW |
|     correspondant au i-ieme parametre actif,     |
|     i etant la valeur en entree de IndA          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : aucune, mais des fonctions|
|  systeme                                         |
| Fonctions appelantes :  TrValW                   |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt CalcTr(IndE, ParamA, ContrE, 
                 IndA,
                 Val)

/* arguments d'entree */
TShortInt IndE;
TVect *ParamA;
TContr *ContrE;

/* arguments d'entree-sortie */
TShortInt *IndA;

/* arguments de sortie */
TDouble *Val;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt bsupn, binfn, egalnn;
/* pointeurs pour ameliorer la performance */
TDouble *bsup, *binf, *egaln, *parama;

/* Ecriture de la trace */
ECRTRACE("CalcTr");

bsup = ContrE->BSup.donnees;
binf = ContrE->BInf.donnees;
egaln = ContrE->EgalN.donnees;
parama = ParamA->donnees;
egalnn = ContrE->EgalN.nbele;
bsupn = ContrE->BSup.nbele;
binfn = ContrE->BInf.nbele;

  while(1)
    {
    if (((egalnn ==0) || myisnan(egaln[IndE])) &&
      ((binfn ==0) || myisnan(binf[IndE])) &&
      ((bsupn ==0) || myisnan(bsup[IndE])))
      {
      /* pas de contrainte numerique */
      *Val = (TDouble)1;
      *IndA = *IndA + 1;
      break;
      }

     if ((ContrE->EgalN.nbele >0) && !myisnan(egaln[IndE]))
        {
        /* contrainte numerique d'egalite sur IndE */
        break;
        }

     if (((binfn >0) && !myisnan(binf[IndE])) &&
          ((bsupn==0) || myisnan(bsup[IndE])))
        {
        /* contrainte : borne inferieure  et pas de borne superieure */
        *Val = (TDouble)2 * parama[*IndA];
        *IndA = *IndA + 1;
        break; /* on continue la boucle sur IndE */
        }

     if (((bsupn >0) && !myisnan(bsup[IndE] )) &&
          ((binfn==0) || myisnan(binf[IndE])))
        {
        /* contrainte : borne superieure  et pas inferieure */
        *Val = -  ((TDouble)2 *  parama[*IndA]);
        *IndA = *IndA + 1;
        break; /* on continue la boucle sur IndE */
        }

      if (((bsupn >0) && !myisnan(bsup[IndE])) &&
          ((binfn  >0) && !myisnan(binf[IndE])))
        {
        /* borne inferieure et superieure */
        *Val = (TDouble)2 * (bsup[IndE] - binf[IndE])
        * sin(parama[*IndA]) * cos(parama[*IndA]);
        *IndA = *IndA + 1;
        break;
        }
    } /* fin du while */

  return(OK);
}

/*--------------- Identification fonction ----------
| Nom de la fonction    : DModAAct                 |
| Role                  : Transformer les          |
|   derivees de la fonction f ou v de la           |
|   dimension "multiples" a la dimension "actifs"  |
|   en passant par la dimension "differents"       |
| Parametres d'entree   :                          |
|  Ctr: VRAI s'il y a des contraintes de n'importe |
|       quel type sur les parametres               |
|  NbObs: nombre d'observations                    |
|  NbObsC: vecteur du nombre d'observations par    |
|   courbe                                         |
|  ParamA: pointeur sur le vecteur   des parametres|
|   en dimension Actifs                            |
|   (le nombre d'elements doit etre positif)       |
|  DMod: pointeur sur la matrice des derivees      |
|   (nombre de lignes=nombre d'observations),      |
|   (nombre de colonnes=nombre de parametres du    |
|    modele)                                       | 
|  Pass: pointeur sur la matrice de passage de la  |
|   dimension multiples a differents               |
|  ContrE: contraintes sur les parametres en       |
|   dimension "differents"                         |
| Parametres d'entree-sortie  :                    |
|   DEff: matrice de travail pour stocker les      |
|    derivees en dimension differents              |
|   DAct: matrice des derivees en dimension actifs |
| En entree: ces structures ont ete allouees       |
| En sortie, leurs valeurs sont affectees          |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|    ATTENTION: appelle les fonctions systemes     |
|   "sin" (argument et retour de type double )     |
| Fonctions appelantes : CModele, CCovNu2          |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt DModAAct(Ctr, NbObs, NbObsC, ParamA, DMod, Pass, ContrE,
                  DEff, DAct)

  /* arguments d'entree */
  TLogic Ctr;
  TLongInt NbObs;
  TVectLong *NbObsC;
  TVect *ParamA;     /* longueur= nombre de parametres actifs de la sorte
                                  consideree */
  TMat *DMod;       /*  NbObs lignes et nombre de parametres du modele
                        colonnes */
  TMatShort *Pass;    /* nombre de lignes= nombre de parametres du modele
                                           de la sorte consideree * nombre
                                           de courbes
                         nombre de colonnes= nombre de parametres differents
                                            de la sorte consideree */
  TContr *ContrE;    

  /* arguments d'entree- sortie */
  TMat *DEff;   /* nombre de colonnes= nombre de parametres differents,
                   nombre de lignes= NbObs (nombre d'observations differentes) */
  TMat *DAct;   /* nombre de colonnes= nombre de parametres actifs
                   nombre de lignes= NbObs (nombre d'observations differentes) */

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TShortInt NbMod;  /* nombre de parametres du modele dans la sorte
                       consideree (Theta ou bien Beta) */
  TShortInt NbEff; /* nombre de parametres differents */
  TShortInt IndM, IndE, IndA; /* respectivement , indices en dimension 
                                 multiples, differents et actif */
  TShortInt IObs, j, ICourbe, IPred, MPred, NbCourbe;
  TChar Message[5], Mess1[20], Mess2[20], Mess3[20]; 
  TShortInt egalnn, binfn, bsupn, deffnl;

  /* pointeurs sur des elements de structure pour ameliorer la performance */
  TDouble *egaln, *binf, *bsup,  *parama;
  TDouble **deff, **dmod, **dact;
  TLongInt *nbobsc;
  TShortInt **pass;


/* mettre son nom dans la trace */
  ECRTRACE("DModAAct");


/* Initialisation */  
  NbMod= DMod->nbcol;
  NbEff=DEff->nbcol;
  NbCourbe=NbObsC->nbele;

   /* Affectation des pointeurs designant des elements de structures */
  egaln = ContrE->EgalN.donnees;
  binf = ContrE->BInf.donnees;
  bsup = ContrE->BSup.donnees;
  parama = ParamA->donnees;
  egalnn = ContrE->EgalN.nbele;
  binfn = ContrE->BInf.nbele;
  bsupn = ContrE->BSup.nbele;

  nbobsc = NbObsC->donnees;
  deff = DEff->donnees;
  dmod = DMod->donnees;
  dact = DAct->donnees;
  pass = Pass->donnees;

/*  calcul des derivees par rapport aux parametres differents */
/* --------------------------------------------------------- */
IPred=0;  /* indice de la 1iere observation de la courbe courante */
MPred=0;  /* indice du 1ier parametre du modele de la courbe courante
          parmi les NbMod*NbCourbe parametres */
for(ICourbe=0; ICourbe<NbCourbe; ICourbe++)
  {  
  for(IObs=0; IObs<(TShortInt)nbobsc[ICourbe]; IObs++)
    {
    for(IndE=0; IndE <NbEff; IndE++)
      {
      deff[IObs+IPred][IndE]=(TDouble)ZERO;
      for(IndM=0; IndM < NbMod; IndM++)
        {
        deff[IObs+IPred][IndE]=deff[IObs+IPred][IndE]+
         (dmod[IObs+IPred][IndM] * (TDouble)pass[IndM+MPred][IndE]);
        }
      } /* fin boucle sur IndE */
    } /* fin boucle sur IObs */
  IPred=IPred+(TShortInt)nbobsc[ICourbe];
  MPred=MPred+ NbMod;
  } /* fin boucle sur ICourbe */


/* calcul des derivees par rapport aux parametres actifs */
/* ----------------------------------------------------- */
  if( Ctr==NON)  /* cas sans contrainte */
    {
    /* derivees par rapport a actifs= derivees par rapport a differents */
    for(IObs=0; IObs<(TShortInt)NbObs; IObs++)
      {
      for(IndE=0; IndE <NbEff; IndE++)
        {
        dact[IObs][IndE]=deff[IObs][IndE];
        }
      }
/* Remarque: ne pas mettre l'instruction suivante:
    *DAct= *DEff;   
 car ce programme va etre appele 2 fois avec le meme DEff :
 si DAct est un pointeur sur la meme zone que DEff, ses valeurs seront
 demolies au 2ieme passage */
    }
  else
    {  /* cas avec contrainte */
    deffnl = DEff->nblig;
    IndA=0;
    for(IndE=0; IndE< NbEff; IndE++)
      {
      if (((egalnn ==0) || myisnan(egaln[IndE])) &&
        ((binfn ==0) || myisnan(binf[IndE])) &&
        ((bsupn ==0) || myisnan(bsup[IndE])))
        {
        /* pas de contrainte numerique */
        for(j=0; j<deffnl; j++)
          {
          dact[j][IndA]= deff[j][IndE]; /* egalite de colonnes */
          }
        IndA=IndA+1;
        continue;
        }

      if ((egalnn >0) && !myisnan(egaln[IndE]))
        {
        /* contrainte numerique d'egalite */
        continue;
        }

      if (((binfn >0) && !myisnan(binf[IndE])) &&
          ((bsupn==0) || myisnan(bsup[IndE])))
        {
        /* contrainte : borne inferieure  et pas de borne superieure */
        for(j=0; j<deffnl; j++)
          {
          dact[j][IndA]= (TDouble)2 * (parama[IndA]) * (deff[j][IndE]);
          }
        IndA=IndA+1;
        continue;
        }

      if (((bsupn >0) && !myisnan(bsup[IndE])) &&
          ((binfn==0) || myisnan(binf[IndE])))
        {
        /* contrainte : borne superieure  et pas inferieure */
        for(j=0; j<deffnl; j++)
          {
          dact[j][IndA]= (TDouble)(-2) *parama[IndA]*deff[j][IndE];
          }
        IndA=IndA+1;
        continue;
        }

      if (((bsupn >0) && !myisnan(bsup[IndE] )) &&
          ((binfn  >0) && !myisnan(binf[IndE])))
        {
        /* borne inferieure et superieure */
        for(j=0; j<deffnl; j++)
          {
          dact[j][IndA]= (bsup[IndE] - binf[IndE]) * sin((TDouble)2*parama[IndA]);
          dact[j][IndA]=dact[j][IndA] * deff[j][IndE];
          if ((errno==EDOM) || (errno== ERANGE))
            {
          /* overflow */
          sprintf(Message, "%hd", (IndA+1));
          sprintf(Mess1, "%g", dact[j][IndA]);
          sprintf(Mess2, "%g", binf[IndE]);
          sprintf(Mess3, "%g", bsup[IndE]);

         /* Actions effectuees par la macro suivante NLERREUR:
          fprintf(stderr,"DModAAct: Overflow dans le calcul  de la derivee du %s ieme parametre actif\n", Message);
          fprintf(stderr," La valeur de la derivee est: %s\n", Mess1);
          fprintf(stderr," La valeur de la borne inferieure est: %s\n", Mess2);
          fprintf(stderr," La valeur de la borne superieure est: %s\n", Mess3);
          return(ERROVFL5);
          fin actions de NLERREUR */
          NLERREUR((ERROVFL5,5,"DModAAct",Message,Mess1,Mess2,Mess3, ERR));
            } /* fin overflow */
          } /* fin du for j */
        IndA=IndA+1;
        continue;
        } /* fin borne inf et sup */
      } /* fin boucle sur IndE */
    } /* fin du else Ctr==NON */

  /* retour */
  return(OK);
  }


/*--------------- Identification fonction ----------
| Nom de la fonction    : ModAAct                  |
| Role                  : Transformer les          |
|   parametres de la dimension "multiples" a la    |
|   dimension "actifs" et "differents"             |
| (fonction appelee successivement sur les Theta et|
|  Beta)                                           |
| Parametres d'entree   :                          |
|  ParamM: pointeur sur le vecteur   des parametres|
|   en dimension multiples                         |
|  ContrM: contraintes sur les parametres en       |
|   dimension "modele"                             |
|  ContrE: contraintes sur les parametres en       |
|   dimension "differents"                         |
| Parametres d'entree-sortie  :                    |
|  ParamE: pointeur sur le vecteur   des parametres|
|   en dimension "differents"                      |
|  ParamA: pointeur sur le vecteur   des parametres|
|   en dimension "actifs"                          |
| En entree: ces structures ont ete allouees       |
| En sortie, leurs valeurs sont affectees          |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   : CopyVect                  |
| Fonctions appelees   : aucune, mais la fonction  |
|  systeme "asin"                                  |
| Fonctions appelantes : MajContP, CCovNu2         |
--------------------------------------------------*/

/*--------------- Definition fonction ------------*/
TShortInt ModAAct(ParamM, ContrM, ContrE,
                  ParamE, ParamA)

  /* arguments d'entree */
  TVect *ParamM;
  TContr *ContrM;
  TContr *ContrE;

  /* arguments d'entree- sortie */
  TVect *ParamE, *ParamA;

/*--------------- Fin identification fonction ----*/
  {

  /* locals */
  TShortInt IndM, IndE, IndA; 
/* respectivement , indices en dimension multiples, differents et actif */
  TChar Mess4[5];
  TChar Message[5], Mess1[20], Mess2[20], Mess3[20]; 

  TShortInt egalnn, binfn, bsupn, NbEff, NbMod;

  /* pointeurs sur des elements de structure pour ameliorer la performance */
  TDouble *egaln, *binf, *bsup, *parame, *parama, *paramm;
  TShortInt *egalp;

/* mettre son nom dans la trace */
  ECRTRACE("ModAAct");



  if(ContrM->Ctr ==NON )
    /* cas ou il n'y a pas de contrainte sur Param  */
    {
    APPEL(CopyVect(ParamM, ParamE));
    APPEL(CopyVect(ParamM, ParamA));
    }

  else
    {
    /* cas ou il y a des contraintes dessus */
    /*  calcul des  parametres differents */

   /* Affectation des pointeurs designant des elements de structures */
    egalp = ContrM->EgalP.donnees;
    egaln = ContrE->EgalN.donnees;
    binf = ContrE->BInf.donnees;
    bsup = ContrE->BSup.donnees;
    parame = ParamE->donnees;
    parama = ParamA->donnees;
    paramm = ParamM->donnees;
    egalnn = ContrE->EgalN.nbele;
    binfn = ContrE->BInf.nbele;
    bsupn = ContrE->BSup.nbele;
    NbEff = ParamE->nbele;
    NbMod = ParamM->nbele;


    IndE=0;
    for ( IndM =0; IndM < NbMod; IndM++)
      {
      if (egalp[IndM] > IndE)
        {
        /* IndM n'est pas egal a un parametre precedent */
        parame[IndE]= paramm[IndM];
        IndE=IndE+1;
        }

      } /*  Fin boucle sur IndM */

    /* Tenir compte des contraintes d'egalite numerique et d'inegalite:
       transformation en dimension actif */
    IndA=0;
    for ( IndE =0; IndE <NbEff; IndE++)
      {
      if (((egalnn ==0) || myisnan(egaln[IndE])) &&
        ((binfn ==0) || myisnan(binf[IndE])) &&
        ((bsupn ==0) || myisnan(bsup[IndE])))
        {
        /* pas de contrainte numerique */
        parama[IndA]= parame[IndE];
        IndA=IndA+1;
        continue; /* on continue la boucle sur IndE */
        }

      if ((egalnn >0) && !myisnan(egaln[IndE]))    
        /* contrainte numerique d'egalite sur IndE */
        {
        continue; /* on continue la boucle sur IndE */
        }

      if (((binfn >0) && !myisnan(binf[IndE] )) &&
          ((bsupn==0) || myisnan(bsup[IndE])))
        {
        /* contrainte : borne inferieure  et pas de borne superieure */
        parama[IndA]= sqrt(parame[IndE]-binf[IndE]);
        IndA=IndA+1;
        continue; /* on continue la boucle sur IndE */
        }
  
      if (((bsupn >0) && !myisnan(bsup[IndE])) &&
          ((binfn==0) || myisnan(binf[IndE])))
        {
        /* contrainte : borne superieure  et pas inferieure */
        parama[IndA]= sqrt(bsup[IndE]-parame[IndE]);
        IndA=IndA+1;
        continue; /* on continue la boucle sur IndE */
        }

      if (((bsupn >0) && !myisnan(bsup[IndE])) &&
          ((binfn  >0) && !myisnan(binf[IndE])))
        {
        /* borne inferieure et superieure: seul cas ou les cas d'overflow
           ne peuvent etre pris en charge en amont (c.a.d lors des verifications
           des contraintes) */
        parama[IndA]= asin(sqrt((parame[IndE]-binf[IndE]) /
                         (bsup[IndE]- binf[IndE])));
        if ((errno==EDOM) || (errno== ERANGE))
          {
          /* overflow */
          sprintf(Message, "%hd", (IndA+1));
          sprintf(Mess4, "%hd", (IndE+1));
          sprintf(Mess1, "%g", parama[IndE]);
          sprintf(Mess2, "%g", binf[IndE]);
          sprintf(Mess3, "%g", bsup[IndE]);

         /* Actions effectuees par la macro suivante NLERREUR:
          fprintf(stderr,
 "ModAAct: Overflow dans la transformation du %s ieme parametre different en le %s ieme param. actif\n", 
 Mess4, Message);
          fprintf(stderr," La valeur du parametre actif est: %s\n", Mess1);
          fprintf(stderr," La valeur de sa borne inferieure est: %s\n", Mess2);
          fprintf(stderr," La valeur de sa borne superieure est: %s\n", Mess3);
          return(ERROVFL4);
          fin actions de NLERREUR */

          NLERREUR((ERROVFL4,6,"ModAAct",Mess4,Message,Mess1,Mess2,Mess3, ERR));
          }
        IndA=IndA+1;
        }

      } /*  Fin boucle sur IndE */
  } /* fin du else Ctr=NON */

  /* retour */
  return(OK);
  }

    

/*--------------- Identification fonction ----------
| Nom de la fonction    : TrValW                   |
| Role                  : Transformer ValW de facon|
|  a tenir compte des contraintes numeriques       |
|  d'inegalite sur les parametres                  |
| Parametres d'entree   :                          |
|  NbTEff : nombre de parametres differents Theta  |
|  NbBEff : nombre de parametres differents Beta   |
|  TAct: les valeurs des parametres actifs Theta   |
|  BAct: les valeurs des parametres actifs Beta    |
|  CThetaE: les contraintes sur les parametres     |
|   differents Theta                               |
|  CBetaE: les contraintes sur les parametres      |
|   differents Beta                                |
|  type: determine la dimension de la matrice ValW |
|       0 si (nTAct+nBAct, nTAct+nBAct)            |
|       1 si (nBAct,nTAct), 2 si (nTAct,nBAct)     |                       
|        (rajout du 3/12/99)                       |
| Parametres d'e/s      :                          |
|  ValW: en entree la matrice ValW calculee sur    |
|    les parametres actifs et, en sortie, elle     |
|    est calculee sur les parametres nu, c.a.d les |
|    parametres actifs apres transformations dues  |
|    aux contraintes numeriques d'inegalite        |
|  Attention: quand ce programme est appele par    |
|  CCovNu2t, le nombre de lignes et de colonnes est|
|  different                                       |
| Parametres de sortie  :                          |
| Retour fonction       : OK                       |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :  CalcTr                   |
| Fonctions appelantes :  CCov2 ,                  |
|                         CCov1                    |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt TrValW(type,
                 NbTEff, NbBEff, TAct, BAct, CThetaE, CBetaE,
                 ValW)

/* arguments d'entree */
TShortInt type, NbTEff, NbBEff;
TVect *TAct, *BAct;
TContr *CThetaE, *CBetaE;

/* arguments d'entree-sortie */
TMat *ValW;

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt IndA, IndAsv, IndE, i, nblig, nbcol, NbTAct;
TDouble Val;

/* pointeurs sur des elements de structure pour ameliorer la performance */
TDouble **valw;

/* Ecriture de la trace */
ECRTRACE("TrValW");

  valw = ValW->donnees;
  nblig = ValW->nblig;
  nbcol = ValW->nbcol;
  NbTAct = TAct->nbele;

  IndA=0;

  /* transformation de la partie concernant les Theta */

  for (IndE=0; IndE< NbTEff; IndE++)
    {
    IndAsv = IndA;
    APPEL(CalcTr(IndE, TAct, CThetaE, &IndA, &Val));
 
    if (IndA != IndAsv)
      {
	if ( (type==0) || (type ==1)) /*  RAJOUT AB 3/12/99 */
	{
	if (IndAsv < nbcol) /*  RAJOUT AB 26/11/99 du test */
	  {
	    for (i=0; i<nblig; i++)
	      {
		valw[i][IndAsv] = valw[i][IndAsv] * Val;
	      }
	  } /* fin if (IndAsv < nbcol) */
	else
	  {
	  fprintf(stderr,
"Problme d'indice dans TrValW (Transfo.c)\n");
	  exit(1);
	  }

	} /* fin if ( (type==0) || (type ==1)) */


      if ((type==0) || (type==2))
	{
	if (IndAsv < nblig) /* RAJOUT AB 26/11/99 du test */
	  {
	    for (i=0; i<nbcol; i++)
	      {
		valw[IndAsv][i] = valw[IndAsv][i] * Val;
	      }
	  } /* fin if (IndAsv < nblig) */
	else
	  {
	  fprintf(stderr,
"Problme d'indice dans TrValW (Transfo.c)\n");
	  exit(1);
	  }
	} /* fin   if ((type==0) || (type==2)) */
          
      } /* fin if (IndA != IndAsv) */
    
    } /* fin boucle sur IndE */




  /* transformation de la partie concernant les Beta */

if (type != 0 )
     NbTAct = 0;

  IndA=0;

  for (IndE=0; IndE< NbBEff; IndE++)
    {
    IndAsv = IndA;
    APPEL(CalcTr(IndE, BAct, CBetaE, &IndA, &Val));

    if (IndA != IndAsv)
      {
	if ((type==0) || (type==2))
	    {
      for (i=0; i<nblig; i++)
        {
	  if (( NbTAct+IndAsv)<nbcol) /* RAJOUT 26/11/99 */
	    {
	      valw[i][NbTAct+IndAsv] = valw[i][NbTAct+IndAsv] * Val;
	    }  /* fin if (( NbTAct+IndAsv)<nbcol) */
	else
	  {
	  fprintf(stderr,
"Problme d'indice dans TrValW (Transfo.c)\n");
	  exit(1);
	  }
        } /* fin for (i=0; i<nblig; i++) */

	    } /* fin type=0 ou 2 */

if ((type==0) || (type==1))
    {
      if ( ( NbTAct+IndAsv)<nblig) /*  RAJOUT 26/11/99 */
	{
	  for (i=0; i<nbcol; i++)
	    {
	      valw[NbTAct+IndAsv][i] = valw[NbTAct+IndAsv][i] * Val;
	    }
	}/* fin if ( ( NbTAct+IndAsv)<nblig) */
	else
	  {
	  fprintf(stderr,
"Problme d'indice dans TrValW (Transfo.c)\n");
	  exit(1);
	  }
    } /* fin  if ((type==0) || (type==1)) */

      } /* fin if (IndA != IndAsv) */
    
    } /* fin for (IndE=0; IndE< NbBEff; IndE++) */

return(OK);
}


/*--------------- Identification fonction ----------
| Nom de la fonction    : VNuAVMod                 |
| Role                  : passer de la matrice de  |
|  variance-covariance asymptotique en dimension   |
|  ``parametres actifs''  a la matrice en          |
|  dimension ``parametres du modele * NbCourbe''   |
| Parametres d'entree   :                          |
|  AsVarNu: la matrice de variance-covariance      |
|   asymptotique en dimension ``parametres actifs''|
|   nblig=nbcol=nombre de parametres actifs        |
|  Theta: la structure des parametres Theta        |
|  Beta: la structure des parametres Beta          |
|  CTheta, CBeta: les contraintes sur les          |
|   parametres du modele                           |
|  Dans celles-ci, le composant Ctr                |
|   =VRAI s'il y a des contraintes de n'importe    |
|   quel type sur les parametres correspondants    |
|  S'il n'y a pas de parametres  Beta,             |
|   Ctr est FAUX                                   |
|  CThetaE: les contraintes sur les parametres     |
|  differents Theta                                |
|  CBetaE: les contraintes sur les parametres      |
|  differents Beta                                 |
| Parametres d'e/s      :                          |
| Parametres de sortie  :                          |
|  AsVar: matrice de variance-covariance           |
|    asymptotique en dimension                     |
|    ``parametres multiples''                      |
|    non allouee avant l'appel                     |
|    nblig=nbcol= NbCourbe*nombre de parametres du |
|                 modele (param. multiples)        |
| Retour fonction       : OK ou code d'erreur      |
| Reference conception  :                          |
--------------------------------------------------*/

/*--------------- Appels croises -------------------
| Fonctions appelees   :                           |
|                       CopyMat,                   |
|                       CreerMat,                  |
|                       CreerVect,                 |
|                       DetruVect,                 |
|                       CreerVectShort,            |
|                       DetruVectShort,            |
| Fonctions appelantes :  CCov1, CCov2             |
--------------------------------------------------*/


/*--------------- Definition fonction ------------*/

TShortInt VNuAVMod( AsVarNu, Theta, Beta, CTheta, CBeta,  CThetaE, CBetaE,
                  AsVar)


/* arguments d'entree */
TMat *AsVarNu; /* dimension NbAct, NbAct */
TParam *Theta , *Beta;
TContr *CTheta, *CBeta;
TContr *CThetaE, *CBetaE;


/* arguments de sortie */
TMat *AsVar; /* dimension NbCourbe*(NbTheta+NbBeta), NbCourbe*(NbTheta+NbBeta) */

/*--------------- Fin identification fonction ----*/

{
/* locals */
TShortInt i,j,IndA, IndAC, IndE, IndEC, IndM, IndMC;
TShortInt NbEff, NbTEff, NbBEff, NbMod, NbTMod, NbBMod;
TVect EgalN;
TVectShort EgalP;
TMat AsVarEff;


/* Ecriture de la trace */
ECRTRACE("VNuAVMod");

/* initialisations */
NbTMod = Theta->Init.nbele;
NbBMod = Beta->Init.nbele;
NbMod = NbTMod + NbBMod;
NbTEff = Theta->Eff.nbele;
NbBEff = Beta->Eff.nbele;
NbEff = NbTEff + NbBEff;

/* allouer la sortie */
APPEL(CreerMat(NbMod, NbMod, AsVar));

if ((CTheta->Ctr == NON) && (CBeta->Ctr == NON))
  {
  /* cas sans contrainte */
  APPEL(CopyMat(AsVarNu, AsVar));
  /* retour */
  return(OK);
  }

/*  tenir compte des egalites numeriques: calcul de AsVarEff */
APPEL(CreerMat(NbEff, NbEff, &AsVarEff));

/* initialisation de AsVarEff */
for (i=0; i<NbEff; i++)
  {
  for(j=0; j<NbEff; j++)
    {
    AsVarEff.donnees[i][j]= mysignaling_nan(n); 
    /* l'argument n ne sert a rien dans cette version du compilateur */
    }
  }


if ((CThetaE->EgalN.nbele == 0) && (CBetaE->EgalN.nbele == 0))
  {
  /* il y a des contraintes mais aucune d'egalite numerique:
   les parametres differents sont ceux actifs */
  APPEL(CopyMat(AsVarNu, &AsVarEff));
  }

else
  {
/* on va concatener les EgalN des Theta et Beta pour faciliter la
   boucle de traitement */
APPEL(CreerVect(NbEff, &EgalN));

for (i=0; i<NbTEff; i++)
  {
  if (CThetaE->EgalN.nbele == 0)
    {
    EgalN.donnees[i]= mysignaling_nan(n);
    }
  else
    {
    EgalN.donnees[i]=CThetaE->EgalN.donnees[i];
    }
  } /* fin boucle sur i */


for (i=0; i<NbBEff; i++)
  {
  if (CBetaE->EgalN.nbele == 0)
    {
    EgalN.donnees[NbTEff + i]= mysignaling_nan(n);
    }
  else
    {
    EgalN.donnees[NbTEff+i]=CBetaE->EgalN.donnees[i];
    }
  } /* fin boucle sur i */

/* les elements AsVarEff(i,j) restent a VALMANQ
  si i ou j est un parametre fixe,
  et on met la valeur qui correspond dans AsVarNu sinon */

IndA = -1; /* indice du parametre actif qui va correspondre
            au IndE-ieme parametre different */

for(IndE=0; IndE<NbEff; IndE++)
  {
  if ( myisnan(EgalN.donnees[IndE]))
    {
    /* pas d'egalite numerique sur le IndE ieme parametre */
    IndA = IndA + 1;
    IndAC= IndA; /* IndAC est l'indice du parametre actif qui
                 va correspondre au IndEC-ieme parametre different */

    /* mise a jour du reste de la ligne et colonne */
    for (IndEC=IndE; IndEC<NbEff; IndEC++)
      {
      if (myisnan(EgalN.donnees[IndEC]))
        {
        /* pas d'egalite numerique sur le IndEC ieme parametre */
        AsVarEff.donnees[IndE][IndEC]=AsVarNu->donnees[IndA][IndAC];
        AsVarEff.donnees[IndEC][IndE]=AsVarNu->donnees[IndAC][IndA];
        IndAC = IndAC + 1;
        }
      } /* fin boucle sur IndEC */
    } /* fin de pas d'egalite numerique */
  } /* fin boucle sur IndE */

DetruVect(&EgalN);
  
} /* fin du else pas de contrainte numerique */


/* tenir compte des egalites entre parametres: calcul de AsVar  */

/* concatener les EgalP pour faciliter la
   boucle de traitement */
/* si pas d'egalite entre parametres, EgalP aura ete initialise par 1,2,3.. */
APPEL(CreerVectShort(NbMod, &EgalP));
for (i=0; i<NbTMod; i++)
  {
  if (CTheta->EgalP.nbele > 0)
    {
    EgalP.donnees[i] = CTheta->EgalP.donnees[i];
    }
  else
    {
    /* l'initialiser a -1: -1= pas de parametres differents de la sorte consideree */
    EgalP.donnees[i]= -1;
    }
  } /* fin boucle sur i<NbTMod */

j = NbTMod;
for (i=0; i<NbBMod; i++)
  {
  if (CBeta->EgalP.nbele > 0)
    {
    /* incrementer les valeurs de EgalP qui concernent les Beta
    par le nombre de Theta differents */
    EgalP.donnees[j] = CBeta->EgalP.donnees[i] + NbTEff;
    }
  else
    {
    /* l'initialiser a -1: -1= pas de parametres differents de la sorte consideree */
    EgalP.donnees[j]= -1;
    }
  j = j + 1;
  } /* fin boucle sur i<NbBMod */


for (IndM=0; IndM<NbMod; IndM++)
  {
  for (IndMC=0; IndMC < NbMod; IndMC++)
    {
    if ((EgalP.donnees[IndM] >0) && (EgalP.donnees[IndMC] > 0))
      {
      AsVar->donnees[IndM][IndMC]= AsVarEff.donnees[EgalP.donnees[IndM]-1][EgalP.donnees[IndMC]-1];
      }
    else
      {
      AsVar->donnees[IndM][IndMC]=mysignaling_nan(n);
      }
    } /* fin boucle sur IndMC */
  } /* fin boucle sur IndM */


/* mettre a VALMANQ les lignes, colonnes correspondant a des parametres
   egaux */

IndE = 1; /* indice courant en dimension differents */
for (IndM=0; IndM<NbMod; IndM++)
  {
  if ((EgalP.donnees[IndM] >= IndE) || (EgalP.donnees[IndM] > 0))
    {
    /* le IndM-ieme parametre n'est pas egal a un precedent */
    IndE = IndE  + 1;
    }
  else
    {
    /* mise a Valmanq de la IndM-ieme ligne et colonne */
    for (i=0; i<NbMod; i++)
      {
      AsVar->donnees[IndM][i] =  mysignaling_nan(n);
      AsVar->donnees[i][IndM] =  AsVar->donnees[IndM][i];
      }
    }
  } /* fin boucle sur IndM */
     

DetruVectShort(&EgalP);


return(OK);
}
